--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with System_Calls;

with Narval.Random_Draw_Thousand;

package body Narval.Communication.Fifos is

   Random_Generator : Random_Draw_Thousand.Draw_Package.Generator;

   -----------------------------------
   -- Recuperer_Info_Initialisation --
   -----------------------------------

   function Get_Init_Info (Link : access Fifo_Link_Type)
                                          return Integer is
   begin
      return Link.Root;
   end Get_Init_Info;

   ------------
   -- Stream --
   ------------

   type Fifos_Type is new Ada.Streams.Root_Stream_Type with record
      Fd_Send : GNAT.OS_Lib.File_Descriptor;
      Fd_Receive : GNAT.OS_Lib.File_Descriptor;
   end record;
   type Fifos_Type_Access is access all Fifos_Type;
   procedure Read
     (Stream : in out Fifos_Type;
      Item   : out Ada.Streams.Stream_Element_Array;
      Last   : out Ada.Streams.Stream_Element_Offset);
   procedure Write
     (Stream : in out Fifos_Type;
      Item   : Ada.Streams.Stream_Element_Array);
   procedure Free is new Ada.Unchecked_Deallocation (Fifos_Type,
                                                     Fifos_Type_Access);

   function Stream (Link : access Fifo_Link_Type) return Stream_Access is
      Fifo : Fifos_Type_Access;
   begin
      Fifo := new Fifos_Type;
      Fifo.Fd_Send := Link.Fd_Send;
      Fifo.Fd_Receive := Link.Fd_Receive;
      return Stream_Access (Fifo);
   end Stream;

   procedure Free (Link : access Fifo_Link_Type;
                   Stream_To_Free : in out Stream_Access) is
      pragma Unreferenced (Link);
      Fifo : Fifos_Type_Access;
   begin
      Fifo := Fifos_Type_Access (Stream_To_Free);
      Free (Fifo);
      Stream_To_Free := null;
   end Free;

   -------------
   -- Envoyer --
   -------------

   procedure Send (Link : access Fifo_Link_Type;
                      Data : Ada.Streams.Stream_Element_Array) is
      Function_Return : Integer;
      Send_Problem : exception;
   begin
      Function_Return := GNAT.OS_Lib.Write (Link.Fd_Send,
                                   Data (Data'First)'Address,
                                   Data'Length);
      if Function_Return /= Data'Length then
         raise Send_Problem;
      end if;
   end Send;

   --------------
   -- Recevoir --
   --------------

   procedure Receive (Link : access Fifo_Link_Type;
                       Data : out Ada.Streams.Stream_Element_Array) is
      Function_Return : Integer;
      Receive_Problem : exception;
   begin
      Function_Return := GNAT.OS_Lib.Read (Link.Fd_Receive,
                                  Data (Data'First)'Address,
                                  Data'Length);
      if Function_Return /= Data'Length then
         raise Receive_Problem;
      end if;
   end Receive;

   -------------
   -- Etablir --
   -------------

   procedure Connect (Link : access Client_Fifo_Link_Type;
                      Bit_Order_To_Send : in out System.Bit_Order) is
      Raw_Channel : Stream_Access := Stream (Link);
   begin
      Log4ada.Loggers.Debug_Out
        (Link.Logger,
         "*** enter : Etablir Type_Lien_Fifo_Client ***" & Link.Root'Img);
      Bit_Order_To_Send := System.Bit_Order'Input (Raw_Channel);
      Log4ada.Loggers.Debug_Out
        (Link.Logger,
         "*** Etablir Type_Lien_Fifo_Client step 1***");
      Free (Fifos_Type_Access (Raw_Channel));
      Log4ada.Loggers.Debug_Out
        (Link.Logger,
         "*** exit : Etablir Type_Lien_Fifo_Client ***");
   end Connect;

   -----------
   -- Clore --
   -----------

   procedure Close (Link : access Client_Fifo_Link_Type) is
   begin
      GNAT.OS_Lib.Close (Link.Fd_Send);
      GNAT.OS_Lib.Close (Link.Fd_Receive);
   end Close;

   -------------
   -- Etablir --
   -------------

   procedure Connect (Link : access Server_Fifo_Link_Type;
                      Bit_Order_To_Send : in out System.Bit_Order) is
      Raw_Channel : Stream_Access := Stream (Link);
   begin
      Log4ada.Loggers.Debug_Out
        (Link.Logger,
         "*** enter : Etablir Type_Lien_Fifo_Serveur ***" & Link.Root'Img);
      System.Bit_Order'Output (Raw_Channel, Bit_Order_To_Send);
      Log4ada.Loggers.Debug_Out
        (Link.Logger,
         "*** Etablir Type_Lien_Fifo_Serveur setp 1 ***");
      Free (Fifos_Type_Access (Raw_Channel));
      Log4ada.Loggers.Debug_Out
        (Link.Logger,
         "*** exit : Etablir Type_Lien_Fifo_Serveur ***");
   end Connect;

   -----------
   -- Clore --
   -----------

   procedure Close (Link : access Server_Fifo_Link_Type) is
      Root : constant String := Integer'Image (Link.Root);
      Last : constant Integer := Root'Last;
      Success : Boolean := False;
   begin
      GNAT.OS_Lib.Close (Link.Fd_Send);
      GNAT.OS_Lib.Close (Link.Fd_Receive);
      GNAT.OS_Lib.Delete_File ("/tmp/fifo_emission_" &
                               Root (2 .. Last),
                               Success);
      if not Success then
         Log4ada.Loggers.Warn_Out (Link.Logger,
                                   "probleme effacement fichier emission");
      end if;
      GNAT.OS_Lib.Delete_File ("/tmp/fifo_reception_" &
                               Root (2 .. Last),
                               Success);
      if not Success then
         Log4ada.Loggers.Warn_Out (Link.Logger,
                                   "probleme effacement fichier reception");
      end if;
   end Close;

   ------------------------
   -- Initialiser_Client --
   ------------------------

   function Init_Client (File : Integer;
                                Logger : Log4ada.Loggers.Logger_Class_Access)
                               return Link_Access is
      Link : Client_Fifo_Link_Access;
      File_Postfix_String : constant String := Integer'Image (File);
      use type GNAT.OS_Lib.File_Descriptor;
   begin
      Link := new Client_Fifo_Link_Type;
      Link.Root := File;
      Link.Logger := Logger;
      Link.Fd_Send := GNAT.OS_Lib.Open_Read_Write
        ("/tmp/fifo_reception_" &
         File_Postfix_String (2 .. File_Postfix_String'Last),
         GNAT.OS_Lib.Binary);
      if Link.Fd_Send = GNAT.OS_Lib.Invalid_FD then
         Free (Link);
         raise Invalid_Link;
      end if;
      Link.Fd_Receive := GNAT.OS_Lib.Open_Read_Write
        ("/tmp/fifo_emission_" &
         File_Postfix_String (2 .. File_Postfix_String'Last),
         GNAT.OS_Lib.Binary);
      if Link.Fd_Receive = GNAT.OS_Lib.Invalid_FD then
         Free (Link);
         raise Invalid_Link;
      end if;
      return Link_Access (Link);
   end Init_Client;

   -------------------------
   -- Initialiser_Serveur --
   -------------------------

   function Init_Server
     (Logger : Log4ada.Loggers.Logger_Class_Access)
     return Link_Access
   is
      Link : Server_Fifo_Link_Access;
      Root : Integer;
      Create_File_Problem : exception;
   begin
      Link := new Server_Fifo_Link_Type;
      Link.Logger := Logger;
      loop
         Root := Random_Draw_Thousand.Draw_Package.Random (Random_Generator);
         declare
            File_Postfix_String : constant String := Integer'Image (Root);
            Send_File_Name : constant String := "/tmp/fifo_emission_" &
              File_Postfix_String (2 .. File_Postfix_String'Last);
            Receive_File_Name : constant String := "/tmp/fifo_reception_" &
              File_Postfix_String (2 .. File_Postfix_String'Last);
            Argument : GNAT.OS_Lib.String_Access;
            Succes : Boolean;
         begin
            if not GNAT.OS_Lib.Is_Regular_File (Send_File_Name) and
              not GNAT.OS_Lib.Is_Regular_File (Receive_File_Name) then
               Argument := new String'(Send_File_Name);
               GNAT.OS_Lib.Spawn ("/usr/bin/mkfifo", (1 => Argument), Succes);
               GNAT.OS_Lib.Free (Argument);
               if not Succes then
                  raise Create_File_Problem;
               else
                  System_Calls.Exec_Command ("ls /tmp/fifo*");
               end if;
               Argument := new String'(Receive_File_Name);
               GNAT.OS_Lib.Spawn ("/usr/bin/mkfifo", (1 => Argument), Succes);
               GNAT.OS_Lib.Free (Argument);
               if not Succes then
                  raise Create_File_Problem;
               else
                  System_Calls.Exec_Command ("ls /tmp/fifo*");
               end if;
               Link.Fd_Send := GNAT.OS_Lib.Open_Read_Write
                 (Send_File_Name, GNAT.OS_Lib.Binary);
               Link.Fd_Receive := GNAT.OS_Lib.Open_Read_Write
                 (Receive_File_Name, GNAT.OS_Lib.Binary);
               exit;
            end if;
         exception
            when Create_File_Problem =>
               null;
         end;
      end loop;
      Link.Root := Root;
      return Link_Access (Link);
   end Init_Server;

   -- fonctions annexes read et write --

   procedure Read
     (Stream : in out Fifos_Type;
      Item   : out Ada.Streams.Stream_Element_Array;
      Last   : out Ada.Streams.Stream_Element_Offset) is
      Function_Return : Integer;
   begin
      Function_Return := GNAT.OS_Lib.Read (Stream.Fd_Receive,
                                  Item (Item'First)'Address,
                                  Item'Length);
      Last := Ada.Streams.Stream_Element_Offset (Function_Return);
   end Read;

   procedure Write
     (Stream : in out Fifos_Type;
      Item   : Ada.Streams.Stream_Element_Array) is
      Function_Return : Integer;
      Write_Fifo_Problem : exception;
   begin
      Function_Return := GNAT.OS_Lib.Write (Stream.Fd_Send,
                                   Item (Item'First)'Address,
                                   Item'Length);
      if Function_Return /= Item'Length then
         raise Write_Fifo_Problem;
      end if;
   end Write;

begin
   Random_Draw_Thousand.Draw_Package.Reset (Random_Generator);
end Narval.Communication.Fifos;
