--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Ada.Exceptions;
with System.Storage_Elements;
with GNAT.Sockets;
with Network_Utils;

package body Narval.Communication is

   use Log4ada.Loggers;

   --------------------
   -- Envoyer_Taille --
   --------------------

   procedure Send_Size (Link : access Link_Type;
                        Size : Interfaces.Unsigned_64) is
      Stream_To_Write : Stream_Access;
   begin
      Stream_To_Write := Stream (Link_Access (Link));
      Interfaces.Unsigned_64'Output (Stream_To_Write,
                                     Network_Utils.To_Network (Size));
      Free (Link_Access (Link), Stream_To_Write);
   end Send_Size;

   ---------------------
   -- Recevoir_Taille --
   ---------------------

   function Receive_Size (Link : access Link_Type)
                         return Interfaces.Unsigned_64 is
      Value_To_Return : Interfaces.Unsigned_64;
      Stream_To_Read : Stream_Access;
   begin
      Stream_To_Read := Stream (Link_Access (Link));
      Value_To_Return := Network_Utils.From_Network
        (Interfaces.Unsigned_64'Input (Stream_To_Read));
      Free (Link_Access (Link), Stream_To_Read);
      return Value_To_Return;
   end Receive_Size;

   ------------------
   -- Tache_Entree --
   ------------------

   task body Input_Task_Type is
      use Ada.Exceptions;
      Requested_Size_To_Hight : exception;
   begin
      loop
         select
            accept Start;
         or
            terminate;
         end select;

         --  NR: Bytes In Count Initialization
         Link.Bytes_In_Count := 0;

         loop
            declare
               use Narval.Protected_Memory;
               use type Ada.Streams.Stream_Element_Offset;
               Used_Memory_Size : Ada.Streams.Stream_Element_Offset;
               Available_Memory_Size : System.Storage_Elements.Storage_Offset;
               Address : System.Address;
            begin
               Used_Memory_Size :=
                 Ada.Streams.Stream_Element_Offset (Receive_Size (Link));
               if Used_Memory_Size /= 0 then
                  Memory.Get_Memory (Address, Available_Memory_Size);
                  if Ada.Streams.Stream_Element_Offset (Available_Memory_Size)
                    < Used_Memory_Size then
                     Ada.Exceptions.Raise_Exception
                       (Requested_Size_To_Hight'Identity,
                        "Available_Size=" & Available_Memory_Size'Img &
                          "< Used_Memory_Size=" & Used_Memory_Size'Img);
                  end if;
                  declare
                     Data : Ada.Streams.Stream_Element_Array
                       (1 .. Used_Memory_Size);
                     for Data'Address use Address;
                  begin
                     Receive (Link, Data);
                     --  NR: Bytes In Count
                     Link.Bytes_In_Count :=
                       Link.Bytes_In_Count +
                       Bytes_Count_Type (Used_Memory_Size);
                  end;
                  Memory.Release_Memory
                    (System.Storage_Elements.Storage_Count
                     (Used_Memory_Size));
               else
                  accept Stop;
                  exit;
               end if;
            end;
         end loop;
      end loop;
   exception
      when E : others =>
         Fatal_Out (Logger, "Tache_Entree", E);
   end Input_Task_Type;

   ------------------
   -- Tache_Sortie --
   ------------------

   task body Output_Task_Type is
      use Ada.Exceptions;
   begin
      loop
         select
            accept Start;
         or
            terminate;
         end select;
         --  NR: Bytes Out Count Initialization
         for I in Links'Range loop
            Links (I).Bytes_Out_Count := 0;
         end loop;

         loop
            declare
               use Narval.Protected_Memory;
               use Ada.Streams;
               Manipulation_Buffer : Buffer_Enumeration_Type;
            begin
               select
                  Memory.Lock_Buffer (Manipulation_Buffer);
                  declare
                     Metadata_Raw_Access : constant Raw_Access_Type :=
                       Memory.Get_Buffer_Structure (Manipulation_Buffer);
                     Size_To_Send : constant Stream_Element_Offset :=
                       Stream_Element_Offset (Metadata_Raw_Access.Size);
                     Data : Ada.Streams.Stream_Element_Array
                       (1 .. Size_To_Send);
                     for Data'Address use Metadata_Raw_Access.Address;
                  begin
                     for I in Links'Range loop
                        if not Links (I).Bypass then
                           begin
                              Send_Size
                                (Links (I),
                                 Interfaces.Unsigned_64 (Size_To_Send));
                              Send (Links (I), Data);

                              --  NR: Bytes Out Count
                              Links (I).Bytes_Out_Count :=
                                Links (I).Bytes_Out_Count +
                                Bytes_Count_Type (Size_To_Send);
                           exception
                              when GNAT.Sockets.Socket_Error =>
                                 Error_Out
                                   (Logger,
                                    "link emission error, bypass set to true");
                                 Links (I).Bypass := True;
                           end;
                        end if;
                     end loop;
                  exception
                     when E : others =>
                        Fatal_Out (Logger, "emission error ", E);
                        raise;
                  end;
                  Memory.Unlock_Buffer (Manipulation_Buffer);
               or
                  delay Default_Waiting_Duration;
                  Memory.Check;
                  if Memory.Is_Empty then
                     select
                        Start_Stop.Is_Stopped;
                        select
                           accept Stop;
                           for I in Links'Range loop
                              if not Links (I).Bypass then
                                 Send_Size (Links (I), 0);
                              end if;
                           end loop;
                           exit;
                        else
                           null;
                        end select;
                     or
                        delay 0.0;
                     end select;
                  end if;
               end select;
            end;
         end loop;
      end loop;
   exception
      when E : others =>
         Fatal_Out (Logger, "Tache_Sortie", E);
   end Output_Task_Type;

   procedure Set_Asker (Link : access Link_Type;
                              Asker : String) is
   begin
      Link.Asker := Ada.Strings.Unbounded.To_Unbounded_String (Asker);
   end Set_Asker;

   function Asker_Link (Link : access Link_Type) return String is
   begin
      return Ada.Strings.Unbounded.To_String (Link.Asker);
   end Asker_Link;

   function Provider_Link (Link : access Link_Type) return String is
   begin
      return Ada.Strings.Unbounded.To_String (Link.Provider);
   end Provider_Link;

end Narval.Communication;
