--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Ada.Characters.Handling;
with System.Storage_Elements;
with GNAT.Traceback.Symbolic;

package body Narval.Actors.Actives.Filters is

   use Log4ada.Loggers;
   use Ada.Strings.Unbounded;

   procedure Dummy (Algo_Data : System.Address;
                    Error_Code : out Error_Code_Type);
   pragma Convention (C, Dummy);
   procedure Dummy (Algo_Data : System.Address;
                    Error_Code : out Error_Code_Type) is
      pragma Unreferenced (Algo_Data);
   begin
      Error_Code := 0;
   end Dummy;

   procedure Test_Error_Code (Object : access Generic_Filter_Type;
                              Error_Code : Error_Code_Type;
                              Procedure_Name : String);

   procedure Set (Object : access Generic_Filter_Type;
                  Parameter : String;
                  Value : String) is
      Parameter_Low_Case : constant String :=
        Ada.Characters.Handling.To_Lower (Parameter);
      Error_Code : aliased Error_Code_Type;
      Library_Problem : exception;
      Max_Size : Integer_Access;
      Algo_Path : Unbounded_String;
      Dl_Close_Return : Integer;
   begin
      Actives.Set (Active_Actor_Access (Object),
                   Parameter, Value);
      if Parameter_Low_Case = "library" then
         if Value /= "clear" then
            Object.Library_Reference :=
              Shared_Library.Open_Library
              (Value, Shared_Library.RTLD_NOW);
            Object.Library_Is_Loaded := True;
            Info_Out (Object.Logger'Access,
                      "library loaded");
            Object.Config := Library_Symbol
              (Object.Library_Reference,
               "process_config");
            if Object.Config = null then
               Fatal_Out (Object.Logger'Access,
                          "process_config undefined");
               raise Library_Problem;
            end if;
            Algo_Path := To_Unbounded_String
              (Actives.Get_Image (Active_Actor_Access (Object),
                                  "algo_path"));
            if Algo_Path = Null_Unbounded_String then
               Warn_Out (Object.Logger'Access,
                         "algo_path not set");
            end if;
            Object.Config
              (To_String (Algo_Path) & ASCII.NUL,
               Error_Code);
            Test_Error_Code (Object, Error_Code, "process_config");
            Object.Block_Producer := Library_Symbol
              (Object.Library_Reference,
               "process_block");
            if Object.Block_Producer = null then
               Fatal_Out (Object.Logger'Access,
                          "process_block undefined");
               raise Library_Problem;
            end if;
            Object.Register := Library_Symbol
              (Object.Library_Reference,
               "process_register");
            if Object.Register = null then
               Fatal_Out (Object.Logger'Access,
                          "process_register undefined");
               raise Library_Problem;
            else
               Object.Common_Reference :=
                 Object.Register (Error_Code'Unchecked_Access);
               Test_Error_Code (Object, Error_Code, "process_register");
            end if;
            Max_Size := Library_Symbol
              (Object.Library_Reference, "max_size");
            if Max_Size = null then
               Info_Out (Object.Logger'Access,
                         "max event size set to 0");
            else
               Object.Max_Size := Max_Size.all;
            end if;
            Object.Initialise := Library_Symbol
              (Object.Library_Reference,
               "process_initialise");
            if Object.Initialise = null then
               Object.Initialise := Dummy'Access;
            end if;
            Object.Reset := Library_Symbol
              (Object.Library_Reference, "process_reset");
            if Object.Reset = null then
               Object.Reset := Dummy'Access;
            end if;
            Object.Start := Library_Symbol
              (Object.Library_Reference, "process_start");
            if Object.Start = null then
               Object.Start := Dummy'Access;
            end if;
            Object.Stop := Library_Symbol
              (Object.Library_Reference, "process_stop");
            if Object.Stop = null then
               Object.Stop := Dummy'Access;
            end if;
            Object.Pause_Ptr := Library_Symbol
              (Object.Library_Reference, "process_pause");
            if Object.Pause_Ptr = null then
               Object.Pause_Ptr := Dummy'Access;
            end if;
            Object.Resume := Library_Symbol
              (Object.Library_Reference, "process_resume");
            if Object.Resume = null then
               Object.Resume := Dummy'Access;
            end if;
            Object.Unload := Library_Symbol
              (Object.Library_Reference, "process_unload");
            if Object.Unload = null then
               Object.Unload := Dummy'Access;
            end if;
         else
            if Object.Library_Is_Loaded then
               Dl_Close_Return := Shared_Library.Close_Library
                 (Object.Library_Reference);
               if Dl_Close_Return /= 0 then
                  Warn_Out (Object.Logger'Access,
                            "close library returned non zero value :" &
                            Dl_Close_Return'Img);
               end if;
               Object.Config := null;
               Object.Register := null;
               Object.Block_Producer := null;
               Object.Initialise := null;
               Object.Reset := null;
               Object.Start := null;
               Object.Stop := null;
               Object.Pause_Ptr := null;
               Object.Resume := null;
               Object.Unload := null;
               Object.Max_Size := 0;
               Object.Library_Is_Loaded := False;
            else
               Warn_Out (Object.Logger'Access, "no library to clear");
            end if;
         end if;
      end if;
   exception
      when E : Shared_Library.Library_Loading_Failed =>
         Error_Out (Object.Logger'Access,
                    "set library :" &
                    Shared_Library.Library_Error,
                    E);
         raise;
   end Set;

   -----------------
   -- Initialiser --
   -----------------

   procedure Initialise
     (Object : access Generic_Filter_Type;
      Actor_Name : String) is
      Parameter : Parameters.Parameter_Access;
      use Parameters;
   begin
      Actives.Initialise (Active_Actor_Access (Object), Actor_Name);
      Parameter := new Parameter_Type'(Container_Kind => String_Type,
                                       Name => To_Unbounded_String
                                         ("algo_path"),
                                       Mode => Read_Write,
                                       Monitor => Request,
                                       Run_Parameter => False,
                                       Editor => None,
                                       String_Value =>
                                         To_Unbounded_String (""));
      Parameter_Vector_Package.Append (Object.Parameters_List,
                                       Parameter);
      Parameter := new Parameter_Type'(Container_Kind => String_Type,
                                       Name => To_Unbounded_String
                                         ("library"),
                                       Mode => Read_Write,
                                       Monitor => Request,
                                       Run_Parameter => False,
                                       Editor => None,
                                       String_Value =>
                                         To_Unbounded_String (""));
      Parameter_Vector_Package.Append (Object.Parameters_List,
                                       Parameter);
      Object.Buffer_Handling_Switchoff := False;
   end Initialise;

   procedure On_Initialise
     (Object : access Generic_Filter_Type) is
      use Shared_Library;
      Name : constant String := To_String (Object.Name);
      No_Library : exception;
      Error_Code : Error_Code_Type;
   begin
      Actives.On_Initialise
        (Active_Actor_Access (Object));
      Object.Waiting_Delay := 0.1;
      if not Object.Library_Is_Loaded then
         Fatal_Out (Object.Logger'Access,
                    "one need to set a library to the " & Name & " actor");
         raise No_Library;
      end if;
      Object.Initialise
        (Object.Common_Reference, Error_Code);
      Test_Error_Code (Object, Error_Code, "process_initialise");
   end On_Initialise;

   ----------------------------
   -- Travail_En_Acquisition --
   ----------------------------

   procedure Buffer_Handling
     (Object : access Generic_Filter_Type) is
      use Protected_Memory;
      Data : Raw_Access_Type;
      Manipulation_Buffer : Buffer_Enumeration_Type;
      Adresse_Sortie : System.Address;
      Available_Memory_Size : System.Storage_Elements.Storage_Count;
      use type System.Storage_Elements.Storage_Count;
      Error_Code : Error_Code_Type := 0;
      Used_Size_Of_Output_Buffer : Interfaces.Unsigned_32 := 0;
   begin
      if Object.Buffer_Handling_Switchoff then
         delay 0.01;
         return;
      end if;
      select
         Object.Inputs (1).Memory.Lock_Buffer
           (Manipulation_Buffer);
         Data := Object.Inputs
           (1).Memory.Get_Buffer_Structure (Manipulation_Buffer);
         loop
            select
               Object.Outputs (1).Memory.Get_Memory
                 (Adresse_Sortie, Available_Memory_Size);
               exit;
            or
               delay 1.0;
               Info_Out (Object.Logger'Access,
                         "generic filter : waiting for free memory");
            end select;
         end loop;
         if Object.Max_Size /= 0 then
            loop
               if Available_Memory_Size < System.Storage_Elements.Storage_Count
                 (Object.Max_Size) then
                  Object.Outputs (1).Memory.Release_Memory (0);
                  Object.Outputs (1).Memory.Check;
                  loop
                     select
                        Object.Outputs (1).Memory.Get_Memory
                          (Adresse_Sortie, Available_Memory_Size);
                        exit;
                     or
                        delay 1.0;
                        Info_Out
                          (Object.Logger'Access,
                           "generic filter : waiting for free memory");
                     end select;
                  end loop;
               else
                  exit;
               end if;
            end loop;
         end if;
         Object.Block_Producer
           (Object.Common_Reference,
            Data.Address,
            Interfaces.Unsigned_32 (Data.Size),
            Adresse_Sortie,
            Interfaces.Unsigned_32 (Available_Memory_Size),
            Used_Size_Of_Output_Buffer,
            Error_Code);
         Test_Error_Code (Object, Error_Code, "process_block");
         if Error_Code >= 100 then
            Object.Buffer_Handling_Switchoff := True;
         end if;
         Object.Outputs (1).Memory.Release_Memory
           (System.Storage_Elements.Storage_Count (Used_Size_Of_Output_Buffer),
            Object.Duplicate);
         Object.Inputs (1).Memory.Unlock_Buffer
           (Manipulation_Buffer);
      or
         delay Object.Waiting_Delay;
      end select;
   exception
      when E : others =>
         Put_Sub_System_In_Error (Object, "buffer_handling");
         Error_Out (Object.Logger'Access,
                    "process_block generic filter", E);
         Error_Out (Object.Logger'Access, "trace " &
                    GNAT.Traceback.Symbolic.Symbolic_Traceback (E));
         Object.Buffer_Handling_Switchoff := True;
         delay 0.1;
   end Buffer_Handling;

   procedure On_Unload
     (Object : access Generic_Filter_Type) is
      use Shared_Library;
      Function_Return : Integer;
      Error_Code : Error_Code_Type;
   begin
      Actives.On_Unload
        (Active_Actor_Access (Object));
      if Object.Unload /= null then
         Object.Unload
           (Object.Common_Reference,
            Error_Code);
         Test_Error_Code (Object, Error_Code, "process_unload");
      end if;
      Function_Return := Close_Library
        (Object.Library_Reference);
      if Function_Return /= 0 then
         Warn_Out (Object.Logger'Access,
                   "close library returned a non value :" &
                     Function_Return'Img);
      end if;
   exception
      when E : others =>
         Put_Sub_System_In_Error (Object, "on_unload");
         Fatal_Out (Object.Logger'Access,
                    "on_unload", E);
         raise;
   end On_Unload;

   procedure On_Reset_Com
     (Object : access Generic_Filter_Type) is
      Error_Code : Error_Code_Type := 0;
   begin
      Actives.On_Reset_Com
        (Active_Actor_Access (Object));
      Object.Reset
        (Object.Common_Reference,
         Error_Code);
      Test_Error_Code (Object, Error_Code, "process_reset");
   exception
      when E : others =>
         Put_Sub_System_In_Error (Object, "on_reset");
         Fatal_Out (Object.Logger'Access,
                    "On_Reset_Com", E);
         raise;
   end On_Reset_Com;

   procedure On_Start
     (Object : access Generic_Filter_Type) is
      Error_Code : Error_Code_Type;
   begin
      Actives.On_Start
        (Active_Actor_Access (Object));
      Object.Start
        (Object.Common_Reference, Error_Code);
      Test_Error_Code (Object, Error_Code, "process_start");
   exception
      when E : others =>
         Put_Sub_System_In_Error (Object, "on_start");
         Fatal_Out (Object.Logger'Access,
                    "On_Start", E);
         raise;
   end On_Start;

   procedure On_Stop
     (Object : access Generic_Filter_Type) is
      Error_Code : Error_Code_Type;
   begin
      Actives.On_Stop
        (Active_Actor_Access (Object));
      Object.Stop
        (Object.Common_Reference, Error_Code);
      Test_Error_Code (Object, Error_Code, "process_stop");
   exception
      when E : others =>
         Put_Sub_System_In_Error (Object, "on_stop");
         Fatal_Out (Object.Logger'Access,
                    "On_Stop", E);
         raise;
   end On_Stop;

   procedure On_Suspend
     (Object : access Generic_Filter_Type) is
      Error_Code : Error_Code_Type;
   begin
      Actives.On_Suspend
        (Active_Actor_Access (Object));
      Object.Pause_Ptr
        (Object.Common_Reference, Error_Code);
      Test_Error_Code (Object, Error_Code, "process_suspend");
   exception
      when E : others =>
         Put_Sub_System_In_Error (Object, "on_suspend");
         Fatal_Out (Object.Logger'Access,
                    "On_Suspend", E);
         raise;
   end On_Suspend;

   procedure On_Resume
     (Object : access Generic_Filter_Type) is
      Error_Code : Error_Code_Type;
   begin
      Actives.On_Resume
        (Active_Actor_Access (Object));
      Object.resume
        (Object.Common_Reference, Error_Code);
      Test_Error_Code (Object, Error_Code, "process_resume");
   exception
      when E : others =>
         Put_Sub_System_In_Error (Object, "on_resume");
         Fatal_Out (Object.Logger'Access,
                    "On_Resume", E);
         raise;
   end On_Resume;

   procedure Test_Error_Code (Object : access Generic_Filter_Type;
                              Error_Code : Error_Code_Type;
                              Procedure_Name : String) is
   begin
      if Error_Code in 1 .. 99 then
         Warn_Out (Object.Logger'Access,
                   "warning in " & Procedure_Name &
                   " call :" & Error_Code'Img);
      elsif Error_Code >= 100 then
         Put_Sub_System_In_Error
           (Object,
            "error in " & Procedure_Name & " call :" & Error_Code'Img);
         Error_Out (Object.Logger'Access,
                    "error in " & Procedure_Name & " call :" & Error_Code'Img);
      end if;
   end Test_Error_Code;
end Narval.Actors.Actives.Filters;
