/*
 *   mwmspcfc.c -- Mwave Modem AT Command Parser
 *
 *  Written By: Paul Schroeder IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
#include <mwmspcfc.h>
#include <mwmcntnd.h>
#include <mwmcapi.h>
#include <time.h>
#include <wtt.h>
#include <mwqservr.h>
#include <mwblapi.h>

#include <errno.h>
#include <pthread.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <port_types.h>
/*#include <pbmplus.h>*/

#include "mww32.h"
extern HIPC hTemphipc;

PMWM_DSPINFO pGlobalDspInfo = 0;
static char szThisFile[] = "MWMSPCFC.C";

/*****************************************************************************/
/* Global variables used by LOAD functions.                                  */
/* Moved out of DSP_INFO to allow restarting modem.                          */
/*****************************************************************************/
USHORT      usDebugWindow = 0;
USHORT      usShowTimeStamp = 0;
PMWM_INSTANCE_INFO pInstanceInfo;
PVOID       INIFile;

 HANDLE      hinstDiscrim;    // MTS 2416
 HANDLE      hinstQsize;

HMEIO       hMEIO;
PFNWRITEDSPSNAPSHOT pfnWriteDSPSnapshot;
ULONG mwmSpcfcUnloadDSPCode(PMWM_DSPINFO pmwmDspInfo);
void  mwmHandleErrorCommon( PMWM_INSTANCE_INFO pInstanceInfo,
                            char *achFile, SHORT sLine,
                            ULONG ulPrimaryCode, ULONG ulSecondaryCode,
                            USHORT usSubcode );

/* Handles to the Servers */
 HMCONV      hMwContend;
 HMWAVE      hMwave;
 HMCONV      hMwNotify;
 HMCONV      hMwWorldTrade;
 HMCONV      hMWQueryClient = 0;



/*****************************************************************************/
/* Function:  mwmHandleError                                                 */
/*****************************************************************************/
/* Puts up an Error Message box                                              */
/* This function handles Mwave (and Mwave Modem) system errors.  Typically,  */
/* these errors should not occur.                                            */
/* Debug information is included in the error message.  The help for this    */
/* message requests the user to copy this information, then call Mwave       */
/* service.  With this information we SHOULD be able to figure out the       */
/* problem.                                                                  */
/*                                                                           */
/* Parameters passed:                                                        */
/*    pInstanceInfo    - Instance information.                               */
/*                         -Window Handle                                    */
/*                         -Anchor Block Handle                              */
/*                         -Instance Handle                                  */
/*                                                                           */
/*    achFile          - Source File name of calling location.               */
/*                                                                           */
/*    sLine            - Source line number calling location.                */
/*                                                                           */
/*    ulPrimaryCode    - Primary return code                                 */
/*                                                                           */
/*    ulSecondaryCode  - Secondary return code                               */
/*                                                                           */
/*                                                                           */
/*****************************************************************************/
void MWM_ENTRY mwmHandleError( PMWM_INSTANCE_INFO pInstanceInfo, char *achFile,
                               SHORT sLine, ULONG ulPrimaryCode, ULONG ulSecondaryCode )
{

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmHandleError entry\n");  }


  mwmHandleErrorCommon( pInstanceInfo, achFile, sLine, ulPrimaryCode, ulSecondaryCode, 0 );

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmHandleError exit\n");  }


}

void MWM_ENTRY mwmHandleErrorWithSubcode( PMWM_INSTANCE_INFO pInstanceInfo,
                                          char *achFile, SHORT sLine,
                                          ULONG ulPrimaryCode,
                                          ULONG ulSecondaryCode,
                                          USHORT usSubcode )
{

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmHandleErrorWithSubcode entry\n");  }


  mwmHandleErrorCommon( pInstanceInfo, achFile, sLine, ulPrimaryCode,
                        ulSecondaryCode, usSubcode );

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmHandleErrorWithSubcode exit\n");  }


}

void mwmHandleErrorCommon( PMWM_INSTANCE_INFO pInstanceInfo, char *achFile,
                           SHORT sLine, ULONG ulPrimaryCode, ULONG ulSecondaryCode,
                           USHORT usSubcode)
{
  unsigned short usStringConstant=0;
  
  { MW_SYSLOG_ERROR(LOG_ERR,"mwmspcfc::mwmHandleErrorCommon::achFile=%s, line=%d, ulPrimaryCode=%lx, ulSecondaryCode=%lx, usSubcode=%u\n", achFile, sLine, ulPrimaryCode, ulSecondaryCode, usSubcode); 
  }
  switch (ulPrimaryCode) {
  case MWM_USER_MESSAGE:
    usStringConstant = (USHORT)ulSecondaryCode;
    break;
  case MWM_DSP_ERROR:
    switch (ulSecondaryCode) {
    case DSP_INSUFF_DATA_MEM:
      MW_SYSLOG_ERROR(LOG_ERR,"Mwave - DSP ERROR, Insufficient Data Memory on DSP\n");
      break;
    case DSP_INSUFF_INST_MEM:
      MW_SYSLOG_ERROR(LOG_ERR,"Mwave - DSP ERROR, Insufficient Instruction Memory on DSP\n");
      break;
    case DSP_INSUFF_CPS:
      MW_SYSLOG_ERROR(LOG_ERR,"Mwave - DSP ERROR, Insufficient CPS on DSP\n");
      break;
    case DSP_INS_IPC_CHANNELS:
      usStringConstant = MWM_STR_MANAGER_OVERCOMMIT;
      MW_SYSLOG_ERROR(LOG_ERR,"Mwave - DSP ERROR, Insufficient IPC Channels on DSP\n");
      break;
    default:
      usStringConstant = MWM_STR_INTERNALERROR;
      MW_SYSLOG_ERROR(LOG_ERR,"Mwave - DSP ERROR, Internal Error on DSP\n");
      break;
    }
    break;
  case MWM_MEIO_ERROR:
    switch (ulSecondaryCode)
      {
      case 0x120:  // MEIO_NC_INVALID_NETWORKINTF  ... this constant is not
	// defined in some meio headers...I'll replace the 0x120
	// with the real constant 'someday'
	usStringConstant = MWM_STR_INVALID_COUNTRY;
	MW_SYSLOG_ERROR(LOG_ERR,"Mwave - MEIO ERROR, Invalid Country\n");
	break;
      default:
	MW_SYSLOG_ERROR(LOG_ERR,"Mwave - MEIO INTERNAL ERROR\n");
	usStringConstant = MWM_STR_INTERNALERROR;
	break;
      }
    break;
  case MWM_DEBUG_MESSAGE:
    usStringConstant = 0;  // Don't load a string
    break;
  default:
    usStringConstant = MWM_STR_INTERNALERROR;
    MW_SYSLOG_ERROR(LOG_ERR,"Mwave - INTERNAL ERROR\n");
    break;
  }
}


ULONG mwmSpcfcAddToVersionList(PMWM_DSPINFO pmwmDspInfo, char *pszFileName,
                               HMOD hMod, char *pszIdentifier, PVOID INIFile)
{
   char      achInstalledVersion[MWM_VERSION_STRING_LENGTH] ;
   char      achLoadedVersion[MWM_VERSION_STRING_LENGTH] ;
   RMODINFO *pModInfoBuffer = NULL;
   USHORT    usSize = 0;
   ULONG     ulRC;
   int       iSuccess;


   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcAddToVersionList entry\n");  }


  memset(achInstalledVersion,0,MWM_VERSION_STRING_LENGTH);
  memset(achLoadedVersion,0,MWM_VERSION_STRING_LENGTH);

  /***************************************************************************/
  /* Query Module Info to get the actual version of this module              */
  /***************************************************************************/
  usSize=0;
  ulRC=dspQueryModuleInfo(hMod,&usSize,pModInfoBuffer);
  if (ulRC != DSP_INS_BUFFER)
  {
     mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                    MWM_DSP_ERROR,
                    ulRC );
     return ulRC;
  }

  /***************************************************************************/
  /* MTS 3525                                                                */
  /* Allocate more than needed...in case someone else is changing the system */
  /* before we get a chance to complete the query.                           */
  /***************************************************************************/
  usSize += 20;


  pModInfoBuffer = calloc( 1, usSize );
  if (!pModInfoBuffer)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_OS_ERROR,
                   0 );
    return ulRC;
  }

  ulRC=dspQueryModuleInfo(hMod,&usSize,pModInfoBuffer);
  if (ulRC)
  {
     mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                    MWM_DSP_ERROR,
                    ulRC );
     return ulRC;
  }


  sprintf(achLoadedVersion,"%d.%d",
          pModInfoBuffer->MODINFO_rver.bVersion,
          pModInfoBuffer->MODINFO_rver.bRelease);


  free(pModInfoBuffer);

  /***************************************************************************/
  /* Get the installed Version information for the Main Modem File from      */
  /* the Profile.                                                            */
  /* Then, add it to the I command display list.                             */
  /* I command will display only the modules/versions in its list.           */
  /***************************************************************************/

/*
   iSuccess= GetPrivateProfileString("VERSIONS",pszIdentifier, "none",achInstalledVersion,
                                     MWM_VERSION_STRING_LENGTH,(PSZ)INIFile);
				     */

  iSuccess = 0; /* paulsch: let's just do this for now instead of the above */
   if ((iSuccess == 0)||((strcmp("none",achInstalledVersion))==0))
   {
   
     /************************************************************************/
     /* If the Correct Version Identifier is NOT in the Versions listed in   */
     /* the INI file, just make the reference version 0.0... this should     */
     /* make the I2 command return "not the installed version" for every     */
     /* DSP file.                                                            */
     /* 11/08/93                                                             */
     /* The user should not see an internal error if version is not listed   */
     /* in INI file.                                                         */
     /************************************************************************/
     strcpy(achInstalledVersion,"0.0\0");
   }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcAddToVersionList exit\n");  }


  return 0;
}





ULONG mwmSpcfcSetAsyncPace(PMWM_DSPINFO pmwmDspInfo, USHORT usLineSpeed)
{
  ULONG ulRC;
  char  achAsyncLabel[10];
  ULONG ulAddressAsyncTable;
  ULONG ulAddress;
  USHORT usAsyncPace = 0;


   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetAsyncPace entry\n");  }


  DPF("mwmSpcfcSetAsyncPace\n");

  if (usLineSpeed <= 2400)
  {
    strcpy(achAsyncLabel,"MAX2400");
  }
  else if (usLineSpeed <= 14400)
  {
    strcpy(achAsyncLabel,"MAX14400");
  }
  else
  {
    if (pmwmDspInfo->bPCMResponseReceived && pmwmDspInfo->usPCMConnection) {
      strcpy( achAsyncLabel, "MAX64K" );
    } else if (!pmwmDspInfo->bPCMResponseReceived && pmwmDspInfo->bPCMConnectionPossible) {
      strcpy( achAsyncLabel, "MAX64K" );
    } else {
      if (pmwmDspInfo->bV34EMResponseReceived) {
        if (pmwmDspInfo->bV34EMConnectionEstablished) {
          strcpy( achAsyncLabel, "MAX33600" );
        } else {
          strcpy( achAsyncLabel, "MAX28800" );
        } /* endif */
      } else {
        if (pmwmDspInfo->bV34EMConnectionPossible) {
          strcpy( achAsyncLabel, "MAX33600" );
        } else {
          strcpy( achAsyncLabel, "MAX28800" );
        } /* endif */
      } /* endif */
    } /* endif */
  }

  ulRC = dspLabelToAddress(pmwmDspInfo->hasyTask,
                           achAsyncLabel,
                           &ulAddressAsyncTable);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

  /***************************************************************************/
  /* Get the pacing value for the selected Datapump                          */
  /***************************************************************************/
  ulRC = dspMemTransfer(pmwmDspInfo->hDSP,
                        ulAddressAsyncTable + MWM_ASYNC_PACING_OFFSET,
                        &usAsyncPace, 1,
                        DSP_MEMXFER_DATA_READ);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }



  /***************************************************************************/
  /* Set ASYNC's Pacing value.                                               */
  /***************************************************************************/
  ulRC = dspLabelToAddress(pmwmDspInfo->hmctlTask, "ASYNPACE",
                           &ulAddress);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

  ulRC = dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                        &usAsyncPace, 1,
                        DSP_MEMXFER_DATA_WRITE);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

  ulRC = dspLabelToAddress(pmwmDspInfo->hasyTask, "PACING", &ulAddress);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

  ulRC = dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                        &usAsyncPace, 1,
                        DSP_MEMXFER_DATA_WRITE);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetAsyncPace exit\n");  }


  return 0;
}





/*****************************************************************************/
/* MTS 2916                                                                  */
/* Change VDP, MNP and V.42 Cycles at LineSpeed Negotiation time.            */
/*****************************************************************************/
ULONG mwmSpcfcModemToLineSpeedCPFAdjust(PMWM_DSPINFO pmwmDspInfo, USHORT usProtocol)
{
//  ULONG ulAddress = 0;
  ULONG ulRC = 0;
  USHORT usMaxLineSpeed = 0;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcModemToLineSpeedCPFAdjust entry\n");  }



  DPF("mwmSpcfcModemToLineSpeedCPFAdjust\n");
  /***************************************************************************/
  /* Find Max Loaded speed.  This must be done to allow for fast rate        */
  /* renegotiate.                                                            */
  /***************************************************************************/
//  switch (pmwmDspInfo->ulFeaturesToLoad & FEATURE_AUTOMODE)
  switch (usProtocol & (MWM_STATUS_V22BIS_CONNECT | MWM_STATUS_V32BIS_CONNECT |
                        MWM_STATUS_V34_CONNECT | MWM_STATUS_PCM_CONNECT) )
  {
    case MWM_STATUS_V22BIS_CONNECT:
      usMaxLineSpeed = 2400;
      break;
//    case FEATURE_V32BIS: /* 14400 */
    case MWM_STATUS_V32BIS_CONNECT:
      usMaxLineSpeed = 14400;
      break;
//    case FEATURE_V32TER: /* 19200 */
    case MWM_STATUS_V34_CONNECT:
      usMaxLineSpeed = 33600;
      break;
    case MWM_STATUS_PCM_CONNECT:
      usMaxLineSpeed = 64000;
      break;
  }

  /***************************************************************************/
  /* Set ASYNC's pacing variable to the Datapump selected.                   */
  /***************************************************************************/
  ulRC = mwmSpcfcSetAsyncPace(pmwmDspInfo, usMaxLineSpeed);
  if (ulRC)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }







  /***************************************************************************/
  /* If V.42 is installed, set the V.42 Cycles appropriately.                */
  /***************************************************************************/
  if (pmwmDspInfo->ulFeaturesToLoad & FEATURE_V42)
  {
    ulRC = mwmSpcfcSetV42MACCycles(pmwmDspInfo, usMaxLineSpeed, MWM_DETECTION_CPF);
    if (ulRC)
      return ulRC;


    ulRC = mwmSpcfcSetV42DLCCycles(pmwmDspInfo, usMaxLineSpeed, MWM_DETECTION_CPF);
    if (ulRC)
      return ulRC;

    ulRC = mwmSpcfcSetV42PKTCycles(pmwmDspInfo, usMaxLineSpeed, MWM_DETECTION_CPF);
    if (ulRC)
      return ulRC;

    ulRC = mwmSpcfcSetV42MACxmitCycles(pmwmDspInfo, usMaxLineSpeed, MWM_DETECTION_CPF);
    if (ulRC)
      return ulRC;

    ulRC = mwmSpcfcSetV42DLCxmitCycles(pmwmDspInfo, usMaxLineSpeed, MWM_DETECTION_CPF);
    if (ulRC)
      return ulRC;


  }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcModemToLineSpeedCPFAdjust exit\n");  }

  return 0;
}


/*****************************************************************************/
/* This entire routine is in place for PTR 1711                              */
/*****************************************************************************/
ULONG mwmSpcfcCarrierDetectToProtocolCPFAdjust(PMWM_DSPINFO pmwmDspInfo, USHORT usProtocol)
{
  ULONG  ulRC;
  USHORT usMaxLineSpeed = 0;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcCarrierDetectToProtocolCPFAdjust entry\n");  }


  switch (usProtocol & (MWM_STATUS_V22BIS_CONNECT | MWM_STATUS_V32BIS_CONNECT |
                        MWM_STATUS_V34_CONNECT | MWM_STATUS_PCM_CONNECT) )
  {
    case MWM_STATUS_V22BIS_CONNECT:    /*  2400 */
      usMaxLineSpeed = 2400;
      break;
    case MWM_STATUS_V32BIS_CONNECT:    /* 14400 */
      usMaxLineSpeed = 14400;
      break;
    case MWM_STATUS_V34_CONNECT:       /* 33600 */
      usMaxLineSpeed = 33600;
      break;
    case MWM_STATUS_PCM_CONNECT:       /* 64000 */
      usMaxLineSpeed = 64000;
      break;
  }

  /***************************************************************************/
  /* Set the V.42 cycles                                                     */
  /***************************************************************************/
  if (usProtocol & MWM_STATUS_V42_CONNECT )
  {
    ulRC = mwmSpcfcSetV42MACCycles(pmwmDspInfo, usMaxLineSpeed, MWM_CONNECTION_CPF);
    if (ulRC)
      return ulRC;

    ulRC = mwmSpcfcSetV42DLCCycles(pmwmDspInfo, usMaxLineSpeed, MWM_CONNECTION_CPF);
    if (ulRC)
      return ulRC;

    ulRC = mwmSpcfcSetV42PKTCycles(pmwmDspInfo, usMaxLineSpeed, MWM_CONNECTION_CPF);
    if (ulRC)
      return ulRC;

    ulRC = mwmSpcfcSetV42MACxmitCycles(pmwmDspInfo, usMaxLineSpeed, MWM_CONNECTION_CPF);
    if (ulRC)
      return ulRC;

    ulRC = mwmSpcfcSetV42DLCxmitCycles(pmwmDspInfo, usMaxLineSpeed, MWM_CONNECTION_CPF);
    if (ulRC)
      return ulRC;

  }

  if (usProtocol & MWM_STATUS_MNP_CONNECT )
  {
    ulRC = mwmSpcfcSetMNPCycles(pmwmDspInfo, usMaxLineSpeed, MWM_CONNECTION_CPF);
    if (ulRC)
      return ulRC;
  }


  ulRC = mwmSpcfcRaiseUARTAsyncCycles(pmwmDspInfo, usProtocol, MWM_MUSTHAVE);
  if (ulRC)
     return ulRC;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcCarrierDetectToProtocolCPFAdjust exit\n");  }


  return 0;
}


/*****************************************************************************/
/* Step up UART and ASYNC cyles as much as possible.                         */
/*****************************************************************************/
ULONG mwmSpcfcRaiseUARTAsyncCycles(PMWM_DSPINFO pmwmDspInfo, USHORT usProtocol, USHORT usMustHave)
{

  ULONG  ulRC;
  USHORT usMaxLineSpeed = 0;
  USHORT usFailed;
  ULONG  ulAddress;
  ULONG  ulNewUARTTasks = 0;


   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcRaiseUARTAsyncCycles entry\n");  }


  switch (usProtocol & (MWM_STATUS_V22BIS_CONNECT | MWM_STATUS_V32BIS_CONNECT |
                        MWM_STATUS_V34_CONNECT | MWM_STATUS_PCM_CONNECT) )
  {
    case MWM_STATUS_V22BIS_CONNECT:    /*  2400 */
      usMaxLineSpeed = 2400;
      break;
    case MWM_STATUS_V32BIS_CONNECT:    /* 14400 */
      usMaxLineSpeed = 14400;
      break;
    case MWM_STATUS_V34_CONNECT:
    case MWM_STATUS_PCM_CONNECT:
      if (pmwmDspInfo->bPCMResponseReceived && pmwmDspInfo->usPCMConnection) {
        usMaxLineSpeed = 64000;
      } else if (!pmwmDspInfo->bPCMResponseReceived && pmwmDspInfo->bPCMConnectionPossible) {
        usMaxLineSpeed = 64000;
      } else {
        usMaxLineSpeed = 33600;
      } /* endif */
      break;
  }

  DPF("Raising UART and Async Cycles");
  /*****************************************************************************/
  /* MTS 3188                                                                  */
  /* Set the UART up to 3 no matter what type of connection.                   */
  /*****************************************************************************/
  if (pmwmDspInfo->usUARTMode & USE_UART)
  {
    ulRC = mwmCntndSetUART(pmwmDspInfo, 3L*11000L/8L, MWM_MUSTHAVE,
                           &usFailed,szThisFile,__LINE__);
    if (usFailed)
      return ulRC;
  }


  /***************************************************************************/
  /* Set the ASYNC Cycles to 1x.                                             */
  /***************************************************************************/
  ulRC = mwmSpcfcSetAsyncCycles(pmwmDspInfo, usMaxLineSpeed, 1, MWM_MUSTHAVE);
  if (ulRC)
  {
    /*******************************************************************/
    /* We MUST get the Cycles up to 1x...so we need to check the rc    */
    /* here.                                                           */
    /*******************************************************************/
    if (usMustHave == MWM_MUSTHAVE) {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC );
    } /* endif */

    return ulRC;
  }



/* paulsch: fixme */
  ulNewUARTTasks = GetPrivateProfileInt("DEBUG","UARTBOOST", 0,
                                        INIFile);
  /* ulNewUARTTasks = 0;  paulsch */
  if (ulNewUARTTasks)
  {
    DPF("INI says to load UARTBOOST");
    /***************************************************************************/
    /* Check to see if the UART believes it needs the "Booster" tasks.         */
    /* They are supposedly not needed on some adapters (ACE)                   */
    /*                                                                         */
    /* Get the address of the UXMIT_EN label.                                  */
    /***************************************************************************/
    ulRC = dspLabelToAddress(pmwmDspInfo->hmtUART, "UXMIT_EN",
                             &ulAddress);
    if (ulRC == DSP_NOERROR)
    {
      ulNewUARTTasks = 0;

      ulRC = dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                            &ulNewUARTTasks, 1,
                            DSP_MEMXFER_DATA_READ);
      if (ulRC!=DSP_NOERROR)
      {
        mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                       MWM_DSP_ERROR, ulRC );
        return ulRC;
      }

      if (ulNewUARTTasks)
      {
        DPF("UART Flag says to load UARTBOOST");
        ulRC=dspConnectITCB(pmwmDspInfo->hmtUART,"UARTITCB",
                            NULL,"U7ITCB");
//        if (ulRC)
//        {
//          if ((ulRC == DSP_INSUFF_CPS) || (ulRC == DSP_INSUFF_INST_MEM))
//            ulRC = 0;
//          else
//            mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
//                           MWM_DSP_ERROR, ulRC );
//          return(ulRC);
//        }

        ulRC=dspConnectITCB(pmwmDspInfo->hmtUART,"UARTITCB",
                            NULL,"U31ITCB");
//        if (ulRC)
//        {
//          if ((ulRC == DSP_INSUFF_CPS) || (ulRC == DSP_INSUFF_INST_MEM))
//            ulRC = 0;
//          else
//            mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
//                           MWM_DSP_ERROR, ulRC );
//          return(ulRC);
//        }

        ulRC=dspConnectITCB(pmwmDspInfo->hmtUART,"UARTITCB",
                            NULL,"UNRTITCB");
//        if (ulRC)
//        {
//          if ((ulRC == DSP_INSUFF_CPS) || (ulRC == DSP_INSUFF_INST_MEM))
//            ulRC = 0;
//          else
//            mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
//                           MWM_DSP_ERROR, ulRC );
//          return(ulRC);
//        }
      }
    }
  }



  if (usProtocol & MWM_STATUS_MNP_CONNECT || usProtocol & MWM_STATUS_V42_CONNECT) {
    if (!((usProtocol & MWM_STATUS_MNP5_CONNECT) ||
          (usProtocol & MWM_STATUS_V42BIS_CONNECT)) ) {

      if (usMaxLineSpeed == 64000) {
        /***********************************************************************/
        /* ASYNC   1 -> 2 (28.8K to 57.6K) bps                                 */
        /***********************************************************************/
        ulRC = mwmSpcfcSetAsyncCycles(pmwmDspInfo, usMaxLineSpeed, 2, MWM_NOTREQD);
        if (ulRC)
        {
          /*********************************************************************/
          /* Insufficient cycles is not a fatal error, so return 0 in that     */
          /* case.                                                             */
          /*********************************************************************/
          if (ulRC == DSP_INSUFF_CPS)
            ulRC = 0;
          else
            mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                           MWM_DSP_ERROR, ulRC );
          return ulRC;
        }

        /***********************************************************************/
        /* UART    3 -> 4 (33k   to 44k) bps                                   */
        /***********************************************************************/
        if (pmwmDspInfo->usUARTMode & USE_UART)
        {
          ulRC = mwmCntndSetUART(pmwmDspInfo, 4L*11000L/8L, MWM_NOTREQD,
                                 &usFailed,szThisFile,__LINE__);
          if (usFailed)
            return ulRC;
        }

        /***********************************************************************/
        /* UART    4 -> 6 (44k   to 66k) bps                                   */
        /***********************************************************************/
        if (pmwmDspInfo->usUARTMode & USE_UART)
        {
          ulRC = mwmCntndSetUART(pmwmDspInfo, 6L*11000L/8L, MWM_NOTREQD,
                                 &usFailed,szThisFile,__LINE__);
          if (usFailed)
            return ulRC;
        }
      } /* endif */
    }
  }

  if (usProtocol & MWM_STATUS_MNP5_CONNECT)
  {
    /***************************************************************************/
    /* Set the ASYNC Cycles to 3x.                                             */
    /***************************************************************************/

    if ( usProtocol & MWM_STATUS_V22BIS_CONNECT )
    {
      ulRC = mwmSpcfcSetAsyncCycles(pmwmDspInfo, usMaxLineSpeed, 3, MWM_NOTREQD);
      if (ulRC)
      {
        /*********************************************************************/
        /* Insufficient cycles is not a fatal error, so return 0 in that     */
        /* case.                                                             */
        /*********************************************************************/
        if (ulRC == DSP_INSUFF_CPS)
          ulRC = 0;
        else
          mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                         MWM_DSP_ERROR, ulRC );
        return ulRC;
      }
    }
    else
    if ( usProtocol & MWM_STATUS_V32BIS_CONNECT )
    {
      ulRC = mwmSpcfcSetAsyncCycles(pmwmDspInfo, usMaxLineSpeed, 3, MWM_NOTREQD);
      if (ulRC)
      {
        /*********************************************************************/
        /* Insufficient cycles is not a fatal error, so return 0 in that     */
        /* case.                                                             */
        /*********************************************************************/
        if (ulRC == DSP_INSUFF_CPS)
          ulRC = 0;
        else
          mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                         MWM_DSP_ERROR, ulRC );
        return ulRC;
      }

      /***********************************************************************/
      /* Step the UART and Async cycles up in small steps....this allows     */
      /* us to get the best thruput possible with the remaining MIPS.        */
      /* Use the following order...                                          */
      /* UART    3 -> 4 (33k   to 44k) bps                                   */
      /* ASYNC   3 -> 4 (44595 to 59460) bps                                 */
      /***********************************************************************/
      /***********************************************************************/
      /* PTR 1711                                                            */
      /* UART    3 -> 4 (33k   to 44k) bps                                   */
      /***********************************************************************/
      if (pmwmDspInfo->usUARTMode & USE_UART)
      {
        ulRC = mwmCntndSetUART(pmwmDspInfo, 4L*11000L/8L, MWM_NOTREQD,
                               &usFailed,szThisFile,__LINE__);
        if (usFailed)
          return ulRC;
      }

      /***********************************************************************/
      /* ASYNC   3 -> 4 (44595 to 59460) bps                                 */
      /***********************************************************************/
      ulRC = mwmSpcfcSetAsyncCycles(pmwmDspInfo, usMaxLineSpeed, 4, MWM_NOTREQD);
      if (ulRC)
      {
        /*********************************************************************/
        /* Insufficient cycles is not a fatal error, so return 0 in that     */
        /* case.                                                             */
        /*********************************************************************/
        if (ulRC == DSP_INSUFF_CPS)
          ulRC = 0;
        else
          mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                         MWM_DSP_ERROR, ulRC );
        return ulRC;
      }
    }
    else
    if ( (usProtocol & MWM_STATUS_V34_CONNECT) || (usProtocol & MWM_STATUS_PCM_CONNECT))
    {
      /***********************************************************************/
      /* ASYNC   1 -> 2 (28.5K to 57.6K) bps                                 */
      /***********************************************************************/
      ulRC = mwmSpcfcSetAsyncCycles(pmwmDspInfo, usMaxLineSpeed, 2, MWM_NOTREQD);
      if (ulRC)
      {
        /*********************************************************************/
        /* Insufficient cycles is not a fatal error, so return 0 in that     */
        /* case.                                                             */
        /*********************************************************************/
        if (ulRC == DSP_INSUFF_CPS)
          ulRC = 0;
        else
          mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                         MWM_DSP_ERROR, ulRC );
        return ulRC;
      }



      /***********************************************************************/
      /* PTR 1711                                                            */
      /* UART    3 -> 4 (33k   to 44k) bps                                   */
      /***********************************************************************/
      if (pmwmDspInfo->usUARTMode & USE_UART)
      {
        ulRC = mwmCntndSetUART(pmwmDspInfo, 4L*11000L/8L, MWM_NOTREQD,
                               &usFailed,szThisFile,__LINE__);
        if (usFailed)
          return ulRC;
      }


//      /***********************************************************************/
//      /* UART    4 -> 5 (44k   to 55k) bps                                   */
//      /***********************************************************************/
//      if (pmwmDspInfo->usUARTMode & USE_UART)
//      {
//        ulRC = mwmCntndSetUART(pmwmDspInfo, 5L*11000L/8L, MWM_NOTREQD,
//                               &usFailed,szThisFile,__LINE__);
//        if (usFailed)
//          return ulRC;
//      }

      /***********************************************************************/
      /* UART    4 -> 6 (44k   to 66k) bps                                   */
      /***********************************************************************/
      if (pmwmDspInfo->usUARTMode & USE_UART)
      {
        ulRC = mwmCntndSetUART(pmwmDspInfo, 6L*11000L/8L, MWM_NOTREQD,
                               &usFailed,szThisFile,__LINE__);
        if (usFailed)
          return ulRC;
      }


      /***********************************************************************/
      /* ASYNC   2 -> 3 (57.6K to 86.4K) bps                                 */
      /***********************************************************************/
      ulRC = mwmSpcfcSetAsyncCycles(pmwmDspInfo, usMaxLineSpeed, 3, MWM_NOTREQD);
      if (ulRC)
      {
        /*********************************************************************/
        /* Insufficient cycles is not a fatal error, so return 0 in that     */
        /* case.                                                             */
        /*********************************************************************/
        if (ulRC == DSP_INSUFF_CPS)
          ulRC = 0;
        else
          mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                         MWM_DSP_ERROR, ulRC );
        return ulRC;
      }

      /***********************************************************************/
      /* UART    6 -> 8 (66k   to 88k) bps                                   */
      /***********************************************************************/
      if (pmwmDspInfo->usUARTMode & USE_UART)
      {
        ulRC = mwmCntndSetUART(pmwmDspInfo, 8L*11000L/8L, MWM_NOTREQD,
                               &usFailed,szThisFile,__LINE__);
        if (usFailed)
          return ulRC;
      }

      /***********************************************************************/
      /* ASYNC   3 -> 4 (86.4K to 115.2K) bps                                */
      /***********************************************************************/
      ulRC = mwmSpcfcSetAsyncCycles(pmwmDspInfo, usMaxLineSpeed, 4, MWM_NOTREQD);
      if (ulRC)
      {
        /*********************************************************************/
        /* Insufficient cycles is not a fatal error, so return 0 in that     */
        /* case.                                                             */
        /*********************************************************************/
        if (ulRC == DSP_INSUFF_CPS)
          ulRC = 0;
        else
          mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                         MWM_DSP_ERROR, ulRC );
        return ulRC;
      }

      if (usMaxLineSpeed == 64000) {
        /***********************************************************************/
        /* ASYNC   4 -> 5 (57.6K to 86.4K) bps                                 */
        /***********************************************************************/
        ulRC = mwmSpcfcSetAsyncCycles(pmwmDspInfo, usMaxLineSpeed, 5, MWM_NOTREQD);
        if (ulRC)
        {
          /*********************************************************************/
          /* Insufficient cycles is not a fatal error, so return 0 in that     */
          /* case.                                                             */
          /*********************************************************************/
          if (ulRC == DSP_INSUFF_CPS)
            ulRC = 0;
          else
            mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                           MWM_DSP_ERROR, ulRC );
          return ulRC;
        }
      } /* endif */

      /***********************************************************************/
      /* UART    8 -> 18 (88k to 198k) bps when ASYNC2X is used (CMVC 1360)  */
      /***********************************************************************/
      if ((pmwmDspInfo->usUARTMode & USE_UART) &&
          (pmwmDspInfo->ulFeaturesToLoad & FEATURE_ASYNC2X))
      {
        ulRC = mwmCntndSetUART(pmwmDspInfo, 18L*11000L/8L, MWM_NOTREQD,
                               &usFailed,szThisFile,__LINE__);
        if (usFailed)
          return ulRC;
      }

    }
  }
  else if (usProtocol & MWM_STATUS_V42BIS_CONNECT)
  {
    if ( usProtocol & MWM_STATUS_V22BIS_CONNECT )
    {
      /***********************************************************************/
      /* Set the ASYNC Cycles to 3x.                                         */
      /***********************************************************************/
      ulRC = mwmSpcfcSetAsyncCycles(pmwmDspInfo, usMaxLineSpeed, 3, MWM_NOTREQD);
      if (ulRC)
      {
        /*********************************************************************/
        /* Insufficient cycles is not a fatal error, so return 0 in that     */
        /* case.                                                             */
        /*********************************************************************/
        if (ulRC == DSP_INSUFF_CPS)
          ulRC = 0;
        else
          mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                         MWM_DSP_ERROR, ulRC );
      }
    }

    /*************************************************************************/
    /* If this is a High Speed Modem, then we will raise CPF even higher...  */
    /*************************************************************************/
    if ( usProtocol & MWM_STATUS_V32BIS_CONNECT )
    {
      /***********************************************************************/
      /* Set the ASYNC Cycles to 3x.                                         */
      /***********************************************************************/
      ulRC = mwmSpcfcSetAsyncCycles(pmwmDspInfo, usMaxLineSpeed, 3, MWM_NOTREQD);
      if (ulRC)
      {
        /*********************************************************************/
        /* Insufficient cycles is not a fatal error, so return 0 in that     */
        /* case.                                                             */
        /*********************************************************************/
        if (ulRC == DSP_INSUFF_CPS)
          ulRC = 0;
        else
          mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                         MWM_DSP_ERROR, ulRC );
      }


      /***********************************************************************/
      /* Step the UART and Async cycles up in small steps....this allows     */
      /* us to get the best thruput possible with the remaining MIPS.        */
      /* Use the following order...                                          */
      /*                                                                     */
      /* ASYNC   1 -> 3 (      to 44595) bps                                 */
      /* UART    3 -> 4 (33k   to 44k) bps                                   */
      /* ASYNC   3 -> 4 (44595 to 59460) bps                                 */
      /* UART    4 -> 5 (44k   to 55k) bps                                   */
      /***********************************************************************/
      /***********************************************************************/
      /* PTR 1711                                                            */
      /* UART    3 -> 4 (33k   to 44k) bps                                   */
      /***********************************************************************/
      if (pmwmDspInfo->usUARTMode & USE_UART)
      {
        ulRC = mwmCntndSetUART(pmwmDspInfo, 4L*11000L/8L, MWM_NOTREQD,
                               &usFailed,szThisFile,__LINE__);
        if (usFailed)
          return ulRC;
      }

      /***********************************************************************/
      /* ASYNC   3 -> 4 (44595 to 59460) bps                                 */
      /***********************************************************************/
      ulRC = mwmSpcfcSetAsyncCycles(pmwmDspInfo, usMaxLineSpeed, 4, MWM_NOTREQD);
      if (ulRC)
      {
        /*********************************************************************/
        /* Insufficient cycles is not a fatal error, so return 0 in that     */
        /* case.                                                             */
        /*********************************************************************/
        if (ulRC == DSP_INSUFF_CPS)
          ulRC = 0;
        else
          mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                         MWM_DSP_ERROR, ulRC );
        return ulRC;
      }


      /***********************************************************************/
      /* PTR 1711                                                            */
      /* UART    4 -> 5 (44k   to 55k) bps                                   */
      /***********************************************************************/
      if (pmwmDspInfo->usUARTMode & USE_UART)
      {
        ulRC = mwmCntndSetUART(pmwmDspInfo, 5L*11000L/8L, MWM_NOTREQD,
                               &usFailed,szThisFile,__LINE__);
        if (usFailed)
          return ulRC;
      }

      /*********************************************************************/
      /* PTR 1711                                                          */
      /* UART    5 -> 6 (55k   to 66k) bps                                 */
      /*********************************************************************/
      if (pmwmDspInfo->usUARTMode & USE_UART)
      {
        ulRC = mwmCntndSetUART(pmwmDspInfo, 6L*11000L/8L, MWM_NOTREQD,
                               &usFailed,szThisFile,__LINE__);
        if (usFailed)
          return ulRC;
      }
    }

    if ( (usProtocol & MWM_STATUS_V34_CONNECT) || (usProtocol & MWM_STATUS_PCM_CONNECT) )
    {
      /***********************************************************************/
      /* ASYNC   1 -> 2 (28.5K to 57.6K) bps                                 */
      /***********************************************************************/
      ulRC = mwmSpcfcSetAsyncCycles(pmwmDspInfo, usMaxLineSpeed, 2, MWM_NOTREQD);
      if (ulRC)
      {
        /*********************************************************************/
        /* Insufficient cycles is not a fatal error, so return 0 in that     */
        /* case.                                                             */
        /*********************************************************************/
        if (ulRC == DSP_INSUFF_CPS)
          ulRC = 0;
        else
          mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                         MWM_DSP_ERROR, ulRC );
        return ulRC;
      }



      /***********************************************************************/
      /* PTR 1711                                                            */
      /* UART    3 -> 4 (33k   to 44k) bps                                   */
      /***********************************************************************/
      if (pmwmDspInfo->usUARTMode & USE_UART)
      {
        ulRC = mwmCntndSetUART(pmwmDspInfo, 4L*11000L/8L, MWM_NOTREQD,
                               &usFailed,szThisFile,__LINE__);
        if (usFailed)
          return ulRC;
      }


//      /***********************************************************************/
//      /* UART    4 -> 5 (44k   to 55k) bps                                   */
//      /***********************************************************************/
//      if (pmwmDspInfo->usUARTMode & USE_UART)
//      {
//        ulRC = mwmCntndSetUART(pmwmDspInfo, 5L*11000L/8L, MWM_NOTREQD,
//                               &usFailed,szThisFile,__LINE__);
//        if (usFailed)
//          return ulRC;
//      }

      /***********************************************************************/
      /* UART    4 -> 6 (44k   to 66k) bps                                   */
      /***********************************************************************/
      if (pmwmDspInfo->usUARTMode & USE_UART)
      {
        ulRC = mwmCntndSetUART(pmwmDspInfo, 6L*11000L/8L, MWM_NOTREQD,
                               &usFailed,szThisFile,__LINE__);
        if (usFailed)
          return ulRC;
      }


      /***********************************************************************/
      /* ASYNC   2 -> 3 (57.6K to 86.4K) bps                                 */
      /***********************************************************************/
      ulRC = mwmSpcfcSetAsyncCycles(pmwmDspInfo, usMaxLineSpeed, 3, MWM_NOTREQD);
      if (ulRC)
      {
        /*********************************************************************/
        /* Insufficient cycles is not a fatal error, so return 0 in that     */
        /* case.                                                             */
        /*********************************************************************/
        if (ulRC == DSP_INSUFF_CPS)
          ulRC = 0;
        else
          mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                         MWM_DSP_ERROR, ulRC );
        return ulRC;
      }


//      /***********************************************************************/
//      /* UART    6 -> 7 (66k   to 77k) bps                                   */
//      /***********************************************************************/
//      if (pmwmDspInfo->usUARTMode & USE_UART)
//      {
//        ulRC = mwmCntndSetUART(pmwmDspInfo, 7L*11000L/8L, MWM_NOTREQD,
//                               &usFailed,szThisFile,__LINE__);
//        if (usFailed)
//          return ulRC;
//      }

      /***********************************************************************/
      /* UART    6 -> 8 (66k   to 88k) bps                                   */
      /***********************************************************************/
      if (pmwmDspInfo->usUARTMode & USE_UART)
      {
        ulRC = mwmCntndSetUART(pmwmDspInfo, 8L*11000L/8L, MWM_NOTREQD,
                               &usFailed,szThisFile,__LINE__);
        if (usFailed)
          return ulRC;
      }


//      if (!(pmwmDspInfo->usUARTMode & USE_UART))
//      {
      /*********************************************************************/
      /* ASYNC   3 -> 4 (86.4K to 115.2K) bps                              */
      /*********************************************************************/
      ulRC = mwmSpcfcSetAsyncCycles(pmwmDspInfo, usMaxLineSpeed, 4, MWM_NOTREQD);
      if (ulRC)
      {
        /*******************************************************************/
        /* Insufficient cycles is not a fatal error, so return 0 in that   */
        /* case.                                                           */
        /*******************************************************************/
        if (ulRC == DSP_INSUFF_CPS)
           ulRC = 0;
        else
          mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                         MWM_DSP_ERROR, ulRC );
        return ulRC;
      }

      if (usMaxLineSpeed == 64000) {

        /*********************************************************************/
        /* ASYNC   4 -> 5 (115.2K to 144.0K) bps                             */
        /*********************************************************************/
        ulRC = mwmSpcfcSetAsyncCycles(pmwmDspInfo, usMaxLineSpeed, 5, MWM_NOTREQD);
        if (ulRC)
        {
          /*******************************************************************/
          /* Insufficient cycles is not a fatal error, so return 0 in that   */
          /* case.                                                           */
          /*******************************************************************/
          if (ulRC == DSP_INSUFF_CPS)
             ulRC = 0;
          else
            mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                           MWM_DSP_ERROR, ulRC );
          return ulRC;
        }
      }

      /***********************************************************************/
      /* UART    8 -> 18 (88k to 198k) bps when ASYNC2X is used (CMVC 1360)  */
      /***********************************************************************/
      if ((pmwmDspInfo->usUARTMode & USE_UART) &&
          (pmwmDspInfo->ulFeaturesToLoad & FEATURE_ASYNC2X))
      {
        ulRC = mwmCntndSetUART(pmwmDspInfo, 18L*11000L/8L, MWM_NOTREQD,
                               &usFailed,szThisFile,__LINE__);
        if (usFailed)
          return ulRC;
      }

    }
  } /* end "if V42bis connection" */
  else
  if (!((usProtocol & MWM_STATUS_MNP_CONNECT) ||
        (usProtocol & MWM_STATUS_V42_CONNECT))   )
  {
    /***************************************************************************/
    /* MTS 3188                                                                */
    /* This is an ASYNC connection, set the ASYNC Cycles to ASYNC Mode.        */
    /***************************************************************************/
    ulRC = mwmSpcfcSetAsyncCycles(pmwmDspInfo, usMaxLineSpeed, 0, MWM_NOTREQD);
    if (ulRC)
    {
      /*************************************************************************/
      /* In this case, it is an error if we are out of DSP space.              */
      /*************************************************************************/
      if (ulRC==DSP_INSUFF_CPS)
        mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                       MWM_DSP_ERROR, ulRC );
      return ulRC;
    }

    if (usMaxLineSpeed == 64000) {
      /***********************************************************************/
      /* UART    3 -> 6 (33k   to 66k) bps                                   */
      /***********************************************************************/
      if (pmwmDspInfo->usUARTMode & USE_UART)
      {
        ulRC = mwmCntndSetUART(pmwmDspInfo, 6L*11000L/8L, MWM_NOTREQD,
                               &usFailed,szThisFile,__LINE__);
        if (usFailed)
          return ulRC;
      }
    } /* endif */
  }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcRaiseUARTAsyncCycles exit\n");  }

  return ulRC;
}

ULONG mwmSpcfcLowerUARTAsyncCycles(PMWM_DSPINFO pmwmDspInfo, USHORT usProtocol)
{
  USHORT usMaxLineSpeed = 0;
  USHORT usFailed;
  ULONG  ulRC;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcLowerUARTAsyncCycles entry\n");  }


  DPF("mwmSpcfcLowerUARTAsyncCycles:  Lowering UART and Async Cycles");

  switch (usProtocol & (MWM_STATUS_V22BIS_CONNECT | MWM_STATUS_V32BIS_CONNECT |
                        MWM_STATUS_V34_CONNECT | MWM_STATUS_PCM_CONNECT) )
  {
    case MWM_STATUS_V22BIS_CONNECT:    /*  2400 */
      usMaxLineSpeed = 2400;
      break;
    case MWM_STATUS_V32BIS_CONNECT:    /* 14400 */
      usMaxLineSpeed = 14400;
      break;
    case MWM_STATUS_V34_CONNECT:       /* 28800 */
      usMaxLineSpeed = 33600;
      break;
    case MWM_STATUS_PCM_CONNECT:
      usMaxLineSpeed = 64000;
      break;
  }

  ulRC = 0;                             /* means we did not lower cycles  */
  if (usMaxLineSpeed != 0)
  {
    DPF("Lowering UART and Async Cycles");

    /*************************************************************************
    ** Adjust the Async Pacing to make sure it is low enough before we
    ** adjust Async's cycles.
    **************************************************************************/
    ulRC = mwmSpcfcSetAsyncPace( pmwmDspInfo, usMaxLineSpeed );
    if (ulRC)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC );
      return ulRC;
    }

    /*************************************************************************/
    /* Set the ASYNC Cycles to 1x.                                           */
    /*************************************************************************/
    ulRC = mwmSpcfcSetAsyncCycles(pmwmDspInfo, usMaxLineSpeed, 1, MWM_NOTREQD);
    if (ulRC)
    {
      if (ulRC == DSP_INSUFF_CPS)
      {
        mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                       MWM_DSP_ERROR, ulRC );
        return ulRC;
      }
    }

    if (pmwmDspInfo->usUARTMode & USE_UART)
    {
      ulRC = mwmCntndSetUART(pmwmDspInfo, 3L*11000L/8L, MWM_MUSTHAVE,
                             &usFailed,szThisFile,__LINE__);
      if (usFailed)
        return ulRC;
    }

    /*************************************************************************/
    /* Disconnect the UARTBOOST ITCBs if they are connected.                 */
    /*************************************************************************/
    dspDisconnectITCB(pmwmDspInfo->hmtUART,"UARTITCB",
                      NULL,"U7ITCB");

    dspDisconnectITCB(pmwmDspInfo->hmtUART,"UARTITCB",
                      NULL,"U31ITCB");

    dspDisconnectITCB(pmwmDspInfo->hmtUART,"UARTITCB",
                      NULL,"UNRTITCB");



    ulRC = 1;   /* means we lowered cycles */
  }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcLowerUARTAsyncCycles exit\n");  }


  return ulRC;
}


ULONG mwmSpcfcSetAsyncCycles(PMWM_DSPINFO pmwmDspInfo, USHORT usLineSpeed,
                             USHORT usCompressionFactor, USHORT usMustHave)
{
  ULONG  ulRC;
  ULONG  ulCurrentAsyncCPF;
  USHORT usOffset=0;
  USHORT usDesiredCPF;
  LONG   lAsyncCPFDelta;
  ULONG  ulAddress;
  char   achAsyncLabels[10];


   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetAsyncCycles entry\n");  }



  ulRC = mwmSpcfcGetTaskCPF(pmwmDspInfo, pmwmDspInfo->hasyTask, &ulCurrentAsyncCPF,szThisFile,__LINE__);
  if (ulRC)
    return ulRC;

  /***************************************************************************/
  /* MTS 3224                                                                */
  /* Allow all line speeds not just discrete 2400, 9600, 14400...            */
  /***************************************************************************/
  if (usLineSpeed <=2400)
  {
    strcpy(achAsyncLabels,"MAX2400");
  }
  else if (usLineSpeed <=9600)
  {
    strcpy(achAsyncLabels,"MAX9600");
  }
  else if (usLineSpeed <=14400)
  {
    strcpy(achAsyncLabels,"MAX14400");
  }
  else
  {
    if (pmwmDspInfo->bPCMResponseReceived && pmwmDspInfo->usPCMConnection) {
      strcpy( achAsyncLabels, "MAX64K" );
    } else if (!pmwmDspInfo->bPCMResponseReceived && pmwmDspInfo->bPCMConnectionPossible) {
      strcpy( achAsyncLabels, "MAX64K" );
    } else {
      if (pmwmDspInfo->bV34EMResponseReceived) {
        if (pmwmDspInfo->bV34EMConnectionEstablished) {
          strcpy( achAsyncLabels, "MAX33600" );
        } else {
          strcpy( achAsyncLabels, "MAX28800" );
        } /* endif */
      } else {
        if (pmwmDspInfo->bV34EMConnectionPossible) {
          strcpy( achAsyncLabels, "MAX33600" );
        } else {
          strcpy( achAsyncLabels, "MAX28800" );
        } /* endif */
      } /* endif */
    } /* endif */
  }


  /***************************************************************************/
  /* If we were passed a linespeed of 0 then we don't go to the Async table. */
  /* Instead, we use the Offline cycles specified by ASYNC                   */
  /***************************************************************************/
  if (usLineSpeed)
  {
    ulRC = dspLabelToAddress(pmwmDspInfo->hasyTask,
                             achAsyncLabels,
                             &ulAddress);
    if (ulRC!=DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC );
      return ulRC;
    }

    switch (usCompressionFactor)
    {
      case 0:
        /*********************************************************************/
        /* Compression Factor of 0 means set cycles to ASYNC MODE.           */
        /*********************************************************************/
        usOffset = MWM_ASYNC_ASYNC_OFFSET;
        break;
      case 1:
        usOffset = MWM_ASYNC_ECC_1X_OFFSET;
        break;
      case 2:
        usOffset = MWM_ASYNC_ECC_2X_OFFSET;
        break;
      case 3:
        usOffset = MWM_ASYNC_ECC_3X_OFFSET;
        break;
      case 4:
        usOffset = MWM_ASYNC_ECC_4X_OFFSET;
        break;
      case 5:
        usOffset = MWM_ASYNC_ECC_5X_OFFSET;
        break;
    }

    ulRC = dspMemTransfer(pmwmDspInfo->hDSP,
                          ulAddress + usOffset,
                          &usDesiredCPF, 1,
                          DSP_MEMXFER_DATA_READ);
    if (ulRC!=DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC );
      return ulRC;
    }
  }
  else
  {
    /*************************************************************************/
    /* Get the Offline cycle count from ASYNC.                               */
    /* We will use this to set the Async Cycles when we are hanging up.      */
    /*************************************************************************/
    ulRC = dspLabelToAddress(pmwmDspInfo->hasyTask,
                             "OFFCYCLE",
                             &ulAddress);
    if (ulRC!=DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC );
      return ulRC;
    }

    ulRC = dspMemTransfer(pmwmDspInfo->hDSP,
                          ulAddress,
                          &usDesiredCPF, 1,
                          DSP_MEMXFER_DATA_READ);
    if (ulRC!=DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC );
      return ulRC;
    }
  }

  lAsyncCPFDelta = (LONG) ((ULONG)usDesiredCPF - ulCurrentAsyncCPF);


  /***************************************************************************/
  /* 04/18/95                                                                */
  /* We have to check to see which way the cycles must be adjusted.          */
  /* if we are raising cycles, then the cycles adjust must come before the   */
  /* notification (Set compression factor).                                  */
  /*                                                                         */
  /***************************************************************************/
  if (lAsyncCPFDelta > 0)
  {
    /*************************************************************************/
    /* Attempt to Change CPF...                                              */
    /*************************************************************************/
//    ulRC=dspChangeCPF(pmwmDspInfo->hasyTask,lAsyncCPFDelta);
    ulRC=mwmCntndChangeCPF(pmwmDspInfo,pmwmDspInfo->hasyTask,lAsyncCPFDelta,
                           usMustHave); /* MTS RADD */
    /*************************************************************************/
    /* If there weren't enough cycles available, return without posting an   */
    /* error message.                                                        */
    /*************************************************************************/
    if (ulRC == DSP_INSUFF_CPS)
      return ulRC;
    if (ulRC!=DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC );
      return ulRC;
    }

    /*************************************************************************/
    /* Since we have made it this far, we haven't had any fatal errors...    */
    /* Write the compression factor to MCTLs copy of the ASYNC CHARPFM variable*/
    /* Compression Factor of 0 means set cycles to ASYNC MODE.               */
    /*************************************************************************/
    if ((usCompressionFactor > 0) && (usLineSpeed > 0))
    {
//      ulRC = dspLabelToAddress(pmwmDspInfo->hmctlTask, "ASYNCMPR", &ulAddress);
//      if (ulRC!=DSP_NOERROR)
//      {
//        mwmHandleError(pmwmDspInfo->pInstanceInfo,szThisFile,__LINE__,
//                       MWM_DSP_ERROR, ulRC );
//        return ulRC;
//      }

      /***********************************************************************/
      /* 04/18/95 Start using Async compression variable directly.           */
      /* rather than allowing an MCTL delay.                                 */
      /***********************************************************************/
      ulRC = dspLabelToAddress(pmwmDspInfo->hasyTask, "COMPRESSION", &ulAddress);
      if (ulRC!=DSP_NOERROR)
      {
        mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                       MWM_DSP_ERROR, ulRC );
        return ulRC;
      }

      ulRC = dspMemTransfer(pmwmDspInfo->hDSP,
                            ulAddress,
                            &usCompressionFactor, 1,
                            DSP_MEMXFER_DATA_WRITE);
      if (ulRC!=DSP_NOERROR)
      {
        mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                       MWM_DSP_ERROR, ulRC );
        return ulRC;
      }
    }
  }
  else
  {
    /*************************************************************************/
    /* We are lowering the Async cycles, so we have to use the reverse order */
    /* Lower the compression factor, then lower the cycles.                  */
    /*************************************************************************/
    /* Write the compression factor to ASYNC's compression factor variable   */
    /* Compression Factor of 0 means set cycles to ASYNC MODE.               */
    /*************************************************************************/
    if ((usCompressionFactor > 0) && (usLineSpeed > 0))
    {
      /***********************************************************************/
      /* 04/18/95 Start using Async compression variable directly.           */
      /* rather than allowing an MCTL delay.                                 */
      /***********************************************************************/
      ulRC = dspLabelToAddress(pmwmDspInfo->hasyTask, "COMPRESSION", &ulAddress);
      if (ulRC!=DSP_NOERROR)
      {
        mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                       MWM_DSP_ERROR, ulRC );
        return ulRC;
      }

      ulRC = dspMemTransfer(pmwmDspInfo->hDSP,
                            ulAddress,
                            &usCompressionFactor, 1,
                            DSP_MEMXFER_DATA_WRITE);
      if (ulRC!=DSP_NOERROR)
      {
        mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                       MWM_DSP_ERROR, ulRC );
        return ulRC;
      }
    }



    /*************************************************************************/
    /* Attempt to Change CPF...                                              */
    /*************************************************************************/
    ulRC=mwmCntndChangeCPF(pmwmDspInfo,pmwmDspInfo->hasyTask,lAsyncCPFDelta,
                           usMustHave); /* MTS RADD */
    /*************************************************************************/
    /* If there weren't enough cycles available, return without posting an   */
    /* error message.                                                        */
    /*************************************************************************/
    if (ulRC == DSP_INSUFF_CPS)
      return ulRC;
    if (ulRC!=DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC );
      return ulRC;
    }

  }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetAsyncCycles exit\n");  }

  return 0;
}


/*****************************************************************************/
/* Step up UART and V80 cycles as much as possible.                        $3*/
/*****************************************************************************/
ULONG mwmSpcfcRaiseUARTV80Cycles(PMWM_DSPINFO pmwmDspInfo, USHORT usProtocol)
{

  ULONG  ulRC;
  USHORT usMaxLineSpeed = 33600;
  USHORT usFailed;
  ULONG  ulAddress;
  ULONG  ulNewUARTTasks = 0;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcRaiseUARTV80Cycles entry\n");  }


  DPF("mwmSpcfcRaiseUARTV80Cycles:  Raising UART and V80 Cycles");

  /*****************************************************************************/
  /* Set the UART to 4x no matter what type of connection.                     */
  /*****************************************************************************/
  if (pmwmDspInfo->usUARTMode & USE_UART)
  {
    ulRC = mwmCntndSetUART(pmwmDspInfo, 4L*11000L/8L, MWM_MUSTHAVE,
                           &usFailed,szThisFile,__LINE__);
    if (usFailed)
      return ulRC;
  }

  /***************************************************************************/
  /* Set the V80 Cycles.                                                     */
  /***************************************************************************/
  ulRC = mwmSpcfcSetV80Cycles(pmwmDspInfo, usMaxLineSpeed, MWM_MUSTHAVE);
  if (ulRC)
  {
    /*******************************************************************/
    /* We MUST get the Cycles...so we need to check the rc here.       */
    /*******************************************************************/
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

  /* paulsch: fixme */
  ulNewUARTTasks = GetPrivateProfileInt("DEBUG","UARTBOOST", 0,
                                        INIFile);
  /* ulNewUARTTasks = 0; paulsch*/
  if (ulNewUARTTasks)
  {
    DPF("  INI says to load UARTBOOST");
    /***************************************************************************/
    /* Check to see if the UART believes it needs the "Booster" tasks.         */
    /* They are supposedly not needed on some adapters (ACE)                   */
    /*                                                                         */
    /* Get the address of the UXMIT_EN label.                                  */
    /***************************************************************************/
    ulRC = dspLabelToAddress(pmwmDspInfo->hmtUART, "UXMIT_EN",
                             &ulAddress);
    if (ulRC == DSP_NOERROR)
    {
      ulNewUARTTasks = 0;

      ulRC = dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                            &ulNewUARTTasks, 1,
                            DSP_MEMXFER_DATA_READ);
      if (ulRC!=DSP_NOERROR)
      {
        mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                       MWM_DSP_ERROR, ulRC );
        return ulRC;
      }

      if (ulNewUARTTasks)
      {
        DPF("UART Flag says to load UARTBOOST");
        ulRC=dspConnectITCB(pmwmDspInfo->hmtUART,"UARTITCB",
                            NULL,"U7ITCB");

        ulRC=dspConnectITCB(pmwmDspInfo->hmtUART,"UARTITCB",
                            NULL,"U31ITCB");

        ulRC=dspConnectITCB(pmwmDspInfo->hmtUART,"UARTITCB",
                            NULL,"UNRTITCB");
      }
    }
  }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcRaiseUARTV80Cycles exit\n");  }


  return ulRC;
}

/*****************************************************************************/
/* Lower UART and V80 cycles.                                              $3*/
/*****************************************************************************/
ULONG mwmSpcfcLowerUARTV80Cycles(PMWM_DSPINFO pmwmDspInfo, USHORT usProtocol)
{
  USHORT usMaxLineSpeed = 33600;
  USHORT usFailed;
  ULONG  ulRC = 1;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcLowerUARTV80Cycles entry\n");  }


  DPF("mwmSpcfcLowerUARTV80Cycles:  Lowering UART and V80 Cycles");

  /*************************************************************************/
  /* Set the V80 Cycles.                                                   */
  /*************************************************************************/
  ulRC = mwmSpcfcSetV80Cycles(pmwmDspInfo, usMaxLineSpeed, MWM_NOTREQD);
  if (ulRC)
  {
    if (ulRC == DSP_INSUFF_CPS)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC );
      return ulRC;
    }
  }

  if (pmwmDspInfo->usUARTMode & USE_UART)
  {
    ulRC = mwmCntndSetUART(pmwmDspInfo, 3L*11000L/8L, MWM_MUSTHAVE,
                           &usFailed,szThisFile,__LINE__);
    if (usFailed)
      return ulRC;
  }

  /*************************************************************************/
  /* Disconnect the UARTBOOST ITCBs if they are connected.                 */
  /*************************************************************************/
  dspDisconnectITCB(pmwmDspInfo->hmtUART,"UARTITCB",
                    NULL,"U7ITCB");

  dspDisconnectITCB(pmwmDspInfo->hmtUART,"UARTITCB",
                    NULL,"U31ITCB");

  dspDisconnectITCB(pmwmDspInfo->hmtUART,"UARTITCB",
                    NULL,"UNRTITCB");

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcLowerUARTV80Cycles exit\n");  }


  return ulRC;
}


/*****************************************************************************/
/* Set the V80 cycles.                                                     $3*/
/*****************************************************************************/
ULONG mwmSpcfcSetV80Cycles(PMWM_DSPINFO pmwmDspInfo, USHORT usLineSpeed,
                           USHORT usMustHave)
{
  ULONG  ulRC;
  ULONG  ulCurrentV80CPF;
  USHORT usDesiredCPF;
  LONG   lV80CPFDelta;
  ULONG  ulAddress;
  char   achV80Labels[10];

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetV80Cycles entry\n");  }


  DPF("mwmSpcfcSetV80Cycles\n");

  strcpy( achV80Labels, "MAX33600" );

  ulRC = mwmSpcfcGetTaskCPF(pmwmDspInfo, pmwmDspInfo->hmtCLASSx, &ulCurrentV80CPF,szThisFile,__LINE__);
  if (ulRC)
    return ulRC;

  /***************************************************************************/
  /* Allow for line speeds of 28800 and 33600.                               */
  /***************************************************************************/
  if (usLineSpeed <=33600)
  {
    if (pmwmDspInfo->bV34EMResponseReceived) {
      if (pmwmDspInfo->bV34EMConnectionEstablished) {
        strcpy( achV80Labels, "MAX33600" );
      } else {
        strcpy( achV80Labels, "MAX28800" );
      } /* endif */
    } else {
      if (pmwmDspInfo->bV34EMConnectionPossible) {
        strcpy( achV80Labels, "MAX33600" );
      } else {
        strcpy( achV80Labels, "MAX28800" );
      } /* endif */
    } /* endif */
  }


  /***************************************************************************/
  /* If we were passed a linespeed then we get the max cycles for the speed  */
  /* specified by V80.                                                       */
  /* Otherwise, we get the offline cycles specified by V80.                  */
  /***************************************************************************/
  if (usLineSpeed != 0)
  {
    ulRC = dspLabelToAddress(pmwmDspInfo->hmtCLASSx,
                             achV80Labels,
                             &ulAddress);
    if (ulRC!=DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC );
      return ulRC;
    }
    ulRC = dspMemTransfer(pmwmDspInfo->hDSP,
                          ulAddress,
                          &usDesiredCPF, 1,
                          DSP_MEMXFER_DATA_READ);
    if (ulRC!=DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC );
      return ulRC;
    }
  }
  else
  {
    /*************************************************************************/
    /* Get the Offline cycle count from V80.                                 */
    /* We will use this to set the V80 Cycles when we are hanging up.        */
    /*************************************************************************/
    ulRC = dspLabelToAddress(pmwmDspInfo->hmtCLASSx,
                             "OFFCYCLE",
                             &ulAddress);
    if (ulRC!=DSP_NOERROR)
    {
      ulRC = DSP_NOERROR;
      usDesiredCPF = 3700;
      DPF("  WARNING: dspLabelToAddress for V80 OFFCYCLE label failed. Default %d used.\n", usDesiredCPF);
//      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
//                     MWM_DSP_ERROR, ulRC );
//      return ulRC;
    }
    else
    {
      ulRC = dspMemTransfer(pmwmDspInfo->hDSP,
                            ulAddress,
                            &usDesiredCPF, 1,
                            DSP_MEMXFER_DATA_READ);
      if (ulRC!=DSP_NOERROR)
      {
        mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                       MWM_DSP_ERROR, ulRC );
        return ulRC;
      }
    }
  }


  lV80CPFDelta = (LONG) ((ULONG)usDesiredCPF - ulCurrentV80CPF);

  if (lV80CPFDelta > 0)
    DPF("  Raising cycles to %d.\n", usDesiredCPF);
  else
    DPF("  Lowering cycles to %d.\n", usDesiredCPF);

  /*************************************************************************/
  /* Attempt to Change CPF...                                              */
  /*************************************************************************/
  ulRC=mwmCntndChangeCPF(pmwmDspInfo,pmwmDspInfo->hmtCLASSx,lV80CPFDelta,
                         usMustHave); /* MTS RADD */
  /*************************************************************************/
  /* If there weren't enough cycles available, return without posting an   */
  /* error message.                                                        */
  /*************************************************************************/
  if (ulRC == DSP_INSUFF_CPS)
    return ulRC;

  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetV80Cycles exit\n");  }


  return 0;

}



ULONG mwmSpcfcSetMNPCycles(PMWM_DSPINFO pmwmDspInfo, USHORT usLineSpeed, USHORT usPhase)
{
  ULONG  ulRC;
  ULONG  ulCurrentCPF;
  ULONG  ulDesiredCPF;
  LONG   lCPFDelta;
  ULONG  ulAddress;
  USHORT usOffset = 0;
  char   achLabel[10];


   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetMNPCycles entry\n");  }


  if (usPhase == MWM_DETECTION_CPF)
    usOffset = 4;
  else
    usOffset = 0;


  ulRC = mwmSpcfcGetTaskCPF(pmwmDspInfo, pmwmDspInfo->hmtMNP, &ulCurrentCPF,szThisFile,__LINE__);
  if (ulRC)
    return ulRC;

  /***************************************************************************/
  /* MTS 3224                                                                */
  /* Allow all line speeds not just discrete 2400, 9600, 14400...            */
  /***************************************************************************/
  if (usLineSpeed <=2400)
  {
    strcpy(achLabel,"MNP2400");
  }
  else if (usLineSpeed <=9600)
  {
    strcpy(achLabel,"MNP9600");
  }
  else if (usLineSpeed <=14400)
  {
    strcpy(achLabel,"MNP14400");
  }
  else if (usLineSpeed <= 64000) {

    if (pmwmDspInfo->bPCMResponseReceived && pmwmDspInfo->usPCMConnection) {
      strcpy( achLabel, "MNP56000" );
    } else if (!pmwmDspInfo->bPCMResponseReceived && pmwmDspInfo->bPCMConnectionPossible) {
      strcpy( achLabel, "MNP56000" );
    } else {
      if (pmwmDspInfo->bV34EMResponseReceived) {
        if (pmwmDspInfo->bV34EMConnectionEstablished) {
          strcpy( achLabel, "MNP33600" );
        } else {
          strcpy( achLabel, "MNP28800" );
        } /* endif */
      } else {
        if (pmwmDspInfo->bV34EMConnectionPossible) {
          strcpy( achLabel, "MNP33600" );
        } else {
          strcpy( achLabel, "MNP28800" );
        } /* endif */
      } /* endif */
    } /* endif */
  }

  ulRC = dspLabelToAddress(pmwmDspInfo->hmtMNP,
                           achLabel,
                           &ulAddress);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

  ulRC = dspMemTransfer(pmwmDspInfo->hDSP,
                        ulAddress + usOffset,
                        &ulDesiredCPF, 2,
                        DSP_MEMXFER_DATA_READ);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

  lCPFDelta = (LONG) (((LONG)ulDesiredCPF) - ((LONG)ulCurrentCPF));

  /***************************************************************************/
  /* Attempt to Change CPF...                                                */
  /***************************************************************************/
//  ulRC=dspChangeCPF(pmwmDspInfo->hmtMNP,lCPFDelta);
  ulRC=mwmCntndChangeCPF(pmwmDspInfo,pmwmDspInfo->hmtMNP,lCPFDelta,
                         MWM_MUSTHAVE); /* MTS RADD */
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }


   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetMNPCycles exit\n");  }


  return 0;
}


ULONG mwmSpcfcSetV42DLCCycles(PMWM_DSPINFO pmwmDspInfo, USHORT usLineSpeed, USHORT usPhase)
{
  ULONG  ulRC;
  ULONG  ulCurrentCPF;
  ULONG  ulAddress;
  USHORT usDesiredCPF;
  LONG   lCPFDelta;
  USHORT usOffset = 0;
  char   achLabel[10];


   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetV42DLCCycles entry\n");  }


  ulRC = mwmSpcfcGetTaskCPF(pmwmDspInfo, pmwmDspInfo->hmtV42LAPDLC, &ulCurrentCPF,szThisFile,__LINE__);
  if (ulRC)
    return ulRC;

  if (usPhase == MWM_DETECTION_CPF)
    usOffset = 2;
  else
    usOffset = 0;

  /***************************************************************************/
  /* MTS 3224                                                                */
  /* Allow all line speeds not just discrete 2400, 9600, 14400...            */
  /***************************************************************************/
  if (usLineSpeed <=2400)
  {
    strcpy(achLabel,"DLC2_4");
  }
  else if (usLineSpeed <=9600)
  {
    strcpy(achLabel,"DLC9_6");
  }
  else if (usLineSpeed <=14400)
  {
    strcpy(achLabel,"DLC14_4");
  }
  else if (usLineSpeed <= 64000) {
    if (pmwmDspInfo->bPCMResponseReceived && pmwmDspInfo->usPCMConnection) {
      strcpy( achLabel, "DLC56_0" );
    } else if (!pmwmDspInfo->bPCMResponseReceived && pmwmDspInfo->bPCMConnectionPossible) {
      strcpy( achLabel, "DLC56_0" );
    } else {
      if (pmwmDspInfo->bV34EMResponseReceived) {
        if (pmwmDspInfo->bV34EMConnectionEstablished) {
          strcpy( achLabel, "DLC33_6" );
        } else {
          strcpy( achLabel, "DLC28_8" );
        } /* endif */
      } else {
        if (pmwmDspInfo->bV34EMConnectionPossible) {
          strcpy( achLabel, "DLC33_6" );
        } else {
          strcpy( achLabel, "DLC28_8" );
        } /* endif */
      } /* endif */
    } /* endif */
  }


  ulRC = dspLabelToAddress(pmwmDspInfo->hmtV42LAPDLC,
                           achLabel,
                           &ulAddress);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

  ulRC = dspMemTransfer(pmwmDspInfo->hDSP,
                        ulAddress + (ULONG)usOffset,
                        &usDesiredCPF, 1,
                        DSP_MEMXFER_DATA_READ);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }


  lCPFDelta = (LONG) (((LONG)usDesiredCPF) - ((LONG)ulCurrentCPF));

  /***************************************************************************/
  /* Attempt to Change CPF...                                                */
  /***************************************************************************/
//  ulRC=dspChangeCPF(pmwmDspInfo->hmtV42LAPDLC,lCPFDelta);
  ulRC=mwmCntndChangeCPF(pmwmDspInfo,pmwmDspInfo->hmtV42LAPDLC,lCPFDelta,
                         MWM_MUSTHAVE); /* MTS RADD */
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetV42DLCCycles exit\n");  }


  return 0;
}


ULONG mwmSpcfcSetV42DLCxmitCycles(PMWM_DSPINFO pmwmDspInfo, USHORT usLineSpeed, USHORT usPhase)
{
  ULONG  ulRC;
  ULONG  ulCurrentCPF;
  ULONG  ulAddress;
  USHORT usDesiredCPF;
  LONG   lCPFDelta;
  USHORT usOffset = 0;
  char   achLabel[10];


   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetV42DLCxmitCycles entry\n");  }


  /***************************************************************************/
  /* This task does not exist in all versions of V42.DSP...                  */
  /***************************************************************************/
  if(pmwmDspInfo->hmtV42LAPDLCxmit == NULL)
     return 0L;

  ulRC = mwmSpcfcGetTaskCPF(pmwmDspInfo, pmwmDspInfo->hmtV42LAPDLCxmit, &ulCurrentCPF,szThisFile,__LINE__);
  if (ulRC)
    return ulRC;

  if (usPhase == MWM_DETECTION_CPF)
    usOffset = 2;
  else
    usOffset = 0;

  /***************************************************************************/
  /* MTS 3224                                                                */
  /* Allow all line speeds not just discrete 2400, 9600, 14400...            */
  /***************************************************************************/
  if (usLineSpeed <=2400)
  {
    strcpy(achLabel,"DLCX2_4");
  }
  else if (usLineSpeed <=9600)
  {
    strcpy(achLabel,"DLCX9_6");
  }
  else if (usLineSpeed <=14400)
  {
    strcpy(achLabel,"DLCX14_4");
  }
  else if (usLineSpeed <= 64000) {
    if (pmwmDspInfo->bPCMResponseReceived && pmwmDspInfo->usPCMConnection) {
      strcpy( achLabel, "DLCX56_0" );
    } else if (!pmwmDspInfo->bPCMResponseReceived && pmwmDspInfo->bPCMConnectionPossible) {
      strcpy( achLabel, "DLCX56_0" );
    } else {
      if (pmwmDspInfo->bV34EMResponseReceived) {
        if (pmwmDspInfo->bV34EMConnectionEstablished) {
          strcpy( achLabel, "DLCX33_6" );
        } else {
          strcpy( achLabel, "DLCX28_8" );
        } /* endif */
      } else {
        if (pmwmDspInfo->bV34EMConnectionPossible) {
          strcpy( achLabel, "DLCX33_6" );
        } else {
          strcpy( achLabel, "DLCX28_8" );
        } /* endif */
      } /* endif */
    } /* endif */
  }


  ulRC = dspLabelToAddress(pmwmDspInfo->hmtV42LAPDLCxmit,
                           achLabel,
                           &ulAddress);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

  ulRC = dspMemTransfer(pmwmDspInfo->hDSP,
                        ulAddress + (ULONG)usOffset,
                        &usDesiredCPF, 1,
                        DSP_MEMXFER_DATA_READ);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }


  lCPFDelta = (LONG) (((LONG)usDesiredCPF) - ((LONG)ulCurrentCPF));

  /***************************************************************************/
  /* Attempt to Change CPF...                                                */
  /***************************************************************************/
  ulRC=mwmCntndChangeCPF(pmwmDspInfo,pmwmDspInfo->hmtV42LAPDLCxmit,lCPFDelta,
                         MWM_MUSTHAVE); /* MTS RADD */
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetV42DLCxmitCycles exit\n");  }


  return 0;
}


ULONG mwmSpcfcSetV42MACCycles(PMWM_DSPINFO pmwmDspInfo, USHORT usLineSpeed, USHORT usPhase)
{
  ULONG  ulRC;
  ULONG  ulCurrentCPF;
  ULONG  ulAddress;
  ULONG  ulDesiredCPF;
  LONG   lCPFDelta;
  USHORT usOffset = 0;
  char   achLabel[10];


   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetV42MACCycles entry\n");  }


  ulRC = mwmSpcfcGetTaskCPF(pmwmDspInfo, pmwmDspInfo->hmtV42MAC, &ulCurrentCPF,szThisFile,__LINE__);
  if (ulRC)
    return ulRC;

  if (usPhase == MWM_DETECTION_CPF)
    usOffset = 4;
  else
    usOffset = 0;


  /***************************************************************************/
  /* MTS 3224                                                                */
  /* Allow all line speeds not just discrete 2400, 9600, 14400...            */
  /***************************************************************************/
  if (usLineSpeed <=2400)
  {
    strcpy(achLabel,"MAC2_4");
  }
  else if (usLineSpeed <=9600)
  {
    strcpy(achLabel,"MAC9_6");
  }
  else if (usLineSpeed <=14400)
  {
    strcpy(achLabel,"MAC14_4");
  }
  else if (usLineSpeed <= 64000) {
    if (pmwmDspInfo->bPCMResponseReceived && pmwmDspInfo->usPCMConnection) {
      strcpy( achLabel, "MAC56_0" );
    } else if (!pmwmDspInfo->bPCMResponseReceived && pmwmDspInfo->bPCMConnectionPossible) {
      strcpy( achLabel, "MAC56_0" );
    } else {
      if (pmwmDspInfo->bV34EMResponseReceived) {
        if (pmwmDspInfo->bV34EMConnectionEstablished) {
          strcpy( achLabel, "MAC33_6" );
        } else {
          strcpy( achLabel, "MAC28_8" );
        } /* endif */
      } else {
        if (pmwmDspInfo->bV34EMConnectionPossible) {
          strcpy( achLabel, "MAC33_6" );
        } else {
          strcpy( achLabel, "MAC28_8" );
        } /* endif */
      } /* endif */
    } /* endif */
  }


  ulRC = dspLabelToAddress(pmwmDspInfo->hmtV42MAC,
                           achLabel,
                           &ulAddress);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

  ulRC = dspMemTransfer(pmwmDspInfo->hDSP,
                        ulAddress + (ULONG)usOffset,
                        &ulDesiredCPF, 2,
                        DSP_MEMXFER_DATA_READ);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

  lCPFDelta = (LONG) (((LONG)ulDesiredCPF) - ((LONG)ulCurrentCPF));

  /***************************************************************************/
  /* Attempt to Change CPF...                                                */
  /***************************************************************************/
//  ulRC=dspChangeCPF(pmwmDspInfo->hmtV42MAC,lCPFDelta);
  ulRC=mwmCntndChangeCPF(pmwmDspInfo,pmwmDspInfo->hmtV42MAC,lCPFDelta,
                         MWM_MUSTHAVE); /* MTS RADD */
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetV42MACCycles exit\n");  }


  return 0;
}



ULONG mwmSpcfcSetV42MACxmitCycles(PMWM_DSPINFO pmwmDspInfo, USHORT usLineSpeed, USHORT usPhase)
{
  ULONG  ulRC;
  ULONG  ulCurrentCPF;
  ULONG  ulAddress;
  ULONG  ulDesiredCPF;
  LONG   lCPFDelta;
  USHORT usOffset = 0;
  char   achLabel[10];


   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetV42MACxmitCycles entry\n");  }


  /***************************************************************************/
  /* This task does not exist in all versions of V42.DSP...                  */
  /***************************************************************************/
  if(pmwmDspInfo->hmtV42MACxmit == NULL)
     return 0L;

  ulRC = mwmSpcfcGetTaskCPF(pmwmDspInfo, pmwmDspInfo->hmtV42MACxmit, &ulCurrentCPF,szThisFile,__LINE__);
  if (ulRC)
    return ulRC;

  if (usPhase == MWM_DETECTION_CPF)
    usOffset = 4;
  else
    usOffset = 0;


  /***************************************************************************/
  /* MTS 3224                                                                */
  /* Allow all line speeds not just discrete 2400, 9600, 14400...            */
  /***************************************************************************/
  if (usLineSpeed <=2400)
  {
    strcpy(achLabel,"MACX2_4");
  }
  else if (usLineSpeed <=9600)
  {
    strcpy(achLabel,"MACX9_6");
  }
  else if (usLineSpeed <=14400)
  {
    strcpy(achLabel,"MACX14_4");
  }
  else if (usLineSpeed <= 64000) {
    if (pmwmDspInfo->bPCMResponseReceived && pmwmDspInfo->usPCMConnection) {
      strcpy( achLabel, "MACX56_0" );
    } else if (!pmwmDspInfo->bPCMResponseReceived && pmwmDspInfo->bPCMConnectionPossible) {
      strcpy( achLabel, "MACX56_0" );
    } else {
      if (pmwmDspInfo->bV34EMResponseReceived) {
        if (pmwmDspInfo->bV34EMConnectionEstablished) {
          strcpy( achLabel, "MACX33_6" );
        } else {
          strcpy( achLabel, "MACX28_8" );
        } /* endif */
      } else {
        if (pmwmDspInfo->bV34EMConnectionPossible) {
          strcpy( achLabel, "MACX33_6" );
        } else {
          strcpy( achLabel, "MACX28_8" );
        } /* endif */
      } /* endif */
    } /* endif */
  }


  ulRC = dspLabelToAddress(pmwmDspInfo->hmtV42MACxmit,
                           achLabel,
                           &ulAddress);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

  ulRC = dspMemTransfer(pmwmDspInfo->hDSP,
                        ulAddress + (ULONG)usOffset,
                        &ulDesiredCPF, 2,
                        DSP_MEMXFER_DATA_READ);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

  lCPFDelta = (LONG) (((LONG)ulDesiredCPF) - ((LONG)ulCurrentCPF));

  /***************************************************************************/
  /* Attempt to Change CPF...                                                */
  /***************************************************************************/
//  ulRC=dspChangeCPF(pmwmDspInfo->hmtV42MAC,lCPFDelta);
  ulRC=mwmCntndChangeCPF(pmwmDspInfo,pmwmDspInfo->hmtV42MACxmit,lCPFDelta,
                         MWM_MUSTHAVE); /* MTS RADD */
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetV42MACxmitCycles exit\n");  }


  return 0;
}



ULONG mwmSpcfcSetV42PKTCycles(PMWM_DSPINFO pmwmDspInfo, USHORT usLineSpeed, USHORT usPhase)
{
  ULONG  ulRC;
  ULONG  ulCurrentCPF;
  ULONG  ulAddress;
  ULONG  ulDesiredCPF;
  LONG   lCPFDelta;
  USHORT usOffset = 0;
  char   achLabel[10];


   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetV42PKTCycles entry\n");  }


  ulRC = mwmSpcfcGetTaskCPF(pmwmDspInfo, pmwmDspInfo->hmtV42PKT, &ulCurrentCPF,szThisFile,__LINE__);
  if (ulRC)
    return ulRC;

  if (usPhase == MWM_DETECTION_CPF)
    usOffset = 4;
  else
    usOffset = 0;


  /***************************************************************************/
  /* MTS 3224                                                                */
  /* Allow all line speeds not just discrete 2400, 9600, 14400...            */
  /***************************************************************************/
  if (usLineSpeed <=2400)
  {
    strcpy(achLabel,"PKT2_4");
  }
  else if (usLineSpeed <=9600)
  {
    strcpy(achLabel,"PKT9_6");
  }
  else if (usLineSpeed <=14400)
  {
    strcpy(achLabel,"PKT14_4");
  }
  else if (usLineSpeed <= 64000) {
    if (pmwmDspInfo->bPCMResponseReceived && pmwmDspInfo->usPCMConnection) {
      strcpy( achLabel, "PKT56_0" );
    } else if (!pmwmDspInfo->bPCMResponseReceived && pmwmDspInfo->bPCMConnectionPossible) {
      strcpy( achLabel, "PKT56_0" );
    } else {
      if (pmwmDspInfo->bV34EMResponseReceived) {
        if (pmwmDspInfo->bV34EMConnectionEstablished) {
          strcpy( achLabel, "PKT33_6" );
        } else {
          strcpy( achLabel, "PKT28_8" );
        } /* endif */
      } else {
        if (pmwmDspInfo->bV34EMConnectionPossible) {
          strcpy( achLabel, "PKT33_6" );
        } else {
          strcpy( achLabel, "PKT28_8" );
        } /* endif */
      } /* endif */
    } /* endif */
  }



  ulRC = dspLabelToAddress(pmwmDspInfo->hmtV42PKT,
                           achLabel,
                           &ulAddress);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

  ulRC = dspMemTransfer(pmwmDspInfo->hDSP,
                        ulAddress + (ULONG)usOffset,
                        &ulDesiredCPF, 2,
                        DSP_MEMXFER_DATA_READ);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }


  lCPFDelta = (LONG) (((LONG)ulDesiredCPF) - ((LONG)ulCurrentCPF));

  /***************************************************************************/
  /* Attempt to Change CPF...                                                */
  /***************************************************************************/
//  ulRC=dspChangeCPF(pmwmDspInfo->hmtV42PKT,lCPFDelta);
  ulRC=mwmCntndChangeCPF(pmwmDspInfo,pmwmDspInfo->hmtV42PKT,lCPFDelta,
                         MWM_MUSTHAVE); /* MTS RADD */
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetV42PKTCycles exit\n");  }


  return 0;
}



/*****************************************************************************/
/*                                                                           */
/* Set Features to load info in parser and modem control                     */
/*                                                                           */
/*****************************************************************************/
ULONG MWM_ENTRY mwmSpcfcSetFeatures(PMWM_DSPINFO pmwmDspInfo, int fTemp)
{
ULONG ulAddress, ulRC;

   { MW_SYSLOG_2(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetFeatures entry ulFeaturesToLoad %lx\n",pmwmDspInfo->ulFeaturesToLoad);  }


  /***************************************************************************/
  /* If not temporarily setting the features, save long term features        */
  /***************************************************************************/
  if (!fTemp)
     pmwmDspInfo->ulLongTermFeatures = pmwmDspInfo->ulFeaturesToLoad;

  /***************************************************************************/
  /* Tell the Parser what features we will be using.                         */
  /***************************************************************************/
  mwmParseSetFeatures(pmwmDspInfo->ulInstalledFeatures, pmwmDspInfo->ulFeaturesToLoad);

  /***************************************************************************/
  /* Tell Modem Control which features are loaded.                           */
  /* If Modem Control is not loaded yet, don't write the features dword.     */
  /***************************************************************************/

  ulRC = dspLabelToAddress(pmwmDspInfo->hmctlTask, "FEATURES", &ulAddress);
  if (ulRC==DSP_NOERROR)
  {
    ulRC = dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                          &pmwmDspInfo->ulFeaturesToLoad, 2,
                          DSP_MEMXFER_DATA_WRITE);
    if (ulRC)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC);
      return ulRC;
    }
  }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetFeatures exit\n");  }


  return 0;
}

/*****************************************************************************/
/* Determine from the SlashN Value which protocols to be loaded.             */
/* MTS 5873                                                                  */
/* 09/13/95                                                                  */
/*****************************************************************************/
ULONG MWM_ENTRY mwmSpcfcSetProtocol(PMWM_DSPINFO pmwmDspInfo,
                                    SHORT sModemCurrSlashNVal, int fTemp)
{
ULONG ulRC=0;


   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetProtocol entry\n");  }
   { MW_SYSLOG_2(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetProtocol sModemCurrSlashNVal=%hd\n", sModemCurrSlashNVal);  }


  switch(sModemCurrSlashNVal)
  {
    case 0:                                                // Case Async only
    case 1:
      if (!(pmwmDspInfo->ulFeaturesToLoad & FEATURE_V42PRELOAD)) {  // If preload not set
             { MW_SYSLOG_2(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetProtocol turning off FEATURE_V42 %lx\n",pmwmDspInfo->ulFeaturesToLoad);  }
            pmwmDspInfo->ulFeaturesToLoad &= ~FEATURE_V42; // Set V42 to 0
      }
      if (!(pmwmDspInfo->ulFeaturesToLoad & FEATURE_MNPPRELOAD))  // If preload not set
            pmwmDspInfo->ulFeaturesToLoad &= ~FEATURE_MNP; // Set MNP to 0
      break;                                               // If either set do nothing
    case 2:                                                // Case MNP,Hangup,
    case 3:                                                //  or MNP,Async,Hangup
      if (!(pmwmDspInfo->ulFeaturesToLoad & FEATURE_V42PRELOAD)) {  // If preload not set
             { MW_SYSLOG_2(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetProtocl, turning off FEATURE_V42 %lx\n",pmwmDspInfo->ulFeaturesToLoad);  }
            pmwmDspInfo->ulFeaturesToLoad &= ~FEATURE_V42; // Set V42 to 0
      }
      break;                                               // If set do nothing
    case 4:                                                // Case V42,Hangup,
    case 5:                                                //  or V42,Async,Hangup
      if (!(pmwmDspInfo->ulFeaturesToLoad & FEATURE_MNPPRELOAD))  // If preload not set
            pmwmDspInfo->ulFeaturesToLoad &= ~FEATURE_MNP; // Set MNP to 0
      break;                                               // If Preload set do nothing
    case 6:                                                // Case V42,MNP,Hangup,
    case 7:                                                //  or V42,MNP,Async,Hangup
      break;                                               // do nothing
    default:
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_USER_MESSAGE,
                     MWM_STR_SPEED_NOT_INSTALLED);         // Handle Error
      ulRC = 1;
      break;
  }

  if (!ulRC)
    mwmSpcfcSetFeatures(pmwmDspInfo, fTemp);   /* tell parser and mctl about change */

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetProtocol exit\n");  }

  return ulRC;
}


/*****************************************************************************/
/* See which Datapump the Command Line wants to use                          */
/*                                                                           */
/* This assumes the Command line has checked that the requested modem is     */
/* actually installed.                                                       */
/*                                                                           */
/*****************************************************************************/
ULONG MWM_ENTRY mwmSpcfcSetSpeed(PMWM_DSPINFO pmwmDspInfo,
                                 SHORT sModemStartSpeed, int fTemp)
{
  ULONG ulRC=0;
  REGISTERS Registers;
  USHORT usFClass, usSubclass;

   { MW_SYSLOG_2(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetSpeed entry ulFeaturesToLoad %lx\n",pmwmDspInfo->ulFeaturesToLoad);  }
   { MW_SYSLOG_2(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetSpeed sModemStartSpeed=%hd\n", sModemStartSpeed);  }


  pmwmDspInfo->ulFeaturesToLoad &= ~(FEATURE_V32        |  /* clear speed */
                                     FEATURE_V32BIS     |
                                     FEATURE_V32BISSTUB |
                                     FEATURE_V32TER     |
                                     FEATURE_V34        |
                                     FEATURE_V34EM      |
                                     FEATURE_PCM        |
                                     FEATURE_V8           );
  pmwmDspInfo->usPCMValue = 0;

  switch(sModemStartSpeed)
  {
    case MW_START2400:
      /***********************************************************************/
      /* Disable VOD because we don't have                                   */
      /* a high enough speed to support it.                                  */
      /***********************************************************************/
      break;
    case MW_START9600:
      /***********************************************************************/
      /* Disable VOD because we don't have                                   */
      /* a high enough speed to support it.                                  */
      /***********************************************************************/
      pmwmDspInfo->ulFeaturesToLoad |= FEATURE_V32;
      break;
    case MW_START14400:
      pmwmDspInfo->ulFeaturesToLoad |= FEATURE_V32BIS;
      break;
    case MW_START19200:
      pmwmDspInfo->ulFeaturesToLoad |= FEATURE_V32TER;
      break;
    case MW_START28800:
      /* V.34 means both V.34 and V.32BIS pumps must be loaded. */
      pmwmDspInfo->ulFeaturesToLoad |= FEATURE_V34;
      pmwmDspInfo->ulFeaturesToLoad |= FEATURE_V32BIS;

      // MTS 7795 - Add Percent TT mode check - 02/17/97 DR
      if ( (pmwmDspInfo->ulInstalledFeatures & FEATURE_V32BISSTUB) &&
           (!pmwmDspInfo->bPercentTTMode) ) {
        pmwmDspInfo->ulFeaturesToLoad |= FEATURE_V32BISSTUB;
      } /* endif */

      /*********************************/
      /* V.34 directly means V.8 Today */
      /*********************************/
      pmwmDspInfo->ulFeaturesToLoad |= FEATURE_V8;
      break;
    case MW_START33600:
      /* V.34 means both V.34 and V.32BIS pumps must be loaded. */
      pmwmDspInfo->ulFeaturesToLoad |= FEATURE_V34;
      pmwmDspInfo->ulFeaturesToLoad |= FEATURE_V32BIS;

      // MTS 7795 - Add Percent TT mode check - 02/17/97 DR
      if ( (pmwmDspInfo->ulInstalledFeatures & FEATURE_V32BISSTUB) &&
           (!pmwmDspInfo->bPercentTTMode) ) {
        pmwmDspInfo->ulFeaturesToLoad |= FEATURE_V32BISSTUB;
      } /* endif */

      /*********************************/
      /* V.34 directly means V.8 Today */
      /*********************************/
      pmwmDspInfo->ulFeaturesToLoad |= FEATURE_V8;
      pmwmDspInfo->ulFeaturesToLoad |= FEATURE_V34EM;
      break;
    case MW_START64000:
      /* V.34 means both V.34 and V.32BIS pumps must be loaded. */
      pmwmDspInfo->ulFeaturesToLoad |= FEATURE_V34;
      pmwmDspInfo->ulFeaturesToLoad |= FEATURE_V32BIS;

      // MTS 7795 - Add Percent TT mode check - 02/17/97 DR
      if ( (pmwmDspInfo->ulInstalledFeatures & FEATURE_V32BISSTUB) &&
           (!pmwmDspInfo->bPercentTTMode) ) {
        pmwmDspInfo->ulFeaturesToLoad |= FEATURE_V32BISSTUB;
      } /* endif */

      /*********************************/
      /* V.34 directly means V.8 Today */
      /*********************************/
      pmwmDspInfo->ulFeaturesToLoad |= FEATURE_V8;
      pmwmDspInfo->ulFeaturesToLoad |= FEATURE_V34EM;

      /*************************************
      ** Enable PCM if we are:
      ** - Not doing a %TT command
      ** - Doing a %TT command and the INI
      **   file says we should use PCM
      ** 08/25/97 - We now also need to
      ** check S52 bit 4. This should be
      ** handled in the data pump, but it
      ** isn't, so we need to do it here.
      ** 11/10/97 - Don't enable the PCM
      ** feature if in V.80 mode.
      **************************************/
      if ( !pmwmDspInfo->bPercentTTMode ||
           (pmwmDspInfo->bPercentTTMode &&
            GetPrivateProfileInt("OPTIONS", "PERCTTPCM", 0,INIFile)) ) {

        if (fTemp) {
          ulRC = mwmParseQueryModemRegisters(&Registers);
          if (ulRC)
            return (ulRC);

          if ((Registers.SV34[3] & 0x0030) != 0x0030) {
            ulRC = mwmFaxGetFCLASS( pmwmDspInfo, &usFClass, &usSubclass );
            if (ulRC)
              return(ulRC);

            // Enable PCM if not in V.80 mode or if in V.80 mode
            // and the INI flag says it's OK to load.
            if ( !(usFClass == 0 && usSubclass == 1) ||
                  (usFClass == 0 && usSubclass == 1 &&
                   GetPrivateProfileInt("OPTIONS", "V80PCM", 0, INIFile) ) ) {
              pmwmDspInfo->ulFeaturesToLoad |= FEATURE_PCM;
              if ((pmwmDspInfo->usLongTermPCMValue & FEATURE_X2) &&
                  !(Registers.SV34[3] & 0x0010)) {
                pmwmDspInfo->usPCMValue |= FEATURE_X2;
              } /* endif */
              if ((pmwmDspInfo->usLongTermPCMValue & FEATURE_V90) &&
                  !(Registers.SV34[3] & 0x0020)) {
                pmwmDspInfo->usPCMValue |= FEATURE_V90;
              } /* endif */
            } /* endif */
          } /* endif */
        } else {
          pmwmDspInfo->ulFeaturesToLoad |= FEATURE_PCM;
          pmwmDspInfo->usPCMValue = pmwmDspInfo->usLongTermPCMValue;
        } /* endif */
      } /* endif */
      break;
    default:
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_USER_MESSAGE,
                     MWM_STR_SPEED_NOT_INSTALLED);
      ulRC = 1;
      break;

  }

  /***************************************************************************/
  /* PRELOADPROBLEM                                                          */
  /* MNP Should be preloaded on NON Automode Modems...without choice.        */
  /* Check the INI File to see what to do for Automode modems.               */
  /***************************************************************************/
  /* MTS 4192 Add V42PRELOAD option.                                         */
  /* V42 must be preloaded in the non-automode case...otherwise, use the flag*/
  /***************************************************************************/
  pmwmDspInfo->ulFeaturesToLoad &= ~(FEATURE_MNPPRELOAD  |  /* clear preload */
                                     FEATURE_V42PRELOAD   );

  if  (!(pmwmDspInfo->ulFeaturesToLoad & FEATURE_AUTOMODE) )
  {
    if (pmwmDspInfo->ulFeaturesToLoad & FEATURE_MNP)
      pmwmDspInfo->ulFeaturesToLoad |= FEATURE_MNPPRELOAD;
    if (pmwmDspInfo->ulFeaturesToLoad & FEATURE_V42)
      pmwmDspInfo->ulFeaturesToLoad |= FEATURE_V42PRELOAD;
  }
  else
  {
    /*************************************************************************/
    /* MTS 5165                                                              */
    /* Don't allow PRELOADs with V.34...                                     */
    /*************************************************************************/
    if  ( !(pmwmDspInfo->ulFeaturesToLoad & FEATURE_V34) &&
          !(pmwmDspInfo->ulFeaturesToLoad & FEATURE_PCM) )
    {
      ULONG ulTypeInstalled;

      ulTypeInstalled = GetPrivateProfileInt("FEATURES","MNPPRELOAD",
                                             0,INIFile );
      if (ulTypeInstalled)
      {
        if (pmwmDspInfo->ulFeaturesToLoad & FEATURE_MNP)
          pmwmDspInfo->ulFeaturesToLoad |= FEATURE_MNPPRELOAD;
      }

      ulTypeInstalled = GetPrivateProfileInt("FEATURES","V42PRELOAD",
                                             0,INIFile );
      if (ulTypeInstalled)
      {
        if (pmwmDspInfo->ulFeaturesToLoad & FEATURE_V42)
          pmwmDspInfo->ulFeaturesToLoad |= FEATURE_V42PRELOAD;
      }
    }
  }

  if (!ulRC)
    mwmSpcfcSetFeatures(pmwmDspInfo, fTemp);   /* tell parser and mctl about change */

   { MW_SYSLOG_2(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetSpeed exit ulFeaturesToLoad %lx\n",pmwmDspInfo->ulFeaturesToLoad);  }

  return ulRC;
}

/*****************************************************************************/
/* DCR 2487                                                                  */
/* This routine Looks at the INI file to determine which features the user   */
/* wishes to load.                                                           */
/* This routine should be called before we attempt to load the modem.        */
/* We will use this routine to determine if the Discriminator is present.    */
/*****************************************************************************/
ULONG MWM_ENTRY mwmSpcfcDetermineFeatures(PMWM_DSPINFO pmwmDspInfo, USHORT usPCMValue)
{
  ULONG  ulInstalledFeatures = 0;
  ULONG  ulTypeInstalled = 0; /* Temporary variable for profiles */
//  USHORT usDiscrimStatus = 0;
  char  achGetSizePath[_MAX_PATH];
  USHORT usPCMForce = 0;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcDetermineFeatures entry\n");  }


  /***************************************************************************/
  /* Initialize the Features To Load Flags                                   */
  /***************************************************************************/
  pmwmDspInfo->ulFeaturesToLoad = 0;


  /***************************************************************************/
  /* Determine which features are installed                                  */
  /* Read the TYPES information from the INI file.                           */
  /***************************************************************************/
  /***************************************************************************/
  /* 04/30/96                                                                */
  /* Is V.23    installed?                                                   */
  /* V.23    is TYPE0                                                        */
  /* V.23 is never in "FeaturesToLoad" by default...only when +MS Command    */
  /* is issued.                                                              */
  /***************************************************************************/
  ulTypeInstalled = GetPrivateProfileInt("TYPES","TYPE0",0,INIFile );
  if (ulTypeInstalled)
  {
    ulInstalledFeatures = ulInstalledFeatures | FEATURE_V23;
  }


  /***************************************************************************/
  /* Is V22bis installed?                                                    */
  /* V.22bis is TYPE1                                                        */
  /***************************************************************************/
  ulTypeInstalled = GetPrivateProfileInt("TYPES","TYPE1",0,INIFile );
  if (ulTypeInstalled)
  {
    /*************************************************************************/
    /* The features flag doesn't even contain V22bis since it is included    */
    /* in EVERY modem.                                                       */
    /*************************************************************************/
    /* The following features are automatically installed if the V.22bis     */
    /* Modem is installed...                                                 */
    /*************************************************************************/
    ulInstalledFeatures = ulInstalledFeatures | FEATURE_MNP;
    ulInstalledFeatures = ulInstalledFeatures | FEATURE_MNP5;
  }


  /***************************************************************************/
  /* Is V32 installed?                                                       */
  /* V.32 is TYPE2                                                           */
  /***************************************************************************/
  ulTypeInstalled = GetPrivateProfileInt("TYPES","TYPE2",0,INIFile );
  if (ulTypeInstalled)
  {
    ulInstalledFeatures = ulInstalledFeatures | FEATURE_V32;
  }



  /***************************************************************************/
  /* Is V32bis installed?                                                    */
  /* V.32bis is TYPE3                                                        */
  /***************************************************************************/
  ulTypeInstalled = GetPrivateProfileInt("TYPES","TYPE3",0,INIFile );
  if (ulTypeInstalled)
  {
    ulInstalledFeatures = ulInstalledFeatures | FEATURE_V32BIS;

    /***************************************************************************/
    /* Is V32bis Stub Task installed?                                          */
    /***************************************************************************/
    ulTypeInstalled = GetPrivateProfileInt("FEATURES","V32BISSTUB",0,INIFile );
    if (ulTypeInstalled) {
      ulInstalledFeatures = ulInstalledFeatures | FEATURE_V32BISSTUB;
    } /* endif */

  }


  /***************************************************************************/
  /* Is V32ter installed?                                                    */
  /* V.32ter is TYPE4                                                        */
  /***************************************************************************/
  ulTypeInstalled = GetPrivateProfileInt("TYPES","TYPE4",0,INIFile );
  if (ulTypeInstalled)
  {
    ulInstalledFeatures = ulInstalledFeatures | FEATURE_V32TER;
  }

  /***************************************************************************/
  /* Is V34    installed?                                                    */
  /* V.34    is TYPE5                                                        */
  /***************************************************************************/
  ulTypeInstalled = GetPrivateProfileInt("TYPES","TYPE5",0,INIFile );
  if (ulTypeInstalled)
  {
    ulInstalledFeatures = ulInstalledFeatures | FEATURE_V34;

    /*******************************************************
    ** Do we want V.34 Enhanced Mode?
    ********************************************************/
    ulTypeInstalled = GetPrivateProfileInt("FEATURES","V34EM", 0, INIFile);
    if (ulTypeInstalled) {
      ulInstalledFeatures |= FEATURE_V34EM;
    } /* endif */

    /*******************************************************
    ** Do we want V.PCM?
    ********************************************************/
    pmwmDspInfo->usLongTermPCMValue = usPCMValue;
    if (GetPrivateProfileInt("OPTIONS", "PCMSELECT", 0, INIFile) ) {
      usPCMForce = GetPrivateProfileInt( "WINDOW", "PCMFORCE", 0, INIFile);
      if (usPCMForce) {
        pmwmDspInfo->usLongTermPCMValue &= usPCMForce;
      } /* endif */
    } /* endif */

    if (usPCMValue) {
      ulInstalledFeatures |= FEATURE_PCM;
    } /* endif */

    /*******************************************************
    ** Initialize the DYNASEGM value from the INI File
    ** MTS 7795 - Added 02/17/97 DR
    ********************************************************/
    pmwmDspInfo->usDynaSegmValue = GetPrivateProfileInt("FEATURES","DYNASEGM", 0, INIFile);
    pmwmDspInfo->usV34RAD = GetPrivateProfileInt("FEATURES","V34RAD", 0, INIFile);          //$8
    pmwmDspInfo->usDSDebugValue = GetPrivateProfileInt( "DEBUG", "DSDEBUG", 0 ,INIFile );

  } /* endif */


  /***************************************************************************/
  /* Is V.42 installed?                                                      */
  /* V.42 is TYPE20                                                          */
  /***************************************************************************/
  ulTypeInstalled = GetPrivateProfileInt("TYPES","TYPE20",0,INIFile );
   { MW_SYSLOG_2(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcDetermineFeatures, ulTypeInstalled %lx for FEATURE_V42\n",ulTypeInstalled);  }
  if (ulTypeInstalled)
  {
    ulInstalledFeatures = ulInstalledFeatures | FEATURE_V42;
     { MW_SYSLOG_2(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcDetermineFeatures, ulInstalledFeatures V42 set %lx\n",ulInstalledFeatures);  }
  }



  /***************************************************************************/
  /* Is V42bis installed?                                                    */
  /* TYPE40=1 means V42bis installed.                                        */
  /* TYPE40=0 or no TYPE40 at all in INI means NO V42bis installed.          */
  /***************************************************************************/
  ulTypeInstalled = GetPrivateProfileInt("TYPES","TYPE40",0,INIFile );
  if (ulTypeInstalled)
  {
    /*************************************************************************/
    /* Using the mask, we "OR" in all of the different V42bis dictionary     */
    /* sizes...all are available to be loaded.                               */
    /*************************************************************************/
    ulInstalledFeatures = ulInstalledFeatures | FEATURE_V42BIS;
  }

  /***************************************************************************/
  /* Is Class 2 Fax installed?                                               */
  /* Class 2 fax is TYPE60                                                   */
  /***************************************************************************/
  ulTypeInstalled = GetPrivateProfileInt("TYPES","TYPE60",0,INIFile );
  if (ulTypeInstalled)
  {
    ulInstalledFeatures = ulInstalledFeatures | FEATURE_CLASS2;
    /*************************************************************************/
    /* If Class 2 Fax is installed, then we automatically enable it.         */
    /*************************************************************************/
    pmwmDspInfo->ulFeaturesToLoad = pmwmDspInfo->ulFeaturesToLoad | FEATURE_CLASS2;
  }



  /***************************************************************************/
  /* Is Class 1 Fax installed?                                               */
  /* Class 1 fax is TYPE100                                                  */
  /***************************************************************************/
  ulTypeInstalled = GetPrivateProfileInt("TYPES","TYPE100",0,INIFile );
  if (ulTypeInstalled)
  {
    ulInstalledFeatures = ulInstalledFeatures | FEATURE_CLASS1;
    /*************************************************************************/
    /* If Class 1 Fax is installed, then we automatically enable it.         */
    /*************************************************************************/
    pmwmDspInfo->ulFeaturesToLoad = pmwmDspInfo->ulFeaturesToLoad | FEATURE_CLASS1;
  }

  /***************************************************************************/
  /* Is V.80 installed?                                                    $1*/
  /* V.80 is TYPE170                                                         */
  /***************************************************************************/
  pmwmDspInfo->bV80HandsetSnoop = 0;
  ulTypeInstalled = GetPrivateProfileInt("TYPES","TYPE170",0,INIFile );
  if (ulTypeInstalled)
  {
    ulInstalledFeatures |= FEATURE_V80;
    pmwmDspInfo->ulFeaturesToLoad |= FEATURE_V80;

    /******************************************************************
    ** 03/11/97 DR - Check for V.80 RAD.
    ******************************************************************/
    pmwmDspInfo->usV80RAD = GetPrivateProfileInt("FEATURES", "V80RAD", 0, INIFile);
  }

  /***************************************************************************/
  /* Do we want V.42 loaded?                                                 */
  /***************************************************************************/
  ulTypeInstalled = GetPrivateProfileInt("FEATURES","V42",
                                         0,INIFile );

   { MW_SYSLOG_2(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcDetermineFeatures ulTypeInstalled %lx FEATURES V42\n",ulTypeInstalled);  }
  if (ulTypeInstalled)
  {
     { MW_SYSLOG_3(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcDetermineFeatures ulInstalledFeatures %lx FEATURE_V42 %x\n",ulInstalledFeatures,FEATURE_V42);  }
    if ( (ulInstalledFeatures & FEATURE_V42) )
    {
      pmwmDspInfo->ulFeaturesToLoad = pmwmDspInfo->ulFeaturesToLoad | FEATURE_V42;
       { MW_SYSLOG_2(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcDetermineFeatures ulFeaturesToLoad %lx\n",pmwmDspInfo->ulFeaturesToLoad);  }
  
    }
  }

  /***************************************************************************/
  /* See what the INI file wants to have loaded                              */
  /* This looks at [FEATURES] words like V42BIS=, MNP=, V42= etc.            */
  /***************************************************************************/
  /***************************************************************************/
  /* Do we want MNP loaded?                                                  */
  /***************************************************************************/
  ulTypeInstalled = GetPrivateProfileInt("FEATURES","MNP",
                                         0,INIFile );
  if (ulTypeInstalled)
  {
    if ( (ulInstalledFeatures & FEATURE_MNP) )
      pmwmDspInfo->ulFeaturesToLoad = pmwmDspInfo->ulFeaturesToLoad | FEATURE_MNP;
      /***********************************************************************/
      /* MNP5 must be selected along with MNP                                */
      /***********************************************************************/
      pmwmDspInfo->ulFeaturesToLoad = pmwmDspInfo->ulFeaturesToLoad | FEATURE_MNP5;
  }

  /***************************************************************************/
  /* Do we want to load V42bis?                                              */
  /***************************************************************************/
  ulTypeInstalled = GetPrivateProfileInt("FEATURES","V42BIS",
                                         0,INIFile );

  /***************************************************************************/
  /* V.42bis will not work is MNP or V42 is not loaded.                      */
  /***************************************************************************/
  if ( (pmwmDspInfo->ulFeaturesToLoad & FEATURE_MNP ) |
       (pmwmDspInfo->ulFeaturesToLoad & FEATURE_V42 ) )
  {
    switch(ulTypeInstalled)
    {
      case 512:
        if ( (ulInstalledFeatures & FEATURE_V42BIS) )
          pmwmDspInfo->ulFeaturesToLoad = pmwmDspInfo->ulFeaturesToLoad | FEATURE_V42BIS_512;
        break;
      case 1024:
        if ( (ulInstalledFeatures & FEATURE_V42BIS) )
          pmwmDspInfo->ulFeaturesToLoad = pmwmDspInfo->ulFeaturesToLoad | FEATURE_V42BIS_1024;
        break;
      case 2048:
        if ( (ulInstalledFeatures & FEATURE_V42BIS) )
          pmwmDspInfo->ulFeaturesToLoad = pmwmDspInfo->ulFeaturesToLoad | FEATURE_V42BIS_2048;
        break;
      case 4096:
        if ( (ulInstalledFeatures & FEATURE_V42BIS) )
          pmwmDspInfo->ulFeaturesToLoad = pmwmDspInfo->ulFeaturesToLoad | FEATURE_V42BIS_4096;
        break;
    }
  }

  /***************************************************************************/
  /* Do we want to load only modem control and asynch?    MTS=SB             */
  /***************************************************************************/
  ulTypeInstalled = GetPrivateProfileInt("FEATURES","SLIMBASE",0,INIFile );
  if (ulTypeInstalled)
  {
     pmwmDspInfo->ulFeaturesToLoad |= FEATURE_SLIMBASE;
  }

  /***************************************************************************/
  /* Do we want to load the ASYNC2X module for connections at V.34 and up?   */
  /***************************************************************************/
  ulTypeInstalled = GetPrivateProfileInt("FEATURES","ASYNC2X",0,INIFile );
  if (ulTypeInstalled)
  {
     pmwmDspInfo->ulFeaturesToLoad |= FEATURE_ASYNC2X;
  }




//  achGetSizePath = GetPrivateProfileString("DEBUG","GETSIZE",
//                                         "\0",INIFile );
  ulTypeInstalled = GetPrivateProfileString("DEBUG","GETSIZE", "\0",
                                            achGetSizePath,
                                            _MAX_PATH,
                                            INIFile);
  DPF("Just got by the GetPrivateProfileString");

  /***************************************************************************/
  /* Save installed features info                                            */
  /***************************************************************************/
  pmwmDspInfo->ulInstalledFeatures = ulInstalledFeatures;


  /***************************************************************************/
  /* Set features to load in parser and modem control                        */
  /***************************************************************************/
  mwmSpcfcSetFeatures(pmwmDspInfo, 0);

  /******************************************************************
  ** 11/20/96 DR - Check for 17 MIPS platform.
  ******************************************************************/
  pmwmDspInfo->us17MIPS = GetPrivateProfileInt( "OPTIONS", "MW17MIPS", 0, INIFile);

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcDetermineFeatures exit\n");  }


  return 0;
}







ULONG mwmSpcfcGetTaskCPF(PMWM_DSPINFO pmwmDspInfo, HMTASK hTask, ULONG *pulTaskCPF,char *achFile, SHORT sLine)

{
  PVOID   pTaskInfoBuffer = 0;
  USHORT  usSize = 0;
  ULONG   ulRC;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcGetTaskCPF entry\n");  }


  /***************************************************************************/
  /* First get Frame Handle for Task.                                        */
  /***************************************************************************/
  ulRC=dspQueryTaskInfo(hTask,&usSize,(PVOID)NULL);
  if (ulRC != DSP_INS_BUFFER)
  {
    mwmHandleError(pInstanceInfo,achFile,sLine,
                   MWM_DSP_ERROR,
                   ulRC );
    return ulRC;
  }

  /***************************************************************************/
  /* MTS 3525                                                                */
  /* Allocate more than needed...in case someone else is changing the system */
  /* before we get a chance to complete the query.                           */
  /***************************************************************************/
  usSize += 20;

  pTaskInfoBuffer = calloc( 1, usSize );
  if (!pTaskInfoBuffer)
  {
    mwmHandleError(pInstanceInfo,achFile,sLine,
                   MWM_OS_ERROR,
                   0 );
    return 1;
  }


  ulRC=dspQueryTaskInfo(hTask,&usSize,pTaskInfoBuffer);
  if (ulRC != DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,achFile,sLine,
                   MWM_DSP_ERROR,
                   ulRC );
    free(pTaskInfoBuffer);
    return ulRC;
  }


  *pulTaskCPF = (((RTASKINFO *)pTaskInfoBuffer) -> TSKINFO_ulCPF);

  free(pTaskInfoBuffer);

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcGetTaskCPF exit\n");  }


  return 0;
}


ULONG mwmSpcfcGetTaskFPS(PMWM_DSPINFO pmwmDspInfo, HMTASK hTask, ULONG *pulTaskFPS)

{
  PVOID   pTaskInfoBuffer;
  PVOID   pMiscInfoBuffer;
  USHORT  usSize = 0;
  HFM     hTaskFrameHandle;
  ULONG   ulRC;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcGetTaskFPS entry\n");  }


  /***************************************************************************/
  /* First get Frame Handle for Task.                                        */
  /***************************************************************************/
  ulRC=dspQueryTaskInfo(hTask,&usSize,(PVOID)NULL);
  if (ulRC != DSP_INS_BUFFER)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC );
    return ulRC;
  }

  /***************************************************************************/
  /* MTS 3525                                                                */
  /* Allocate more than needed...in case someone else is changing the system */
  /* before we get a chance to complete the query.                           */
  /***************************************************************************/
  usSize += 20;

  pTaskInfoBuffer = calloc( 1, usSize );
  if (!pTaskInfoBuffer)
  {
    if (ulRC != DSP_NOERROR)
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_OS_ERROR,
                     0 );
    return 1;
  }


  ulRC=dspQueryTaskInfo(hTask,&usSize,pTaskInfoBuffer);
  if (ulRC != DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC );
    free(pTaskInfoBuffer);
    return ulRC;
  }


  hTaskFrameHandle = (((RTASKINFO *)pTaskInfoBuffer) -> TSKINFO_hFM);

  free(pTaskInfoBuffer);

  /***************************************************************************/
  /* Next get Frame Manager Misc Info...                                     */
  /***************************************************************************/
  usSize=0;
  ulRC=dspQueryMiscInfo(MISC_FM, hTaskFrameHandle,&usSize,NULL);
  if (ulRC!=DSP_INS_BUFFER)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return ulRC;
  }

  /***************************************************************************/
  /* MTS 3525                                                                */
  /* Allocate more than needed...in case someone else is changing the system */
  /* before we get a chance to complete the query.                           */
  /***************************************************************************/
  usSize += 20;

  pMiscInfoBuffer = calloc( 1, usSize );
  if (!pMiscInfoBuffer)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_OS_ERROR,
                   0);
    return 1;
  }


  ulRC=dspQueryMiscInfo(MISC_FM, hTaskFrameHandle,&usSize,pMiscInfoBuffer);
  if (ulRC != DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC );
    free(pMiscInfoBuffer);
    return ulRC;
  }


  *pulTaskFPS=( (((RFMINFO *)pMiscInfoBuffer) -> FMINFO_ulSampleRate) /
               (((RFMINFO *)pMiscInfoBuffer) -> FMINFO_usSamplesPerFrame) );


  free(pMiscInfoBuffer);

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcGetTaskFPS exit\n");  }


  return 0;
}



/*****************************************************************************/
/* mwmSpcfcInitialize                                                        */
/*****************************************************************************/
/* Initialize the Modem specific code.                                       */
/*                                                                           */
/* -Allocate the MWM_DSPINFO structure...return a handle to it.              */
/*****************************************************************************/
PMWM_DSPINFO MWM_ENTRY mwmSpcfcInitialize( PVOID pCallerINIFile, PMWM_INSTANCE_INFO pCallerInstanceInfo, USHORT usPCMValue )
{
  PMWM_DSPINFO pmwmDspInfo = 0;
  ULONG        ulRC;
  USHORT       usSize = 0;
  RMGRINFO     *pMgrInfoBuffer = NULL;
  USHORT       usTempData;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcInitialize entry\n");  }

  /***************************************************************************/
  /* Save the Caller's INI File Handle (OS/2) or INI Filename                */
  /* and pointer to the callers instance info.                               */
  /***************************************************************************/
  INIFile = pCallerINIFile;
  pInstanceInfo = pCallerInstanceInfo;

  /***************************************************************************/
  /* Allocate The DSP Info structure.                                        */
  /***************************************************************************/
  pmwmDspInfo = (PMWM_DSPINFO)calloc( 1, sizeof(MWM_DSPINFO) );
  if (!pmwmDspInfo)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_OS_ERROR,
                   0);
    return 0;
  }



  pGlobalDspInfo = pmwmDspInfo;

  /***************************************************************************/
  /* Get the DSP handle                                                      */
  /*   -First get the size of the buffer to be returned.                     */
  /*   -Allocate the buffer                                                  */
  /*   -Query the information again supplying the correct buffer size        */
  /***************************************************************************/

  ulRC = dspQueryManagerInfo(&usSize, pMgrInfoBuffer);
  if (ulRC != DSP_INS_BUFFER)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC );
    return 0;
  }

  /***************************************************************************/
  /* MTS 3525                                                                */
  /* Allocate more than needed...in case someone else is changing the system */
  /* before we get a chance to complete the query.                           */
  /***************************************************************************/
  usSize += 20;


  pMgrInfoBuffer = (RMGRINFO *)calloc( 1, usSize );
  if (!pMgrInfoBuffer)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_OS_ERROR,
                   0);
    return 0;
  }


  ulRC = dspQueryManagerInfo(&usSize, pMgrInfoBuffer);

  if (ulRC != DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC);
    free(pMgrInfoBuffer);
    return 0;
  }

  /***************************************************************************/
  /* Put the DSP handle into the shared memory area.                         */
  /***************************************************************************/
  pmwmDspInfo->hDSP = pMgrInfoBuffer -> MGRINFO_phDSP[0];

  /***************************************************************************/
  /* Free the Manager Info buffer...we are finished with it.                 */
  /***************************************************************************/
  free(pMgrInfoBuffer);


  /***************************************************************************/
  /* Do a dummy operation to the dsp to make sure it is enabled.             */
  /***************************************************************************/
  ulRC = dspMemTransfer(pmwmDspInfo->hDSP, 0x1000,
                        &usTempData, 1,
                        DSP_MEMXFER_DATA_READ);
  if (ulRC != DSP_NOERROR)
  {
     return 0;
  }




  /***************************************************************************/
  /* Open the MEIO  , get the MEIO handle, put it directly into the          */
  /* shared memory area.                                                     */
  /***************************************************************************/
  ulRC = dspMeioOpen(pmwmDspInfo->hDSP, &hMEIO, 0);
  if (ulRC != MEIO_NOERROR)    // Check the error codes
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_MEIO_ERROR,
                   ulRC);
    return 0;
  }



  /***************************************************************************/
  /* 6/7/94 Move this from the OS Specific code.                             */
  /* Allocate Parser Global Memory, and initialize parts of it.              */
  /***************************************************************************/
  ulRC = mwmParseInitializeParser( pInstanceInfo->hwndMainWindow,
                                   INIFile );
  if (ulRC)
  {
    return 0;
  }

  /***************************************************************************/
  /* Query the Debug Window setting from the parser                          */
  /***************************************************************************/
  ulRC = mwmParseQueryDebugSettings(&usDebugWindow);
  if (ulRC)
    return 0;


  ulRC = mwmSpcfcSetWTParms(pmwmDspInfo);
  if (ulRC)
    return 0;

  /***************************************************************************/
  /* Determine which features that we should load.                           */
  /* This routine compares the installed features with the user's requested  */
  /* features to determine what this run of the modem will actually look like*/
  /***************************************************************************/

  ulRC = mwmSpcfcDetermineFeatures(pmwmDspInfo, usPCMValue);
  if (ulRC)
    return 0;

  /***************************************************************************/
  /* 03/14/97 - CB                                                           */
  /* Take the blacklisting out of the modem                                  */
  /***************************************************************************/

  ulRC = BL_APIversion();
  if (ulRC != BL_API_VERSION) {
    mwmHandleError( pInstanceInfo, szThisFile, __LINE__,
                    MWM_BL_ERROR, -1 );
  } else {
    ulRC = BL_RegisterDevice ( DEVICE_MODEM, "IBM DSP Data Modem",
                               &pmwmDspInfo->ulBlcklistModemHandle, 0);
    if (ulRC != BL_SUCCESSFUL) {
      mwmHandleError(pInstanceInfo, szThisFile, __LINE__,
                     MWM_BL_ERROR, ulRC );
    } else {
      ulRC = BL_RegisterDevice ( DEVICE_FAX, "IBM DSP Fax Modem",
                                 &pmwmDspInfo->ulBlcklistFaxHandle, 0);
      if (ulRC != BL_SUCCESSFUL) {
        mwmHandleError(pInstanceInfo, szThisFile, __LINE__,
                       MWM_BL_ERROR, ulRC );
      } /* endif */
    } /* endif */
  }
  { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcInitialize exit\n");  }

  return pmwmDspInfo;
}/* mwmSpcfcInitialize */




ULONG MWM_ENTRY mwmSpcfcLoadModem(PMWM_DSPINFO pmwmDspInfo )
{


   RC      ulRC;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcLoadModem entry\n");  }


  /***************************************************************************/
  /* Make necessary MEIO connections.                                        */
  /* This routine checks to see if we are in discriminator mode.             */
  /***************************************************************************/
  ulRC = mwmLoadMakeMEIOConnections(pmwmDspInfo);
  if (ulRC)
    return ulRC;


  /***************************************************************************/
  /* Load the configuration requested.                                       */
  /* This function is also called when a power management "resume" event     */
  /* is received.                                                            */
  /***************************************************************************/
  ulRC = mwmLoadConfiguration(pmwmDspInfo);
  if (ulRC)
    return ulRC;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcLoadModem exit\n");  }


  return 0;
}



#define fixAPIENTRY  _far _pascal _export _loadds

/* paulsch:  we won't deal with power management for now...  so we'll ifdef it out for
 * linux...  */
#ifndef LINUX_BLD

/*****************************************************************************/
/* DCR 2456 Power Management.                                                */
/*****************************************************************************/
ULONG   MWM_ENTRY mwmSpcfcNSClientProc(ULONG  dwConversationId,
                                       USHORT wMessage,
                                       ULONG  lParam,
                                       USHORT wSize,
                                       void FAR *pBuffer)

{
  ULONG ulRC = 0;
  PMWM_DSPINFO pmwmDspInfo;
  USHORT usUARTMode;
  REGISTERS Registers;
  HMOD   hmodModem;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcNSClientProc entry\n");  }


  pmwmDspInfo = (PMWM_DSPINFO)dwConversationId;
  switch (wMessage)
  {
    case CLI_CONNECT:
      /***********************************************************************/
      /* dwConversationId contains the instance data passed as the last      */
      /* parameter from the mwConnectToServer call.                          */
      /* In our case, the instance data is the pointer to global data (pgd)  */
      /***********************************************************************/
      ulRC = dwConversationId;
      break;
    case NS_EVENTNOTIFICATION:
      switch(lParam)
      {
        case NS_DSPINIT:
          /*******************************************************************/
          /* MTS 4593                                                        */
          /* Close modem on INIT rather than trying to "resume"              */
          /* The attempt to resume was causing lots of developers problems.  */
          /* For instance, the intermetrics tools do an INITDSP expecting the*/
          /* DSP to be empty afterwards.  We were quickly reloading, and     */
          /* getting in the way of the debugger.                             */
          /*******************************************************************/
            mwmParsePostMessage(WM_MWM_ERROR,NS_DSPINIT,0);
            mwmParsePostMessage(WM_MWM_ERROR,NS_DSPINIT,0);
          break;
        case NS_PWRRESUME:
        case NS_PWRCRITICALRESUME:
            DPF("Processing Resume...");

            ulRC = dspNameToModuleHandle("MODEM", &hmodModem);
            if (ulRC == DSP_NAME_NOT_FOUND) {

              /*****************************************************************/
              /* 1/11/95                                                       */
              /* MTS 4362 Resume into DMA Mode if that is how we went down...  */
              /* First, save away the Mode we were in when we went down.       */
              /* Then, after loading Modem, if we were in DMA Mode..switch.    */
              /*****************************************************************/
              usUARTMode = pmwmDspInfo->usUARTMode;

              /*****************************************************************/
              /* MTS 3988                                                      */
              /* 10/03/94                                                      */
              /* Handles need to be set to 0 since if we were suspended during */
              /* a connection, we might not reload the same modules...         */
              /* Just to be sure, reset all of the handles.                    */
              /*****************************************************************/
              pmwmDspInfo->ulAddressLEDS = 0;
              pmwmDspInfo->hmodInterpolator = 0;
              pmwmDspInfo->hmodCallProg = 0;          // MTS 3361
              pmwmDspInfo->hmodV22 = 0;
              pmwmDspInfo->hmodVDP = 0;
              pmwmDspInfo->hmodT30 = 0;
              pmwmDspInfo->hmodFAXPMP = 0;
              pmwmDspInfo->hmodCL1_HDLC = 0;
              pmwmDspInfo->hmodCLASSx = 0;
              pmwmDspInfo->hmodSILDET = 0;
              pmwmDspInfo->hmodModem = 0;
              pmwmDspInfo->hmodAsync = 0;
              pmwmDspInfo->hmodMNP = 0;
              pmwmDspInfo->hmodMNP5 = 0;
              pmwmDspInfo->hmodV42bis = 0;
              pmwmDspInfo->hmodV42 = 0;
              pmwmDspInfo->hmtV42PKT = 0;             // V.42 Packet task handle
              pmwmDspInfo->hmtV42LAPDLC = 0;          // V.42 LAP DLC Task handle
              pmwmDspInfo->hmtV42MAC = 0;             // V.42 MAC Task handle
              pmwmDspInfo->hmtV42bis = 0;             // V.42bis task handle
              pmwmDspInfo->hmtMNP5 = 0;               // MNP5 task handle
              pmwmDspInfo->hmtProtSupr = 0;
              pmwmDspInfo->hmtCLASSx = 0;
              pmwmDspInfo->hmtT30 = 0;
              pmwmDspInfo->hmtMHINT = 0;
              pmwmDspInfo->hmctlTask = 0;             // modem control task handle
              pmwmDspInfo->hv22Task = 0;              // v22 task handle
              pmwmDspInfo->hvdpTask = 0;              // vdp task handle
              pmwmDspInfo->hasyTask = 0;              // async task handle
              pmwmDspInfo->hmtMNP = 0;                // mnp task handle
              pmwmDspInfo->hinterpTask = 0;           // interpolator task handle
              pmwmDspInfo->hmtUART = 0;               // UART Task Handle
              pmwmDspInfo->hmodFakeUART = 0;          // MTS 4362 DMA UART Handle...
              pmwmDspInfo->hmtFakeUART = 0;           // MTS 4362 DMA UART Handle...
              pmwmDspInfo->hmtCallProg = 0;           // MTS 3361
              pmwmDspInfo->hmtCallProgMove = 0;       // MTS 3361
              pmwmDspInfo->hsegPercentTT = 0;         // MTS 4086
              pmwmDspInfo->hmodRAMMBASE = 0;          // MTS 4092
              pmwmDspInfo->hmodGSM     = 0;
//              pmwmDspInfo->hmodINTERP8 = 0;           // MTS 4092
//              pmwmDspInfo->hmodDECIMA  = 0;           // MTS 4092
              pmwmDspInfo->hmodSPKVOD  = 0;
              pmwmDspInfo->hmtRAMMBASE = 0;           // MTS 4092
              pmwmDspInfo->hmtGSMCOD   = 0;           // MTS 4092
              pmwmDspInfo->hmtGSMDEC   = 0;           // MTS 4092
              pmwmDspInfo->hmtMOVE79   = 0;           // MTS 4092
              pmwmDspInfo->hmtMOVE97   = 0;           // MTS 4092
//              pmwmDspInfo->hmtINTERP8  = 0;           // MTS 4092
//              pmwmDspInfo->hmtDECIMA   = 0;           // MTS 4092
              pmwmDspInfo->hmtSPKVOD   = 0;
              pmwmDspInfo->hmtV8       = 0;           // MTS 4416
              pmwmDspInfo->hmodV8      = 0;           // MTS 4416
              pmwmDspInfo->hmodV34     = 0;
              pmwmDspInfo->hmtV34      = 0;
              pmwmDspInfo->hsegV42BTempGPC1 = 0;
              pmwmDspInfo->hsegV42BTempGPC2 = 0;
              pmwmDspInfo->hsegV42BTempDSEG = 0;

              // 06/12/97 - Clear voice-related handles.
              pmwmDspInfo->hmodCANCEL       = 0;
              pmwmDspInfo->hmtCANCEL        = 0;
              pmwmDspInfo->hmodSPK076       = 0;
              pmwmDspInfo->hmtSPK076        = 0;
              pmwmDspInfo->hmtSPK076DBM944  = 0;
              pmwmDspInfo->hmodDTMFRM       = 0;
              pmwmDspInfo->hmtDTMFRMMove    = 0;
              pmwmDspInfo->hmodDISC         = 0;
              pmwmDspInfo->hmtDISCFILT      = 0;
              pmwmDspInfo->hmodINT9608      = 0;
              pmwmDspInfo->hmtINT9608       = 0;
              pmwmDspInfo->hmodINT0896      = 0;
              pmwmDspInfo->hmtINT0896       = 0;
              pmwmDspInfo->hmodMIX9LINE     = 0;
              pmwmDspInfo->hmtMIX9LINE      = 0;
              pmwmDspInfo->hmodMIX9RECORD   = 0;
              pmwmDspInfo->hmtMIX9RECORD    = 0;
              pmwmDspInfo->hmodGAINWT       = 0;
              pmwmDspInfo->hmtGAINWT        = 0;
              pmwmDspInfo->hmodAGC9         = 0;
              pmwmDspInfo->hmtAGC9          = 0;
              pmwmDspInfo->hmodAGC9WT       = 0;
              pmwmDspInfo->hmtAGC9WT        = 0;
              pmwmDspInfo->hmodAGC8         = 0;
              pmwmDspInfo->hmtAGC8          = 0;

              pmwmDspInfo->hmodCALLERID      = 0;
              pmwmDspInfo->hmtCALLERID_ASYNC = 0;

              pmwmDspInfo->hmodVVPROT       = 0;
              pmwmDspInfo->hmodVVHDLC       = 0;
              pmwmDspInfo->hmodTONEGEN      = 0;
              pmwmDspInfo->hmodROUTER       = 0;
              pmwmDspInfo->hmodVV0896       = 0;
              pmwmDspInfo->hmodVV9608       = 0;
              pmwmDspInfo->hmodV80MAC       = 0;
              pmwmDspInfo->hmtV80MAC        = 0;

              pmwmDspInfo->hmodpc2gpc       = 0;
              pmwmDspInfo->htpc2gpc         = 0;
              pmwmDspInfo->hmodvplay        = 0;
              pmwmDspInfo->htvplay          = 0;
              pmwmDspInfo->hmodgpc2pc       = 0;
              pmwmDspInfo->htgpc2pc         = 0;
              pmwmDspInfo->hmodvrec         = 0;
              pmwmDspInfo->htvrec           = 0;
              pmwmDspInfo->hmodDRing        = 0;
              pmwmDspInfo->hmtDRing         = 0;
              pmwmDspInfo->hmodMHINT        = 0;

              pmwmDspInfo->hmodADPCM           = 0;
              pmwmDspInfo->hmtADPCM_COMPRESS   = 0;
              pmwmDspInfo->hmtADPCM_DECOMPRESS = 0;

              // CMVC 559 - Clear memory pool segment handle
              pmwmDspInfo->ahsegMemPool[0]  = 0;

              mwmCidlInit(pmwmDspInfo);               // reset caller id  MTS6619
              mwmCidlCheck(pmwmDspInfo);              // re-init caller idMTS6619


              /*****************************************************************/
              /* Load Modem                                                    */
              /*****************************************************************/
              ulRC=mwmLoadConfiguration(pmwmDspInfo);

              /*****************************************************************/
              /* 1/11/95 MTS 4362                                              */
              /* If we were in DMA Mode before suspend, switch to DMA Mode Now.*/
              /*****************************************************************/
              if (usUARTMode & USE_DMA)
              {
                mwmcapiMoveUARTToDMA(pmwmDspInfo);
              }

              mwmParsePostMessage( WM_MWM_UPDATE_STATUS, MWM_STATE_READY, 0);


            } else {
              mwmParseSetSuspendFlag(0);
            } /* endif */

            mwmParsePostMessage(WM_MWM_DISCRIM_STATUS, TRUE, 0);

// 04/04/95  Now do this in the mwmParseInitializeModem "last case"
// This was a race ... moving it to InitializeModem eliminates the race.
//            if (!ulRC)
//            {
              /***************************************************************/
              /* Reset the Suspend Flag.                                     */
              /***************************************************************/
//              mwmParseSetSuspendFlag(0);
//            }
          break;
        case NS_PWRSUSPEND:

            DPF("Processing Suspend...");

            /**********************************************************
            ** NOTE: Certain systems have added Wake-Up-On-Ring
            ** capability where the DSP is not shut down on a suspend.
            ** For those systems, we do not want to disconnect IPCs.
            ***********************************************************/
            pmwmDspInfo->bSuspending = TRUE;
            if (GetPrivateProfileInt("OPTIONS", "WAKEONRING", 0, INIFile) == 0) {


              // Disconnect the IPC to modem control
              ulRC = dspDisconnectIPC(pmwmDspInfo->hmctlIPC);
            } /* endif */

            // Check to see if we suspended during a call, if so restore from
            // saved registers when we went of hook.
            if(pmwmDspInfo->bSuspendingDuringCall == TRUE)
              mwmParseSetModemRegisters(&(pmwmDspInfo->SavedRegisters));

            pmwmDspInfo->bSuspendingDuringCall = FALSE;

            /***************************************************************/
            /* CMVC 573, 574 - Reset Blacklisting devices on a suspend.    */
            /***************************************************************/
            ulRC = BL_ResetDevice(pmwmDspInfo->ulBlcklistFaxHandle, 0L);
            if (ulRC != BL_SUCCESSFUL)
            {
              mwmHandleError(pInstanceInfo, szThisFile, __LINE__,
                             MWM_BL_ERROR, ulRC);
              return ulRC;
            }
            ulRC = BL_ResetDevice(pmwmDspInfo->ulBlcklistVoiceHandle, 0L);
            if (ulRC != BL_SUCCESSFUL)
            {
              mwmHandleError(pInstanceInfo, szThisFile, __LINE__,
                             MWM_BL_ERROR, ulRC);
              return ulRC;
            }
            ulRC = BL_ResetDevice(pmwmDspInfo->ulBlcklistModemHandle, 0L);
            if (ulRC != BL_SUCCESSFUL)
            {
              mwmHandleError(pInstanceInfo, szThisFile, __LINE__,
                             MWM_BL_ERROR, ulRC);
              return ulRC;
            }

            mwmParseSaveModemRegisters( );

            /*****************************************************************/
            /* Set the Suspend Flag                                          */
            /*****************************************************************/
            mwmParseSetSuspendFlag(1);

          break;
      }
      break;

  } /* endswitch */

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcNSClientProc exit\n");  }

  return ulRC;
}
/*****************************************************************************/
/* End ... DCR 2456 Power Management.                                        */
/*****************************************************************************/
#endif

/* paulsch: this seems to deal with WT table events..  We're US only right now.. */
#ifndef LINUX_BLD

ULONG fixAPIENTRY mwmSpcfcWTClientProc(ULONG  dwConversationId,
                                    USHORT wMessage,
                                    ULONG  lParam,
                                    USHORT wSize,
                                    void FAR *pBuffer)

{
  ULONG ulRC = 0;
  PMWM_DSPINFO pmwmDspInfo;
  DWORD        dwWaitStatus;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcWTClientProc entry\n");  }


//  pmwmDspInfo = (PMWM_DSPINFO)dwConversationId;
  pmwmDspInfo = pGlobalDspInfo;
  switch (wMessage)
  {
    case WT_TABLECHANGED:
      DPF("Received WT_TABLECHANGED event");
      /***********************************************************************/
      /* Unload the entire modem.                                            */
      /* Don't disconnect the Server connections.                            */
      /***********************************************************************/

      DPF("pmwmDspInfo: %x",pmwmDspInfo );
      dwWaitStatus = WaitForSingleObject(hModemMutex,MODEM_TIMEOUT);
      if((dwWaitStatus !=     WAIT_OBJECT_0) && (dwWaitStatus != WAIT_ABANDONED))
      {
        mwmHandleError(pInstanceInfo,szThisFile,__LINE__,    //$1
                       MWM_OS_ERROR,                        //$1
                       0);                                   //$1
        return 0;
      }

      // Before resetting, treat this is a Suspend so we can restore later.
      mwmParseSetSuspendFlag(1);

      ulRC = mwmSpcfcUnloadDSPCode(pmwmDspInfo);

      /***********************************************************************/
      /* Clear all of the handles etc...                                     */
      /***********************************************************************/
//      memset(pmwmDspInfo,0,sizeof(MWM_DSPINFO));

      pmwmDspInfo->ulAddressLEDS = 0;

      /***********************************************************************/
      /* Refresh the parser's world trade parameters.                        */
      /***********************************************************************/
      ulRC = mwmSpcfcSetWTParms(pmwmDspInfo);

      /***********************************************************************/
      /* Now reload the modem.                                               */
      /***********************************************************************/
          if(!ulRC)
                ulRC = mwmSpcfcLoadModem(pmwmDspInfo);
                if(!ReleaseMutex(hModemMutex))
                {
            mwmHandleError(pInstanceInfo,szThisFile,__LINE__,    //$1
                           MWM_OS_ERROR,                        //$1
                           0);                                   //$1
                }

      /***********************************************************
      ** Reset the blacklister
      ************************************************************/
      ulRC = BL_Reset(0);
      if (ulRC) {
        mwmHandleError( pInstanceInfo, szThisFile, __LINE__,
                        MWM_BL_ERROR, ulRC );
        return(ulRC);
      } /* endif */

      mwmParsePostMessage( WM_MWM_UPDATE_STATUS, MWM_STATE_READY, 0);

      if(!mwmParseWTChangeFlagValue())
        mwmParsePostMessage( WM_MWM_TABLE_CHANGED, 0, 0);


      mwmParsePostMessage(WM_MWM_DISCRIM_STATUS, TRUE, 0);

      break;

  } /* endswitch */

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcWTClientProc exit\n");  }

  return ulRC;
}

#endif


ULONG mwmSpcfcPlayRingWave(PMWM_DSPINFO pmwmDspInfo)
{
   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcPlayRingWave exit\n");  }

  return 0;
}

ULONG MWM_ENTRY mwmSpcfcQueryModemStatus( PMWM_DSPINFO pmwmDspInfo, PMWM_MODEM_STATUS pModemStatus,
                                          ULONG *pulAddressLEDS)
{
  ULONG  ulRC = 0;


   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcQueryModemStatus entry\n");  }


  /***************************************************************************/
  /* Call this the first time with the address set a 0.  This routine will   */
  /* convert the label to an address. From then on, call this function with  */
  /* the correct address.  This allows the function to only convert the label*/
  /* on the first call.                                                      */
  /***************************************************************************/
  if (!*pulAddressLEDS)
  {
    ulRC = dspLabelToAddress(pmwmDspInfo->hmctlTask, "MDMSTAT", pulAddressLEDS);
    if (ulRC!=DSP_NOERROR)
    {
      /***********************************************************************/
      /* Must NOT show any error message boxes in this routine.              */
      /* If we do, the shutdown will fail, since this routine is called      */
      /* from the second thread.                                             */
      /***********************************************************************/
//      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
//                     MWM_DSP_ERROR, ulRC);
      return(ulRC);
    }
  }

  ulRC = dspMemTransfer(pmwmDspInfo->hDSP, *pulAddressLEDS,
                        pModemStatus, 4,
                        DSP_MEMXFER_DATA_READ);
  if (ulRC!=DSP_NOERROR)
  {
//    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
//                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcQueryModemStatus exit\n");  }

  return 0;
}






ULONG MWM_ENTRY mwmSpcfcQueryPhoneLine( PMWM_DSPINFO pmwmDspInfo, PUSHORT pusLineAvailable )
{
  PUSHORT pMeioQueryBuffer = NULL;
  ULONG   ulRC;
  ULONG   ulSize;
  USHORT  usCount;


   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcQueryPhoneLine entry\n");  }


  *pusLineAvailable = 1;

  /***************************************************************************/
  /* Using MEIO, query the phone line to make sure it is not already         */
  /* in use.                                                                 */
  /***************************************************************************/
  ulSize = 0;
  ulRC = dspMeioQuery(hMEIO,QACONNECTIONS,&ulSize,(PVOID)pMeioQueryBuffer);

  pMeioQueryBuffer = (PUSHORT)calloc( 1, ulSize );
  if (!pMeioQueryBuffer)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_OS_ERROR,
                   0);
    return ulRC;
  }
//  memset(pMeioQueryBuffer,0,ulSize);

  ulRC = dspMeioQuery(hMEIO,QACONNECTIONS,&ulSize,(PVOID)pMeioQueryBuffer);
  if (ulRC != MEIO_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_MEIO_ERROR,
                   ulRC );
    free(pMeioQueryBuffer);
    pMeioQueryBuffer = NULL;
    return ulRC;
  }

  usCount = 0;
  while( pMeioQueryBuffer[usCount])
  {
    if (pMeioQueryBuffer[usCount]==TELEOUT_1)
    {
      *pusLineAvailable = 0;
    }
    usCount++;
  }

  /***************************************************************************/
  /* Free the query buffer...we are finished with it.                        */
  /***************************************************************************/
  if (pMeioQueryBuffer)
  {
    free(pMeioQueryBuffer);
    pMeioQueryBuffer = NULL;
  }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcQueryPhoneLine exit\n");  }


  return 0;

}




ULONG mwmSpcfcQueryUARTTaskHandle( PMWM_DSPINFO pmwmDspInfo, HMTASK  *phmtTask )
{
  RDSPINFO *pDSPInfoBuffer = NULL;
  ULONG   ulRC;
  USHORT  usSize;
  HMOD    hmodMWAVEOS;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcQueryUARTTaskHandle entry\n");  }


  /***************************************************************************/
  /*  Query the Manager to get the MWAVEOS module Handle.                    */
  /*  We will store the handle to the UART task in the DspInfo Structure.    */
  /*   -First get the size of the buffer to be returned.                     */
  /*   -Allocate the buffer                                                  */
  /*   -Query the information again supplying the correct buffer size        */
  /***************************************************************************/
  usSize = 0;
  ulRC = dspQueryDSPInfo(pmwmDspInfo->hDSP, &usSize, pDSPInfoBuffer);
  if (ulRC != DSP_INS_BUFFER)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC );
    return ulRC;
  }

  /***************************************************************************/
  /* MTS 3525                                                                */
  /* Allocate more than needed...in case someone else is changing the system */
  /* before we get a chance to complete the query.                           */
  /***************************************************************************/
  usSize += 20;

  pDSPInfoBuffer = (RDSPINFO *)calloc( 1, usSize );
  if (!pDSPInfoBuffer)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_OS_ERROR,
                   0);
    return 1;
  }


  ulRC = dspQueryDSPInfo(pmwmDspInfo->hDSP, &usSize, pDSPInfoBuffer);
  if (ulRC != DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC);
    free(pDSPInfoBuffer);
    return ulRC;
  }

  /***************************************************************************/
  /* Put the DSP handle into the shared memory area.                         */
  /***************************************************************************/
  hmodMWAVEOS = pDSPInfoBuffer -> DSPINFO_hMod_MwaveOS;

  /***************************************************************************/
  /* Free the DSP Info buffer...we are finished with it.                     */
  /***************************************************************************/
  free(pDSPInfoBuffer);

  /***************************************************************************/
  /* Now that we have the Module Handle to the MwaveOS, get the Task Handle  */
  /* to UART000                                                              */
  /***************************************************************************/
  ulRC = dspNameToTaskHandle(hmodMWAVEOS, "UART000", phmtTask);
  if (ulRC)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC );
    return (ulRC);
  }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcQueryUARTTaskHandle exit\n");  }



  return 0;
}






ULONG mwmSpcfcUnloadDSPCode( PMWM_DSPINFO pmwmDspInfo )
{
  ULONG ulRC = 0;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcUnloadDSPCode entry\n");  }


//  printf("Putting Modules in Standby...\n");
  /***************************************************************************/
  /* First, put all active modules in standby...                             */
  /***************************************************************************/
  if (pmwmDspInfo->hmodT30)
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodT30, DSP_ACTIVATE_STANDBY );
  }

  if (pmwmDspInfo->hmodMHINT)
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodMHINT, DSP_ACTIVATE_STANDBY );
  }

  if ( (pmwmDspInfo->hmodFAXPMP) && (ulRC != 0x0034) )
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodFAXPMP, DSP_ACTIVATE_STANDBY );
  }

  /* 01/15/97 DR - Silence Detection */
  if ( (pmwmDspInfo->hmodSILDET) && (ulRC != 0x0034) )
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodSILDET, DSP_ACTIVATE_STANDBY );
  }

  if ( (pmwmDspInfo->hmodCLASSx) && (ulRC != 0x0034) )
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodCLASSx, DSP_ACTIVATE_STANDBY );
  }

  if ( (pmwmDspInfo->hmodV80MAC)   && (ulRC != 0x0034) )          //$4
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodV80MAC, DSP_ACTIVATE_STANDBY );
  }

//$7 begin change
  if ((pmwmDspInfo->hmodpc2gpc)  && (ulRC != 0x0034) )
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodpc2gpc, DSP_ACTIVATE_STANDBY);
  }

  if ((pmwmDspInfo->hmodvplay)  && (ulRC != 0x0034) )
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodvplay, DSP_ACTIVATE_STANDBY);
  }

  if ( (pmwmDspInfo->hmodINT0896)  && (ulRC != 0x0034) )
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodINT0896, DSP_ACTIVATE_STANDBY);
  }

  if ( (pmwmDspInfo->hmodgpc2pc)  && (ulRC != 0x0034) )
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodgpc2pc, DSP_ACTIVATE_STANDBY);
  }

  if ( (pmwmDspInfo->hmodvrec)  && (ulRC != 0x0034) )
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodvrec, DSP_ACTIVATE_STANDBY);
  }

  if ( (pmwmDspInfo->hmodINT9608)  && (ulRC != 0x0034) )
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodINT9608, DSP_ACTIVATE_STANDBY);
  }
//$7 end change

  if ( (pmwmDspInfo->hmodV22)  && (ulRC != 0x0034) )
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodV22, DSP_ACTIVATE_STANDBY );
  }

  if ( (pmwmDspInfo->hmodV8)  && (ulRC != 0x0034) )
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodV8, DSP_ACTIVATE_STANDBY );
  }

  if ( (pmwmDspInfo->hmodVDP)  && (ulRC != 0x0034) )
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodVDP, DSP_ACTIVATE_STANDBY );
  }

  if ( (pmwmDspInfo->hmodAsync) && (ulRC != 0x0034) )
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodAsync, DSP_ACTIVATE_STANDBY );
  }

  if ( (pmwmDspInfo->hmodMNP)   && (ulRC != 0x0034) )
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodMNP, DSP_ACTIVATE_STANDBY );
  }

  if ( (pmwmDspInfo->hmodMNP5)  && (ulRC != 0x0034) )
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodMNP5, DSP_ACTIVATE_STANDBY );
  }

  if ( (pmwmDspInfo->hmodV42bis)  && (ulRC != 0x0034) )
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodV42bis, DSP_ACTIVATE_STANDBY );
  }

  if ( (pmwmDspInfo->hmodV42)   && (ulRC != 0x0034) )
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodV42, DSP_ACTIVATE_STANDBY );
  }

  if ( (pmwmDspInfo->hmodRAMMBASE)   && (ulRC != 0x0034) )
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodRAMMBASE, DSP_ACTIVATE_STANDBY );
  }

  if ( (pmwmDspInfo->hmodGSM)   && (ulRC != 0x0034) )
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodGSM, DSP_ACTIVATE_STANDBY );
  }

//  if ( (pmwmDspInfo->hmodINTERP8)   && (ulRC != 0x0034) )
//  {
//    ulRC = dspChangeModuleState(pmwmDspInfo->hmodINTERP8, DSP_ACTIVATE_STANDBY );
//  }

//  if ( (pmwmDspInfo->hmodDECIMA)   && (ulRC != 0x0034) )
//  {
//    ulRC = dspChangeModuleState(pmwmDspInfo->hmodDECIMA, DSP_ACTIVATE_STANDBY );
//  }

  if ( (pmwmDspInfo->hmodSPKVOD)   && (ulRC != 0x0034) )
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodSPKVOD, DSP_ACTIVATE_STANDBY );
  }

//if ( (pmwmDspInfo->hmodCLASS1)   && (ulRC != 0x0034) )
//{
//  ulRC = dspChangeModuleState(pmwmDspInfo->hmodCLASS1, DSP_ACTIVATE_STANDBY );
//}

  if ( (pmwmDspInfo->hmodCL1_HDLC)   && (ulRC != 0x0034) )
  {
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodCL1_HDLC, DSP_ACTIVATE_STANDBY );
  }

if (hTemphipc != NULL)
{
  ulRC = dspDisconnectIPC(hTemphipc);
    if (ulRC)
    {
       mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC);
      return(ulRC);
    }
}


  /***************************************************************************/
  /* Next, unload all of the modem/fax modules...                            */
  /***************************************************************************/

  /***************************************************************************/
  /* MTS 4086                                                                */
  /***************************************************************************/
  if ( (pmwmDspInfo->hsegPercentTT != NULL) && (ulRC != 0x0034) )
  {
    ulRC = dspFreeSegment(pmwmDspInfo->hsegPercentTT);
    if (ulRC)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC);
      return(ulRC);
    }
    pmwmDspInfo->hsegPercentTT = 0;
  }



  if ( (pmwmDspInfo->hmodModem != NULL) && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodModem);
    if (!ulRC)
    {
      pmwmDspInfo->hmodModem = NULL;
    }
  }

  /* Unload caller ID if it's loaded - 03/28/96 MTS6619 */
  mwmCidpInitCid( FALSE, 0l, 0l );
  if(( !( pmwmDspInfo->hmodCALLERID == NULL ) ) && (ulRC != 0x0034) )
  { /* there is a module to uload */
    ulRC = dspFreeModule( pmwmDspInfo->hmodCALLERID );
    pmwmDspInfo->hmodCALLERID = NULL;
    pmwmDspInfo->hmtCALLERID_ASYNC = NULL;
  }

  /* Unload distinctive ring if it's loaded  */
  if(( !( pmwmDspInfo->hmodDRing == NULL ) ) && (ulRC != 0x0034) )
  { /* there is a module to uload */
    ulRC = dspFreeModule( pmwmDspInfo->hmodDRing );
    mwmCidlInitDRing( pmwmDspInfo );
  }

  if ( (pmwmDspInfo->hmodT30 != NULL)  && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodT30);
    if (!ulRC)
    {
      pmwmDspInfo->hmodT30 = NULL;
      mwmParseSetCAPBUFFAddress(0);
      mwmParseSetMYIDAddress(0);
      mwmParseSetMYCIGAddress(0);
    }
  }

  if ( (pmwmDspInfo->hmodMHINT != NULL)  && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodMHINT);
    if (!ulRC)
    {
      pmwmDspInfo->hmodMHINT = NULL;
    }
  }


  if ( (pmwmDspInfo->hmodFAXPMP != NULL)  && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodFAXPMP);
    if (!ulRC)
    {
      pmwmDspInfo->hmodFAXPMP = NULL;
    }
  }

  /* 01/15/97 DR - Silence Detection */
  if ( (pmwmDspInfo->hmodSILDET != NULL)  && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodSILDET);
    if (!ulRC)
    {
      pmwmDspInfo->hmodSILDET = NULL;
    }
  }


  if ( (pmwmDspInfo->hmodCLASSx != NULL)  && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodCLASSx);
    if (!ulRC)
    {
      pmwmDspInfo->hmodCLASSx = NULL;
      pmwmDspInfo->hmtCLASSx = NULL;                              //$4
    }
  }

  if ( (pmwmDspInfo->hmodV80MAC != NULL)  && (ulRC != 0x0034) )   //$4
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodV80MAC);
    if (!ulRC)
    {
      pmwmDspInfo->hmodV80MAC = NULL;
      pmwmDspInfo->hmtV80MAC = NULL;
    }
  }

  if ((pmwmDspInfo->hmodpc2gpc)  && (ulRC != 0x0034) )             //$5
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodpc2gpc);                 //$7
    if (!ulRC)
    {
      pmwmDspInfo->hmodpc2gpc = NULL;
      pmwmDspInfo->htpc2gpc = NULL;
    }
  }

  if ((pmwmDspInfo->hmodvplay)  && (ulRC != 0x0034) )              //$5
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodvplay);                  //$7
    if (!ulRC)
    {
      pmwmDspInfo->hmodvplay = NULL;
      pmwmDspInfo->htvplay = NULL;
    }
  }

  if ( (pmwmDspInfo->hmodINT0896)  && (ulRC != 0x0034) )           //$7
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodINT0896);

    if (!ulRC)
    {
      pmwmDspInfo->hmodINT0896 = 0;
      pmwmDspInfo->hmtINT0896   = 0;
    }
  }

  if ( (pmwmDspInfo->hmodgpc2pc)  && (ulRC != 0x0034) )             //$5
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodgpc2pc);                 //$7
    if (!ulRC)
    {
      pmwmDspInfo->hmodgpc2pc = NULL;
      pmwmDspInfo->htgpc2pc = NULL;
    }
  }

  if ( (pmwmDspInfo->hmodvrec)  && (ulRC != 0x0034) )               //$5
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodvrec);                   //$7
    if (!ulRC)
    {
      pmwmDspInfo->hmodvrec = NULL;
      pmwmDspInfo->htvrec = NULL;
    }
  }

  if ( (pmwmDspInfo->hmodINT9608)  && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodINT9608);
    if (!ulRC)
    {
      pmwmDspInfo->hmodINT9608 = 0;
      pmwmDspInfo->hmtINT9608   = 0;
    }
  }

  if ( (pmwmDspInfo->hmodInterpolator != NULL) && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodInterpolator);
    if (!ulRC)
    {
      pmwmDspInfo->hmodInterpolator = NULL;
    }
  }

  /* MTS 3361 */
  if ( (pmwmDspInfo->hmodCallProg != NULL) && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodCallProg);
    if (!ulRC)
    {
      pmwmDspInfo->hmodCallProg = NULL;
      pmwmDspInfo->hmtCallProg = NULL;
    }
  }


  if ( (pmwmDspInfo->hmodV22 != NULL) && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodV22);
    if (!ulRC)
    {
      pmwmDspInfo->hmodV22 = NULL;
      pmwmDspInfo->hv22Task = NULL;
    }
  }


  if ( (pmwmDspInfo->hmodV8 != NULL) && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodV8);
    if (!ulRC)
    {
      pmwmDspInfo->hmodV8 = NULL;
      pmwmDspInfo->hmtV8  = NULL;
    }
  }


  if ( (pmwmDspInfo->hmodVDP != NULL) && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodVDP);
    if (!ulRC)
    {
      pmwmDspInfo->hmodVDP  = NULL;
      pmwmDspInfo->hvdpTask = NULL;
    }
  }

  if ( (pmwmDspInfo->hmodV34 != NULL) && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodV34);
    if (!ulRC)
    {
      pmwmDspInfo->hmodV34 = NULL;
      pmwmDspInfo->hmtV34  = NULL;
    }
  }


  if ( (pmwmDspInfo->hmodAsync != NULL) && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodAsync);

    if (!ulRC)
    {
      pmwmDspInfo->hmodModem   = NULL;
      pmwmDspInfo->hmctlTask   = NULL;
      pmwmDspInfo->hmtProtSupr = NULL;
    }
  }


  if ( (pmwmDspInfo->hmodFakeUART != NULL) && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodFakeUART);
    if (!ulRC)
    {
      pmwmDspInfo->hmodFakeUART = NULL;
      pmwmDspInfo->hmtFakeUART = NULL;
    }
  }


  if ( (pmwmDspInfo->hmodMNP != NULL)   && (ulRC != 0x0034) )
  {
     ulRC = dspFreeModule(pmwmDspInfo->hmodMNP);

     if (!ulRC)
     {
       pmwmDspInfo->hmodMNP = NULL;
     }
  }

  if ( (pmwmDspInfo->hmodMNP5 != NULL)  && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodMNP5);

    if (!ulRC)
    {
      pmwmDspInfo->hmodMNP = NULL;
    }
  }

  if ( (pmwmDspInfo->hmodV42bis != NULL) && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodV42bis);

    if (!ulRC)
    {
      pmwmDspInfo->hmodV42bis = NULL;
    }
  }

  if ( (pmwmDspInfo->hmodV42 != NULL)   && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodV42);

    if (!ulRC)
    {
      pmwmDspInfo->hmodV42 = NULL;
    }
  }

  if ( (pmwmDspInfo->hmodRAMMBASE != NULL)   && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodRAMMBASE);

    if (!ulRC)
    {
      pmwmDspInfo->hmodRAMMBASE = NULL;
      pmwmDspInfo->hmtRAMMBASE = NULL;
    }
  }

  if ( (pmwmDspInfo->hmodGSM != NULL)   && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodGSM);

    if (!ulRC)
    {
      pmwmDspInfo->hmodGSM = NULL;
      pmwmDspInfo->hmtGSMCOD = NULL;
      pmwmDspInfo->hmtGSMDEC = NULL;
      pmwmDspInfo->hmtMOVE79 = NULL;
      pmwmDspInfo->hmtMOVE97 = NULL;
    }
  }

//  if ( (pmwmDspInfo->hmodINTERP8 != NULL)   && (ulRC != 0x0034) )
//  {
//    ulRC = dspFreeModule(pmwmDspInfo->hmodINTERP8);
//
//    if (!ulRC)
//    {
//      pmwmDspInfo->hmodINTERP8 = NULL;
//      pmwmDspInfo->hmtINTERP8 = NULL;
//    }
//  }

//  if ( (pmwmDspInfo->hmodDECIMA != NULL)   && (ulRC != 0x0034) )
//  {
//    ulRC = dspFreeModule(pmwmDspInfo->hmodDECIMA);
//
//    if (!ulRC)
//    {
//      pmwmDspInfo->hmodDECIMA = NULL;
//      pmwmDspInfo->hmtDECIMA  = NULL;
//    }
//  }

  if ( (pmwmDspInfo->hmodSPKVOD != NULL)   && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodSPKVOD);

    if (!ulRC)
    {
      pmwmDspInfo->hmodSPKVOD = NULL;
      pmwmDspInfo->hmtSPKVOD  = NULL;
    }
  }

//if ( (pmwmDspInfo->hmodCLASS1 != NULL)   && (ulRC != 0x0034) )
//{
//  ulRC = dspFreeModule(pmwmDspInfo->hmodCLASS1);
//
//  if (!ulRC)
//  {
//    pmwmDspInfo->hmodCLASS1 = NULL;
//    pmwmDspInfo->hmtCLASS1  = NULL;
//  }
//}


  if ( (pmwmDspInfo->hmodCL1_HDLC != NULL)   && (ulRC != 0x0034) )
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodCL1_HDLC);

    if (!ulRC)
    {
      pmwmDspInfo->hmodCL1_HDLC = NULL;
      pmwmDspInfo->hmtCL1_HDLC  = NULL;
    }
  }



  if (pmwmDspInfo->hConnect1)
  {
    ulRC = dspMeioDisconnect(pmwmDspInfo->hConnect1,0);
    pmwmDspInfo->hConnect1 = 0;
  }

  if (pmwmDspInfo->hConnect2)
  {
    ulRC = dspMeioDisconnect(pmwmDspInfo->hConnect2,0);
    pmwmDspInfo->hConnect2 = 0;
  }


  /***************************************************************************/
  /* Free all speaker MEIO connections                                       */
  /***************************************************************************/
  if (pmwmDspInfo->hConnect3)
  {
    ulRC = dspMeioDisconnect(pmwmDspInfo->hConnect3,0);
    pmwmDspInfo->hConnect3 = 0;
  }
  if (pmwmDspInfo->hConnect4)
  {
    ulRC = dspMeioDisconnect(pmwmDspInfo->hConnect4,0);
    pmwmDspInfo->hConnect4 = 0;
  }


  /***************************************************************************/
  /* Disconnect the UARTBOOST ITCBs if they are connected.                   */
  /***************************************************************************/
  dspDisconnectITCB(pmwmDspInfo->hmtUART,"UARTITCB",
                    NULL,"U7ITCB");

  dspDisconnectITCB(pmwmDspInfo->hmtUART,"UARTITCB",
                    NULL,"U31ITCB");

  dspDisconnectITCB(pmwmDspInfo->hmtUART,"UARTITCB",
                    NULL,"UNRTITCB");




  /***************************************************************************/
  /* PTR 1711                                                                */
  /* 12/03/93 ... Added so that the driver can update the UART cycles        */
  /* do the disconnect here...                                               */
  /***************************************************************************/
  if (pmwmDspInfo->hConnectUart)
  {
    ulRC = dspMeioDisconnect(pmwmDspInfo->hConnectUart,0);
    pmwmDspInfo->hConnectUart = 0;
  }
/*
#ifdef DEBUG
  {
          char dstr[200];
                        sprintf(dstr,"pmwmDspInfo->hmtUART = %x\n",pmwmDspInfo->hmtUART);
                MessageBox(NULL,dstr,"Mwave Modem",
                          MB_ICONSTOP | MB_OK | MB_SETFOREGROUND);
  }
#endif
*/

  pmwmDspInfo->hmtUART = 0;


   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcUnloadDSPCode exit\n");  }


  return ulRC;
}





/*******************************************************/
/* Function:  mwmSpcfcUnloadModem                      */
/* Unloads COMMSYS and frees all references            */
/*                                                     */
/*******************************************************/

ULONG MWM_ENTRY mwmSpcfcUnloadModem(PMWM_DSPINFO pmwmDspInfo, USHORT CloseApp)
{
   ULONG    ulRC = 0;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcUnloadModem entry\n");  }


   if (!pmwmDspInfo)
     return 0;


  if (CloseApp)
  {
  } /* Close app */


  ulRC = mwmSpcfcUnloadDSPCode(pmwmDspInfo);



  /***************************************************************************/
  /* Free up the MEIO handle                                                 */
  /***************************************************************************/
  if (hMEIO)
  {
    ulRC=dspMeioClose(hMEIO,0);
    hMEIO = 0;
  }



      if (pmwmDspInfo->ulFeaturesToLoad & FEATURE_GETSIZE)
      {
      }


  ulRC = mwmParseCleanupParser();

  if (pmwmDspInfo)
    free(pmwmDspInfo);

  pmwmDspInfo = 0;
  pGlobalDspInfo = 0;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcUnloadModem exit\n");  }


  return(ulRC);
}







ULONG mwmSpcfcGetIPSUsed(PMWM_DSPINFO pmwmDspInfo, ULONG *pulIPSUsed)
{
  ULONG        ulRC;
  USHORT       usSize = 0;
  RDSPINFO     *pDSPInfoBuffer = NULL;
  RDSP_ABILITIES_INFO  *pAbilitiesBuffer = NULL;


   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcGetIPSUsed entry\n");  }


  /***************************************************************************/
  /***************************************************************************/

  ulRC = dspQueryDSPInfo(pmwmDspInfo->hDSP, &usSize, pDSPInfoBuffer);
  if (ulRC != DSP_INS_BUFFER)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC );
    return ulRC;
  }

  /***************************************************************************/
  /* MTS 3525                                                                */
  /* Allocate more than needed...in case someone else is changing the system */
  /* before we get a chance to complete the query.                           */
  /***************************************************************************/
  usSize += 20;

  pDSPInfoBuffer = (RDSPINFO *)calloc( 1, usSize );
  if (!pDSPInfoBuffer)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_OS_ERROR,
                   0);
    return 1;
  }


  ulRC = dspQueryDSPInfo(pmwmDspInfo->hDSP, &usSize, pDSPInfoBuffer);
  if (ulRC != DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC);
    free(pDSPInfoBuffer);
    return ulRC;
  }


  usSize = 0;
  ulRC = dspAbilities(pmwmDspInfo->hDSP,&usSize, pAbilitiesBuffer);
  if (ulRC != DSP_INS_BUFFER)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC );
    free(pDSPInfoBuffer);
    return ulRC;
  }

  pAbilitiesBuffer = (RDSP_ABILITIES_INFO *)calloc( 1, usSize );
  if (!pDSPInfoBuffer)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_OS_ERROR,
                   0);
    free(pDSPInfoBuffer);
    return 1;
  }


  ulRC = dspAbilities(pmwmDspInfo->hDSP,&usSize, pAbilitiesBuffer);
  if (ulRC != DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC);
    free(pAbilitiesBuffer);
    free(pDSPInfoBuffer);
    return ulRC;

  }


  /***************************************************************************/
  /* Put the DSP handle into the shared memory area.                         */
  /***************************************************************************/
  *pulIPSUsed = ((RDSP_ABILITIES_INFO *)pAbilitiesBuffer)->DSPABILITIES_ulIPS -
                ((RDSPINFO *)pDSPInfoBuffer)->DSPINFO_ulIPSLeft;

  /***************************************************************************/
  /* Free the Manager Info buffer...we are finished with it.                 */
  /***************************************************************************/
  free(pAbilitiesBuffer);
  free(pDSPInfoBuffer);

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcGetIPSUsed exit\n");  }

  return 0;
}




ULONG mwmSpcfcSetWTParms(PMWM_DSPINFO pmwmDspInfo)
{
  LONG         lRC        = 0;
  ULONG        ulRC       = 0;
//  ULONG        ulWTTAF1   = 0;
//  ULONG        ulWTTAF2   = 0;
//  ULONG        ulWTTDTADS = 0;
//  ULONG        ulWTTDMODE = 0;
//  ULONG        ulWTTCPM   = 0;
//  ULONG        ulWTTAMD   = 0;
  MWM_WTPARMS wt;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetWTParms entry\n");  }


  /*************************************************************************/
  /* Query the WTTAF1 Variable from the world trade table access library   */
  /*************************************************************************/
  lRC = WtQueryItem(NULL,
                    WT_ALLOWED_FUNC_1,
                    &wt.ulWTTAF1);
  if (lRC < 0)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_WT_ERROR,
                   lRC );
    return (ULONG)lRC;
  }

  /*************************************************************************/
  /* Query the WTTAF2 Variable from the world trade table access library   */
  /*************************************************************************/
  lRC = WtQueryItem(NULL,
                    WT_ALLOWED_FUNC_2,
                    &wt.ulWTTAF2);
  if (lRC < 0)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_WT_ERROR,
                   lRC );
    return (ULONG)lRC;
  }

  /*************************************************************************/
  /* Query the WTTDTADS Variable from the world trade table access library */
  /*************************************************************************/
  lRC = WtQueryItem(NULL,
                    WT_DTMF_ALLOWED_SET,
                    &wt.ulWTTDTADS);
  if (lRC < 0)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_WT_ERROR,
                   lRC );
    return (ULONG)lRC;
  }

  /*************************************************************************/
  /* Query the WTTDMODE Variable from the world trade table access library.      */
  /*************************************************************************/
  lRC = WtQueryItem(NULL,
                    WT_PULSE_DIAL_MODE,
                    &wt.ulWTTDMODE);
  if (lRC < 0)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_WT_ERROR,
                   lRC );
    return (ULONG)lRC;
  }

  /*************************************************************************/
  /* Query the WTTCPM Variable from the world trade table access library.      */
  /*************************************************************************/
  lRC = WtQueryItem(NULL,
                    WT_CALL_PROG_CTRL,
                    &wt.ulWTTCPM);
  if (lRC < 0)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_WT_ERROR,
                   lRC );
    return (ULONG)lRC;
  }

  /*************************************************************************/
  /* MTS 3444  Query the WTTAMD word.                                      */
  /*************************************************************************/
  lRC = WtQueryItem(NULL,
                    WT_AUTO_MODEM_DISCONNECT_TIME,
                    &wt.ulWTTAMD);
  if (lRC < 0)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_WT_ERROR,
                   lRC );
    return (ULONG)lRC;
  }

  /*************************************************************************/
  /* MTS 3444  Query the WTTRNGS word.                                     */
  /*************************************************************************/
  lRC = WtQueryItem(NULL,
                    WT_MIN_RINGS,
                    &wt.ulWTTRNGS);
  if (lRC < 0)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_WT_ERROR,
                   lRC );
    return (ULONG)lRC;
  }


  /*************************************************************************/
  /* MTS 3444  Query the WTTBDWT word.                                     */
  /*************************************************************************/
  lRC = WtQueryItem(NULL,
                    WT_BLIND_DIAL_WAIT_TIME,
                    &wt.ulWTTBDWT);
  if (lRC < 0)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_WT_ERROR,
                   lRC );
    return (ULONG)lRC;
  }


  /*************************************************************************/
  /* MTS 5080                                                              */
  /*           Query the WTTCW word.                                       */
  /*************************************************************************/
  lRC = WtQueryItem(NULL,
                    WT_CARRIER_WAIT_TIME,
                    &wt.ulWTTCW);
  if (lRC < 0)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_WT_ERROR,
                   lRC );
    return (ULONG)lRC;
  }

  /*************************************************************************/
  /* MTS 5080                                                              */
  /*           Query the WTTLCW word.                                      */
  /*************************************************************************/
  lRC = WtQueryItem(NULL,
                    WT_LOSS_OF_CARRIER_WAVE_TIME,
                    &wt.ulWTTLCW);
  if (lRC < 0)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_WT_ERROR,
                   lRC );
    return (ULONG)lRC;
  }

  /*************************************************************************/
  /*                                                                       */
  /*           Query the WTTLCWD word.                                     */
  /*************************************************************************/
  wt.ulWTTCWD = 0;
  lRC = WTT_QueryInfo( NULL, WTBLOCK_WTITCB, WTPARAM_WTTCWD, WTLENGTH_ONEPARAM,
                       (short *)&wt.ulWTTCWD, 0L);
  if (lRC != 0)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_WT_ERROR,
                   lRC );
    return (ULONG)lRC;
  }





  /*************************************************************************/
  /* MTS 5080                                                              */
  /* Query the WTTBDWTX word.                                              */
  /*************************************************************************/
  lRC = WtQueryItem(NULL,
                    WT_BLIND_DIAL_WAIT_TIME_MAX   ,
                    &wt.ulWTTBDWTX);
  if (lRC < 0)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_WT_ERROR,
                   lRC );
    return (ULONG)lRC;
  }


  /*************************************************************************/
  /* MTS 5080                                                              */
  /* Query the WTTCWX word.                                                */
  /*************************************************************************/
  lRC = WtQueryItem(NULL,
                    WT_CARRIER_WAIT_TIME_MAX    ,
                    &wt.ulWTTCWX);
  if (lRC < 0)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_WT_ERROR,
                   lRC );
    return (ULONG)lRC;
  }

  /*************************************************************************/
  /* MTS 5080                                                              */
  /* Query the WTTLCWX word.                                               */
  /*************************************************************************/
  lRC = WtQueryItem(NULL,
                    WT_LOSS_OF_CARRIER_WAVE_MAX   ,
                    &wt.ulWTTLCWX);
  if (lRC < 0)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_WT_ERROR,
                   lRC );
    return (ULONG)lRC;
  }


  /*************************************************************************/
  /* MTS 5080                                                              */
  /* Query the WTTDTSD word.                                               */
  /*************************************************************************/
  lRC = WtQueryItem(NULL,
                    WT_DTMF_SIGNAL_DURATION   ,
                    &wt.ulWTTDTSD);
  if (lRC < 0)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_WT_ERROR,
                   lRC );
    return (ULONG)lRC;
  }

  /*************************************************************************/
  /*                                                                       */
  /*           Query the WTT34FBR word.                                    */
  /*************************************************************************/
  wt.ulWTT34FBR = 0;
  lRC = WTT_QueryInfo( NULL, WTBLOCK_WTITCB, WTPARAM_WTT34FBR, WTLENGTH_ONEPARAM,
                       (short *)&wt.ulWTT34FBR, 0L);
  if (lRC != 0)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_WT_ERROR,
                   lRC );
    return (ULONG)lRC;
  }



  ulRC = mwmParseSetWTParms(&wt);
  if (ulRC)
    return ulRC;


   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetWTParms exit\n");  }


  return 0;
}


ULONG mwmSpcfcSetUpCP_Parms(PMWM_DSPINFO pmwmDspInfo, HMTASK hCpTask)
{
   HSEG       hCpSeg=0;
   long              BlockSize = 0;
   unsigned short far *pPackedBlock=0;
   unsigned long  ulWtAddr;
   unsigned long  ulCpAddr;
//   unsigned long  ulLoadedAddr;
   unsigned short usWtSegment;
   unsigned long  ulRC;
   unsigned short DataValid = 0;
//   unsigned short Loaded  = 1;
   unsigned short TonesValid = 0x1f;  // set of valid tones

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetUpCP_Parms entry\n");  }


   //
   //  get the data block from WT Access library
   //
   BlockSize = WtQueryField(NULL,
                            WT_FIELD_PSTN_CALL_PROGRESS,
                            &pPackedBlock);
   if (BlockSize < 0)
   {
    //  error("Error: bad return from WT Access LIBRARY. RC = %d.\n",BlockSize);
     mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                    MWM_WT_ERROR,
                    BlockSize);
     return BlockSize;
   }


   ulRC = dspLabelToAddress(hCpTask,"_CP_PTR",&ulCpAddr);
   if (ulRC)
   {
   //   error("Error: can not get CP_PTR address. RC=%lX.\n",ulRC);
     mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                    MWM_DSP_ERROR,
                    ulRC);
     return ulRC;
   }


   //
   //  OK, now get label address for WT_PTR
   //
   ulRC = dspLabelToAddress(hCpTask,"_WT_PTR",&ulWtAddr);
   if (ulRC)
   {
   //   error("Error: can not get WT_PTR address. RC=%lX.\n",ulRC);
     mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                    MWM_DSP_ERROR,
                    ulRC);
     return ulRC;
   }

   ulRC = dspMemTransfer(pmwmDspInfo->hDSP,ulWtAddr+2,&DataValid,1,DSP_MEMXFER_DATA_WRITE);
   if (ulRC)
   {
     mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                    MWM_DSP_ERROR,
                    ulRC);
     return ulRC;
   }

   //
   // now allocate segment for parameters
   //
   if (hCpSeg)
   {
      ulRC = dspFreeSegment(hCpSeg);      // free the previous segment
      if (ulRC)
      {
        mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                       MWM_DSP_ERROR,
                       ulRC);
        return ulRC;
      }
   } /* endif */

   ulRC =  mwmCntndAllocSegment(pmwmDspInfo,
                                hCpTask,
                                "WT_PARMS",
                                BlockSize,
                                1,
                                ulWtAddr,
                                &hCpSeg );
// ulRC = dspAllocateSegment(hCpTask,
//                           "WT_PARMS",
//                           BlockSize,
//                           1,
//                           SEG_DATA,
//                           ulWtAddr,
//                           &hCpSeg);
   if (ulRC)
   {
     mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                    MWM_DSP_ERROR,
                    ulRC);
     return ulRC;
   }

   //
   // get address of segment
   //
   ulRC = dspMemTransfer(pmwmDspInfo->hDSP,ulWtAddr,&usWtSegment,1,DSP_MEMXFER_DATA_READ);
   if (ulRC)
   {
     mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                    MWM_DSP_ERROR,
                    ulRC);
     return ulRC;
   }

   //
   //  and transfer block to segment
   //
   ulRC = dspMemTransfer(pmwmDspInfo->hDSP,
                         usWtSegment,
                         pPackedBlock,
                         (unsigned short)BlockSize,
                         DSP_MEMXFER_DATA_WRITE);
   if (ulRC)
   {
     mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                    MWM_DSP_ERROR,
                    ulRC);
     return ulRC;
   }

   WtFree(&pPackedBlock);
   if (ulRC)
   {
     mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                    MWM_DSP_ERROR,
                    ulRC);
     return ulRC;
   }

   //
   //  finally, tell CallProg that data is valid
   //

   ulRC = dspMemTransfer(pmwmDspInfo->hDSP,ulCpAddr+2,&TonesValid,1,DSP_MEMXFER_DATA_WRITE);
   if (ulRC)
   {
     mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                    MWM_DSP_ERROR,
                    ulRC);
     return ulRC;
   }


   DataValid = 1;

   ulRC = dspMemTransfer(pmwmDspInfo->hDSP,ulWtAddr+2,&DataValid,1,DSP_MEMXFER_DATA_WRITE);
   if (ulRC)
   {
     mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                    MWM_DSP_ERROR,
                    ulRC);
     return ulRC;
   }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetUpCP_Parms exit\n");  }

   return 0;
}


ULONG mwmSpcfcSetDPCPF(PMWM_DSPINFO pmwmDspInfo, HMTASK hmtDatapump, char *achLabel)
{
  ULONG  ulRC = 0;
  ULONG  ulAddress = 0;
  ULONG  ulCurrentCPF = 0;
  USHORT usDesiredCPF = 0;
  LONG lCPFDelta = 0;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetDPCPF entry\n");  }


  ulRC = mwmSpcfcGetTaskCPF(pmwmDspInfo, hmtDatapump, &ulCurrentCPF,szThisFile,__LINE__);
  if (ulRC)
    return ulRC;


  ulRC = dspLabelToAddress(hmtDatapump,
                           achLabel,
                           &ulAddress);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

  ulRC = dspMemTransfer(pmwmDspInfo->hDSP,
                        ulAddress,
                        &usDesiredCPF, 1,
                        DSP_MEMXFER_DATA_READ);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }


  lCPFDelta = (LONG) ((ULONG)usDesiredCPF - ulCurrentCPF);

  /*************************************************************************/
  /* Attempt to Change CPF...                                              */
  /*************************************************************************/
//  ulRC=dspChangeCPF(hmtDatapump,lCPFDelta);
  ulRC=mwmCntndChangeCPF(pmwmDspInfo,hmtDatapump,lCPFDelta, MWM_MUSTHAVE); /* MTS RADD */
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetDPCPF exit\n");  }

  return 0;
}

USHORT MWM_ENTRY mwmSpcfcCheckSpeedParm(PMWM_DSPINFO pmwmDspInfo, USHORT sRequestedSpeed)
{
  SHORT    sPotentialModemStartSpeed = MW_START2400;

   { MW_SYSLOG_3(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcCheckSpeedParm entry sRequestedSpeed %x ulInstalledFeatures %lx\n",sRequestedSpeed,pmwmDspInfo->ulInstalledFeatures);  }

  /***********************************************************************/
  /* Check the profile to see if the requested modem speed is installed. */
  /***********************************************************************/
  if (( sRequestedSpeed == MW_BADPARAMETER) ||
      (( sRequestedSpeed == MW_START2400) ))
  {
      sPotentialModemStartSpeed = MW_START2400;
  }
  if (( sRequestedSpeed == MW_BADPARAMETER) ||
     (( sRequestedSpeed == MW_START9600)))
  {
    if ( (pmwmDspInfo->ulInstalledFeatures & FEATURE_V32 ) )
      sPotentialModemStartSpeed = MW_START9600;
  }
  if (( sRequestedSpeed == MW_BADPARAMETER) ||
     (( sRequestedSpeed == MW_START14400) ))
  {
    if ( (pmwmDspInfo->ulInstalledFeatures & FEATURE_V32BIS ) )
      sPotentialModemStartSpeed = MW_START14400;
  }
  if (( sRequestedSpeed == MW_BADPARAMETER) ||
     (( sRequestedSpeed == MW_START19200) ))
  {
    if ( (pmwmDspInfo->ulInstalledFeatures & FEATURE_V32TER ) )
      sPotentialModemStartSpeed = MW_START19200;
  }

  if (( sRequestedSpeed == MW_BADPARAMETER) ||
     (( sRequestedSpeed == MW_START28800) ))
  {
    if ( (pmwmDspInfo->ulInstalledFeatures & FEATURE_V34 ) )
      sPotentialModemStartSpeed = MW_START28800;
  }

  if (( sRequestedSpeed == MW_BADPARAMETER) ||
     (( sRequestedSpeed == MW_START33600) ))
  {
    if ( (pmwmDspInfo->ulInstalledFeatures & FEATURE_V34EM ) ) {
      sPotentialModemStartSpeed = MW_START33600;
    } else if ( (pmwmDspInfo->ulInstalledFeatures & FEATURE_V34 ) ) {
      sPotentialModemStartSpeed = MW_START28800;
    } /* endif */
  }

  if (( sRequestedSpeed == MW_BADPARAMETER) ||
     (( sRequestedSpeed == MW_START64000) ))
  {
    if ( (pmwmDspInfo->ulInstalledFeatures & FEATURE_PCM ) )
      sPotentialModemStartSpeed = MW_START64000;
  }

  if ( sRequestedSpeed == MW_BADPARAMETER)
  {
     sRequestedSpeed = sPotentialModemStartSpeed;
  }
  else if ( sRequestedSpeed != sPotentialModemStartSpeed)
  {
    sRequestedSpeed = MW_BADPARAMETER;
  }

   { MW_SYSLOG_2(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcCheckSpeedParm exit sRequestedSpeed %x\n",sRequestedSpeed);  }

  return (sRequestedSpeed);
}

ULONG MWM_ENTRY mwmSpcfcReleaseLine(PMWM_DSPINFO pmwmDspInfo )
{
   ULONG ulRC=0;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcReleaseLine entry\n");  }


   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcReleaseLine exit\n");  }



   return ulRC;
}




ULONG mwmSpcfcSetMaxS28ComPortSpeed(PMWM_DSPINFO pmwmDspInfo)
{
  ULONG  ulRC = 0;
  ULONG  ulAddress = 0;
  USHORT usTempValue;
  REGISTERS Registers;
  ULONG  ulComPortBPS = 0;
  ULONG  ulS28BPS = 0;
  ULONG  ulComboBPS = 0;
  USHORT usComS28Min = 0;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetMaxS28ComPortSpeed entry\n");  }


  /***************************************************************************/
  /* Get Address of UART registers...                                        */
  /***************************************************************************/
  ulRC = dspLabelToAddress(pmwmDspInfo->hmctlTask, "UARTREGS", &ulAddress);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }

  /***************************************************************************/
  /* Read UART_DL location.                                                  */
  /***************************************************************************/
  ulRC = dspMemTransfer(pmwmDspInfo->hDSP, ulAddress + 4,
                        &usTempValue, 1,
                        DSP_MEMXFER_DATA_READ);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }

  ulComPortBPS = 115200 / usTempValue;


  /***************************************************************************/
  /* Query the S Register settings.                                          */
  /***************************************************************************/
  ulRC = mwmParseQueryModemRegisters(&Registers);
  if (ulRC)
    return (ulRC);

  /***************************************************************************/
  /* Determine the MAX bit rate from S28                                     */
  /***************************************************************************/
  switch(Registers.S[28])
  {
    case 1:
      ulS28BPS = 50;
      break;
    case 2:
      ulS28BPS = 75;
      break;
    case 3:
      ulS28BPS = 110;
      break;
    case 4:
      ulS28BPS = 135;
      break;
    case 5:
      ulS28BPS = 150;
      break;
    case 6:
    case 7:
      ulS28BPS = 300;
      break;
    case 8:
      ulS28BPS = 1200;
      break;
    case 9:
      ulS28BPS = 2400;
      break;
    case 10:
      ulS28BPS = 4800;
      break;
    case 11:
      ulS28BPS = 7200;
      break;
    case 12:
      ulS28BPS = 9600;
      break;
    case 13:
      ulS28BPS = 12000;
      break;
    case 14:
      ulS28BPS = 14400;
      break;
    case 15:
      ulS28BPS = 16800;
      break;
    case 16:
      ulS28BPS = 19200;
      break;
    case 17:
      ulS28BPS = 21600;
      break;
    case 18:
      ulS28BPS = 24000;
      break;
    case 19:
      ulS28BPS = 26400;
      break;
    case 20:
      ulS28BPS = 28800;
      break;
    case 21:
      ulS28BPS = 31200;
      break;
    case 22:
      ulS28BPS = 33600;
      break;
    case 23:
      ulS28BPS = 29333;
      break;
    case 24:
      ulS28BPS = 30000;
      break;
    case 25:
      ulS28BPS = 30666;
      break;
    case 26:
      ulS28BPS = 32000;
      break;
    case 27:
      ulS28BPS = 33333;
      break;
    case 28:
      ulS28BPS = 34000;
      break;
    case 29:
      ulS28BPS = 34666;
      break;
    case 30:
      ulS28BPS = 36000;
      break;
    case 31:
      ulS28BPS = 37333;
      break;
    case 32:
      ulS28BPS = 38000;
      break;
    case 33:
      ulS28BPS = 38666;
      break;
    case 34:
      ulS28BPS = 40000;
      break;
    case 35:
      ulS28BPS = 41333;
      break;
    case 36:
      ulS28BPS = 42000;
      break;
    case 37:
      ulS28BPS = 42666;
      break;
    case 38:
      ulS28BPS = 44000;
      break;
    case 39:
      ulS28BPS = 45333;
      break;
    case 40:
      ulS28BPS = 46000;
      break;
    case 41:
      ulS28BPS = 46666;
      break;
    case 42:
      ulS28BPS = 48000;
      break;
    case 43:
      ulS28BPS = 49333;
      break;
    case 44:
      ulS28BPS = 50000;
      break;
    case 45:
      ulS28BPS = 50666;
      break;
    case 46:
      ulS28BPS = 52000;
      break;
    case 47:
      ulS28BPS = 53333;
      break;
    case 48:
      ulS28BPS = 54000;
      break;
    case 49:
      ulS28BPS = 54666;
      break;
    case 50:
      ulS28BPS = 56000;
      break;
  }


  /***************************************************************************/
  /* Determine the MAX Bit Rate from the combination of S28 and Com Port.    */
  /* S28=0 means rely only on the com port setting.                          */
  /***************************************************************************/
  if (ulS28BPS)
  {
    ulComboBPS = min(ulComPortBPS,ulS28BPS);
  }
  else
    ulComboBPS = ulComPortBPS;

  if (ulComboBPS < 1200)
  {
    if (ulComboBPS >= 300)
      usComS28Min = 6;
    else if (ulComboBPS >= 150)
      usComS28Min = 5;
    else if (ulComboBPS > 134)
      usComS28Min = 4;
    else if (ulComboBPS >= 110)
      usComS28Min = 3;
    else if (ulComboBPS >= 75)
      usComS28Min = 2;
    else if (ulComboBPS >= 50)
      usComS28Min = 1;
  }
  else
  {
    usComS28Min = (USHORT)ulComboBPS / 50;

    /************************************************
    ** If there's any remainder from the division
    ** (i.e. the value is not a multiple of 50),
    ** then round up to the next highest number.
    *************************************************/
    if (ulComboBPS % 50) {
      usComS28Min++;
    } /* endif */
  }

  DPF("ComPort: %lu  S28: %d  Max MCTLRATE: %d",
       ulComPortBPS,
       Registers.S[28],
       usComS28Min);


  /***************************************************************************/
  /* Get Address of COMS28MIN registers...                                   */
  /***************************************************************************/
  ulRC = dspLabelToAddress( pmwmDspInfo->hmctlTask, "COMS28MIN",
                            &pmwmDspInfo->ulDSPAddrComS28Min);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }

  /***************************************************************************/
  /* Write Value...                                                          */
  /***************************************************************************/
  ulRC = dspMemTransfer(pmwmDspInfo->hDSP, pmwmDspInfo->ulDSPAddrComS28Min,
                        &usComS28Min, 1,
                        DSP_MEMXFER_DATA_WRITE);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetMaxS28ComPortSpeed exit\n");  }


  return 0;
}

/*************************************************************************
** Function: mwmSpcfcIsV34EMConnectionPossible
**
** Purpose: Determines if the conditions are present for a possible V.34
**          Enhanced Mode connection (>28.8).
**
** Input: PMWM_DSPINFO pmwmDspInfo
**
** Returns: ulRC - return code from DSP calls
**************************************************************************/
ULONG mwmSpcfcIsV34EMConnectionPossible(PMWM_DSPINFO pmwmDspInfo)
{

  REGISTERS Registers;
  ULONG     ulRC = 0;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcIsV34EMConnectionPossible entry\n");  }


  /******************************************************************
  ** Initialize the flag.
  *******************************************************************/
  pmwmDspInfo->bV34EMConnectionPossible = FALSE;

  /******************************************************************
  ** The user must have selected 33.6 from the Modem GUI and
  ** V34EM=1 must be set in the INI file.
  *******************************************************************/
  if (pmwmDspInfo->ulInstalledFeatures & FEATURE_V34EM) {

    if (pmwmDspInfo->ulFeaturesToLoad & FEATURE_V34EM) {

      /******************************************************************
      ** Query the S Register settings.
      *******************************************************************/
      ulRC = mwmParseQueryModemRegisters(&Registers);
      if (ulRC)
        return (ulRC);

      /******************************************************************
      ** S52 Bit 2 must be 0 to allow > 28.8 connections.
      *******************************************************************/
      if (!(Registers.SV34[3] & 0x0004)) {

        /**************************************************************
        ** The COM port must be set to handle > 28.8 connections.
        ** MTS 7515 - Do not check the COM port speed. V.34 does
        ** not rely on it for enabling faster baud rates.
        ***************************************************************/
//        ulRC = dspMemTransfer( pmwmDspInfo->hDSP,
//                               pmwmDspInfo->ulDSPAddrComS28Min,
//                               &usComS28Min, 1,
//                               DSP_MEMXFER_DATA_READ);
//        if (ulRC)
//          return (ulRC);
//
//        if (usComS28Min > 576) {
          pmwmDspInfo->bV34EMConnectionPossible = TRUE;
          DPF("Conditions are all there for a V.34 Enhanced Mode connection.");
//        } /* endif */
      } /* endif */
    } /* endif */
  } /* endif */

   { MW_SYSLOG_2(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcIsV34EMConnectionPossible exit ulRC %lx\n",ulRC);  }


  return(ulRC);

}

/*************************************************************************
** Function: mwmSpcfcIsPCMConnectionPossible
**
** Purpose: Determines if the conditions are present for a possible
**          PCM connection (>33.6).
**
** Input: PMWM_DSPINFO pmwmDspInfo
**
** Returns: ulRC - return code from DSP calls
**************************************************************************/
ULONG mwmSpcfcIsPCMConnectionPossible(PMWM_DSPINFO pmwmDspInfo)
{

  REGISTERS Registers;
  ULONG     ulRC = 0;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcIsPCMConnectionPossible entry\n");  }


  /******************************************************************
  ** Initialize the flag.
  *******************************************************************/
  pmwmDspInfo->bPCMConnectionPossible = FALSE;

  /******************************************************************
  ** The user must have selected 33.6 from the Modem GUI and
  ** V34EM=1 must be set in the INI file.
  *******************************************************************/
  if (pmwmDspInfo->ulInstalledFeatures & FEATURE_PCM) {

    if (pmwmDspInfo->ulFeaturesToLoad & FEATURE_PCM) {

      /******************************************************************
      ** Query the S Register settings.
      *******************************************************************/
      ulRC = mwmParseQueryModemRegisters(&Registers);
      if (ulRC)
        return (ulRC);

      /******************************************************************
      ** S52 Bit 4 must be 0 to allow x2 connections.
      ** S52 Bit 5 must be 0 to allow V.90 connections.
      *******************************************************************/
      if ((Registers.SV34[3] & 0x0030) != 0x0030 ) {

        /**************************************************************
        ** The COM port must be set to handle > 28.8 connections.
        ** MTS 7515 - Do not check the COM port speed. V.34 does
        ** not rely on it for enabling faster baud rates.
        ***************************************************************/
//        ulRC = dspMemTransfer( pmwmDspInfo->hDSP,
//                               pmwmDspInfo->ulDSPAddrComS28Min,
//                               &usComS28Min, 1,
//                               DSP_MEMXFER_DATA_READ);
//        if (ulRC)
//          return (ulRC);
//
//        if (usComS28Min > 576) {
          pmwmDspInfo->bPCMConnectionPossible = TRUE;
          DPF("Conditions are all there for a PCM connection.");
//        } /* endif */
      } /* endif */
    } /* endif */
  } /* endif */

   { MW_SYSLOG_2(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcIsPCMConnectionPossible exit ulRC %lx\n",ulRC);  }


  return(ulRC);

}

/*************************************************************************
** Function: mwmSpcfcAccessDSPData
**
** Purpose: Reads or writes from the requested DSP Address
**
** Input:
**
** Returns: ulRC - return code from DSP calls
**************************************************************************/
ULONG mwmSpcfcAccessDSPData( HDSP hDSP, HMTASK hTask, char *pszLabel,
                             USHORT usReadWriteFlag, USHORT usNumWords,
                             USHORT *pBuffer, char *pszSourceFilename,
                             USHORT usLine)
{

  ULONG ulRC = 0;
  ULONG ulAddress;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcAccessDSPData entry\n");  }


  ulRC = dspLabelToAddress( hTask, pszLabel, &ulAddress );
  if (ulRC != DSP_NOERROR) {
    mwmHandleErrorWithSubcode( pInstanceInfo, pszSourceFilename, usLine,
                               MWM_DSP_ERROR, ulRC, MWM_SUBCODE_1);
    return(ulRC);
  } /* endif */

  ulRC = dspMemTransfer( hDSP, ulAddress, pBuffer, usNumWords, usReadWriteFlag );
  if (ulRC != DSP_NOERROR) {
    mwmHandleErrorWithSubcode( pInstanceInfo, pszSourceFilename, usLine,
                               MWM_DSP_ERROR, ulRC, MWM_SUBCODE_2);
    return(ulRC);
  } /* endif */

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcAccessDSPData exit\n");  }


  return 0;


}

/********************************************************************
** Used to set DRVRMSG after a country change during IPC 20, 30, 1030
********************************************************************/
ULONG MWM_ENTRY mwmSpcfcSetDRVRMSG(PMWM_DSPINFO pmwmDspInfo)
{
  ULONG ulRC;
  ULONG ulDRVRMSG;

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetDRVRMSG entry\n");  }


  /***************************************************************************/
  /* 04/10/99 Get Address of DRVRMSG                                         */
  /***************************************************************************/
  ulRC = dspLabelToAddress(pmwmDspInfo->hmctlTask, "DRVRMSG", &ulDRVRMSG);
  if (ulRC)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return (ulRC);
  }

  ulRC = dspMemTransfer(pmwmDspInfo->hDSP, ulDRVRMSG,
                        &pmwmDspInfo->usWTChangeDRVRSTATValue,
                        1,
                        DSP_MEMXFER_DATA_WRITE);

  if (ulRC != DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return ulRC;
  }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcSetDRVRMSG exit\n");  }

  return 0;
}

/*****************************************************************************
** Function: mwmSpcfcActivateDebugWindow
**
** Purpose: Called by the main window when the user starts the debug output
**          window. Turns on the flag indicating that there is a debug window
**          present.
******************************************************************************/
void MWM_ENTRY mwmSpcfcActivateDebugWindow( BOOL bActivateFlag )
{

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcActivateDebugWindow entry\n");  }

    usDebugWindow = bActivateFlag;
    mwmParseActivateDebugWindow(bActivateFlag);

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcActivateDebugWindow exit\n");  }

}

/*****************************************************************************
** Function: mwmSpcfcActivateTimeStamp
**
** Purpose: Called by the debug window when the user selects the Show
**          Timestamp option under settings in the debug window.
******************************************************************************/
void MWM_ENTRY mwmSpcfcActivateTimeStamp( BOOL bTimeStampFlag )
{

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcActivateTimeStamp entry\n");  }

  usShowTimeStamp = bTimeStampFlag;
  mwmParseActivateTimeStamp(bTimeStampFlag);

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmSpcfcActivateTimeStamp exit\n");  }

}

struct mw_msg_elem {
  ULONG		uMsg;
  ULONG        	lParam;
  ULONG 	wParam;
  ULONG         inUse;
};

#define MAX_MESSAGES 16

struct mw_msg_elem      mwMsgList[MAX_MESSAGES];
ULONG 	                mwProducer=0,mwConsumer=0,mwCount=0;
pthread_mutex_t		mutex;
pthread_mutexattr_t	mutex_attr;
pthread_condattr_t	cond_attr;
pthread_cond_t		cond;

void MWM_ENTRY mwmHandleMessage(PMWM_DSPINFO pmwmDspInfo) {
  ULONG i;
  int rc;
  
  rc=pthread_mutex_init(&mutex, 0 /* &mutex_attr */);
   { MW_SYSLOG_2(TRACE_MWMLW32,"mwmspcfc::mwmHandleMessages pthread_mutex_init rc %x\n",rc);  }

  rc=pthread_cond_init(&cond, 0 /* &cond_attr */);
   { MW_SYSLOG_2(TRACE_MWMLW32,"mwmspcfc::mwmHandleMessages pthread_cond_init rc %x\n",rc);   }

  for(i=0;i<MAX_MESSAGES;i++) {
    mwMsgList[i].inUse=0;
  }


  while (1) {
    rc=pthread_mutex_lock(&mutex);
    //MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmHandleMessages pthread_mutex_lock rc %x\n",rc);    
    //MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmHandleMessages tid priority %x\n",getpriority(PRIO_PROCESS,0));
    
    if (!mwCount) {
      rc=pthread_cond_wait(&cond, &mutex);
      //MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmHandleMessages pthread_cond_wait rc %x\n",rc);
      //MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmHandleMessages awake mCount %x mConsumer %x mProducer %x\n",mwCount,mwConsumer,mwProducer);
       { MW_SYSLOG_5(TRACE_MWMLW32,"mwmspcfc::mwmHandleMessages mwMsgList[mwConsumer] uMsg %lx wParam %lx lParam %lx inUse %lx\n",
	 mwMsgList[mwConsumer].uMsg, mwMsgList[mwConsumer].wParam, mwMsgList[mwConsumer].lParam, mwMsgList[mwConsumer].inUse);  }
    } else {
       { MW_SYSLOG_2(TRACE_MWMLW32,"mwmspcfc::mwmHandleMessages, mwCount %lx > 0, skipping wait\n",mwCount);  }
    }

    rc=pthread_mutex_unlock(&mutex);
    //MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmHandleMessages pthread_mutex_unlock rc %x\n",rc);
    
    switch(mwMsgList[mwConsumer].uMsg) {
      case WM_MWM_MODEM_IPC:
	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmHandleMessages, handing WM_MWM_MODEM_IPC case\n");  }
	if (mwmIPC( pmwmDspInfo, mwMsgList[mwConsumer].wParam, mwMsgList[mwConsumer].lParam  ) )  {
	   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmHandleMessages, mwmIPC returns TRUE\n");  }
	} 
	break;
	
      default:
	 { MW_SYSLOG_2(TRACE_MWMLW32,"mwmspcfc::mwmHandleMessages default case, add code to handle uMsg %lx!!!\n",mwMsgList[mwConsumer].uMsg);  }
	break;
    }
    mwMsgList[mwConsumer].inUse=0; // handled
    mwConsumer++; mwCount--;
    if (mwConsumer == MAX_MESSAGES) mwConsumer=0;

  }
  rc=pthread_mutex_unlock(&mutex);
  //MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmHandleMessages pthread_mutex_unlock rc %x\n",rc);

  pthread_mutex_destroy(&mutex);
  pthread_cond_destroy(&cond);
}


void MWM_ENTRY mwmPostMessage(UINT uMsg, LONG wParam, LONG lParam) {
  int rc;
  
  { MW_SYSLOG_4(TRACE_MWMLW32,"mwmspcfc::mwmPostMessage entry uMsg %x wParam %lx lParam %lx\n",uMsg,wParam,lParam);   }
  // { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmPostMessage tid priority %x\n",getpriority(PRIO_PROCESS,0));  }
  rc=pthread_mutex_lock(&mutex);
  //MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmPostMessage pthread_mutex_lock rc %x\n",rc);
  if (mwCount < MAX_MESSAGES) {
    mwCount++;
    mwMsgList[mwProducer].uMsg=uMsg;
    mwMsgList[mwProducer].wParam=wParam;
    mwMsgList[mwProducer].lParam=lParam;
    mwMsgList[mwProducer].inUse=1;
    mwProducer++;
    if (mwProducer == MAX_MESSAGES) mwProducer=0;
    rc=pthread_cond_signal(&cond);
    //MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmPostMessage, rc %x from pthread_cond_signal\n",rc);
  } else {
     { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspfc::mwmPostMessage ERROR, message queue overflow\n");  }
  }
  //MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmPostMessage releasing mutex\n");     
  rc=pthread_mutex_unlock(&mutex); 
   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmspcfc::mwmPostMessage exit\n");   }

}
