/*************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 gnunode.cpp  -  Definition of a node in the gnutella network

 the original version of this file was taken from Gnucleus (http://gnucleus.sourceforge.net)

    begin                : Tue May 29 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/ioctl.h>
#include <fcntl.h>

#include <ctype.h>

#include <zlib.h>

#include "mutella.h"
#include "structures.h"

#include "asyncsocket.h"
#include "packet.h"

#include "rcobject.h"
#include "event.h"
#include "messages.h"
#include "gnuupload.h"
#include "gnushare.h"
#include "gnucache.h"
#include "gnuhash.h"
#include "gnudirector.h"
#include "gnunode.h"
#include "conversions.h"
#include "common.h"
#include "property.h"
#include "preferences.h"
#include "gnuwordhash.h"

MGnuNode::MGnuNode(MGnuDirector* pComm, IP ipHost, UINT Port, bool bInbound, bool bForceV4)
{
    m_pComm  = pComm;
    m_pPrefs = m_pComm->GetPrefs();
    m_pHostCatcher  = m_pComm->GetHostCatcher();
    m_pHostStore    = m_pComm->GetHostStore();
    m_pUltraCatcher = m_pComm->GetUltraCatcher();

    m_ExtraLength  = 0;

    m_dwID = 0;

    // Node Info
    m_nStatus           = SOCK_CONNECTING;
    m_nSecsTrying       = 0;
    m_nSecsRefresh      = 0;
    m_nSecsDead         = 0;
    m_ClosedSecs        = 0;
    m_IntervalPing      = 0;
    m_nSecsAlive        = 0;
    m_nOrigDistance     = 0;
    m_NextReSearchWait  = 0;

    m_ipHost    = ipHost;
    m_sHost  = Ip2Str(ipHost);

    m_nPeerMode          = CM_NORMAL;
    m_nPort              = Port;
    m_bInbound           = bInbound;

    m_NodeFileCount     = 0;
    m_NodeLeafMax       = 0;

    m_DowngradeRequest  = false;
    m_PatchUpdateNeeded = false;

    m_UltraPongSent = false;

    //if(m_pPrefs->m_LanMode)
    //      m_HostName = m_pCore->ResolveIP( StrtoIP(m_ipHost));

    // Ultrapeers
    m_TableInfinity = 2;
    m_TableLength   = 64 * 1024;

    m_PatchTable    = NULL;
    m_TableNextPos  = 0;

    m_CompressedTable = NULL;
    m_CompressedSize  = 0;

    m_CurrentSeq      = 1;


    // Regular ping pong
    for(int i = 0; i < MAX_TTL_SELF; i++)
    {
        m_dwFriends[i]      = 0;
        m_dwLibrary[i]      = 0;
        m_dwFiles[i]        = 0;
    }

    m_dwFriendsTotal = 0;
    m_dwSharingTotal = 0;
    m_llLibraryTotal = 0;
    m_dwFilesTotal   = 0;

    // Packet priority
    for(int i = 0; i < 6; i++)
        m_PacketListLength[i] = 0;
    m_BackBuffLength = 0;
    m_pBackBuff = (BYTE*) malloc(PACKET_BUFF);
    m_pBuff = (BYTE*) malloc(PACKET_BUFF);

    // Packet Stats
    for(int i = 0; i < PACKETCACHE_SIZE; i++)
        for(int j = 0; j < 2; j++)
            m_StatPackets[i][j] = 0;

    m_StatPos       = 0;
    m_StatElements  = 0;
    m_nEfficiency   = 0;

    for(int i = 0; i < 2; i++)
    {
        m_StatPings[i]      = 0;
        m_StatPongs[i]      = 0;
        m_StatQueries[i]    = 0;
        m_StatQueryHits[i]  = 0;
        m_StatPushes[i]     = 0;
    }


	// Statistics
	for(int i = 0; i < NR_LAST; ++i)
	{
		m_llStatisticsTotal   [i] = 0;
		m_dStatisticsRate     [i] = 0.0;
		m_nStatisticsSec      [i] = 0;
		m_nStatisticsHistTotal[i] = 0;
		for(int j = 0; j < 180; j++)
			m_nStatisticsHist[i][j] = 0;
	}

    m_nSecNum = 0;

    //m_pComm->NodeUpdate(m_NodeID);

    m_dwMaxInBytesPerSec = 0;
    
    m_nSecCounter = 0;
	m_nUptime = xtime();
	m_nPeerUptime = m_nUptime;
	m_dwVersion = 4;

	m_nSendQueueSize = 0;
	m_nDroppedPackets = 0;
	m_nAvgDroppedPackets = 0;
	m_nSecDroppedPackets = 0;
	m_nBadPackets = 0;
	m_nRoutingErrors = 0;
	m_nLoopBackErrors = 0;
	m_nDuplicatePackets = 0;

	m_bReceiveBlocked = false;
}

MGnuNode::~MGnuNode()
{
    m_nStatus = SOCK_CLOSED;

    //m_pComm->NodeUpdate(m_NodeID);

    //m_pComm->m_NodeIDMap.erase( m_pComm->m_NodeIDMap.find(m_NodeID) );


    // Flush receive buffer
    //BYTE pBuff[4096];
    //while(Receive(pBuff, 4096) > 0);

    if(m_hSocket != INVALID_SOCKET)
        AsyncSelect(0);

	// Add to cache if we are connected
	if( m_nSecsAlive > 30 )
	{
		// the host has proven its gnutella abilities
		/*if (!m_bInbound || m_pUltraCatcher->IsHalfEmpty())
			m_pUltraCatcher->Add(m_ipHost, m_nPort);
		else*/ if (!m_bInbound || m_pHostStore->IsHalfEmpty())
			m_pHostStore->Add(m_ipHost, m_nPort);
	}

    // Delete query routing tables
    if(m_TableLength)
        m_pComm->GetWordTable()->ResetTable(m_dwID);

    if(m_PatchTable)
        delete [] m_PatchTable;

    if(m_CompressedTable)
        delete [] m_CompressedTable;

    free(m_pBackBuff);
    free(m_pBuff);
}

#define VERSION_4_CONNECT    "GNUTELLA CONNECT/0.4\n\n"
#define VERSION_4_CONNECT_OK "GNUTELLA OK\n\n"

void MGnuNode::OnConnect(int nErrorCode)
{
    CString Handshake;

    // Get Remote host
    //GetPeerName(m_ipHost, m_nPort);

	if (!m_pPrefs->m_bAcceptRemoteIpHeader)
	{
		CString sLocalHost;
		u_int nPort;
		VERIFY(GetSockName(sLocalHost, nPort));
		VERIFY(Str2Ip(sLocalHost,m_pPrefs->m_ipLocalHost));
	}

    if(true)
    {
		CString sMuteVersion = VERSION;

        Handshake =  "GNUTELLA CONNECT/0.6\r\n";
        Handshake += "User-Agent: Mutella-" + sMuteVersion + "\r\n";

        // Ultrapeer Header
        if(m_pComm->GetHostMode() == CM_ULTRAPEER)
        {
            Handshake += "X-Ultrapeer: True\r\n";
            Handshake += "X-Leaf-Max: " + DWrdtoStr(m_pPrefs->m_nMaxLeaves) + "\r\n";
        }
        else
        	Handshake += "X-Ultrapeer: False\r\n";

        // Query Routing Header
        Handshake += "X-Query-Routing: 0.1\r\n";

        // Uptime Header
        int nSpan  = xtime() - m_pComm->GetClientStartTime();
        int nMins  = nSpan  / 60;
        int nHours = nMins  / 60;
        int nDays  = nHours / 24;
        nMins  %= 60;
        nHours %= 24;
        CString sTime;
        sTime.format("%02dD %02dH %02dM", nDays, nHours, nMins);
        Handshake += "Uptime: " + sTime + "\r\n";

        // Bye packet
        Handshake += "Bye-Packet: 0.1\r\n";

        // Authentication
        /*m_pCore->m_comCore->m_comNetwork->FireAuthenticate(m_NodeID);
        if( !m_Challenge.IsEmpty() && !m_ChallengeAnswer.IsEmpty() )
            Handshake += "X-Auth-Challenge: " + m_Challenge + "\r\n";*/

        Handshake += "\r\n";
    }
    else
        Handshake = VERSION_4_CONNECT ;

    Send(Handshake.c_str(), Handshake.length());

    // Add to log
    ReplaceSubStr(Handshake, "\n\n", "\r\n\r\n");
    m_WholeHandshake += Handshake;

    MAsyncSocket::OnConnect(nErrorCode);
}

void MGnuNode::OnReceive(int nErrorCode)
{
    int BuffLength = 0;

    ASSERT(m_ExtraLength >= 0);
    ASSERT(m_ExtraLength < PACKET_BUFF);
    BuffLength = Receive(m_pBuff + m_ExtraLength, PACKET_BUFF - m_ExtraLength);

    // Handle Errors
    if(!BuffLength || BuffLength == SOCKET_ERROR)
    {
        ForceDisconnect();
        return;
    }

    // Bandwidth stats
    m_nStatisticsSec[NR_BYTE_IN] += BuffLength;
    m_pComm->m_dwConnBytesRecv += BuffLength;


    BuffLength += m_ExtraLength;
    m_ExtraLength = 0;


    // Connected to node, sending and receiving packets
	if(m_nStatus == SOCK_CONNECTED)
	{
		ASSERT(BuffLength > 0);
		ASSERT(BuffLength <= PACKET_BUFF);
		
		if (SplitBundle(m_pBuff, BuffLength, &m_ExtraLength))
		{
			if (0 != m_ExtraLength)
			{
				ASSERT(m_ExtraLength < PACKET_BUFF);
				memmove(m_pBuff, m_pBuff + (BuffLength - m_ExtraLength), m_ExtraLength);
			}
		}
		else
		{
			// got bad packets there -- drop
			TRACE2("*** Got bad packets, closing the connection with ", m_sRemoteClient);
			ForceDisconnect();
			return;
		}
	}

    // Still in handshake mode
    else
    {
        CString MoreData((char*) m_pBuff, BuffLength);

        m_InitData += MoreData;

        // Gnutella 0.4 Handshake
        if( m_InitData.find(VERSION_4_CONNECT_OK) != -1)
        {
            ParseHandshake04(m_InitData, m_pBuff, BuffLength);
        }
        // Gnutella 0.6 Handshake
        else if( m_InitData.find("\r\n\r\n") != -1 )
        {
            if(m_bInbound)
                ParseIncomingHandshake06(m_InitData, m_pBuff, BuffLength, true);
            else
                ParseOutboundHandshake06(m_InitData, m_pBuff, BuffLength);
        }


        if(m_InitData.length() > 4096)
        {
            m_WholeHandshake += m_InitData;
            ForceDisconnect();
        }
    }

    // Block further receives if
	if (m_dwMaxInBytesPerSec > 0 &&
		m_dwMaxInBytesPerSec <= m_nStatisticsSec[NR_BYTE_IN])
	{
		//cout << "node: blocking receive because m_dwSecBytes[0] = " << m_dwSecBytes[0] << " while limit is " << m_dwMaxInBytesPerSec << "\n";
		ModifySelectFlags(0, FD_READ);
		m_bReceiveBlocked = true;
	}

    MAsyncSocket::OnReceive(nErrorCode);
}


/////////////////////////////////////////////////////////////////////////////
// New connections

void MGnuNode::ParseHandshake04(CString Data, BYTE* Stream, int StreamLength)
{
    // Node making an inbound 0.4 connection
    if(m_bInbound)
    {
        if(m_pComm->IsShieldedLeaf())
        {
            ForceDisconnect();
            return;
        }

        bool Busy = false;

        if(m_pPrefs->m_nMaxConnects >= 0)
            if(m_pComm->CountNormalConnects() >= m_pPrefs->m_nMaxConnects)
            {
                m_pHostCatcher->SendCache(this, 10);
                ForceDisconnect();
                return;
            }


        m_WholeHandshake += VERSION_4_CONNECT;

        // Send Connect OK
        CString Response = VERSION_4_CONNECT_OK ;
        Send(Response.c_str(), Response.length());

        ReplaceSubStr(Response, "\n\n", "\r\n");
        m_WholeHandshake += Response;

        SetConnected();
    }


    // This node making an outbound 0.4 connect
    else
    {
        m_WholeHandshake += "GNUTELLA OK\r\n";

        // Stream begins
        for(int i = 0; i < StreamLength - 2; i++)
            if(strncmp((char*) &Stream[i], "\n\n", 2) == 0)
            {
                m_ExtraLength = StreamLength - (i + 2);
                memcpy(m_pBuff, &Stream[i + 2], m_ExtraLength);
            }

        SetConnected();
    }
}

bool MGnuNode::ParseIncomingHandshake06(CString Data, BYTE* pStream, int StreamLength, bool bHandshakeInBuf)
{
    m_Handshake = Data.substr(0, Data.find("\r\n\r\n") + 4);
    m_WholeHandshake += m_Handshake;

    m_lowHandshake = m_Handshake;
    m_lowHandshake.make_lower();

    // Connect string, GNUTELLA CONNECT/0.6\r\n
    if(m_Handshake.find("GNUTELLA CONNECT/") != -1)
    {
        // Parse User-Agent header
        m_sRemoteClient = find_header("User-Agent", m_Handshake, m_lowHandshake, "Unknown");
        if (m_sRemoteClient.length() > 25)
		{
			int nPos = m_sRemoteClient.find(' ', 15);
			if (nPos>0)
				m_sRemoteClient.cut(nPos);
		}

        // Parse X-Query-Routing
        bool QueryRouting = false;
        CString RoutingHeader = find_header("X-Query-Routing", m_Handshake, m_lowHandshake);
        if(RoutingHeader == "0.1")
            QueryRouting = true;

        // Parse Uptime
        int days = 0, hours = 0, minutes = 0;
        CString UptimeHeader = find_header("Uptime", m_Handshake, m_lowHandshake);
        if(!UptimeHeader.empty())
        {
            sscanf(UptimeHeader.c_str(), "%dD %dH %dM", &days, &hours, &minutes);
            m_nPeerUptime -= days*86400 + hours*3600 + minutes*60;
        }

        // Parse Authentication Challenge
        /*CString ChallengeHeader = FindHeader("X-Auth-Challenge");
        if( !ChallengeHeader.IsEmpty() )
        {
            m_RemoteChallenge = ChallengeHeader;
            m_pCore->m_comCore->m_comNetwork->FireChallenge(m_NodeID, m_RemoteChallenge);
        }
        // Authenticate connection
        m_pCore->m_comCore->m_comNetwork->FireAuthenticate(m_NodeID);*/


        //Parse Ultrapeer header
        CString UltraHeader = find_header("X-Ultrapeer", m_Handshake, m_lowHandshake);
        if(!UltraHeader.empty())
        {
            UltraHeader.make_lower();

            // Connecting client an ultrapeer
            if(UltraHeader == "true")
                m_nPeerMode = CM_ULTRAPEER;
            else
                m_nPeerMode = CM_LEAF;
        }

        // Parse leaf max header
        CString LeafMax = find_header("X-Leaf-Max", m_Handshake, m_lowHandshake);
        if(!LeafMax.empty())
            m_NodeLeafMax = atoi(LeafMax.c_str());
        else
            m_NodeLeafMax = 75;

        if(m_NodeLeafMax > 1500)
            m_NodeLeafMax = 1500;

        if(m_lowHandshake.find("bearshare 2.") != -1)
            m_nPeerMode = CM_NORMAL;

		switch (m_nPeerMode) {
			case CM_ULTRAPEER: // Connecting node a SuperNode
				switch (m_pComm->GetHostMode()) {
					case CM_ULTRAPEER: // A supernode connecting to our supernode
						if( m_pPrefs->m_nMaxConnects >= 0 && m_pComm->CountNormalConnects() >= m_pPrefs->m_nMaxConnects          ||
							m_pPrefs->m_nMaxIncomingConns >= 0 && m_pComm->CountIncomingConns() >= m_pPrefs->m_nMaxIncomingConns )
						{
							Send_ConnectBusy();
							return false;
						}
						
						// If we have 66% free room of their max leaf count, request downgrade
						if(m_pComm->FreeUltraCapacity(m_NodeLeafMax) > 66)
							m_DowngradeRequest = true;
						// break means OK
						break;
					case CM_ULTRALEAF: // A supernode connecting to our supernode in leaf mode
					case CM_LEAF: // A supernode connecting to our leaf
						if( QueryRouting &&
							( m_pPrefs->m_nLeafModeConnects < 0 || m_pComm->CountNormalConnects() < m_pPrefs->m_nLeafModeConnects ) &&
							( m_pPrefs->m_nMaxIncomingConns < 0 || m_pComm->CountIncomingConns() < m_pPrefs->m_nMaxIncomingConns )  )
						{
							Send_ConnectOK(false, CM_LEAF);
							return true;
						}
						Send_ConnectBusy();
						return false;
				}
				break;
			case CM_LEAF: // Connecting node is a Leaf
				switch (m_pComm->GetHostMode()) {
					case CM_ULTRAPEER:
						if (!QueryRouting                                                ||
							m_pComm->RunningUltraCapacity(m_pPrefs->m_nMaxLeaves) >= 100 )
						{
							Send_ConnectBusy();
							return false;
						}
						// break means OK
						break;
					case CM_ULTRALEAF: 
					case CM_LEAF: // A leaf connecting to our leaf
						if (m_pComm->IsShieldedLeaf() ||
							m_pPrefs->m_nMaxIncomingConns >= 0 && m_pComm->CountIncomingConns() >= m_pPrefs->m_nMaxIncomingConns )
						{
							Send_ConnectBusy();
							return false;
						}
						// break means OK
						break;
				}
				break;
			case CM_NORMAL: // Connecting Node a Normal Node
			default:
				if (m_pComm->IsShieldedLeaf())
				{
					Send_ConnectBusy();
					return false;
				}

				if( m_pPrefs->m_nMaxConnects >= 0 && m_pComm->CountNormalConnects() >= m_pPrefs->m_nMaxConnects          ||
					m_pPrefs->m_nMaxIncomingConns >= 0 && m_pComm->CountIncomingConns() >= m_pPrefs->m_nMaxIncomingConns )
				{
					Send_ConnectBusy();
					return false;
				}
		}

		// we are going to accept the connections
		Send_ConnectOK(false, m_pComm->GetHostMode());
		return true;
	}
    // Ok string, GNUTELLA/0.6 200 OK\r\n
    else if(m_Handshake.find(" 200 OK\r\n") != -1)
    {
        // Parse Authentication Response
        //if( !m_Challenge.empty() )
        //    if(m_ChallengeAnswer.Compare( FindHeader("X-Auth-Response") ) != 0 )
        //    {
        //        Close();
        //        return;
        //    }

        // A SuperNode finishing its connection with our leaf
        if(m_nPeerMode == CM_ULTRAPEER)
        {
            if(m_pComm->IsLeaf() && !m_pComm->IsShieldedLeaf())
            {
				// we were in leaf mode but did not have superpeer connects
				m_pComm->CloseAllOtherNodes(this);
				TRACE("Close All Other (1)");
            }
            else if (m_pComm->GetHostMode() == CM_ULTRAPEER)
            {
                // Supernode to Supernode connect finishing ...
                //Parse Ultrapeer header
                CString UltraHeader = find_header("X-Ultrapeer", m_Handshake, m_lowHandshake);
                if(!UltraHeader.empty())
                    UltraHeader.make_lower();

                if(m_DowngradeRequest)
                {
                    if(UltraHeader == "false")
                        m_nPeerMode = CM_LEAF;
                    else if(m_pPrefs->m_nMaxConnects >= -1 && m_pComm->CountNormalConnects() >= m_pPrefs->m_nMaxConnects)
                    {
						#warning we are a bit rude with this Close() here!
                        ForceDisconnect();
                        return false;
                    }
                }
            }
        }

        SetConnected();

        // Stream begins
        if (bHandshakeInBuf)
        {
            for(int i = 0; i < StreamLength - 4; i++)
                if(strncmp((char*) (pStream + i), "\r\n\r\n", 4) == 0)
                {
                    m_ExtraLength = StreamLength - (i + 4);
                    memcpy(m_pBuff, pStream + i + 4, m_ExtraLength);
                }
        }
        else
        {
            m_ExtraLength = StreamLength;
            memcpy(m_pBuff, pStream, m_ExtraLength);
        }
        return true;
    }

    // Error string
    else
    {
        // Parse X-Try header
        CString TryHeader = find_header("X-Try", m_Handshake, m_lowHandshake);
        if(!TryHeader.empty())
            ParseTryHeader( TryHeader );


        // Parse X-Try-Ultrapeers header
        CString UltraTryHeader = find_header("X-Try-Ultrapeers", m_Handshake, m_lowHandshake);
        if(!UltraTryHeader.empty())
            ParseTryHeader( UltraTryHeader, true );

        ForceDisconnect();

		CString sMsg = "\n" + m_WholeHandshake;
		ReplaceSubStr(sMsg, "\n", "\n  ");
		POST_EVENT( MStringEvent(
			ET_MESSAGE, ES_NONE,
			sMsg,
			MSG_CONNECTION_HANDSHAKE_FAIL,
			MSGSRC_NODE
		));
        
        return false;
    }
}

void MGnuNode::ParseOutboundHandshake06(CString Data, BYTE* Stream, int StreamLength)
{
    m_Handshake = Data.substr(0, Data.find("\r\n\r\n") + 4);
    m_WholeHandshake += m_Handshake;

    m_lowHandshake = m_Handshake;
    m_lowHandshake.make_lower();


    // Ok string, GNUTELLA/0.6 200 OK\r\n
    if(m_Handshake.find(" 200 OK\r\n") != -1)
    {
        // Parse Remote-IP header
        if (m_pPrefs->m_bAcceptRemoteIpHeader)
        {
            CString RemoteIP = find_header("Remote-IP", m_Handshake, m_lowHandshake);
            if(!RemoteIP.empty())
                Str2Ip(RemoteIP, m_pPrefs->m_ipLocalHost);
        }

        // Parse User-Agent header
        m_sRemoteClient = find_header("User-Agent", m_Handshake, m_lowHandshake, "Unknown");
        if (m_sRemoteClient.length() > 25)
		{
			int nPos = m_sRemoteClient.find(' ');
			if (nPos>0)
				m_sRemoteClient.cut(nPos);
		}

        // Parse X-Query-Routing header
        bool QueryRouting = false;
        CString RoutingHeader = find_header("X-Query-Routing", m_Handshake, m_lowHandshake);
        if(RoutingHeader == "0.1")
            QueryRouting = true;

        // Parse Uptime
        int days = 0, hours = 0, minutes = 0;
        CString UptimeHeader = find_header("Uptime", m_Handshake, m_lowHandshake);
        if(!UptimeHeader.empty())
        {
            sscanf(UptimeHeader.c_str(), "%dD %dH %dM", &days, &hours, &minutes);
            m_nPeerUptime -= days*86400 + hours*3600 + minutes*60;
        }
        //if (days+hours+minutes == 0)
        //	cout << m_Handshake;

        // Parse Authentication Response
        //if( !m_Challenge.IsEmpty() )
        //    if(m_ChallengeAnswer.Compare( FindHeader("X-Auth-Response") ) != 0 )
        //    {
        //        Close();
        //        return;
        //    }

        // Parse Authentication Challenge
        //CString ChallengeHeader = FindHeader("X-Auth-Challenge");
        //if( !ChallengeHeader.IsEmpty() )
        //{
        //    m_RemoteChallenge = ChallengeHeader;
        //    m_pCore->m_comCore->m_comNetwork->FireChallenge(m_NodeID, m_RemoteChallenge);
        //}

        // Parse Ultrapeer header
        CString UltraHeader = find_header("X-Ultrapeer", m_Handshake, m_lowHandshake);
        if(!UltraHeader.empty())
        {
            UltraHeader.make_lower();

            if(UltraHeader == "true")
                m_nPeerMode = CM_ULTRAPEER;
            else
                m_nPeerMode = CM_LEAF;
        }

        // Parse leaf max header
        CString LeafMax = find_header("X-Leaf-Max", m_Handshake, m_lowHandshake);
        if(!LeafMax.empty())
            m_NodeLeafMax = atoi(LeafMax.c_str());
        else
            m_NodeLeafMax = 75;

        if(m_NodeLeafMax > 1500)
            m_NodeLeafMax = 1500;

        if(m_lowHandshake.find("bearshare 2.") != -1)
            m_nPeerMode = CM_NORMAL;

        CString NeededHeader;
		// Parse the Ultrapeers Needed header
		NeededHeader = find_header("X-Ultrapeer-Needed", m_Handshake, m_lowHandshake);
		NeededHeader.make_lower();

		switch (m_nPeerMode) {
			case CM_ULTRAPEER: // Connecting to a SuperNode
				switch (m_pComm->GetHostMode()) {
					case CM_ULTRAPEER: // our supernode is connection to another supernode
						// what if the other peer thinks that there's enough ultrapeers around
						if(NeededHeader == "false")
						{
							// We are an ultrapeer, lets see if we shall be
							if (!m_pComm->IsForcedUltraPeer())
							{
								// participate de-election process
								if (m_nPeerUptime > m_pComm->GetClientStartTime()    &&
									m_pComm->RunningUltraCapacity(m_NodeLeafMax) < 33)
								{
									// the other guy is deffinetely better than we are
									m_pComm->DecUltrapeerCredit();
									if (m_pComm->GetUltrapeerCredit() < 0       &&
										xtime() > m_pComm->HostModeValidUntil() )
									{
										// we do not deserve to be an ultrapeer
										m_DowngradeRequest        = true;
										m_pComm->CloseAllOtherNodes(this);
										m_pComm->SetHostMode(CM_ULTRALEAF);
										TRACE("Downgraded (1) ...");
									}
								}
							}
						}
						// what if the other peer thinks that there's not enough ultrapeers around
						if(NeededHeader == "true")
						{
							if (!m_pComm->IsForcedUltraPeer())
							{
								// participate election process
								if (m_nPeerUptime < m_pComm->GetClientStartTime())
								{
									// we are older means we are good
									m_pComm->IncUltrapeerCredit();
								}
							}
						}
						// break means OK
						break;                
					case CM_ULTRALEAF: // Our supernode in leaf mode is connecting to a supernode
						// This SuperNode needs more supernodes
						if( NeededHeader == "true" &&
							xtime() > m_pComm->HostModeValidUntil())
						{
							// Become a supernode
							m_pComm->CloseAllOtherNodes(this);
							m_pComm->SetHostMode(CM_ULTRAPEER);
							TRACE("Upgraded (1) ...");
							break;
						}
					case CM_LEAF: // Our leaf is connecting to a supernode
						if (!m_pComm->IsShieldedLeaf()) // we were in a leaf mode, but did not have connections with UPs
						{
							// close all other connections
							m_pComm->CloseAllOtherNodes(this);
							TRACE("Close all other (2)");
						}
						// break is OK
						break;
				}
				// break is OK
				break;
			case CM_LEAF: // Connecting to a node which is a Leaf
				switch (m_pComm->GetHostMode()) {
					case CM_ULTRAPEER: // our supernode is conecting to a leaf
						if (!QueryRouting)
						{
							Send_ConnectBusy();
							return;
						}
						// break means OK
						break;
					case CM_ULTRALEAF:
					case CM_LEAF: // Our leaf is connecting a leaf
						if (m_pComm->IsShieldedLeaf())
						{
							Send_ConnectBusy();
							return;
						}
						// break means OK
						break;
				}
				break;
			case CM_NORMAL: // Connecting Node a Normal Node
			default:
				if (m_pComm->IsShieldedLeaf())
				{
					Send_ConnectBusy();
					return;
				}
		}

		Send_ConnectOK(true, m_pComm->GetHostMode());
		SetConnected();
    }

    // Connect failed, 200 OK not received
    else
    {
        // Parse X-Try header
        CString TryHeader = find_header("X-Try", m_Handshake, m_lowHandshake);
        if(!TryHeader.empty())
            ParseTryHeader( TryHeader );

        // Parse X-Try-Ultrapeers header
        CString UltraTryHeader = find_header("X-Try-Ultrapeers", m_Handshake, m_lowHandshake);
        if(!UltraTryHeader.empty())
            ParseTryHeader( UltraTryHeader, true );


        ForceDisconnect();

		CString sMsg = "\n" + m_WholeHandshake;
		ReplaceSubStr(sMsg, "\n", "\n  ");
		POST_EVENT( MStringEvent(
			ET_MESSAGE, ES_NONE,
			sMsg,
			MSG_CONNECTION_HANDSHAKE_FAIL,
			MSGSRC_NODE
		));
    }
}

void MGnuNode::ParseTryHeader(CString TryHeader, bool Super)
{
    TryHeader += ",";
    ReplaceSubStr(TryHeader, ",,", ",");
    ReplaceSubStr(TryHeader, " ", "");

    int tryFront = 0,
        tryMid = TryHeader.find(":"),
        tryBack = TryHeader.find(",");

    while(tryBack != -1 && tryMid != -1)
    {
        Node tryNode;
        if (Str2Ip(TryHeader.substr(tryFront, tryMid - tryFront), tryNode.Host))
        {
        	tryNode.Port = atoi( TryHeader.substr(tryMid + 1, tryBack - tryMid + 1).c_str());

        	m_pUltraCatcher->UpdateCache(tryNode);
        }

        tryFront  = tryBack + 1;
        tryMid    = TryHeader.find(":", tryFront);
        tryBack   = TryHeader.find(",", tryFront);
    }
}

void MGnuNode::Send_ConnectOK(bool Reply, int HeaderMode)
{
    CString Handshake;

    // Reply to CONNECT OK
    if(Reply)
    {
        Handshake = "GNUTELLA/0.6 200 OK\r\n";


        // We are converting from supernode to a leaf
        if(m_DowngradeRequest)
            Handshake += "X-Ultrapeer: False\r\n";


        // Send Answer to Challenge
        if( !m_RemoteChallengeAnswer.empty() )
            Handshake += "X-Auth-Response: " + m_RemoteChallengeAnswer + "\r\n";


        Handshake += "\r\n";
    }

    // Sending initial CONNECT OK
    else
    {
        Handshake =  "GNUTELLA/0.6 200 OK\r\n";
        Handshake += CString("User-Agent: Mutella-") + VERSION + "\r\n";

        // Remote IP header
        UINT nTrash;
        CString sPeerAddress;
        GetPeerName(sPeerAddress, nTrash);
        Str2Ip(sPeerAddress, m_ipHost);
        Handshake += "Remote-IP: " + sPeerAddress + "\r\n";

        // Query Routing Header
        Handshake += "X-Query-Routing: 0.1\r\n";

        // Ultrapeer header
        if(HeaderMode == CM_ULTRAPEER)
        {
            Handshake += "X-Ultrapeer: True\r\n";
            Handshake += "X-Leaf-Max: " + DWrdtoStr(m_pPrefs->m_nMaxLeaves) + "\r\n";

            if(m_DowngradeRequest)
                Handshake += "X-Ultrapeer-Needed: False\r\n";
        }
        else
            Handshake += "X-Ultrapeer: False\r\n";


        // Uptime header
        int nSpan  = xtime() - m_pComm->GetClientStartTime();
        int nMins  = nSpan  / 60;
        int nHours = nMins  / 60;
        int nDays  = nHours / 24;
        nMins  %= 60;
        nHours %= 24;
        CString sTime;
        sTime.format("%02dD %02dH %02dM", nDays, nHours, nMins);
        Handshake += "Uptime: " + sTime + "\r\n";

        // Send authentication response
        //if( !m_RemoteChallengeAnswer.IsEmpty() )
        //    Handshake += "X-Auth-Response: " + m_RemoteChallengeAnswer + "\r\n";


        // Send authentication challenge
        //if( !m_Challenge.IsEmpty() && !m_ChallengeAnswer.IsEmpty() )
        //    Handshake += "X-Auth-Challenge: " + m_Challenge + "\r\n";

        // X-Try header
        CString HostsToTry;
        if(GetAlternateHostList(HostsToTry))
            Handshake += "X-Try: " + HostsToTry + "\r\n";

        // X-Try-Ultrapeers header
        CString SuperHostsToTry;
        if(GetAlternateSuperList(SuperHostsToTry))
            Handshake += "X-Try-Ultrapeers: " + SuperHostsToTry + "\r\n";

        Handshake += "\r\n";
    }

    Send(Handshake.c_str(), Handshake.length());

    ReplaceSubStr(Handshake, "\n\n", "\r\n");
    m_WholeHandshake += Handshake;
}

void MGnuNode::Send_ConnectBusy()
{
    CString Handshake;

    Handshake =  "GNUTELLA/0.6 503 BUSY\r\n";
    Handshake += CString("User-Agent: Mutella-") + VERSION + "\r\n";


    // LAN header
    //if(m_pPrefs->m_LanMode)
    //    Handshake += "LAN: " + m_pPrefs->m_LanName + "\r\n";

    // Remote-IP header
    UINT nTrash;
    CString sPeerAddress;
    GetPeerName(sPeerAddress, nTrash);
    Str2Ip(sPeerAddress, m_ipHost);
    Handshake += "Remote-IP: " + sPeerAddress + "\r\n";

    // X-Try header
    CString HostsToTry;
    if(GetAlternateHostList(HostsToTry))
        Handshake += "X-Try: " + HostsToTry + "\r\n";

    // X-Try-Ultrapeers header
    CString SuperHostsToTry;
    if(GetAlternateSuperList(SuperHostsToTry))
        Handshake += "X-Try-Ultrapeers: " + SuperHostsToTry + "\r\n";

    Handshake += "\r\n";

    Send(Handshake.c_str(), Handshake.length());

    ReplaceSubStr(Handshake, "\n\n", "\r\n");
    m_WholeHandshake += Handshake;
}

void MGnuNode::SetConnected()
{
    // Get Remote host
    CString sHostAddress;
    UINT Port;
    GetPeerName(sHostAddress, Port);
    Str2Ip(sHostAddress, m_ipHost);

    m_nStatus = SOCK_CONNECTED;
    //m_pComm->NodeUpdate(m_NodeID);

    // We are a leaf send index to supernode
    if(m_pComm->IsLeaf())
        m_PatchUpdateNeeded  = true;

    Send_Ping(1);

	CString sMsg = "\n" + m_WholeHandshake;
	ReplaceSubStr(sMsg, "\n", "\n  ");
	POST_EVENT( MStringEvent(
		ET_MESSAGE, ES_NONE,
		sMsg,
		MSG_CONNECTION_HANDSHAKE_OK,
		MSGSRC_NODE
	));
}

void MGnuNode::SendPacket(void* packet, int length, int type, bool local)
{

    list<PriorityPacket>::iterator itPacket;

    if(length > 65536 || length <= 0 || type > 5)
        return;

    if(local)
        type = 0;

    m_BufferAccess.lock();

        // Remove old packets if back list too large (does not apply to local packets)
        if(type != 0 && m_PacketList[type].size() > 50)
            while(m_PacketList[type].size() > 25)
            {
				// get pointer to the last packet
                itPacket = m_PacketList[type].end();
                itPacket--;
				
				// do not dequeue PONG packets with TTL<=1
				if (type == PACKET_PONG && ((packet_Header*)(itPacket->Packet))->TTL <= 1)
				{
					TRACE2("MGnuNode::SendPacket() : canceling drop for a packet with TTL %d", ((packet_Header*)(itPacket->Packet))->TTL);
					break;
				}
				
				// dequeue the packet
                m_PacketListLength[type] -= itPacket->Length;
                m_PacketList[type].pop_back();

                ASSERT(m_PacketListLength[type] >= 0);
                // increment dropped packet counter
                m_nDroppedPackets++;
                m_nSecDroppedPackets++;
            }

        ASSERT(packet);

        // Build a priority packet
        PriorityPacket SendPacket(packet, length);
        m_PacketListLength[type] += length;
        m_PacketList[type].push_front(SendPacket);

		ModifySelectFlags(FD_WRITE,0);
    m_BufferAccess.unlock();

}

void MGnuNode::FlushSendBuffer()
// *** Must be called with in a BufferAccess Lock
{
    list<PriorityPacket>::iterator itPacket;

    int AttemptSend = 0;

    if(m_nStatus != SOCK_CONNECTED)
        return;

    // Send back buffer
    if (m_BackBuffLength > 0)
    {
        AttemptSend = Send(m_pBackBuff, m_BackBuffLength);

        if(AttemptSend == SOCKET_ERROR || AttemptSend == 0)
        {
            if(AttemptSend == SOCKET_ERROR || GetLastError() != EWOULDBLOCK)
                ForceDisconnect();
            return;            
        }
        // we have probably sent something, but not all
        if (AttemptSend != m_BackBuffLength)
            memmove(m_pBackBuff, m_pBackBuff + AttemptSend, m_BackBuffLength - AttemptSend);
        m_BackBuffLength -= AttemptSend;
        ASSERT(m_BackBuffLength >= 0);
        ASSERT(m_BackBuffLength < PACKET_BUFF);
        if (m_BackBuffLength)
        	return; // we still have data to send in m_BackBuff
    }

    // Send prioritized buffer
    list<PriorityPacket>::iterator itType;

    // Each Packet type
    for(int i = 0; i < 6; i++)
        // while list not empty
        while(!m_PacketList[i].empty())
        {
            itPacket = m_PacketList[i].begin();

            //m_pComm->PacketOutgoing(m_NodeID, (*itPacket).Packet, (*itPacket).Length, (*itPacket).Type == 0);

            ASSERT(itPacket->Length >= 0 && itPacket->Length < 65536);

            // Send packet
            AttemptSend = Send(itPacket->Packet, itPacket->Length);

            // If send fails, copy to back buffer so it is the first to be sent
            if(AttemptSend == SOCKET_ERROR || AttemptSend == 0)
            {
                if(AttemptSend == SOCKET_ERROR || GetLastError() != EWOULDBLOCK)
                {
                    ForceDisconnect();
                    return;
                }

                memcpy(m_pBackBuff, itPacket->Packet, itPacket->Length - AttemptSend);
                m_BackBuffLength = itPacket->Length - AttemptSend;
                ASSERT(m_BackBuffLength > 0);
                ASSERT(m_BackBuffLength < PACKET_BUFF);
                
                // Delete packet once copied
                m_PacketListLength[i] -= itPacket->Length;
                m_PacketList[i].pop_front();

                m_nStatisticsSec[NR_PACKET_OUT]++;

                return;
            }
            // Delete packet once sent
            m_PacketListLength[i] -= itPacket->Length;
            m_PacketList[i].pop_front();

            m_nStatisticsSec[NR_PACKET_OUT]++;
        }
    ASSERT(m_PacketList[0].empty());
    ASSERT(m_PacketListLength[0] == 0);
    ASSERT(m_PacketList[1].empty());
    ASSERT(m_PacketListLength[1] == 0);
    ASSERT(m_PacketList[2].empty());
    ASSERT(m_PacketListLength[2] == 0);
    ASSERT(m_PacketList[3].empty());
    ASSERT(m_PacketListLength[3] == 0);
    ASSERT(m_PacketList[4].empty());
    ASSERT(m_PacketListLength[4] == 0);
    ASSERT(m_PacketList[5].empty());
    ASSERT(m_PacketListLength[5] == 0);
    // ensure that OnSend event will not be polled
    ModifySelectFlags(0, FD_WRITE);    
}

int MGnuNode::Send(const void* lpBuf, int nBuffLen, int nFlags)
{
    int BytesSent = 0;

    BytesSent = MAsyncSocket::Send(lpBuf, nBuffLen, nFlags);

    if(BytesSent > 0)
    {
        m_nStatisticsSec[NR_BYTE_OUT] += BytesSent;
        m_pComm->m_dwConnBytesSend += BytesSent;
    }

    return BytesSent;
}

void MGnuNode::OnSend(int nErrorCode)
{
    m_BufferAccess.lock();
    FlushSendBuffer();
    // ensure that OnSend event will not be polled
	ModifySelectFlags(0, FD_WRITE);
    m_BufferAccess.unlock();

    MAsyncSocket::OnSend(nErrorCode);
}

void MGnuNode::OnClose(int nErrorCode)
{
    /* Add node to cache if it returned other hosts
    Node PermNode;
    PermNode.Host = m_ipHost;
    PermNode.Port = m_nPort;

    if(m_pComm->NotLocal(PermNode) && m_pPrefs->AllowedIP(StrtoIP(PermNode.Host)) && m_nPort != 0)
    {
        while(m_pCache->m_PermNodes.size() > NODECACHE_SIZE)
            m_pCache->m_PermNodes.pop_back();

        if(PermNode.Port)
            m_pCache->m_PermNodes.push_front(PermNode);
    }*/


    ForceDisconnect();

    MAsyncSocket::OnClose(nErrorCode);
}

void MGnuNode::Close()
{
	//TRACE2("MGnuNode::Close() : xtime() = ", xtime());
	//struct timespec rqtp, rmtp;
	//rqtp.tv_sec = 1;
	//rqtp.tv_nsec = rmtp.tv_sec = rmtp.tv_nsec = 0;
	//safe_nanosleep(&rqtp, &rmtp);

    if(m_hSocket != INVALID_SOCKET)
    {
        // Close socket
        AsyncSelect(0);
        ShutDown(1);

        MAsyncSocket::Close();
    }


    m_nStatus = SOCK_CLOSED;
    //m_pComm->NodeUpdate(m_NodeID);
}

void  MGnuNode::ForceDisconnect(bool bExernal /*=false*/)
{
	//TRACE("MGnuNode::ForceDisconnect()");
	Close();
}

/////////////////////////////////////////////////////////////////////////////
// Packet handlers

bool MGnuNode::SplitBundle(BYTE* bundle, DWORD length, DWORD* pExtraLength)
{
    UINT Payload = 0;
    UINT nextPos = 0;

    packet_Header* packet;

    enum status { status_DONE,       status_CONTINUE,
                  status_BAD_PACKET, status_INCOMPLETE_PACKET };

    status theStatus = status_CONTINUE;

    do
    {
        if (nextPos + sizeof(packet_Header) > length)
            theStatus = status_INCOMPLETE_PACKET;
        else
        {
            packet = (packet_Header*) (bundle + nextPos);

            Payload = packet->le_Payload;

            if((packet->Function == 0x00 && Payload ==  0)                  ||
               (packet->Function == 0x01 && Payload == 14)                  ||
               (packet->Function == 0x02 && Payload < 32768)                ||
               (packet->Function == 0x30 && Payload > 2 && Payload < 32768) ||
               (packet->Function == 0x40 && Payload == 26)                  ||
               (packet->Function == 0x80 && Payload >  2 && Payload <= 230) ||
               (packet->Function == 0x81 && Payload > 26 && Payload <= 32768))
            {
                if (nextPos + sizeof(packet_Header) + Payload <= length)
                {
                    HandlePacket(packet, 23 + Payload);

                    nextPos += 23 + Payload;
                    if (nextPos == length)
                        theStatus = status_DONE;
                }
                else
                    theStatus = status_INCOMPLETE_PACKET;
            }
            else
            {
            if (nextPos < length - sizeof(packet_Header))
                    nextPos++;
            else
            {
                    theStatus = status_BAD_PACKET;
                    TRACE4("Bad Packet: code= ", (int) packet->Function, "  playload=", Payload);
            }
            }
        }
    } while(status_CONTINUE == theStatus);

    if (pExtraLength)
    {
        *pExtraLength = length - nextPos;
    }
    return status_BAD_PACKET != theStatus;
}

void MGnuNode::HandlePacket(packet_Header* packet, DWORD length)
{
    m_nStatisticsSec[NR_PACKET_IN]++;

    switch(packet->Function)
    {
    case 0x00:
        Receive_Ping((packet_Ping*) packet, length);
        break;

    case 0x01:
        Receive_Pong((packet_Pong*) packet, length);
        break;

    case 0x02:
        Receive_Bye((packet_Bye*) packet, length);
        break;

    case 0x30:
        if( ((packet_RouteTableReset*) packet)->PacketType == 0x0)
            Receive_RouteTableReset((packet_RouteTableReset*) packet, length);
        else if(((packet_RouteTableReset*) packet)->PacketType == 0x1)
            Receive_RouteTablePatch((packet_RouteTablePatch*) packet, length);

        break;
    case 0x40:
        Receive_Push((packet_Push*) packet, length);
        break;

    case 0x80:
        Receive_Query((packet_Query*) packet, length);
        break;

    case 0x81:
        Receive_QueryHit((packet_QueryHit*) packet, length);
        break;

    default:
        // Disable unknowns
        // Receive_Unknown((byte *) packet, length);
        break;
    }
}

bool MGnuNode::InspectPacket(packet_Header* packet)
{
    if(packet->TTL == 0 ||  packet->Hops >= MAX_TTL_ACCEPTED)
        return false;

    packet->Hops++;
    packet->TTL--;

    // Reset TTL if too high
    if(packet->TTL >= MAX_TTL_ACCEPTED)
        packet->TTL = MAX_TTL_ACCEPTED - packet->Hops;

    return true;
}


/////////////////////////////////////////////////////////////////////////////
// Receiving packets

void MGnuNode::Receive_Ping(packet_Ping* Ping, int nLength)
{
    int nError = 0;

    // Packet stats
    int StatPos = UpdateStats(Ping->Header.Function);

    // Inspect
    if(!InspectPacket(&Ping->Header))
    {
        //nError = ERROR_HOPS;
        //m_pComm->PacketIncoming(m_NodeID, (byte*) Ping, nLength, nError, false);
        return;
    }

    key_Value* pRouteKey      = m_pComm->m_TableRouting.FindValue(&Ping->Header.Guid);
    key_Value* pLocalRouteKey = m_pComm->m_TableLocal.FindValue(&Ping->Header.Guid);

    if(pLocalRouteKey)
    {
        //nError = ERROR_LOOPBACK;
        //m_pComm->PacketIncoming(m_NodeID, (byte*) Ping, nLength, nError, false);
        return;
    }

    // Fresh Ping?
    if(!pRouteKey)
    {
        m_pComm->m_TableRouting.Insert(&Ping->Header.Guid, m_dwID);

        // Ping from child node
        if(m_pComm->GetHostMode() == CM_ULTRAPEER && m_nPeerMode == CM_LEAF)
        {
            if(Ping->Header.Hops == 1)
            {
                Send_Pong(Ping->Header.Guid, Ping->Header.Hops);
                Ping->Header.TTL = 0;
            }
            else
            {
                //nError = ERROR_ROUTING;
                //m_pComm->PacketIncoming(m_NodeID, (byte*) Ping, nLength, nError, false);
                return;
            }
        }
        // always reply to pings with Hops = 1 
        else if(Ping->Header.Hops == 1)
            Send_Pong(Ping->Header.Guid, Ping->Header.Hops);

        // else if connections below min or hops below 3
        else if(Ping->Header.Hops < 3 || (m_pPrefs->m_nMinConnects >= 0 && m_pComm->GetNormalConnectsApprox() < m_pPrefs->m_nMinConnects))
            Send_Pong(Ping->Header.Guid, Ping->Header.Hops);

        // Send 1kb of extended pongs a minute (advertise our ultrapeer)
        else if(m_pComm->GetExtPongBytes() < 22 && m_pComm->GetHostMode() == CM_ULTRAPEER )
        {
            Send_Pong(Ping->Header.Guid, Ping->Header.Hops);
            m_pComm->IncExtPongBytes(23);
        }

        // Broadcast if still alive
        if(Ping->Header.Hops < MAX_TTL_SELF && Ping->Header.TTL > 0)
            m_pComm->Broadcast_Ping(Ping, nLength, this);

        //m_pComm->PacketIncoming(m_NodeID, (byte*) Ping, nLength, nError, false);

        AddGoodStat(Ping->Header.Function);

        return;
    }
    else
    {
        if(pRouteKey && pRouteKey->ID == m_dwID)
        {
            //nError = ERROR_DUPLICATE;
            //m_pComm->PacketIncoming(m_NodeID, (byte*) Ping, nLength, nError, false);
            return;
        }
        else
        {
            //nError = ERROR_ROUTING;
            //m_pComm->PacketIncoming(m_NodeID, (byte*) Ping, nLength, nError, false);
            return;
        }
    }
}


void MGnuNode::Receive_Pong(packet_Pong* Pong, int nLength)
{
    int nError = 0;

    // Packet stats
    int StatPos = UpdateStats(Pong->Header.Function);

    // Inspect
    if(!InspectPacket(&Pong->Header))
    {
        //nError = ERROR_HOPS;
        //m_pComm->PacketIncoming(m_NodeID, (byte*) Pong, nLength, nError, false);
        return;
    }

    // Detect if this pong is from an ultrapeer
    bool Ultranode = false;

    // TODO: check this
    UINT Marker = 8;
    while(Marker <= Pong->le_FileSize && Marker)
    {
        if(Marker == Pong->le_FileSize)
        {
            Ultranode = true;
            break;
        }
        else
        {
            Marker *= 2;
        }
    }

    // Add to host cache
    if(Pong->Header.Hops > 2)
    {
        Node NetNode;
        NetNode.Host    = Pong->Host;
        NetNode.Port    = Pong->le_Port;

        // Ultranode
        if (Ultranode)
        	m_pUltraCatcher->UpdateCache(NetNode);
        else
        	m_pHostCatcher->UpdateCache(NetNode);
    }

    key_Value* pRouteKey      = m_pComm->m_TableRouting.FindValue(&Pong->Header.Guid);
    key_Value* pLocalRouteKey = m_pComm->m_TableLocal.FindValue(&Pong->Header.Guid);

    if(pLocalRouteKey)
    {
        // If this pong is one we sent out
        if(pLocalRouteKey->ID) //(ID != 0) ???
        {
            if(pLocalRouteKey->ID == m_dwID)
            {
                //nError = ERROR_LOOPBACK;
                //m_pComm->PacketIncoming(m_NodeID, (byte*) Pong, nLength, nError, false);
                return;
            }
        }
        else
        {
            // Nodes file count
            if(Pong->Header.Hops == 1)
                m_NodeFileCount = Pong->le_FileCount;

            // Mapping
            //if(m_pCore->m_LocalMapping)
            //    MapPong(Pong, false);
            //if(Pong->Header.Hops < 3)
            //    MapPong(Pong, true);

            if (Pong->Header.Hops > MAX_TTL_SELF)
            	return; // we accept but ignore it

            m_dwFriendsTotal++;
			ASSERT(Pong->Header.Hops <= MAX_TTL_SELF);
			m_dwFriends[Pong->Header.Hops - 1]++;

			DWORD dwLibrary = Pong->le_FileSize;
			DWORD dwFiles = Pong->le_FileCount;
			if (dwLibrary>MAX_SHARE)
				dwLibrary = MAX_SHARE;
			if(dwLibrary)
			{
				m_dwLibrary[Pong->Header.Hops - 1] += dwLibrary;
				m_dwFiles[Pong->Header.Hops - 1] += dwFiles;
				m_llLibraryTotal += dwLibrary;
				m_dwFilesTotal += dwFiles;
				m_dwSharingTotal++;
			}

            AddGoodStat(Pong->Header.Function);
            //m_pComm->PacketIncoming(m_NodeID, (byte*) Pong, nLength, nError, true);

            return;
        }
    }

    if(pRouteKey)
    {
        // Send it out
        if(Pong->Header.Hops < MAX_TTL_SELF && Pong->Header.TTL > 0)
        {
            m_pComm->Route_Pong(Pong, nLength, pRouteKey->ID);

            if(Ultranode && m_nPeerMode != CM_LEAF)
                m_pComm->Route_UltraPong(Pong, nLength, pRouteKey->ID);
        }

        AddGoodStat(Pong->Header.Function);
        //m_pComm->PacketIncoming(m_NodeID, (byte*) Pong, nLength, nError, false);

        return;
    }
    else
    {
        // If pong advertising ultrapeer it is good
        if(m_pComm->IsLeaf() && Ultranode)
        {
            AddGoodStat(Pong->Header.Function);
            //m_pComm->PacketIncoming(m_NodeID, (byte*) Pong, nLength, nError, true);
            return;
        }

        //nError = ERROR_ROUTING;
        //m_pComm->PacketIncoming(m_NodeID, (byte*) Pong, nLength, nError, false);
        return;
    }
}

void MGnuNode::Receive_Bye(packet_Bye* Bye, int nLength)
{
	CString sStr(((LPSTR) Bye)+25, Bye->Header.le_Payload-2);
	WORD nCode = Bye->le_Code;
	CString sMsg;
	sMsg.format("Vendor: %s  Code: %d  \"%s\"", m_sRemoteClient.c_str(), nCode, sStr.c_str());
	POST_EVENT( MStringEvent(
		ET_MESSAGE, ES_NONE,
		sMsg,
		MSG_CONNECTION_BYE,
		MSGSRC_NODE
	));

	ForceDisconnect();
}

void MGnuNode::Receive_Push(packet_Push* Push, int nLength)
{
    int nError = 0;

    // Packet stats
    int StatPos = UpdateStats(Push->Header.Function);

    // Host Cache
    Node NetNode;
    NetNode.Host        = Push->Host;
    NetNode.Port        = Push->le_Port;
    m_pHostCatcher->UpdateCache(NetNode);

    // Inspect
    if(!InspectPacket(&Push->Header))
    {
        //nError = ERROR_HOPS;
        //m_pComm->PacketIncoming(m_NodeID, (byte*) Push, nLength, nError, false);
        return;
    }


    // Find packet in hash tables
    key_Value* pRouteKey      = m_pComm->m_TableRouting.FindValue(&Push->Header.Guid);
    key_Value* pLocalRouteKey = m_pComm->m_TableLocal.FindValue(&Push->Header.Guid);
    key_Value* pPushRouteKey  = m_pComm->m_TablePush.FindValue(&Push->ServerID);

    if(pLocalRouteKey)
    {
        //nError = ERROR_LOOPBACK;
        //m_pComm->PacketIncoming(m_NodeID, (byte*) Push, nLength, nError, false);
        return;
    }

    int i = 0;

    // Check ServerID of Push with ClientID of the client
    if(*m_pComm->GetClientID() == Push->ServerID)
    {
		// Make we aren't already pushing the file
		if (m_pComm->CheckIfPushung(Push))
		{
			//nError = ERROR_DUPLICATE;
			//m_pComm->PacketIncoming(m_NodeID, (byte*) Push, nLength, nError, false);
			return;
		}

		if (m_pPrefs->m_bQuietMode)
			return;
		TRACE("MGnuNode::Receive_Push(): processing request...");

		// Create upload
		MGnuUpload* pUpload = new MGnuUpload(m_pComm);
		ASSERT(pUpload);
		pUpload->m_ipHost     = Push->Host;
		pUpload->m_nPort      = Push->le_Port;
		pUpload->m_nFileIndex = Push->le_Index;

		if(!pUpload->Create(0, SOCK_STREAM, FD_READ | FD_CONNECT | FD_CLOSE)) // FD_WRITE shouldn't harm here
		{
			POST_ERROR(ES_UNIMPORTANT, CString("Upload Create Error: ") + DWrdtoStr(pUpload->GetLastError()));
		}

		// Add the node to the list
		m_pComm->AddUpload(pUpload);
		pUpload->PushFile();

        AddGoodStat(Push->Header.Function);
        //m_pComm->PacketIncoming(m_NodeID, (byte*) Push, nLength, nError, true);
        return;
    }

    if(!pRouteKey)
    {
        m_pComm->m_TableRouting.Insert(&Push->Header.Guid, m_dwID);
    }
    else
    {
        if(pRouteKey->ID == m_dwID)
        {
            //nError = ERROR_DUPLICATE;
            //m_pComm->PacketIncoming(m_NodeID, (byte*) Push, nLength, nError, false);
            return;
        }
        else
        {
            //nError = ERROR_ROUTING;
            //m_pComm->PacketIncoming(m_NodeID, (byte*) Push, nLength, nError, false);
            return;
        }
    }

    if(pPushRouteKey)
    {
        if(Push->Header.Hops < MAX_TTL_SELF && Push->Header.TTL > 0)
            m_pComm->Route_Push(Push, nLength, pPushRouteKey->ID);

        AddGoodStat(Push->Header.Function);
        //m_pComm->PacketIncoming(m_NodeID, (byte*) Push, nLength, nError, false);

        return;
    }
    else
    {
        //nError = ERROR_ROUTING;
        //m_pComm->PacketIncoming(m_NodeID, (byte*) Push, nLength, nError, false);
        return;
    }
}


void MGnuNode::Receive_Query(packet_Query* Query, int nLength)
{
    int nError = 0;

    // Packet stats
    int StatPos = UpdateStats(Query->Header.Function);

    // Inspect
    int QuerySize  = Query->Header.le_Payload - 2;
    int TextSize   = strlen((char*) Query + 25) + 1;

    // Bad packet, text bigger than payload
    if (TextSize > QuerySize)
    {
        //nError = ERROR_ROUTING;
        //m_pComm->PacketIncoming(m_NodeID, (byte*) Query, nLength, nError, false);
        //TRACE0("Text Query too big " + CString((char*) Query + 25) + "\n");
        return;
    }

    CString ExtendedQuery;

    if (TextSize < QuerySize)
    {
        int ExtendedSize = strlen((char*) Query + 25 + TextSize);

        if(ExtendedSize)
        {
            ExtendedQuery = CString((char*) Query + 25 + TextSize, ExtendedSize);

            /*int WholeSize = TextSize + HugeSize + 1;

            TRACE("Huge Query, " + DWrdtoStr(WholeSize) + " bytes\n");
            TRACE("     " + CString((char*)Query + 25 + TextSize) + "\n");

            if(WholeSize > QuerySize)
            {
                TRACE("   Huge Query too big " + CString((char*) Query + 25 + TextSize) + "\n");
            }

            if(WholeSize < QuerySize)
            {
                TRACE("   Huge Query too small " + CString((char*) Query + 25 + TextSize) + "\n");

                byte* j = 0;
                for(int i = WholeSize; i < QuerySize; i++)
                    j = (byte*) Query + 25 + i;
            }*/
        }
        else
        {
            // Query with double nulls, wtf
        }

    }

    if(!InspectPacket(&Query->Header))
    {
        //nError = ERROR_HOPS;
        //m_pComm->PacketIncoming(m_NodeID, (byte*) Query, nLength, nError, false);
        return;
    }

    key_Value* pRouteKey      = m_pComm->m_TableRouting.FindValue(&Query->Header.Guid);
    key_Value* pLocalRouteKey = m_pComm->m_TableLocal.FindValue(&Query->Header.Guid);

    if(pLocalRouteKey)
    {
        //nError = ERROR_LOOPBACK;
        //m_pComm->PacketIncoming(m_NodeID, (byte*) Query, nLength, nError, false);
        return;
    }

    // Fresh Query?
    if(!pRouteKey)
    {
        m_pComm->m_TableRouting.Insert(&Query->Header.Guid, m_dwID);

        AddGoodStat(Query->Header.Function);

        // Broadcast if still alive
        if(Query->Header.Hops < MAX_TTL_SELF && Query->Header.TTL > 0)
            m_pComm->Broadcast_Query(Query, nLength, this);

        // Queue to be compared with local files
        BYTE* pSearch = (BYTE*) &Query->Header;

        MShareReq* pSR = new MShareReq;

        pSR->m_query.OriginID  = m_dwID;
        pSR->m_query.nHops     = Query->Header.Hops;
        pSR->m_query.QueryGuid = Query->Header.Guid;
        strncpy(pSR->m_query.Text, ((char*)Query) + 25, MAX_QUERY_LEN);

        if (!ExtendedQuery.empty())
            pSR->m_query.ExtendedPart = ExtendedQuery;

        //if(Compare.QueryText.GetLength() > 200)
        //    return;

        ED().PostEvent( pSR );
        
        //m_pComm->PacketIncoming(m_NodeID, (byte*) Query, nLength, nError, false);
        return;
    }
    else
    {
        if(pRouteKey->ID == m_dwID)
        {
            //nError = ERROR_DUPLICATE;
            //m_pComm->PacketIncoming(m_NodeID, (byte*) Query, nLength, nError, false);
            return;
        }
        else
        {
            //nError = ERROR_ROUTING;
            //m_pComm->PacketIncoming(m_NodeID, (byte*) Query, nLength, nError, false);
            return;
        }
    }
}

void MGnuNode::Receive_QueryHit(packet_QueryHit* QueryHit, DWORD nLength)
{
    int nError = 0;

    // Packet stats
    int StatPos = UpdateStats(QueryHit->Header.Function);

    // Host Cache
    Node NetNode;
    NetNode.Host        = QueryHit->Host;
    NetNode.Port        = QueryHit->le_Port;
    m_pHostCatcher->UpdateCache(NetNode);

    // Inspect
    if(!InspectPacket(&QueryHit->Header))
    {
		// still makes sence to look inside...
		m_pComm->OnQueryHit(QueryHit, this, false);
        //nError = ERROR_HOPS;
        //m_pComm->PacketIncoming(m_NodeID, (byte*) QueryHit, nLength, nError, false);
        return;
    }

    // we will in any case check the hit against searches
    m_pComm->OnQueryHit(QueryHit, this, false);
    ++m_nStatisticsSec[NR_QREPLY_IN];

    // go on with standard routing stuff
    key_Value* pRouteKey      = m_pComm->m_TableRouting.FindValue(&QueryHit->Header.Guid);
    key_Value* pLocalRouteKey = m_pComm->m_TableLocal.FindValue(&QueryHit->Header.Guid);

    if(pLocalRouteKey)
    {
        int i = 0;

        // Check for query hits we sent out
        if(pLocalRouteKey->ID != 0) //(ID != 0) ???
        {
            if(pLocalRouteKey->ID == m_dwID)
            {
                //nError = ERROR_LOOPBACK;
                //m_pComm->PacketIncoming(m_NodeID, (byte*) QueryHit, nLength, nError, false);
                return;
            }

            //nError = ERROR_ROUTING;
            //m_pComm->PacketIncoming(m_NodeID, (byte*) QueryHit, nLength, nError, false);

            return;
        }
        else
        {
            //if(m_pCore->m_LocalMapping)
            //    MapQueryHit(QueryHit);

            AddGoodStat(QueryHit->Header.Function);
            //m_pComm->PacketIncoming(m_NodeID, (byte*) QueryHit, nLength, nError, true);

            return;
        }
    }

    if(pRouteKey)
    {
        // Add ClientID of packet to push table
        if(!m_pComm->m_TablePush.FindValue( ((GUID*) ((char*)QueryHit + (nLength - 16)))))
            m_pComm->m_TablePush.Insert( ((GUID*) ((char*)QueryHit + (nLength - 16))) , m_dwID);

        // Send it out
        if(QueryHit->Header.Hops < MAX_TTL_SELF && QueryHit->Header.TTL > 0)
            m_pComm->Route_QueryHit(QueryHit, nLength, pRouteKey->ID);

        // Mapping
        //if(m_pCore->m_LocalMapping)
        //    MapQueryHit(QueryHit);

        AddGoodStat(QueryHit->Header.Function);
        //m_pComm->PacketIncoming(m_NodeID, (byte*) QueryHit, nLength, nError, false);

        return;
    }
    else
    {
        //nError = ERROR_ROUTING;
        //m_pComm->PacketIncoming(m_NodeID, (byte*) QueryHit, nLength, nError, false);
        return;
    }
}

void MGnuNode::Receive_RouteTableReset(packet_RouteTableReset* TableReset, UINT Length)
{
    if(m_nPeerMode != CM_LEAF || TableReset->Header.Hops > 0)
        return;

    m_TableInfinity = TableReset->Infinity;
    m_TableLength   = TableReset->le_TableLength;


    // Reset main patch table
    m_pComm->GetWordTable()->ResetTable(m_dwID);

    m_CurrentSeq = 1;
}

void MGnuNode::Receive_RouteTablePatch(packet_RouteTablePatch* TablePatch, UINT Length)
{
    if(m_nPeerMode != CM_LEAF || TablePatch->Header.Hops > 0)
        return;

    // Connection must be closed if patch is out of synch
    if(m_CurrentSeq != TablePatch->SeqNum)
    {
        ForceDisconnect();
        return;
    }

    // If first patch in sequence, reset table
    if(TablePatch->SeqNum == 1)
    {
        if(m_PatchTable)
            delete [] m_PatchTable;

        m_PatchTable   = new char[m_TableLength];
        m_TableNextPos = 0;

        memset(m_PatchTable, 0, m_TableLength);

        // Compressed patch table
        if(m_CompressedTable)
            delete [] m_CompressedTable;

        m_CompressedTable = new BYTE[m_TableLength];
        m_CompressedSize  = 0;
    }


    if(TablePatch->SeqNum <= TablePatch->SeqSize)
    {
        UINT PatchSize = TablePatch->Header.le_Payload - 5;

        // As patches come in, build buffer of data
        if(m_CompressedSize + PatchSize <= m_TableLength)
        {
            memcpy(m_CompressedTable + m_CompressedSize, (BYTE*) TablePatch + 28, PatchSize);

            m_CompressedSize += PatchSize;
        }
        else
        {
            ForceDisconnect();
            TRACE("MGnuNode::Receive_RouteTablePatch(): Patch is too big");
        }
    }

    // Final patch received
    if(TablePatch->SeqNum == TablePatch->SeqSize)
    {
        DWORD UncompressedSize = m_TableLength;

        // Uncompress packet if needed
        if(TablePatch->Compression == 0x1)
        {
            if(uncompress((BYTE*) m_PatchTable, &UncompressedSize, m_CompressedTable, m_CompressedSize) != Z_OK)
            {
                TRACE("MGnuNode::Receive_RouteTablePatch(): Uncompress patch error");
            }
        }
        else
        {
            UncompressedSize = m_CompressedSize;
            memcpy(m_PatchTable, m_CompressedTable, UncompressedSize);
        }


        // Add patch to main patch table
        m_pComm->GetWordTable()->ApplyPatch(this, TablePatch->EntryBits);

        delete [] m_PatchTable;
        delete [] m_CompressedTable;

        m_PatchTable      = NULL;
        m_CompressedTable = NULL;

        m_CurrentSeq = 1;
    }
    else
    {
        m_CurrentSeq++;
    }
}

void MGnuNode::Receive_Unknown(BYTE*, DWORD dwLength)
{

}

/////////////////////////////////////////////////////////////////////////////
// Sending packets

void MGnuNode::Send_Ping(int TTL)
{
    GUID Guid = GUID_NULL;
	CreateGuid(&Guid);
	if (Guid == GUID_NULL)
	{
		POST_ERROR(ES_UNIMPORTANT, "Failed to create a GUID to send a ping");
		return;
	}

    packet_Ping Ping;

    Ping.Header.Guid     = Guid;
    Ping.Header.Function = 0;
    Ping.Header.Hops     = 0;
    Ping.Header.TTL      = TTL;
    Ping.Header.le_Payload  = 0;


    m_pComm->m_TableLocal.Insert(&Guid, 0ul);

    SendPacket(&Ping, 23, PACKET_PING, true);
}

void MGnuNode::Send_Pong(const GUID& Guid, int nHops)
{
    // Build the packet
    packet_Pong Pong;

    Pong.Header.Guid        = Guid;
    Pong.Header.Function    = 0x01;
    Pong.Header.TTL         = nHops;
    Pong.Header.Hops        = 0;
    Pong.Header.le_Payload  = 14;
    Pong.le_Port            = m_pComm->GetPublicPort();
    Pong.Host               = m_pComm->GetPublicIP();
    Pong.le_FileCount       = m_pComm->GetLocalSharedFiles();

    // If we are an ultrapeer, the size field is used as a marker send that info
    if(m_pComm->GetHostMode() == CM_ULTRAPEER)
        Pong.le_FileSize = m_pComm->GetUltrapeerSizeMarker();
    else
        Pong.le_FileSize = m_pComm->GetLocalSharedSize();

    m_pComm->m_TableLocal.Insert(&Guid, m_dwID);

    SendPacket(&Pong, 37, PACKET_PONG, true);
}

void MGnuNode::Send_Host(const Node& Host)
{
	GUID Guid = GUID_NULL;
	CreateGuid(&Guid);
	if (Guid == GUID_NULL)
		return;

	// Build the packet
	packet_Pong Pong;

	Pong.Header.Guid		= Guid;
	Pong.Header.Function	= 0x01;
	Pong.Header.TTL			= 1;
	Pong.Header.Hops		= Host.Distance;
	Pong.Header.le_Payload	= 14;

	Pong.le_Port			= Host.Port;
	Pong.Host				= Host.Host;

	Pong.le_FileCount		= Host.ShareCount;
	Pong.le_FileSize		= Host.ShareSize;

#ifdef _PACKET_DEBUG
	POST_EVENT( MPacketEvent(
		(packet_Header*) &Pong,
		37,
		m_ipHost,
		MSG_PACKET_SENT
	));
#endif

	SendPacket(&Pong, 37, PACKET_PONG, true);
}


void MGnuNode::Send_QueryHit(const QueryComp& Query, const BYTE* pQueryHit, DWORD ReplyLength, BYTE ReplyCount, const CString &MetaTail)
{
    packet_QueryHit*    QueryHit = (packet_QueryHit*)   pQueryHit;
    packet_QueryHitEx*  QHD      = (packet_QueryHitEx*) (pQueryHit + 34 + ReplyLength);


    // Build Query Packet
    int packetLength = 34 + ReplyLength;

    QueryHit->Header.Guid = Query.QueryGuid;
    m_pComm->m_TableLocal.Insert(&Query.QueryGuid, m_dwID);

    QueryHit->Header.Function = 0x81;
    QueryHit->Header.TTL      = Query.nHops;
    QueryHit->Header.Hops     = 0;

    QueryHit->TotalHits     = ReplyCount;
    QueryHit->le_Port       = (WORD) m_pComm->GetPublicPort();
    QueryHit->le_Speed      = GetSpeed();
    QueryHit->Host          = m_pComm->GetPublicIP();

    // Add Query Hit Descriptor
    packetLength += sizeof(packet_QueryHitEx);

    bool Busy = false;
    if(m_pPrefs->m_nMaxUploads)
        if(m_pComm->CountUploads() >= m_pPrefs->m_nMaxUploads)
            Busy = true;

    strcpy((char*) QHD->VendorID, "MUTE");
    QHD->Length     = 4;
    QHD->FlagPush   = true;
    QHD->FlagBad    = true;
    QHD->FlagBusy   = true;
    QHD->FlagStable = true;
    QHD->FlagSpeed  = true;
    QHD->FlagTrash  = 0;

    QHD->Push       = m_pPrefs->m_bBehindFirewall;
    QHD->Bad        = 0;
    QHD->Busy       = Busy;
    QHD->Stable     = true; //m_pCore->m_HaveUploaded;
    QHD->Speed      = false; //m_pCore->m_RealSpeedUp ? true : false;
    QHD->Trash      = 0;

    // Add Metadata to packet
    strcpy((char*) pQueryHit + packetLength, "{deflate}");
    packetLength += 9;

    DWORD MetaSize  = MetaTail.length() + 1; // Plus one for null
    DWORD CompSize  = (DWORD) (MetaSize * 1.2) + 12;

    if(compress((BYTE*)pQueryHit + packetLength, &CompSize, (const BYTE*) MetaTail.c_str(), MetaSize) == Z_OK)
    {
        packetLength += CompSize;
        QHD->le_MetaSize = 9 + CompSize;
    }

    // Add ClientID of this node
    memcpy((BYTE*)pQueryHit + packetLength, m_pComm->GetClientID(), 16);

    packetLength += 16;

    // Send the packet
    QueryHit->Header.le_Payload  = packetLength - 23;

    SendPacket((BYTE*)pQueryHit, packetLength, PACKET_QUERYHIT, true);


    // Testing...
    //byte pTest[4096];
    //memcpy(pTest, pQueryHit, packetLength);
}

void MGnuNode::Send_ForwardQuery(const QueryComp& QueryInfo)
{
	if (QueryInfo.nHops >=  MAX_TTL_SELF)
		return;

	BYTE* rawPacket = (BYTE*) alloca(strlen(QueryInfo.Text)+QueryInfo.ExtendedPart.length()+28);

    packet_Query* Query = (packet_Query*) rawPacket;

    Query->Header.Guid          = QueryInfo.QueryGuid;
    Query->Header.Function      = 0x80;
    Query->Header.TTL           = MAX_TTL_SELF - QueryInfo.nHops;
    Query->Header.Hops          = QueryInfo.nHops;
    //Query->Header.Payload = ?;

    Query->le_Speed = 0;

    UINT PacketSize = 25;
    int nTextLen = strlen(QueryInfo.Text);
    memcpy(rawPacket + PacketSize, QueryInfo.Text, nTextLen);
    PacketSize += nTextLen;

    rawPacket[PacketSize] = 0;
    PacketSize++;

    //#warning wtf is that 
    /*for(int i = 0; i < QueryInfo.Extended.size(); i++)
    {
        for(int j = 0; j < QueryInfo.Extended[i].length(); j++)
        {
            rawPacket[PacketSize] = QueryInfo.Extended[i][j];
            PacketSize++;
        }

        rawPacket[PacketSize] = 0x1C;
        PacketSize++;
    }*/
	if (QueryInfo.ExtendedPart.length())
	{
		memcpy(rawPacket + PacketSize, QueryInfo.ExtendedPart.c_str(), QueryInfo.ExtendedPart.length());
		PacketSize += QueryInfo.ExtendedPart.length();
		rawPacket[PacketSize] = 0x1C;
		PacketSize++;
	}

    rawPacket[PacketSize - 1] = 0;

    Query->Header.le_Payload   = PacketSize - 23;

    ASSERT(PacketSize <= strlen(QueryInfo.Text)+QueryInfo.ExtendedPart.length()+28);

    SendPacket(rawPacket, PacketSize, PACKET_QUERY, true);
}


void MGnuNode::Send_PatchTable()
{
    BYTE* PacketBuff   = new BYTE[1 << TABLE_BITS]; // Used so everything is sent in the correct order
    UINT  NextPos      = 0;

    GUID Guid = GUID_NULL;
    CreateGuid(&Guid);
    if (Guid == GUID_NULL)
    {
		delete [] PacketBuff;
        return;
    }

    // Build the packet
    packet_RouteTableReset Reset;

    Reset.Header.Guid       = Guid;
    Reset.Header.Function   = 0x30;
    Reset.Header.TTL        = 1;
    Reset.Header.Hops       = 0;
    Reset.Header.le_Payload = 6;

    Reset.PacketType        = 0x0;
    Reset.le_TableLength    = 1 << TABLE_BITS;
    Reset.Infinity          = TABLE_INFINITY;

    // Send reset packet so remote host clears entries for us
    memcpy(PacketBuff + NextPos, &Reset, 29);
    NextPos += 29;


    // Compress patch table
    DWORD PatchSize  = 1 << TABLE_BITS;
    DWORD CompSize   = (DWORD) (PatchSize * 1.2) + 12;

    BYTE* CompBuff   = new BYTE[CompSize];

    if(compress(CompBuff, &CompSize, (BYTE*) m_pComm->GetWordTable()->m_PatchTable, PatchSize) != Z_OK)
    {
        TRACE("MGnuNode::Send_PatchTable(): patch compression error");
        delete [] CompBuff;
        delete [] PacketBuff;
        return;
    }

     // Determine how many 1024 byte packets to send
    int SeqSize = (CompSize + (1024 - 1)) >> 10;

    int CopyPos  = 0;
    int CopySize = 0;

    BYTE* RawPacket = new BYTE[1024 + 28];
    packet_RouteTablePatch* PatchPacket = (packet_RouteTablePatch*) RawPacket;


    for(int SeqNum = 1; SeqNum <= SeqSize; SeqNum++)
    {
        if(CompSize - CopyPos < 1024)
            CopySize = CompSize - CopyPos;
        else
            CopySize = 1024;

        // Build packet
        CreateGuid(&Guid);

        PatchPacket->Header.Guid        = Guid;
        PatchPacket->Header.Function    = 0x30;
        PatchPacket->Header.TTL         = 1;
        PatchPacket->Header.Hops        = 0;
        PatchPacket->Header.le_Payload  = 5 + CopySize;

        PatchPacket->PacketType = 0x1;
        PatchPacket->SeqNum         = SeqNum;
        PatchPacket->SeqSize    = SeqSize;

        PatchPacket->Compression = 0x1;
        PatchPacket->EntryBits   = 8;

        memcpy(RawPacket + 28, CompBuff + CopyPos, CopySize);

        memcpy(PacketBuff + NextPos, RawPacket, 28 + CopySize);
        NextPos += 28 + CopySize;

        CopyPos += 1024;
    }

    // This mega packet includes the reset and all patches
    SendPacket(PacketBuff, NextPos, PACKET_QRP, true);

    delete [] PacketBuff;
    delete [] CompBuff;
    delete [] RawPacket;
}

bool MGnuNode::GetAlternateHostList(CString &HostList)
{
    // Give 5 hosts from actual cache and 5 hosts from perm cache

    HostList = m_pHostCatcher->GetNodeList(5);
    if (HostList.size())
    	HostList += ',';

    HostList = m_pHostStore->GetNodeList(5, false);

    return !HostList.empty();
}

bool MGnuNode::GetAlternateSuperList(CString &HostList)
{
    HostList = m_pHostCatcher->GetNodeList(5);
    if (HostList.size())
    	HostList += ',';

    HostList = m_pUltraCatcher->GetNodeList(5, false);

    return !HostList.empty();
}


/////////////////////////////////////////////////////////////////////////////
// Misc functions

void MGnuNode::Refresh()
{
    for(int i = 0; i < MAX_TTL_SELF; i++)
    {
        m_dwFriends[i]      = 0;
        m_dwLibrary[i]      = 0;
        m_dwFiles[i]        = 0;

        // Mapping
        //LocalMap[i].clear();
    }

    //MapLinkList.clear();

    m_dwFriendsTotal = 0;
    m_llLibraryTotal = 0;
    m_dwFilesTotal   = 0;
    m_dwSharingTotal = 0;
    m_nSecsRefresh   = 0;

    //m_pComm->NodeUpdate(m_NodeID);

    Send_Ping(MAX_TTL_SELF);
}

DWORD MGnuNode::GetSpeed()
{
	if(m_pPrefs->m_dwSpeedStat)
		return m_pPrefs->m_dwSpeedStat;
	else
		return m_pPrefs->m_dwSpeedDyn;
}

void MGnuNode::OnTimer()
{
    // Decrement time till next ReSearch is allowed on this socket
    if(m_NextReSearchWait)
        m_NextReSearchWait--;

    // Increment recent queries
    std::vector<RecentQuery>::iterator itRecent;
    for(itRecent = m_RecentQueryList.begin(); itRecent != m_RecentQueryList.end(); itRecent++)
    {
        (*itRecent).SecsOld++;

        if((*itRecent).SecsOld >= 5 * 60)
        {
            m_RecentQueryList.erase(itRecent);
            break;
        }
    }

    // Minute counter
    if(m_nSecNum < 60)
        m_nSecNum++;
    int nHistCounter = m_nSecCounter % 180;
	++m_nSecCounter;
    // Statistics
    for(int i = 0; i < NR_LAST; i++)
    {
		m_llStatisticsTotal   [i] += m_nStatisticsSec[i];
		m_nStatisticsHistTotal[i] += m_nStatisticsSec[i];
		m_nStatisticsHistTotal[i] -= m_nStatisticsHist[i][nHistCounter];
		m_nStatisticsHist[i][nHistCounter] = m_nStatisticsSec[i];
		// rates are always calculated per second
		m_dStatisticsRate[i] = m_nStatisticsHistTotal[i] / min(180.0, 1.0*m_nSecCounter);
	}

	// calculate avg SendQueueSize by weighting the prev value with the current one
	m_nSendQueueSize *= 0x0f;
	for (int i = 0; i < 6; ++i)
		m_nSendQueueSize += m_PacketListLength[i];
	m_nSendQueueSize >>= 4;
	// calculate m_nAvgDroppedPackets in the same way
	m_nAvgDroppedPackets *= 0x0f;
	m_nAvgDroppedPackets += m_nSecDroppedPackets;
	m_nAvgDroppedPackets >>= 4;
	m_nSecDroppedPackets = 0;
	
	/////////////////////////////////////

    // Efficiency calculation
    UINT dPart  = m_StatPings[1] + m_StatPongs[1] + m_StatQueries[1] + m_StatQueryHits[1] + m_StatPushes[1];
    UINT dWhole = m_StatPings[0] + m_StatPongs[0] + m_StatQueries[0] + m_StatQueryHits[0] + m_StatPushes[0];

    if(dWhole)
        m_nEfficiency = dPart * 100 / dWhole;
    else
        m_nEfficiency = 0;
    if (m_nEfficiency > 100)
    	m_nEfficiency = 100;

    //cout << "m_dwSecBytes[0] = " << m_dwSecBytes[0]  << "  m_dwSecBytes[1] = " << m_dwSecBytes[1] << "  m_nEfficiency = " << m_nEfficiency << endl;
    NodeManagement();

    // Reset statistics counters
    for(int i = 0; i < NR_LAST; i++)
		m_nStatisticsSec[i] = 0;

	// may be it's time to unblock receive?
	if (m_bReceiveBlocked &&
		m_dwMaxInBytesPerSec > m_dStatisticsRate[NR_BYTE_IN])
	{
		//cout << "node: unblocking receive because m_dByteRate[0] = " << m_dByteRate[0] << "\n";
		ModifySelectFlags(FD_READ,0);
		m_bReceiveBlocked = false;
	}
}

void MGnuNode::NodeManagement()
{
    if(SOCK_CONNECTING == m_nStatus || SOCK_NEGOTIATING == m_nStatus)
    {
        m_nSecsTrying++;

        if(m_nSecsTrying > CONNECT_TIMEOUT)
        {
            ForceDisconnect();
            return;
        }
    }
    else if(SOCK_CONNECTED == m_nStatus)
    {
        // Check if we need to update the patch table
        if(m_pComm->IsLeaf())
            if(m_PatchUpdateNeeded)
            {
                Send_PatchTable();
            }

        m_PatchUpdateNeeded = false;

        // Normal ping at 10 secs
        if(!m_pComm->IsLeaf() && m_nPeerMode != CM_LEAF)
            if(m_IntervalPing == 45)
            {
                AvoidTriangles();

                Send_Ping(2);

                m_IntervalPing = 0;
            }
            else
                m_IntervalPing++;


        // Drop if not socket gone mute 30 secs
        if(m_nStatisticsSec[NR_BYTE_IN] == 0)
        {
            m_nSecsDead++;
            if(m_nSecsDead == 15)
                Send_Ping(1); // alife ping

            if(m_nSecsDead > 30)
            {
                ForceDisconnect();
                return;
            }
        }
        else
            m_nSecsDead = 0;


        // Re-Search on new connect after 10 seconds
        if(m_nSecsAlive == 10)
        {
            Send_Ping(MAX_TTL_SELF);

            /*for(int i = 0; i < m_pTrans->m_DownloadList.size(); i++)
                m_pTrans->m_DownloadList[i]->IncomingNode(this);*/

            //m_pComm->NodeUpdate(m_NodeID);

            m_nSecsAlive++;
        }
        else if(m_nSecsAlive < 60)
            m_nSecsAlive++;
    }
}

void MGnuNode::AvoidTriangles()
{
    /*int i, j, k;

    // Clean old nodes from vector
    for(i = 0; i < 2; i++)
    {
        std::list<MapNode> TempNodes;

        // erase doesnt work so we have to manually move through list
        while(NearMap[i].size())
        {
            if(NearMap[i].back().MapID <= 3)
            {
                NearMap[i].back().MapID++; // MapID is used as an age
                TempNodes.push_front(NearMap[i].back());
            }

            NearMap[i].pop_back();
        }

        while(TempNodes.size())
        {
            NearMap[i].push_back(TempNodes.front());
            TempNodes.pop_front();
        }
    }


    // Check for collisions with nodes
    bool Active = false;

    for(i = 0; i < m_pComm->m_NodeList.size(); i++)
    {
        MGnuNode* pNode = m_pComm->m_NodeList[i];

        // Only check with nodes that are after this node's position in the list
        if(pNode == this)
            Active = true;

        else if(Active && pNode->m_nPeerMode != CLIENT_LEAF)
        {
            for(j = 0; j < NearMap[0].size(); j++)
                for(k = 0; k < pNode->NearMap[0].size(); k++)
                    if(NearMap[0][j].Host.S_addr == pNode->NearMap[0][k].Host.S_addr && NearMap[0][j].Port == pNode->NearMap[0][k].Port)
                    {
                        m_pCache->RemoveIP(pNode->m_ipHost);
                        pNode->Close();
                    }

            for(j = 0; j < NearMap[0].size(); j++)
                for(k = 0; k < pNode->NearMap[1].size(); k++)
                    if(NearMap[0][j].Host.S_addr == pNode->NearMap[1][k].Host.S_addr && NearMap[0][j].Port == pNode->NearMap[1][k].Port)
                    {
                        m_pCache->RemoveIP(pNode->m_ipHost);
                        pNode->Close();
                    }

            for(j = 0; j < NearMap[1].size(); j++)
                for(k = 0; k < pNode->NearMap[0].size(); k++)
                    if(NearMap[1][j].Host.S_addr == pNode->NearMap[0][k].Host.S_addr && NearMap[1][j].Port == pNode->NearMap[0][k].Port)
                    {
                        m_pCache->RemoveIP(pNode->m_ipHost);
                        pNode->Close();
                    }

            for(j = 0; j < NearMap[1].size(); j++)
                for(k = 0; k < pNode->NearMap[1].size(); k++)
                    if(NearMap[1][j].Host.S_addr == pNode->NearMap[1][k].Host.S_addr && NearMap[1][j].Port == pNode->NearMap[1][k].Port)
                    {
                        m_pCache->RemoveIP(pNode->m_ipHost);
                        pNode->Close();
                    }
        }
    }*/
}

void MGnuNode::MapPong(packet_Pong* Pong, bool Near)
{
    /*MapNode Node;
    Node.Host      = Pong->Host;
    Node.Port      = Pong->Port;
    Node.FileSize  = Pong->FileSize;
    Node.FileCount = Pong->FileCount;

    // For pongs 2 hops away or less
    if(Near)
    {
        if(Pong->Header.Hops == 1)
            if(m_nPort != Pong->Port)
            {
                m_nPort = Pong->Port;
                m_pComm->NodeUpdate(m_NodeID);
            }

        // Using MapID as age counter
        Node.MapID = 0;

        for(int i = 0; i < NearMap[Pong->Header.Hops - 1].size(); i++)
            if(NearMap[Pong->Header.Hops - 1][i].Host.S_addr == Pong->Host.S_addr && NearMap[Pong->Header.Hops - 1][i].Port == Pong->Port)
            {
                NearMap[Pong->Header.Hops - 1][i].MapID = 0;
                return;
            }

        NearMap[Pong->Header.Hops - 1].push_back(Node);
        return;
    }


    // Check if node has already been substituted with a temp node
    // If so replace with this one
    if(LocalMap[Pong->Header.Hops - 1].size() == 1)
    {
        // Gets the node from the table with the corresponding ID of the 1st node in the local map list
        MapNode TempNode = m_pComm->MapTable[m_pComm->GetNodeMap(LocalMap[Pong->Header.Hops - 1][0])];

        if(TempNode.Host.S_addr == StrtoIP("255.255.255.255").S_addr &&
           TempNode.Port == 6666)
        {
            // Replace the node
            Node.MapID = TempNode.MapID;
            m_pComm->ReplaceNodeMap(Node);

            return;
        }
    }

    // Add node to local and global map list
    m_pComm->AddNodeMap(Node);
    LocalMap[Pong->Header.Hops - 1].push_back(Node.MapID);

    // If this node should be mapped
    if(Pong->Header.Hops > 1 && Pong->Header.Hops < MAX_TTL)
    {

        UINT parent = 0;

        // Check to make sure this node will have parents
        if(LocalMap[Pong->Header.Hops - 2].size())
            parent = rand() % LocalMap[Pong->Header.Hops - 2].size() + 0;

        // If not make a temp parent
        else
        {
            MapNode RandNode;
            RandNode.Host        = StrtoIP("255.255.255.255");
            RandNode.Port    = 6666;
            RandNode.FileSize    = rand() % 5000000 + 10000;
            RandNode.FileCount   = rand() % 5000 + 0;

            // Add the temp
            m_pComm->AddNodeMap(RandNode);
            LocalMap[Pong->Header.Hops - 2].push_back(RandNode.MapID);

            // Connect the temp with the one before it
            if(Pong->Header.Hops > 2)
                if(LocalMap[Pong->Header.Hops - 3].size())
                {
                    UINT grandparent = rand() % LocalMap[Pong->Header.Hops - 3].size() + 0;

                    MapLink MapEntry;
                    MapEntry.ID           = RandNode.MapID;
                    MapEntry.ParentID = LocalMap[Pong->Header.Hops - 3][grandparent];
                    MapEntry.Hops     = Pong->Header.Hops - 1;

                    MapLinkList.push_back(MapEntry);
                }
        }

        MapLink MapEntry;
        MapEntry.ID           = Node.MapID;
        MapEntry.ParentID = LocalMap[Pong->Header.Hops - 2][parent];
        MapEntry.Hops     = Pong->Header.Hops;

        MapLinkList.push_back(MapEntry);
    }*/
}

void MGnuNode::MapQueryHit(packet_QueryHit* QueryHit)
{
    // Find node in table
    /*for(int i = 0; i < m_pComm->MapTable.size(); i++)
        if(QueryHit->Host.S_addr == m_pComm->MapTable[i].Host.S_addr)
            if(QueryHit->Port == m_pComm->MapTable[i].Port)
            {
                // Get Host of packet
                byte* Packet   = (byte*) QueryHit;

                int    HitsLeft = QueryHit->TotalHits, pos;
                DWORD  NextPos  = 34;
                DWORD  Length   = QueryHit->Header.Payload + 23;

                // Find start of QHD
                int ItemCount = 0;
                for(pos = 42; pos < Length - 16; pos++)
                    if(Packet[pos] == 0 && Packet[pos + 1] == 0)
                    {
                        ItemCount++;

                        if(ItemCount != QueryHit->TotalHits)
                            pos += 9;
                        else
                            break;
                    }


                packet_QueryHitEx* QHD = (packet_QueryHitEx*) &Packet[pos + 2];

                CString Vendor( (char*) QHD->VendorID, 4);

                if(ValidVendor(Vendor))
                    memcpy(m_pComm->MapTable[i].Client, (LPCSTR) Vendor, 4);

                return;
            }*/
}

// Pong Caching
#warning Pong Caching !!!
/*struct MapPongList
{
        IP   Host;
        UINT Port;

        UINT FileCount;
        UINT FileSize;

        MGnuNode* PrimNode;
};
void MGnuNode::Send_Pong(MapPongList* pPongCache)
{
    packet_Pong Pong;

    Pong.Header.Guid        = m_LastGuid;
    Pong.Header.Function    = 0x01;
    Pong.Header.TTL         = m_LastHops;
    Pong.Header.Hops        = 0;
    Pong.Header.le_Payload  = 14;

    Pong.le_Port            = pPongCache->Port;
    Pong.Host               = pPongCache->Host;

    Pong.le_FileCount       = pPongCache->FileCount;
    Pong.FileSize           = pPongCache->FileSize;

    m_pComm->m_TableLocal.Insert(&m_LastGuid, m_NodeID);
    SendPacket(&Pong, 37, PACKET_PONG, true);
}*/

bool MGnuNode::IsRecentQuery(GUID Guid)
{
    for(int i = 0; i < m_RecentQueryList.size(); i++)
        if(m_RecentQueryList[i].Guid == Guid)
            return true;

    return false;
}

int MGnuNode::UpdateStats(int type)
{
    bool Clean = true;
    if(m_StatElements < PACKETCACHE_SIZE)
    {
        Clean = false;
        m_StatElements++;
    }

    int StatPos = m_StatPos;

    if(Clean)
        RemovePacket(m_StatPos);

    ASSERT(m_StatPos>=0);
    ASSERT(m_StatPos<PACKETCACHE_SIZE);
    m_StatPackets[m_StatPos][0] = type;

    m_StatPos = (m_StatPos + 1) % PACKETCACHE_SIZE;

    switch(type)
    {
    case 0x00:
        m_StatPings[0]++;
        break;
    case 0x01:
        m_StatPongs[0]++;
        break;
    case 0x80:
        m_StatQueries[0]++;
        break;
    case 0x81:
        m_StatQueryHits[0]++;
        break;
    case 0x40:
        m_StatPushes[0]++;
        break;
    }


    return StatPos;
}

void MGnuNode::AddGoodStat(int type)
{
	ASSERT(m_StatPos>=0);
    ASSERT(m_StatPos<PACKETCACHE_SIZE);
    m_StatPackets[m_StatPos][1] = 1;

    switch(type)
    {
    case 0x00:
        m_StatPings[1]++;
        break;
    case 0x01:
        m_StatPongs[1]++;
        break;
    case 0x80:
        m_StatQueries[1]++;
        break;
    case 0x81:
        m_StatQueryHits[1]++;
        break;
    case 0x40:
        m_StatPushes[1]++;
        break;
    }
}

void MGnuNode::RemovePacket(int pos)
{
	ASSERT(pos>=0);
    ASSERT(pos<PACKETCACHE_SIZE);
    switch(m_StatPackets[pos][0])
    {
    case 0x00:
        m_StatPings[0]--;

        if(m_StatPackets[pos][1])
            m_StatPings[1]--;

        break;
    case 0x01:
        m_StatPongs[0]--;

        if(m_StatPackets[pos][1])
            m_StatPongs[1]--;

        break;
    case 0x80:
        m_StatQueries[0]--;

        if(m_StatPackets[pos][1])
            m_StatQueries[1]--;

        break;
    case 0x81:
        m_StatQueryHits[0]--;

        if(m_StatPackets[pos][1])
            m_StatQueryHits[1]--;

        break;
    case 0x40:
        m_StatPushes[0]--;

        if(m_StatPackets[pos][1])
            m_StatPushes[1]--;

        break;
    }

    m_StatPackets[pos][1] = 0;
}

#if 0

#include <sys/ioctl.h>
#include <fcntl.h>

#include <ctype.h>

#include "mutella.h"
#include "structures.h"

#include "asyncsocket.h"
#include "packet.h"

#include "rcobject.h"
#include "gnuupload.h"
#include "gnushare.h"
#include "gnucache.h"
#include "gnuhash.h"
#include "gnudirector.h"
#include "gnunode.h"
#include "event.h"
#include "conversions.h"
#include "common.h"
#include "property.h"
#include "preferences.h"
#include "messages.h"

#ifdef _PACKET_DEBUG
MPacketEvent::MPacketEvent(packet_Header* pPH, DWORD length, IP host, DWORD dwID) :
		MEvent(ET_MESSAGE, ES_DEBUG, dwID, MSGSRC_NODE),
		m_dwLength(length),
		m_ipHost(host),
		m_nTTL(pPH->TTL),
		m_nHops(pPH->Hops),
		m_nFunction(pPH->Function)
{
}

CString MPacketEvent::Format()
{
		CString s;
		CString s1;
		switch(m_dwID)
		{
			case MSG_PACKET_RECEIVED: s = ">><<"; break;
			case MSG_PACKET_SENT:     s = "<<>>"; break;
			case MSG_PACKET_REMOVED_GOOD: s = "####"; break;
			case MSG_PACKET_REMOVED_BAD:  s = "-!!-"; break;
			case MSG_PACKET_RECEIVED_UNKNOWN: s = ">??<"; break;
		}
		switch(m_nFunction)
		{
			case 0x00: s += " PING"; break;
			case 0x01: s += " PONG"; break;
			case 0x40: s += " PUSH"; break;
			case 0x80: s += " QUER"; break;
			case 0x81: s += " REPL"; break;
			default:  s1.format(" [%02x]", m_nFunction); s += s1; break;
		}
		s += " ";
		s1.format("len:%6d ip:%15s ttl:%3d hops:%d", m_dwLength, Ip2Str(m_ipHost).c_str(), m_nTTL, m_nHops);
		s += s1;
		return s;
}
#endif //_PACKET_DEBUG

/////////////////////////////////////////////////////////////////////////////
// MGnuNode

MGnuNode::MGnuNode(MGnuDirector* pComm, IP ipHost, UINT Port, bool bInbound, bool bForceV4)
{
	m_pDirector  = pComm;
	m_pPrefs = m_pDirector->GetPrefs();
	m_pHostCatcher = m_pDirector->GetHostCatcher();
	m_pHostStore = m_pDirector->GetHostStore();
	m_pShare = m_pDirector->GetShare();

	m_dwExtraLength  = 0;

	// Node Info
	m_nStatus	= SOCK_CONNECTING;
	m_nSecsTrying   = 0;
	m_nSecsDead		= 0;

	m_sHost		= Ip2Str(ipHost);
	m_ipHost	= ipHost;
	m_nPort		= Port;
	m_bInbound  = bInbound;
	m_dwVersion = 4;
	m_bHaveReceivedPongs = false;
	m_bHaveReceivedPings = false;
	m_dwUptime	= xtime();

	m_dwID = 0;

	m_bForceV4 = bForceV4;

	int i;
	for(i = 0; i < MAX_TTL_SELF; i++)
	{
		m_dwFriends[i]		= 0;
		m_dwLibrary[i]		= 0;
		m_dwFiles[i]		= 0;
	}

	m_dwFriendsTotal = 0;
	m_llLibraryTotal = 0;
	m_dwFilesTotal = 0;
	m_dwSharingTotal = 0;
	// bandwidth and and packet statistics
	// [0] is Received, [1] is Sent
	for(i = 0; i < 2; i++)
	{
		m_dwSecPackets[i] = 0;
		m_dwHistTotalPackets[i] = 0;
		m_dwSecBytes[i] = 0;
		m_dwHistTotalBytes[i] = 0;
		for(int j = 0; j < 180; j++)
		{
			m_adwHistPackets[i][j] = 0;
			m_adwHistBytes[i][j] = 0;
		}
		m_nTotalPackets[i] = 0;
		m_nllTotalBytes[i] = 0;
		m_dPacketRate[i] = 0;
		m_dByteRate[i] = 0;
	}
	m_dwSecReplyPackets = 0;
	for(i = 0; i < 180; i++)
		m_adwHistReplyPackets[i] = 0;
	m_dwHistTotalReplyPackets = 0;
	m_dReplyReceiveRate = 0;
	//
	m_nBadPackets = 0;
	m_nRoutingErrors = 0;
	m_nLoopBackErrors = 0;
	m_nDuplicatePackets = 0;
	m_nDroppedPackets = 0;
	m_nSendQueueSize = 0;

	m_nSecCounter = 0;

    // bandwidth limiting
	m_dwMaxInBytesPerSec = 0;
	m_bReceiveBlocked = false;

	// buffers
	m_pReceiveBuffer = new BYTE[65536];
	ASSERT(m_pReceiveBuffer);
	m_pExtraBuffer = new BYTE[65536];
	ASSERT(m_pExtraBuffer);

	m_sRemoteClient = "Unknown";
}

MGnuNode::~MGnuNode()
{
	ASSERT(m_pDirector);
	ASSERT(!m_pDirector->IsInTheList(this));
	MLock lock(m_sendQueueMutex);
	while (m_sendQueue.size())
	{
		delete [] m_sendQueue.front().pData;
		m_sendQueue.pop();
	}
	delete [] m_pReceiveBuffer;
	delete [] m_pExtraBuffer;
}

int MGnuNode::Send(const void* lpBuf, int nBufLen, int nFlags /*=0*/ )
{
	ASSERT(nFlags == 0);
	MLock lock(m_sendQueueMutex);
	// dont send packets, only enqueue them
	if (m_sendQueue.size()>=256)
	{
		// drop
		m_nDroppedPackets++;
		return nBufLen;
	}
	QueuedPacket qp;
	qp.size = nBufLen;
	qp.pData = new char[nBufLen];
	ASSERT(qp.pData);
	memcpy(qp.pData, lpBuf, nBufLen);
	m_sendQueue.push(qp);
	// ensure that OnSend event will be polled
	ModifySelectFlags(FD_WRITE,0);
	return nBufLen;
}

void MGnuNode::OnSend(int nErrorCode)
{
	MLock lock(m_sendQueueMutex);
	// how to measure ammount of bytes to send?
	int nRes;
	int olderrno = 0;
	errno = 0;
	int nError;
	socklen_t nSize;

#ifdef _DEBUG
	int nFlags = fcntl(m_hSocket,F_GETFL,0);
	ASSERT(nFlags == -1 || (O_NONBLOCK & nFlags));
#endif

	while (m_sendQueue.size())
	{
		nRes = MAsyncSocket::Send(m_sendQueue.front().pData, m_sendQueue.front().size);

		if (olderrno!=errno)
		{
			//printf("OnSend error changed: %d, was %d\n",errno,olderrno);
			ForceDisconnect();
			return;
		}
		olderrno = errno;
		// addidional checks
		nSize = sizeof(nError);
		getsockopt(m_hSocket, SOL_SOCKET,SO_ERROR,(char*)&nError,&nSize);
		ASSERT(nSize == sizeof(int));
		if (nRes != m_sendQueue.front().size || nError)
			if (nError==EAGAIN || nError==EWOULDBLOCK)
			{
				TRACE4("wouldblock, nRes=",nRes," size=",m_sendQueue.front().size);
				return;
			}
			else
			{
				//printf("OnSend: nError=%d, errno=%d\n", nError, errno);
				ForceDisconnect();
				return;
			}
		delete [] m_sendQueue.front().pData;
		m_pDirector->m_dwConnBytesSend += m_sendQueue.front().size;
		m_dwSecBytes[1] += m_sendQueue.front().size;
		m_dwSecPackets[1]++;

		m_sendQueue.pop();
	}
	// save cpu on callbacks when we have nothing to send
	ModifySelectFlags(0, FD_WRITE);
}

/////////////////////////////////////////////////////////////////////////////
// MGnuNode member functions

void MGnuNode::OnConnect(int nErrorCode)
{
	// get localhost IP
	UINT    Port;
	CString sLocalHost;
	if (!GetSockName(sLocalHost, Port))
	{
		ForceDisconnect();
		return;
	}

	if (!m_pPrefs->m_bAcceptRemoteIpHeader)
	{
		IP ipLocalHost;
		VERIFY(Str2Ip(sLocalHost,ipLocalHost));
		m_pPrefs->m_ipLocalHost = ipLocalHost;
	}

	// Get Remote host
	if (!GetPeerName(m_sHost, m_nPort))
	{
		ForceDisconnect();
		return;
	}
	VERIFY(Str2Ip(m_sHost, m_ipHost));

	// Send handshake
	Send_Hello(!m_bForceV4);
	//
	m_nSecsTrying = 0;
	m_nStatus = SOCK_NEGOTIATING;

	MAsyncSocket::OnConnect(nErrorCode);
}

void MGnuNode::OnReceive(int nErrorCode)
{
	int dwOffset = 0, dwBuffLength = 0;
	BYTE* pDataToSplit;

	// Add any unprocessed BYTEs from last receive
	if(m_dwExtraLength)
	{
		ASSERT(m_dwExtraLength>0);
		ASSERT(m_dwExtraLength<=65536);
		//memcpy(m_pReceiveBuffer, m_pExtra, m_dwExtraLength);
		swap(m_pReceiveBuffer, m_pExtraBuffer);
		dwOffset = m_dwExtraLength;
		m_dwExtraLength = 0;
	}
	// check if we are limiting the bandwidth
	int nBytesToReceive = 65536 - dwOffset;
	if (m_dwMaxInBytesPerSec > 0 &&
		m_dwMaxInBytesPerSec - m_dwSecBytes[0] < nBytesToReceive )
		nBytesToReceive = m_dwMaxInBytesPerSec - m_dwSecBytes[0];
	// receive
	dwBuffLength = Receive(m_pReceiveBuffer+dwOffset, nBytesToReceive);
    //printf("node received %d bytes\n", dwBuffLength);
	// Handle Errors
	switch (dwBuffLength)
	{
	case 0:
		//printf("on receive: no data. errno=%d\n", errno);
		ForceDisconnect();
		return;
	case SOCKET_ERROR:
		ForceDisconnect();
		return;
	}
	dwBuffLength += dwOffset;
	pDataToSplit = m_pReceiveBuffer;

	if(m_nStatus == SOCK_NEGOTIATING)
	{
		// we are in handshake mode
		CString Data((char*) m_pReceiveBuffer, dwBuffLength);
		if(!ParseHandshake(Data, m_pReceiveBuffer, dwBuffLength, true))
		{
			ASSERT(m_nStatus != SOCK_CONNECTED);
			if( dwBuffLength > 4096)
			{
				ForceDisconnect();
				return;
			}
			else
			{
				swap(m_pReceiveBuffer, m_pExtraBuffer);
				m_dwExtraLength = dwBuffLength;
			}
		}
	} else if(m_nStatus == SOCK_CONNECTED)
	{
		m_dwExtraLength = SplitBundle(pDataToSplit, dwBuffLength);
		ASSERT(m_dwExtraLength <= dwBuffLength);
		if (0 != m_dwExtraLength)
		{
			if(65536 > m_dwExtraLength)
			{
				if (dwBuffLength == m_dwExtraLength && pDataToSplit == m_pReceiveBuffer)
					swap(m_pReceiveBuffer, m_pExtraBuffer);
				else
				{
					ASSERT(dwBuffLength<=65536);
					memcpy(m_pExtraBuffer, pDataToSplit + (dwBuffLength - m_dwExtraLength), m_dwExtraLength); //TODO: check bounds!!
				}
			}
			else
			{
				TRACE("OnReceive: failed to split the bundle");
				ForceDisconnect();
				return;
			}
		}
	}

	// Bandwidth stats
	m_dwSecBytes[0] += dwBuffLength;
	m_pDirector->m_dwConnBytesRecv += dwBuffLength;

	// Block further receives if
	if (m_dwMaxInBytesPerSec > 0 &&
		m_dwMaxInBytesPerSec <= m_dwSecBytes[0])
	{
    //cout << "node: blocking receive\n";
		ModifySelectFlags(0,FD_READ);
		m_bReceiveBlocked = true;
	}

	MAsyncSocket::OnReceive(nErrorCode);
}

void MGnuNode::OnClose(int nErrorCode)
{
	m_nStatus = SOCK_CLOSED;

	m_pDirector->RemoveNode(this);

	MAsyncSocket::OnClose(nErrorCode);
}

/////////////////////////////////////////////////////////////////////////////
// New connections

#define NEW_CONNECT_STRING     "GNUTELLA CONNECT/0."
#define OLD_CONNECT_STRING     "GNUTELLA CONNECT/0.4"
#define NEW_HANDSHAKE_TERM     "\r\n\r\n"
#define OLD_HANDSHAKE_TERM     "\n\n"
#define NEW_CONNECT_REPLY1     "GNUTELLA/0."
#define NEW_CONNECT_REPLY2     " 200 OK\r\n"
#define NEW_CONNECT_REPLY_FULL " 503"
#define OLD_CONNECT_REPLY      "GNUTELLA OK"
#define SELF_VERSION       "6"
#define X_TRY_HEADER       "X-Try:"

bool MGnuNode::ParseHandshake(const CString& Data, BYTE* pStream, int nStreamLength, bool bHandshakeInBuf)
{
	int i,n;
	CString sHandshake;
	CString s;

	// Version 6 or later
	if((i=Data.find(NEW_HANDSHAKE_TERM)) != -1)
	{
		i += strlen(NEW_HANDSHAKE_TERM);
		sHandshake = Data.substr(0, i);
		m_sHandshake += sHandshake;
		// Connect string, GNUTELLA CONNECT/0.x\r\n
		if(m_bInbound && ((n=sHandshake.find(NEW_CONNECT_STRING)) != -1))
		{
			m_dwVersion = sHandshake[n+strlen(NEW_CONNECT_STRING)]-'0';
			//TRACE4("incomming connection from version ", m_dwVersion, " IP:",m_sHost);
			Send_ConnectOK(true, false);
			return true;
		}
		// Ok string, GNUTELLA/0.6 200 OK\r\n
		else if((n=sHandshake.find(NEW_CONNECT_REPLY1)) != -1)
		{
			n += strlen(NEW_CONNECT_REPLY1);
			m_dwVersion = sHandshake[n]-'0';
			//TRACE4("accepting connection with version ", m_dwVersion, " IP:",m_sHost);
			s = sHandshake.substr(n+1);
			if (s.find(NEW_CONNECT_REPLY2) != -1)
			{
				if (m_bInbound)
				{
					// Stream begins
					if (bHandshakeInBuf)
					{
						ASSERT(i <= nStreamLength);
						m_dwExtraLength = nStreamLength - i;
						ASSERT(65536>=m_dwExtraLength);
						memcpy(m_pExtraBuffer, pStream + i, m_dwExtraLength);
					}
					else
					{
						m_dwExtraLength = 0;
						if (nStreamLength)
						{
							m_dwExtraLength = nStreamLength;
							memcpy(m_pExtraBuffer, pStream, nStreamLength);
						}
					}
					//
					//TRACE("connection accepted");
				}
				else
				{
					Send_ConnectOK(true, true);
				}
				SetConnected();
				return true;
			} else if (s.find(NEW_CONNECT_REPLY_FULL) != -1)
			{
				//X-Try: 24.98.16.79:6346,140.209.103.162:6346,129.119.162.121:6346,12.226.214.103:6346,12.248.69.129:6346,65.162.85.213:6346,208.180.252.92:6346,209.208.96.151:6346,80.143.218.228:6641,62.246.104.201:6346,\r\n
				int nPosXTry = s.find(X_TRY_HEADER);
				if (nPosXTry>=0)
				{
					nPosXTry += strlen(X_TRY_HEADER);

					CString sServers = StripAnyOf(s.substr(nPosXTry, s.find("\r\n",nPosXTry)-nPosXTry), ", ");
					sServers += ','; // we cannot count on trailig comma, but it's presence makes parsing easier

					// now split the line by , and add those hosts to the cache
					Node newNode;
					newNode.Ping	   = 0;
					newNode.Speed      = 0;
					newNode.ShareSize  = 0;
					newNode.ShareCount = 0;
					newNode.Distance   = 0;
					CString sHost;
					//
					int nPC = 0;
					int nPtmp;
					bool bOK;
					for (int nP = -1; nPC != -1; nP = nPC)
					{
						nPC = sServers.find(',', nP+1);
						if (nPC>0)
						{
							sHost = sServers.substr(nP+1, nPC-nP-1);
							// get IP and port
							nPtmp = sHost.find(':');
							if (nPtmp > 0)
							{
								newNode.Port       = atol(sHost.substr(nPtmp+1).c_str());
								bOK = Str2Ip(sHost.substr(0,nPtmp), newNode.Host);
							}
							else
							{
								newNode.Port       = 6346;
								bOK = Str2Ip(sHost, newNode.Host);
							}
							if (bOK)
								m_pHostCatcher->UpdateCache(newNode);
						}
					}
				}
			}
		}
	}
	// Version 4 ?
	else if((i=Data.find(OLD_HANDSHAKE_TERM)) != -1)
	{
		i += strlen(OLD_HANDSHAKE_TERM);
		sHandshake = Data.substr(0, i);
		m_sHandshake += sHandshake;
		if(m_bInbound && (sHandshake.find(OLD_CONNECT_STRING) != -1))
		{
			Send_ConnectOK(false, false);
			SetConnected();
			return true;
		}
		else if((!m_bInbound) && (sHandshake.find(OLD_CONNECT_REPLY) != -1))
		{
			// Stream begins
			if (bHandshakeInBuf)
			{
				ASSERT(i <= nStreamLength);
				m_dwExtraLength = nStreamLength - i;
				ASSERT(65536>=m_dwExtraLength);
				memcpy(m_pExtraBuffer, pStream + i, m_dwExtraLength);
			}
			else
			{
				m_dwExtraLength = 0;
				if (nStreamLength)
				{
					m_dwExtraLength = nStreamLength;
					memcpy(m_pExtraBuffer, pStream, nStreamLength);
				}
			}
			SetConnected();
			return true;
		}
	}
	return false;
}

void MGnuNode::SetConnected()
{
	// get localhost IP
	UINT    Port;
	CString sLocalHost;
	if (!GetSockName(sLocalHost, Port))
	{
		ForceDisconnect();
		return;
	}

	if (!m_pPrefs->m_bAcceptRemoteIpHeader)
	{
		IP ipLocalHost;
		VERIFY(Str2Ip(sLocalHost,ipLocalHost));
		m_pPrefs->m_ipLocalHost = ipLocalHost;
	}

	// Get Remote host
	if (!GetPeerName(m_sHost, m_nPort))
	{
		ForceDisconnect();
		return;
	}
	VERIFY(Str2Ip(m_sHost, m_ipHost));

	if (m_bInbound)
	{
		// the host has proven its gnutella abilities
		// however, it is not clear if it is reacheable for us
		// anyway, we dont know the listening port

		// If too may nodes are connected, send list of pongs
		if((m_pPrefs->m_nMaxConnects > 0 && m_pDirector->CountConnections() > m_pPrefs->m_nMaxConnects) ||
		   (m_pPrefs->m_nMaxIncomingConns > 0 && m_pDirector->CountIncomingConns() > m_pPrefs->m_nMaxIncomingConns) ||
		   (m_pPrefs->m_nMaxConnPerSubnetA > 0 && m_pDirector->CountConnPerSubnet(8, m_ipHost) > m_pPrefs->m_nMaxConnPerSubnetA) ||
		   (m_pPrefs->m_nMaxConnPerSubnetB > 0 && m_pDirector->CountConnPerSubnet(16,m_ipHost) > m_pPrefs->m_nMaxConnPerSubnetB) ||
		   (m_pPrefs->m_nMaxConnPerSubnetC > 0 && m_pDirector->CountConnPerSubnet(24,m_ipHost) > m_pPrefs->m_nMaxConnPerSubnetC) )
		{
			Send_HostCache(); // this also disconnects the node
			return;
		}
	}
	else
	{
		// the host has proven its gnutella abilities
		m_pHostStore->Add(m_ipHost, m_nPort);
	}

	m_nStatus = SOCK_CONNECTED;
	//m_pComm->NodeMessage(SOCK_UPDATE, NULL);

	Send_Ping(MAX_TTL_SELF);

	//
	//cout << "HANDSHAKE:\n" << m_sHandshake << ":END\n";
	// parse the handshake buffer to find out whatever we find approapiate
	if (m_dwVersion >= 6)
	{
		int nStartPos = -1;
		int nEndPos = -1;
		if (m_bInbound)
		{
			CString sTmp = NEW_CONNECT_STRING;
			sTmp += DWrdtoStr(m_dwVersion);
			nStartPos = m_sHandshake.find(sTmp);
			if (nStartPos>=0)
				nStartPos = m_sHandshake.find("\r\n", nStartPos + sTmp.length());
			if (nStartPos>=0)
			{
				nEndPos = m_sHandshake.find(NEW_CONNECT_REPLY1, nStartPos);
			}
		}
		else
		{
			CString sTmp = NEW_CONNECT_REPLY1;
			sTmp += DWrdtoStr(m_dwVersion);
			nStartPos = m_sHandshake.find(sTmp);
			if (nStartPos>=0)
				nStartPos = m_sHandshake.find("\r\n", nStartPos + sTmp.length());
			if (nStartPos>=0)
			{
				nEndPos = m_sHandshake.find(NEW_CONNECT_REPLY1, nStartPos);
			}
		}
		//
		if (nStartPos>=0 && nEndPos>=0)
		{
			CString sRemoteCaps = m_sHandshake.substr(nStartPos, nEndPos - nStartPos);
			//cout << "REMOTE CAPS:" << sRemoteCaps << ":END\n";
			sRemoteCaps.make_lower();
			//
			map<CString, CString> values;
			if (parse_params(sRemoteCaps, "\r\n", ": ", values))
			{
				// copy parameters to the internal structures
				map<CString, CString>::iterator it = values.find("user-agent");
				if (it != values.end())
				{
					m_sRemoteClient = it->second;
					if (m_sRemoteClient.length() > 25)
					{
						int nPos = m_sRemoteClient.find(' ');
						if (nPos>0)
							m_sRemoteClient.cut(nPos);
					}
				}
			}
		}
	}
	else
	{
		// 0.4 client
		m_sRemoteClient = "unknown 0.4";
	}
}

void MGnuNode::Send_Hello(bool bVersion6)
{
	CString sHandshake;
	if (bVersion6)
	{
		CString sMuteVersion = VERSION;
		//
		sHandshake = NEW_CONNECT_STRING SELF_VERSION;
		sHandshake += "\r\n";
		sHandshake += CString("Node: ") +
		          Ip2Str(m_pDirector->GetPublicIP()) + ":" +
		          DWrdtoStr(m_pDirector->GetPublicPort()) + "\r\n";
		sHandshake += CString("User-Agent: Mutella-") + sMuteVersion + "\r\n";
		sHandshake += "X-Ultrapeer: False\r\n";
		sHandshake += "\r\n";
	}
	else
	{
		sHandshake += OLD_CONNECT_STRING OLD_HANDSHAKE_TERM;
	}
	Send(sHandshake.c_str(), sHandshake.length());
	m_sHandshake += sHandshake;
}

//"Node: %s\r\n"
//"User-Agent: %s\r\n"
//"Pong-Caching: 0.1\r\n"
//"X-Live-Since: %s\r\n"

void MGnuNode::Send_ConnectOK(bool bVersion6, bool bReply)
{
	CString sHandshake;
	CString sMuteVersion = VERSION;

	if(bVersion6 && bReply)
	{
		sHandshake = NEW_CONNECT_REPLY1 SELF_VERSION NEW_CONNECT_REPLY2;
		sHandshake += "\r\n";
	}
	else if(bVersion6 && !bReply)
	{
		sHandshake = NEW_CONNECT_REPLY1 SELF_VERSION NEW_CONNECT_REPLY2;
		sHandshake += CString("User-Agent: Mutella-") + sMuteVersion + "\r\n";
		sHandshake += "\r\n";
	}
	else
	{
		sHandshake = OLD_CONNECT_REPLY OLD_HANDSHAKE_TERM;
	}
	Send(sHandshake.c_str(), sHandshake.length());

	m_sHandshake += sHandshake;
}

/*void MGnuNode::Receive_ConnectOK()
{
	m_nStatus = SOCK_CONNECTED;
	m_pDirector->NodeMessage(SOCK_UPDATE, 0);
	Send_Ping(MAX_TTL_SELF);
	//m_pDirector->SendAllLocalQueries(this);
}*/


////////////////////////////
// packet.handlers

DWORD MGnuNode::SplitBundle(BYTE* bundle, DWORD length)
{
	DWORD Payload = 0;
	DWORD nextPos = 0;

	packet_Header* packet;

	enum status { status_DONE,       status_CONTINUE,
				  status_BAD_PACKET, status_INCOMPLETE_PACKET };

	status theStatus = status_CONTINUE;

	do
	{
		if (nextPos + sizeof(packet_Header) > length)
			theStatus = status_INCOMPLETE_PACKET;
		else
		{
			packet = (packet_Header*) (bundle + nextPos);
			Payload = packet->le_Payload;

			if((packet->Function == 0x00 && Payload ==  0) ||
			   (packet->Function == 0x01 && Payload == 14) ||
			   (packet->Function == 0x40 && Payload == 26) ||
			   (packet->Function == 0x80 && Payload >  2 && Payload < 256) ||
			   (packet->Function == 0x81 && Payload > 26 && Payload < 65536) ||
			   (packet->Function == 0x31 && Payload == 9)  || // comes from bearshare
			   (packet->Function == 0x99 && Payload == 22) || // comes from some 0.4 client
			   (packet->Function == 0x02 && Payload < 256) || // bye packet (as sent by gtk-gnutella)
			   (packet->Function == 0x80 && Payload < 65536) ) // sent by limewire and morpheus
			{
				if (nextPos + sizeof(packet_Header) + Payload <= length)
				{
					HandlePacket(packet, sizeof(packet_Header) + Payload);

					nextPos += sizeof(packet_Header) + Payload;
					if (nextPos == length)
						theStatus = status_DONE;
				}
				else
				{
					theStatus = status_INCOMPLETE_PACKET;
					if (sizeof(packet_Header) + Payload > 65536)
					{
						TRACE("SplitBundle: too big packet has arrived");
						ForceDisconnect();
					}
				}
			}
			else
			{
		    /*if (nextPos < length - sizeof(packet_Header))
					nextPos++;
		    else*/
				theStatus = status_BAD_PACKET;
				TRACE2("SplitBundle: bad or unknown packet from ", m_sRemoteClient);
				TRACE4("   Function: ", (int) packet->Function, "  Playload: ", Payload);
				ForceDisconnect();
				//TRACE("SplitBundle: sync lost");
			}
		}
	} while(status_CONTINUE == theStatus);


	return length - nextPos;
}

void MGnuNode::HandlePacket(packet_Header* packet, DWORD length)
{
	m_dwSecPackets[0]++;

#ifdef _PACKET_DEBUG
	POST_EVENT( MPacketEvent(
		packet,
		length,
		m_ipHost,
		MSG_PACKET_RECEIVED
	));
#endif

	switch(packet->Function)
	{
	case 0x00:
		Receive_Ping((packet_Ping*) packet, length);
		break;

	case 0x01:
		Receive_Pong((packet_Pong*) packet, length);
		break;

	case 0x40:
		Receive_Push((packet_Push*) packet, length);
		break;

	case 0x80:
		Receive_Query((packet_Query*) packet, length);
		break;

	case 0x81:
		Receive_QueryHit((packet_QueryHit*) packet, length);
		break;

	default:
		Receive_Unknown((BYTE *) packet, length);
		break;
	}
}

bool MGnuNode::InspectPacket(packet_Header* packet, DWORD length)
{
	// check if the packet should be dead by now
	if(packet->TTL == 0)
	{
#ifdef _PACKET_DEBUG
		POST_EVENT( MPacketEvent(
			packet,
			length,
			m_ipHost,
			MSG_PACKET_REMOVED_BAD
		));
#endif
		return false;
	}
	if (packet->Hops > MAX_TTL_ACCEPTED)
	{
#ifdef _PACKET_DEBUG
		POST_EVENT( MPacketEvent(
			packet,
			length,
			m_ipHost,
			MSG_PACKET_REMOVED_BAD
		));
#endif
		return false;
	}
	// Increment hops
	packet->Hops++;
	packet->TTL--;
	// If packet has hopped more than MAX_TTL_SELF times kill it
	if(packet->Hops > MAX_TTL_SELF)
	{
#ifdef _PACKET_DEBUG
		POST_EVENT( MPacketEvent(
			packet,
			length,
			m_ipHost,
			MSG_PACKET_REMOVED_BAD
		));
#endif
		return false;
	}
	// Bad TTL or turned over TTL
	if (packet->TTL > MAX_TTL_ACCEPTED)
	{
#ifdef _PACKET_DEBUG
		POST_EVENT( MPacketEvent(
			packet,
			length,
			m_ipHost,
			MSG_PACKET_REMOVED_BAD
		));
#endif
		return false;
	}
	// Reset TTL of packet if it needs to be done
	if(packet->TTL > MAX_TTL_SELF - packet->Hops)
		packet->TTL = MAX_TTL_SELF - packet->Hops;
	return true;
}


/////////////////////////////////////////////////////////////////////////////
// Receiving packets


void MGnuNode::Receive_Ping(packet_Ping* Ping, int nLength)
{
	//printf("Receive Ping\n");
	// part of the hanbshake procedure
	if (!m_bHaveReceivedPings && Ping->Header.Hops == 0)
	{
		m_bHaveReceivedPings = true;
		if ( Ping->Header.Guid.a[8] == 0xFF)
		{
			m_dwVersion = 5+Ping->Header.Guid.a[15];
		}
	}

	// Inspect TTL & Hops
	if(!InspectPacket((packet_Header*) Ping, nLength))
	{
		m_nBadPackets++;
		return;
	}
#ifdef _NO_PING_PONG
	return;
#endif //_NO_PING_PONG

	key_Value* key = m_pDirector->m_TablePing.FindValue(&Ping->Header.Guid);
	key_Value* LocalKey = m_pDirector->m_TableLocal.FindValue(&Ping->Header.Guid);

	if(LocalKey)
	{
#ifdef _PACKET_DEBUG
		POST_EVENT( MPacketEvent(
			(packet_Header*)Ping,
			nLength,
			m_ipHost,
			MSG_PACKET_REMOVED_BAD
		));
#endif
		m_nLoopBackErrors++;
		return;
	}

	// Fresh Ping?
	if(key == NULL)
	{
		m_pDirector->m_TablePing.Insert(&Ping->Header.Guid, this);
		if (m_pPrefs->m_nMaxConnects < 0 || m_pDirector->GetLastConnCount() <= m_pPrefs->m_nMaxConnects * 0.8 ||
			Ping->Header.Hops == 1 /* always reply to the 'alife' ping */ )
			Send_Pong(Ping->Header.Guid, Ping->Header.Hops);
		// Broadcast if still alive
		if(Ping->Header.TTL)
		{
			m_pDirector->Broadcast_Ping(Ping, nLength, this);
			return;
		}
	}
	else
	{
		if(key->Origin == this)
		{
#ifdef _PACKET_DEBUG
			POST_EVENT( MPacketEvent(
				(packet_Header*) Ping,
				nLength,
				m_ipHost,
				MSG_PACKET_REMOVED_BAD
			));
#endif
			m_nDuplicatePackets++;
			return;
		}
		else
		{
#ifdef _PACKET_DEBUG
			POST_EVENT( MPacketEvent(
				(packet_Header*) Ping,
				nLength,
				m_ipHost,
				MSG_PACKET_REMOVED_BAD
			));
#endif
			m_nRoutingErrors++;
			return;
		}
	}
}


void MGnuNode::Receive_Pong(packet_Pong* Pong, int nLength)
{
	//printf("Receive Pong\n");
	// Host Cache
	Node NetNode;
	NetNode.Host		= Pong->Host;
	NetNode.Port		= Pong->le_Port;
	NetNode.Ping		= 0;
	NetNode.Speed		= 0;
	NetNode.Distance	= Pong->Header.Hops;
	NetNode.ShareCount  = Pong->le_FileCount;
	NetNode.ShareSize	= Pong->le_FileSize;
	m_pHostCatcher->UpdateCache(NetNode);
	// part of the hanbshake procedure
	if (!m_bHaveReceivedPongs && Pong->Header.Hops == 0 && Pong->Host.S_addr == m_ipHost.S_addr)
	{
		m_bHaveReceivedPongs = true;
		m_nPort = Pong->le_Port;
		// at this point we find out the listening port for
		// the incomming connections. however, there's no guarantee
		// that the host is reacheble
		// so we only add it if we really need more hosts
		if (m_bInbound && m_pHostStore->IsHalfEmpty())
			m_pHostStore->Add(m_ipHost, m_nPort);
	}

	// Inspect
	if(!InspectPacket((packet_Header*) Pong, nLength))
	{
		m_nBadPackets++;
		return;
	}

	key_Value* key = m_pDirector->m_TablePing.FindValue(&Pong->Header.Guid);
	key_Value* LocalKey = m_pDirector->m_TableLocal.FindValue(&Pong->Header.Guid);

	if(LocalKey)
	{
		// If Pong GUID matches a GUID of a Ping we sent out, its for us
		// Else, its a loop back error

		m_dwFriendsTotal++;
		ASSERT(Pong->Header.Hops <= MAX_TTL_SELF);
		m_dwFriends[Pong->Header.Hops - 1]++;

		DWORD dwLibrary = Pong->le_FileSize;
		DWORD dwFiles = Pong->le_FileCount;
		if (dwLibrary>MAX_SHARE)
			dwLibrary = MAX_SHARE;
		if(dwLibrary)
		{
			m_dwLibrary[Pong->Header.Hops - 1] += dwLibrary;
			m_dwFiles[Pong->Header.Hops - 1] += dwFiles;
			m_llLibraryTotal += dwLibrary;
			m_dwFilesTotal += dwFiles;
			m_dwSharingTotal++;
		}
		return;
	}

	if(key)
	{
		// Send it out
		if(Pong->Header.TTL)
			m_pDirector->Route_Pong(Pong, nLength, key);
		return;
	}
	else
	{
#ifdef _PACKET_DEBUG
		POST_EVENT( MPacketEvent(
			(packet_Header*) Pong,
			nLength,
			m_ipHost,
			MSG_PACKET_REMOVED_BAD
		));
#endif
		m_nRoutingErrors++;
		return;
	}
}

void MGnuNode::Receive_Push(packet_Push* Push, int nLength)
{
	// Host Cache
	Node NetNode;
	NetNode.Host		= Push->Host;
	NetNode.Port		= Push->le_Port;
	NetNode.Ping		= 0;
	NetNode.Speed		= 0;
	NetNode.Distance	= Push->Header.Hops;
	NetNode.ShareCount  = 0;
	NetNode.ShareSize	= 0;
	m_pHostCatcher->UpdateCache(NetNode);

	// Inspect
	if(!InspectPacket((packet_Header*) Push, nLength))
	{
		m_nBadPackets++;
		return;
	}

	// Find packet in hash tables
	key_Value* LocalKey = m_pDirector->m_TableLocal.FindValue(&Push->Header.Guid);
	key_Value* PushKey  = m_pDirector->m_TablePush.FindValue(&Push->ServerID);

	if(LocalKey)
	{
#ifdef _PACKET_DEBUG
		POST_EVENT( MPacketEvent(
			(packet_Header*) Push,
			nLength,
			m_ipHost,
			MSG_PACKET_REMOVED_BAD
		));
#endif
		m_nBadPackets++;
		return;
	}

	// Check ServerID of Push with ClientID of the client
	if(*m_pDirector->GetClientID() == Push->ServerID)
	{
		// Make sure not already pushing file
		if (m_pDirector->CheckIfPushung(Push))
		{
#ifdef _PACKET_DEBUG
			POST_EVENT( MPacketEvent(
				(packet_Header*) Push,
				nLength,
				m_ipHost,
				MSG_PACKET_REMOVED_BAD
			));
#endif
			m_nDuplicatePackets++; // not sure if it is appropriate
			return;
		}

		if (m_pPrefs->m_bQuietMode)
			return;
    TRACE("processing push request");

		// Create upload
		MGnuUpload* pUpload = new MGnuUpload(m_pDirector);
		ASSERT(pUpload);
		pUpload->m_ipHost     = Push->Host;
		pUpload->m_nPort      = Push->le_Port;
		pUpload->m_nFileIndex = Push->le_Index;

		if(!pUpload->Create(0, SOCK_STREAM, FD_READ | FD_CONNECT | FD_CLOSE)) // FD_WRITE shouldn't harm here
		{
			POST_ERROR(ES_UNIMPORTANT, CString("Upload Create Error: ") + DWrdtoStr(pUpload->GetLastError()));
		}

		// Add the node to the linked list
		m_pDirector->AddUpload(pUpload);
		pUpload->PushFile();
		return;
	}

	if(PushKey)
	{
		if(Push->Header.TTL)
			m_pDirector->Route_Push(Push, nLength, PushKey);
		return;
	}
	else
	{
#ifdef _PACKET_DEBUG
		POST_EVENT( MPacketEvent(
			(packet_Header*) Push,
			nLength,
			m_ipHost,
			MSG_PACKET_REMOVED_BAD
		));
#endif
		m_nRoutingErrors++;
		return;
	}
}

void MGnuNode::Receive_Query(packet_Query* Query, int nLength)
{
	// Inspect
	if(!InspectPacket((packet_Header*) Query, nLength))
	{
		m_nBadPackets++;
		return;
	}

	key_Value* key = m_pDirector->m_TableQuery.FindValue(&Query->Header.Guid);
	key_Value* LocalKey = m_pDirector->m_TableLocal.FindValue(&Query->Header.Guid);

	if(LocalKey)
	{
#ifdef _PACKET_DEBUG
		POST_EVENT( MPacketEvent(
			(packet_Header*) Query,
			nLength,
			m_ipHost,
			MSG_PACKET_REMOVED_BAD
		));
#endif
		m_nLoopBackErrors++;
		return;
	}

	// Fresh Query?
	if(key == NULL)
	{
		m_pDirector->m_TableQuery.Insert(&Query->Header.Guid, this);
		m_pDirector->m_nQGood++;
		// spam protection
		int nAlpha = 0;
		int nLen = MIN(Query->Header.le_Payload - 2, MAX_QUERY_LEN);
		char* pSearch = (char*) Query+25;
		for (int i=0;i<nLen;++i)
			if (isalnum(pSearch[i]))
				++nAlpha;
		if (nAlpha<4 || nAlpha < nLen/2)
		{
#ifdef _PACKET_DEBUG
		POST_EVENT( MPacketEvent(
			(packet_Header*) Query,
			nLength,
			m_ipHost,
			MSG_PACKET_REMOVED_BAD
		));
#endif
			// spam
			return;
		}
		// Broadcast if still alive
		if(Query->Header.TTL)
			m_pDirector->Broadcast_Query(Query, nLength, this);

		// Queue to be compared with local files
		if(Query->le_Speed <= GetOurSpeed())
		{
			// WATCH OUT!!! IT CRASHED HERE ONCE
			QueryComp* pCompare = new QueryComp;
			ASSERT(pCompare);
			pCompare->OriginID  = m_dwID;
			pCompare->nHops     = Query->Header.Hops;
			pCompare->QueryGuid = Query->Header.Guid;

			memcpy(pCompare->Query,(char*)pSearch,nLen);
			pCompare->Query[nLen]=0;

			if( (!m_pPrefs->m_bSendOnlyAvail) || // dont check quieries against the file list if busy
			     m_pPrefs->m_nMaxUploads < 0  ||
			     m_pDirector->CountUploads() < m_pPrefs->m_nMaxUploads )
			{
				m_pShare->AppendQuery(pCompare);
			}
			else
				delete pCompare;
		}
		return;
	}
	else
	{
		if(key->Origin == this)
		{
#ifdef _PACKET_DEBUG
			POST_EVENT( MPacketEvent(
				(packet_Header*) Query,
				nLength,
				m_ipHost,
				MSG_PACKET_REMOVED_BAD
			));
#endif
			//printf("R_Q: duplicate packet, conflict with a packet from %d sec before\n   query=`%s'\n", time(NULL)-key->time,((char*)Query)+25);
			m_nDuplicatePackets++;
			m_pDirector->m_nQRepeat++;
			return;
		}
		else
		{
#ifdef _PACKET_DEBUG
			POST_EVENT( MPacketEvent(
				(packet_Header*) Query,
				nLength,
				m_ipHost,
				MSG_PACKET_REMOVED_BAD
			));
#endif
			//printf("R_Q: error routing, conflict with a packet from %d sec before\n   query=`%s'\n", time(NULL)-key->time,((char*)Query)+25);
			m_nRoutingErrors++;
			m_pDirector->m_nQError++;
			return;
		}
	}
}


void MGnuNode::Receive_QueryHit(packet_QueryHit* QueryHit, DWORD dwLength)
{
	// Host Cache
	Node NetNode;
	NetNode.Host		= QueryHit->Host;
	NetNode.Port		= QueryHit->le_Port;
	NetNode.Ping		= 0;
	NetNode.Speed       = QueryHit->le_Speed;
	NetNode.Distance	= QueryHit->Header.Hops;
	NetNode.ShareCount  = 0;
	NetNode.ShareSize   = 0;

	m_pHostCatcher->UpdateCache(NetNode);

	// Inspect
	if(!InspectPacket((packet_Header*) QueryHit, dwLength))
	{
		m_nBadPackets++;
		//
		// still makes sence to check what has been found
		m_pDirector->OnQueryHit(QueryHit, this, false);
		return;
	}
	// make note of the query-hit for statistics
	m_dwSecReplyPackets++;
	// prepare for routing
	key_Value* key = m_pDirector->m_TableQuery.FindValue(&QueryHit->Header.Guid);
	key_Value* LocalKey = m_pDirector->m_TableLocal.FindValue(&QueryHit->Header.Guid);

	if(LocalKey)
	{
		// If QueryHit GUID matches a GUID of a Query we sent out its for us
		m_pDirector->OnQueryHit(QueryHit, this, true);
		return;
	}
	else
	{
		// but we will be little bit nosy and check what the others find
		m_pDirector->OnQueryHit(QueryHit, this, false);
	}

	if(key)
	{
		// Send it out
		if(QueryHit->Header.TTL)
		{
			// Add ClientID of packet to push table
			m_pDirector->m_TablePush.Insert( (GUID*) ((BYTE*)QueryHit + (dwLength - 16)), this);
			m_pDirector->Route_QueryHit(QueryHit, dwLength, key);
			return;
		}
		else
		{
#ifdef _PACKET_DEBUG
			POST_EVENT( MPacketEvent(
				(packet_Header*) QueryHit,
				dwLength,
				m_ipHost,
				MSG_PACKET_REMOVED_BAD
			));
#endif
			m_nBadPackets++;
			return;
		}
	}
	else
	{
#ifdef _PACKET_DEBUG
		POST_EVENT( MPacketEvent(
			(packet_Header*) QueryHit,
			dwLength,
			m_ipHost,
			MSG_PACKET_REMOVED_BAD
		));
#endif
		m_nRoutingErrors++;
		return;
	}
}


void MGnuNode::Receive_Unknown(BYTE* pPacket, DWORD dwLength)
{
	m_nBadPackets++;
#ifdef _PACKET_DEBUG
	POST_EVENT( MPacketEvent(
		(packet_Header*) pPacket,
		dwLength,
		m_ipHost,
		MSG_PACKET_RECEIVED_UNKNOWN
	));
#endif

}


/////////////////////////////////////////////////////////////////////////////
// Sending packets

void MGnuNode::Send_Ping(int TTL)
{
	//printf("Send Ping, TTL=%d\n", TTL);

	GUID Guid = GUID_NULL;
	CreateGuid(&Guid);
	if (Guid == GUID_NULL)
	{
		POST_ERROR(ES_UNIMPORTANT, "Failed to create a GUID to send a ping");
		return;
	}

	packet_Ping Ping;

	Ping.Header.Guid = Guid;
	Ping.Header.Function = 0;
	Ping.Header.Hops = 0;
	Ping.Header.TTL = TTL;
	Ping.Header.le_Payload = 0;

	// TTL of 1 means we are checking if node is responding only
	if(TTL != 1)
		m_pDirector->m_TableLocal.Insert(&Guid, NULL);

#ifdef _PACKET_DEBUG
	POST_EVENT( MPacketEvent(
		(packet_Header*) &Ping,
		23,
		m_ipHost,
		MSG_PACKET_SENT
	));
#endif

	Send(&Ping, 23);
}

void MGnuNode::Send_Pong(GUID Guid, int nHops)
{
    //printf("Send Pong\n");
	// Build the packet
	packet_Pong Pong;

	Pong.Header.Guid		= Guid;
	Pong.Header.Function	= 0x01;
	Pong.Header.TTL			= nHops;
	Pong.Header.Hops		= 0;
	Pong.Header.le_Payload	= 14;

	Pong.le_Port			= m_pDirector->GetPublicPort();
	Pong.Host				= m_pDirector->GetPublicIP();;

	Pong.le_FileCount		= m_pShare->m_dwTotalFiles;
	Pong.le_FileSize		= m_pShare->m_dwTotalSize;

#ifdef _PACKET_DEBUG
	POST_EVENT( MPacketEvent(
		(packet_Header*) &Pong,
		37,
		m_ipHost,
		MSG_PACKET_SENT
	));
#endif

	Send(&Pong, 37);
}

void MGnuNode::Send_QueryHit(QueryComp* pQuery, BYTE* pQueryHit, DWORD &ReplyLength, BYTE &ReplyCount)
{
	int packetLength = 34 + ReplyLength + sizeof(packet_QueryHitEx) + 16;

	packet_QueryHit*  QueryHit = (packet_QueryHit*) pQueryHit;
	packet_QueryHitEx QHD;

	// Build Query Packet
	QueryHit->Header.Guid	    = pQuery->QueryGuid;
	QueryHit->Header.Function   = 0x81;
	QueryHit->Header.TTL	    = pQuery->nHops;
	QueryHit->Header.Hops	    = 0;
	QueryHit->Header.le_Payload = packetLength - 23;

	QueryHit->TotalHits		    = ReplyCount;
	QueryHit->le_Port		    = (WORD) m_pDirector->GetPublicPort();
	QueryHit->Host			    = m_pDirector->GetPublicIP();
	QueryHit->le_Speed		    = GetOurSpeed();

	// Add Query Hit Descriptor
	bool Busy = false;
	if(m_pPrefs->m_nMaxUploads >= 0)
		if(m_pDirector->CountUploads() >= m_pPrefs->m_nMaxUploads)
			Busy = true;

	strcpy((char*) QHD.VendorID, "MUTE");
	QHD.Length		= 2;
	QHD.FlagPush	= 0;//true;
	QHD.FlagBad		= 1;//true;
	QHD.FlagBusy	= 1;//true;
	QHD.FlagStable	= 1;//true;
	QHD.FlagSpeed	= 1;//true;
	QHD.FlagTrash   = 0;
	QHD.Push		= m_pDirector->BehindFirewall();
	QHD.Bad			= 0;//false;
	QHD.Busy		= 0;//false;//Busy;
	QHD.Stable		= 1;//false;//true;//m_pDirector->HaveUploaded();
	QHD.Speed		= 0;//false;//true;//m_pDirector->GetRealSpeed() ? true : false;
	QHD.Trash		= 0;
	//QHD.bug1 = 0xcc;
	//QHD.bug2= 0xcc;

	int pos = 34 + ReplyLength;
	pQueryHit += pos;

	memcpy(pQueryHit, &QHD, sizeof(packet_QueryHitEx)); //TODO: check bounds!!!
	pQueryHit -= pos;

	// Add ID of this node
	pos += sizeof(packet_QueryHitEx);
	pQueryHit += pos;
	memcpy(pQueryHit, m_pDirector->GetClientID(), 16);
	pQueryHit -= pos;

#ifdef _PACKET_DEBUG
	POST_EVENT( MPacketEvent(
		(packet_Header*) pQueryHit,
		packetLength,
		m_ipHost,
		MSG_PACKET_SENT
	));
#endif

	// Send the packet
	Send(pQueryHit,packetLength);
}

void MGnuNode::Send_HostCache()
{
	m_pHostCatcher->SendCache(this, 10);
	//ForceDisconnect();
	m_nStatus = SOCK_CLOSED; // this delays disconnection to the next OnTimer event
}

void MGnuNode::Send_Host(Node Host)
{
	GUID Guid = GUID_NULL;
	CreateGuid(&Guid);
	if (Guid == GUID_NULL)
		return;

	// Build the packet
	packet_Pong Pong;

	Pong.Header.Guid		= Guid;
	Pong.Header.Function	= 0x01;
	Pong.Header.TTL			= 1;
	Pong.Header.Hops		= Host.Distance;
	Pong.Header.le_Payload	= 14;

	Pong.le_Port			= Host.Port;
	Pong.Host				= Host.Host;

	Pong.le_FileCount		= Host.ShareCount;
	Pong.le_FileSize		= Host.ShareSize;

#ifdef _PACKET_DEBUG
	POST_EVENT( MPacketEvent(
		(packet_Header*) &Pong,
		37,
		m_ipHost,
		MSG_PACKET_SENT
	));
#endif

	Send(&Pong, 37);
}

/////////////////////////////////////////////////////////////////////////////
// Misc functions


DWORD MGnuNode::GetOurSpeed()
{
	if(m_pPrefs->m_dwSpeedStat)
		return m_pPrefs->m_dwSpeedStat;
	else
		return m_pPrefs->m_dwSpeedDyn;
}

void MGnuNode::ForceDisconnect(bool bExernal /*=false*/)
{
	//TRACE4("closing connection with version ", m_dwVersion, " IP:",m_sHost);
	if(m_hSocket != INVALID_SOCKET)
	{
		AsyncSelect(FD_CLOSE);
		ShutDown(2);
	}
	Close();
	if (!bExernal)
	{
		if (!m_bForceV4 && !m_bInbound && SOCK_NEGOTIATING == m_nStatus)
			m_pDirector->AddNode(m_ipHost, m_nPort, true);
	}
	m_nStatus = SOCK_CLOSED;
}

void MGnuNode::OnTimer()
{
	m_nSecCounter++;
	int nHistCounter = m_nSecCounter % 180;
	int nLastSecInPackets = m_dwSecPackets[0];
	// When i = 0 its receive stats
    //      i = 1 its send stats
	// Easier to read without the [i]'s, but oh well
	for(int i = 0; i < 2; i++)
	{
		// Packets
		m_nTotalPackets[i] += m_dwSecPackets[i];
		m_dwHistTotalPackets[i] += m_dwSecPackets[i];
		m_dwHistTotalPackets[i] -= m_adwHistPackets[i][nHistCounter];
		m_adwHistPackets[i][nHistCounter] = m_dwSecPackets[i];
		m_dwSecPackets[i] = 0;
		// packet rate
		m_dPacketRate[i] = m_dwHistTotalPackets[i] / min(180.0, 1.0*m_nSecCounter);
		// Bytes
		m_nllTotalBytes[i] += m_dwSecBytes[i];
		m_dwHistTotalBytes[i] += m_dwSecBytes[i];
		m_dwHistTotalBytes[i] -= m_adwHistBytes[i][nHistCounter];
		m_adwHistBytes[i][nHistCounter] = m_dwSecBytes[i];
		m_dwSecBytes[i] = 0;
		// byte rate
		m_dByteRate[i] = m_dwHistTotalBytes[i] / min(180.0, 1.0*m_nSecCounter);
	}
	// calculate query-reply rate
	m_dwHistTotalReplyPackets += m_dwSecReplyPackets;
	m_dwHistTotalReplyPackets -= m_adwHistReplyPackets[nHistCounter];
	m_adwHistReplyPackets[nHistCounter] = m_dwSecReplyPackets;
	m_dwSecReplyPackets = 0;
	// reply rate
	m_dReplyReceiveRate = m_dwHistTotalReplyPackets / min(3.0, 1.0/60.0*m_nSecCounter);

	// calculate avg SendQueueSize by weighting the prev value with the current one
	m_nSendQueueSize *= 0x0f;
	m_nSendQueueSize += m_sendQueue.size();
	m_nSendQueueSize >>= 4;

	if(SOCK_CLOSED == m_nStatus)
		ForceDisconnect();

	// may be it's time to unblock receive?
	if (m_bReceiveBlocked &&
		m_dwMaxInBytesPerSec > m_dByteRate[0])
	{
    //cout << "node: unblocking receive\n";
    ModifySelectFlags(FD_READ,0);
		m_bReceiveBlocked = false;
	}

	NodeManagement(nLastSecInPackets);
}

void MGnuNode::NodeManagement(int nLastSecPackets)
{
	if(SOCK_CONNECTING == m_nStatus || SOCK_NEGOTIATING == m_nStatus)
	{
		m_nSecsTrying++;
		
		if(m_nSecsTrying > CONNECT_TIMEOUT)
		{
			m_pDirector->RemoveNode(this);
			return;
		}
	}
	else if(SOCK_CONNECTED == m_nStatus)
	{
		// Drop if not transfering anything
		if(nLastSecPackets == 0)
		{
			m_nSecsDead++;

			if(m_nSecsDead == 30)
				Send_Ping(1);

			if(m_nSecsDead > 120)
			{
				m_pDirector->RemoveNode(this);
				return;
			}
		}
		else
			m_nSecsDead = 0;

		// Drop if Minimum friends isnt met
		if(m_pPrefs->m_nFriendsMin > 0)
		{
			if(m_nSecCounter>60 && m_dwFriendsTotal < m_pPrefs->m_nFriendsMin)
			{
				m_pDirector->RemoveNode(this);
				return;
			}
		}
		// once a second send a host
		//Node nd;
		//if (m_pHostCatcher->GetRandomNode(nd, false /* leave host in the cache */ ))
		//	Send_Host(nd);
		// once a minute send a ping to estimate the horizon
		if (m_nSecCounter % 60 == 30) // tricky offset of 30 secs. see min.friends condition above
			Send_Ping(MAX_TTL_SELF);
	}
	else
	{
		m_pDirector->RemoveNode(this);
		return;
	}
}

#endif // 0
