/*
 * Implementation of the proxy base class for communicating via D-Bus.
 *
 * Music Applet
 * Copyright (C) 2006 Paul Kuliniewicz <paul.kuliniewicz@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02111-1301, USA.
 *
 */

#include <config.h>

#include "ma-dbus-proxy.h"

#include <string.h>


#define GET_PRIVATE(o) 			(G_TYPE_INSTANCE_GET_PRIVATE ((o), MA_TYPE_DBUS_PROXY, MaDBusProxyPrivate))


typedef struct _MaDBusProxyPrivate	MaDBusProxyPrivate;


struct _MaDBusProxyPrivate
{
	gchar *dbus_name;

	DBusGProxyCall *call_owner;
	DBusGProxyCall *call_start;
};

typedef enum
{
	PROP_NONE,
	PROP_DBUS_NAME,
} MaDBusProxyProperty;

static MaProxyClass *parent_class;


/*********************************************************************
 *
 * Function declarations
 *
 *********************************************************************/

static void ma_dbus_proxy_class_init (MaDBusProxyClass *klass);
static void ma_dbus_proxy_init (MaDBusProxy *dproxy);

static void ma_dbus_proxy_dispose (GObject *object);
static void ma_dbus_proxy_finalize (GObject *object);

static void ma_dbus_proxy_get_property (GObject *object,
					guint prop_id,
					GValue *value,
					GParamSpec *pspec);

static void ma_dbus_proxy_set_property (GObject *object,
					guint prop_id,
					const GValue *value,
					GParamSpec *pspec);

static void ma_dbus_proxy_enable (MaProxy *proxy);
static void ma_dbus_proxy_disable (MaProxy *proxy);

static void name_owner_changed_cb (DBusGProxy *bus_proxy,
				   const gchar *name,
				   const gchar *old_owner,
				   const gchar *new_owner,
				   MaDBusProxy *dproxy);

static void name_has_owner_cb (DBusGProxy *bus_proxy,
			       DBusGProxyCall *call,
			       gpointer proxy);

static void start_service_by_name_cb (DBusGProxy *bus_proxy,
				      DBusGProxyCall *call,
				      gpointer proxy);


/*********************************************************************
 *
 * GType stuff
 *
 *********************************************************************/

GType
ma_dbus_proxy_get_type (void)
{
	static GType type = 0;

	if (type == 0)
	{
		/* Even though the objects stored in the Class struct are
		 * refcounted, we don't bother with base_init and
		 * base_finalized since, being static, this class is
		 * guaranteed never to be unloaded.
		 */

		static const GTypeInfo info = {
			sizeof (MaDBusProxyClass),			/* class_size */
			NULL,						/* base_init */
			NULL,						/* base_finalize */
			(GClassInitFunc) ma_dbus_proxy_class_init,	/* class_init */
			NULL,						/* class_finalize */
			NULL,						/* class_data */
			sizeof (MaDBusProxy),				/* instance_size */
			0,						/* n_preallocs */
			(GInstanceInitFunc) ma_dbus_proxy_init,		/* instance_init */
			NULL
		};

		type = g_type_register_static (MA_TYPE_PROXY, "MaDBusProxy", &info, G_TYPE_FLAG_ABSTRACT);
	}

	return type;
}

static void
ma_dbus_proxy_class_init (MaDBusProxyClass *klass)
{
	GError *error = NULL;

	GObjectClass *object_class = (GObjectClass *) klass;
	MaProxyClass *proxy_class = (MaProxyClass *) klass;
	parent_class = g_type_class_peek_parent (klass);

	object_class->dispose = ma_dbus_proxy_dispose;
	object_class->finalize = ma_dbus_proxy_finalize;

	object_class->get_property = ma_dbus_proxy_get_property;
	object_class->set_property = ma_dbus_proxy_set_property;

	proxy_class->enable = ma_dbus_proxy_enable;
	proxy_class->disable = ma_dbus_proxy_disable;

	klass->connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
	if (klass->connection != NULL)
	{
		klass->bus_proxy = dbus_g_proxy_new_for_name (klass->connection,
							      DBUS_SERVICE_DBUS,
							      DBUS_PATH_DBUS,
							      DBUS_INTERFACE_DBUS);
		dbus_g_proxy_add_signal (klass->bus_proxy, "NameOwnerChanged",
					 G_TYPE_STRING,
					 G_TYPE_STRING,
					 G_TYPE_STRING,
					 G_TYPE_INVALID);
	}
	else
	{
		/* TODO: Is this the best way to handle this error? */
		g_warning ("Failed to open connection to D-Bus session bus: %s",
			   error->message);
		g_error_free (error);
		klass->bus_proxy = NULL;
	}

	g_object_class_install_property (object_class,
					 PROP_DBUS_NAME,
					 g_param_spec_string ("dbus-name",
						 	      "dbus-name",
							      "DBus name to connect to",
							      NULL,
							      G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

	g_type_class_add_private (klass, sizeof (MaDBusProxyPrivate));
}

static void
ma_dbus_proxy_init (MaDBusProxy *dproxy)
{
	MaDBusProxyPrivate *priv = GET_PRIVATE (dproxy);

	priv->dbus_name = NULL;

	priv->call_owner = NULL;
}


/*********************************************************************
 *
 * Protected interface
 *
 *********************************************************************/

void
_ma_dbus_proxy_launch (MaDBusProxy *dproxy)
{
	MaDBusProxyPrivate *priv;

	g_return_if_fail (dproxy != NULL);
	g_return_if_fail (MA_IS_DBUS_PROXY (dproxy));

	priv = GET_PRIVATE (dproxy);

	if (priv->call_start == NULL)
	{
		priv->call_start = dbus_g_proxy_begin_call (MA_DBUS_PROXY_GET_CLASS (dproxy)->bus_proxy,
							    "StartServiceByName",
							    start_service_by_name_cb, dproxy, NULL,
							    G_TYPE_STRING, priv->dbus_name,
							    G_TYPE_UINT, 0,
							    G_TYPE_INVALID);
	}
}

void
_ma_dbus_proxy_cancel_call (DBusGProxy *dbus_proxy, DBusGProxyCall **call)
{
	if (*call != NULL)
	{
		dbus_g_proxy_cancel_call (dbus_proxy, *call);
		*call = NULL;
	}
}

void
_ma_dbus_proxy_report_error (const gchar *method, GError **error)
{
	if (*error != NULL)
	{
		g_warning ("%s failed: %s", method, (*error)->message);
		g_error_free (*error);
		*error = NULL;
	}
}


/*********************************************************************
 *
 * GObject overrides
 *
 *********************************************************************/

static void
ma_dbus_proxy_dispose (GObject *object)
{
	ma_dbus_proxy_disable (MA_PROXY (object));

	G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
ma_dbus_proxy_finalize (GObject *object)
{
	g_free (GET_PRIVATE (object)->dbus_name);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
ma_dbus_proxy_get_property (GObject *object,
			    guint prop_id,
			    GValue *value,
			    GParamSpec *pspec)
{
	MaDBusProxyPrivate *priv = GET_PRIVATE (object);

	switch (prop_id)
	{
	case PROP_DBUS_NAME:
		g_value_set_string (value, priv->dbus_name);
		break;

	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static void
ma_dbus_proxy_set_property (GObject *object,
			    guint prop_id,
			    const GValue *value,
			    GParamSpec *pspec)
{
	MaDBusProxyPrivate *priv = GET_PRIVATE (object);

	switch (prop_id)
	{
	case PROP_DBUS_NAME:
		g_free (priv->dbus_name);
		priv->dbus_name = g_value_dup_string (value);
		break;

	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


/*********************************************************************
 *
 * MaProxy overrides
 *
 *********************************************************************/

static void
ma_dbus_proxy_enable (MaProxy *proxy)
{
	MaDBusProxyPrivate *priv;
	DBusGProxy *bus_proxy;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_DBUS_PROXY (proxy));

	priv = GET_PRIVATE (proxy);
	bus_proxy = MA_DBUS_PROXY_GET_CLASS (proxy)->bus_proxy;

	dbus_g_proxy_connect_signal (bus_proxy,
				     "NameOwnerChanged",
				     G_CALLBACK (name_owner_changed_cb), proxy, NULL);

	priv->call_owner = dbus_g_proxy_begin_call (bus_proxy,
						    "NameHasOwner",
						    name_has_owner_cb, proxy, NULL,
						    G_TYPE_STRING, priv->dbus_name,
						    G_TYPE_INVALID);
}

static void
ma_dbus_proxy_disable (MaProxy *proxy)
{
	MaDBusProxyPrivate *priv;
	DBusGProxy *bus_proxy;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_DBUS_PROXY (proxy));

	priv = GET_PRIVATE (proxy);
	bus_proxy = MA_DBUS_PROXY_GET_CLASS (proxy)->bus_proxy;

	if (ma_proxy_get_state (proxy) == MA_PROXY_STATE_CONNECTED)
	{
		MA_DBUS_PROXY_GET_CLASS (proxy)->disconnect (MA_DBUS_PROXY (proxy));
		_ma_proxy_set_connected (proxy, FALSE);
	}

	_ma_dbus_proxy_cancel_call (bus_proxy, &priv->call_owner);
	_ma_dbus_proxy_cancel_call (bus_proxy, &priv->call_start);

	dbus_g_proxy_disconnect_signal (bus_proxy,
					"NameOwnerChanged",
					G_CALLBACK (name_owner_changed_cb), proxy);
}


/*********************************************************************
 *
 * Callbacks
 *
 *********************************************************************/

static void
name_owner_changed_cb (DBusGProxy *bus_proxy,
		       const gchar *name,
		       const gchar *old_owner,
		       const gchar *new_owner,
		       MaDBusProxy *dproxy)
{
	MaDBusProxyPrivate *priv = GET_PRIVATE (dproxy);
	MaDBusProxyClass *klass = MA_DBUS_PROXY_GET_CLASS (dproxy);

	if (strcmp (name, priv->dbus_name) == 0)
	{
		MaProxyState state = ma_proxy_get_state (MA_PROXY (dproxy));

		if (state == MA_PROXY_STATE_DISCONNECTED && new_owner[0] != '\0')
		{
			_ma_proxy_set_connected (MA_PROXY (dproxy), TRUE);
			klass->connect (dproxy, klass->connection);
		}
		else if (state == MA_PROXY_STATE_CONNECTED && new_owner[0] == '\0')
		{
			_ma_proxy_set_connected (MA_PROXY (dproxy), FALSE);
			klass->disconnect (dproxy);
		}
	}
}

static void
name_has_owner_cb (DBusGProxy *bus_proxy,
		   DBusGProxyCall *call,
		   gpointer proxy)
{
	MaDBusProxyPrivate *priv = GET_PRIVATE (proxy);
	MaDBusProxyClass *klass = MA_DBUS_PROXY_GET_CLASS (proxy);

	gboolean has_owner = FALSE;
	GError *error = NULL;

	if (!dbus_g_proxy_end_call (bus_proxy, call, &error,
				    G_TYPE_BOOLEAN, &has_owner,
				    G_TYPE_INVALID))
	{
		if (error != NULL)
		{
			g_warning ("NameHasOwner failed: %s", error->message);
			g_error_free (error);
		}
	}

	if (has_owner)
	{
		_ma_proxy_set_connected (MA_PROXY (proxy), TRUE);
		klass->connect (MA_DBUS_PROXY (proxy), klass->connection);
	}

	priv->call_owner = NULL;
}

static void
start_service_by_name_cb (DBusGProxy *bus_proxy,
			  DBusGProxyCall *call,
			  gpointer proxy)
{
	MaDBusProxyPrivate *priv = GET_PRIVATE (proxy);

	guint result;
	GError *error = NULL;

	if (!dbus_g_proxy_end_call (bus_proxy, call, &error,
				    G_TYPE_UINT, &result,
				    G_TYPE_INVALID))
	{
		if (error != NULL)
		{
			gchar *primary = g_strdup_printf (_("Failed to launch %s"),
							  ma_proxy_get_name (MA_PROXY (proxy)));
			_ma_proxy_report_error (MA_PROXY (proxy), primary, error->message, TRUE);
			g_free (primary);
			g_error_free (error);
		}
	}

	priv->call_start = NULL;
}
