//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: midithread.cpp,v 1.6 2002/02/27 15:53:45 muse Exp $
//
//  (C) Copyright 2001 Werner Schweer (ws@seh.de)
//=========================================================

#include <config.h>

#include <poll.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <sys/time.h>

#include <linux/version.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
#include <linux/spinlock.h>
#include <linux/mc146818rtc.h>
#else
#include <linux/rtc.h>
#endif

#include <qsocketnotifier.h>
#include <qtimer.h>

#include "synth.h"
#include "alsamidi.h"
#include "midithread.h"
#include "globals.h"
#include "song.h"
#include "midictrl.h"
#include "event.h"
#include "tempo.h"
#include "midithreadp.h"
#include "app.h"
#include "utils.h"

static const unsigned char mmcStopMsg[] =         { 0x7f, 0x7f, 0x06, 0x01 };
static const unsigned char mmcDeferredPlayMsg[] = { 0x7f, 0x7f, 0x06, 0x03 };

static int TICK_SLICE;

//---------------------------------------------------------
//   recTimeStamp
//---------------------------------------------------------

int MidiThread::recTimeStamp() const
      {
      return tempomap.time2tick(curTime() - data->startTime)
         + data->recStartTick;
      }

//---------------------------------------------------------
//   heartBeat
//---------------------------------------------------------

void MidiThread::heartBeat()
      {
      song->beat(data->playTickPos);
      if (data->controlChanged) {
            muse->ctrlChanged();
            data->controlChanged = false;
            }
      }

//---------------------------------------------------------
//   MidiThread
//---------------------------------------------------------

MidiThread::MidiThread(bool rt=true, int prio=50, bool ml = true)
   : QObject(0, "midiThread"), Thread(rt, prio, ml)
      {
      data               = new MidiThreadPrivate;
      data->tempoSN      = -1;
      data->timerFd      = -1;
      data->state        = IDLE;
      data->midiTick     = 0;
      data->rtcTick      = 0;
      data->midiSync     = division/24;
      data->playTickPos  = 0;
      data->rtcTick      = 0;
      data->controlChanged = false;
      data->playEvents   = new EventList;
      data->stuckNotes   = new EventList;
      data->realRtcTicks = rtcTicks;
      TICK_SLICE         = division/10;
      data->endSlice     = 0;

      heartBeatTimer    = new QTimer(this, "timer");
      connect(heartBeatTimer, SIGNAL(timeout()), this, SLOT(heartBeat()));

      //---------------------------------------------------
      //  establish socket to gui
      //---------------------------------------------------

      int filedes[2];         // 0 - reading   1 - writing
      if (pipe(filedes) == -1) {
            perror("creating pipe1");
            exit(-1);
            }
      data->sigFd = filedes[1];
      QSocketNotifier* ss = new QSocketNotifier(filedes[0], QSocketNotifier::Read);
      connect(ss, SIGNAL(activated(int)), this, SLOT(seqSignal(int)));
      }

//---------------------------------------------------------
//   ~MidiThread
//---------------------------------------------------------

MidiThread::~MidiThread()
      {
      if (data->timerFd) {
            ioctl(data->timerFd, RTC_PIE_OFF, 0);
            close(data->timerFd);
            }
      delete data->stuckNotes;
      delete data->playEvents;
      delete data;
      }

//---------------------------------------------------------
//   isPlaying
//---------------------------------------------------------

bool MidiThread::isPlaying() const
      {
      return data->state == PLAY;
      }

//---------------------------------------------------------
//   curTickPos
//    current play position in midi ticks;
//    provided for gui "heartBeat()"
//---------------------------------------------------------

int MidiThread::curTickPos() const
      {
      return data->playTickPos;
      }

//---------------------------------------------------------
//   SEvent
//---------------------------------------------------------

struct SEvent {
      MidiEvent* ev;
      bool operator<(SEvent&);
      SEvent(MidiEvent* me) { ev = me; }
      };

//---------------------------------------------------------
//   SEventList
//---------------------------------------------------------

class SEventList : public std::list<SEvent> {
   public:
      SEventList() {}
      };
typedef std::list<SEvent>::iterator iSEvent;

bool SEvent::operator<(SEvent& e)
      {
      if (ev->port() != e.ev->port())
            return ev->port() < e.ev->port();


      // play note off events first to prevent overlapping
      // notes

      if (ev->channel() == e.ev->channel())
            return ev->type() == MidiEvent::NoteOff;

      int map[16] = { 1, 2, 3, 4, 5, 6, 7, 8, 9, 0, 10, 11, 12, 13, 14, 15 };
      return map[ev->channel()] < map[e.ev->channel()];
      }

//---------------------------------------------------------
//   recordStop
//    execution environment: gui thread
//---------------------------------------------------------

void MidiThread::recordStop()
      {
      TrackList* tl = song->tracks();
      int lastTick = tempomap.time2tick(curTime() - data->startTime);
// printf("record STOP, starttick %d - enddtick %d\n",
//  data->recStartTick, lastTick);
      for (iTrack it = tl->begin(); it != tl->end(); ++it) {
            if ((*it)->type() != Track::MIDI && (*it)->type() != Track::DRUM)
                  continue;
            MidiTrack* mt = (MidiTrack*)*it;
            EventList* el = mt->events();

            //---------------------------------------------------
            //    resolve NoteOff events
            //---------------------------------------------------

            for (iEvent i = el->begin(); i != el->end(); ++i) {
                  MidiEvent* ev  = (MidiEvent*)i->second;
                  if (ev->isNote()) {
                        iEvent k = i;
                        for (++k; k != el->end(); ++k) {
                              if (ev->isNoteOff((MidiEvent*)k->second)) {
                                    int t = k->first - i->first;
                                    if (t == 0) {
                                          t = 1;
                                          printf("len == 0!\n");
                                          }
                                    ev->setLenTick(t);
                                    ev->setVeloOff(((MidiEvent*)i->second)->veloOff());
                                    el->erase(k);
                                    break;
                                    }
                              }
                        if (k == el->end()) {
                              int len = lastTick - k->first;
                              if (len < 1)
                                    len = 1;
                              ev->setLenTick(len);
                              }
                        }
                  }

            //-------------------------------------------------------------
            //    assign events to parts
            //-------------------------------------------------------------

            song->cmdAddRecordedEvents(mt, el, data->recStartTick);
            el->clear();
            }
      song->setRecord(false);
      }

//---------------------------------------------------------
//   seqSignal
//    sequencer message to GUI
//    execution environment: gui thread
//---------------------------------------------------------

void MidiThread::seqSignal(int fd)
      {
      char buffer[16];

      int n = ::read(fd, buffer, 1);
      if (n != 1) {
            printf("MidiThread: seqSignal: READ PIPE returns %d\n", n);
            return;
            }
      switch(buffer[0]) {
            case '0':         // STOP
                  if (song->record())
                        recordStop();
                  song->setPlay(false);
                  break;
            case '2':   // record
                  song->setRecord(true);
            case '1':
                  song->setPlay(true);
                  break;
            case 'P':   // alsa ports changed
                  song->rescanAlsaPorts();
                  break;
            case 'I':   // event received
                  emit midiNote(data->curPitch, data->curVelo);
                  break;
            case 'G':
                  song->setPos(0, data->playTickPos, true, true, true);
                  break;
            default:
                  printf("Seq Signal <%c>\n", buffer[0]);
                  break;
            }
      }

//---------------------------------------------------------
//   readMsg
//---------------------------------------------------------

static void readMsg(void* p, void*)
      {
// printf("midithread: readMsg\n");
      MidiThread* at = (MidiThread*)p;
      at->readMsg();
      }

//---------------------------------------------------------
//   midiTick
//---------------------------------------------------------

void MidiThread::midiTick(void* p, void*)
      {
      MidiThread* at = (MidiThread*)p;
      at->data->processTimerTick();
      }

//---------------------------------------------------------
//   alsaMidiRead
//---------------------------------------------------------

static void alsaMidiRead(void*, void*)
      {
      alsaProcessMidiInput();
      }

//---------------------------------------------------------
//   midiRead
//---------------------------------------------------------

static void midiRead(void*, void* d)
      {
      MidiDevice* dev = (MidiDevice*) d;
      dev->processInput();
      }

//---------------------------------------------------------
//   synthIRead
//---------------------------------------------------------

static void synthIRead(void*, void* d)
      {
      SynthI* syn = (SynthI*) d;
      syn->processInput();
      }

//---------------------------------------------------------
//   midiWrite
//---------------------------------------------------------

static void midiWrite(void*, void* d)
      {
      MidiDevice* dev = (MidiDevice*) d;
      dev->flush();
      }

//---------------------------------------------------------
//   setHeartBeat
//---------------------------------------------------------

void MidiThread::setHeartBeat()
      {
      heartBeatTimer->start(1000/guiRefresh);
      }

//---------------------------------------------------------
//   start
//---------------------------------------------------------

void MidiThread::start()
      {
      setHeartBeat();

      //---------------------------------------------------
      // initialize timer
      //---------------------------------------------------

#ifndef RTCAP
      doSetuid();
#endif
      data->timerFd = ::open("/dev/rtc", O_RDWR);
#ifndef RTCAP
      undoSetuid();
#endif
      if (data->timerFd == -1)
            perror("cannot open rtc clock /dev/rtc");
      if (setRtcTicks()) {
#ifndef RTCAP
      doSetuid();
#endif
            if (ioctl(data->timerFd, RTC_PIE_ON, 0) == -1)
                  perror("RTC_PIE_ON failed");
#ifndef RTCAP
      undoSetuid();
#endif
            }

      updatePollFd();
      Thread::start();
      }

//---------------------------------------------------------
//   updatePollFd
//---------------------------------------------------------

void MidiThread::updatePollFd()
      {
      clearPollFd();
      if (data->timerFd)
            addPollFd(data->timerFd, POLLIN, midiTick, this, 0);
// printf("update poll %d\n", toThreadFdr);
      addPollFd(toThreadFdr, POLLIN, ::readMsg, this, 0);

      //---------------------------------------------------
      //  midi ports
      //---------------------------------------------------

      for (int i = 0; i < MIDI_PORTS; ++i) {
            MidiPort* mp = &midiPorts[i];
            MidiDevice* dev = mp->device();
            if (dev) {
                  if ((mp->rwFlags()&0x2) || (extSyncFlag.value() && (extSyncPort==i))) {
                        int fd = dev->selectRfd();
                        if (fd != -1)
                              addPollFd(fd, POLLIN, ::midiRead, this, dev);
                        }
                  if (dev->bytesToWrite()) {
                        int fd = dev->selectWfd();
                        if (fd != -1)
                              addPollFd(fd, POLLIN, ::midiWrite, this, dev);
                        }
                  }
            }
#ifdef ALSA
      // special handling for alsa midi:
      // (one fd for all devices)
      //    this allows for processing of some alsa events
      //    even if no alsa driver is active (assigned to a port)
      int fd = alsaSelectRfd();
      if (fd != -1) {
            addPollFd(fd, POLLIN, ::alsaMidiRead, this, 0);
            }
#endif

      //---------------------------------------------------
      //   connect Synthi instances (GUI)
      //---------------------------------------------------

      for (iSynthI i = synthiInstances.begin(); i != synthiInstances.end(); ++i) {
            int fd = (*i)->readFd();
            if (fd != -1)
                  addPollFd(fd, POLLIN, ::synthIRead, this, *i);
            }
      }

//---------------------------------------------------------
//   setRtcTicks
//    return true on success
//---------------------------------------------------------

bool MidiThread::setRtcTicks()
      {
      if (data->timerFd == -1) {
            setPollWait(10);  // 10 msec
            data->realRtcTicks = 100;
            return false;
            }
#ifndef RTCAP
      doSetuid();
#endif
      int rc = ioctl(data->timerFd, RTC_IRQP_SET, rtcTicks);
#ifndef RTCAP
      undoSetuid();
#endif

      if (rc == -1) {
            fprintf(stderr, "cannot set tick on /dev/rtc: %s\n",
               strerror(errno));
            fprintf(stderr, "precise timer not available\n");
            close(data->timerFd);
            data->timerFd = -1;
            setPollWait(10);  // 10 msec
            data->realRtcTicks = 100;
            return false;
            }
      data->realRtcTicks = rtcTicks;
      return true;
      }

//---------------------------------------------------------
//   processMsg
//---------------------------------------------------------

void MidiThread::processMsg(const ThreadMsg* m)
      {
      const MidiMsg* msg = (MidiMsg*)m;

      switch (msg->id) {
            case SEQM_RESET_DEVICES:
                  for (int i = 0; i < MIDI_PORTS; ++i)
                        midiPorts[i].instrument()->reset(i, song->mtype());
                  break;
            case SEQM_INIT_DEVICES:
                  data->initDevices();
                  break;
            case SEQM_MIDI_LOCAL_OFF:
                  sendLocalOff();
                  break;
            case SEQM_MIDI_CTRL:
                  midiPorts[msg->port].setCtrl(msg->channel, msg->ctrl, msg->a);
                  break;
            case SEQM_MIDI_MASTER_VOLUME:
                  for (int i = 0; i < MIDI_PORTS; ++i)
                        midiPorts[i].setMasterVol(msg->b);
                  break;
            case SEQM_SET_MIDI_DEVICE:
                  ((MidiPort*)(msg->p1))->setMidiDevice((MidiDevice*)(msg->p2));
                  updatePollFd();
                  break;
            case SEQM_PLAY_MIDI_EVENT:
                  data->playEvent((MidiEvent*)(msg->p1));
                  break;
            case SEQM_SET_RTC_TICKS:
                  setRtcTicks();
                  break;
            case SEQM_SEEK:
                  if (data->state == PLAY)
                        data->seek(msg->a);
                  break;
            case SEQM_SCAN_ALSA_MIDI_PORTS:
                  alsaScanMidiPorts();
                  break;
            case SEQM_PLAY:
                  if (data->state == IDLE && msg->a)
                        data->startPlay();
                  else if ((data->state == PLAY || data->state == PRECOUNT)
                     && !msg->a)
                        data->stopPlay();
                  break;
            case MIDI_ADD_SYNTHI:
                  updatePollFd();
                  break;
            case MIDI_SHOW_INSTR_GUI:
                  ((MidiInstrument*)(msg->p1))->showGui(msg->a);
                  updatePollFd();
                  break;
            default:
                  song->processMsg(msg);
                  break;
            }
      }

//---------------------------------------------------------
//   playEvent
//---------------------------------------------------------

void MidiThreadPrivate::playEvent(MidiEvent* event)
      {
      MidiTrack* t   = event->trk();
      MidiPort* port = &midiPorts[event->port()];
      switch (event->type()) {
            case MidiEvent::Note:
                  {
                  int len   = event->lenTick();
                  int pitch = event->pitch();
                  int velo  = event->velo();
                  if (t) {
                        if (song->mute(t))
                              return;
                        pitch += t->transposition;
                        if (pitch > 127)
                              pitch = 127;
                        if (pitch < 0)
                              pitch = 0;

                        velo += t->velocity;
                        velo = (velo * t->compression) / 100;
                        if (velo > 127)
                              velo = 127;
                        if (velo < 1)           // no off event
                              velo = 1;

                        len = (len *  t->len) / 100;
                        t->addActivity(velo);
                        }
                  if (len <= 0)     // dont allow zero length
                        len = 1;
                  MidiEvent* noteOff = new MidiEvent(*event);
                  noteOff->setType(MidiEvent::NoteOff);
                  noteOff->setPosTick(event->posTick() + len);
                  noteOff->setPitch(pitch);
                  stuckNotes->add(noteOff, noteOff->posTick());
                  if (len != event->lenTick() || pitch != event->pitch()
                     | velo != event->velo()) {
                        //CHECK
                        MidiEvent* ev = new MidiEvent(*event);
                        ev->setLenTick(len);
                        ev->setPitch(pitch);
                        ev->setVelo(velo);
                        port->putEvent(ev);
                        return;
                        }
                  }
                  break;

            case MidiEvent::Ctrl7:
            case MidiEvent::Ctrl14:
                  if (song->mute(t))
                        return;
                  controlChanged = true;
                  break;
            case MidiEvent::NoteOff:
                  if (t == 0)
                        break;
                  t->addActivity(-(event->velo()));
                  break;
            default:
                  if (t && song->mute(t))
                        return;
                  break;
            }
      port->putEvent(event);
      if (port->instrument()->hasGui())
            port->instrument()->writeToGui(event);
      }

//---------------------------------------------------------
//   stop
//---------------------------------------------------------

void MidiThreadPrivate::stopPlay()
      {
      //---------------------------------------------------
      //    end all notes
      //---------------------------------------------------

      for (iEvent i = stuckNotes->begin(); i != stuckNotes->end(); ++i)
            playEvent((MidiEvent*)(i->second));
      stuckNotes->clear();

      //---------------------------------------------------
      //    reset sustain
      //---------------------------------------------------

      for (int i = 0; i < MIDI_PORTS; ++i) {
            for (int ch = 0; ch < MIDI_CHANNELS; ++ch) {
                  if (midiPorts[i].cState(ch)->controller[CTRL_SUSTAIN] != -1)
                        midiPorts[i].setCtrl(ch, CTRL_SUSTAIN, 0);
                  }
            }

      audioThread->stopPlay();

      //---------------------------------------------------
      //    clear track activity display
      //---------------------------------------------------

      TrackList* tl = song->tracks();
      for (iTrack i = tl->begin(); i != tl->end(); ++i) {
            SoundSource* ss = dynamic_cast<SoundSource*>(*i);
            ss->resetMeter();
            ss->setCurActivity(1);
            }

      MidiDevice* syncDev = midiPorts[extSyncPort].device();

      if (genMTCSync) {       // MTC
            unsigned char mmcPos[] = {
                  0x7f, 0x7f, 0x06, 0x44, 0x06, 0x01,
                  0, 0, 0, 0, 0
                  };
            MTC mtc(tempomap.tick2time(playTickPos));
            mmcPos[6] = mtc.h() | (mtcType << 5);
            mmcPos[7] = mtc.m();
            mmcPos[8] = mtc.s();
            mmcPos[9] = mtc.f();
            mmcPos[10] = mtc.sf();
            MidiEvent ev0(0, 0, 0, MidiEvent::Sysex, sizeof(mmcStopMsg), mmcStopMsg);
            midiPorts[extSyncPort].putEvent(&ev0);
            MidiEvent ev1(0, 0, 0, MidiEvent::Sysex, sizeof(mmcPos), mmcPos);
            midiPorts[extSyncPort].putEvent(&ev1);
            }
      if (genMCSync) {        // Midi Clock
            // send STOP and
            // "set song position pointer"
            if (syncDev) {
                  syncDev->putStop();
                  int beat = playTickPos * 4 / division;
                  syncDev->putSongpos(beat);
                  }
            }
      state = IDLE;
      write(sigFd, "0", 1);   // STOP
      }

//---------------------------------------------------------
//   defaultTick
//---------------------------------------------------------

void MidiThread::defaultTick()
      {
      data->processTimerTick();
      }

//---------------------------------------------------------
//   midiPortsChanged
//---------------------------------------------------------

void MidiThread::midiPortsChanged()
      {
      write(data->sigFd, "P", 1);
      }

//---------------------------------------------------------
//   sendLocalOff
//---------------------------------------------------------

void MidiThread::sendLocalOff()
      {
      for (int k = 0; k < MIDI_PORTS; ++k) {
            for (int i = 0; i < 16; ++i)
                  midiPorts[k].setCtrl(i, 122, 0);
            }
      }

//---------------------------------------------------------
//   initDevices
//---------------------------------------------------------

void MidiThreadPrivate::initDevices()
      {
// printf("init Devices\n");
      //
      // mark all used devices in song as active
      //
      bool activeDevices[MIDI_PORTS];
      for (int i = 0; i < MIDI_PORTS; ++i)
            activeDevices[i] = false;

      TrackList* tracks = song->tracks();
      for (iTrack it = tracks->begin(); it != tracks->end(); ++it) {
            MidiTrack* track = dynamic_cast<MidiTrack*>(*it);
            if (track == 0)
                  continue;
            activeDevices[track->outPort()] = true;
            }

      //
      // damit Midi-Devices, die mehrere Ports besitzen, wie z.B.
      // das Korg NS5R, nicht mehrmals zwischen GM und XG/GS hin und
      // hergeschaltet werden, wird zunchst auf allen Ports GM
      // initialisiert, und dann erst XG/GS
      //
      for (int i = 0; i < MIDI_PORTS; ++i) {
            if (activeDevices[i] == false)
                  continue;
            MidiPort* port = &midiPorts[i];
            switch(song->mtype()) {
                  case MT_GS:
                  case MT_UNKNOWN:
                        break;
                  case MT_GM:
                  case MT_XG:
                        port->gmOn();
                        usleep(200000);   // wait 200ms
                        break;
                  }
            }
      for (int i = 0; i < MIDI_PORTS; ++i) {
            if (activeDevices[i] == false)
                  continue;
            MidiPort* port = &midiPorts[i];
            switch(song->mtype()) {
                  case MT_UNKNOWN:
                        break;
                  case MT_GM:
                        break;
                  case MT_GS:
                        port->gsOn();
                        usleep(50000);    // wait 50 ms
                        break;
                  case MT_XG:
                        port->xgOn();
                        usleep(50000);    // wait 50 ms
                        break;
                  }
            }

      for (int i = 0; i < MIDI_PORTS; ++i) {
            if (activeDevices[i] == false)
                  continue;
            MidiPort* port = &midiPorts[i];
            //
            // alle channel-states resetten wg. gmOn
            //
            for (int chan = 0; chan < MIDI_CHANNELS; ++chan) {
                  ChannelState* istate = port->iState(chan);
                  //
                  //  Drum- und Percussion Channel bei GM nicht switchen!
                  //    mein Korg NS5R schalten dann auf Voice um!
                  //
                  if (song->mtype() == MT_GM) {
                        ChannelState* cstate = port->cState(chan);
                        cstate->controller[CTRL_HBANK] = istate->controller[CTRL_HBANK];
                        cstate->controller[CTRL_LBANK] = istate->controller[CTRL_LBANK];
                        }
                  for (int k = 0; k < 128; ++k) {
                        int val = istate->controller[k];
                        if ((val != -1) && (val != port->cState(chan)->controller[k])) {
//DEBUG
//printf("init port %d dev %d ctrl %d val %d\n",
//   i, chan, k, val);
                              port->setCtrl(chan, k, val);
                              }
                        }

                  if (!((chan == 9) && (song->mtype() == MT_GM))) {
                        int val = istate->program;
                        if (val != -1) {
                              port->programChange(chan, val);
                              }
                        }
                  else {
                        // GM-File tries to change drum channel
//                        printf("bad GM\n");
                        }
                  // RPN's
                  // NRPN's
                  }
            }
      }

//---------------------------------------------------------
//   startPlay
//---------------------------------------------------------

void MidiThreadPrivate::startPlay()
      {
      //
      // reset runstate of midi devices for more
      // robust handling
      //
      for (int i = 0; i < MIDI_PORTS; ++i)
            midiPorts[i].resetRunstate();
      if (song->cpos() == 0 && !song->record())
            initDevices();
      audioThread->play(song->cpos());

      seek(song->cpos());
      if (genMTCSync) {
            MidiEvent ev(0, 0, 0, MidiEvent::Sysex,
              sizeof(mmcDeferredPlayMsg), mmcDeferredPlayMsg);
            midiPorts[extSyncPort].putEvent(&ev);
            }
      if (genMCSync) {
            if (playTickPos)
                  midiPorts[extSyncPort].putContinue();
            else
                  midiPorts[extSyncPort].putStart();
            }
      if (precountEnableFlag
         && song->click()
         && !extSyncFlag.value()
         && song->record()) {
            state = PRECOUNT;
            int z, n;
            if (precountFromMastertrackFlag)
                  sigmap.timesig(playTickPos, z, n);
            else {
                  z = precountSigZ;
                  n = precountSigN;
                  }
            clickno       = z * preMeasures;
            clicksMeasure = z;
            ticksBeat     = (division * 4)/n;
            midiClick     = midiTick;         // start now
            }
      else {
            int bar, beat, tick;
            sigmap.tickValues(playTickPos, &bar, &beat, &tick);
            if (tick)
                  beat += 1;
            midiClick = sigmap.bar2tick(bar, beat, 0);
// printf("tick %d(%d)  %d %d %d = %d\n",
//   midiTick, song->cpos(), bar, beat, tick, midiClick);
            }
      // start real play on next midi sync
      // current state is START_PLAY
      }

//---------------------------------------------------------
//   seek
//---------------------------------------------------------

void MidiThreadPrivate::seek(int pos)
      {
      playTickPos = pos;

      //---------------------------------------------------
      //    end all notes
      //---------------------------------------------------

      for (iEvent i = stuckNotes->begin(); i != stuckNotes->end(); ++i)
            playEvent((MidiEvent*)(i->second));
      stuckNotes->clear();

      MidiDevice* syncDev = midiPorts[extSyncPort].device();
      if (genMCSync && syncDev) {
            int beat = (playTickPos * 4) / division;
            syncDev->putStop();           // ??
            syncDev->putSongpos(beat);
            syncDev->putContinue();       // ??
            }
      audioThread->seek(playTickPos);
      loopPassed = true;   // for record loop mode
      playEvents->clear();
      state = START_PLAY;
      }

//---------------------------------------------------------
//   processMidiTick
//---------------------------------------------------------

void MidiThreadPrivate::processMidiTick()
      {
      if (genMCSync)
            midiPorts[extSyncPort].clock();
      if (state == START_PLAY) {
            // start play on sync
            state         = PLAY;
            midiTickStart = playTickPos;
            midiTick      = playTickPos;
            midiSync      = playTickPos;
            rtcTickStart  = rtcTick;
            endSlice      = playTickPos;
            recTick       = playTickPos;
            lastTickPos   = playTickPos;
            startTime     = curTime();
            recStartTick  = playTickPos;
            write(sigFd, "1", 1);   // PLAY
            }
      }

//---------------------------------------------------------
//   processTimerTick
//---------------------------------------------------------

void MidiThreadPrivate::processTimerTick()
      {
      int nn = 1;
      if (timerFd != -1) {
            unsigned long nn;
            if (read(timerFd, &nn, sizeof(unsigned long)) != sizeof(unsigned long)) {
                  perror("illegal timer return");
                  exit(-1);
                  }
            nn >>= 8;
            // rtcTicks  (ticks per second);
            }
      if (tempoSN != tempomap.tempoSN()) {
            midiTickStart = midiTick;
            rtcTickStart  = rtcTick;
            tempoSN       = tempomap.tempoSN();
            }
      rtcTick += nn;
      midiTick = tempomap.time2tick(1.0/realRtcTicks
                  * (rtcTick - rtcTickStart)) + midiTickStart;

      if (!extSyncFlag.value() && midiTick >= midiSync) {
            processMidiTick();
            midiSync += division/24;
            }

      if (genMTCSync) {
            // printf("Midi Time Code Sync generation not impl.\n");
            }

      if (state == PLAY) {
            playTickPos = midiTick;
            if (song->loop() && !extSyncFlag.value()
               && (playTickPos >= song->rpos())) {
                  seek(song->lpos());
                  // loop on next midiSync
                  return;
                  }
            if (playTickPos >= endSlice) {
                  int next = endSlice + TICK_SLICE;
                  song->nextEvents(endSlice, next, playEvents);
                  endSlice = next;
                  }
            if (playTickPos >= song->len() && !song->record() && !song->loop()) {
                  stopPlay();
                  }
            }
      if ((state == PRECOUNT || (song->click() && (state == PLAY)))
         && (midiTick >= midiClick)) {
            MidiEvent* ev = new MidiEvent(clickPort, clickChan,
               midiTick, MidiEvent::Note,
               beatClickNote, beatClickVelo, 0, 4);
            bool isMeasure = false;
            int bar, beat, tick;
            switch (state) {
                  case PRECOUNT:
                        if ((clickno % clicksMeasure) == 0)
                              isMeasure = true;
                        if (clickno) {
                              --clickno;
                              midiClick += ticksBeat;
                              break;
                              }
                        sigmap.tickValues(playTickPos, &bar, &beat, &tick);
                        if (beat == 0 && tick == 0)
                              isMeasure = true;
                        midiClick = sigmap.bar2tick(bar, beat+1, 0);
                        state = START_PLAY;
                        break;
                  case PLAY:
                        sigmap.tickValues(midiTick, &bar, &beat, &tick);
//tick==0?*/            if (beat == 0 && tick == 0)
                              isMeasure = true;
                        midiClick = sigmap.bar2tick(bar, beat+1, 0);
                        break;
                  default:
                        break;
                  }
            if (isMeasure) {
                  ev->setPitch(measureClickNote);
                  ev->setVelo(measureClickVelo);
                  }
            playEvents->add(ev);
            }
      SEventList sevents;
// printf("midiTick %d\n", midiTick);
      EventRange range1 = stuckNotes->equal_range(midiTick);
//      for (iEvent k = range1.first; k != range1.second; ++k)
      for (iEvent k = stuckNotes->begin(); k != range1.second; ++k)
            sevents.push_back(SEvent((MidiEvent*)(k->second)));
      EventRange range2 = playEvents->equal_range(midiTick);
//      for (iEvent k = range2.first; k != range2.second; ++k)
      for (iEvent k = playEvents->begin(); k != range2.second; ++k)
            sevents.push_back(SEvent((MidiEvent*)(k->second)));
      if (!sevents.empty()) {
            sevents.sort();
            for (iSEvent sk = sevents.begin(); sk != sevents.end(); ++sk)
                  playEvent(sk->ev);
            EventRange range1 = stuckNotes->equal_range(midiTick);
//            stuckNotes->erase(range1.first, range1.second);
//            playEvents->erase(range2.first, range2.second);
            stuckNotes->erase(stuckNotes->begin(), range1.second);
            playEvents->erase(playEvents->begin(), range2.second);
            }
      }

