//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: audionode.cpp,v 1.1 2002/01/30 14:54:03 muse Exp $
//
//  (C) Copyright 2000 Werner Schweer (ws@seh.de)
//=========================================================

#include <math.h>
#include "audionode.h"
#include "globals.h"
#include "song.h"
#include "xml.h"

//---------------------------------------------------------
//   AudioNode
//---------------------------------------------------------

AudioNode::AudioNode()
      {
      _route  = 0;
      _volume = 0.0;
      _pan    = 0.0;
//      buffer  = new float[segmentSize * channels()];
      }

//---------------------------------------------------------
//   ~AudioNode
//---------------------------------------------------------

AudioNode::~AudioNode()
      {
      }

//---------------------------------------------------------
//   node2String
//    create string name representation for audio node
//---------------------------------------------------------

QString node2String(AudioNode* n)
      {
      if (n == song->master())
            return "Master";
      for (int i = 0; i < song->mixerGroups(); ++i) {
            if (n == song->group(i)) {
                  QString s;
                  s.sprintf("Group %c", i + 'A');
                  return s;
                  }
            }
      return "None";
      }

//---------------------------------------------------------
//   string2Node
//---------------------------------------------------------

AudioNode* string2Node(const QString& s)
      {
      if (s == "Master")
            return song->master();
      if (s.left(6) == "Group ") {
            int i = s.at(6).latin1() - 'A';
            return song->group(i);
            }
      return 0;
      }

//---------------------------------------------------------
//   segmentSizeChanged
//---------------------------------------------------------

void AudioNode::segmentSizeChanged()
      {
//      delete buffer;
//      buffer  = new float[segmentSize * channels()];
      }

//---------------------------------------------------------
//   fillBuffer
//---------------------------------------------------------

void AudioNode::fillBuffer(int dstChannels, float* dstBuffer, int idx)
      {
      int srcChannels = channels();
      if (mute() || soloMute()) {
            for (int i = 0; i < srcChannels; ++i)
                  _activity[i] = 0;
            return;
            }
      float buffer[segmentSize * srcChannels];
      getData(buffer, idx);

      //---------------------------------------------------
      // apply plugin chain
      //---------------------------------------------------

      _efxPipe.apply(buffer, segmentSize, srcChannels);

      //---------------------------------------------------
      //    prefader metering
      //---------------------------------------------------

      float meter[srcChannels];
      if (_prefader) {
            float* p = buffer;
            for (int i = 0; i < srcChannels; ++i) {
                  meter[i] = 0.0;
                  for (int k = 0; k < segmentSize; ++k) {
                        double f = fabs(*p);
                        if (f > meter[i])
                              meter[i] = f;
                        ++p;
                        }
                  _activity[i] = int(meter[i] * 32767.0);
                  if (_activity[i] > _peak[i])
                        _peak[i] = _activity[i];
                  }
            }

      //---------------------------------------------------
      // apply volume
      //    postfader metering
      //---------------------------------------------------

      double vol[2];
      vol[0] = _volume * (1.0 - _pan);
      vol[1] = _volume * (1.0 + _pan);

      if (srcChannels == dstChannels) {
            float* sp = buffer;
            float* dp = dstBuffer;
            if (_prefader) {
                  for (int c = 0; c < dstChannels; ++c) {
                        for (int k = 0; k < segmentSize; ++k) {
                              *dp++ += (*sp++ * vol[c]);
                              }
                        }
                  }
            else {
                  for (int c = 0; c < dstChannels; ++c) {
                        meter[c] = 0.0;
                        for (int k = 0; k < segmentSize; ++k) {
                              float val = *sp++ * vol[c];
                              *dp++ += val;
                              double f = fabs(val);
                              if (f > meter[c])
                                    meter[c] = f;
                              }
                        _activity[c] = int(meter[c] * 32767.0);
                        if (_activity[c] > _peak[c])
                              _peak[c] = _activity[c];
                        }
                  }
            }
      else if (srcChannels == 1 && dstChannels == 2) {
            float* dp = dstBuffer;
            if (_prefader) {
                  for (int c = 0; c < dstChannels; ++c) {
                        float* sp = buffer;
                        for (int k = 0; k < segmentSize; ++k) {
                              *dp++ += (*sp++ * vol[c]);
                              }
                        }
                  }
            else {
                  for (int c = 0; c < dstChannels; ++c) {
                        float* sp = buffer;
                        meter[c] = 0.0;
                        for (int k = 0; k < segmentSize; ++k) {
                              float val = *sp++ * vol[c];
                              double f = fabs(val);
                              if (f > meter[c])
                                    meter[c] = f;
                              *dp++ += val;
                              }
                        _activity[c] = int(meter[c] * 32767.0);
                        if (_activity[c] > _peak[c])
                              _peak[c] = _activity[c];
                        }
                  }
            }
      if (srcChannels == 2 && dstChannels == 1) {
            float* sp1 = buffer;
            float* sp2 = buffer + segmentSize;
            if (_prefader) {
                  float* dp = dstBuffer;
                  for (int k = 0; k < segmentSize; ++k) {
                        *dp++ += (*sp1++ * vol[0] + *sp2++ * vol[1]);
                        }
                  }
            else {
                  float* dp = dstBuffer;
                  meter[0] = 0.0;
                  for (int k = 0; k < segmentSize; ++k) {
                        float val = (*sp1++ * vol[0] + *sp2++ * vol[1]);
                        double f = fabs(val);
                        if (f > meter[0])
                              meter[0] = f;
                        *dp++ += val;
                        }
                  _activity[0] = int(meter[0] * 32767.0);
                  if (_activity[0] > _peak[0])
                        _peak[0] = _activity[0];
                  }
            }
      }

//---------------------------------------------------------
//   setChannel
//---------------------------------------------------------

void AudioNode::setChannels(int c)
      {
      SoundSource::setChannels(c);
      segmentSizeChanged();
      }

//---------------------------------------------------------
//   setRoute
//---------------------------------------------------------

void AudioNode::setRoute(AudioNode* src)
      {
      _route = src;
      }

//---------------------------------------------------------
//   writeConfiguration
//---------------------------------------------------------

void AudioNode::writeConfiguration(int level, Xml& xml) const
      {
      xml.tag(level++, "audionode");
      xml.intTag(level, "channels", channels());
      QString s = node2String(_route);
      xml.strTag(level, "connect", s);
      xml.put(level, "<volume>%f</volume>", _volume);
      xml.doubleTag(level, "pan", _pan);
      xml.intTag(level, "mute", mute());
      xml.intTag(level, "solomute", soloMute());
      xml.intTag(level, "prefader", prefader());
      for (ciPluginI ip = _efxPipe.begin(); ip != _efxPipe.end(); ++ip) {
            if (*ip)
                  (*ip)->writeConfiguration(level, xml);
            }
      xml.etag(level, "audionode");
      }

//---------------------------------------------------------
//   readVolume
//---------------------------------------------------------

void AudioNode::readVolume(Xml& xml)
      {
      int ch = 0;
      for (;;) {
            Xml::Token token = xml.parse();
            switch (token) {
                  case Xml::Error:
                  case Xml::End:
                        return;
                  case Xml::TagStart:
                        xml.unknown("readVolume");
                        break;
                  case Xml::Text:
                        setVolume(xml.s1().toDouble());
                        break;
                  case Xml::Attribut:
                        if (xml.s1() == "ch")
                              ch = xml.s2().toInt();
                        break;
                  case Xml::TagEnd:
                        if (xml.s1() == "volume")
                              return;
                  default:
                        break;
                  }
            }
      }

//---------------------------------------------------------
//   readConfiguration
//---------------------------------------------------------

void AudioNode::readConfiguration(Xml& xml)
      {
      for (;;) {
            Xml::Token token = xml.parse();
            const QString& tag = xml.s1();
            switch (token) {
                  case Xml::Error:
                  case Xml::End:
                        return;
                  case Xml::TagStart:
                        if (tag == "connect") {
                              QString s = xml.parse1();
                              _route = string2Node(s);
                              }
                        else if (tag == "channels") {
                              int chan = xml.parseInt();
                              setChannels(chan);
                              }
                        else if (tag == "volume")
                              readVolume(xml);
                        else if (tag == "pan")
                              setPan(xml.parseDouble());
                        else if (tag == "plugin") {
                              PluginI* pi = new PluginI;
                              pi->readConfiguration(xml);
                              _efxPipe.push_back(pi);
                              }
                        else if (tag == "mute")
                              setMute(xml.parseInt());
                        else if (tag == "solomute")
                              setSolo(xml.parseInt());
                        else if (tag == "prefader")
                              setPrefader(xml.parseInt());
                        else
                              xml.unknown("audiosource");
                        break;
                  case Xml::Attribut:
                        break;
                  case Xml::TagEnd:
                        if (tag == "audionode")
                              return;
                  default:
                        break;
                  }
            }
      }

//---------------------------------------------------------
//   setPrefader
//---------------------------------------------------------

void AudioNode::setPrefader(bool flag)
      {
      _prefader = flag;
      for (int i = 0; i < channels(); ++i) {
            _activity[i] = 0;
            _peak[i] = 0;
            }
      }

