//clients.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2010
 *
 *  This file is part of RoarD,
 *  a sound server daemon for using the RoarAudio protocol.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  or (at your option) any later version as published by
 *  the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include "muroard.h"

#define _FILTER_ANY 0

int client_init(void) {
 memset(g_client, 0, sizeof(g_client)); // state unused = 0!

#ifdef MUROAR_FEATURE_INTERNAL_CLIENT
 g_client[0].state  = CLIENT_STATE_INTERNAL;
 g_client[0].sock   = -1;
 g_client[0].stream = -1;
#endif

 return 0;
}

int client_free(void) {
 int i;
 int ret = 0;

 for (i = 0; i < MUROAR_MAX_CLIENTS; i++) {
  if ( g_client[i].state != CLIENT_STATE_UNUSED ) {
   if ( client_delete(i) == -1 )
    ret = -1;
  }
 }

 return ret;
}

int client_new(int sock) {
 int i;

 if ( network_nonblock(sock) == -1 )
  return -1;

 for (i = 0; i < MUROAR_MAX_CLIENTS; i++) {
  if ( g_client[i].state == CLIENT_STATE_UNUSED ) {
   g_client[i].state  = CLIENT_STATE_NEW;
   g_client[i].sock   = sock;
   g_client[i].stream = -1;
   return i;
  }
 }

 return -1;
}

int client_delete(int id) {
 int ret = 0;

 if ( id >= MUROAR_MAX_CLIENTS || id < 0 )
  return -1;

#ifdef MUROAR_FEATURE_INTERNAL_CLIENT
 if ( g_client[id].state == CLIENT_STATE_INTERNAL  )
  return 0;
#endif

 if ( g_client[id].state == CLIENT_STATE_UNUSED   ||
      g_client[id].state == CLIENT_STATE_CLOSING   )
  return 0;

 g_client[id].state = CLIENT_STATE_CLOSING;

 if ( g_client[id].stream != -1 ) {
  if ( stream_delete(g_client[id].stream) == -1 )
   ret = -1;
 }

 if ( g_client[id].sock != -1 ) {
  network_close(g_client[id].sock);
 }

 g_client[id].state = CLIENT_STATE_UNUSED;

 return ret;
}

int client_exec(int id) {
 int sock = g_client[id].sock;

 g_client[id].sock  = -1;
 g_client[id].state = CLIENT_STATE_EXECED;

 return sock;
}

int client_handle(int id) {
 struct muroard_message m;
 int sock = g_client[id].sock;
 int ret  = 0;

 if ( proto_recv(sock, &m) == -1 ) {
  client_delete(id);
  return -1;
 }

 if ( m.stream > MUROAR_MAX_STREAMS && m.stream != (uint16_t)-1 ) {
  // FIXME: this code is redundant :(
  m.cmd     = MUROAR_CMD_ERROR;
  m.stream  = -1;
  m.pos     =  0;
  m.datalen =  0;

  if ( proto_send(sock, &m) == -1 ) {
   client_delete(id);
   return -1;
  }
  return 0;
 }

 switch (m.cmd) {
  case MUROAR_CMD_NOOP:
  case MUROAR_CMD_IDENTIFY:
  case MUROAR_CMD_AUTH:
#ifdef MUROAR_FEATURE_NOOP_SIUCMDS
  case MUROAR_CMD_SET_META:
#endif
    // no need to do anything
    m.datalen =  0;
   break;
  case MUROAR_CMD_QUIT:
    m.cmd = MUROAR_CMD_OK;
    m.datalen =  0;
    proto_send(sock, &m);
    client_delete(id);
    return 0;
   break;
  case MUROAR_CMD_EXEC_STREAM:
    m.datalen =  0;
    if ( m.stream != (uint16_t)-1 && m.stream == g_client[id].stream ) {
     if ( stream_exec(m.stream) == -1 )
      ret = -1;
    } else {
     ret = -1;
    }
   break;
#ifdef MUROAR_FEATURE_CMD_WHOAMI
  case MUROAR_CMD_WHOAMI:
    m.datalen =  1;
    m.data[0] =  id;
   break;
#endif
#ifdef MUROAR_FEATURE_CMD_EXIT
  case MUROAR_CMD_EXIT:
    // we simply set g_alive to zero so we will die with the next cycle.
    if ( m.datalen == 0 ) { // default: exit
     g_alive  = 0;
    } else if ( m.datalen == 1 && m.data[0] == 0 ) { // explicit: exit
     g_alive  = 0;
#ifndef MUROAR_FEATURE_TERMINATE
    } else if ( m.datalen == 1 && m.data[0] == 1 ) { // explicit: terminate
     ret = network_prefree();
#endif
    } else { // terminate or unsupported command version
     ret      = -1;
    }
    m.datalen = 0;
   break;
#endif
#ifdef MUROAR_FEATURE_CMD_GET_STANDBY
  case MUROAR_CMD_GET_STANDBY:
    m.datalen = 2;
    m.data[0] = 0;
    m.data[1] = 0;
   break;
#endif
#ifdef MUROAR_FEATURE_CMD_GET_VOL
  case MUROAR_CMD_GET_VOL:
    // we support only the most up to date version of the command:
    if ( m.datalen < 2 || m.data[0] != 0 || m.data[1] != 1 ) {
     ret = -1;
    } else {
     m.datalen = 6*2;
     // version: 1
     m.data[ 0] = 0;
     m.data[ 1] = 1;
     // channels: 1
     m.data[ 2] = 0;
     m.data[ 3] = 1;
     // scale:
     m.data[ 4] = 0xFF;
     m.data[ 5] = 0xFF;
     // rpg mul:
     m.data[ 6] = 0;
     m.data[ 7] = 1;
     // rpg div:
     m.data[ 8] = 0;
     m.data[ 9] = 1;
     // our ownly channel:
     m.data[10] = 0xFF;
     m.data[11] = 0xFF;
    }
   break;
#endif
#ifdef MUROAR_FEATURE_CMD_KICK
  case MUROAR_CMD_KICK:
    if ( m.datalen != 4 ) {
     ret = -1;
    } else {
     if ( m.data[0] != 0 ) { // type >= 256
      ret = -1;
     } else {
      if ( m.data[2] != 0 ) { // id >= 256
       ret = -1;
      } else if ( m.data[1] == 1 ) { // CLIENT
       ret = client_delete(m.data[3]);
      } else if ( m.data[1] == 2 ) { // STREAM
       ret = stream_delete(m.data[3]);
      } else { // unknown type
       ret = -1;
      }
     }
    }
   break;
#endif
#ifdef MUROAR_FEATURE_CMD_ATTACH
  case MUROAR_CMD_ATTACH:
    ret = client_handle_attach(id, &m);
   break;
#endif
#ifdef MUROAR_FEATURE_CMD_SERVER_OINFO
  case MUROAR_CMD_SERVER_OINFO:
    ret = client_handle_server_oinfo(id, &m);
   break;
#endif
#ifdef MUROAR_FEATURE_CMD_PASSFH
  case MUROAR_CMD_PASSFH:
    ret = client_handle_passfh(id, &m);
   break;
#endif
#ifdef MUROAR_FEATURE_CMD_LIST_CLIENTS
  case MUROAR_CMD_LIST_CLIENTS:
    ret = client_handle_list_clients(id, &m);
   break;
#endif
#ifdef MUROAR_FEATURE_CMD_LIST_STREAMS
  case MUROAR_CMD_LIST_STREAMS:
    ret = client_handle_list_streams(id, &m);
   break;
#endif
#ifdef MUROAR_FEATURE_CMD_GET_CLIENT
  case MUROAR_CMD_GET_CLIENT:
    ret = client_handle_get_client(id, &m);
   break;
#endif
#ifdef MUROAR_FEATURE_CMD_GET_STREAM
  case MUROAR_CMD_GET_STREAM:
    ret = client_handle_get_stream(id, &m);
   break;
#endif
  case MUROAR_CMD_NEW_STREAM:
    ret = client_handle_new_stream(id, &m);
   break;
  default:
    ret = -1; // we do not know this command
 }

 if ( ret == 0 ) {
  m.cmd     = MUROAR_CMD_OK;
 } else {
  m.cmd     = MUROAR_CMD_ERROR;
  m.stream  = -1;
  m.pos     =  0;
  m.datalen =  0;
 }

 if ( proto_send(sock, &m) == -1 ) {
  client_delete(id);
  return -1;
 }

 return 0;
}

int client_handle_new_stream(int id, struct muroard_message * mes) {
 struct muroar_audio_info info;
 uint32_t * data = (uint32_t*)mes->data;
 int ret;
 int i;

 if ( mes->datalen != (6*4) )
  return -1;

 mes->datalen = 0;

 for (i = 0; i < 6; i++) {
  data[i] = ntohl(data[i]);
 }

 if ( data[1] != (uint32_t)-1 )
  return -1;

 if ( data[2] != (uint32_t)g_sa_rate )
  return -1;

 info.bits     = data[3];
 info.channels = data[4];
 info.codec    = data[5];

 ret = stream_new(id, data[0], &info);

 if ( ret == -1 )
  return -1;

 mes->stream = ret;

 return 0;
}

#ifdef MUROAR_FEATURE_CMD_ATTACH
int client_handle_attach(int id, struct muroard_message * mes) {
 int client;

 (void)id; // we do not need tis parameter

 if ( mes->datalen != 6 )
  return -1;

 // we do a quick and dirty check:
 // 0) 16 bit version must be zero
 // 1) we only support SIMPLE attach, const 16 bit, too.
 // 2) upper 8 bit of client must be zero (we do not support > 256 clients!)

 if ( mes->data[0] != 0 || mes->data[1] != 0 ||
      mes->data[2] != 0 || mes->data[3] != 1 ||
      mes->data[4] != 0                       )
  return -1;

 client = mes->data[5];

 if ( client < 0 )
  return -1;

 if ( (client + 1) > MUROAR_MAX_CLIENTS )
  return -1;

 if ( g_client[client].state != CLIENT_STATE_NEW      &&
#ifdef MUROAR_FEATURE_INTERNAL_CLIENT
      g_client[client].state != CLIENT_STATE_INTERNAL &&
#endif
      g_client[client].state != CLIENT_STATE_OLD       )
  return -1;

 if ( g_client[client].stream != -1 )
  return -1;

 if ( stream_move_client(mes->stream, client) == -1 )
  return -1;

#ifdef MUROAR_FEATURE_INTERNAL_CLIENT
 if ( g_client[client].state != CLIENT_STATE_INTERNAL )
#endif
  g_client[client].stream = mes->stream;

 mes->datalen = 0;

 return 0;
}
#endif

#ifdef MUROAR_FEATURE_CMD_SERVER_OINFO
int client_handle_server_oinfo(int id, struct muroard_message * mes) {
 uint32_t * data = (uint32_t*) mes->data;
 int        i;

 (void)id; // we do not need tis parameter

 mes->datalen = 6*4;
 mes->stream  =  -1;

 data[0] =  6; // TODO: do not use magic number. this is ROAR_DIR_MIXING
 data[1] = -1;
 data[2] = g_sa_rate;
 data[3] = 16;
 data[4] = g_sa_channels;
 data[5] = MUROAR_CODEC_PCM;

 for (i = 0; i < 6; i++) {
  data[i] = htonl(data[i]);
 }

 return 0;
}
#endif

#ifdef MUROAR_FEATURE_CMD_PASSFH
int client_handle_passfh(int id, struct muroard_message * mes) {
 int fh;

 mes->datalen = 0;

 fh = network_recvfh(g_client[id].sock);

 if ( fh == -1 )
  return -1;

 if ( !stream_exist(mes->stream) ) {
  network_close(fh);
  return -1;
 }

 if ( stream_get_sock(mes->stream) != -1 ) {
  network_close(fh);
  return -1;
 }

 if ( stream_set_sock(mes->stream, fh) == -1 ) {
  network_close(fh);
  return -1;
 }

 return 0;
}
#endif

#ifdef MUROAR_FEATURE_CMD_LIST_CLIENTS
int client_handle_list_clients(int id, struct muroard_message * mes) {
 int len = 0;
 int i;

 (void)id; // we do not need tis parameter

 if ( mes->datalen != 7 )
  return -1;

 if ( mes->data[0] != 0 )
  return -1;

 if ( mes->data[1] != _FILTER_ANY )
  return -1;

 for (i = 0; i < MUROAR_MAX_CLIENTS; i++) {
  if ( g_client[i].state != CLIENT_STATE_UNUSED ) {
   mes->data[len++] = i;
  }
 }

 mes->datalen = len;

 return 0;
}
#endif
#ifdef MUROAR_FEATURE_CMD_LIST_STREAMS
int client_handle_list_streams(int id, struct muroard_message * mes) {
 int len = 0;
 int i;

 (void)id; // we do not need tis parameter

 if ( mes->datalen != 7 )
  return -1;

 if ( mes->data[0] != 0 )
  return -1;

 if ( mes->data[1] != _FILTER_ANY )
  return -1;

 for (i = 0; i < MUROAR_MAX_STREAMS; i++) {
  if ( g_stream[i].state != STREAM_STATE_UNUSED ) {
   mes->data[len++] = i;
  }
 }

 mes->datalen = len;

 return 0;
}
#endif

#ifdef MUROAR_FEATURE_CMD_GET_CLIENT
int client_handle_get_client(int id, struct muroard_message * mes) {
 struct muroard_client * c;
 int client;
 int len = 0;

 (void)id; // we do not need tis parameter

 if ( mes->datalen != 1 )
  return -1;

 client = (unsigned char)mes->data[0];

 if ( client >= MUROAR_MAX_CLIENTS )
  return -1;

 if ( (c = &(g_client[client]))->state == CLIENT_STATE_UNUSED )
  return -1;

 mes->data[len++] = 0; // version

 if ( c->stream == -1 ) {
  mes->data[len++] = -1; // not execed because we do not have a stream
 } else {
  if ( g_stream[c->stream].state == STREAM_STATE_EXECED ) {
   mes->data[len++] = c->stream;
  } else {
   mes->data[len++] = -1; // not execed
  }
 }

 if ( c->stream == -1 ) {
  mes->data[len++] = 0; // no streams
 } else {
  mes->data[len++] = 1; // one stream
  mes->data[len++] = c->stream;
 }

 // TODO: support some dummy like "Client %i" or something
 mes->data[len++] = 1; // name has size of one byte
 mes->data[len++] = 0; // \0

 memset(&(mes->data[len]), 0xFF, 3*4); // PID, UID, GID
 len += 3*4;

 // Protocol: RoarAudio = 1, 32 bit network byte order
 mes->data[len++] = 0;
 mes->data[len++] = 0;
 mes->data[len++] = 0;
 mes->data[len++] = 1;

 // Byte order we use: Network byte order = 0x02, 32 bit network byte order
 mes->data[len++] = 0;
 mes->data[len++] = 0;
 mes->data[len++] = 0;
 mes->data[len++] = 0x02;

 mes->datalen = len;

 return 0;
}
#endif
#ifdef MUROAR_FEATURE_CMD_GET_STREAM
int client_handle_get_stream(int id, struct muroard_message * mes) {
 struct muroard_stream * s;
 uint32_t * data = (uint32_t*) mes->data;
 int        i;

 (void)id; // we do not need tis parameter

 if ( !stream_exist(mes->stream) ) {
  return -1;
 }

 s = &(g_stream[mes->stream]);

 //fprintf(stderr, "%s:%i: dir=%i\n", __FILE__, __LINE__, s->dir);

 mes->datalen = 6*4;
 mes->pos     = -1;

 data[0] = s->dir;
 data[1] = -1;
 data[2] = g_sa_rate;
 data[3] = s->info.bits;
 data[4] = s->info.channels;
 data[5] = s->info.codec;

 for (i = 0; i < 6; i++) {
  data[i] = htonl(data[i]);
 }

 return 0;
}
#endif

//ll
