/*--------------------------------*-C-*---------------------------------*
 * File:	pixmap.c
 *----------------------------------------------------------------------*
 * Copyright (c) 1999 Ethan Fischer <allanon@crystaltokyo.com>
 * Copyright (c) 1999 Sasha Vasko   <sashav@sprintmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*
 * Originally written:
 *    1999	Sasha Vasko <sashav@sprintmail.com>
 *
 *    2002 Alexis <materm@tele2.fr>
 *              - modifications for multi-terms support
 *----------------------------------------------------------------------*/

/*
 * $Id: pixmap.c,v 1.11 2004/08/15 16:17:26 alexis Exp $
 */

#include "rxvt.h"		/* NECESSARY */
#ifdef DEBUG
#define DEBUG_XPM_ALEXIS 1
#else
#define DEBUG_XPM_ALEXIS 0
#endif

#if DEBUG_XPM_ALEXIS
#define DXPM_ALEXIS(x) fprintf x; fputc('\n', stderr)
#else
#define DXPM_ALEXIS(x)
#endif


#if defined(BACKGROUND_IMAGE) || defined(TRANSPARENT)
# include <X11/Xatom.h>
# ifdef HAVE_LIBXPM
#   include <X11/xpm.h>
# endif
#endif

/*#define BENCHMARK_SHADING 200 */
/*#define DO_CLOCKING       */
/*#define GETPIXEL_PUTPIXEL */
/*#define DEBUG_IMAGING     */

#if defined(DO_CLOCKING) || defined(BENCHMARK_SHADING)
#include <time.h>
#endif

#ifdef DEBUG_X
#define XGetGeometry(dpy,win,r,x,y,w,h,b,d) \
	trace_XGetGeometry(__FILE__,__LINE__,dpy,win,r,x,y,w,h,b,d)
#endif

#if defined(BACKGROUND_IMAGE) || defined(TRANSPARENT) || defined(_MYSTYLE_)
/* Xdisplay, Xroot, Xvisual and Xdepth needs to be defined if code is moved to some other prog */
#define CREATE_TRG_PIXMAP(w,h) XCreatePixmap(Xdisplay, Xroot, w, h, Xdepth)

#ifdef USE_JPEG
/* this comes from jpg.c */
extern long JpegReadFileToPixmap(Display* display,Window window,GC gc,
             char* filename,Pixmap* pixmap,long* w,long* h);
#endif

#ifdef USE_PNG
/* this comes from jpg.c */
extern long PngReadFileToPixmap(Display* display,Window window,GC gc,
             char* filename,Pixmap* pixmap,long* w,long* h);
#endif

/* this comes from libasimage or from ximage_utils.c */
XImage *ScaleXImageToSize(XImage * src, Position width,Position height);
void ShadeXImage(XImage * srcImage, ShadingInfo * shading, GC gc);

int pixmap_error_handler(Display * dpy, XErrorEvent * error)
{
#ifdef DEBUG_IMAGING
	fprintf(stderr,
		"aterm caused XError # %u, in resource %lu, Request: %d.%d",
		error->error_code, error->resourceid, error->request_code,
		error->minor_code);
#endif
	return 0;
}

#ifndef USE_LIBASIMAGE		/* used to be LIBASIMAGE_HEADERS */
void
CopyAndShadeArea(Drawable src, Pixmap trg,
		 int x, int y, int w, int h,
		 int trg_x, int trg_y, GC gc, ShadingInfo * shading)
{
	int             (*oldXErrorHandler) (Display *, XErrorEvent *);

    /* we need to check if pixmap is still valid */
	oldXErrorHandler = XSetErrorHandler(pixmap_error_handler);

	if (shading)
	{
		XImage         *img;

		if (x < 0 || y < 0)
			return;
		if ((img = XGetImage(Xdisplay, src, x, y, w, h, AllPlanes,ZPixmap)) != NULL)
		{
			ShadeXImage(img, shading, gc);
			XPutImage(Xdisplay, trg, gc, img, 0, 0, trg_x, trg_y,
				  w, h);
#ifdef BENCHMARK_SHADING
			{
				int             i;
				time_t          before, after;
				double          diff;

				before = time(NULL);
				for (i = 0; i < BENCHMARK_SHADING; i++)
					ShadeXImage(img, shading, gc);
				after = time(NULL);

				diff = difftime(after, before);
				printf("CopyAndShadeArea(): %d shading runs took %.0f seconds\n",
				     BENCHMARK_SHADING, diff);
			}
#endif
			XDestroyImage(img);
			return;
		}
	}
	if (!XCopyArea(Xdisplay, src, trg, gc, x, y, w, h, trg_x, trg_y))
		XFillRectangle(Xdisplay, trg, gc, trg_x, trg_y, w, h);
	XSetErrorHandler(oldXErrorHandler);
}

int
FillPixmapWithTile(Pixmap pixmap, Pixmap tile, int x, int y, int width,
		   int height, int tile_x, int tile_y)
{
	if (tile != None && pixmap != None)
	{
		GC              gc;
		XGCValues       gcv;

		gcv.tile = tile;
		gcv.fill_style = FillTiled;
		gcv.ts_x_origin = -tile_x;
		gcv.ts_y_origin = -tile_y;
		gc = XCreateGC(Xdisplay, tile,
			       GCFillStyle | GCTile | GCTileStipXOrigin |
			       GCTileStipYOrigin, &gcv);
		XFillRectangle(Xdisplay, pixmap, gc, x, y, width, height);
		XFreeGC(Xdisplay, gc);
		return 1;
	}
	return 0;
}

void
ShadeTiledPixmap(Pixmap src, Pixmap trg, int src_w, int src_h, int x, int y,
		 int w, int h, GC gc, ShadingInfo * shading)
{
	int             tile_x, tile_y, left_w, bott_h;

  DXPM_ALEXIS((stderr,"ShadeTiledPixmap: x=%d,src_w=%d",x,src_w));
	tile_x = x % src_w;
	tile_y = y % src_h;
	left_w = min(src_w - tile_x, w);
	bott_h = min(src_h - tile_y, h);
/*fprintf( stderr, "\nShadeTiledPixmap(): tile_x = %d, tile_y = %d, left_w = %d, bott_h = %d, SRC = %dx%d TRG=%dx%d", tile_x, tile_y, left_w, bott_h, src_w, src_h, w, h);*/
	CopyAndShadeArea(src, trg, tile_x, tile_y, left_w, bott_h, 0, 0, gc,
			 shading);
	if (bott_h < h)
	{			/* right-top parts */
		CopyAndShadeArea(src, trg, tile_x, 0, left_w, h - bott_h, 0,
				 bott_h, gc, shading);
	}
	if (left_w < w)
	{			/* left-bott parts */
		CopyAndShadeArea(src, trg, 0, tile_y, w - left_w, bott_h,
				 left_w, 0, gc, shading);
		if (bott_h < h)	/* left-top parts */
			CopyAndShadeArea(src, trg, 0, 0, w - left_w,
					 h - bott_h, left_w, bott_h, gc,
					 shading);
	}
}

Pixmap
ShadePixmap(Pixmap src, int x, int y, int width, int height, GC gc,
	    ShadingInfo * shading)
{
	Pixmap          trg = CREATE_TRG_PIXMAP(width, height);

	if (trg != None)
	{
		CopyAndShadeArea(src, trg, x, y, width, height, 0, 0, gc,shading);
	}
	return trg;
}

Pixmap
ScalePixmap(Pixmap src, int src_w, int src_h, int width, int height, GC gc,
	    ShadingInfo * shading)
{
	XImage         *srcImage, *trgImage;
	Pixmap          trg = None;

	if (src != None)
	{
		if ((srcImage = XGetImage(Xdisplay, src, 0, 0, src_w, src_h, AllPlanes,ZPixmap)) != NULL)
		{
			if ((trgImage =
			     ScaleXImageToSize(srcImage, (Dimension) width,
					       (Dimension) height)) != NULL)
			{
				if ((trg =
				     CREATE_TRG_PIXMAP(width, height)) != 0)
				{
					if (shading)
						ShadeXImage(trgImage, shading,gc);
					XPutImage(Xdisplay, trg, gc, trgImage,
						  0, 0, 0, 0, width, height);
#ifdef BENCHMARK_SHADING
					{
						int             i;
						time_t          before, after;
						double          diff;

						before = time(NULL);
						for (i = 0;i < BENCHMARK_SHADING;i++)
							ShadeXImage(trgImage,shading,gc);
						after = time(NULL);

						diff = difftime(after, before);
						printf
						    ("ScalePixmap(): %d shading runs took %.0f seconds\n",
						     BENCHMARK_SHADING, diff);
					}
#endif
				}
				XDestroyImage(trgImage);
			}
			XDestroyImage(srcImage);
		}
	}
	return trg;
}

Pixmap
CenterPixmap(Pixmap src, int src_w, int src_h, int width, int height, GC gc,
	     ShadingInfo * shading)
{
	int             x, y, w, h, src_x = 0, src_y = 0;
	Pixmap          trg;

    /* create target pixmap of the size of the window */
	trg = CREATE_TRG_PIXMAP(width, height);
	if (trg != None)
	{
	    /* fill it with background color */
		XFillRectangle(Xdisplay, trg, gc, 0, 0, width, height);
	    /* place image at the center of it */
		x = (width - src_w) >> 1;
		y = (height - src_h) >> 1;
		if (x < 0)
		{
			src_x -= x;
			w = min(width, src_w + x);
			x = 0;
		} else
			w = min(width, src_w);
		if (y < 0)
		{
			src_y -= y;
			h = min(height, src_h + y);
			y = 0;
		} else
			h = min(height, src_h);

		CopyAndShadeArea(src, trg, src_x, src_y, w, h, x, y, gc,
				 shading);
	}

	return trg;
}

Pixmap
GrowPixmap(Pixmap src, int src_w, int src_h, int width, int height, GC gc,
	   ShadingInfo * shading)
{
	int             w, h;
	Pixmap          trg;

    /* create target pixmap of the size of the window */
	trg = CREATE_TRG_PIXMAP(width, height);
	if (trg != None)
	{
	    /* fill it with background color */
		XFillRectangle(Xdisplay, trg, gc, 0, 0, width, height);
	    /* place image at the center of it */
		w = min(width, src_w);
		h = min(height, src_h);

		CopyAndShadeArea(src, trg, 0, 0, w, h, 0, 0, gc, shading);
	}

	return trg;
}

/* PROTO */
int
GetRootDimensions(int *width, int *height)
{
	Window          root;
	int             w_x, w_y;
	unsigned int    w_w, w_h, border_w, w_depth;

	if (!XGetGeometry(Xdisplay, Xroot, &root,
			  &w_x, &w_y, width, height, &border_w, &w_depth))
	{
		*width = 0;
		*height = 0;
	}
	return (*width > 0 && *height > 0) ? 1 : 0;
}

int
GetWinPosition(Window win, int *x, int *y)
{
	Window          root, parent, *children;
	unsigned int    nchildren;
	static int      rootWidth = 0, rootHeight = 0;
	XWindowAttributes attr;
	int             my_x, my_y, dumm;
	unsigned int    w, h, udumm;

	XGetWindowAttributes(Xdisplay, win, &attr);
	if (attr.map_state != IsViewable)
		return 0;

	if (!x)
		x = &my_x;
	if (!y)
		y = &my_y;

	*x = 0;
	*y = 0;

	if (!rootWidth || !rootHeight)
		if (!GetRootDimensions(&rootWidth, &rootHeight))
			return 0;

	XGetGeometry(Xdisplay, win, &root, &dumm, &dumm, &w, &h, &udumm,&udumm);

	while (XQueryTree
	       (Xdisplay, win, &root, &parent, &children, &nchildren))
	{
		int             w_x, w_y;
		unsigned int    border_w;

		if (children)
			XFree(children);
		if (!XGetGeometry(Xdisplay, win, &root, &w_x, &w_y, &udumm, &udumm,
		     &border_w, &udumm))
			break;
		(*x) += w_x + (int)border_w;
		(*y) += w_y + (int)border_w;

		if (parent == root)
		{		/* taking in to consideration virtual desktopping */
			int             bRes = 1;

			if (*x >= rootWidth || (*x + (int)w) <= 0 ||
			    *y >= rootHeight || (*y + (int)h) <= 0)
				bRes = 0;
		    /* don't want to return position outside the screen even if we fail */
		    /*      while( *x < 0 ) *x += rootWidth ;
		     * while( *y < 0 ) *y += rootHeight ;
		     * while( *x+(int)w > rootWidth ) *x -= rootWidth ;
		     * while( *y+(int)h > rootHeight) *y -= rootHeight ;
		     */ return bRes;
		}
		win = parent;
	}
	*x = 0;
	*y = 0;
	return 0;
}

void
sleep_a_little(int n)
{
	struct timeval  value;

	if (n <= 0)
		return;

	value.tv_usec = n % 1000000;
	value.tv_sec = n / 1000000;

	(void)select(1, 0, 0, 0, &value);
}

static          Pixmap
CutPixmap(Pixmap src, Pixmap trg,
	  int x, int y,
	  unsigned int src_w, unsigned int src_h,
	  unsigned int width, unsigned int height,
	  GC gc, ShadingInfo * shading)
{
	Bool            my_pixmap = (trg == None) ? True : False;
	int             screen_w, screen_h;
	int             offset_x = 0, offset_y = 0;

	screen_w = DisplayWidth(Xdisplay, Xscreen);
	screen_h = DisplayHeight(Xdisplay, Xscreen);

	while (x + (int)width < 0)
		x += screen_w;
	while (x >= screen_w)
		x -= screen_w;
	while (y + (int)height < 0)
		y += screen_h;
	while (y >= screen_h)
		y -= screen_h;

	if (width < 2 || height < 2)
		return trg;
	if (x < 0)
	{
		offset_x = (-x);
		x = 0;
		width -= offset_x;
	}
	if (y < 0)
	{
		offset_y = (-y);
		y = 0;
		height -= offset_y;
	}
	if (x + width >= screen_w)
		width = screen_w - x;
	if (y + height >= screen_h)
		height = screen_h - y;

	if (src == None)	/* we don't have root pixmap ID */
	{			/* we want to create Overrideredirect window overlapping out window
				 * with background type of Parent Relative and then grab it */
		XSetWindowAttributes attr;
		XEvent          event;
		int             tick_count = 0;
		Bool            grabbed = False;

		attr.background_pixmap = ParentRelative;
		attr.backing_store = Always;
		attr.event_mask = ExposureMask;
		attr.override_redirect = True;
		src = XCreateWindow(Xdisplay, Xroot, x, y, width, height,
				    0,
				    CopyFromParent, CopyFromParent,
				    CopyFromParent,
				    CWBackPixmap | CWBackingStore |
				    CWOverrideRedirect | CWEventMask, &attr);

		if (src == None)
			return trg;
		XGrabServer(Xdisplay);
		grabbed = True;
		XMapRaised(Xdisplay, src);
		XSync(Xdisplay, False);
	    /* now we have to wait for our window to become mapped - waiting for Expose */
		for (tick_count = 0;
		     !XCheckWindowEvent(Xdisplay, src, ExposureMask, &event)
		     && tick_count < 100; tick_count++)
			sleep_a_little(100);

		if (tick_count < 100)
		{
			if (trg == None)
				trg = CREATE_TRG_PIXMAP(width + offset_x,height + offset_y);
			if (trg != None)
			{	/* custom code to cut area, so to ungrab server ASAP */
				if (shading)
				{
					XImage         *img;

					img = XGetImage(Xdisplay, src, 0, 0,
						      width, height, AllPlanes,
						      ZPixmap);
					XDestroyWindow(Xdisplay, src);
					src = None;
					XUngrabServer(Xdisplay);
					grabbed = False;
					if (img != NULL)
					{
						ShadeXImage(img, shading, gc);
						XPutImage(Xdisplay, trg, gc,
							  img, 0, 0, offset_x,
							  offset_y, width,
							  height);
#ifdef BENCHMARK_SHADING
						{
							int             i;
							time_t          before,
							    after;
							double          diff;

							before = time(NULL);
							for (i = 0;i <BENCHMARK_SHADING; i++)
								ShadeXImage(img,shading,gc);
							after = time(NULL);

							diff = difftime(after,before);
							printf("CutPixmap(): %d shading runs took %.0f seconds\n",
							     BENCHMARK_SHADING,diff);
						}
#endif
						XDestroyImage(img);
					} else if (my_pixmap)
					{

						XFreePixmap(Xdisplay, trg);
						trg = None;
					}
				} else
					XCopyArea(Xdisplay, src, trg, gc, 0, 0,width, height, offset_x,
						  offset_y);
			}
		}

		if (src)
			XDestroyWindow(Xdisplay, src);
		if (grabbed)
			XUngrabServer(Xdisplay);
		return trg;
	}
    /* we have root pixmap ID */
    /* find out our coordinates relative to the root window */
	if (x + width > src_w || y + height > src_h)
	{			/* tiled pixmap processing here */
		Pixmap          tmp;

		width = min(width, src_w);
		height = min(height, src_h);

		tmp = CREATE_TRG_PIXMAP(width, height);
		if (tmp != None)
		{
			ShadeTiledPixmap(src, tmp, src_w, src_h, x, y, width,
					 height, gc, shading);
			if (trg == None)
				trg = CREATE_TRG_PIXMAP(width + offset_x,height + offset_y);
			if (trg != None)
				XCopyArea(Xdisplay, tmp, trg, gc, 0, 0, width,
					  height, offset_x, offset_y);

			XFreePixmap(Xdisplay, tmp);
			return trg;
		}
	}

    /* create target pixmap of the size of the window */
	if (trg == None)
		trg = CREATE_TRG_PIXMAP(width + offset_x, height + offset_y);
	if (trg != None)
	{
	    /* cut area */
		CopyAndShadeArea(src, trg, x, y, width, height, offset_x,
				 offset_y, gc, shading);

	}

	return trg;
}

Pixmap
CutWinPixmap(Window win, Drawable src, int src_w, int src_h, int width,
	     int height, GC gc, ShadingInfo * shading)
{
	unsigned int    x = 0, y = 0;

	if (src == None)
	{
#ifndef TRANSPARENT
		return None;
#else
		if (!(Options & Opt_transparent))
    //if( ! (Options & TermWin.vts[TermWin.active_page].bg.transparent))
			return None;
#endif
	}

	if (!GetWinPosition(win, &x, &y))
		return None;

	return CutPixmap(src, None, x, y, src_w, src_h, width, height, gc,
			 shading);
}

/* PROTO */
Pixmap
GetRootPixmap(Atom id)
{
	Pixmap          currentRootPixmap = None;

	if (id == None)
		id = XInternAtom(Xdisplay, "_XROOTPMAP_ID", True);

	if (id != None)
	{
		Atom            act_type;
		int             act_format;
		unsigned long   nitems, bytes_after;
		unsigned char  *prop = NULL;

/*fprintf(stderr, "\n aterm GetRootPixmap(): root pixmap is set");    		    */
		if (XGetWindowProperty(Xdisplay, Xroot, id, 0, 1, 
        False, XA_PIXMAP, &act_type,
		     &act_format, &nitems, &bytes_after, &prop) == Success)
		{
			if (prop)
			{
				currentRootPixmap = *((Pixmap *) prop);
				XFree(prop);
/*fprintf(stderr, "\n aterm GetRootPixmap(): root pixmap is [%lu]", currentRootPixmap); */
			}
		}
	}
	return currentRootPixmap;
}

Pixmap
ValidatePixmap(Pixmap p, int bSetHandler, int bTransparent,
	       unsigned int *pWidth, unsigned int *pHeight)
{
	int             (*oldXErrorHandler) (Display *, XErrorEvent *);

    /* we need to check if pixmap is still valid */
	Window          root;
	int             junk;

	if (bSetHandler)
		oldXErrorHandler = XSetErrorHandler(pixmap_error_handler);

	if (bTransparent) {
		p = GetRootPixmap(None);
  }
	if (!pWidth)
		pWidth = &junk;
	if (!pHeight)
		pHeight = &junk;

	if (p != None)
	{
		if (!XGetGeometry(Xdisplay, p, &root, &junk, 
      &junk, pWidth, pHeight, &junk,&junk))
			p = None;
	}
	if (bSetHandler)
		XSetErrorHandler(oldXErrorHandler);

	return p;
}

#endif				/* LIBASIMAGE_HEADERS */
/***************************************************************************/
/*     Down below goes aterm specific functions                            */
/***************************************************************************/

//#define BG TermWin.vts[0].bg	/*for convinience */
#define BGpage TermWin.vts[page].bg

/* PROTO */
int 
f_GetMyPosition(int page, int *x, int *y) 
{
  int bRet = 0;
  int (*old) (Display *, XErrorEvent *) = XSetErrorHandler(pixmap_error_handler);
  
  bRet = GetWinPosition(TermWin.vts[page].vt,x,y);
  XSetErrorHandler(old);
  
  return bRet;
}

/* PROTO */
void 
f_FreeTargetPixmap(int page) 
{
  if(TermWin.vts[page].bg.trgPixmap != None) {
    XFreePixmap(Xdisplay, TermWin.vts[page].bg.trgPixmap);
    TermWin.vts[page].bg.trgPixmap = None;
  }
}

/* PROTO */
Pixmap f_SetSrcPixmap(int page,Pixmap p) 
{
  
  if(TermWin.vts[page].bg.srcPixmap != None && TermWin.vts[page].bg.bMySource ) {
    XFreePixmap(Xdisplay, TermWin.vts[page].bg.srcPixmap);
    TermWin.vts[page].bg.srcPixmap = p;
    TermWin.vts[page].bg.bMySource = 0;
  }
  
  TermWin.vts[page].bg.srcPixmap = p;
  TermWin.vts[page].bg.Width = 0;
  TermWin.vts[page].bg.Height = 0;
  
  if(TermWin.vts[page].bg.srcPixmap != None) {
    Window root;
    unsigned int dum,w,h;
    int dummy;
    
    if (XGetGeometry(Xdisplay, TermWin.vts[page].bg.srcPixmap, &root, &dummy, &dummy,
				 &w, &h, &dum, &dum)) {
			TermWin.vts[page].bg.Width = w;
			TermWin.vts[page].bg.Height = h;
		}
	}
}


void f_ValidateSrcPixmap(int page, int bSetHandler) {
  if( !BGpage.bMySource) {
    Pixmap new_p;
    
    new_p = ValidatePixmap(BGpage.srcPixmap,bSetHandler,
      ((Options & Opt_transparent) && BGpage.trgType != BGT_None), NULL,NULL);
    //((Options & TermWin.vts[page].bg.transparent) && BGpage.trgType != BGT_None), NULL,NULL);
    
    if( new_p != BGpage.srcPixmap) {
      f_SetSrcPixmap(page,new_p);
    }
  }
}

#ifdef _MYSTYLE_
Pixmap
RenderMyStylePixmap(MyStyle * style, Pixmap root_pmap,
		    unsigned int root_pmap_width,
		    unsigned int root_pmap_height, unsigned int width,
		    unsigned int height)
{
/*  fprintf( stderr, "Entering RenderMyStylePixmap : texture_type = %d\n", style->texture_type );
*/
	if (style->texture_type == TEXTURE_SOLID)
		return None;
	else if (style->texture_type < TEXTURE_PIXMAP)	/* gradients */
		return mystyle_make_pixmap(style, width, height, None);
	else if (style->texture_type == TEXTURE_PIXMAP)
		return style->back_icon.pix;
	else if (style->texture_type > TEXTURE_PIXMAP)
	{
		int             real_x, real_y;

		GetMyPosition(&real_x, &real_y);
		return mystyle_make_pixmap_overlay(style, real_x, real_y,
						   width, height, None);
	}
	return None;
}
#endif

/* PROTO */
void
f_RenderPixmap(int page, int DontCheckSource) 
{
	XGCValues       gcvalue;
	GC              gc;
	unsigned int    width = TermWin_TotalWidth();
	unsigned int    height = TermWin_TotalHeight();
	unsigned int    fin_width, fin_height;
	int             (*oldXErrorHandler) (Display *, XErrorEvent *);

  /* for convinience only */
	fin_width = width;
	fin_height = height;
  DXPM_ALEXIS((stderr, "f_RenderPixmap(%d), window size is %dx%d, trg_type = %d", 
    page,width, height, BGpage.trgType ));
  
  gcvalue.foreground = PixColors[Color_bg];
	gc = XCreateGC(Xdisplay, TermWin.vts[page].vt, GCForeground, &gcvalue);
  
  /* we have some nice processing of all the X errors built in */
  /* so let's not let us crash if anything goes wrong          */
	oldXErrorHandler = XSetErrorHandler(pixmap_error_handler);
  
  if (!DontCheckSource)
		f_ValidateSrcPixmap(page,0);

	if (BGpage.srcPixmap == None) {
#ifdef TRANSPARENT
		if (!(Options & Opt_transparent) || BGpage.trgType == BGT_None)
    //if( ! (Options & TermWin.vts[page].bg.transparent) || BGpage.trgType == BGT_None)
#endif
		{
			XSetErrorHandler(oldXErrorHandler);
			return;	/* nothing to do here */
		}
	}
	if (BGpage.trgPixmap != BGpage.srcPixmap) {
		f_FreeTargetPixmap(page);
  }

	switch (BGpage.trgType)
	{
#define SHADINGpage ((BGpage.bMySource)? NULL : &(BGpage.Shading))
	case BGT_Tile:		/* just copying source PixampID into trgPixmapID */
    DXPM_ALEXIS((stderr,"Tiling pixmap ...\n"));
		if (BGpage.bMySource || NO_NEED_TO_SHADE(BGpage.Shading)) {
      
			BGpage.trgPixmap = BGpage.srcPixmap;
			fin_width = BGpage.Width;
			fin_height = BGpage.Height;
      
		} else if ((BGpage.finWidth != width || BGpage.finHeight != height) &&
			   (BGpage.Width != BGpage.finWidth || BGpage.Height != BGpage.finHeight)) {
           
			fin_width = min(BGpage.Width, width);
			fin_height = min(BGpage.Height, height);
			BGpage.trgPixmap = ShadePixmap(BGpage.srcPixmap, 0, 0, fin_width,
					fin_height, gc, SHADINGpage);
		}
		break;
	case BGT_Center:	/* more complicated case */
		if (BGpage.finWidth != width || BGpage.finHeight != height)	{
      
DXPM_ALEXIS((stderr, "aterm RenderPixmap(): (BGT_Center)src pixmap is [%lu]",BGpage.srcPixmap));	
			BGpage.trgPixmap = CenterPixmap(BGpage.srcPixmap,BGpage.Width, BGpage.Height,
        width, height, gc, SHADINGpage);
		}
		break;
	case BGT_Scale:
		if (BGpage.finWidth != width || BGpage.finHeight != height) {
      DXPM_ALEXIS((stderr, "aterm RenderPixmap(): (BGT_Scale) src pixmap is [%lu]",BGpage.srcPixmap));
			BGpage.trgPixmap = ScalePixmap(BGpage.srcPixmap,
        BGpage.Width, BGpage.Height,width, height, gc, SHADINGpage);
		}
		break;
	case BGT_ScaleH:
		if (BGpage.finWidth != width)	{
			BGpage.trgPixmap = ScalePixmap(BGpage.srcPixmap,
        BGpage.Width, BGpage.Height,width, BGpage.Height, gc,SHADINGpage);
			fin_height = BGpage.Height;
		}
		break;
	case BGT_ScaleV:
		if (BGpage.finHeight != height) {
			BGpage.trgPixmap = ScalePixmap(BGpage.srcPixmap,
						   BGpage.Width, BGpage.Height,BGpage.Width, height, gc,SHADINGpage);
			fin_width = BGpage.Width;
		}
		break;
	case BGT_NoTile:
		if (BGpage.finWidth != width || BGpage.finHeight != height) {
			BGpage.trgPixmap = GrowPixmap(BGpage.srcPixmap,BGpage.Width, BGpage.Height,
        width, height, gc, SHADINGpage);
		}
		break;
	case BGT_NoTileH:
		if (BGpage.finWidth != width) {
			BGpage.trgPixmap = GrowPixmap(BGpage.srcPixmap,BGpage.Width, BGpage.Height,
        width, BGpage.Height, gc,SHADINGpage);
			fin_height = BGpage.Height;
		}
		break;
	case BGT_NoTileV:
		if (BGpage.finHeight != height)	{
			BGpage.trgPixmap = GrowPixmap(BGpage.srcPixmap,
        BGpage.Width, BGpage.Height,
        BGpage.Width, height, gc,SHADINGpage);
			fin_width = BGpage.Width;
		}
		break;
	case BGT_Cut:

DXPM_ALEXIS((stderr, "aterm RenderPixmap(): (BG_Cut)src pixmap is [%lu]",BGpage.srcPixmap)); 
		BGpage.trgPixmap = CutWinPixmap(TermWin.vts[page].vt, BGpage.srcPixmap,
      BGpage.Width, BGpage.Height,width, height, gc, SHADINGpage);
		break;
	}

	XFreeGC(Xdisplay, gc);	/* don't need anymore */
	if (BGpage.trgPixmap != None) {
		BGpage.finWidth = fin_width;
		BGpage.finHeight = fin_height;
		XSync(Xdisplay, 0);
		XSetWindowBackgroundPixmap(Xdisplay, TermWin.vts[page].vt, BGpage.trgPixmap);
		TermWin.vts[page].LastPixmapUsed = BGpage.srcPixmap;
		BGpage.trgPixmapSet = 1;

		if (BGpage.trgPixmap != BGpage.srcPixmap)
		{		/* don't need it anymore server has it */
			XFreePixmap(Xdisplay, BGpage.trgPixmap);
			XSync(Xdisplay, 0);
		}
		BGpage.trgPixmap = None;
	}
    /* restore old handler so we can crash again ;) */
	XSetErrorHandler(oldXErrorHandler);
  DXPM_ALEXIS((stderr,"Done with f_RenderPixmap\n"));
}

//******************************* RenderPixmap **********************/
#endif				// BACKGROUND_IMAGE || TRANSPARENT */

#ifdef BACKGROUND_IMAGE
//* we need this stuff only to load background image from file */

#ifdef USE_LIBASIMAGE
#ifndef LoadImage
Pixmap          LoadImageWithMask(Display * dpy, Window w,
				  unsigned long max_colors,
				  const char *realfilename, Pixmap * pMask);
#endif
#endif

int f_parse_pixmap_geom(int page,const char *geom) {
	int w = 0, h = 0, x = 0, y = 0;
	int flags, changed = 0;
  
  if(geom == NULL || !strlen(geom)) {
    return 0;
  }
  if( !strcmp(geom,"?")) {
    static char     str[] = "[10000x10000+10000+10000]";	/* should be big enough */
		sprintf(str, "[%dx%d+%d+%d]",BGpage.srcWidth, BGpage.srcHeight, BGpage.srcX, BGpage.srcY);
		f_xterm_seq(page,XTerm_title, str);
		return 0;
	}
  
  // pour recuperer les dimensions du pixmap correspondant a geom */
	flags = XParseGeometry(geom, &x, &y, (unsigned int *)&w, (unsigned int *)&h);
	if (!(flags & XValue))
		x = 0;
	if (!(flags & YValue))
		y = 0;
	if (!(flags & WidthValue))
		w = -1;
	if (!(flags & HeightValue))
		h = -1;
	MIN_IT(x, 10000);
	MIN_IT(y, 10000);
	MIN_IT(w, 10000);
	MIN_IT(h, 10000);
  // MAJ des dimensions du background */
	if (w != BGpage.srcWidth)
	{
		changed++;
		BGpage.srcWidth = w;
	}
	if (h != BGpage.srcHeight)
	{
		changed++;
		BGpage.srcHeight = h;
	}
	if (x != BGpage.srcX)
	{
		changed++;
		BGpage.srcX = x;
	}
	if (y != BGpage.srcY)
	{
		changed++;
		BGpage.srcY = y;
	}
  DXPM_ALEXIS(( stderr, "parse_pixmap_geom(): geometry is [%dx%d+%d+%d]", w,h,x,y ));
  return changed;
}

/* PROTO */
void 
f_LoadBGPixmap(int page, const char *file) 
{
  XpmAttributes xpmAttr;
  char *realfilename;
  
  assert(file != NULL);
  if(BGpage.srcPixmap != None && BGpage.bMySource) {
    XFreePixmap(Xdisplay,BGpage.srcPixmap);
    BGpage.srcPixmap = None;
  }
  
  if( *file != '\0') {
    if((realfilename = (char *)File_Find(file,".xpm")) == NULL) {
#ifndef USE_LIBASIMAGE
			print_error("couldn't load XPM file [%s]", file);
#endif
    } else {
      int filenamelen=strlen(realfilename);
      xpmAttr.closeness = 30000;
			xpmAttr.colormap = Xcmap;
			xpmAttr.visual = Xvisual;
			xpmAttr.depth = Xdepth;
			xpmAttr.valuemask = XpmCloseness | XpmColormap | XpmVisual | XpmDepth
        | XpmSize | XpmReturnPixels;
      
#ifdef USE_LIBASIMAGE
			if ((BGpage.srcPixmap = LoadImageWithMask(Xdisplay, Xroot, 256,
        realfilename, NULL)) != None) {
          
				Window          root;
				int             x, y;
				unsigned int    bw, depth;

				if (XGetGeometry(Xdisplay, BGpage.srcPixmap, &root, &x, &y,
				     &(xpmAttr.width), &(xpmAttr.height), &bw, &depth) == 0) {
        
					xpmAttr.width = 0;
					xpmAttr.height = 0;
				}
			}
#else
#ifdef USE_JPEG
			if ((filenamelen >= 4 && strncasecmp(realfilename + (filenamelen - 4), ".jpg", 4)==0) ||
			    (filenamelen >= 5 && strncasecmp(realfilename + (filenamelen - 5), ".jpeg", 5)==0)) {
			  long w = 0, h=0;
			  if (JpegReadFileToPixmap (Xdisplay, Xroot,  DefaultGC(Xdisplay, Xscreen),
			       realfilename, &BGpage.srcPixmap, &w, &h) != 0) {
		            print_error("couldn't load JPEG file [%s]",realfilename);
			  } else {
			    xpmAttr.width = w;
			    xpmAttr.height = h;
			  }
                        } else
#endif
#ifdef USE_PNG
			if ((filenamelen >= 4 && strncasecmp(realfilename + (filenamelen - 4), ".png", 4)==0)) {
			  long w = 0, h=0;
			  if (PngReadFileToPixmap (Xdisplay, Xroot,  DefaultGC(Xdisplay, Xscreen),
			       realfilename, &BGpage.srcPixmap, &w, &h) != 0) {
		            print_error("couldn't load PNG file [%s]",realfilename);
			  } else {
			    xpmAttr.width = w;
			    xpmAttr.height = h;
			  }
                        } else
#endif
			
			if (XpmReadFileToPixmap (Xdisplay, Xroot, realfilename, &BGpage.srcPixmap,
			     NULL, &xpmAttr) != 0) {
        print_error("couldn't load XPM file [%s]",realfilename);
      }
             
#endif
		}
	}
    /* need to add geometry processing code here */

	if (BGpage.srcPixmap == None)
	{
		XSetWindowBackground(Xdisplay, TermWin.vts[page].vt,PixColors[Color_bg]);
		BGpage.bMySource = 0;
		BGpage.Width = 0;
		BGpage.Height = 0;
		BGpage.trgPixmapSet = 0;
	} else {
		// printf ("Pixmap: %dx%d (%dx%d) %dx%d\n",xpmAttr.width,xpmAttr.height, BGpage.srcWidth, BGpage.srcHeight, BGpage.srcX, BGpage.srcY);
		BGpage.bMySource = 1;
		BGpage.Width = xpmAttr.width;
		BGpage.Height = xpmAttr.height;
		if (BGpage.srcWidth > 0 || BGpage.srcHeight > 0 || BGpage.srcX > 0 || BGpage.srcY > 0) {
			int             w, h;
			Pixmap          tmp;

			if (BGpage.srcWidth <= 0)
				w = BGpage.Width;
			else
				w = min(BGpage.Width - BGpage.srcX, BGpage.srcWidth);

			if (BGpage.srcHeight <= 0)
				h = BGpage.Height;
			else
				h = min(BGpage.Height - BGpage.srcY, BGpage.srcHeight);

			tmp = ShadePixmap(BGpage.srcPixmap, BGpage.srcX, BGpage.srcY, w, h,
					  DefaultGC(Xdisplay, Xscreen), &(BGpage.Shading));
			if (tmp != None)
			{
				XFreePixmap(Xdisplay, BGpage.srcPixmap);
				BGpage.srcPixmap = tmp;
				BGpage.Width = w;
				BGpage.Height = h;
			}
		} else if (!NO_NEED_TO_SHADE(BGpage.Shading))	{
			Pixmap tmp;
			GC     gc = DefaultGC(Xdisplay, Xscreen);

			tmp = ShadePixmap(BGpage.srcPixmap, 0, 0, BGpage.Width,
					BGpage.Height, DefaultGC(Xdisplay,Xscreen),
					&(BGpage.Shading));
			if (tmp != None)
			{
				XFreePixmap(Xdisplay, BGpage.srcPixmap);
				BGpage.srcPixmap = tmp;
			}
		}
		f_RenderPixmap(page,1);
	}

	//f_scr_touch(page);
  DXPM_ALEXIS((stderr,"Done with f_LoadBGPixmap\n"));

}
#endif				/* BACKGROUND_IMAGE */
