#include <stdio.h>
#include <math.h>

#include "tiff.h"
#include "MRW_Loader.h"
#include "MRW_Private.h"

/* Name must be exactly three characters long.
 */
static int
MRW_TypeCode (const unsigned char *name)
{
	int	retval;

	retval = *name++;
	retval = (retval << 8) | *name++;
	retval = (retval << 8) | *name;
	return retval;
}

int
LoadMRW (mrw_header *hdr, FILE *f, char **errmsg)
{
	int i, posn;
	int		exif_offset, manu_offset;
	int		settings_offset, pim_offset;
	unsigned char	mrmhdr[8];
static char errmsgbuf[200];

	/* Read MRM block header. */
	if (fread (&mrmhdr, sizeof(mrmhdr), 1, f) != 1)
		return 0;
	hdr->mrm.type = mrw_val (mrmhdr, 0, 4);
	hdr->mrm.length = mrw_val (mrmhdr, 4, 4);
	if (hdr->mrm.type != MRW_TypeCode ("MRM")) {
		*errmsg = "LoadMRW: MRM block had bad magic number";
		return 0;
	}

#ifdef DEBUG
	fprintf (stderr, "MRM block length: %d\n", hdr->mrm.length);
#endif
	/* Read MRM block data. */
	if ((hdr->mrm.data = (unsigned char *)malloc(hdr->mrm.length)) == (unsigned char *)0)
		return 0;
	if ((i = fread (hdr->mrm.data, 1, hdr->mrm.length, f)) != hdr->mrm.length) {
		sprintf (errmsgbuf, "LoadMRW: Unable to read MRM block data (count = %d, not %d)", i, hdr->mrm.length);
		*errmsg = errmsgbuf;
		return 0;
	}

	/* Break out PRD, TTW, WBG, and RIF subblocks. */
	posn = 0;

	hdr->prd.type = mrw_val (hdr->mrm.data, posn, 4);
	if (hdr->prd.type != MRW_TypeCode ("PRD")) {
		*errmsg = "LoadMRW: PRD block had bad magic number";
		return 0;
	}
	hdr->prd.length = mrw_val (hdr->mrm.data, posn+4, 4);
	hdr->prd.data = hdr->mrm.data + posn + 8;
	posn += hdr->prd.length + 8;

	hdr->ttw.type = mrw_val (hdr->mrm.data, posn, 4);
	if (hdr->ttw.type != MRW_TypeCode ("TTW")) {
		*errmsg = "LoadMRW: TTW block had bad magic number";
		return 0;
	}
	hdr->ttw.length = mrw_val (hdr->mrm.data, posn+4, 4);
	hdr->ttw.data = hdr->mrm.data + posn + 8;
	posn += hdr->ttw.length + 8;

#if 0
	fprintf (stderr, "WBG begins at offset %d\n", posn+8);
#endif
	hdr->wbg.type = mrw_val (hdr->mrm.data, posn, 4);
	if (hdr->wbg.type != MRW_TypeCode ("WBG")) {
		*errmsg = "LoadMRW: WBG block had bad magic number";
		return 0;
	}
	hdr->wbg.length = mrw_val (hdr->mrm.data, posn+4, 4);
	hdr->wbg.data = hdr->mrm.data + posn + 8;
	posn += hdr->wbg.length + 8;

#if 0
	fprintf (stderr, "RIF begins at offset %d\n", posn+8);
#endif
	hdr->rif.type = mrw_val (hdr->mrm.data, posn, 4);
	if (hdr->rif.type != MRW_TypeCode ("RIF")) {
		*errmsg = "LoadMRW: RIF block had bad magic number";
		return 0;
	}
	hdr->rif.length = mrw_val (hdr->mrm.data, posn+4, 4);
	hdr->rif.data = hdr->mrm.data + posn + 8;
	posn += hdr->rif.length + 8;

	/* Parse IFD's in the ttw data. */
	hdr->mrw_ifd = ParseIFD (hdr->ttw.data,
				mrw_val (hdr->ttw.data, MRW_TIFFIFD0_OFFSET, MRW_TIFFIFD0_LENGTH));
	if (hdr->mrw_ifd == (IFD *)0) {
		sprintf (errmsgbuf, "LoadMRW: Unable to read image directory");
		*errmsg = errmsgbuf;
		return 0;
	}

	exif_offset = IFD_GetLong (hdr->mrw_ifd, IFDTAG_EXIFOFFSET);
#ifdef DEBUG
	fprintf (stderr, "EXIF_OFFSET = %d\n", exif_offset);
#endif
	hdr->exif_ifd = ParseIFD (hdr->ttw.data, exif_offset);
	if (hdr->exif_ifd == (IFD *)0) {
		sprintf (errmsgbuf, "LoadMRW: Unable to read EXIF directory");
		*errmsg = errmsgbuf;
		return 0;
	}

	pim_offset = IFD_GetUndefined (hdr->mrw_ifd, IFDTAG_PIMOFFSET);
#ifdef DEBUG
	fprintf (stderr, "PIM_OFFSET = %d\n", pim_offset);
#endif
	if (pim_offset == 0)
		hdr->pimdata = NULL;
	else
		hdr->pimdata = hdr->ttw.data + pim_offset;

	manu_offset = IFD_GetUndefined (hdr->exif_ifd, EXIFTAG_MANUFNOTES);
#ifdef DEBUG
	fprintf (stderr, "MANU_OFFSET = %d\n", manu_offset);
#endif

	hdr->manu_ifd = ParseIFD (hdr->ttw.data, manu_offset);
	if (hdr->manu_ifd == (IFD *)0) {
		sprintf (errmsgbuf, "LoadMRW: Unable to read Manufacturer directory\n");
		*errmsg = errmsgbuf;
		return 0;
	}
	settings_offset = IFD_GetUndefined (hdr->manu_ifd, MANUTAG_CAMERASETTINGS);
#ifdef DEBUG
	fprintf (stderr, "SETTINGS_OFFSET = %d\n", settings_offset);
#endif
	if (settings_offset == 0) {
#ifdef DEBUG
		fprintf (stderr, "Trying tag 0x0003\n");
#endif
		settings_offset = IFD_GetUndefined (hdr->manu_ifd, MANUTAG_CAMERASETTINGS2);
#ifdef DEBUG
		fprintf (stderr, "SETTINGS_OFFSET = %d\n", settings_offset);
#endif
	}
	if (settings_offset == 0) {
		sprintf (errmsgbuf, "LoadMRW: Unable to find camera settings info");
		*errmsg = errmsgbuf;
		return 0;
	}
	hdr->settings = hdr->ttw.data + settings_offset;

	return 1;
}

unsigned
MRW_Width (const mrw_header *hdr)
{
	return mrw_val (hdr->prd.data, MRW_PRD_CCDWIDTH_OFFSET, MRW_PRD_CCDWIDTH_LENGTH);
}

unsigned
MRW_Height (const mrw_header *hdr)
{
	return mrw_val (hdr->prd.data, MRW_PRD_CCDHEIGHT_OFFSET, MRW_PRD_CCDHEIGHT_LENGTH);
}

const char *
MRW_DCFVer (const mrw_header *hdr)
{
	return IFD_GetString (hdr->mrw_ifd, IFDTAG_DCFVER);
}

const char *
MRW_Manufacturer (const mrw_header *hdr)
{
	return IFD_GetString (hdr->mrw_ifd, IFDTAG_MANUF);
}

const char *
MRW_CameraName (const mrw_header *hdr)
{
	return IFD_GetString (hdr->mrw_ifd, IFDTAG_CAMERA);
}

const char *
MRW_Firmware (const mrw_header *hdr)
{
	return IFD_GetString (hdr->mrw_ifd, IFDTAG_FIRMWARE);
}

const char *
MRW_DateTime (const mrw_header *hdr)
{
	return IFD_GetString (hdr->mrw_ifd, IFDTAG_DATETIME);
}

int
MRW_ISO (const mrw_header *hdr)
{
	return IFD_GetShort (hdr->exif_ifd, EXIFTAG_ISOSPEED);
}

int
MRW_ThumbNail (const mrw_header *hdr)
{
	return IFD_GetShort (hdr->manu_ifd, MANUTAG_THUMBNAIL);
}

#if 0
static struct _shutter_table {
	int	val;
	char   *speed;
} shutter_table[] = {
	{ -20, "4" },
	{ -15, "3" },
	{ -10, "2" },
	{ -5, "1.5" },
	{ 0, "1" },
	{ 5, "0.7" },
	{ 10, "1/2" },
	{ 15, "1/3" },
	{ 20, "1/4" },
	{ 25, "1/6" },
	{ 30, "1/8" },
	{ 35, "1/10" },
	{ 40, "1/15" },
	{ 45, "1/20" },
	{ 50, "1/30" },
	{ 55, "1/45" },
	{ 60, "1/60" },
	{ 65, "1/90" },
	{ 70, "1/125" },
	{ 75, "1/180" },
	{ 80, "1/250" },
	{ 85, "1/350" },
	{ 90, "1/500" },
	{ 95, "1/750" },
	{ 100, "1/1000" },
	{ 105, "1/1500" },
	{ 110, "1/2000" },
};
#endif

double
MRW_Shutter (const mrw_header *hdr)
{
	int num, denom;

	if (IFD_GetSRational (hdr->exif_ifd, EXIFTAG_SHUTTERSPEED, &num, &denom))
		return pow (2.0, -num/(double)denom);
	else if (IFD_GetRational (hdr->exif_ifd, EXIFTAG_EXPOSURETIME, &num, &denom))
		return num/(double)denom;
	else
		return 0;
}

#if 0
static struct _aperture_table {
	int	val;
	char	*aperture;
} aperture_table[] = {
	{ 30, "2.8" },
	{ 35, "3.5" },
	{ 40, "4.0" },
	{ 45, "4.5" },
	{ 50, "5.6" },
	{ 55, "6.7" },
	{ 60, "8.0" },
	{ 65, "9.5" },
};
#endif

double
MRW_Aperture (const mrw_header *hdr)
{
	int num, denom;
	if (IFD_GetRational (hdr->exif_ifd, EXIFTAG_APERTURE, &num, &denom))
                return pow (2.0, 0.5 * num/(double)denom);
	else if (IFD_GetRational (hdr->exif_ifd, EXIFTAG_FNUMBER, &num, &denom))
                return num/(double)denom;
	else
		return 0;
}

double
MRW_FocalLen (const mrw_header *hdr)
{
	int num, denom;
	if (IFD_GetRational (hdr->exif_ifd, EXIFTAG_FOCALLEN, &num, &denom))
                return num/(double)denom;
	return 0;
}

double
MRW_FocalLen35mm (const mrw_header *hdr)
{
	double truelen = MRW_FocalLen (hdr);
	double CCDsize;  /* In mm. */

	if (strcmp (MRW_CameraName (hdr), "DiMAGE 5") == 0)
		CCDsize = 8.933;
	else
		CCDsize = 11.0;
	return 43.2666153 / CCDsize * truelen;
}

double
MRW_FocusLen (const mrw_header *hdr)
{
	return mrw_val (hdr->settings, MRW_FOCUS_OFFSET, MRW_FOCUS_LENGTH);
}

int
MRW_Macro (const mrw_header *hdr)
{
	return mrw_val (hdr->settings, MRW_MACRO_OFFSET, MRW_MACRO_LENGTH);
}

int
MRW_Flash (const mrw_header *hdr)
{
	return mrw_val (hdr->settings, MRW_FLASH_OFFSET, MRW_FLASH_LENGTH);
}

int
MRW_Pictnum (const mrw_header *hdr)
{
	return mrw_val (hdr->settings, MRW_PICTNUM_OFFSET, MRW_PICTNUM_LENGTH);
}

MRI_balance
MRW_WBalance (const mrw_header *hdr)
{
	MRI_balance b;

#ifdef OLD_GAIN_INFO
	b.rgain = mrw_val (hdr->settings, MRW_BALANCE1_OFFSET, MRW_BALANCE1_LENGTH);
	b.ggain = mrw_val (hdr->settings, MRW_BALANCE2_OFFSET, MRW_BALANCE2_LENGTH);
	b.bgain = mrw_val (hdr->settings, MRW_BALANCE3_OFFSET, MRW_BALANCE3_LENGTH);
#else
	b.rgain = mrw_val (hdr->wbg.data, MRW_WBG_RGAIN_OFFSET, MRW_WBG_RGAIN_LENGTH);
	b.ggain = mrw_val (hdr->wbg.data, MRW_WBG_G1GAIN_OFFSET, MRW_WBG_G1GAIN_LENGTH);
	b.bgain = mrw_val (hdr->wbg.data, MRW_WBG_BGAIN_OFFSET, MRW_WBG_BGAIN_LENGTH);
#endif
	return b;
}

int
MRW_Color (const mrw_header *hdr)
{
	return mrw_val (hdr->settings, MRW_COLOR_OFFSET, MRW_COLOR_LENGTH) - 3;
}

int
MRW_Contrast (const mrw_header *hdr)
{
	return mrw_val (hdr->settings, MRW_CONTRAST_OFFSET, MRW_CONTRAST_LENGTH) - 3;
}

int
MRW_FocusMode (const mrw_header *hdr)
{
	int		afOffset, afLength;

	afOffset = IFD_GetUndefined (hdr->manu_ifd, MANUTAG_AUTOFOCUS);
	afLength = IFD_GetUndefinedLength (hdr->manu_ifd, MANUTAG_AUTOFOCUS);
	if (mrw_val (hdr->ttw.data, afOffset, 4) == (signed int)0xAFAFAFAF)
		return MRI_FOCUS_AUTO;
	else
		return MRI_FOCUS_MANUAL;
}

char *preset_table[] = {
	"camera",
	"tungsten",
	"daylight",
	"cloudy",
	"fluorescent",
	"flash",
	"preset6",
};

int
MRW_GetPresetBalance (const MRI *mri, const char *presetname, MRI_balance *retval)
{
    unsigned	idx;
    
    for (idx = 0; idx < sizeof(preset_table)/sizeof(preset_table[0]); idx++)
      if (strcmp (presetname, preset_table[idx]) == 0) {
	mrw_header *hdr  = (mrw_header *)mri->hdr;
        if (idx == 0)
          *retval = MRW_WBalance (hdr);
	else {
	  idx = 8 + (idx - 1) * 4;
	  retval->rgain = mrw_val (hdr->rif.data, idx, 2);
	  retval->ggain = 256;
	  retval->bgain = mrw_val (hdr->rif.data, idx+2, 2);
	}
	return 1;
      }
    return 0;
}

struct mrw_preset_iterator {
	const MRI *mri;
	unsigned index;
};

void *
preset_iterator_next (MRI_Iterator *pi)
{
	struct mrw_preset_iterator *priv;
	priv = (struct mrw_preset_iterator *)pi->private;
	if (priv->index < sizeof(preset_table)/sizeof(preset_table[0])) {
	  return (void *)preset_table[priv->index++];
	}
	else
	  return (void *)0;
}

void
preset_iterator_close (MRI_Iterator *pi)
{
	free (pi->private);
	free (pi);
}

MRI_Iterator *
MRW_PresetIterator (const MRI *mri)
{
	MRI_Iterator *rv;
	struct mrw_preset_iterator *priv;

	priv = (struct mrw_preset_iterator *)malloc(sizeof(*priv));
	priv->mri = mri;
	priv->index = 0;
	rv->private = (void *)priv;
	rv->next = preset_iterator_next;
	rv->close = preset_iterator_close;
	return rv;
}

