/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <unistd.h>

#include "MRI.h"

struct PPMWriterData {
	int	outputBits;
	int	width;
	int	height;
	int	freedata;
	int	reversed;
	int	xvmini;
	int	errorR, errorG, errorB;
	FILE	*f;
	MRI_Link *next;
};

void
PPMWriterStart (void *private, int width, int height, int freedata)
{
	struct PPMWriterData *wd = private;
	wd->width = width;
	wd->height = height;
	wd->freedata = freedata;
	if (wd->outputBits == 16)
		fprintf (wd->f, "%d %d %d\n", width, height, 65535);
	else if (wd->outputBits == 8)
		fprintf (wd->f, "%d %d %d\n", width, height, 255);
	else {
		fprintf (stderr, "MRI_WritePPM: ERROR: Cannot output PPM file with depth %d\n", wd->outputBits);
	}
	if (wd->next)
	    (*wd->next->start) (wd->next->private, width, height, freedata);
}

static inline void
putpixel (unsigned short p, FILE *f)
{
	putc (p >> 8, f);
	putc (p & 0xFF, f);
}

void
putxvminipix (struct PPMWriterData *wd, struct MRI_ScanLine *sl, int x)
{
	int r, g, b;

	if (sl->scanLineType != LINETYPE_SHORT) {
		fprintf (stderr, "putxvminipix: unexpected line type %d\n", sl->scanLineType);
		exit (1);
	}
	wd->errorR += ((unsigned short *)(sl->R))[x];
	if (wd->errorR < 0) wd->errorR = 0;
	else if (wd->errorR > 65535) wd->errorR = 65535;
	wd->errorG += ((unsigned short *)(sl->G))[x];
	if (wd->errorG < 0) wd->errorG = 0;
	else if (wd->errorG > 65535) wd->errorG = 65535;
	wd->errorB += ((unsigned short *)(sl->B))[x];
	if (wd->errorB < 0) wd->errorB = 0;
	else if (wd->errorB > 65535) wd->errorB = 65535;
	r = (wd->errorR + (1 << 12) > 65535 ? 65535 : (wd->errorR + (1<<12))) * 7 / 65535;
	g = (wd->errorG + (1 << 12) > 65535 ? 65535 : (wd->errorG + (1<<12))) * 7 / 65535;
	b = (wd->errorB + (1 << 13) > 65535 ? 65535 : (wd->errorB + (1<<13))) * 3 / 65535;
	fputc ((r << 5) | (g << 2) | b, wd->f);
	#if 1
	wd->errorR -= r * 65535 / 7;
	wd->errorG -= g * 65535 / 7;
	wd->errorB -= b * 65535 / 3;
	#else
	wd->errorR = 0;
	wd->errorG = 0;
	wd->errorB = 0;
	#endif
}

void
PPMWriterRowShort (void *private, void *data)
{
	struct PPMWriterData *wd = private;
	struct MRI_ScanLine *sl = data;
	int x;

	if (wd->xvmini) {
	    wd->errorR = wd->errorG = wd->errorB = 0;
	    if (wd->reversed) {
		    for (x = wd->width; --x >= 0; )
			putxvminipix (wd, sl, x);
	    }
	    else {
		    for (x = 0; x < wd->width; x++)
			putxvminipix (wd, sl, x);
	    }
	}
	else if (wd->outputBits == 16) {
	    unsigned short *R = sl->R;
	    unsigned short *G = sl->G;
	    unsigned short *B = sl->B;
	    if (wd->reversed) {
		    for (x = wd->width; --x >= 0; ) {
			putpixel (R[x], wd->f);
			putpixel (G[x], wd->f);
			putpixel (B[x], wd->f);
		    }
	    }
	    else {
		    for (x = 0; x < wd->width; x++) {
			putpixel (R[x], wd->f);
			putpixel (G[x], wd->f);
			putpixel (B[x], wd->f);
		    }
	    }
	}
	else if (wd->outputBits == 8) {
	    unsigned short *R = sl->R;
	    unsigned short *G = sl->G;
	    unsigned short *B = sl->B;
	    if (wd->reversed) {
		    for (x = wd->width; --x >= 0; ) {
			putc (R[x] >> 8, wd->f);
			putc (G[x] >> 8, wd->f);
			putc (B[x] >> 8, wd->f);
		    }
	    }
	    else {
		    for (x = 0; x < wd->width; x++) {
			putc (R[x] >> 8, wd->f);
			putc (G[x] >> 8, wd->f);
			putc (B[x] >> 8, wd->f);
		    }
	    }
	}
	else {
		fprintf (stderr, "PPMWriterRow: unknown bits %d\n", wd->outputBits);
		exit (1);
	}
}

static inline unsigned short FtoS (float f)
{
	if (f < 0) return 0;
	if (f > 65535) return 65535;
	return f;
}

static inline unsigned char FtoC (float f)
{
	f /= 256.0;
	if (f < 0) return 0;
	if (f > 255) return 255;
	return f;
}


void
PPMWriterRowFloat (void *private, void *data)
{
	struct PPMWriterData *wd = private;
	struct MRI_ScanLine *sl = data;
	int x;

	if (wd->xvmini) {
	    wd->errorR = wd->errorG = wd->errorB = 0;
	    if (wd->reversed) {
		    for (x = wd->width; --x >= 0; )
			putxvminipix (wd, sl, x);
	    }
	    else {
		    for (x = 0; x < wd->width; x++)
			putxvminipix (wd, sl, x);
	    }
	}
	else if (wd->outputBits == 16) {
	    float *R = sl->R;
	    float *G = sl->G;
	    float *B = sl->B;
	    if (wd->reversed) {
		    for (x = wd->width; --x >= 0; ) {
			putpixel (FtoS(R[x]), wd->f);
			putpixel (FtoS(G[x]), wd->f);
			putpixel (FtoS(B[x]), wd->f);
		    }
	    }
	    else {
		    for (x = 0; x < wd->width; x++) {
			putpixel (FtoS(R[x]), wd->f);
			putpixel (FtoS(G[x]), wd->f);
			putpixel (FtoS(B[x]), wd->f);
		    }
	    }
	}
	else if (wd->outputBits == 8) {
	    float *R = sl->R;
	    float *G = sl->G;
	    float *B = sl->B;
	    if (wd->reversed) {
		    for (x = wd->width; --x >= 0; ) {
			putc (FtoC(R[x]), wd->f);
			putc (FtoC(G[x]), wd->f);
			putc (FtoC(B[x]), wd->f);
		    }
	    }
	    else {
		    for (x = 0; x < wd->width; x++) {
			putc (FtoC(R[x]), wd->f);
			putc (FtoC(G[x]), wd->f);
			putc (FtoC(B[x]), wd->f);
		    }
	    }
	}
	else {
		fprintf (stderr, "PPMWriterRow: unknown bits %d\n", wd->outputBits);
		exit (1);
	}
}

void
PPMWriterRow (void *private, void *data)
{
	struct PPMWriterData *wd = private;
	struct MRI_ScanLine *sl = data;
	int x;

	if (sl->scanLineType == LINETYPE_SHORT) {
	    PPMWriterRowShort (private, data);
	}
	else if (sl->scanLineType == LINETYPE_FLOAT) {
	    PPMWriterRowFloat (private, data);
	}
	else {
		fprintf (stderr, "PPMWriterRow: unexpected line type %d\n", sl->scanLineType);
		exit (1);
	}
	if (wd->next != NULL)
	    (*wd->next->row) (wd->next->private, sl);
	else if (wd->freedata)
	    MRI_FreeScanLine (sl);
}

void
PPMWriterClose (void *private)
{
    struct PPMWriterData *wd = private;
    if (wd->next != NULL) {
	(*wd->next->close) (wd->next->private);
        free (wd->next);
    }
    free (wd);
}

MRI_Link *
MRI_GenPPMWriterTee (MRI_Link *next, int outputBits, int reversed, FILE *f)
{
	struct PPMWriterData *wd = malloc (sizeof (struct PPMWriterData));
	struct link *ep = malloc (sizeof (*ep));
	if (wd == (struct PPMWriterData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	ep->start = PPMWriterStart;
	ep->row = PPMWriterRow;
	ep->close = PPMWriterClose;
	ep->private = wd;
	wd->xvmini = 0;
	wd->outputBits = outputBits;
	if (outputBits != 8 && outputBits != 16) {
		fprintf (stderr, "PPMWriterRow: unknown bits %d\n", wd->outputBits);
		exit (1);
	}
	fprintf (f, "P6 ");
	wd->reversed = reversed;
	wd->f = f;
	wd->next = next;
	return ep;
}

MRI_Link *
MRI_GenPPMWriter (int outputBits, int reversed, FILE *f)
{
	return MRI_GenPPMWriterTee (NULL, outputBits, reversed, f);
}


struct link *
MRI_GenXVMiniWriter (MRI *mri, int reversed, FILE *f)
{
	struct PPMWriterData *wd = malloc (sizeof (struct PPMWriterData));
	struct link *ep = malloc (sizeof (*ep));
	if (wd == (struct PPMWriterData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	ep->start = PPMWriterStart;
	ep->row = PPMWriterRow;
	ep->close = PPMWriterClose;
	ep->private = wd;
	wd->xvmini = 1;
	wd->outputBits = 8;
	wd->reversed = reversed;
	wd->f = f;
	fprintf (f, "P7 332\n");
	fprintf (f, "#IMGINFO:%dx%d RGB (%d bytes)\n", MRI_GetWidth(mri), MRI_GetHeight(mri), MRI_GetWidth(mri)*MRI_GetHeight(mri)*3);
	fprintf (f, "#END_OF_COMMENTS\n");
	return ep;
}
