/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <math.h>
#include <string.h>
#include "MRI.h"
#include "vmedian.h"

struct ColorBalanceData {
	struct link *next;
	int	width;
	int	height;
	MRI_balance bal;
};

static void
ColorBalanceStart (void *private, int width, int height, int freedata)
{
	struct ColorBalanceData *wd = private;

	wd->width = width;
	wd->height = height;
	(*wd->next->start) (wd->next->private, width, height, freedata);
}

static void
ColorBalanceRow (void *private, void *data)
{
	struct ColorBalanceData *wd = private;
	struct MRI_ScanLine *sl = data;
	int x;
	unsigned short *R = sl->R;
	unsigned short *G = sl->G;
	unsigned short *B = sl->B;

	for (x = 0; x < wd->width; x++) {
		int tr = (R[x] * (int)wd->bal.rgain) >> 8;
		if (tr > 0xFFFF) tr = 0xFFFF;
		R[x] = tr;
	}
	for (x = 0; x < wd->width; x++) {
		int tg = (G[x] * (int)wd->bal.ggain) >> 8;
		if (tg > 0xFFFF) tg = 0xFFFF;
		G[x] = tg;
	}
	for (x = 0; x < wd->width; x++) {
		int tb = (B[x] * (int)wd->bal.bgain) >> 8;
		if (tb > 0xFFFF) tb = 0xFFFF;
		B[x] = tb;
	}
	(*wd->next->row) (wd->next->private, sl);
}

struct LimitCorrectData {
	struct link *next;
	int	width;
	int	height;
	MRI_balance bal;
	struct MRI_ScanLine *P, *C, *N;
	int	redGreen;
	int	row;
	int	freedata;
};

static void
LimitCorrectStart (void *private, int width, int height, int freedata)
{
	struct LimitCorrectData *wd = private;

	wd->width = width;
	wd->height = height;
	wd->freedata = freedata;
	wd->row = 0;
	wd->P = MRI_NewScanLine (LINETYPE_FLOAT, width);
	wd->C = MRI_NewScanLine (LINETYPE_FLOAT, width);
	wd->N = MRI_NewScanLine (LINETYPE_FLOAT, width);
	(*wd->next->start) (wd->next->private, width, height, TRUE);
}

static void
LimitRedGreenRow (struct MRI_ScanLine *out,
		  int width,
		  struct MRI_ScanLine *P,
		  struct MRI_ScanLine *C,
		  struct MRI_ScanLine *N)
{
    int x;

    for (x = 0; x < width; x++) {
	    if ((x % 2) == 0) {
		/* Red. */
		float b, m = ((float *) C->R)[x];
		if (C->masks[x] & 7) {
		    int n = 0; double tot = 0.0;
		    if (P && (b = ((float *) P->G)[x]) > 0.0)
			{ n++; tot += b; }
		    if (N && (b = ((float *) N->G)[x]) > 0.0)
			{ n++; tot += b; }
		    if (x > 0) {
			if (P && (b = ((float *) P->B)[x - 1]) > 0.0)
			    { n++; tot += b; }
			if ((b = ((float *) C->G)[x - 1]) > 0.0)
			    { n++; tot += b; }
			if (N && (b = ((float *) N->B)[x - 1]) > 0.0)
			    { n++; tot += b; }
		    }
		    if (x < width - 1) {
			if (P && (b = ((float *) P->B)[x + 1]) > 0.0)
			    { n++; tot += b; }
			if ((b = ((float *) C->G)[x + 1]) > 0.0)
			    { n++; tot += b; }
			if (N && (b = ((float *) N->B)[x + 1]) > 0.0)
			    { n++; tot += b; }
		    }
		    if (n && tot/n > m) m = tot/n;
		}
		((float *) out->R)[x] = m;
		((float *) out->G)[x] = 0.0;
		((float *) out->B)[x] = 0.0;
	    } else {
		/* Green. */
		float b, m = ((float *) C->G)[x];
		if (C->masks[x] & 7) {
		    int n = 0; double tot = 0.0;
		    if (P && (b = ((float *) P->B)[x]) > 0.0)
			{ tot += b; n++; }
		    if (N && (b = ((float *) N->B)[x]) > 0.0)
			{ tot += b; n++; }
		    if (x > 0) {
			if ((b = ((float *) C->R)[x - 1]) > 0.0)
			    { tot += b; n++; }
		    }
		    if (x < width - 1) {
			if ((b = ((float *) C->R)[x + 1]) > 0.0)
			    { tot += b; n++; }
		    }
		    if (n && tot/n > m) m = tot/n;
		}
		((float *) out->R)[x] = 0.0;
		((float *) out->G)[x] = m;
		((float *) out->B)[x] = 0.0;
	    }
    }
}

static void
LimitGreenBlueRow (struct MRI_ScanLine *out,
		   int width,
		   struct MRI_ScanLine *P,
		   struct MRI_ScanLine *C,
		   struct MRI_ScanLine *N)
{
    int x;

    for (x = 0; x < width; x++) {
	    if ((x % 2) == 1) {
		/* Blue. */
		float b, m = ((float *) C->B)[x];
		if (C->masks[x] & 7) {
		    int n = 0; double tot = 0.0;
		    if (P && (b = ((float *) P->G)[x]) > 0.0)
			{ n++; tot += b; }
		    if (N && (b = ((float *) N->G)[x]) > 0.0)
			{ n++; tot += b; }
		    if (x > 0) {
			if (P && (b = ((float *) P->R)[x - 1]) > 0.0)
			    { n++; tot += b; }
			if ((b = ((float *) C->G)[x - 1]) > 0.0)
			    { n++; tot += b; }
			if (N && (b = ((float *) N->R)[x - 1]) > 0.0)
			    { n++; tot += b; }
		    }
		    if (x < width - 1) {
			if (P && (b = ((float *) P->R)[x + 1]) > 0.0)
			    { n++; tot += b; }
			if ((b = ((float *) C->G)[x + 1]) > 0.0)
			    { n++; tot += b; }
			if (N && (b = ((float *) N->R)[x + 1]) > 0.0)
			    { n++; tot += b; }
		    }
		    if (n && tot/n > m) m = tot / n;
		}
		((float *) out->R)[x] = 0.0;
		((float *) out->G)[x] = 0.0;
		((float *) out->B)[x] = m;
	    } else {
		/* Green. */
		float b, m = ((float *) C->G)[x];
		if (C->masks[x] & 7) {
		    int n = 0; double tot = 0.0;
		    if (P && (b = ((float *) P->R)[x]) > 0.0)
			{ n++; tot += b; }
		    if (N && (b = ((float *) N->R)[x]) > 0.0)
			{ n++; tot += b; }
		    if (x > 0) {
			if ((b = ((float *) C->B)[x - 1]) > 0.0)
			    { n++; tot += b; }
		    }
		    if (x < width - 1) {
			if ((b = ((float *) C->B)[x + 1]) > 0.0)
			    { n++; tot += b; }
		    }
		    if (n && tot/n > m) m = tot/n;
		}
		((float *) out->R)[x] = 0.0;
		((float *) out->G)[x] = m;
		((float *) out->B)[x] = 0.0;
	    }
    }
}

static void
LimitCorrectRow (void *private, void *data)
{
	struct LimitCorrectData *wd = private;
	struct MRI_ScanLine *sl = data;
	int x;

	unsigned short *R = sl->R;
	unsigned short *G = sl->G;
	unsigned short *B = sl->B;
	float *oR = wd->N->R;
	float *oG = wd->N->G;
	float *oB = wd->N->B;

	if (wd->redGreen) {
	    for (x = 0; x < wd->width; x+=2) {
		oR[x] = R[x] * (wd->bal.rgain / 256.0);
		oG[x] = 0.0;
		oB[x] = 0.0;
		oR[x+1] = 0.0;
		oG[x+1] = G[x+1] * (wd->bal.ggain / 256.0);
		oB[x+1] = 0.0;
		wd->N->masks[x] = sl->masks[x];
		wd->N->masks[x+1] = sl->masks[x+1];
	    }
	    if (wd->row > 0) {
	        struct MRI_ScanLine *ssl = MRI_NewScanLine (LINETYPE_FLOAT, wd->width);
	        /* Previous line was GreenBlue. */
	        LimitGreenBlueRow (ssl, wd->width, wd->row > 1 ? wd->P : NULL, wd->C, wd->N);
	        (*wd->next->row) (wd->next->private, ssl);
	    }
	    wd->redGreen = FALSE;
	}
	else {
	    for (x = 0; x < wd->width; x+=2) {
		oR[x] = 0.0;
		oG[x] = G[x] * (wd->bal.ggain / 256.0);
		oB[x] = 0.0;
		oR[x+1] = 0.0;
		oG[x+1] = 0.0;
		oB[x+1] = B[x+1] * (wd->bal.bgain / 256.0);
		wd->N->masks[x] = sl->masks[x];
		wd->N->masks[x+1] = sl->masks[x+1];
	    }
	    if (wd->row > 0) {
	        struct MRI_ScanLine *ssl = MRI_NewScanLine (LINETYPE_FLOAT, wd->width);
	        /* Previous line was RedGreen. */
	        LimitRedGreenRow (ssl, wd->width, wd->row > 1 ? wd->P : NULL, wd->C, wd->N);
	        (*wd->next->row) (wd->next->private, ssl);
	    }
	    wd->redGreen = TRUE;
	}
	wd->row++;

	/* Rotate line pointers. */
	{
	    struct MRI_ScanLine *tmp = wd->P;
	    wd->P = wd->C;
	    wd->C = wd->N;
	    wd->N = tmp;
	}
	if (wd->freedata)
	    MRI_FreeScanLine (sl);
}

static void
ColorBalanceClose (void *private)
{
	struct ColorBalanceData *wd = private;
	(*wd->next->close) (wd->next->private);
	free (wd->next);
	free (wd);
}

static void
LimitCorrectClose (void *private)
{
	struct LimitCorrectData *wd = private;
	if (wd->row > 0) {
	    struct MRI_ScanLine *ssl = MRI_NewScanLine (LINETYPE_FLOAT, wd->width);
	    if (!wd->redGreen)
	        LimitRedGreenRow (ssl, wd->width, wd->row > 1 ? wd->P : NULL, wd->C, NULL);
	    else
	        LimitGreenBlueRow (ssl, wd->width, wd->row > 1 ? wd->P : NULL, wd->C, NULL);
	    (*wd->next->row) (wd->next->private, ssl);
	}
	(*wd->next->close) (wd->next->private);
	MRI_FreeScanLine (wd->P);
	MRI_FreeScanLine (wd->C);
	MRI_FreeScanLine (wd->N);
	free (wd->next);
	free (wd);
}

struct link *
MRI_GenColorBalancer (MRI_balance bal, struct link *next)
{
	struct ColorBalanceData *wd = malloc (sizeof (struct ColorBalanceData));
	struct link *ep = malloc (sizeof (*ep));
	if (wd == (struct ColorBalanceData *)0 ||
	    ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory!\n");
		exit (1);
	}
	ep->start = ColorBalanceStart;
	ep->row = ColorBalanceRow;
	ep->close = ColorBalanceClose;
	ep->private = wd;
	wd->next = next;
	wd->bal = bal;

	return ep;
}

struct link *
MRI_GenLimitCorrecter (MRI_balance bal, struct link *next, int redGreen)
{
	struct LimitCorrectData *wd = malloc (sizeof (struct LimitCorrectData));
	struct link *ep = malloc (sizeof (*ep));
	if (wd == NULL || ep == NULL) {
		fprintf (stderr, "Error: unable to allocate memory!\n");
		exit (1);
	}
	ep->start = LimitCorrectStart;
	ep->row = LimitCorrectRow;
	ep->close = LimitCorrectClose;
	ep->private = wd;
	wd->next = next;
	wd->bal = bal;
	wd->redGreen = redGreen;

	return ep;
}
