/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Richard Hult <rhult@codefactory.se>
 * Copyright (C) 2001 Anders Carlsson <andersca@codefactory.se>
 * Copyright (C) 2001 Mikael Hallendal <micke@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <bonobo.h>
#include <bonobo/bonobo-shlib-factory.h>
#include <glib.h>
#include <gal/e-table/e-cell-combo.h>
#include <gal/e-table/e-cell-popup.h>
#include <gal/e-table/e-cell-spin-button.h>
#include <gal/e-table/e-cell-text.h>
#include <gal/e-table/e-table.h>
#include <gal/e-table/e-table-extras.h>
#include <gal/e-table/e-table-scrolled.h>
#include <gal/widgets/e-scroll-frame.h>
#include <liboaf/liboaf.h>
#include <liboaf/oaf-async.h>
#include "util/corba-utils.h"
#include "libmrproject/GNOME_MrProject.h"
#include "client/manager-clients/resource-manager-client.h"
#include "resource-component.h"
#include "print-resources.h"
#include "resource-table-model.h"
#include "group-dialog.h"
#include "client/widgets/mr-message-box.h"

#ifdef HAVE_EVOLUTION_COMPOSER_IDL
#include "libmrproject/Composer.h"
#endif

#define OAFIID "OAFIID:GNOME_MrProject_ResourceComponent"
#define COMPOSER_IID "OAFIID:GNOME_Evolution_Mail_Composer"

#define DEBUG 0
#include "util/debug.h"

/* GtkObject. */
static void resource_component_init       (ResourceComponent       *component);
static void resource_component_class_init (ResourceComponentClass  *klass);

static BonoboControl * rc_control_new     (ResourceComponent       *component);

static void rc_cmd_insert                 (BonoboUIComponent       *component, 
					   gpointer                 data, 
					   const char              *cname);
static void rc_cmd_remove                 (BonoboUIComponent       *component, 
					   gpointer                 data, 
					   const char              *cname);
static void rc_cmd_print                  (BonoboUIComponent       *component, 
					   gpointer                 data, 
					   const char              *cname);
static void rc_cmd_group                  (BonoboUIComponent       *component, 
					   gpointer                 data, 
					   const char              *cname);

static void rc_resource_inserted_cb       (ResourceComponent       *component,
					   GM_Resource             *resource,
					   gpointer                 user_data);
static void rc_resources_removed_cb       (ResourceComponent       *component,
					   GSList                  *resources,
					   gpointer                 user_data);
static void rc_resource_updated_cb        (ResourceComponent       *component,
					   GM_Resource             *resource,
					   gpointer                 user_data);
static void rc_items_clear_cb             (ResourceComponent       *component);
static void rc_group_inserted_cb          (ResourceComponent       *component,
					   GM_ResourceGroup        *group,
					   gpointer                 user_data);
static void rc_group_updated_cb           (ResourceComponent       *component,
					   GM_ResourceGroup        *group,
					   gpointer                 user_data);
static void rc_group_removed_cb           (ResourceComponent       *component,
					   GM_Id                    group_id,
					   gpointer                 user_data);
static void rc_group_default_set_cb       (ResourceComponent       *component,
					   GM_Id                    group_id,
					   gpointer                 user_data);
static gboolean rc_add_to_shell           (ShellComponent          *component,
					   GM_Shell                 shell);

static void rc_remove_from_shell          (ShellComponent          *component);

static void rc_set_debug_flags            (ShellComponent          *component,
					   glong                    flags);

static void rc_resource_gui_updated_cb    (GtkObject               *obj,
					   gpointer                 resource,
					   gpointer                 user_data);

static void rc_group_gui_updated_cb       (GtkObject               *obj,
					   GM_ResourceGroup        *group,
					   gpointer                 user_data);

static void rc_group_gui_added_cb         (GtkObject               *obj,
					   gpointer                 user_data);

static void rc_group_gui_deleted_cb       (GtkObject               *obj,
					   gint                     group_id,
					   gpointer                 user_data);

static void rc_group_gui_default_set_cb   (GtkObject               *obj,
					   GM_Id                    group_id,
					   gpointer                 user_data);
static gint rc_float_compare              (gconstpointer            a,
					   gconstpointer            b);

static gboolean rc_create_table           (ResourceComponent       *component);

static void rc_set_sensitivity            (ResourceComponent       *component);

#ifdef HAVE_EVOLUTION_COMPOSER_IDL
static void rc_right_click_cb             (ETable                  *table,
					   int                      row,
					   int                      col,
					   GdkEvent                *event,
					   ResourceComponent       *rc);

#endif /* HAVE_EVOLUTION_COMPOSER_IDL */

static void rc_table_selection_change     (ETable                  *table,
					   ResourceComponent       *rc);


#define PARENT_TYPE SHELL_COMPONENT_TYPE
static GtkObjectClass *parent_class;

/* Private members. */
struct _ResourceComponentPriv {
	BonoboControl         *control;
	ResourceManagerClient *rm_client;
	GtkWidget	      *table;
	ResourceTableModel    *rtm;
	GtkWidget             *group_dialog;
	ECell                 *group_cell;
	gint                   resource_changed_id;
};

/* Send information to project-server */
static void
rc_cmd_insert (BonoboUIComponent *component, gpointer data, const char *cname)
{
 	ResourceComponent     *resource_component; 
	ResourceComponentPriv *priv;
	CORBA_Environment      ev;
	
	g_return_if_fail (data != NULL);
	g_return_if_fail (IS_RESOURCE_COMPONENT (data));
	
	d(puts (__FUNCTION__));

	resource_component = RESOURCE_COMPONENT (data);
	priv               = resource_component->priv;

	CORBA_exception_init (&ev);

	resource_mc_insert_resource (priv->rm_client, NULL, &ev);
	
	CORBA_exception_free (&ev);
}

typedef struct {
	ResourceComponent *rc;
	GSList            *list;
} SelectionData;

static void
rc_foreach_remove (gint row, SelectionData *data)
{
	ResourceComponentPriv *priv;
	GM_Resource           *resource;
	
	priv = data->rc->priv;
	
	resource = resource_tm_get_resource_on_row (priv->rtm, row);
	
	if (!resource) {
		return;		
	}
	
	data->list = g_slist_prepend (data->list, 
				      GINT_TO_POINTER (resource->resourceId));
}

/* Send information to project-server */
static void
rc_cmd_remove (BonoboUIComponent *component, gpointer data,  const char *cname)
{
	ResourceComponent     *resource_component;
	ResourceComponentPriv *priv;
	ETable                *etable;
	CORBA_Environment      ev;
	SelectionData         *sd;
	
	g_return_if_fail (data != NULL);
	g_return_if_fail (IS_RESOURCE_COMPONENT (data));

	d(puts (__FUNCTION__));

	resource_component = RESOURCE_COMPONENT (data);
	priv               = resource_component->priv;

	etable = e_table_scrolled_get_table (E_TABLE_SCROLLED (priv->table));

	sd       = g_new0 (SelectionData, 1);
	sd->rc   = resource_component;
	sd->list = NULL;

	e_table_selected_row_foreach (etable, 
				      (EForeachFunc) rc_foreach_remove,
				      sd);
	
	if (!sd->list) {
		return;
	}

	CORBA_exception_init (&ev);
	
	resource_mc_remove_resources (priv->rm_client, sd->list, &ev);
	
	if (BONOBO_EX (&ev)) {
		g_warning ("Exception when removing resources");
	}
	
	g_slist_free (sd->list);
	CORBA_exception_free (&ev);

	g_free (sd);
}

static void
rc_cmd_print (BonoboUIComponent *component, gpointer data, const char *cname)
{
	ResourceComponent     *resource_component;
	ResourceComponentPriv *priv;
	ETable                *e_table;
	
	g_return_if_fail (data != NULL);
	g_return_if_fail (IS_RESOURCE_COMPONENT (data));
	
	d(puts (__FUNCTION__));

	resource_component = RESOURCE_COMPONENT (data);
	priv               = resource_component->priv;

	e_table = e_table_scrolled_get_table (E_TABLE_SCROLLED (priv->table));
	resource_print_dialog (resource_component, e_table, FALSE);
}

static void
rc_cmd_group (BonoboUIComponent *component, gpointer data, const char *cname)
{
	ResourceComponent     *resource_component;
	ResourceComponentPriv *priv;
	GSList                *groups = NULL;
	CORBA_Environment      ev;
	GM_Id                 default_group;
	
	g_return_if_fail (data != NULL);
	g_return_if_fail (IS_RESOURCE_COMPONENT (data));

	d(puts (__FUNCTION__));
	
	resource_component = RESOURCE_COMPONENT (data);
	priv               = resource_component->priv;
	
	if (priv->group_dialog && GTK_WIDGET (priv->group_dialog)->window) {
		gdk_window_raise (GTK_WIDGET(priv->group_dialog)->window);
	} else {
		
		CORBA_exception_init (&ev);
		
		groups = resource_mc_get_all_groups (priv->rm_client, &ev);
		default_group = resource_mc_get_default_group (priv->rm_client,
							       &ev);
		
		CORBA_exception_free (&ev);

		priv->group_dialog = group_dialog_new (groups);
		
		if (default_group > 0) {
			group_dialog_set_default_group (GROUP_DIALOG (priv->group_dialog),
							default_group);
		}
		
		
		gtk_signal_connect (GTK_OBJECT (priv->group_dialog),
				    "group_add",
				    GTK_SIGNAL_FUNC (rc_group_gui_added_cb),
				    resource_component);

		gtk_signal_connect (GTK_OBJECT (priv->group_dialog),
				    "group_delete",
				    GTK_SIGNAL_FUNC (rc_group_gui_deleted_cb),
				    resource_component);

		gtk_signal_connect (GTK_OBJECT (priv->group_dialog),
				    "group_updated",
				    GTK_SIGNAL_FUNC (rc_group_gui_updated_cb),
				    resource_component);

		gtk_signal_connect (GTK_OBJECT (priv->group_dialog),
				    "group_default_set",
				    GTK_SIGNAL_FUNC (rc_group_gui_default_set_cb),
				    resource_component);
		
		gtk_widget_show (priv->group_dialog);
		
		g_slist_CORBA_free (groups);
 	}
}

/* Got signal from project-server */
static void
rc_resource_inserted_cb (ResourceComponent *component, 
			 GM_Resource       *resource,
			 gpointer           user_data)
{
	ResourceComponentPriv *priv;
	guint                  row;
	
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_RESOURCE_COMPONENT (component));

	d(puts (__FUNCTION__));
	
	priv = component->priv;
	row  = resource_tm_add_resource (priv->rtm, resource);

     	e_table_set_cursor_row (E_TABLE_SCROLLED (priv->table)->table, row);
	gtk_widget_grab_focus (priv->table);
}

static void
rc_resources_removed_cb (ResourceComponent *component,
			 GSList            *resources,
			 gpointer           user_data)
{
	ResourceComponentPriv *priv;
	gint                   row;
	GSList                *node;
	GM_Id                  id;
	ETable                *table;
	
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_RESOURCE_COMPONENT (component));

	d(puts (__FUNCTION__));

	priv = component->priv;

	for (node = resources; node; node = node->next) {
		id = GPOINTER_TO_INT (node->data);
		
		row = resource_tm_get_row (priv->rtm, id);
	
		if (row >= e_table_model_row_count (
			    E_TABLE_MODEL (priv->rtm)) - 1) {
			row = row - 1;
		}

		resource_tm_remove_resource (priv->rtm, id);

		table = e_table_scrolled_get_table (
			E_TABLE_SCROLLED (priv->table));

		if (row >= 0) {
			e_table_set_cursor_row (table, row);
		}
	}
}

/* Got signal from project-server */
static void
rc_resource_updated_cb (ResourceComponent *component,
			GM_Resource       *resource,
			gpointer           user_data)
{
	ResourceComponentPriv *priv;
	
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_RESOURCE_COMPONENT (component));
	
	d(puts (__FUNCTION__));

	priv = component->priv;
	
	resource_tm_update_resource (priv->rtm, resource);
}

static void
rc_items_clear_cb (ResourceComponent *component)
{
	ResourceComponentPriv *priv;
	
	g_return_if_fail (IS_RESOURCE_COMPONENT (component));

	priv = component->priv;

	resource_tm_remove_all_resources (priv->rtm);
	resource_tm_remove_all_groups    (priv->rtm);
}

static void
rc_group_inserted_cb (ResourceComponent *component,
		      GM_ResourceGroup  *group,
		      gpointer           user_data)
{
	ResourceComponentPriv *priv;
	
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_RESOURCE_COMPONENT (component));
	
	d(puts (__FUNCTION__));

	priv = component->priv;
	
	resource_tm_add_group (priv->rtm, group);

	e_cell_combo_set_popdown_strings (
		E_CELL_COMBO (priv->group_cell),
		resource_tm_get_group_strings (priv->rtm));

	if (priv->group_dialog && IS_GROUP_DIALOG (priv->group_dialog)) {
		group_dialog_add_group (GROUP_DIALOG(priv->group_dialog),
					group);
	}
}

static void
rc_group_removed_cb (ResourceComponent *component,
		     GM_Id              group_id,
		     gpointer           user_data)
{
	ResourceComponentPriv *priv;
	
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_RESOURCE_COMPONENT(component));

	d(puts (__FUNCTION__));

	priv = component->priv;

	resource_tm_remove_group (priv->rtm, group_id);

	e_cell_combo_set_popdown_strings (
		E_CELL_COMBO (priv->group_cell),
		resource_tm_get_group_strings (priv->rtm));

	if (priv->group_dialog && IS_GROUP_DIALOG (priv->group_dialog)) {
		group_dialog_remove_group (GROUP_DIALOG (priv->group_dialog),
					   group_id);
	}
}

static void
rc_group_default_set_cb (ResourceComponent *component,
			 GM_Id              group_id,
			 gpointer           user_data)
{
	ResourceComponentPriv *priv;
	
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_RESOURCE_COMPONENT(component));

	d(puts (__FUNCTION__));

	priv = component->priv;

	if (priv->group_dialog && IS_GROUP_DIALOG (priv->group_dialog)) {
		group_dialog_set_default_group (
			GROUP_DIALOG (priv->group_dialog),
			group_id);
	}
}

static void 
rc_group_updated_cb (ResourceComponent *component,
		     GM_ResourceGroup  *group,
		     gpointer           user_data)
{
	ResourceComponentPriv *priv;
	
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_RESOURCE_COMPONENT(component));
	
	d(puts (__FUNCTION__));

	priv = component->priv;
	
	resource_tm_update_group (priv->rtm, group);

	e_cell_combo_set_popdown_strings (
		E_CELL_COMBO (priv->group_cell),
		resource_tm_get_group_strings (priv->rtm));

}


static void
rc_resource_gui_updated_cb (GtkObject *obj, 
			    gpointer   resource, 
			    gpointer   user_data)
{
	ResourceComponent     *resource_component;
	ResourceComponentPriv *priv;
	CORBA_Environment      ev;
	
	g_return_if_fail (user_data != NULL);
	g_return_if_fail (IS_RESOURCE_COMPONENT (user_data));
	g_return_if_fail (obj != NULL);
	g_return_if_fail (IS_RESOURCE_TABLE_MODEL (obj));
	g_return_if_fail (resource != NULL);
	
	d(puts (__FUNCTION__));

	resource_component = RESOURCE_COMPONENT (user_data);
	priv               = resource_component->priv;

	CORBA_exception_init (&ev);

	resource_mc_update_resource (priv->rm_client, 
				     (GM_Resource *) resource,
				     &ev);
	
	CORBA_exception_free (&ev);
}

static void 
rc_group_gui_added_cb (GtkObject *obj, gpointer user_data)
{
	ResourceComponent     *resource_component;
	ResourceComponentPriv *priv;
	CORBA_Environment      ev;
	
 	g_return_if_fail (user_data != NULL);
 	g_return_if_fail (IS_RESOURCE_COMPONENT (user_data));   
	
	d(puts (__FUNCTION__));

	resource_component = RESOURCE_COMPONENT (user_data);
	priv               = resource_component->priv;

	CORBA_exception_init (&ev);

	resource_mc_insert_group (priv->rm_client, NULL, &ev);

	CORBA_exception_free (&ev);
}

static void 
rc_group_gui_deleted_cb (GtkObject *obj, gint gid, gpointer user_data)
{
	ResourceComponent     *resource_component;
	ResourceComponentPriv *priv;
	CORBA_Environment      ev;
	
 	g_return_if_fail (user_data != NULL);
 	g_return_if_fail (IS_RESOURCE_COMPONENT (user_data));   

	d(puts (__FUNCTION__));

	resource_component = RESOURCE_COMPONENT (user_data);
	priv               = resource_component->priv;
	
	CORBA_exception_init (&ev);

	resource_mc_remove_group (priv->rm_client, gid, &ev);

	CORBA_exception_free (&ev);
}

static void
rc_group_gui_updated_cb (GtkObject        *obj, 
			 GM_ResourceGroup *group, 
			 gpointer          user_data)
{
	ResourceComponent     *resource_component;
	ResourceComponentPriv *priv;
	CORBA_Environment      ev;
	
 	g_return_if_fail (user_data != NULL);
 	g_return_if_fail (IS_RESOURCE_COMPONENT (user_data));   
 	g_return_if_fail (obj != NULL);                         
 	g_return_if_fail (IS_GROUP_DIALOG (obj));               

	d(puts (__FUNCTION__));

	resource_component = RESOURCE_COMPONENT (user_data);
	priv               = resource_component->priv;

	CORBA_exception_init (&ev);

	resource_mc_update_group (priv->rm_client, 
				  (GM_ResourceGroup *) group,
				  &ev);
	
	CORBA_exception_free (&ev);
}

static void
rc_group_gui_default_set_cb (GtkObject *obj,
			     GM_Id      group_id,
			     gpointer   user_data)
{
	ResourceComponent     *resource_component;
	ResourceComponentPriv *priv;
	CORBA_Environment      ev;
	
 	g_return_if_fail (user_data != NULL);
 	g_return_if_fail (IS_RESOURCE_COMPONENT (user_data));
 	g_return_if_fail (obj != NULL);   
 	g_return_if_fail (IS_GROUP_DIALOG (obj));

	d(puts (__FUNCTION__));

	resource_component = RESOURCE_COMPONENT (user_data);
	priv               = resource_component->priv;

	CORBA_exception_init (&ev);

	resource_mc_set_default_group (priv->rm_client, 
				       group_id,
				       &ev);
	
	CORBA_exception_free (&ev);
}



static BonoboUIVerb verbs [] = {
	BONOBO_UI_VERB ("ResourceRemove",   rc_cmd_remove),
	BONOBO_UI_VERB ("ResourceInsert",   rc_cmd_insert),
	BONOBO_UI_VERB ("ResourcePrint",    rc_cmd_print),
        BONOBO_UI_VERB ("ResourceGroup",    rc_cmd_group),
	BONOBO_UI_VERB_END
};

static gboolean
rc_add_to_shell (ShellComponent *component, GM_Shell shell)
{
	ResourceComponentPriv *priv;
	CORBA_Environment      ev;
	
	g_return_val_if_fail (component != NULL, FALSE);
	g_return_val_if_fail (IS_RESOURCE_COMPONENT (component), FALSE);
	
	d(puts (__FUNCTION__));

	CORBA_exception_init (&ev);
	
	priv = (RESOURCE_COMPONENT(component))->priv;

	/* Create a control. */
	priv->control = rc_control_new (RESOURCE_COMPONENT (component));
	
	GNOME_MrProject_Shell_addControl (shell,
					  BONOBO_OBJREF (component),
					  BONOBO_OBJREF (priv->control),
					  OAFIID,
					  _("Resources"),
					  MRPROJECT_IMAGEDIR "resources.png",
					  NULL);
	
	priv->rm_client = RESOURCE_MANAGER_CLIENT (resource_mc_new (shell, 
								    FALSE, 
								    &ev));

	gtk_signal_connect_object (GTK_OBJECT (priv->rm_client),
				   "resource_inserted",
				   GTK_SIGNAL_FUNC (rc_resource_inserted_cb),
				   GTK_OBJECT (component));
	
	gtk_signal_connect_object (GTK_OBJECT (priv->rm_client),
				   "resource_updated",
				   GTK_SIGNAL_FUNC (rc_resource_updated_cb),
				   GTK_OBJECT (component));
	
	gtk_signal_connect_object (GTK_OBJECT (priv->rm_client),
				   "resources_removed",
				   GTK_SIGNAL_FUNC (rc_resources_removed_cb),
				   GTK_OBJECT (component));

	gtk_signal_connect_object (GTK_OBJECT (priv->rm_client),
				   "items_clear",
				   GTK_SIGNAL_FUNC (rc_items_clear_cb),
				   GTK_OBJECT (component));

	gtk_signal_connect_object (GTK_OBJECT (priv->rm_client),
				   "group_inserted",
				   GTK_SIGNAL_FUNC (rc_group_inserted_cb),
				   GTK_OBJECT (component));
	
	gtk_signal_connect_object (GTK_OBJECT (priv->rm_client),
				   "group_updated",
				   GTK_SIGNAL_FUNC (rc_group_updated_cb),
				   GTK_OBJECT (component));
	
	gtk_signal_connect_object (GTK_OBJECT (priv->rm_client),
				   "group_removed",
				   GTK_SIGNAL_FUNC (rc_group_removed_cb),
				   GTK_OBJECT (component));

	gtk_signal_connect_object (GTK_OBJECT (priv->rm_client),
				   "group_default_set",
				   GTK_SIGNAL_FUNC (rc_group_default_set_cb),
				   GTK_OBJECT (component));

	CORBA_exception_free (&ev);

	return TRUE;
}

static void
rc_remove_from_shell (ShellComponent *component)
{
	ResourceComponentPriv *priv;
	
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_RESOURCE_COMPONENT (component));
	
	d(puts (__FUNCTION__));

	priv = RESOURCE_COMPONENT (component)->priv;

	gtk_object_destroy (GTK_OBJECT (priv->rm_client));
	priv->rm_client = NULL;
}

static void
rc_set_debug_flags (ShellComponent *component, glong flags)
{
	d(puts (__FUNCTION__));
}

static void
rc_set_pixmap (BonoboUIComponent *uic, const char *xml_path, const char *icon)
{
	char      *path;
	GdkPixbuf *pixbuf;

	d(puts (__FUNCTION__));

	path   = g_concat_dir_and_file (MRPROJECT_IMAGEDIR, icon);
	pixbuf = gdk_pixbuf_new_from_file (path);

	if (pixbuf == NULL) {
		g_warning ("Cannot load image -- %s", path);
		g_free (path);
		return;
	}

	bonobo_ui_util_set_pixbuf (uic, xml_path, pixbuf);

	gdk_pixbuf_unref (pixbuf);

	g_free (path);
}

static void
rc_control_activate (BonoboControl *control, ResourceComponent *rc)
{
	Bonobo_UIContainer  remote_uih;
	BonoboUIComponent  *uic;

	d(puts (__FUNCTION__));

	uic = bonobo_control_get_ui_component (control);
	g_assert (uic != NULL);

	remote_uih = bonobo_control_get_remote_ui_container (control);
	bonobo_ui_component_set_container (uic, remote_uih);
	bonobo_object_release_unref (remote_uih, NULL);

	bonobo_ui_component_add_verb_list_with_data (uic, verbs, rc);

	bonobo_ui_component_freeze (uic, NULL);

	bonobo_ui_util_set_ui (uic, GNOME_DATADIR,
			       "GNOME_MrProject_ResourceComponent.ui",
			       "mrproject-resource-component");

	rc_set_pixmap (uic, "/Toolbar/ToolInsertResource", 
		       "24_add_resource.png");
	rc_set_pixmap (uic, "/Toolbar/ToolRemoveResource", 
		       "24_delete_resource.png");
	rc_set_pixmap (uic, "/Toolbar/ToolGroupResource", "24_groups.png");

	rc_set_sensitivity (rc);
	
	bonobo_ui_component_thaw (uic, NULL);
}

static void
rc_control_deactivate (BonoboControl *control)
{
	BonoboUIComponent *uic = bonobo_control_get_ui_component (control);
	g_assert (uic != NULL);
	
	d(puts (__FUNCTION__));

	bonobo_ui_component_rm (uic, "/", NULL);
 	bonobo_ui_component_unset_container (uic);
}

static void
rc_control_activate_cb (BonoboControl *control, 
			gboolean       activate, 
			gpointer       user_data)
{
	d(puts (__FUNCTION__));

	if (activate) {
		rc_control_activate (control, user_data);
	} else {
		rc_control_deactivate (control);
	}
}

static BonoboControl *
rc_control_new (ResourceComponent *component)
{
	ResourceComponentPriv   *priv;
	
	d(puts (__FUNCTION__));

	priv = component->priv;

	if (!rc_create_table (component)) {
		g_warning ("Could not create the resource table");
		return NULL;
	}
	
	
	priv->control = bonobo_control_new (priv->table);

	gtk_widget_show (bonobo_control_get_widget (priv->control));

	gtk_signal_connect (GTK_OBJECT (priv->control), "activate",
			    GTK_SIGNAL_FUNC (rc_control_activate_cb),
			    component);
	
 	priv->resource_changed_id = 
		gtk_signal_connect (GTK_OBJECT (priv->rtm), 
				    "resource_changed",
				    GTK_SIGNAL_FUNC (rc_resource_gui_updated_cb), 
				    component); 

	bonobo_control_set_automerge (priv->control, FALSE);

	return priv->control;
}

static void
rc_destroy (GtkObject *object)
{
	ResourceComponent     *component;
	ResourceComponentPriv *priv;

	d(puts (__FUNCTION__));

	component = RESOURCE_COMPONENT (object);
	priv      = component->priv;

	gtk_object_destroy (GTK_OBJECT (priv->rm_client));

	g_free (priv);

	(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
resource_component_class_init (ResourceComponentClass *klass)
{
	GtkObjectClass      *object_class;
	ShellComponentClass *shell_component_class;

	d(puts (__FUNCTION__));

	object_class                             = (GtkObjectClass *) klass;
	shell_component_class                    = SHELL_COMPONENT_CLASS (klass);
	parent_class                             = gtk_type_class (PARENT_TYPE);

	object_class->destroy                    = rc_destroy;

	shell_component_class->add_to_shell      = rc_add_to_shell;
	shell_component_class->remove_from_shell = rc_remove_from_shell;
	shell_component_class->set_debug_flags   = rc_set_debug_flags;
}

static void
resource_component_init (ResourceComponent *component)
{
	ResourceComponentPriv *priv;
	
	d(puts (__FUNCTION__));

	priv = g_new0 (ResourceComponentPriv, 1);
	priv->rm_client = NULL;
	
	component->priv = priv;
}

static void
rc_set_sensitivity (ResourceComponent *rc)
{
	ResourceComponentPriv *priv;
	BonoboUIComponent     *uic;
	gint                   count;
	gchar                 *one, *at_least_one;
	
	priv  = rc->priv;
	count = e_table_selected_count (E_TABLE_SCROLLED (priv->table)->table);
	
	one          = (count == 1) ? "1" : "0";
	at_least_one = (count > 0)  ? "1" : "0";

	uic = bonobo_control_get_ui_component (priv->control);

	if (bonobo_ui_component_get_container (uic) != CORBA_OBJECT_NIL) {
		/* Note: add the freeze/thaw when we get more than one set_prop here. */
		/*bonobo_ui_component_freeze (uic, NULL);*/
		
		bonobo_ui_component_set_prop (uic, "/commands/ResourceRemove",
					      "sensitive", at_least_one, NULL);
		
		/*bonobo_ui_component_thaw (uic, NULL);*/
	}
}

#ifdef HAVE_EVOLUTION_COMPOSER_IDL
typedef enum {
	RC_SEND_TYPE_RESOURCE,
	RC_SEND_TYPE_GROUP
} RCSendType;

typedef struct {
	ResourceComponent *rc;
	gint               row;
	RCSendType         type;
} SendToData;

static void
rc_composer_component_loaded (CORBA_Object  composer,
			      const gchar  *error_reason,
			      SendToData   *sdata)
{
	ResourceComponentPriv                  *priv;
	GNOME_Evolution_Composer_RecipientList *list;
 	GNOME_Evolution_Composer_RecipientList *empty_list;
	GM_Resource                            *resource;
	GSList                                 *r_list;
	CORBA_Environment                       ev;
	gchar                                  *subject;
	GSList                                 *node;
	gint                                    len1, len2;
	
	g_return_if_fail (composer != NULL);
	g_return_if_fail (sdata != NULL);
	g_return_if_fail (sdata->rc != NULL);
	g_return_if_fail (IS_RESOURCE_COMPONENT (sdata->rc));
	
	priv = sdata->rc->priv;

	if (composer == CORBA_OBJECT_NIL) {
		g_warning ("Unable to start mail composer component!");
		return;
	}

	resource = resource_tm_get_resource_on_row (priv->rtm, sdata->row);
	
	empty_list = GNOME_Evolution_Composer_RecipientList__alloc ();
	empty_list->_length = 0;
	empty_list->_maximum = 0;
	empty_list->_buffer = CORBA_sequence_GNOME_Evolution_Composer_Recipient_allocbuf(0);

	list = GNOME_Evolution_Composer_RecipientList__alloc ();

	switch (sdata->type) {
	case RC_SEND_TYPE_RESOURCE:
		list->_length  = 1;
		list->_maximum = 1;
		list->_buffer  = CORBA_sequence_GNOME_Evolution_Composer_Recipient_allocbuf (1);
		list->_buffer[0].name = CORBA_string_dup (resource->name);
		list->_buffer[0].address = CORBA_string_dup (resource->email);
		break;
	case RC_SEND_TYPE_GROUP:
		r_list = resource_tm_get_resources_with_group (priv->rtm,
							       resource->groupId);
		len1 = g_slist_length (r_list);

		/* Remove all resources without an email address */
		node = r_list;
		do {
			resource = (GM_Resource *) node->data;
			
			if (!resource->email || 
			    !strcmp (resource->email, "")) {
				node = node->next;
				r_list = g_slist_remove (r_list, resource);
			} else {
				node = node->next;
			}
		} while (node);

		if (!r_list) {
			mr_message_box_show_message (
				GNOME_MESSAGE_BOX_WARNING,
				_("One or more resources of choosen group doesn't have email addresses."));
			g_free (sdata);
			return;
		}
		
		/* Setup the to-list */
		len2 = g_slist_length (r_list);

		if (len2 < len1) { 
			/* We had to remove resources without email */

			mr_message_box_show_message (
				GNOME_MESSAGE_BOX_WARNING,
				_("Not all of the resources of choosen group have an email address."));
		}

		list->_length  = len2;
		list->_maximum = len2;
		list->_buffer  = CORBA_sequence_GNOME_Evolution_Composer_Recipient_allocbuf (len2);
			
		for (node = r_list; node; node = node->next) {
			resource = (GM_Resource *) node->data;
			list->_buffer[--len2].name  = CORBA_string_dup (resource->name);
			list->_buffer[len2].address = CORBA_string_dup (resource->email);
		}

		g_slist_free (r_list);
		break;
	default:
		break;
	};
	
	CORBA_exception_init (&ev);

	subject = g_strdup ("");
	
	GNOME_Evolution_Composer_setHeaders (composer, 
					     list,               /* To  */
					     empty_list,         /* Cc  */
					     empty_list,         /* Bcc */
					     subject,
					     &ev);

	if (!BONOBO_EX (&ev)) {
		GNOME_Evolution_Composer_show (composer, &ev);
	}
	
	g_free (sdata);
	g_free (subject);
	CORBA_exception_free (&ev);
}


static void
rc_send_to_activate_cb (GtkMenuItem *menu_item, SendToData *sdata)
{
	CORBA_Environment ev;
	
	g_return_if_fail (sdata != NULL);
	g_return_if_fail (sdata->rc != NULL);
	g_return_if_fail (IS_RESOURCE_COMPONENT (sdata->rc));

	CORBA_exception_init (&ev);
	
	oaf_activate_from_id_async (COMPOSER_IID,
  				    0,
  				    (OAFActivationCallback) rc_composer_component_loaded,
 				    sdata,
 				    &ev);
	
	if (BONOBO_EX (&ev)) {
		mr_message_box_show_message (GNOME_MESSAGE_BOX_ERROR,
					     "Unable to start mail composer component!"); 
		g_free (sdata);
	}

	CORBA_exception_free (&ev);
}

static GtkWidget *
rc_setup_right_click_menu (ResourceComponent *rc, gint row)
{
	ResourceComponentPriv *priv;
	GtkWidget             *menu;
	GtkWidget             *menu_item;
	GM_Resource           *resource;
	SendToData            *sdata;
	
	priv = rc->priv;

	resource = resource_tm_get_resource_on_row (priv->rtm, row);
	
	if (!resource) {
		g_warning ("No such resource");
		return NULL;
	}

	menu = gtk_menu_new ();

	sdata = g_new0 (SendToData, 1);
	sdata->row  = row;
	sdata->rc   = rc;
	sdata->type = RC_SEND_TYPE_RESOURCE;

	menu_item = gtk_menu_item_new_with_label (_("Mail to..."));
	gtk_signal_connect (GTK_OBJECT (menu_item), 
			    "activate",
			    rc_send_to_activate_cb,
			    sdata);

	if (!resource->email || !strcmp (resource->email, "")) {
		gtk_widget_set_sensitive (menu_item, FALSE);
	}

	gtk_widget_show (menu_item);
	gtk_menu_append (GTK_MENU (menu), menu_item);

	sdata = g_new0 (SendToData, 1);
	sdata->row  = row;
	sdata->rc   = rc;
	sdata->type = RC_SEND_TYPE_GROUP;

	menu_item = gtk_menu_item_new_with_label (_("Mail to group..."));
	gtk_signal_connect (GTK_OBJECT (menu_item), 
			    "activate",
			    rc_send_to_activate_cb,
			    sdata);

	if (resource->groupId == -1) {
		gtk_widget_set_sensitive (menu_item, FALSE);
	}
	
	gtk_widget_show (menu_item);
	gtk_menu_append (GTK_MENU (menu), menu_item);

	return menu;
}

static void
rc_right_click_cb (ETable            *table,
		   int                row,
		   int                col,
		   GdkEvent          *event,
		   ResourceComponent *rc)
{
	GtkWidget      *menu;
	GdkEventButton *button_event;
	
	menu = rc_setup_right_click_menu (rc, row);
	button_event = (GdkEventButton *) event;

	gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL, 
			button_event->button, button_event->time);
		
}
#endif /* HAVE_EVOLUTION_COMPOSER_IDL */

static void
rc_set_sensitivity_and_status_message (ResourceComponent *component)
{
	ResourceComponentPriv *priv;
	BonoboUIComponent     *ui_component;
	gchar                 *str;
	gint                   count;
	ETable                *etable;
	gint                   row;
	GM_Resource           *resource;

	priv = component->priv;
	
	rc_set_sensitivity (component);

	ui_component = bonobo_control_get_ui_component (priv->control);

	/* Don't bother if we're not activated. */
	if (!bonobo_ui_component_get_container (ui_component)) {
		return;
	}

	/* FIXME: This does not seem to work. My guess is that the reason
	 * is that we do this in the selection changed callback. If so,
	 * could we fix GAL or is there a workaround? It works for ETree
	 * (as used in the gantt component).
	 * The problem is that when a row is first clicked, we get a
	 * garbage value for row number, the second time it's clicked,
	 * we get the real row number.
	 */
	return;

	count = e_table_selected_count (E_TABLE_SCROLLED (priv->table)->table);

	if (count > 1) {
		str = g_strdup_printf (_("%d resources selected"), count);
	}
	else if (count == 0) {
		str = g_strdup (" "); /* "" does not seem to work. */
	}
	else { /* We have exactly one. */
		etable = e_table_scrolled_get_table (
			E_TABLE_SCROLLED (priv->table));

		row    = e_table_get_cursor_row (etable);

		if (row < 0) {
			return;
		}
	
		resource = resource_tm_get_resource_on_row (priv->rtm, row);
	
		if (!resource) {
			return;
		}
		
		str = g_strdup_printf (_("Selected resource: %s"), 
				       resource->name);
	}

	bonobo_ui_component_set_status (ui_component, str, NULL);
	g_free (str);
}

static void
rc_table_selection_change (ETable              *table,
			   ResourceComponent   *rc)
{
	rc_set_sensitivity_and_status_message (rc);	
}

static BonoboObject *
resource_component_factory (BonoboGenericFactory *this,
			    const char           *object_id,
			    void                 *data)
{
	g_return_val_if_fail (object_id != NULL, NULL);

	if (!strcmp (object_id, OAFIID)) {
		return BONOBO_OBJECT (gtk_type_new (RESOURCE_COMPONENT_TYPE));
	} else {
		g_warning ("Failing to manufacture a '%s'", object_id);
	}
	
	return NULL;
}

static gboolean
rc_create_table (ResourceComponent *component)
{
	ResourceComponentPriv *priv;
	ETableExtras          *extras;
	GList                 *strings = NULL;
	ECell                 *combo_cell;
	ECell                 *text_cell;
	ECell                 *cell;
	
	g_return_val_if_fail (component != NULL, FALSE);
	g_return_val_if_fail (IS_RESOURCE_COMPONENT (component), FALSE);
	
	priv      = component->priv;
	extras    = e_table_extras_new ();
	priv->rtm = RESOURCE_TABLE_MODEL(resource_tm_new ());

	/* Resource Type cell */
	text_cell = e_cell_text_new (NULL, GTK_JUSTIFY_LEFT);

	gtk_object_set (GTK_OBJECT (text_cell),
			"editable", FALSE,
			NULL);

	strings    = resource_tm_get_type_strings (priv->rtm);
	combo_cell = e_cell_combo_new ();
	e_cell_popup_set_child (E_CELL_POPUP (combo_cell), text_cell);
	gtk_object_unref (GTK_OBJECT (text_cell));
	
	e_cell_combo_set_popdown_strings (E_CELL_COMBO(combo_cell), 
					  strings);
 	e_table_extras_add_cell (extras, "resource_type", combo_cell);

	/* Resource Group cell */
	text_cell = e_cell_text_new (NULL, GTK_JUSTIFY_LEFT);

	gtk_object_set (GTK_OBJECT (text_cell),
			"editable", FALSE,
			NULL);

	strings          = resource_tm_get_group_strings (priv->rtm);
	priv->group_cell = e_cell_combo_new ();

	e_cell_popup_set_child (E_CELL_POPUP (priv->group_cell), text_cell);
	gtk_object_unref (GTK_OBJECT (text_cell));
	
	e_cell_combo_set_popdown_strings (E_CELL_COMBO(priv->group_cell), 
					  strings);

 	e_table_extras_add_cell (extras, "resource_group", priv->group_cell);

	/* Unit cell */
	cell = e_cell_spin_button_new (50, 1000, 50, NULL);
 	e_table_extras_add_cell (extras, "resource_unit", cell);

	/* Rate cell */
	cell = e_cell_spin_button_new_float (50, 2000, 50, NULL);
 	e_table_extras_add_cell (extras, "resource_rate", cell);

	/* Float compare */
	e_table_extras_add_compare (extras, "float_compare", rc_float_compare);
	
	priv->table = e_table_scrolled_new_from_spec_file (
		E_TABLE_MODEL (priv->rtm),
		extras,
		MRPROJECT_DATADIR "resource-component.etspec",
		MRPROJECT_DATADIR "resource-component.etstate");

#ifdef HAVE_EVOLUTION_COMPOSER_IDL
	if (corba_util_evolution_composer_exist ()) {
		gtk_signal_connect (GTK_OBJECT (E_TABLE_SCROLLED (priv->table)->table),
				    "right_click",
				    rc_right_click_cb,
				    component);
	}
#endif /* HAVE_EVOLUTION_COMPOSER_IDL */		    

	gtk_object_unref (GTK_OBJECT (extras));

	e_scroll_frame_set_shadow_type (E_SCROLL_FRAME (priv->table),
                                        GTK_SHADOW_NONE);

	gtk_signal_connect (GTK_OBJECT (E_TABLE_SCROLLED (priv->table)->table),
			    "selection_change",
			    rc_table_selection_change,
			    component);
	return TRUE;
}

static gint
rc_float_compare (gconstpointer a, gconstpointer b)
{
	float f1 = *(float *) a;
	float f2 = *(float *) b;
	
	if (f1 < f2) {
		return -1;
	}
	else if (f1 > f2) {
		return 1;
	}

	return 0;
}

BONOBO_OAF_SHLIB_FACTORY_MULTI ("OAFIID:GNOME_MrProject_ResourceComponentFactory",
				"Mr Project resource component factory",
				resource_component_factory,
				NULL);

BONOBO_X_TYPE_FUNC (ResourceComponent, 
		    PARENT_TYPE,
		    resource_component);
