/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Mouse Gestures (Mappings Service).
 *
 * The Initial Developer of the Original Code is Jochen Schlehuber.
 * Portions created by the Initial Developer are Copyright (C) 2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *  Jochen <bugs@krickelkrackel.de>
 *
 * ***** END LICENSE BLOCK ***** */

const mozGestContractID      = "@mousegestures.org/mgMappingsService;1";
const mozGestCID             = Components.ID("{fea06ea8-4621-469c-a85c-eedc1f654a4f}");
const mozGestMappingsService = Components.interfaces.mgIMappingsService;

const mozGestObs             = Components.classes["@mozilla.org/observer-service;1"]
                               .getService(Components.interfaces.nsIObserverService);

const mozGestDirService      = Components.classes['@mozilla.org/file/directory_service;1']
                               .getService(Components.interfaces.nsIProperties);

const mozGestPrefService     = Components.classes["@mozilla.org/preferences-service;1"]
                               .getService(Components.interfaces.nsIPrefService);

const mozGestPrefBranch      = mozGestPrefService.getBranch("mozgest.");


var mgMappingsService = {
  QueryInterface : function(iid) {
    if (!iid.equals(mozGestMappingsService) &&
        !iid.equals(Components.interfaces.nsISupports))
      throw Components.results.NS_ERROR_NO_INTERFACE;

    return this;
  },

  init                   : false,
  activeMappings         : new Array(),
  //actTemp will act as temporary activeMappings while customizing or importing
  actTemp                : new Array(),
  defaultMappings        : new Array(),
  functions              : new Array(),
  supportedWindows       : new Array(),
  jsLoader               : Components.classes["@mozilla.org/moz/jssubscript-loader;1"]
                           .getService(Components.interfaces.mozIJSSubScriptLoader),
  history                : new Array(),
  //import...
  curBookmark            : null,
  curBookmarkName        : null,
  curLocation            : null,
  removedWinTypes        : new Array(),
  bundle                 : Components.classes["@mozilla.org/intl/stringbundle;1"].getService()
                           .QueryInterface(Components.interfaces.nsIStringBundleService)
                           .createBundle("chrome://mozgest/locale/mozgest.properties"),
  prompt                 : Components.classes["@mozilla.org/embedcomp/prompt-service;1"]
                           .getService(Components.interfaces.nsIPromptService),

  initMappings : function() {
    if (this.init)
      return;

    this.init = true;

    //register shutdown and uninstall observer
    mozGestObs.addObserver(mgMappingsService.shutDownObserver, "quit-application", false);
    mozGestObs.addObserver(mgMappingsService.uninstallObserver, "em-action-requested", false);

    if (!("@mozilla.org/extensions/manager;1" in Components.classes)) {
      try {
        Components.utils.import("resource://gre/modules/AddonManager.jsm");
        AddonManager.addAddonListener(mgMappingsService.addonObserver);
      }
      catch (err) {}
    }

    this.removedWinTypes["browser_printpreview"] = true;
    this.removedWinTypes["chatzilla"] = true;
    this.removedWinTypes["window"] = true;

    this.jsLoader.loadSubScript('chrome://mozgest/content/defaults.js');
    this.jsLoader.loadSubScript('chrome://mozgest/content/gestimp.js');

    //comes from defaults.js
    mgDefaultSettings.defaultMappings.init();
    this.defaultMappings = mgDefaultSettings.defaultMappings.getMappings();
    //load window list
    this.getWindows();
    //load function list
    this.getFunctions();
    //load active mappings
    this.getActiveMappings(false, true);
  },

  shutDownObserver : {
    observe: function(subject, topic, data) {
      var removeAction = 99;

      try {
        removeAction = mozGestPrefBranch.getIntPref("misc.uninstall.action");
      }
      catch (err) {}

      mozGestPrefBranch.deleteBranch("misc.uninstall.");

      if (removeAction != 99) {
        if (removeAction == 0 || removeAction == 2)
          mozGestPrefBranch.deleteBranch("");

        if (removeAction == 0 || removeAction == 1) {
          var mozgestDir = mgMappingsService.getMappingsFile();
          mozgestDir = mozgestDir.parent;

          if (mozgestDir.exists()) {
            try {
              mozgestDir.remove(true);
            }
            catch (err) {}
          }
        }
      }
      else
        mgMappingsService.writeMappingsToFile(false);

      mozGestObs.removeObserver(mgMappingsService.shutDownObserver, "quit-application");
      mozGestObs.removeObserver(mgMappingsService.uninstallObserver, "em-action-requested");
    }
  },

  uninstallObserver : {
    observe: function(subject, topic, data) {
      if (subject) {
        subject.QueryInterface(Components.interfaces.nsIUpdateItem);

        if (subject.id == "{FFA36170-80B1-4535-B0E3-A4569E497DD0}") {
          if (data == "item-uninstalled") {
            var gs = mgMappingsService.bundle.GetStringFromName;

            if (mgMappingsService.prompt.confirm(null, subject.name, gs("uninstDesc")))
              mozGestPrefBranch.setIntPref("misc.uninstall.action", 0);
          }
          else if (data == "item-cancel-action")
            mozGestPrefBranch.deleteBranch("misc.uninstall.");
        }
      }
    }
  },

  addonObserver : {
    onUninstalling : function(ext, b, c, d) {
      if (ext.id == "{FFA36170-80B1-4535-B0E3-A4569E497DD0}") {
        var gs = mgMappingsService.bundle.GetStringFromName;

        if (mgMappingsService.prompt.confirm(null, ext.name, gs("uninstDesc")))
          mozGestPrefBranch.setIntPref("misc.uninstall.action", 0);
      }
    },

    onOperationCancelled : function(ext, b, c, d) {
      if (ext.id == "{FFA36170-80B1-4535-B0E3-A4569E497DD0}") {
        if (!(ext.pendingOperations & AddonManager.PENDING_UNINSTALL))
          mozGestPrefBranch.deleteBranch("misc.uninstall.");
      }
    }
  },

  getWindows : function() {
    this.jsLoader.loadSubScript('chrome://mozgest/content/windowTypes.js');

    for (winType in mgWindowTypes.wTypes)
      this.supportedWindows[mgWindowTypes.wTypes[winType][0]] = true;

    this.supportedWindows["unknownWindow"] = true;

    for (each in this.supportedWindows)
      this.history[each] = new Array();
  },

  getFunctions : function() {
    //comes from defaults.js
    mgDefaultSettings.defaultFunctions.init();
    var funcTemp = new Array();
    funcTemp = mgDefaultSettings.defaultFunctions.getFunctions();
    this.defFuncs = mgDefaultSettings.defaultFunctions.getFunctions();

    for (wType in funcTemp) {           // read all "native" functions for this windowtype
      this.functions[wType] = new Array();

      var silent = funcTemp[wType];
      for (func in silent)
        this.functions[wType][func] = func + "()";
    }

    for (wType in funcTemp) {           // read all "foreign" functions for this windowtype
      var silent = funcTemp[wType];

      for (func in silent) {
        if (funcTemp[wType][func].length > 1) {
          for (var x = 1; x < funcTemp[wType][func].length; x++) {
            var wt = false;

            if (funcTemp[wType][func][x] == "B")
              wt = "browser";
            else if (funcTemp[wType][func][x] == "VS")
              wt = "viewsource";
            else if (funcTemp[wType][func][x] == "M")
              wt = "messenger";
            else if (funcTemp[wType][func][x] == "MC")
              wt = "mailcompose";
            else if (funcTemp[wType][func][x] == "U")
              wt = "unknownWindow";

            if (wt)
              this.functions[wt][func] = func + "()";
          }
        }
      }
    }
  },

  getImportMappings : function(importFile) {
    var ioService  = Components.classes["@mozilla.org/network/io-service;1"]
                     .getService(Components.interfaces.nsIIOService);

    try {
      var mgFileSpec = ioService.newFileURI(importFile).spec
      this.jsLoader.loadSubScript(mgFileSpec);
      this.getActiveMappings(true);
      this.writeMappingsToFile();
    }
    catch (err) {
      this.getActiveMappings();
      return false;
    }

    return true;
  },

  getMappingsFile : function() {
    var mappingsFile = mozGestDirService.get('ProfD', Components.interfaces.nsILocalFile);
    mappingsFile.append("mozgest3")

    try {
      mappingsFile.create(1, 0755);
    }
    catch (e) {}

    mappingsFile.append("gestures3.js");
    return mappingsFile;
  },

  getActiveMappings : function(isImport, start) {
    this.cleanUp();

    if (!isImport) {
      var mappingsFile = this.getMappingsFile();

      if (start && !mappingsFile.exists()) {
        var oldMappingsFile = mozGestDirService.get('ProfD', Components.interfaces.nsILocalFile);
        oldMappingsFile.append("mousegestures.js");

        if (!oldMappingsFile.exists())
          this.writeMappingsToFile(true);
        else {
          var gs = this.bundle.GetStringFromName;

          if (this.prompt.confirm(null, gs("mozgest"), gs("instOldMappings")))
            mappingsFile = oldMappingsFile;
          else
            this.writeMappingsToFile(true);
        }
      }

      var ioService  = Components.classes["@mozilla.org/network/io-service;1"]
                       .getService(Components.interfaces.nsIIOService);
      var mgFileSpec = ioService.newFileURI(mappingsFile).spec

      //mgActiveMappings comes from the just loaded subscript
      //if there is a problem restore defaults
      try {
        this.jsLoader.loadSubScript(mgFileSpec);
        mgActiveMappings.init();
      }
      catch (e) {
        this.writeMappingsToFile(true);  //restore default
        var gs = this.bundle.GetStringFromName;
        this.prompt.alert(null, gs("mozgest"), gs("fileCorrupt"));
        this.getActiveMappings();
        return;
      }
    }

    // we have a 'valid' file. Maybe.
    mgActiveMappings.init();
    this.activeMappings = new Array();
    this.actTemp = mgActiveMappings.getMappings();

    //test for added "window types"
    //"unknownWindow" was added with 3.0
    if (!("unknownWindow" in this.actTemp))
      this.actTemp["unknownWindow"] = this.defaultMappings["unknownWindow"];

    this.fillMappings(this.actTemp, this.activeMappings);
    this.actTemp = new Array();

    if (!start)
      mozGestObs.notifyObservers(null, "mozgestControl", "mappingsUpdated");
  },

  fillMappings : function(array1, array2) {
    for (winType in array1) {
      // check for removed window types
      if (winType in this.removedWinTypes)
        continue;

      array2[winType] = new Array();

      for (mapping in array1[winType]) {
        if (!array1[winType][mapping].type) {
          var func = array1[winType][mapping].func;
          // "window" has been removed in MGR 3
          if (winType == "window" || !(func in this.functions[winType]))
            continue;

          array1[winType][mapping].type = "0";
        }

        if (array1[winType][mapping].type == "0")
          array1[winType][mapping].name = encodeURIComponent(this.bundle.GetStringFromName(array1[winType][mapping].func));

        if (array1[winType][mapping].type == "1") {
        // Since MG3.0 lots of functions are no longer global
        // ugly hack to not break compatibility with existing scripts
          var customFunc = array1[winType][mapping].func;
          array1[winType][mapping].customFunc = customFunc;

          try {
            var newString = "let mgAppInfo = mgCommon.appInfo;" +
                            "let mgShowStatus = mgGestureHelper.showStatus;";

            //mgBuiltInFunctions comes from gestimp.js
            for (each in mgBuiltInFunctions) {
              var result = customFunc.search(each);

              if (result != -1)
                newString = newString + "let " + each + " = mgBuiltInFunctions." + each + ";";
            }
            array1[winType][mapping].customFunc = newString + customFunc;
          }
          catch (err) {
            array1[winType][mapping].customFunc = array1[winType][mapping].func;
          }
        }

        // brute force in case of gestures3.js without categories
        if (array1[winType][mapping].cat == null) {
          try {
            var funcTemp = new Array();
            funcTemp = mgDefaultSettings.defaultFunctions.getFunctions();
            var cat = 0;

            for (wT in this.functions) {
              var silent = funcTemp[wT];
              var xFunc = array1[winType][mapping].func;

              if (xFunc in silent)
                cat = silent[xFunc][silent[xFunc].length -1].substring(3);
            }
            array1[winType][mapping].cat = cat;
          }
          catch (e) {
            array1[winType][mapping].cat = 0;
          }
        }

        array2[winType][mapping] = array1[winType][mapping];
      }
    }
  },

  addMappingToTemp : function(wType, code, type, name, aFunc, count, cat) {
    var mapping = this.actTemp[wType][code] = {func : aFunc};
    mapping.type = type;
    mapping.name = name;
    mapping.count = count;
    mapping.cat = cat;
    mozGestObs.notifyObservers(null, "mozgestControl", "tempMappingsUpdated");
  },

  updateMappingsFromTemp : function() {
    this.activeMappings = this.actTemp;
    this.writeMappingsToFile(false);
    this.getActiveMappings();
  },

  cleanUp : function() {
    this.actTemp = new Array();
    var mgActiveMappings = null;
  },

  writeMappingsToFile : function(newFile) {
    var gestureSet;

    if (newFile)
      gestureSet = this.defaultMappings;
    else
      gestureSet = this.activeMappings;

    try {
      var txt =
      "var mgActiveMappings =\n"+
      "{\n"+
      "  mappings :\n"+
      "  {\n"+
      "    browser       : new Array(),\n"+
      "    viewsource    : new Array(),\n"+
      "    messenger     : new Array(),\n"+
      "    mailcompose   : new Array(),\n"+
      "    unknownWindow : new Array()\n"+
      "  },\n\n"+
      "  /* for description: chrome://mozgest/content/defaults.js */\n\n"+
      "  init : function()\n"+
      "  {\n";

      for (wType in gestureSet) {
        var wt;

        if (wType == "browser")
          wt = "b";
        else if (wType == "viewsource")
          wt = "vs";
        else if (wType == "messenger")
          wt = "m";
        else if (wType == "mailcompose")
          wt = "mc";
        else if (wType == "unknownWindow")
          wt = "u";

        txt = txt + "    var " + wt + " = this.mappings." + wType +";\n";
        var silent = gestureSet[wType];

        for (mapping in silent) {
          txt = txt + "    " + wt;
          txt = txt + '["' + mapping +'"] = {';
          txt = txt + 'func: "' + silent[mapping].func + '"';

          if (silent[mapping].type)
            txt = txt + ', type: "' + silent[mapping].type + '"';

          if (silent[mapping].type != "0" && silent[mapping].name)
            txt = txt + ', name: "' + silent[mapping].name + '"';

          if (silent[mapping].count)
            txt = txt + ', count: ' + silent[mapping].count;
          else
            txt = txt + ', count: 0';

          if (silent[mapping].cat)
            txt = txt + ', cat: ' + silent[mapping].cat;
          else
            txt = txt + ', cat: 0';

          txt = txt + "}\n";

        }
        txt = txt + "\n";
      }
      txt = txt + "  },\n\n";
      txt = txt + "  getMappings : function()\n" +
                    "  {\n" +
                    "    return this.mappings\n" +
                    "  }\n}";

      var mappingsFile = this.getMappingsFile();
      var stream = Components.classes["@mozilla.org/network/file-output-stream;1"]
                             .createInstance(Components.interfaces.nsIFileOutputStream);

      stream.init(mappingsFile, 0x02 | 0x08 | 0x20, 0644, 0);
      stream.write(txt, txt.length);
      stream.close();
    }
    catch (e) {dump("MozGest: Error writing mappings.\n")}
  }
}

mgMappingsService.wrappedJSObject = mgMappingsService;

var mozgestModule = {
  QueryInterface : function(iid) {
    if (!iid.equals(Components.interfaces.nsIModule) &&
        !iid.equals(Components.interfaces.nsIFactory) &&
        !iid.equals(Components.interfaces.nsISupports))
      throw Components.results.NS_ERROR_NO_INTERFACE;

    return this;
  },

  getClassObject : function(compMgr, cid, iid) {
    if (!cid.equals(mozGestCID))
      throw Components.results.NS_ERROR_NO_INTERFACE;

    return this.QueryInterface(iid);
  },

  registerSelf : function(compMgr, fileSpec, location, type) {
    var compReg = compMgr.QueryInterface(Components.interfaces.nsIComponentRegistrar)
    compReg.registerFactoryLocation(mozGestCID, "MozGest Mappings Service",
                                    mozGestContractID, fileSpec, location, type);
  },

  unregisterSelf : function(compMgr, location, type) {
    var compReg = compMgr.QueryInterface(Components.interfaces.nsIComponentRegistrar);
    compReg.unregisterFactoryLocation(mozGestCID, fileSpec);
  },

  canUnload : function(a) {return true},

  createInstance : function(outer, iid) {
    if (outer != null)
      throw Components.results.NS_ERROR_NO_AGGREGATION;

    return mgMappingsService.QueryInterface(iid);
  },

  lockFactory : function(a) {}
}

function NSGetModule(compMgr, fileSpec) {
  return mozgestModule;
}