/*	track.c
 *
 *	Experimental motion tracking.
 *
 *	Copyright 2000, Jeroen Vreeken
 *	This program is published under the GNU Public license
 */

#include <termios.h>
#include "motion.h"
#include "video.h"
#include "conf.h"
#include "track.h"
#include "alg.h"

extern struct config conf;

struct trackoptions track={
	0,	/* dev open */
	NULL,	/* char *port */
	-1,	/* iomojo id */
	0,	/* int motorx */
	0,	/* int maxx; */
};


/* Add your own center and move functions here: */
static int stepper_center(void);
static int stepper_move(int dev, int pipe, int mpipe, struct coord cent, struct images *imgs);
static int iomojo_center(void);
static int iomojo_move(int dev, int pipe, int mpipe, struct coord cent, struct images *imgs);

/* Add a call to your functions here: */
int track_center(void)
{
	if (track.iomojo_id==-1)
		return stepper_center();
	else
		return iomojo_center();
}

/* Add a call to your functions here: */
int track_move(int dev, int pipe, int mpipe, struct coord cent, struct images *imgs)
{
	if (track.iomojo_id==-1)
		return stepper_move(dev, pipe, mpipe, cent, imgs);
	else
		return iomojo_move(dev, pipe, mpipe, cent, imgs);
}


/******************************************************************************

	Stepper motor on serial port

******************************************************************************/


static int stepper_command(int motor, int command, int n)
{
	char buffer[3];
	time_t timeout=time(NULL);

	buffer[0]=motor;
	buffer[1]=command;
	buffer[2]=n;
	if (write(track.dev, buffer, 3)!=3)
		return -1;

	while (read(track.dev, buffer, 1)!=1 && time(NULL) < timeout+1);
	if (time(NULL) >= timeout+2) {
		perror("Status byte timeout!");
		return 0;
	}
	return buffer[0];
}


static int stepper_status(int motor)
{
	return stepper_command(motor, STEPPER_COMMAND_STATUS, 0);
}


static int stepper_center(void)
{
	struct termios adtio;

	if (!track.dev) {
		if ((track.dev=open(track.port, O_RDWR | O_NOCTTY)) < 0) {
			perror("Unable to open serial device");
			exit(1);
		}
		bzero (&adtio, sizeof(adtio));
		adtio.c_cflag= STEPPER_BAUDRATE | CS8 | CLOCAL | CREAD;
		adtio.c_iflag= IGNPAR;
		adtio.c_oflag= 0;
		adtio.c_lflag= 0;	/* non-canon, no echo */
		adtio.c_cc[VTIME]=0;	/* timer unused */
		adtio.c_cc[VMIN]=0;	/* blocking read until 1 char */
		tcflush (track.dev, TCIFLUSH);
		if (tcsetattr(track.dev, TCSANOW, &adtio) < 0) {
			perror("Unable to initialize serial device");
			exit(1);
		}
	}
	stepper_command(track.motorx, STEPPER_COMMAND_SPEED, 255);
	stepper_command(track.motorx, STEPPER_COMMAND_LEFT_N, track.maxx);
	while ((stepper_status(track.motorx) & STEPPER_STATUS_LEFT));
	stepper_command(track.motorx, STEPPER_COMMAND_RIGHT_N, track.maxx/2);
	while (stepper_status(track.motorx) & STEPPER_STATUS_RIGHT);

	return 0;
}

static int stepper_move(int dev, int pipe, int mpipe, struct coord cent, struct images *imgs)
{
	int command=0;
	int n=0;
	
	if (cent.x < imgs->width/2) {
		command=STEPPER_COMMAND_LEFT_N;
		n=imgs->width/2-cent.x;
	}
	if (cent.x > imgs->width/2) {
		command=STEPPER_COMMAND_RIGHT_N;
		n=cent.x-imgs->width/2;
	}
	n=n*40/imgs->width;
	if (n) {
		stepper_command(track.motorx, command, n);
		while(stepper_status(track.motorx) & (STEPPER_STATUS_RIGHT|STEPPER_STATUS_LEFT) &&
		    vid_keepalive(imgs, dev, pipe, mpipe));
		return 1;
	}
	return 0;
}

/******************************************************************************

	Iomojo Smilecam on serial port

******************************************************************************/

static char iomojo_command(char *command, int len, int ret)
{
	char buffer[1];
	time_t timeout=time(NULL);

	if (write(track.dev, command, len)!=len)
		return 0;

	if (ret) {
		while (read(track.dev, buffer, 1)!=1 && time(NULL) < timeout+2);
		if (time(NULL) >= timeout+2) {
			perror("Return byte timeout!");
			return 0;
		}
	}
	return buffer[0];
}

static void iomojo_setspeed(int speed)
{
	char command[3];
	
	command[0]=IOMOJO_SETSPEED_CMD;
	command[1]=track.iomojo_id;
	command[2]=speed;
	
	if (iomojo_command(command, 3, 1)!=IOMOJO_SETSPEED_RET)
		perror("Unable to set camera speed");
}

static void iomojo_movehome(void)
{
	char command[2];
	
	command[0]=IOMOJO_MOVEHOME;
	command[1]=track.iomojo_id;

	iomojo_command(command, 2, 0);
}

static int iomojo_center(void)
{
	struct termios adtio;

	if (!track.dev) {
		if ((track.dev=open(track.port, O_RDWR | O_NOCTTY)) < 0) {
			perror("Unable to open serial device");
			exit(1);
		}
		bzero (&adtio, sizeof(adtio));
		adtio.c_cflag= IOMOJO_BAUDRATE | CS8 | CLOCAL | CREAD;
		adtio.c_iflag= IGNPAR;
		adtio.c_oflag= 0;
		adtio.c_lflag= 0;	/* non-canon, no echo */
		adtio.c_cc[VTIME]=0;	/* timer unused */
		adtio.c_cc[VMIN]=0;	/* blocking read until 1 char */
		tcflush (track.dev, TCIFLUSH);
		if (tcsetattr(track.dev, TCSANOW, &adtio) < 0) {
			perror("Unable to initialize serial device");
			exit(1);
		}
	}
	iomojo_setspeed(40);
	iomojo_movehome();

	return 0;
}

static int iomojo_move(int dev, int pipe, int mpipe, struct coord cent, struct images *imgs)
{
	char command[5];
	int direction=0;
	int nx=0, ny=0;
	int i;
	
	if (cent.x < imgs->width/2) {
		direction|=IOMOJO_DIRECTION_LEFT;
		nx=imgs->width/2-cent.x;
	}
	if (cent.x > imgs->width/2) {
		direction|=IOMOJO_DIRECTION_RIGHT;
		nx=cent.x-imgs->width/2;
	}
	if (cent.y < imgs->height/2) {
		direction|=IOMOJO_DIRECTION_LEFT;
		ny=imgs->height/2-cent.y;
	}
	if (cent.y > imgs->height/2) {
		direction|=IOMOJO_DIRECTION_RIGHT;
		ny=cent.y-imgs->height/2;
	}
	nx=nx*72/imgs->width;
	ny=ny*72/imgs->height;
	if (nx || ny) {
		if (nx > 180) nx=180;
		if (ny > 60) ny=60;
		command[0]=IOMOJO_MOVEOFFSET_CMD;
		command[1]=track.iomojo_id;
		command[2]=direction;
		command[3]=nx;
		command[4]=ny;
		iomojo_command(command, 5, 0);
		/* Number of frames to skip while moving */
		if (ny >= nx)
			i=25*ny/90;
		else
			i=25*nx/90;
		while (i--)
			vid_keepalive( imgs, dev, pipe, mpipe);
		return 1;
	}
	return 0;
}

