# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright_cap (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All right_caps reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.

from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.graph import IMAGE_FILLED, \
                                        IMAGE_NEAREST, \
                                        IMAGE_EMPTY, \
                                        IMAGE_LEFT, \
                                        IMAGE_RIGHT
from elisa.plugins.pigment.graph.image import Image


class SlicedImageHorizontal(Widget):

    def __init__(self):
        super(SlicedImageHorizontal, self).__init__()

        self.body = Image()
        self.add(self.body)
        self.body.bg_a = 0
        self.body.layout = IMAGE_FILLED
        self.body.interp = IMAGE_NEAREST
        self._body_id = self.body.connect("file-loaded",
                                          lambda x: self._layout())

        self.right_cap = Image()
        self.add(self.right_cap)
        self.right_cap.bg_a = 0
        self.right_cap.layout = IMAGE_FILLED
        self.right_cap.interp = IMAGE_NEAREST
        self._right_cap_id = self.right_cap.connect("file-loaded",
                                               lambda x: self._layout())

        self.left_cap = Image()
        self.add(self.left_cap)
        self.left_cap.bg_a = 0
        self.left_cap.layout = IMAGE_FILLED
        self.left_cap.interp = IMAGE_NEAREST
        self._left_cap_id = self.left_cap.connect("file-loaded",
                                                 lambda x: self._layout())

    def clean(self):
        self.left_cap.disconnect(self._left_cap_id)
        self.left_cap.clean()
        self.left_cap = None

        self.right_cap.disconnect(self._right_cap_id)
        self.right_cap.clean()
        self.right_cap = None

        self.body.disconnect(self._body_id)
        self.body.clean()
        self.body = None

        return super(SlicedImageHorizontal, self).clean()

    def width__get(self):
        return super(SlicedImageHorizontal, self).width__get()

    def width__set(self, width):
        super(SlicedImageHorizontal, self).width__set(width)
        self._layout()

    width = property(width__get, width__set)

    def height__get(self):
        return super(SlicedImageHorizontal, self).height__get()

    def height__set(self, height):
        super(SlicedImageHorizontal, self).height__set(height)
        self._layout()

    height = property(height__get, height__set)

    def _layout(self):
        if not self.left_cap.storage_type == IMAGE_EMPTY and \
           not self.right_cap.storage_type == IMAGE_EMPTY and \
           not self.body.storage_type == IMAGE_EMPTY:
            self._left_cap_layout()
            self._right_cap_layout()
            self._body_layout()
            self.left_cap.visible = True
            self.right_cap.visible = True
            self.body.visible = True

    def _left_cap_layout(self):
        self.left_cap.alignment = IMAGE_LEFT
        self.left_cap.x = 0.0
        self.left_cap.y = 0.0
        self.left_cap.height = 1.0
        self._respect_aspect_ratio(self.left_cap)

    def _body_layout(self):
        self.body.x = self.left_cap.x + self.left_cap.width
        self.body.y = 0.0
        self.body.height = 1.0
        self.body.width = 1.0-self.left_cap.width-self.right_cap.width

    def _right_cap_layout(self):
        self.right_cap.alignment = IMAGE_RIGHT
        self.right_cap.height = 1.0
        self._respect_aspect_ratio(self.right_cap)
        self.right_cap.x = 1.0-self.right_cap.width
        self.right_cap.y = 0.0

    def _respect_aspect_ratio(self, image):
        real_aspect_ratio = image.aspect_ratio[0]/float(image.aspect_ratio[1])
        current_aspect_ratio = image.absolute_width/image.absolute_height
        deformation = real_aspect_ratio/current_aspect_ratio

        # current_aspect_ratio is made equal to real_aspect_ratio
        image.width *= deformation

    @classmethod
    def _demo_widget(cls, *args, **kwargs):
        widget = cls()
        widget.visible = True
        return widget


if __name__ == '__main__':
    import logging
    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    i = SlicedImageHorizontal.demo()

    try:
        __IPYTHON__
    except NameError:
        import pgm
        pgm.main()
