# copyright (C) 1997-2001 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: modules.tcl,v 2.34 2001/02/18 17:44:32 jfontain Exp $}


class modules {

    class instance {

        proc instance {this module index} {
            set ($this,module) $module
            set ($this,loaded) [new module $module $index]
        }

        proc ~instance {this} {
            delete $($this,loaded)
        }

        proc load {this} {                               ;# load module since its switches need to be known for command line parsing
            set loaded $($this,loaded)
            module::load $loaded
            set namespace $module::($loaded,namespace)
            set ($this,namespace) $namespace
            catch {set ($this,switches) [set ::${namespace}::data(switches)]}                          ;# module may take no options
            set ($this,initialize) $module::($loaded,initialize)
            set ($this,version) $module::($loaded,version)
            initialize $this
        }

        proc initialize {this} {
            set namespace $($this,namespace)
            if {![catch {set ${namespace}::data(identifier)} identifier]} {            ;# store identifier if it exists and is valid
                if {![modules::validName $identifier]} {
                    foreach {name index} [modules::decoded $namespace] {}
                    puts stderr "\"$name\" module identifier: \"$identifier\" contains invalid characters"
                    exit 1
                }
                set ($this,identifier) $identifier
            }
            catch {set ($this,times) [set ${namespace}::data(pollTimes)]}                          ;# may not be available initially
            catch {set ($this,views) [set ${namespace}::data(views)]}                                     ;# there could be no views
        }

        proc synchronize {this} {
            module::synchronize $($this,loaded)
            initialize $this   ;# reinitialize in case a few variables have been set or reset in the module initialization procedure
        }

        proc empty {this} {
            module::clear $($this,loaded)
        }

    }


    set (instances) {}

    proc modules {this} error                                                                                   ;# object-less class

    proc source {interpreter package file} {
        if {[string equal [file extension $file] .pm]} {                                            ;# the module is written in Perl
            # if we got here, act as if the package was provided
            if {[catch {package require tclperl 2}]} return                                              ;# cannot load Perl modules
            set perl [perl::interp new]                                                       ;# create a temporary Perl interpreter
            $perl eval "use $package"         ;# use the Perl module, create a Tcl namespace and eventually copy the updates counter
            $interpreter eval "namespace eval $package {catch {set data(updates) [$perl eval \$${package}::data{updates}]}}"
            $interpreter eval "package provide $package [$perl eval \$${package}::VERSION]"                            ;# must exist
            perl::interp delete $perl                                                                 ;# delete the Perl interpreter
        } else {                                                                                                  ;# normal sourcing
            $interpreter eval _source [list $file]
        }
    }

    # using Tcl built-in package management facilities, seek available moodss modules
    proc available {{command {}}} {     ;# in command string, %M is replaced by module name and %S by switches list from module code
        set directory [pwd]
        set packages {}
        foreach package [package names] {
            if {[regexp\
{^(BLT|Tk|Tkined|Tktable|Tnm|apacheutilities|http|internationalization|mime|msgcat|opt|scwoop|smtp|stooop|switched|tcllib|tclperl|tcltest|tkpiechart)$}\
                $package\
            ]} continue                        ;# filter out a few well known packages, such as tclperl which causes problems anyway
            if {![info exists ::packageDirectory($package)]} continue                                        ;# for Tcl, for example
            cd $::packageDirectory($package)                                 ;# switch to module directory only during loading phase
            set interpreter [interp create]              ;# use a separate interpreter in order not to interfere with loaded modules
            $interpreter eval "set auto_path [list $::auto_path]"                    ;# set packages paths list in child interpreter
            catch {$interpreter eval {package require {}}}  ;# preload all packages locations (many pkgIndex.tcl files sourced here)
            # then intercept source command to be able to detect non Tcl modules:
            $interpreter eval {rename source _source}; $interpreter alias source ::modules::source $interpreter $package
            if {\
                ![catch {$interpreter eval "package require $package"}]&&\
                [$interpreter eval info exists ::${package}::data(updates)]\
            } {                                                                                           ;# ignore invalid packages
                lappend packages $package
                set switches {}                                                                          ;# there may be no switches
                # module package name and module namespace are identical
                catch {set switches [$interpreter eval "set ::${package}::data(switches)"]}
                set switches [list $switches]                                                                ;# make it a valid list
                if {[string length $command]>0} {
                    regsub -all %M $command $package string
                    regsub -all %S $string $switches string
                    uplevel #0 $string                                                      ;# always invoke command at global level
                }
            }
            interp delete $interpreter
        }
        cd $directory                                                                                   ;# restore current directory
        return [lsort $packages]
    }

    proc printAvailable {} {            ;# using Tcl built-in package management facilities, seek and print available moodss modules
        puts {searching for module packages, please wait...}
        foreach package [available] {
            puts -nonewline "$package: possibly in"
            set count 0
            foreach directory $::auto_path {
                if {[file readable [file join $directory $package pkgIndex.tcl]]} {
                    if {$count>0} {
                        puts -nonewline ,
                    }
                    puts -nonewline " [file join $directory $package]"
                    incr count
                }
            }
            puts {}
        }
    }

    # recursive procedure: eventually initialize next module and its eventual options in command line arguments
    proc parse {arguments} {                  ;# arguments list format is: module [-option [value] -option ...] module [-option ...]
        if {[llength $arguments]==0} return                                                                                  ;# done
        set name [lindex $arguments 0]
        set arguments [lrange $arguments 1 end]                                         ;# point to start of switches or next module
        foreach {name index} [decoded $name] {}  ;# eventually split module into its name and its index (if coming from a save file)
        if {![info exists ::packageDirectory($name)]} {                               ;# not a valid module (usually a wrong switch)
            error "error: \"$name\" is not a valid moodss module name"
        }
        if {![validName $name]} {
            error "\"$name\" module name contains invalid characters"
        }
        lifoLabel::push $global::messenger "loading $name..."
        update idletasks
        set instance [new instance $name $index]
        if {[catch {instance::load $instance} message]} {   ;# load module since its switches need be known for command line parsing
            if {$global::debug} {
                set information $::errorInfo
            }
            lifoLabel::pop $global::messenger
            delete $instance                                                                                             ;# clean up
            if {$global::debug} {
                error $information
            } else {
                error "module \"$name\" load error:\n$message"
            }
        }
        lifoLabel::pop $global::messenger
        set help [expr {[lsearch -exact $arguments --help]>=0}]                                         ;# help requested for module
        if {[info exists instance::($instance,switches)]} {                                                  ;# module takes options
            if {[llength $instance::($instance,switches)]==0} {
                error "module \"$name\" switches are empty"             ;# design error: no need to recover when dynamically loading
            }
            if {$help} {
                displayHelpMessage $name $instance::($instance,switches)
                exit
            }
            if {[catch {set next [parseCommandLineArguments $instance::($instance,switches) $arguments options]} message]} {
                delete $instance                                                                                         ;# clean up
                error "module \"$name\" options error: $message"
            }
            if {!$instance::($instance,initialize)} {
                error "module \"$name\" has no initialize procedure"    ;# design error: no need to recover when dynamically loading
            }
            set instance::($instance,options) [array get options]
            # save module arguments for eventual saving in file
            set instance::($instance,arguments) [lrange $arguments 0 [expr {[llength $arguments]-[llength $next]-1}]]
            set arguments $next
        } else {                                                                                          ;# module takes no options
            if {$help} {
                displayHelpMessage $name
                exit
            }
            set instance::($instance,arguments) {}                              ;# save module arguments for eventual saving in file
        }
        lappend (instances) $instance                                              ;# add module to successfully loaded modules list
        parse $arguments                                                                                         ;# process the rest
        update idletasks                                       ;# make sure latest loading message is not left showing meaninglessly
    }

    proc helpHTMLData {name} {                                                                          ;# module name with no index
        set noHelpText {no help available}                                               ;# in case module code does not handle help
        foreach instance $(instances) {
            set namespace $instance::($instance,namespace)
            foreach {module index} [decoded $namespace] {}
            if {[string compare $module $name]} continue
            if {![info exists text]} {                                             ;# retrieve help text from first module namespace
                set text $noHelpText
                catch {set text [set ${namespace}::data(helpText)]}
                set version $instance::($instance,version)
            }
            lappend arguments $instance::($instance,arguments)
        }
        set header "<b>$name </b>"
        append header {module version}
        if {[info exists text]} {                                                           ;# at least 1 loaded module of that name
            append header " <i>$version</i>, "
            append header instances:
            append header <ol>
            foreach argument $arguments {
                append header <li>
                if {[llength $argument]==0} {
                    append header {<i>with no options</i>}
                } else {
                    append header <b>$argument</b>
                }
                append header </li>
            }
            append header </ol>
        } else {
            foreach {version text} [versionAndHelpText $name] {}                                     ;# retrieve data in another way
            if {[string length $text]==0} {
                set text $noHelpText
            }
            append header " <i>$version</i>"
        }
        append header <br><br>
        if {[regsub -nocase <body> $text <body>$header text]>0} {                            ;# insert header if HTML formatted help
            # eventually remove title which appears on viewer, since we already generated a title
            regsub -nocase {<title>.*</title>} $text {} text
            return $text
        } else {
            regsub -all \n $text <br> text                                                 ;# regular help, keep original formatting
            return ${header}$text
        }
    }

    proc versionAndHelpText {name} {                                               ;# returns module version and help text in a list
        set directory [pwd]
        cd $::packageDirectory($name)                                        ;# switch to module directory only during loading phase
        set interpreter [interp create]                  ;# use a separate interpreter in order not to interfere with loaded modules
        $interpreter eval "set auto_path [list $::auto_path]"         ;# duplicate in case module interpreter requires some packages
        $interpreter eval "package require $name"
        set version [$interpreter eval "package provide $name"]
        set text {}                                                                           ;# there may be no help for the module
        catch {set text [$interpreter eval "namespace eval $name {set data(helpText)}"]}
        interp delete $interpreter
        cd $directory                                                                                   ;# restore current directory
        return [list $version $text]
    }

    proc initialize {} {      ;# eventually invoke modules initialization procedure once. modules must be loaded first (see parse{})
        foreach instance $(instances) {
            if {!$instance::($instance,initialize)} continue
            set namespace $instance::($instance,namespace)
            lifoLabel::push $global::messenger "initializing $namespace..."
            update idletasks
            set error 0
            if {[info exists instance::($instance,options)]} {
                if {[catch {::${namespace}::initialize $instance::($instance,options)} message]} {
                    if {$global::debug} {
                        set information $::errorInfo
                    }
                    set error 1
                }
            } else {                                                                                      ;# module takes no options
                if {[catch ::${namespace}::initialize message]} {
                    if {$global::debug} {
                        set information $::errorInfo
                    }
                    set error 1
                }
            }
            lifoLabel::pop $global::messenger
            if {$error} {
                unload $instance                                                                                         ;# clean up
                if {$global::debug} {
                    error $information
                } else {
                    error "module \"$namespace\" initialize error:\n$message"
                }
            } else {
                instance::synchronize $instance                                                          ;# in case data was updated
            }
            set instance::($instance,initialize) 0                                ;# a module instance must be initialized once only
        }
        update idletasks                                ;# make sure latest initialization message is not left showing meaninglessly
    }

    proc setPollTimes {{override {}}} {
        if {[llength $(instances)]==0} {
            set global::pollTimes {}
            set global::pollTime 0
            return
        }
        set default 0
        set minimum 0
        foreach instance $(instances) {
            set times $instance::($instance,times)
            if {[llength $times]==0} {
                error "module $instance::($instance,namespace) poll times list is empty"
            }
            # for an asynchronous module, the sole time value would be negative and is used as graph interval, for example
            set time [lindex $times 0]
            if {$time<0} {                                          ;# asynchronous module, poll time is a viewer interval (negated)
                set intervals($time) {}
                continue
            }
            if {$time>$default} {                                                              ;# default value is the first in list
                set default $time                                                    ;# keep the greater default time of all modules
            }
            set times [lsort -integer $times]                                                                     ;# sort poll times
            set time [lindex $times 0]
            if {$time>$minimum} {
                set minimum $time                                                    ;# keep the greater minimum time of all modules
                set minimumModule $instance::($instance,namespace)
            }
            foreach time $times {                                    ;# poll times list is the combination of all modules poll times
                set data($time) {}
            }
        }
        # sort and restrict poll times above maximum module minimum poll time
        set global::pollTimes [lsort -integer [array names data]]
        set global::pollTimes [lrange $global::pollTimes [lsearch -exact $global::pollTimes $minimum] end]
        set global::pollTime $default
        if {[string length $override]>0} {                                              ;# eventually validate command line override
            if {$override<$minimum} {
                puts stderr "$::argv0: minimum poll time is $minimum seconds for module $minimumModule"
                exit 1
            }
            set global::pollTime $override
        }
        if {$global::pollTime==0} { 
            # all modules are asynchronous, so use an average time as a viewer interval for viewers that need it, such as graphs.
            # the poll times list is empty at this point so the user cannot change the poll time.
            # note that the viewer interval can still be forced by the command line poll time option.
            set sum 0
            set number 0
            foreach interval [array names intervals] {
                incr sum $interval
                incr number
            }
            set global::pollTime [expr {round(double($sum)/-$number)}]
        }
    }

    proc identifier {array} {  ;# from an array name, eventually deduce a unique module identifier if needed (used in viewer labels)
        variable nextIndex

        set namespace [namespaceFromArray $array]
        foreach instance $(instances) {
            if {[string equal $namespace $instance::($instance,namespace)]} {                              ;# this is a module array
                if {[info exists instance::($instance,identifier)]} {
                    return $instance::($instance,identifier)                                        ;# favor identifier if it exists
                }
                foreach {module index} [decoded $namespace] {}
                if {[string length $index]>0} {        ;# there are more than 1 instance of this module, so identification is needed
                    return $namespace
                }
                break
            }
        }
        return {}                                                                   ;# not a module array or identification unneeded
    }

    proc decoded {name} {     ;# return module and index list (index may be empty if module is not indexed: name instead of name<N>)
        set index {}
        scan $name {%[^<]<%u>} name index                                     ;# eventually split module into its name and its index
        return [list $name $index]
    }

    proc validName {string} {                                                                               ;# includes iso-8859 set
        return [regexp {^[\w @%&*()=+:.-]+$} $string]
    }

    proc displayHelpMessage {name {switches {}}} {
        puts -nonewline "$name module usage:"
        if {[llength $switches]==0} {
            puts -nonewline { <no arguments allowed>}
        } else {
            foreach {switch argument} $switches {
                puts -nonewline " \[$switch"
                if {$argument} {                                                                        ;# option takes one argument
                    puts -nonewline { argument}
                }
                puts -nonewline \]
            }
        }
        puts {}
    }

    proc loaded {} {    ;# return a list of namespaces with options, options being of list of switch, argument required and argument
        if {[llength $(instances)]==0} {
            return {}
        }
        foreach instance $(instances) {
            lappend list [list $instance $instance::($instance,namespace)]
        }
        set return {}
        foreach list [lsort -dictionary -index 1 $list] {                                    ;# sort in namespace alphabetical order
            foreach {instance namespace} $list {}
            lappend return $namespace
            set switches {}                                                                       ;# in case module takes no options
            catch {set switches $instance::($instance,switches)}
            if {[llength $switches]==0} {
                lappend return {}
            } else {
                catch {unset argument}
                foreach {switch required} $switches {
                    set argument($switch) $required
                }
                set arguments $instance::($instance,arguments)
                set length [llength $arguments]
                set list {}
                for {set index 0} {$index<$length} {incr index} {
                    set item [lindex $arguments $index]
                    lappend list $item
                    if {$argument($item)} {
                        lappend list 1 [lindex $arguments [incr index]]                    ;# append value then point to next switch
                    } else {
                        lappend list 0 1                                  ;# value is true for boolean options since switch was used
                    }
                }
                lappend return $list
            }
        }
        return $return
    }

    proc names {} {  ;# return of list of all different loaded module names (a name is unique even if several instances were loaded)
        set list {}
        foreach instance $(instances) {
            set module $instance::($instance,module)
            if {[lsearch -exact $list $module]<0} {
                lappend list $module
            }
        }
        return $list
    }

    proc unload {instance} {
        ldelete (instances) $instance
        delete $instance
    }

    proc namespaceFromArray {name} {
        return [string trimleft [namespace qualifiers [namespace which -variable $name]] :]
    }

    proc trace {module namespace message} {                                              ;# destined to trace module if instantiated
        foreach instance $(instances) {
            if {[string equal $instance::($instance,module) trace]} {
                set traceNamespace $instance::($instance,namespace)
                ::${traceNamespace}::update $module $namespace $message
            }
        }
    }

    proc flashMessage {module namespace message {seconds 1}} {
        ::lifoLabel::flash $::global::messenger $message $seconds
        trace $module $namespace $message                                                        ;# also eventually display in trace
    }

    proc pushMessage {module namespace message} {
        ::lifoLabel::push $::global::messenger $message
        trace $module $namespace $message                                                        ;# also eventually display in trace
    }

    proc popMessage {} {
        ::lifoLabel::pop $::global::messenger
    }

}
