# copyright (C) 1997-2001 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: cpustats.tcl,v 2.5 2001/02/04 10:58:07 jfontain Exp $}


package provide cpustats [lindex {$Revision: 2.5 $} 1]
package require network 1

namespace eval cpustats {

    array set data {
        updates 0
        0,label CPU 0,type ascii 0,message {CPU number (0 for all or a single CPUs)}
        1,label user 1,type real 1,message {percentage spent in user mode}
        2,label system 2,type real 2,message {percentage spent in system mode}
        3,label nice 3,type real 3,message {percentage spent in nice mode}
        4,label idle 4,type real 4,message {percentage spent in idle mode}
        sort {0 increasing}
        switches {-r 1 --remote 1}
    }
    set file [open cpustats.htm]
    set data(helpText) [read $file]                                                           ;# initialize HTML help data from file
    close $file

    proc initialize {optionsName} {
        upvar $optionsName options
        variable remote
        variable data
        variable statisticsFile

        set lookup [expr {![info exists options(-n)]&&![info exists options(--numeric)]}]            ;# host or network names lookup
        if {![catch {set locator $options(--remote)}]||![catch {set locator $options(-r)}]} {                   ;# remote monitoring
            set data(pollTimes) {20 10 30 60 120 300 600}                                ;# poll less often when remotely monitoring
            foreach {remote(protocol) remote(user) remote(host)} [network::parseRemoteLocator $locator] {}
            network::checkRemoteOutputEmptiness $remote(protocol) $remote(user) $remote(host)
            set data(identifier) cpustats($remote(host))
            set file [open "| /usr/bin/$remote(protocol) -n -l $remote(user) $remote(host) cat /proc/stat"]
            fileevent $file readable {set ::cpustats::remote(busy) 0}
            vwait ::cpustats::remote(busy)
            # detect errors early (but avoid write on pipe with no readers errors by reading whole data)
            if {[catch {read $file} message]||[catch {close $file} message]} {
                error "on remote host $remote(host) as user $remote(user): $message"
            }
        } else {
            set data(pollTimes) {10 5 20 30 60 120 300 600}
            set statisticsFile [open /proc/stat]                                      ;# keep local file open for better performance
        }
    }

    proc update {} {                                               ;# gather cpu statistics (based on the proc man page information)
        variable remote
        variable statisticsFile
        variable data
        variable last

        if {[info exists remote]} {
            if {![info exists statisticsFile]} {                           ;# start data gathering process in a non blocking fashion
                if {$remote(busy)} return                                           ;# core invocation while waiting for remote data
                set remote(busy) 1
                set file [open "| /usr/bin/$remote(protocol) -n -l $remote(user) $remote(host) cat /proc/stat"]
                # do not hang GUI, allow other modules updates
                fileevent $file readable "set ::cpustats::statisticsFile $file; ::cpustats::update"
                return                                                                                       ;# wait for remote data
            }                                                                                 ;# else continue below to process data
        } else {
            seek $statisticsFile 0                                                                  ;# rewind before retrieving data
        }
        gets $statisticsFile firstLine
        set lines {}
        while {([gets $statisticsFile line]>=0)&&[string match cpu* $line]} {                            ;# only save CPU data lines
            lappend lines $line
        }
        if {[info exists remote]} {
            read $statisticsFile                             ;# avoid write on pipe with no readers errors by reading remaining data
            if {[catch {close $statisticsFile} message]} {                               ;# communication error can be detected here
                flashMessage "cpustats error: $message"
                unset firstLine                                                                ;# consider data corrupted as a whole
            }
            unset statisticsFile
            set remote(busy) 0
        }
        if {[info exists firstLine]} {
            set index 0                                                 ;# first line is for all CPUs, data is in 100ths of a second
            scan $firstLine {cpu %d %d %d %d} user nice system idle
            updateRow $index $user $nice $system $idle
            if {[llength $lines]>1} {                                 ;# display per CPU statistics only if there is more than 1 CPU
                foreach line $lines {
                    if {[scan $line {cpu%*u %d %d %d %d} user nice system idle]==4} {
                        updateRow [incr index] $user $nice $system $idle
                    }
                }
            }
        } else {                                                                                                ;# data is corrupted
            catch {array unset data {[0-9]*,[0-9]*}}
            catch {unset last}
            array set data {0,0 0 0,1 ? 0,2 ? 0,3 ? 0,4 ?}              ;# only display first row (for all CPUs) with unknown values
        }
        incr data(updates)
    }

    proc updateRow {index user nice system idle} {
        variable last
        variable data

        if {![info exists last($index,user)]} {        ;# calculate statistics during the last poll period except for the first pass
            set data($index,0) $index
            set last($index,user) 0
            set last($index,system) 0
            set last($index,nice) 0
            set last($index,idle) 0
        }
        set multiplier\
            [expr {100.0/($user-$last($index,user)+$nice-$last($index,nice)+$system-$last($index,system)+$idle-$last($index,idle))}]
        array set data [list\
            $index,1 [format %.1f [expr {$multiplier*($user-$last($index,user))}]]\
            $index,2 [format %.1f [expr {$multiplier*($system-$last($index,system))}]]\
            $index,3 [format %.1f [expr {$multiplier*($nice-$last($index,nice))}]]\
            $index,4 [format %.1f [expr {$multiplier*($idle-$last($index,idle))}]]\
        ]
        set last($index,user) $user
        set last($index,system) $system
        set last($index,nice) $nice
        set last($index,idle) $idle
    }

}
