/* mondo-prep.c             Hugo Rabson                           03/10/2002


03/10
- mkreiserfs -ff instead of -q (Andy Glass)

02/19
- don't resize drive if drive not found (or if its size cannot be det'd)
- when generating list of drives from mountlist, skip the 'p' at the end
  of drive name if /dev/ida/ or /dev/cciss/; just do it (Michael Eisenberg)

02/14
- don't specify journal size when formatting ext3
  (used to have -Jsize=10 in the call to mkfs)

01/28
- handles files >2GB in size

01/20
- format interactively, if Interactive Mode

01/08
- removed reference to g_tape_size

12/28
- when executing /tmp/i-want-my-lvm, only record the error# if the command
  was _not_ an 'insmod' command

12/24
- pass partition size to fdisk in Kilobytes now, not Megabytes
- log fdisk's output to /tmp/mondo-restore.log if it fails

12/19
- don't try to format partitions of type 'image'
- don't type to set types of 'image' partitions
- if format code is 1 or 2 chars then assume it is a hex string

12/11
- took out all '/ /' comments

12/09
- don't extrapolate/add partition from RAID dev to mountlist if it's already
  present in mountlist
- less repetitive logging in the event of vacuum-packing of last part'n
- no extrapolation at all: RAID partitions should be listed in mountlist
  already, thanks to either Mindi v0.5x or the mountlist editor itself
- no longer say, 'and logging to...' when setting a partition's type

12/07
- don't run mkfs on RAID partitions (/dev/hd*, /dev/sd*); just set type

12/04
- the setting of a partition's type now takes place in a separate subroutine
  from the subroutine that actually creates the partition
- no need to set type if 83: 83 is the default (under fdisk)
- turned on '-Wall'; cleaned up some cruft
- if vacuum-packing partition (i.e. size=0MB --> max) then say, "(maximum)"
  not, "(0 MB)"

11/22
- preliminary code review


Created on Nov 22nd, 2001
*/


#include "my-stuff.h"


#define ARCHIVES_PATH "/mnt/cdrom/archives"

/* global vase - er, vars */
extern long g_maximum_progress, g_current_progress, g_start_time;
extern int g_currentY, g_current_cd_number;
extern char *g_tape_device;


/* externs */
extern void finish(int);
extern void setup_newt_stuff(void);
extern char which_restore_mode(void);
extern int ask_me_yes_or_no(char*);
extern long get_phys_size_of_drive(char*);
extern void log_to_screen(char*);
extern void update_progress_form(char*);
extern void open_progress_form(char*,char*,char*,char*,long);
extern void close_progress_form(void);
extern void popup_and_OK(char*);
extern int  popup_and_get_string(char*,char*,char*);
extern long get_time(void);
extern int run_program_and_log_output(char*);
extern int is_this_device_mounted(char*);
extern int does_partition_exist(char*device,int partno);
extern int strcmp_inc_numbers(char*,char*);
extern long count_lines_in_file(char*);
extern long length_of_file(char*);
extern long noof_lines_that_match_wildcard(char*,char*);
extern char *slice_fname(long,long,bool,char*);
extern char *last_line_of_file(char*);
extern void log_file_end_to_screen(char*,char*);
extern void log_it(char*);
extern int zero_out_a_device(char*);
extern void mvaddstr_and_log_it(int, int, char *);
extern bool does_file_exist(char*);


/* ------ my stuff ------ */
int extrapolate_mountlist_to_include_raid_partitions(struct mountlist_itself*, struct mountlist_itself*);
bool mountlist_contains_raid_devices(struct mountlist_itself*);
int stop_raid_device(char*);
int stop_all_raid_devices(struct mountlist_itself*);
int format_everything(struct mountlist_itself*, bool);
int partition_device(char*,int,int,char*,long long);
int format_device(char*,char*);
int partition_drive(struct mountlist_itself*, char*);
int partition_everything(struct mountlist_itself*);
int do_my_funky_lvm_stuff(struct mountlist_itself*);
int which_format_command_do_i_need(char*,char*);
int make_dummy_partitions(char*,int);
int make_list_of_drives(struct mountlist_itself*, char drivelist[ARBITRARY_MAXIMUM][MAX_STR_LEN]);
int set_partition_type(char*, int, char*, long);
void resize_drive_proportionately_to_suit_new_drives(struct mountlist_itself *mountlist, char*drive_name);
void resize_mountlist_proportionately_to_suit_new_drives(struct mountlist_itself *mountlist);




/* ----------------------------------------------------------------------- */




int do_my_funky_lvm_stuff(struct mountlist_itself *mountlist)
{
  char tmp[MAX_STR_LEN],incoming[MAX_STR_LEN],command[MAX_STR_LEN],*p;
  int retval=0,res=0;
  FILE*fin;
  fin=fopen("/tmp/i-want-my-lvm","r");
  if (!fin) {return(1);}
  log_it("OK, opened i-want-my-lvm. Doing funky stuff...");
  for(fgets(incoming,512,fin); !feof(fin); fgets(incoming,512,fin))
    {
      if (incoming[0]!='#') {continue;}
      for(p=incoming+1;*p==' ';p++);
      strcpy(command,p);
      for(p=command;*p!='\0';p++);
      for(;*(p-1)<32;p--);
      *p='\0';
      res=run_program_and_log_output(command);
      sprintf(tmp,"'%s' returned %d",command,res);
      if (!strstr(command,"insmod")) {retval+=res;}
      log_it(tmp);
    }
  fclose(fin);
  log_it("Closed i-want-my-lvm. Finished doing funky stuff.");
  return(retval);
}




int extrapolate_mountlist_to_include_raid_partitions(struct mountlist_itself *new_mountlist, struct mountlist_itself *old_mountlist)
{
  FILE*fin;
  int lino,j;
  char incoming[MAX_STR_LEN],*p,tmp[MAX_STR_LEN];
  new_mountlist->entries = 0;
  for(lino=0; lino < old_mountlist->entries; lino++)
    {
      if (strstr(old_mountlist->el[lino].device,"/dev/md"))
	{
	  if (!does_file_exist("/etc/raidtab")) { log_to_screen("Cannot find /etc/raidtab - cannot extrapolate the fdisk entries"); finish(1); }
	  fin=fopen("/etc/raidtab","r");
	  if (!fin) { log_to_screen("Cannot open /etc/raidtab"); finish(1); }
	  for(fgets(incoming,MAX_STR_LEN-1,fin); !feof(fin) && !strstr(incoming,old_mountlist->el[lino].device); fgets(incoming,MAX_STR_LEN-1,fin));
	  if (!feof(fin))
	    {
	      sprintf(tmp,"Investigating %s",old_mountlist->el[lino].device);
	      log_it(tmp);
	      for(fgets(incoming,MAX_STR_LEN-1,fin); !feof(fin) && !strstr(incoming,"raiddev"); fgets(incoming,MAX_STR_LEN-1,fin))
		{
		  if (strstr(incoming,"device") && !strchr(incoming,'#'))
		    {
		      for(p=incoming+strlen(incoming); *(p-1) <= 32; p--);
		      *p='\0';
		      for(p--; p>incoming && *(p-1)>32; p--);
		      sprintf(tmp,"Extrapolating %s",p);
		      log_it(tmp);
		      for(j=0; j<new_mountlist->entries && strcmp(new_mountlist->el[j].device, p); j++);
		      if (j >= new_mountlist->entries)
			{
			  strcpy(new_mountlist->el[new_mountlist->entries].device,p);
			  strcpy(new_mountlist->el[new_mountlist->entries].mountpoint,"raid");
			  strcpy(new_mountlist->el[new_mountlist->entries].format,"raid");
			  new_mountlist->el[new_mountlist->entries].size = old_mountlist->el[lino].size;
			  new_mountlist->entries++;
			}
		      else
			{
			  sprintf(tmp,"Not adding %s to mountlist: it's already there",p);
			  log_it(tmp);
			}
		    }
		}
	    }
	  fclose(fin);
	}
      else
	{
	  strcpy(new_mountlist->el[new_mountlist->entries].device, old_mountlist->el[lino].device);
	  strcpy(new_mountlist->el[new_mountlist->entries].mountpoint, old_mountlist->el[lino].mountpoint);
	  strcpy(new_mountlist->el[new_mountlist->entries].format, old_mountlist->el[lino].format);
	  new_mountlist->el[new_mountlist->entries].size = old_mountlist->el[lino].size;
	  new_mountlist->entries++;
	}
    }
  return(0);
}






int format_device(char*device, char*format)
{
  int res,retval=0;
  char program[MAX_STR_LEN],tmp[MAX_STR_LEN];

  if (strcmp(format,"lvm")==0) /* do not format LVM PV's */
    {
      sprintf(tmp,"Not formatting %s (it is an LVM PV)",device);
      log_it(tmp);
      return(0);
    }
  if (strcmp(format,"raid")==0) /* do not form RAID disks; do it to /dev/md* instead */
    {
      sprintf(tmp,"Not formatting %s (it is a RAID disk)",device);
      log_it(tmp);
      return(0);
    }
  if (strlen(format)<=2)
    {
      sprintf(tmp,"%s has a really small format type ('%s') - this is probably a hexadecimal string, which would suggest the partition is an image --- I shan't format it",device,format);
      log_it(tmp);
      return(0);
    }
  if (is_this_device_mounted(device))
    {
      sprintf(tmp,"%s is mounted - cannot format it       ",device);
      log_to_screen(tmp);
      return(1);
    }
  if (strstr(device,"/dev/md"))
    {
      sprintf(tmp,"Initializing RAID device %s",device);
      log_to_screen(tmp);
      stop_raid_device(device);
      /* format raid partition */
      sprintf(program,"mkraid --really-force %s",device);
      res=run_program_and_log_output(program);
      /* retval+=res; */
    }
  res=which_format_command_do_i_need(format,program);
  sprintf(tmp,"%s %s",program,device);
  if (strstr(program,"kludge")) { strcat(tmp," /"); }
  sprintf(program,"sh -c 'echo -en \"y\\ny\\ny\\n\" | %s'",tmp);
  sprintf(tmp,"Formatting %s as %s",device,format);
  update_progress_form(tmp);
  res=run_program_and_log_output(program);
  if (res && strstr(program,"kludge"))
    {
      sprintf(tmp,"Kludge failed; using regular mkfs.%s to format %s",format,device);
      sprintf(program,"mkfs -t %s -F 32 %s",format,device);
      res=run_program_and_log_output(program);
    }
  retval+=res;
  if (retval)
    {
      strcat(tmp,"...failed");
    }
  else
    {
      strcat(tmp,"...OK");
    }
  log_to_screen(tmp);
  return(retval);
}



int format_everything(struct mountlist_itself *mountlist, bool interactively)
{
  int res,retval=0,lino;
  bool do_it;
  char tmp[MAX_STR_LEN], question[MAX_STR_LEN];
  mvaddstr_and_log_it(g_currentY,0,"Formatting partitions     ");
  open_progress_form("Formatting partitions","I am now formatting your hard disk partitions.", "This may take up to five minutes.","",mountlist->entries);
  /* do my funky LVM stuff if necessary */
  if (does_file_exist("/tmp/i-want-my-lvm"))
    {
      log_to_screen("I want my... I want my LVM... *cue Dire Straits*");
      res=do_my_funky_lvm_stuff(mountlist);
      retval+=res;
      if (res) { log_to_screen("do_my_funky_lvm_stuff returned errors"); }
      else { log_it("do_my_funky_lvm_stuff ran OK"); }
    }
  for(lino=0; lino < mountlist->entries; lino++)
    {
      if (!strcmp(mountlist->el[lino].mountpoint,"image"))
	{ sprintf(tmp,"Not formatting %s - it's an image",mountlist->el[lino].device); log_it(tmp); res=0; }
      else
	{
	  if (interactively)
	    {
	      sprintf(question,"Shall I format %s (%s) ?", mountlist->el[lino].device, mountlist->el[lino].mountpoint);
	      do_it = ask_me_yes_or_no(question);
	    }
	  else
	    {
	      do_it = TRUE;
	    }
	  if (do_it)
	    {
	      res=format_device(mountlist->el[lino].device,mountlist->el[lino].format);
	    }
	  else
	    {
	      res=0; 
	    }
	}
      retval+=res;
      g_current_progress++;
    }
  close_progress_form();
  if (retval)
    {
      mvaddstr_and_log_it(g_currentY++,74,"Failed.");
      log_to_screen("Errors occurred during the formatting of your hard drives.");
    }
  else
    {
      mvaddstr_and_log_it(g_currentY++,74,"Done.");
    }
  return(retval);
}





int make_dummy_partitions(char *drivename, int devno_we_must_allow_for)
{
  int current_devno, previous_devno, retval=0, res;
  char tmp[MAX_STR_LEN];
  if (devno_we_must_allow_for >= 5)
    {
      sprintf(tmp,"Making dummy primary %s%d",drivename,1);
      log_it(tmp);
      g_maximum_progress++;
      res=partition_device(drivename,1,0,"ext2",32000);
      retval+=res;
      previous_devno=1;
      current_devno=5;
    }
  else
    {
      previous_devno=0;
      current_devno=1;
    }
  for(; current_devno < devno_we_must_allow_for; current_devno++)
    {
      sprintf(tmp,"Creating dummy partition %s%d",drivename,current_devno);
      log_it(tmp);
      g_maximum_progress++;
      res=partition_device(drivename,current_devno,previous_devno,"ext2",32000);
      retval+=res;
      previous_devno=current_devno;
    }
  return(previous_devno);
}



int make_list_of_drives(struct mountlist_itself *mountlist, char drivelist[ARBITRARY_MAXIMUM][MAX_STR_LEN])
{
  int lino, noof_drives=0, i, j;
  char drive[MAX_STR_LEN];
  for(lino=0; lino < mountlist->entries; lino++)
    {
      strcpy(drive,mountlist->el[lino].device);
      for(i=strlen(drive); isdigit(drive[i-1]); i--);
/*      if ((strncmp("/dev/cciss/",drive,11)==0 || strncmp("/dev/ida/",drive,9)==0) && isalpha(drive[i-1]))
        { i--; }
*/
      drive[i]='\0';
      if (get_phys_size_of_drive(drive)<=0 && drive[i-1]=='p') { i--; drive[i]='\0'; }
      for(j=0; j<noof_drives && strcmp(drivelist[j],drive)!=0; j++);
      if (j==noof_drives)
	{
	  strcpy(drivelist[noof_drives++],drive);
	}
    }
  return(noof_drives);
}





bool mountlist_contains_raid_devices(struct mountlist_itself *mountlist)
{
  int i, matching=0;
  for(i=0; i<mountlist->entries; i++)
    {
      if (strstr(mountlist->el[i].device,"/dev/md")) { matching++; }
    }
  if (matching)
    { return(TRUE); }
  else
    { return(FALSE); }
}




int partition_drive(struct mountlist_itself *mountlist, char *drivename)
{
  int current_devno, previous_devno=0, lino, retval=0, res;
  long long partsize;
  char devicestring[MAX_STR_LEN], format[MAX_STR_LEN], tmp[MAX_STR_LEN];

  sprintf(tmp,"Partitioning drive %s",drivename);
  log_it(tmp);
  for(current_devno=1; current_devno<99; current_devno++)
    {
      sprintf(devicestring,"%s%d",drivename,current_devno);
      for(lino=0; lino < mountlist->entries && strcmp(mountlist->el[lino].device,devicestring)!=0; lino++);
      if (lino >= mountlist->entries) {continue;}
      /* OK, we've found partition /dev/hdxN in mountlist; let's prep it */
      if (!previous_devno)
	{
	  sprintf(tmp,"Wiping %s",drivename);
	  log_it(tmp);
	  if (zero_out_a_device(drivename))
	    {
	      sprintf(tmp,"Warning - unable to zero %s",drivename);
	      log_to_screen(tmp);
	    }
	  if (current_devno > 1)
	    { previous_devno=make_dummy_partitions(drivename,current_devno); }
	}
      strcpy(format,mountlist->el[lino].format);
      partsize=mountlist->el[lino].size;
      if (current_devno<=4)
	{
	  res=partition_device(drivename,current_devno,previous_devno,format,partsize);
	  retval+=res;
	}
      if (current_devno>=5)
	{
	  if (current_devno==5 && previous_devno==4) { log_to_screen("You must leave at least one partition spare as the Extended partition."); return(1); }
	  res=partition_device(drivename,current_devno,previous_devno,format,partsize);
	  retval+=res;
	}
      if (previous_devno+1 < current_devno)
	{
	  /*
	  for(i=previous_devno+1;i<current_devno;i++)
	    {
	      sprintf(tmp,"%s%d not mentioned in mountlist.",drivename,i);
	      log_to_screen(tmp);
	    }
	  */
	}
      previous_devno=current_devno;
    }
  return(retval);
}


int partition_device(char*drive, int partno, int prev_partno, char *format, long long partsize)
{
  int retval, res;
  char program[MAX_STR_LEN], devicestring[MAX_STR_LEN], tmp[MAX_STR_LEN],
	logfile[MAX_STR_LEN], *p, output[MAX_STR_LEN];
  FILE*fout;

  sprintf(devicestring,"%s%d",drive,partno);
  if (partsize<=0)
    { sprintf(tmp,"Partitioning device %s (max size)",devicestring); }
  else
    { sprintf(tmp,"Partitioning device %s (%ld MB)",devicestring,(long)partsize/1024); }
  update_progress_form(tmp);
  log_it(tmp);
  if (is_this_device_mounted(devicestring))
    {
      sprintf(tmp,"%s is mounted, and should not be partitioned",devicestring);
      log_to_screen(tmp);
      return(1);
    }
  else if (does_partition_exist(drive,partno))
    {
      sprintf(tmp,"%s already has a partition",devicestring);
      log_to_screen(tmp);
      return(1);
    }

  /*  sprintf(tmp,"Partitioning %s  ",devicestring); */
  /*  mvaddstr_and_log_it(g_currentY+1,30,tmp); */
  p=(char*)strrchr(devicestring,'/');
  sprintf(logfile,"/tmp/fdisk.log.%s",++p);
  sprintf(program,"fdisk %s > %s 2> %s.err",drive,logfile,logfile);

  output[0]='\0';
  /* make it a primary/extended/logical */
  if (partno<=4)
    {
      sprintf(output+strlen(output),"n\np\n%d\n",partno); 
    }
  else
    {
      if (partno==5)
	{
	  if (prev_partno >= 4)
	    {
	      log_to_screen("You need to leave at least one partition free, for 'extended/logical'");
	      return(1);
	    }
	  else
	    {
	      sprintf(output+strlen(output),"n\ne\n%d\n\n\n",prev_partno+1);
	    }
	}
      strcat(output+strlen(output),"n\nl\n");
    }
  strcat(output+strlen(output),"\n");  /*start block (ENTER for next free blk*/
  if (partsize>0)
    {
      sprintf(output+strlen(output), "+%lld",(long long)(partsize));
      strcat(output+strlen(output),"K");
    }
  strcat(output+strlen(output),"\n");
  strcat(output+strlen(output),"w\n\n");
/*
  sprintf(tmp,"PARTSIZE = +%ld",(long)partsize/1024);
  log_it(tmp);
  log_it("---fdisk command---");
  log_it(output);
  log_it("---end of fdisk---");
*/
  /* write to disk; close fdisk's stream */
  fout=popen(program,"w");
  fputs(output,fout);
  pclose(fout);

  if (!does_partition_exist(drive,partno) && partsize>0)
    {
      log_it("Vaccum-packing");
      g_current_progress--;
      res=partition_device(drive,partno,prev_partno,format,-1);
      if (res)
	{
	  sprintf(tmp,"Failed to vacuum-pack %s%d",drive,partno);
	  log_it(tmp);
	  retval++;
	}
      else
        {
          retval=0;
        }
    }
  if (does_partition_exist(drive,partno))
    {
      retval=set_partition_type(drive,partno,format,partsize);
      if (retval)
        {
          sprintf(tmp,"Partitioned %s%d but failed to set its type",drive,partno);
          log_it(tmp);
	}
      else
        {
          if (partsize>0)
            {
              sprintf(tmp,"Partition %s%d created+configured OK",drive,partno);
              log_to_screen(tmp);
            }
          else
            {
              log_it("Returning from a successful vacuum-pack");
            }
        }
    }
  else
    {
      sprintf(tmp,"Failed to partition %s%d",drive,partno);
      if (partsize>0) 
	{
	  log_to_screen(tmp); 
	  sprintf(tmp,"cat %s %s.err >> %s",logfile,logfile,"/tmp/mondo-restore.log");
	  system(tmp);
	}
      else { log_it(tmp); }
      retval++;
    }
  /*
  log_it("------------------------start of fdisk log------------------------");
  sprintf(tmp,"cat %s >> %s",logfile,MONDO_LOGFILE);
  if (system(tmp)) { log_it("Failed to append fdisk's output to the log file"); }
  log_it("--------------------------end of fdisk log------------------------");
  */
  g_current_progress++;
  return(retval);
}



int partition_everything(struct mountlist_itself *mountlist)
{
  int lino, retval=0, noof_drives=0, i, res;
  char drivelist[ARBITRARY_MAXIMUM][MAX_STR_LEN];
  /*  struct mountlist_itself new_mtlist, *mountlist; */

  log_it("partition_everything() --- starting");
  mvaddstr_and_log_it(g_currentY,0,"Partitioning hard drives      ");
  /*  mountlist=orig_mtlist;*/
  if (mountlist_contains_raid_devices(mountlist))
    {
      /*      mountlist=&new_mtlist; */
      /*      extrapolate_mountlist_to_include_raid_partitions(mountlist,orig_mtlist); */
      log_it("Mountlist, including the partitions incorporated in RAID devices:-");
      for(i=0;i<mountlist->entries;i++)
	{ log_it(mountlist->el[i].device); }
      log_it("End of mountlist.");
      res=stop_all_raid_devices(mountlist);
    }
  open_progress_form("Partitioning devices","I am now going to partition all your drives.", "This should not take more than five minutes.","",mountlist->entries);

  noof_drives = make_list_of_drives(mountlist,drivelist);

  /* partition each drive */
  for(lino = 0; lino < noof_drives; lino++)
    {
      res=partition_drive(mountlist, drivelist[lino]);
      retval+=res;
    }
  close_progress_form();
  if (retval)
    {
      mvaddstr_and_log_it(g_currentY++,74,"Failed.");
      log_to_screen("Errors occurred during the partitioning of your hard drives.");
    }
  else
    {
      mvaddstr_and_log_it(g_currentY++,74,"Done.");
      system("rm -f /tmp/fdisk*.log 2> /dev/null");
    }
  return(retval);
}






int set_partition_type(char*drive, int partno, char*format, long partsize)
{
  char device[MAX_STR_LEN];
  char command[MAX_STR_LEN], output[MAX_STR_LEN], tmp[MAX_STR_LEN], partcode[MAX_STR_LEN], logfile[MAX_STR_LEN], *p;
  FILE*fout;
  int res=0;

  sprintf(device,"%s%d",drive,partno);
  p=(char*)strrchr(device,'/');
  sprintf(logfile,"/tmp/fdisk-set-type.%s.log",++p);
  if (strcmp(format,"swap")==0) { strcpy(partcode,"82"); }
  else if (strcmp(format,"vfat")==0)
    {
      if (partsize/1024 > 8192) { strcpy(partcode,"c"); }
      else                      { strcpy(partcode,"b"); }
    }
  else if (strcmp(format,"ext2")==0 || strcmp(format,"reiserfs")==0 || strcmp(format,"ext3")==0 || strcmp(format,"xfs")==0 || strcmp(format,"jfs")==0)
    { strcpy(partcode,"83"); }
  else if (strcmp(format,"minix")==0)
    { strcpy(partcode,"81"); }
  else if (strcmp(format,"raid")==0)
    { strcpy(partcode,"fd"); } /* raid autodetect */
  else if (strcmp(format,"lvm")==0)
    { strcpy(partcode,"8e"); } /* LVM physical device */
  else if (format[0]=='\0')
    { partcode[0]='\0'; }
  else if (strlen(format)>=1 && strlen(format)<=2)
    { strcpy(partcode,format); } /* probably an image */
  else
    {
      sprintf(tmp,"Unknown format ('%s') - using code 83",format);
      mvaddstr_and_log_it(g_currentY++,0,tmp);
      strcpy(partcode,"83");
    }
  sprintf(tmp,"Setting %s's type to %s (%s)",device,format,partcode);
  log_it(tmp);
  if (partcode[0]!='\0' && strcmp(partcode,"83")) /* no need to set type if 83: 83 is default */
    {
      sprintf(output,"t\n%d\n%s\nw\n",partno,partcode); 
      sprintf(command,"fdisk %s > %s 2> %s.err",drive,logfile,logfile);
      fout=popen(command,"w");
      if (!fout)
        {
	  res=1; 
	  sprintf(tmp,"cat %s %s.err >> %s",logfile,logfile,"/tmp/mondo-restore.log");
	  system(tmp);
	}
      else
        {
	  res=0;
          fprintf(fout,output);
          pclose(fout);
        }
    }
  return(res);
}



int stop_raid_device(char *raid_device)
{
  int res,retval=0;
  char program[MAX_STR_LEN];
  sprintf(program,"raidstop %s",raid_device);
  res=run_program_and_log_output(program);
  retval+=res;
  return(retval);
}



int stop_all_raid_devices(struct mountlist_itself *mountlist)
{
  int retval=0, res;
  char incoming[MAX_STR_LEN], dev[MAX_STR_LEN], *p;
  FILE *fin;
  fin=fopen("/proc/mdstat","r");
  if (!fin) { return (1); }
  for(fgets(incoming,MAX_STR_LEN-1,fin); !feof(fin); fgets(incoming,MAX_STR_LEN-1,fin))
    {
      for(p=incoming;*p!='\0'&&(*p!='m'||*(p+1)!='d'||!isdigit(*(p+2))); p++);
      if (*p!='\0')
	{
	  sprintf(dev,"/dev/%s",p);
	  for(p=dev;*p>32;p++);
	  *p='\0';
	  res=stop_raid_device(dev);
	}
    }
  fclose(fin);
  return(retval);
}




int which_format_command_do_i_need(char*format,char*program)
{
  int res=0;
  char tmp[MAX_STR_LEN];
  if (strcmp(format,"swap")==0)      
    { strcpy(program,"mkswap"); }
  else if (strcmp(format,"vfat")==0) 
    { 
	  strcpy(program,"format-and-kludge-vfat"); 
    }
  else if (strcmp(format,"reiserfs")==0) 
    { strcpy(program,"mkreiserfs -ff"); }
  else if (strcmp(format,"xfs")==0)
    { strcpy(program,"mkfs.xfs -f -q"); }
  else if (strcmp(format,"jfs")==0)
    { strcpy(program,"mkfs.jfs"); }
  else if (strcmp(format,"ext3")==0)
    { strcpy(program,"mkfs -t ext2 -F -j -q"); }
  else if (strcmp(format,"ext2")==0)
    { strcpy(program,"mkfs -t ext2 -F -q"); }
  else if (strcmp(format,"minix")==0)
    { strcpy(program,"mkfs.minix"); }
  else
    {
      sprintf(program,"mkfs -t %s",format);
      sprintf(tmp,"Unknown format (%s) - assuming '%s' will do",format,program);
      log_it(tmp);
      res=0;
    }
  return(res);
}






long calc_orig_size_of_drive_from_mountlist(struct mountlist_itself *mountlist, char*drive_name)
{
  long original_size_of_drive;
  int partno;
  char tmp[MAX_STR_LEN];

  for(original_size_of_drive=0,partno=0; partno<mountlist->entries; partno++)
    {
      if (strncmp(mountlist->el[partno].device, drive_name, strlen(drive_name))==0)
        {
          original_size_of_drive += mountlist->el[partno].size/1024;
        }
      else
        {
          sprintf(tmp,"Skipping %s", mountlist->el[partno].device);
//          log_to_screen(tmp);
        }
    }
  return(original_size_of_drive);
}



void resize_drive_proportionately_to_suit_new_drives(struct mountlist_itself *mountlist, char*drive_name)
{
  char tmp[MAX_STR_LEN];
  int noof_drives, driveno, partno;
  float factor, new_size;
  long newsizL, current_size_of_drive, original_size_of_drive, final_size; /* in Megabytes */

  if (strncmp(drive_name, "/dev/md", 6)==0) { return; }
  sprintf(tmp,"cp -f %s %s.pre-resize", MOUNTLIST_FNAME, MOUNTLIST_FNAME);
  run_program_and_log_output(tmp);
  current_size_of_drive = get_phys_size_of_drive(drive_name);
  sprintf(tmp, "Expanding entries to suit drive %s (%ld MB)", drive_name, current_size_of_drive);
  log_to_screen(tmp);
  original_size_of_drive = calc_orig_size_of_drive_from_mountlist(mountlist, drive_name);
  if (original_size_of_drive<=0)
    {
      sprintf(tmp,"Cannot resize %s's entries. Drive not found.", drive_name);
      log_to_screen(tmp);
      return;
    }
  factor = (float)(current_size_of_drive) / (float)(original_size_of_drive);
  sprintf(tmp,"Disk was %ld MB; is now %ld MB; factor = %f", original_size_of_drive, current_size_of_drive, factor);
  log_to_screen(tmp);
  for(partno=0; partno<mountlist->entries; partno++)
    {
/* the 'atoi' thing is to make sure we don't try to resize _images_, whose formats will be numeric */
      if (atoi(mountlist->el[partno].format)<=0 && strncmp(mountlist->el[partno].device, drive_name, strlen(drive_name))==0)
        {
          new_size = (float)(mountlist->el[partno].size) * factor;
          newsizL = (long) new_size;
          sprintf(tmp,"Changing %s from %ld KB to %ld KB", mountlist->el[partno].device, mountlist->el[partno].size, newsizL);
          log_to_screen(tmp);
          mountlist->el[partno].size = newsizL;
        }
    }
  final_size = get_phys_size_of_drive(drive_name);
  sprintf(tmp,"final_size = %ld MB", final_size);
  log_to_screen(tmp);
}






void resize_mountlist_proportionately_to_suit_new_drives(struct mountlist_itself *mountlist)
{
  char drivelist[ARBITRARY_MAXIMUM][MAX_STR_LEN], tmp[MAX_STR_LEN];
  int noof_drives, driveno;

  noof_drives = make_list_of_drives(mountlist, drivelist);
  for(driveno=0; driveno<noof_drives; driveno++)
    {
      resize_drive_proportionately_to_suit_new_drives(mountlist, drivelist[driveno]);
    }
  log_to_screen("Mountlist adjusted to suit current hard drive(s)");
}












