# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Compare different peaklists with main.

# load modules
from nucleus import commfce


class mCompCount:
    """ Compare imported peaklists with main. """

    # ----
    def __init__(self):
        self.ctrlData = {}
    # ----


    # ----
    def matchDataToPeaklist(self, mainPeaklist, loadedPeaklists):
        """ Compare loaded peaklists with the main one. """

        # delete old match
        for x, peaklist in enumerate(loadedPeaklists):
            for y, peak in enumerate(peaklist[0]):
                loadedPeaklists[x][0][y][2] = ''

        # walk in main peaklist
        match = False
        for mainPeak in range(len(mainPeaklist)):
            mainMass = mainPeaklist[mainPeak][0]

            # select peaklist
            for x, peaklist in enumerate(loadedPeaklists):

                # walk in selected peaklist
                for y, peak in enumerate(peaklist[0]):

                    # count absolute tolerance from peak mass and error
                    massTolerance = commfce.countTolerance(peak[0], self.ctrlData['tolerance'], self.ctrlData['errortype'])

                    # if in mass range
                    if (peak[0] - massTolerance) <= mainMass <= (peak[0] + massTolerance):
                        loadedPeaklists[x][0][y][2] += str(mainPeak)+';'
                        match = True

        return loadedPeaklists, match
    # ----


    # ----
    def getMatchInfo(self, mainPeaklist, loadedPeaklists, errorType, digits):
        """ Get match-info for each peak in main peaklist. """

        #init params
        data={}
        data['params'] = []
        data['errors'] = []
        data['hidden'] = {}

        matchedPeaks = []
        errorList = []

        # select peaklist
        for peaklist in loadedPeaklists:

            # walk in selected peaklist
            for peak in peaklist[0]:

                # matched peak
                if peak[2] != '':

                    # get error list
                    peaks = peak[2].split(';')
                    for peakIndex in peaks:
                        if peakIndex != '':

                            # get peak
                            if peakIndex not in matchedPeaks:
                                matchedPeaks.append(peakIndex)

                            # get masses
                            peakIndex = int(peakIndex)
                            loadedMass = peak[0]
                            mainMass = mainPeaklist[peakIndex][0]

                            # get error
                            errorList.append([mainMass, mainMass-loadedMass])

        # append data
        data['params'].append(['Tolerance: ', str(self.ctrlData['tolerance']) + ' ' + self.ctrlData['errortype']])
        data['params'].append(["Peaks in main peaklist: ", str(len(mainPeaklist))])
        data['params'].append(["Matched peaks: ", str(len(matchedPeaks))])
        data['params'].append(["Missed peaks: ", str(len(mainPeaklist) - len(matchedPeaks))])
        data['params'].append(["Loaded peaklists: ", str(len(loadedPeaklists))])
        data['errors'] = errorList
        data['hidden']['errortype'] = errorType

        return data
    # ----
