/* Copyright (C) 2000, Luca Padovani <luca.padovani@cs.unibo.it>.
 *
 * This file is part of mlminidom, the Ocaml binding for minidom.
 * 
 * mlminidom is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * mlminidom is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with mlminidom; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * For details, send a mail to the author.
 */

#include <assert.h>
#include <mlvalues.h>
#include <memory.h>
#include <custom.h>

#include "minidom.h"

#define Val_ptr(p)        ((value) (p))
#define Val_option(p,f)   ((p != NULL) ? ml_some(f(p)) : Val_unit)
#define mDOMString_val(v) ((mDOMStringRef) String_val(v))
#define mDOMDocRef_val(r) (*((mDOMDocRef *)Data_custom_val(r)))

static value
Val_mDOMConstString(mDOMConstStringRef s)
{
  return copy_string((char *) s);
}

static value
Val_mDOMString(mDOMStringRef s)
{
  value r = copy_string((char *) s);
  mdom_string_free(s);
  return r;
}

static value
ml_some(value v)
{
  CAMLparam1(v);
  value ret = alloc_small(1,0);
  Field(ret,0) = v;
  CAMLreturn(ret);
}

value
ml_string_of_mDOMString(value s)
{
  CAMLparam1(s);
  CAMLreturn(s);
}

value
ml_mDOMString_of_string(value s)
{
  CAMLparam1(s);
  CAMLreturn(s);
}

static void
ml_doc_free(value doc)
{
	mdom_doc_free(mDOMDocRef_val(doc));
}

static struct custom_operations ops =
  {"it.unibo.cs.helm.gtkmathview.mDOMDocRef",
   ml_doc_free,
   custom_compare_default,
   custom_hash_default,
   custom_serialize_default,
   custom_deserialize_default
  };

value
ml_doc_load(value file_name)
{
  mDOMDocRef doc_ref;

  CAMLparam1(file_name);
  CAMLlocal1(val_doc_ref);

  doc_ref = mdom_load(String_val(file_name), FALSE, NULL);
  if (doc_ref == NULL) failwith("minidom: could not load document");
  val_doc_ref = alloc_custom(&ops, sizeof(mDOMDocRef), 1, 1);

  *((mDOMDocRef *)Data_custom_val(val_doc_ref)) = doc_ref;

  CAMLreturn(val_doc_ref);
}

value
ml_doc_new(value s)
{
  mDOMDocRef doc_ref;

  CAMLparam1(s);
  CAMLlocal1(val_doc_ref);

  doc_ref = mdom_doc_new(mDOMString_val(s));
  if (doc_ref == NULL) failwith("minidom: could not create new document");
  val_doc_ref = alloc_custom(&ops, sizeof(mDOMDocRef), 1, 1);

  *((mDOMDocRef *)Data_custom_val(val_doc_ref)) = doc_ref;

  CAMLreturn(val_doc_ref);
}


value
ml_doc_get_root_node(value doc)
{
  mDOMNodeRef root;

  CAMLparam1(doc);
  root = mdom_doc_get_root_node(mDOMDocRef_val(doc));
  if (root == NULL) failwith("minidom: document has no root node!");

  CAMLreturn((value) root);
}

value
ml_doc_add_entity(value doc, value name, value content)
{
  mDOMEntityRef ent;

  CAMLparam3(doc, name, content);
  ent = mdom_doc_add_entity(mDOMDocRef_val(doc), mDOMString_val(name), mDOMString_val(content));
  if (ent == NULL) failwith("minidom: could not add entity");

  CAMLreturn((value) ent);
}

value
ml_doc_get_entity(value doc, value name)
{
  mDOMEntityRef ent;

  CAMLparam2(doc, name);
  ent = mdom_doc_get_entity(mDOMDocRef_val(doc), mDOMString_val(name));

  CAMLreturn(Val_option(ent, Val_ptr));
}

value
ml_doc_get_predefined_entity(value name)
{
  mDOMEntityRef ent;

  CAMLparam1(name);
  ent = mdom_get_predefined_entity(mDOMString_val(name));

  CAMLreturn(Val_option(ent, Val_ptr));
}

value
ml_entity_get_content(value ent)
{
  CAMLparam1(ent);
  CAMLreturn(Val_mDOMConstString(mdom_entity_get_content((mDOMEntityRef) ent)));
}

value
ml_node_is_text(value node)
{
  CAMLparam1(node);
  CAMLreturn(Val_bool(mdom_node_is_text((mDOMNodeRef) node)));
}

value
ml_node_is_element(value node)
{
  CAMLparam1(node);
  CAMLreturn(Val_bool(mdom_node_is_element((mDOMNodeRef) node)));
}

value
ml_node_is_blank(value node)
{
  CAMLparam1(node);
  CAMLreturn(Val_bool(mdom_node_is_blank((mDOMNodeRef) node)));
}

value
ml_node_is_entity_ref(value node)
{
  CAMLparam1(node);
  CAMLreturn(Val_bool(mdom_node_is_entity_ref((mDOMNodeRef) node)));
}

value
ml_node_get_type(value node)
{
  CAMLparam1(node);
  CAMLreturn(Val_int(mdom_node_get_type((mDOMNodeRef) node)));
}

value
ml_node_get_name(value node)
{
  CAMLparam1(node);
  CAMLreturn(Val_option(mdom_node_get_name((mDOMNodeRef) node), Val_mDOMConstString));
}

value
ml_node_has_attribute(value node, value name)
{
  CAMLparam2(node,name);
  CAMLreturn(Val_bool(mdom_node_has_attribute((mDOMNodeRef) node, String_val(name))));
}

value
ml_node_has_attribute_ns(value node, value name, value uri)
{
  CAMLparam3(node,name,uri);
  CAMLreturn(Val_bool(mdom_node_has_attribute_ns((mDOMNodeRef) node, String_val(name), String_val(uri))));
}

value
ml_node_get_content(value node)
{
  CAMLparam1(node);
  CAMLreturn(Val_option(mdom_node_get_content((mDOMNodeRef) node), Val_mDOMString));
}

value
ml_node_get_ns_uri(value node)
{
  CAMLparam1(node);
  CAMLreturn(Val_option(mdom_node_get_ns_uri((mDOMNodeRef) node), Val_mDOMConstString));
}

value
ml_node_get_attribute(value node, value name)
{
  CAMLparam2(node,name);
  CAMLreturn(Val_option(mdom_node_get_attribute((mDOMNodeRef) node, String_val(name)), Val_mDOMString));
}

value
ml_node_get_attribute_ns(value node, value name, value ns_uri)
{
  CAMLparam2(node,name);
  CAMLreturn(Val_option(mdom_node_get_attribute_ns((mDOMNodeRef) node,
						   String_val(name),
						   String_val(ns_uri)), Val_mDOMString));
}

value
ml_node_get_parent(value node)
{
  CAMLparam1(node);
  CAMLreturn(Val_option(mdom_node_get_parent((mDOMNodeRef) node), Val_ptr));
}

value
ml_node_get_prev_sibling(value node)
{
  CAMLparam1(node);
  CAMLreturn(Val_option(mdom_node_get_prev_sibling((mDOMNodeRef) node), Val_ptr));
}

value
ml_node_get_next_sibling(value node)
{
  CAMLparam1(node);
  CAMLreturn(Val_option(mdom_node_get_next_sibling((mDOMNodeRef) node), Val_ptr));
}

value
ml_node_get_first_child(value node)
{
  CAMLparam1(node);
  CAMLreturn(Val_option(mdom_node_get_first_child((mDOMNodeRef) node), Val_ptr));
}

value
ml_node_get_first_attribute(value node)
{
  CAMLparam1(node);
  CAMLreturn(Val_option(mdom_node_get_first_attribute((mDOMNodeRef) node), Val_ptr));
}

value
ml_node_is_first(value node)
{
  CAMLparam1(node);
  CAMLreturn(Val_bool(mdom_node_is_first((mDOMNodeRef) node)));
}

value
ml_node_is_last(value node)
{
  CAMLparam1(node);
  CAMLreturn(Val_bool(mdom_node_is_last((mDOMNodeRef) node)));
}

value
ml_attr_get_name(value attr)
{
  CAMLparam1(attr);
  CAMLreturn(Val_option(mdom_attr_get_name((mDOMAttrRef) attr), Val_mDOMConstString));
}

value
ml_attr_get_ns_uri(value attr)
{
  CAMLparam1(attr);
  CAMLreturn(Val_option(mdom_attr_get_ns_uri((mDOMAttrRef) attr), Val_mDOMConstString));
}

value
ml_attr_get_value(value attr)
{
  CAMLparam1(attr);
  CAMLreturn(Val_option(mdom_attr_get_value((mDOMAttrRef) attr), Val_mDOMString));
}

value
ml_attr_get_prev_sibling(value attr)
{
  CAMLparam1(attr);
  CAMLreturn(Val_option(mdom_attr_get_prev_sibling((mDOMAttrRef) attr), Val_ptr));
}

value
ml_attr_get_next_sibling(value attr)
{
  CAMLparam1(attr);
  CAMLreturn(Val_option(mdom_attr_get_next_sibling((mDOMAttrRef) attr), Val_ptr));
}

value
ml_attr_get_parent(value attr)
{
  CAMLparam1(attr);
  CAMLreturn(Val_option(mdom_attr_get_parent((mDOMAttrRef) attr), Val_ptr));
}

