/*
 * Written by Bastien Chevreux (BaCh)
 * Copyright (C) 2007 by Bastien Chevreux
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 *
 */

#ifndef _bas_skim_h_
#define _bas_skim_h_

#include <boost/thread/mutex.hpp>
#include <boost/thread/condition.hpp>
#include <boost/regex.hpp> 

#include "stdinc/defines.H"
#include "stdinc/stlincludes.H"

#include "util/progressindic.H"
#include "mira/readpool.H"
#include "ads.H"


//typedef unsigned long vhash_t;
typedef uint64 vhash_t;

struct vhrap_t {
  vhash_t vhash;       // vhash,
  uint32 readid;       // readid
  uint16 hashpos;      // (hash)position  (lowest pos: basesperhash-1)

  Read::bhashstat_t bhashstats; // baseflags for this hash

  //bool operator<(const vhrap_t & other) const {return vhash < other.vhash;};
  friend ostream & operator<<(ostream &ostr, const vhrap_t & v){
    ostr << "hash: " << hex << v.vhash
	 << "\trid: " << dec << v.readid
	 << "\thpos: " << v.hashpos
	 << "\tbhs: " << v.bhashstats;
    return ostr;
  }
};


struct diskhash_t {
  vhash_t vhash;       // vhash,
  uint16  hashpos;     // hash position (lowest pos: 0)
  int8    dir;         // direction
  uint8   seqtype;     // sequencing type

  //bool operator<(const vhrap_t & other) const {return vhash < other.vhash;};
  friend ostream & operator<<(ostream &ostr, const diskhash_t & v){
    ostr << "hash: " << hex << v.vhash
	 << "\thpos: " << v.hashpos
	 << "\tdir: " << dec << static_cast<int16>(v.dir)
	 << "\tseqtype: " << static_cast<uint16>(v.seqtype);
    return ostr;
  }
};


// now almost same as hashstat? takes up 16 bytes anyway (like diskhash)
struct newdiskhash_t {
  vhash_t vhash;       // vhash,
  uint32  count;       // how often this hash appeared
  uint16  lowpos;      // lowest hash position (lowest pos: 0)
  uint8   seqtype;     // sequencing type

  bool    hasfwd:1;
  bool    hasrev:1;
  bool    hasmultipleseqtype:1;

  //bool operator<(const vhrap_t & other) const {return vhash < other.vhash;};
  friend ostream & operator<<(ostream &ostr, const newdiskhash_t & v){
    ostr << "hash: " << hex << v.vhash
	 << "\tcount: " << v.count
	 << "\tlpos: " << v.lowpos
	 << "\tseqtype: " << static_cast<uint16>(v.seqtype)
	 << "\tfwd: " << v.hasfwd
	 << "\trev: " << v.hasrev
	 << "\tmseq: " << v.hasmultipleseqtype;
    return ostr;
  }
};



struct hashstat_t {
  vhash_t vhash;       // vhash,
  uint32  count;       // how often this hash appeared
  uint16  lowpos;      // lowest hash position seen (lowest pos: 0)
  bool    hasfwdrev:1;            // whether it's there in forward and reverse
  bool    hasmultipleseqtype:1;   // found in multiple sequencing types?

  friend ostream & operator<<(ostream &ostr, const hashstat_t & v){
    ostr << "hash: " << hex << v.vhash
	 << "\tcount: " << dec << v.count
	 << "\tlowpos: " << v.lowpos
	 << "\tfwdrev: " << v.hasfwdrev
	 << "\tmseqtype: " << v.hasmultipleseqtype;
    return ostr;
  }
};

struct matchwithsorter_t{
  uint32 otherid;
  int32  eoffset;
  int32  percent_in_overlap;
  uint32 numhashes;                 // try to deprecate?

  uint32 estimscore;

  bool   taken;
  
  bool ol_stronggood:1;  // frequency: 2*bph-1 pos at 3, thereof bph-1 contiguous
  bool ol_weakgood:1;   // frequency: bph-1 positions contiguous at 3
  bool ol_belowavgfreq:1;   // frequency: bph-1 positions contiguous at <=3
  bool ol_norept:1;      // nothing >3 (but can contain 1 (single hashes == errors)
  bool ol_rept:1;      // bph-1 positions >=5 frequency

  // not saved to disk but used internally
  bool ol_fulllength:1;   // overlap is full length of one of the reads
  bool ol_fullencased:1;  /* got the status of fullencased:
			     100% match, no rept, len difference >=8
			  */

  friend ostream & operator<<(ostream &ostr, const matchwithsorter_t & m){
    ostr << "MWS:\t" << m.otherid
	 << '\t' << m.eoffset
	 << '\t' << m.percent_in_overlap
	 << '\t' << m.numhashes
	 << '\t' << m.estimscore
	 << "\ttak " << m.taken
	 << "\tsg  " << m.ol_stronggood
	 << "\twg  " << m.ol_weakgood
	 << "\tbaf " << m.ol_belowavgfreq
	 << "\tnrp " << m.ol_norept
	 << "\trep " << m.ol_rept
	 << "\tfll " << m.ol_fulllength
	 << "\tfle " << m.ol_fullencased
	 << '\n';
    return ostr;
  }
};


// only similar to matchwithsorter_t
struct skimhitforsave_t{
  uint32 rid1;
  uint32 rid2;
  int32  eoffset;
  int32  percent_in_overlap;
  uint32 numhashes;
  
  bool ol_stronggood:1;  // frequency: 2*bph-1 pos at 3, thereof bph-1 contiguous
  bool ol_weakgood:1;   // frequency: bph-1 positions contiguous at 3
  bool ol_belowavgfreq:1;   // frequency: bph-1 positions contiguous at <=3
  bool ol_norept:1;      // nothing >3 (but can contain 1 (single hashes == errors)
  bool ol_rept:1;      // bph-1 positions >=5 frequency

  friend ostream & operator<<(ostream &ostr, const skimhitforsave_t & e){
    ostr << "SHFS:\t" << e.rid1
	 << '\t' << e.rid2
	 << '\t' << e.eoffset
	 << '\t' << e.percent_in_overlap
	 << '\t' << e.numhashes
	 << "\tsg " << e.ol_stronggood
	 << "\twg " << e.ol_weakgood
	 << "\tbaf " << e.ol_belowavgfreq
	 << "\trp" << e.ol_norept
	 << "\tnrp" << e.ol_rept
	 << '\n';
    return ostr;
  }

};



struct matchwith_t{
  uint32 otherid;
  int32  eoffset;
};


struct readhashmatch_t{
  uint32 rid2;
  int32  eoffset;
  uint16 hashpos1;
  uint16 hashpos2;
  Read::bhashstat_t bhashstats; // baseflags for this hash

  friend ostream & operator<<(ostream &ostr, const readhashmatch_t & rhm){
    ostr << "rid2: " << rhm.rid2
	 << "\teoffset: " << rhm.eoffset
	 << "\thp1: " << rhm.hashpos1
	 << "\thp2: " << rhm.hashpos2
	 << "\tbhs: " << rhm.bhashstats
	 << '\n';
    return ostr;
  }

};

typedef multimap< int32, matchwith_t> possible_overlaps_t;
typedef possible_overlaps_t::value_type posoverlap_pair_t;


// bop_t is organised as follows:
//  each ban of a pair of reads is stored in the vector of the read with the lower ID
// bop_t was initially vector<set<uint32> >, but these beasts are extremely expensive,
//  both in time and in memory (48 per set + 48 per element in set!)
// using a vector is way quicker (around 3 times), uses way less memory (24 + 4 per element)
//  and the growth strategy ensures that not too much memory is wasted (has a small hit on
//  run-time efficiency though as we might need to copy quite a couple of times)

struct bannedoverlappairs_t {
  typedef vector<vector<uint32> > bop_t;
  bop_t          bop;

  inline void nuke() {nukeSTLContainer(bop);};
  inline size_t size() {return bop.size();};
  inline void resize(size_t sz) {bop.resize(sz);};
  inline void insertBan(uint32 id1, uint32 id2) {
    if(id1>id2) swap(id1,id2);
    vector<uint32>::iterator iI=upper_bound(bop[id1].begin(),bop[id1].end(),id2);
    if(iI==bop[id1].end() || iI==bop[id1].begin() || (iI != bop[id1].begin() && (*(iI-1) != id2))){
      // must insert
      // moderate growth strategy if needed ... most reads will not have thousands of bans
      if(bop[id1].capacity()>=128
	 && bop[id1].capacity() == bop[id1].size()){
	// if we take over memory management ourselves, need to be careful:
	//  reserve() will invalidate our iI pointer (insert() handles that
	//  automatically)
	// So, just make sure we keep a valid pointer
	size_t distance=iI-bop[id1].begin();
	bop[id1].reserve(bop[id1].size()+128);
	iI=bop[id1].begin()+distance;
      }
      bop[id1].insert(iI,id2);
    }
  }
  inline bool checkIfBanned(uint32 id1, uint32 id2) {
    if(id1>id2) swap(id1,id2);
    return binary_search(bop[id1].begin(),bop[id1].end(),id2);
  }
  void getNumBans(size_t & banned, size_t & numsets) {
    banned=0;
    numsets=0;
    for(size_t i=0; i<bop.size(); i++){
      if(!bop[i].empty()){
	banned+=bop[i].size();
	++numsets;
      }
    }
  }
};

typedef map< uint32, uint32> simplematches_t;
typedef simplematches_t::value_type simplematch_pair_t;

class Skim
{
private:
  bool SKIM3_logflag_purgeunnecessaryhits;
  bool SKIM3_logflag_save2;

  ReadPool * SKIM3_readpool;

  vector<vhrap_t> SKIM3_vhraparray;
  vector<vector<vhrap_t>::const_iterator > SKIM3_vashortcuts_begin;
  vector<vector<vhrap_t>::const_iterator > SKIM3_vashortcuts_end;

  // the "NULL" replacement for the shortcut arrays above
  vector<vhrap_t>::const_iterator SKIM3_completevhraparray_end;

  vector<uint8> SKIM3_megahubs;

  // TODO: eventuall compress this with megahubs into uint8 having 8 boolean 
  vector<uint8> SKIM3_hasMNRr;
  vector<uint8> SKIM3_hasSRMr;
  vector<uint8> SKIM3_hasFpAS;

  // counts how many times a read has excellent hits with other reads
  vector<uint32> * SKIM3_overlapcounter;

  // count how many times a read was saved as fully encased
  vector<uint8> SKIM3_fullencasedcounter;

  vector<uint32> * SKIM3_writtenhitsperid;

  bannedoverlappairs_t * SKIM3_bannedoverlaps;
  
  ofstream SKIM3_posfmatchfout;
  ofstream SKIM3_poscmatchfout;
  string SKIM3_posfmatchfname;
  string SKIM3_poscmatchfname;
  uint64 SKIM3_posfmatchnextchecksize;
  uint64 SKIM3_poscmatchnextchecksize;

  uint32 SKIM3_possiblehits;
  uint32 SKIM3_acceptedhits;

  uint32 SKIM3_numthreads;
  uint8  SKIM3_basesperhash;
  uint8  SKIM3_hashsavestepping;

  //int32  SKIM3_percentrequired;

  vector<int32>  SKIM3_percentrequired;
  vector<int32>  SKIM3_overlaplenrequired; // min len per sequencing type

  // if vector.size() != 0
  // skimGo() also hunts chimeras
  // outer vector size of readpool
  // inner vectors size of length of clipped sequences
  //  0 initially, set to 1 for consecutive hash matches
  //  chimeras show up as a 0 stretch within the 1s
  vector<vector<uint8> > SKIM3_chimerahunt;
  vector<int32> * SKIM3_chuntleftcut;
  vector<int32> * SKIM3_chuntrightcut;

  // overlap criterion levels for left and right extend
  // used to take overlaps only if they reach a given level
  vector<uint8> * SKIM3_overlapcritlevell;
  vector<uint8> * SKIM3_overlapcritlevelr;

  vector<uint32> SKIM3_largestencasementscoretodate;


  uint32 SKIM3_maxhitsperread;


  uint64 SKIM3_totalhitschosen;
  uint64 SKIM3_totalpermbans;


  uint32 SKIM_partfirstreadid;
  uint32 SKIM_partlastreadid;

  ProgressIndicator<int64> * SKIM_progressindicator;
  int64 SKIM_progressend;


  bool SKIM3_onlyagainstrails;

  double SKIM3_freqest_minnormal;
  double SKIM3_freqest_maxnormal;
  double SKIM3_freqest_repeat;
  double SKIM3_freqest_heavyrepeat;
  double SKIM3_freqest_crazyrepeat;
  uint32 SKIM3_nastyrepeatratio;


  // for provideHashStatistics()
  vector<hashstat_t> SKIM3_hs_hashstats;
  vector<vector<hashstat_t>::const_iterator > SKIM3_hs_hsshortcuts_begin;
  vector<vector<hashstat_t>::const_iterator > SKIM3_hs_hsshortcuts_end;
  uint8 SKIM3_hs_basesperhash;

  vector<vhrap_t> SKIM3_baiting_singlereadvhraparray;
  vector<uint8>   SKIM3_baiting_tagmaskvector;

  // each findAdaptorRightClip thread needs a couple of data structures ...
  struct farc_threaddata_t {
    vector<readhashmatch_t> readhashmatches;
    vector<vhrap_t> singlereadvhraparray;
    vector<uint8> tagmaskvector;
    vector<matchwithsorter_t> tmpmatchwith;
    list<boost::regex> addregexes;
    boost::match_results<std::string::const_iterator> rematches;
  };

  farc_threaddata_t  SKIM3_farcdata_fornonmultithread;
  ReadPool         * SKIM3_farc_searchpool;
  vector<int32>    * SKIM3_farc_results;
  uint32             SKIM3_farc_minhashes;
  uint8              SKIM3_farc_seqtype;
//  list<boost::regex> SKIM3_farc_addregexes_templates;
  string             SKIM3_farc_addregexes_file;

  vector<farc_threaddata_t> SKIM3_farcd_vector;

//  vector<readhashmatch_t> SKIM3_streamskim_readhashmatches;
//  vector<matchwithsorter_t> SKIM3_streamskim_tmpmatchwith;

  // each checkForHashes thread needs a couple of data structures ...
  struct cfh_threaddata_t {
    vector<readhashmatch_t> readhashmatches;
    vector<uint32> smallhist4repeats;
    vector<vhrap_t> singlereadvhraparray;
    vector<matchwithsorter_t> tmpmatchwith;
    vector<uint8> tagmaskvector;
    vector<skimhitforsave_t> shfsv;
    ofstream * posmatchfout;
    // this vector is used to collect read ids which have a match
    //  and then quickly update SKIM3_writtenhitsperid inside a mutex
    vector<uint32> ridswithmatches;
    vector<uint32> uidswithnewcritlevell;
    vector<uint32> uidswithnewcritlevelr;
    vector<uint8> critlevellofnewuids;
    vector<uint8> critlevelrofnewuids;
  };

  vector<cfh_threaddata_t> SKIM3_cfhd_vector;


  struct threadworkercontrol_t {
    uint32 from;
    uint32 to;
    int8   direction;
    bool   flag_datavalid;
    bool   flag_endthread;
  };

  vector<threadworkercontrol_t> SKIM3_threadcontrol;

  boost::mutex SKIM3_coutmutex;
  boost::mutex SKIM3_resultfileoutmutex;
  boost::mutex SKIM3_globalclassdatamutex;
  boost::mutex SKIM3_whpid_mutex;

  boost::mutex SKIM3_critlevelwrite_mutex;


  //uint32 SKIM3_whpid_counter;

  boost::mutex SKIM3_mutex;
  boost::condition SKIM3_master2slavesignal;
  boost::condition SKIM3_slave2mastersignal;

public:


  //Functions
private:
  void foolCompiler();

//  void prepareSkim(bool alsocheckreverse);
  void prepareSkim(uint32 fromid, uint32 toid, vector<vhrap_t> & vhraparray, bool assemblychecks);
  void purgeMatchFileIfNeeded(int8 direction);
  void purgeUnnecessaryHitsFromSkimFile(string & filename, const int8 rid2dir);

  uint32 transformSeqToVariableHash(
    const uint32 readid, 
    const Read & actread,
    const char * seq, 
    uint32 slen,
    const uint8 basesperhash,
    vector<vhrap_t>::iterator & vhraparrayI,
    const bool countonly,
    const uint8 hashsavestepping,
    vector<uint8> & maskvector,
    const vector<Read::bposhashstat_t> & bposhashstats, 
    int32 bfpos, 
    const int32 bfposinc
    );
  
  void fillTagMaskVector(const uint32 readid, vector<uint8> & tagmaskvector);
  void fillTagStatusInfoOfReads();

  void reverseTagMaskVector(vector<uint8> & tagmaskvector);


  void startMultiThreading(const int8 direction, 
			   const uint32 numthreads, 
			   const uint32 readsperthread, 
			   const uint32 firstid,
			   const uint32 lastid,
			   boost::function<void(uint32_t)> initfunc,
			   boost::function<void(uint32_t)> callfunc);
  void cfhThreadsDataInit(const uint32 numthreads);
  void cfhThreadLoop(const uint32 threadnr);
  void checkForHashes_fromto(const int8 direction,
			     const uint32 fromid,
			     const uint32 toid,
			     cfh_threaddata_t & cfhd);
  void checkForPotentialHits(const int8 direction,
			     const uint32 actreadid, 
			     vector<matchwithsorter_t> & tmpmatchwith,
			     vector<readhashmatch_t> & readhashmatches,
			     vector<uint32> & smallhist4repeats);

  void farcThreadsDataInit(const uint32 threadnr);
  void farcThreadLoop(const uint32 threadnr);
  void checkForPotentialAdaptorHits(const int8 direction,
				    const uint32 actreadid,
				    Read & actread,
				    vector<matchwithsorter_t> & tmpmatchwith,
				    vector<readhashmatch_t> & readhashmatches);
  
  void selectPotentialHitsForSave2(const int8 direction,
				   const uint32 actreadid,
				   cfh_threaddata_t & cfhd);
  void updateCriterionLevels(const int8 direction,
			     const uint32 actreadid,
			     cfh_threaddata_t & cfhd);

  void chimeraHuntStoreOverlapCoverage(
    const int8 direction, 
    const uint32 actreadid, 
    const uint32 rid2, 
    uint16 hp1min,
    uint16 hp1max,
    uint16 hp2min,
    uint16 hp2max);
  void chimeraHuntLocateChimeras();

  void makeVHRAPArrayShortcuts(vector<vhrap_t> & vhraparray,
			       const uint8 basesperhash);

//////////////////////////////////

  void init();
  
  uint32 computePartition(uint32 maxmemusage, 
			  bool computenumpartitions);

  void sFR_makeHashCounts(vector<uint32> & hashcounter, uint32 basesperhash);

////////////////////////////////////////

  void hashes2disk(vector<string> & hashfilenames,
		   vector<size_t> & elementsperfile,
		   ReadPool & rp, 
		   bool checkusedinassembly,
		   bool onlyagainstrails,
		   bool fwdandrev,
		   uint8  basesperhash,
		   uint8  hashsavestepping,
		   const string & directory);
  size_t createHashStatisticsFile(string & hashstatfilename, 
				  vector<string> & hashfilenames, 
				  vector<size_t> & elementsperfile, 
				  uint8 basesperhash,
				  uint32 fwdrevmin,
				  ReadPool & rp, 
				  bool onlyagainstrails, 
				  bool alsosavesinglehashes, 
				  const string & directory);
  size_t loadHashStatisticsFile(string & hashstatfilename, 
				vector<hashstat_t> & hashstats);

  size_t prepareHashStatistics(const string & directory, 
			       ReadPool & rp, bool checkusedinassembly, bool onlyagainstrails, bool alsosavesinglehashes,
			       bool fwdandrev,
			       uint32 fwdrevmin,
			       uint8  basesperhash,
			       uint8  hashsavestepping,
			       string & hashstatfilename,
			       vector<hashstat_t> & hashstats,
			       vector<vector<hashstat_t>::const_iterator> & hsshortcuts_begin,
			       vector<vector<hashstat_t>::const_iterator> & hsshortcuts_end);

  size_t calcMidHashStatIndex(const vector<hashstat_t> & hashstats,
			      size_t dontcarepercent);

  void dumpHashStatisticsInfo(size_t avgcov,
			      vector<hashstat_t> & hashstats);

  void makeHashStatArrayShortcuts(
    vector<hashstat_t> & hashstats,
    const uint8 basesperhash, 
    vector<vector<hashstat_t>::const_iterator > & hsshortcuts_begin,
    vector<vector<hashstat_t>::const_iterator > & hsshortcuts_end);
  
  void assignReadBaseStatistics(
    ReadPool & rp, 
    size_t avgcov,
    vector<hashstat_t> & hashstats, 
    const uint8 basesperhash, 
    vector<vector<hashstat_t>::const_iterator > & hsshortcuts_begin,
    vector<vector<hashstat_t>::const_iterator > & hsshortcuts_end,
    bool masknastyrepeats);

  void correctReadBaseStatisticsByRMB(ReadPool & rp,
				      const uint8 basesperhash);

  int32 findAdaptorRightClip_internal(Read & actread, uint32 minhashes, farc_threaddata_t & farcd);


public:
  Skim();
  Skim(Skim const &other);
  ~Skim();

  Skim const & operator=(Skim const & other);
  friend ostream & operator<<(ostream &ostr, Skim const &theskim);

  void discard();


  void setHashFrequencyRatios(double freqest_minnormal,
			      double freqest_maxnormal,
			      double freqest_repeat,
			      double freqest_heavyrepeat,
			      double freqest_crazyrepeat,
			      uint32 nastyrepeatratio);

  uint32 skimGo (ReadPool & rp, 
		 string               & posfmatchname,
		 string               & poscmatchname,
		 string               & megahublogname,
		 bannedoverlappairs_t & bannedoverlaps,
		 vector<uint32>       & overlapcounter,
		 vector<uint32>       & writtenhitsperid,
		 vector<int32>        & chuntleftcut,
		 vector<int32>        & chuntrightcut,
		 vector<uint8>        & overlapcritlevell,
		 vector<uint8>        & overlapcritlevelr,
		 
		 uint32 numthreads,           //2
		 uint32 maxmemusage,          //  = 15000000
		 
		 bool onlyagainstrails,       // =false
		 bool alsocheckreverse,       // =true
		 
		 uint8  basesperhash,         // = 16
		 uint8  hashsavestepping,     // = 4
		 //int32  percentrequired,      // = 50
		 const vector<int32> & percentrequired,   // = 30 for each st
		 const vector<int32> & overlaplenrequired,   // = 30 for each st
		 uint32 maxhitsperread);      // = 200

  void skimStreamPrepare(ReadPool & rp, uint8  bph, uint8  hss, const char * additionalregexp=NULL);
  int32 findAdaptorRightClip(Read & actread, uint32 minhashes);
  void findAdaptorRightClip(ReadPool & searchpool, vector<int32> & results, int8 seqtype, uint32 minhashes, uint32 numthreads);


  uint32 maskNastyRepeats(ReadPool & rp, 
			  uint32 repeatthreshold,
			  ostream * logostr = NULL);

  void analyseHashes(const string & directory,
		     ReadPool & rp, 
		     bool checkusedinassembly,
		     bool onlyagainstrails,
		     bool alsosavesinglehashes,
		     bool fwdandrev,
		     uint32 fwdrevmin,
		     uint8  basesperhash,
		     uint8  hashsavestepping,
		     bool masknastyrepeats);

  void provideHashStatistics(const string & directory,
			     ReadPool & rp, 
			     bool checkusedinassembly,
			     bool onlyagainstrails,
			     bool alsosavesinglehashes,
			     bool fwdandrev,
			     uint32 fwdrevmin,
			     uint8  basesperhash,
			     uint8  hashsavestepping);
  void showHashStatisticsInfo(string & filename);

  uint32 checkBaitHit(Read & actread);

  static void getOverlapCriterionLevel(const uint32 actreadid,
				       const uint8 seqtype,
				       const ADSEstimator & adse,
				       const uint8 relscore,
				       uint8 & levell,
				       uint8 & levelr);

  void setExtendedLog(bool f) {
    SKIM3_logflag_purgeunnecessaryhits=f;
    SKIM3_logflag_save2=f;
  }

};


#endif

