;;; mew-scan.el --- Scanning messages for Mew

;; Author:  Kazu Yamamoto <Kazu@Mew.org>
;; Created: Oct  2, 1996

;;; Code:

(require 'mew)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Pre-defined functions for mew-scan-form
;;;

(defun mew-scan-form-num ()
  "A function to return a message number."
  (if VIRTUALP
      (prog1
	(int-to-string (mew-vinfo-get-count))
	(mew-vinfo-set-count (1+ (mew-vinfo-get-count))))
    (MEW-NUM)))

(defun mew-scan-form-mark ()
  "A function to return a mark.
'mew-scan-form-mark-delete' and 'mew-scan-form-mark-review'
effect to this function."
  (let ((id (MEW-ID)) duplicated review)
    (when mew-scan-form-mark-delete
      (when (string-match mew-regex-id id)
	(setq id (mew-match 1 id))
	(if (member id (mew-sinfo-get-scan-id)) ;; in Summary mode
	    (setq duplicated t)
	  (mew-sinfo-set-scan-id (cons id (mew-sinfo-get-scan-id))))))
    (when mew-scan-form-mark-review
      (let* ((mew-header-max-depth nil)
	     (to (mew-addrstr-parse-address-list (MEW-TO))))
	(catch 'loop
	  (while to
	    (if (mew-is-my-address mew-regex-my-address-list (car to))
		(throw 'loop (setq review t)))
	    (setq to (cdr to))))))
    (cond
     (duplicated (char-to-string mew-mark-delete))
     (review     (char-to-string mew-mark-review))
     (t " "))))

(defun mew-scan-form-type ()
  "A function to return a mark of content type."
  (let ((ct (MEW-CT))
	(uid (mew-scan-usc-uid (mew-scan-split-usc (MEW-UIDL))))
	(case-fold-search t))
    (cond
     ((not (string= uid ""))                       "T")
     ((string-match "Multipart/Signed"    ct)      "S")
     ((string-match "Multipart/Encrypted" ct)      "E")
     ((string-match "Application/X-Pkcs7-Mime" ct) "E")
     ((mew-ct-multipartp ct)                       "M")
     ((string-match "Message/Partial"     ct)      "P")
     (t " "))))

(defun mew-scan-form-time ()
  "A function to return a message time, HH:MM"
  (let ((s (MEW-DATE)))
    (if (or (string= s "")
	    (not (string-match mew-time-rfc-regex s)))
	"00:00"
      (format "%02d:%2s"
	      (or (mew-time-rfc-hour) 0)
	      (or (mew-time-rfc-min) "00")))))

(defun mew-scan-form-date ()
  "A function to return a date, MM/DD."
  (let ((s (MEW-DATE)))
    (when (or (string= s "")
	      (not (string-match mew-time-rfc-regex s)))
      (setq s (mew-time-ctz-to-rfc
	       (mew-file-get-time (mew-expand-folder (MEW-FLD) (MEW-NUM))))))
    (if (string-match mew-time-rfc-regex s)
	(format "%02d/%02d"
		(mew-time-mon-str-to-int (mew-time-rfc-mon))
		(mew-time-rfc-day))
      "")))

(defun mew-scan-form-year ()
  "A function to return a message year, YYYY"
  (let ((s (MEW-DATE)) year)
    (if (or (string= s "")
	    (not (string-match mew-time-rfc-regex s)))
	"0000"
      (setq year (mew-time-rfc-year))
      (cond
       ((< year 50)
	(setq year (+ year 2000)))
       ((< year 100)
	(setq year (+ year 1900))))
      (int-to-string year))))

(defvar mew-scan-form-size-unit '("" "k" "M" "G" "T"))

(defun mew-scan-form-size ()
  "A function to return the size of the message. Should be used
with -4. See also 'mew-scan-form-size-0k' and 'mew-scan-form-size-huge'."
  (let ((len-1 (1- (length mew-scan-form-size-unit)))
	(SIZE (mew-scan-usc-size (mew-scan-split-usc (MEW-UIDL))))
	(i 0) size unit)
    (if (string-match "^[0-9]+$" SIZE)
	(setq size (string-to-int SIZE))
      (setq size (mew-file-get-size (mew-expand-folder (MEW-FLD) (MEW-NUM)))))
    (while (and (< i len-1) (>= size 1000))
      (setq size (/ size 1000))
      (setq i (1+ i)))
    (if (and mew-scan-form-size-huge (>= size 1000))
	"HUGE"
      (setq unit (nth i mew-scan-form-size-unit))
      (if (and mew-scan-form-size-0k (string= unit ""))
	  "0k"
	(concat
	 (if (integerp size)
	     (int-to-string size)
	   (format "%.0f" size))
	 unit)))))

(defun mew-scan-form-extract-addr (addr)
  "Extract addr according to 'mew-scan-form-extract-rule'."
  (let* ((func (if mew-addrbook-for-summary
		   (mew-addrbook-func mew-addrbook-for-summary)))
	 (raw (or (mew-addrstr-parse-address addr) ""))
	 (nickname (if func (funcall func raw)))
	 (rules mew-scan-form-extract-rule)
	 rule ret)
    (catch 'matched
      (while rules
	(setq rule (car rules))
	(setq rules (cdr rules))
	(cond
	 ((and (eq rule 'name)
	       (or (string-match "^\"\\([^\"]+\\)\"[ \t]*<[^>]+>" addr)
		   (string-match "^\\([^<]+\\)<[^>]+>" addr)))
	  (throw 'matched (setq ret (mew-chop (mew-match 1 addr)))))
	 ((and (eq rule 'comment)
	       (string-match "^[^(]+(\\(.+\\))" addr))
	  (throw 'matched (setq ret (mew-chop (mew-match 1 addr)))))
	 ((eq rule 'address)
	  (throw 'matched (setq ret raw)))
	 ((and (eq rule 'nickname) nickname)
	  (throw 'matched (setq ret nickname)))
	 ((and (stringp rule)
	       (string-match rule addr))
	  (throw 'matched (setq ret (mew-chop (mew-match 1 addr))))))))
    (or ret addr)))

(defun mew-scan-form-from ()
  "A function to return an address.
If the message is destined to me AND 'mew-scan-form-from-me-prefix'
is a string, an address on To:, is returned. In this
case, 'mew-scan-form-from-me-prefix' is prepended to the address. 

Otherwise, an address on From: is returned.

Address is converted by 'mew-scan-form-extract-addr'. See also
'mew-scan-form-extract-rule'."
  (let* ((FROM (MEW-FROM)) (TO (MEW-TO))
	 (from (or (mew-addrstr-parse-address FROM) "")))
    (cond
     ((string= FROM "")
      "")
     ((and (stringp mew-scan-form-from-me-prefix)
	   (not (string= TO ""))
	   (mew-is-my-address mew-regex-my-address-list from))
      (mew-replace-white-space
       (concat mew-scan-form-from-me-prefix (mew-scan-form-extract-addr TO))))
     (t
      (mew-replace-white-space (mew-scan-form-extract-addr FROM))))))

(defun mew-scan-form-subj ()
  "A function to return Subject:. Unnecessary white spaces are removed."
  ;; The beginning white spaces have been removed in mew-scan-header
  ;; (mew-keyval).
  (let ((subj (MEW-SUBJ)))
    (if (string= subj "") (setq subj mew-error-no-subject))
    (if mew-decode-broken
	subj
      ;; already well-formatted
      (mew-replace-white-space subj))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; The engine function to call mew-scan-form-*
;;;

(defsubst mew-scan-split-usc (xuidl)
  (mew-split xuidl 32))

(defsubst mew-scan-usc-uid (usc)
  (or (nth 0 usc) ""))

(defsubst mew-scan-usc-size (usc)
  (or (nth 1 usc) ""))

(defsubst mew-scan-usc-case (usc)
  (or (nth 2 usc) ""))

(defun mew-scan-get-piece (spec)
  (let (func width str rightp)
    (if (stringp spec)
	(progn
	  (setq TOTAL (+ TOTAL (length spec))) ;; xxx length
	  spec)
      (if (symbolp spec)
	  (setq width 1 func spec)
	(setq width (nth 0 spec) func (nth 1 spec)))
      (when (and (integerp width) (symbolp func))
	(if (= width 0) (setq width (- WIDTH TOTAL 1)))
	(if (< width 0) (setq width (abs width) rightp t))
	(setq TOTAL (+ TOTAL width))
	(setq func (intern-soft
		    (concat mew-scan-form-func-prefix (symbol-name func))))
	(when (fboundp func)
	  (setq str (funcall func))
	  (if rightp
	      (if (<= (length str) width)
		  (format (format "%%%ds" width) str)
		(setq TOTAL (+ (- TOTAL width) (length str)))
		str) ;; width may exceed.
	    (mew-substring str width)))))))

(defun mew-scan-get-line (mew-vec WIDTH &optional VIRTUALP)
  (let* ((TOTAL 0)
	 (line (mapconcat (function mew-scan-get-piece)
			  (mew-sinfo-get-scan-form) ""))
	 (my-id "") (par-id "") (first-irt "") fld msg ld onlyone start)
    (if (string-match mew-regex-id (MEW-ID))
	(setq my-id (mew-match 1 (MEW-ID))))
    ;; RFC 2822 says: the "In-Reply-To:" field may be used to identify
    ;; the message (or messages) to which the new message is a reply,
    ;; while the "References:" field may be used to identify a
    ;; "thread" of conversation.
    ;;
    ;; However, even if the References field exists, it may not contain
    ;; a parent's ID. So, if the In-Reply-To field contain one ID, 
    ;; we use it for thread.
    ;;
    ;; (1) The In-Reply-To contains one ID, use it.
    ;; (2) The References contains one or more IDs, use the last one.
    ;; (3) The In-Reply-To contains two or more IDs, use the first one.
    (when (string-match mew-regex-id (MEW-IRT))
      (setq start (match-end 0))
      (setq first-irt (mew-match 1 (MEW-IRT)))
      (unless (string-match mew-regex-id (MEW-IRT) start)
	(setq onlyone t)))
    (cond
     (onlyone
      (setq par-id first-irt))
     ((string-match mew-regex-id (MEW-REF))
      (setq start (match-end 0))
      (setq par-id (mew-match 1 (MEW-REF)))
      (while (string-match mew-regex-id (MEW-REF) start)
	(setq start (match-end 0))
	(setq par-id (mew-match 1 (MEW-REF)))))
     (t
      (setq par-id first-irt)))
    (if (null VIRTUALP)
	(setq ld (format "\r <%s> <%s>\n" my-id par-id))
      (setq fld (mew-scan-get-folder mew-vec))
      (setq msg (mew-scan-get-message mew-vec))
      (setq ld (format "\r %s %s <%s> <%s>\n" fld msg my-id par-id)))
    (cons line ld)))

;;

(defvar mew-vec [0 1 2 3 4 5 6 8 9 10 11 12 13 14 15 16 17 18 19 20]
  "Just for test of (MEW-FOO).")

(defun mew-scan-setup ()
  "Define functions (MEW-FOO) according 'mew-scan-fields-alias'."
  (let ((n (length mew-scan-fields-alias))
	(i 0))
    (while (< i n)
      (fset (intern (concat "MEW-" (nth i mew-scan-fields-alias)))
	    (list 'lambda () (list 'aref 'mew-vec i)))
      (setq i (1+ i)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;
;;;

(defun mew-summary-scan-form (folder &optional column)
  "Get scan-form from 'mew-summary-refine-scan-form'.
See also 'mew-scan-form-list-list'."
  (let ((lst mew-scan-form-list)
	(str-func (mew-scan-form-list-string-func))
	(lst-func (mew-scan-form-list-list-func))
	key keys values ret form col)
    (catch 'loop
      (while lst
	(setq keys (car (car lst)))
	(setq values (cdr (car lst)))
	(setq lst (cdr lst))
	(cond
	 ((eq keys t)
	  (throw 'loop (setq ret values)))
	 ((stringp keys)
	  (if (funcall str-func keys folder)
	      (throw 'loop (setq ret values))))
	 ((listp keys)
	  (while keys
	    (setq key (car keys))
	    (setq keys (cdr keys))
	    (if (funcall lst-func key folder)
		(throw 'loop (setq ret values))))))))
    (if ret
	(setq form (nth 0 ret) col (nth 1 ret))
      (setq form mew-scan-form))
    (setq form (mew-summary-final-scan-form form))
    (if column
	(if (numberp col) col (mew-thread-column form))
      form)))

(defun mew-summary-final-scan-form (form)
  "Canonicalize the scan-form specified by FORM.
If the first element of FORM is an integer N,
append '((N num) mark) to the FORM.
Otherwise append 'mew-scan-form-header' to FORM."
  (if (integerp (car form))
      (cons (list (car form) 'num) (cons 'mark (cdr form)))
    (append mew-scan-form-header form)))

(defun mew-thread-column (form)
  (let ((col 0) ret ent)
    (catch 'loop
      (while form
	(setq ent (car form))
	(setq form (cdr form))
	(cond
	 ((consp ent)
	  (setq col (+ col (abs (car ent)))))
	 ((stringp ent)
	  (setq col (+ col (string-width ent))))
	 ((eq ent t)
	  (throw 'loop (setq ret col)))
	 (t
	  (setq col (1+ col))))))
    (or ret mew-thread-column)))

(defun mew-summary-scan-form-num (folder)
  (abs (car (rassoc '(num) (mew-summary-scan-form folder)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Exchanging the cursor
;;;

(defvar mew-summary-inbox-position (make-marker))

(defun mew-summary-exchange-point ()
  "Get back to the position before typing '\\<mew-summary-mode-map>\\[mew-summary-retrieve]'."
  (interactive)
  (mew-summary-only
   (cond
    ((not (mew-folder-inboxp (mew-summary-folder-name 'ext)))
     (message "Cannot be used in this folder"))
    ((not (marker-position mew-summary-inbox-position))
     (message "No previous folder"))
    (t
     (let* ((buf (marker-buffer mew-summary-inbox-position))
	    (pos (marker-position mew-summary-inbox-position))
	    (folder (buffer-name buf))
	    getback)
       (cond
	((equal (current-buffer) buf)
	 (goto-char pos))
	((not (get-buffer buf))
	 (message "Previous folder not exist"))
	((mew-folder-virtualp folder)
	 (if (mew-virtual-thread-p folder)
	     (if (mew-thread-cache-valid-p folder)
		 (setq getback t)
	       (message "%s is old. " folder))
	   (setq getback t)))
	(t
	 (setq getback t)))
       (when getback
	 (switch-to-buffer buf)
	 (goto-char pos)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Getting messages
;;;

(defun mew-range-erase-p (range)
  (or (char-equal (aref range 0) (aref mew-range-all 0))
      (string-match ":" range)))

(defun mew-summary-retrieve (&optional no-flush)
  "Retrieve messages to +inbox asynchronously.
If called with '\\[universal-argument]', +queue is not flushed."
  (interactive "P")
  (let* ((case mew-case-input)
	 (inbox (mew-inbox-folder case))
	 (mailbox (mew-mailbox-type case))
	 range)
    ;; for mew-summary-exchange-point.
    (cond
     ((mew-summary-or-virtual-p)
      (set-marker mew-summary-inbox-position (point) (current-buffer))
      (mew-summary-switch-to-folder inbox))
     (t
      (mew-summary-switch-to-folder inbox)
      (set-marker mew-summary-inbox-position (point) (current-buffer))))
    ;; in the inbox
    (when (mew-summary-exclusive-p)
      (when (mew-summary-folder-dir-newp)
	;; scan the gap
	(goto-char (point-max))
	(setq range (mew-input-range inbox mew-ask-range))
	(if (mew-range-erase-p range)
	    (mew-erase-buffer))
	(mew-scan (mew-scan-mewls-src inbox range))
	(mew-rendezvous mew-summary-buffer-process))
      (mew-window-configure 'summary)
      (mew-current-set nil nil nil)
      (mew-decode-syntax-delete)
      (goto-char (point-max))
      (mew-sinfo-set-start-point (point))
      (mew-sinfo-set-direction 'down)
      (mew-sinfo-set-scan-form (mew-summary-scan-form inbox))
      (cond
       ((eq mailbox 'pop)
	(mew-pop-retrieve case 'inc (not no-flush)))
       ((eq mailbox 'mbox)
	(mew-mbox-retrieve case))))))

(defun mew-header-get-xuidl-from-file (fld msg)
  (let ((buf (generate-new-buffer mew-buffer-prefix))
	(xmu mew-x-mew-uidl:)
	xuidl)
    (save-excursion
      (set-buffer buf)
      (mew-erase-buffer)
      (mew-insert-message
       fld msg mew-cs-text-for-read mew-header-reasonable-size)
      (setq xuidl (mew-header-get-value xmu)))
    (mew-remove-buffer buf)
    xuidl))

(defun mew-summary-retrieve-message (&optional rev-del)
  "Retrieve the rest of a truncated('T') message.
If 'mew-pop-delete' is 't', delete the message from the mailbox.
When executed with '\\[universal-argument]', 'mew-pop-delete' is considered reversed."
  (interactive "P")
  (mew-summary-msg
   (mew-summary-only
    (mew-summary-not-in-mdrop
     (when (mew-summary-exclusive-p)
       (let* ((fld (mew-summary-folder-name 'ext))
	      (msg (mew-summary-message-number))
	      (xuidl (mew-header-get-xuidl-from-file fld msg))
	      usc uid siz case del rtr rtrs)
	 (if (null xuidl)
	     (message "No %s field" mew-x-mew-uidl:)
	   (setq usc (mew-scan-split-usc xuidl))
	   (setq uid (mew-scan-usc-uid usc))
	   (setq siz (mew-scan-usc-size usc))
	   (setq case (mew-scan-usc-case usc))
	   (setq del (eq (mew-pop-delete case) t))
	   (if rev-del (setq del (not del)))
	   (setq rtr (list uid siz del (list fld msg)))
	   (setq rtrs (list rtr))
	   (mew-sinfo-set-scan-form (mew-summary-scan-form fld))
	   (mew-pop-retrieve case 'get fld rtrs))))))))

(defun mew-summary-mark-retrieve-message (&optional rev-del)
  "Retrieve the rest of truncated('T') messages marked with '*'.
If 'mew-pop-delete' is 't', delete the message from the mailbox.
When executed with '\\[universal-argument]', 'mew-pop-delete' is considered reversed."
  (interactive "P")
  (mew-summary-only
   (mew-summary-not-in-mdrop
    (let* ((fld (mew-summary-folder-name 'ext))
	   (msgs (mew-summary-mark-collect mew-mark-review))
	   msg usc xuidl uid siz case del rtr rtrs)
      (if (null msgs)
	  (message "No marked messages")
	(if rev-del (setq del (not del)))
	(while msgs
	  (setq msg (car msgs))
	  (setq msgs (cdr msgs))
	  (setq xuidl (mew-header-get-xuidl-from-file fld msg))
	  (when xuidl
	    (setq usc (mew-scan-split-usc xuidl))
	    (setq uid (mew-scan-usc-uid usc))
	    (setq siz (mew-scan-usc-size usc))
	    (setq case (mew-scan-usc-case usc)) ;; xxx
	    (setq del (eq (mew-pop-delete case) t))
	    (if rev-del (setq del (not del)))
	    (setq rtr (list uid siz del (list fld msg)))
	    (setq rtrs (cons rtr rtrs)))))
      (if (null rtrs)
	  (message "No %s field" mew-x-mew-uidl:)
	(mew-sinfo-set-scan-form (mew-summary-scan-form fld))
	(mew-pop-retrieve case 'get fld rtrs))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Retrieving messages from mbox
;;;

(defun mew-mbox-retrieve (case)
  (let ((mbox-command (mew-mbox-command case))
	(mbox-command-arg (mew-mbox-command-arg case))
	(inbox (mew-inbox-folder case))
	args)
    (if (not (mew-which-exec mbox-command))
	(message "'%s' not found!" mbox-command)
      (if mbox-command-arg
	  (setq args (list "-e" mbox-command "-m" mbox-command-arg))
	(setq args (list "-e" mbox-command)))
      (setq args (append args (mew-scan-mewls-src inbox)))
      (mew-scan args  nil nil 'flush))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Scanning a folder
;;;

(defun mew-summary-ls (&optional jump)
  "List this folder asynchronously.
If executed with '\\[universal-argument]', the cursor stays in the
current position."
  (interactive "P")
  (when (mew-summary-exclusive-p)
    (mew-summary-only
     (let* ((folder (mew-summary-folder-name 'ext))
	    (askp mew-ask-range)
	    scanp range mkdb oldformatp)
       (mew-summary-folder-cache-load)
       (save-excursion
	 (goto-char (point-min))
	 (if (and (not (eobp))
		  (not (looking-at mew-regex-thread))
		  (y-or-n-p "The format of this scan cache is old. Type 'y' when ready. "))
	     (setq oldformatp t)))
       (if oldformatp
	   (progn
	     (setq range mew-range-all)
	     (setq mkdb (mew-summary-mark-collect4))
	     (mew-erase-buffer)
	     (mew-scan (mew-scan-mewls-src folder range) mkdb))
	 (cond
	  ((interactive-p);; "s"
	   (setq askp t)
	   (setq scanp t))
	  ((mew-summary-folder-dir-newp);; "g"
	   (setq askp nil)
	   (setq scanp t)))
	 (set-buffer-modified-p nil)
	 (if (or (interactive-p) jump) (goto-char (point-max)))
	 (if (not scanp)
	     (run-hooks 'mew-summary-ls-no-scan-hook)
	   ;; asking range when "s"
	   (setq range (mew-input-range folder askp))
	   (when (mew-range-erase-p range)
	     (setq mkdb (mew-summary-mark-collect4))
	     (mew-erase-buffer))
	   (if (mew-folder-mdropp folder)
	       (mew-scan-mdrop folder)
	     (mew-scan (mew-scan-mewls-src folder range) mkdb))))))))

(defun mew-scan-mdrop (mdrop)
  (let* ((mdropdir (mew-expand-folder mdrop))
	 (msgs (mew-dir-messages mdropdir)))
    (mew-summary-clean-msgs mdrop msgs))
  (mew-sinfo-set-scan-form (mew-summary-scan-form mdrop))
  (mew-pop-retrieve mew-case-input 'scan mdrop))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Scan info
;;;

(defvar mew-scan-info-list '("folder" "message"))

(mew-info-defun "mew-scan-" mew-scan-info-list)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Process name
;;;

(defconst mew-scan-info-prefix "mew-scan-info-")

(defsubst mew-scan-info-name (bnm)
  (format "%s<%s>" mew-scan-info-prefix bnm))

(defsubst mew-scan-pnm-to-bnm (pnm)
  (if (string-match mew-regex-id pnm) (mew-match 1 pnm)))

(defsubst mew-scan-buffer-name (folder)
  (concat mew-buffer-prefix folder))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Sub-functions for Scan
;;;

(defun mew-scan-header (&optional draftp)
  (let (vec key med str n)
    (setq vec (make-vector (length mew-scan-fields) ""))
    (save-restriction
      (narrow-to-region (point-min) (point))
      (goto-char (point-min))
      (while (not (eobp))
	(if (not (looking-at mew-keyval))
	    (forward-line)
	  (setq key (mew-capitalize (mew-match 1)))
	  (setq med (match-end 0))
	  (forward-line)
	  (mew-header-goto-next)
	  (when (setq n (mew-member-case-equal key mew-scan-fields))
	    ;; xxx this may be able to tune up.
	    (mew-header-decode-region key med (point) draftp)
	    ;; Need to inherit properties.
	    ;; So, we use buffer-substring, not mew-buffer-substring.
	    (setq str (buffer-substring med (1- (point))))
	    (aset vec n str))))
      (goto-char (point-max)))
    vec))

(defun mew-scan-width ()
  (if (and (integerp mew-summary-scan-width)
	   (> mew-summary-scan-width 40)) ;; xxx
      mew-summary-scan-width
    (if (< (window-width) 80)
	80
      (window-width))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Scan body
;;;

(defun mew-scan-mewls-src (folder &optional range)
  (if range
      (list "-s" (format "%s %s" folder range))
    (list "-s" folder)))

(defun mew-scan (args &optional mkdb virtualp flush)
  (cond
   ((not (mew-summary-exclusive-p))
    ())
   ((not (mew-which-exec mew-prog-mewls))
    (message "'%s' not found!" mew-prog-mewls))
   (t
    (let ((process-connection-type mew-connection-type2)
	  process buf bnm)
      (condition-case nil
	  (progn
	    (cond
	     (virtualp
	      (or (mew-virtual-p) (mew-virtual-mode)))
	     (t 
	      (or (mew-summary-p) (mew-summary-mode))))
	    (mew-window-configure 'summary)
	    (mew-current-set nil nil nil)
	    (mew-decode-syntax-delete)
	    (setq bnm (mew-summary-folder-name 'ext))
	    (mew-sinfo-set-scan-form (mew-summary-scan-form bnm))
	    (mew-sinfo-set-start-point (point))
	    (mew-sinfo-set-direction 'down)
	    (mew-sinfo-set-scan-id nil)
	    (mew-sinfo-set-mkdb mkdb)
	    (mew-sinfo-set-flush flush)
	    (message "Scanning %s..." bnm)
	    (setq buf (get-buffer-create (mew-scan-buffer-name bnm)))
	    (save-excursion
	      (set-buffer buf)
	      (mew-erase-buffer))
	    (setq args (append (list "-b" mew-mail-path "-c" mew-news-path
				     "-w" (int-to-string mew-scan-wait-for)
				     "-f" (mapconcat (function identity)
						     (nthcdr 2 mew-scan-fields)
						     ","))
			       args))
	    (setq process (apply (function start-process)
				 (mew-scan-info-name bnm)
				 buf mew-prog-mewls args))
	    (mew-summary-lock process "Scanning")
	    (mew-set-process-cs process mew-cs-text-for-read mew-cs-dummy)
	    ;; text may be broken, so undecided is very dangerous!
	    (set-process-filter   process 'mew-scan-filter)
	    (set-process-sentinel process 'mew-scan-sentinel)
	    (process-kill-without-query process))
	(quit
	 (set-process-sentinel process nil)
	 (mew-sinfo-set-start-point nil)
	 (mew-sinfo-set-mkdb nil)
	 (mew-sinfo-set-flush nil)
	 (mew-summary-unlock)))))))

;; See also mew-summary-cook-region
(defun mew-scan-insert-line (folder vec width lmsg virtualp)
  (when (get-buffer folder)
    (save-excursion
      (set-buffer folder)
      (let* ((after-change-functions nil)
	     (line (mew-scan-get-line vec width virtualp))
	     (opos (point))
	     (omax (point-max))
	     beg med face mark)
	(mew-elet
	 (if (null lmsg)
	     (goto-char (point-max))
	   ;; a message marked with 'T'.
	   (goto-char (point-min))
	   (if (not (re-search-forward (mew-regex-jmp-msg lmsg) nil t))
	       (goto-char (point-max));; xxx
	     (beginning-of-line)
	     (setq beg (point))
	     (forward-line)
	     ;; To avoid inserting a line AFTER the cursor underline,
	     ;; keep this line and make it invisible.
	     (put-text-property beg (point) 'invisible t)
	     (forward-line -1)))
	 (setq beg (point))
	 ;; To "insert" just after mew-marker-decode-syntax-end.
	 (insert (car line))
	 (setq med (point))
	 (goto-char beg)
	 (mew-front-nonsticky beg med);; for XEmacs
	 (if (and mew-use-highlight-mark
		  (looking-at mew-regex-msg-mark)
		  (setq mark (string-to-char (mew-match 2)))
		  (setq face (mew-highlight-mark-get-face mark)))
	     (put-text-property beg med 'face face))
	 (if mew-use-highlight-mouse-line
	     (put-text-property
	      beg med 'mouse-face mew-highlight-mouse-line-face))
	 (goto-char med)
	 (insert (cdr line))
	 (put-text-property med (1- (point)) 'invisible t)
	 ;; Removing the invisible line.
	 (when lmsg
	   (setq beg (point))
	   (forward-line)
	   (delete-region beg (point))))
	(if (or (eq opos (mew-sinfo-get-start-point))
		(/= opos omax))
	    ;; move the cursor to the original position.
	    (goto-char opos))))))

(defun mew-scan-filter (process string)
  (let* ((width (1- (mew-scan-width))) ;; xxx
	 (pnm (process-name process))
	 (folder (mew-scan-pnm-to-bnm pnm))
	 (virtualp (mew-folder-virtualp folder))
	 (draftp (mew-folder-draftp folder))
	 vec)
    (mew-filter
     (goto-char (point-max))
     (mew-set-buffer-multibyte nil)
     (mew-elet (insert string))
     (goto-char (point-min))
     (while (and (re-search-forward mew-eoh nil t) (not (eobp)))
       (mew-set-buffer-multibyte t)
       (setq vec (mew-scan-header draftp))
       (mew-set-buffer-multibyte nil)
       (mew-scan-insert-line folder vec width nil virtualp)
       (forward-line)
       (delete-region (point-min) (point))))))

(defun mew-scan-sentinel (process event)
  (let* ((pnm (process-name process))
	 (folder (mew-scan-pnm-to-bnm pnm))
	 (virtualp (mew-folder-virtualp folder))
	 opos mdb ent msg mrk flush)
    (mew-filter
     (mew-set-buffer-multibyte t)
     (set-buffer folder)
     (setq opos (point))
     (setq mdb (mew-sinfo-get-mkdb))
     (setq flush (mew-sinfo-get-flush))
     (goto-char (point-min))
     (while mdb
       (setq ent (car mdb))
       (setq mdb (cdr mdb))
       (setq msg (car ent))
       (setq mrk (cdr ent))
       (when (re-search-forward (mew-regex-jmp-msg msg) nil t)
	 (mew-summary-mark-as mrk 'force)
	 (forward-line)))
     (goto-char opos)
     (if (and virtualp (mew-vinfo-get-func))
	 (funcall (mew-vinfo-get-func)))
     (mew-vinfo-set-func nil)
     (mew-sinfo-set-start-point nil)
     (mew-sinfo-set-mkdb nil)
     (mew-sinfo-set-flush nil)
     (mew-summary-folder-cache-save)
     (set-buffer-modified-p nil)
     (mew-summary-unlock)
     (message "Scanning %s...done" folder)
     (run-hooks 'mew-scan-sentinel-hook)
     (when (and mew-auto-flush-queue flush)
       (mew-smtp-flush-queue
	(mew-queue-folder mew-case-output) mew-case-output)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Summary file cache
;;;

(defun mew-summary-compare-times (type)
  (let* ((folder (mew-summary-folder-name 'ext))
	 (dir (file-chase-links (mew-expand-folder folder)))
	 (tdir (if mew-touch-folder-p
		   (mew-file-get-time
		    (expand-file-name mew-summary-touch-file
				      (mew-expand-folder dir)))
		 (mew-file-get-time dir)))
	 (cache (expand-file-name mew-summary-cache-file dir))
	 (tcache (mew-file-get-time cache))
	 (tbuf (mew-sinfo-get-cache-time))
	 t1 t2)
    (cond
     ((eq type 'dir-cache)
      (setq t1 tdir)
      (setq t2 tcache))
     ((eq type 'cache-buf)
      (setq t1 tcache)
      (setq t2 tbuf)))
    (cond
     ((null t1) nil)
     ((null t2) t) ;; do update
     ((> (nth 0 t1) (nth 0 t2)) t)
     ((= (nth 0 t1) (nth 0 t2))
      (if (> (nth 1 t1) (nth 1 t2)) t nil)) ;; nil if equal
     (t nil))))

(defsubst mew-summary-folder-dir-newp ()
  (mew-summary-compare-times 'dir-cache))

(defsubst mew-summary-folder-cache-newp ()
  (mew-summary-compare-times 'cache-buf))

(defsubst mew-summary-folder-cache-updatep (folder)
  (mew-folder-localp folder))

(defun mew-summary-folder-cache-load ()
  (let* ((folder (mew-summary-folder-name 'ext))
	 (cache (mew-expand-folder folder mew-summary-cache-file)))
    (when (and (mew-summary-folder-cache-updatep folder)
	       (file-readable-p cache)
	       (mew-summary-folder-cache-newp))
      (mew-elet
       (mew-erase-buffer)
       (mew-frwlet
	mew-cs-m17n mew-cs-dummy
	(insert-file-contents cache))
       (mew-sinfo-set-cache-time (mew-file-get-time cache))
       (setq mew-summary-buffer-raw t)
       (mew-mark-undo-mark mew-mark-refile 'no-msg)
       (set-buffer-modified-p nil)))))

(defun mew-summary-folder-cache-save ()
  (let* ((folder (mew-summary-folder-name 'ext))
	 (cache (mew-expand-folder folder mew-summary-cache-file)))
    (when (and (mew-summary-folder-cache-updatep folder)
	       (file-writable-p cache))
      (mew-touch-folder folder)
      (save-restriction
	(widen)
	(if (mew-decode-syntax-p)
	    (let ((cbuf (current-buffer))
		  (min (point-min))
		  (max (point-max))
		  (beg (mew-decode-syntax-begin))
		  (end (mew-decode-syntax-end))
		  (tmpbuf (generate-new-buffer mew-buffer-prefix)))
	      (set-buffer tmpbuf)
	      (mew-erase-buffer)
	      (mew-insert-buffer-substring cbuf min beg)
	      (mew-insert-buffer-substring cbuf end max)
	      (mew-frwlet
	       mew-cs-dummy mew-cs-m17n
	       (write-region (point-min) (point-max) cache nil 'no-msg))
	      (set-buffer cbuf)
	      (mew-remove-buffer tmpbuf))
	  (mew-frwlet
	   mew-cs-dummy mew-cs-m17n
	   (write-region (point-min) (point-max) cache nil 'no-msg))
	  (set-file-modes cache mew-file-mode))
	(mew-sinfo-set-cache-time (mew-file-get-time cache))))))

(provide 'mew-scan)

;;; Copyright Notice:

;; Copyright (C) 1996-2002 Mew developing team.
;; All rights reserved.

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;; 
;; 1. Redistributions of source code must retain the above copyright
;;    notice, this list of conditions and the following disclaimer.
;; 2. Redistributions in binary form must reproduce the above copyright
;;    notice, this list of conditions and the following disclaimer in the
;;    documentation and/or other materials provided with the distribution.
;; 3. Neither the name of the team nor the names of its contributors
;;    may be used to endorse or promote products derived from this software
;;    without specific prior written permission.
;; 
;; THIS SOFTWARE IS PROVIDED BY THE TEAM AND CONTRIBUTORS ``AS IS'' AND
;; ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;; PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE TEAM OR CONTRIBUTORS BE
;; LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
;; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
;; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
;; BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
;; OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
;; IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; mew-scan.el ends here
