/* audio.c -- sound warnings for Melon 
 *
 * Copyright(C) 2001-2002 Elisa Manara <e@entropika.net>
 * This code is released under the GPL License version 2 */

#include <stdio.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <dirent.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include "melon.h"

static void set_beep ( GtkWidget	*p_list,
			gint            row,
			GdkEventButton  *event );


void play_beep(void)
{
	int dev = -1;
	int fd = -1;
	int b_read;
	int retval;
	char buf[1024];
	char *error_string;

	/* No beep is set */
	if (cfg_data.beep_file[0] == '\0')
		return;

	if((dev = open(cfg_data.beep_device, O_WRONLY)) == -1) {
		error_string=strerror(errno);
		melon_error("Cannot open %s: %s\n", cfg_data.beep_device, error_string);
		goto error;
	}

	if((fd = open(cfg_data.beep_file, O_RDONLY)) == -1) {
		error_string=strerror(errno);
		melon_error("Cannot open beep file: %s\n", error_string);
		goto error;
	}

	while ((b_read = read(fd, buf, 1024))) 
	{
		if (b_read == -1) {
			error_string=strerror(errno);
			melon_error("Error on reading beep file: %s\n", error_string);
			goto error;
		}
		retval = write(dev, buf, b_read);
		if (retval == -1) {
			error_string=strerror(errno);
			melon_error("Cannot write to %s: %s\n", cfg_data.beep_device, error_string);
			goto error;
		}
	}
error:
	if (dev != -1) close (dev);
	if (fd != -1) close (fd);
}

gint select_beep (GtkWidget *widget, gpointer data)
{
	GtkWidget *sel_window;
	GtkWidget *vbox;
	GtkWidget *scroll_win, *a_list;
	GdkPixmap *gdk_pix1, *gdk_pix2;
	GtkWidget *button;
	GdkBitmap *mask1, *mask2;
	GtkStyle *style;
	DIR *beep_dir;
	struct dirent *dr;
	char beep_path[MAX_PLEN];
	char beep_dir_path[MAX_PLEN];
	char sound_pix[MAX_PLEN], no_sound_pix[MAX_PLEN];
	gchar *a_data[2];
	int row = 0;

	if((strcmp(cfg_data.beep_device, "/dev/audio")) == 0) {
		snprintf(beep_dir_path, MAX_PLEN, "%s", BEEP_DIR_AU);
	} else if ((strcmp(cfg_data.beep_device, "/dev/dsp")) == 0) {
		snprintf(beep_dir_path, MAX_PLEN, "%s", BEEP_DIR_VOC);
	} else {
		dialog_message(" Please, set your audio device first! ");
		return 0;
	}
		
	/* window */
	sel_window = gtk_window_new( GTK_WINDOW_DIALOG);
	gtk_window_set_position(GTK_WINDOW (sel_window), GTK_WIN_POS_MOUSE);
	gtk_widget_set_usize( GTK_WIDGET (sel_window), 200, 240);
	gtk_window_set_title(GTK_WINDOW (sel_window), "Beep Selection");
	gtk_widget_show(sel_window);

	/* vbox */
	vbox = gtk_vbox_new( FALSE, 0 );
	gtk_container_add( GTK_CONTAINER( sel_window ), vbox );
	gtk_widget_show( vbox );

	/* scrolled window */
	scroll_win = gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW( scroll_win ),  
		GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);
	gtk_box_pack_start( GTK_BOX( vbox ), scroll_win, FALSE, FALSE, 0 );
	gtk_widget_show(scroll_win);

	/* sound pixmap */
	snprintf(sound_pix, MAX_PLEN, "%s/%s", beep_dir_path, ".sound.xpm");
	snprintf(no_sound_pix, MAX_PLEN, "%s/%s", beep_dir_path, ".nosound.xpm");

	style = gtk_widget_get_default_style();
	gdk_pix1 = gdk_pixmap_create_from_xpm( sel_window->window, &mask1,
		&style->bg[GTK_STATE_NORMAL],
		sound_pix);
	gdk_pix2 = gdk_pixmap_create_from_xpm( sel_window->window, &mask2,
		&style->bg[GTK_STATE_NORMAL],
		no_sound_pix);

	/* clist */
	a_list = gtk_clist_new( 2 );

	gtk_signal_connect(GTK_OBJECT(a_list), "select_row",
		GTK_SIGNAL_FUNC(set_beep), NULL);

	gtk_widget_set_usize( GTK_WIDGET (a_list), 200, 200);
	gtk_clist_set_row_height( GTK_CLIST (a_list), 20);
	gtk_clist_set_shadow_type (GTK_CLIST(a_list), GTK_SHADOW_OUT);
	gtk_clist_set_selection_mode( GTK_CLIST( a_list ),
		GTK_SELECTION_SINGLE);

	gtk_clist_set_column_width(GTK_CLIST( a_list ), 0, 25);
	gtk_clist_set_column_width(GTK_CLIST( a_list ), 1, 175);

	gtk_container_add( GTK_CONTAINER( scroll_win ), a_list );

	/* no sound line */
	a_data[0] = NULL;
	a_data[1] = "No beep";

	gtk_clist_append(GTK_CLIST (a_list), a_data);
	gtk_clist_set_pixmap(GTK_CLIST( a_list ), row, 0, gdk_pix2, mask2);
	row++;

	/* fill clist */
	if((beep_dir = opendir(beep_dir_path)) == NULL) {
		char msg[1024];
		snprintf(msg, 1024, " Cannot open %s ", beep_dir_path);
		dialog_message(msg);
		return 0;
        }
	while((dr = readdir(beep_dir)) != NULL)
	{
		if ((!strncmp(dr->d_name, ".", 1)))
			continue;

		/* path of every au file in beep dir */
		snprintf(beep_path, MAX_PLEN, "%s/%s", beep_dir_path, dr->d_name);
		
		a_data[0] = NULL;
		a_data[1] = strdup(dr->d_name);

		gtk_clist_append(GTK_CLIST (a_list), a_data);
		free(a_data[1]);

		gtk_clist_set_pixmap(GTK_CLIST( a_list ), row, 0, gdk_pix1, mask1);
		row++;
	}
	closedir(beep_dir);

	gtk_widget_show(a_list);

	/* free sound pixmap */
	gdk_pixmap_unref(gdk_pix1);
	gdk_pixmap_unref(gdk_pix2);

	/* CLOSE button */
	button = gtk_button_new_with_label ("Close");
	gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
		GTK_SIGNAL_FUNC(gtk_widget_destroy),
		GTK_OBJECT (sel_window));
	gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
	gtk_widget_show (button);

	return 1;

}

void set_beep ( GtkWidget	*a_list,
		gint            row,
		GdkEventButton  *event )
{
	char *beep_file;
	char beep_dir_path[MAX_PLEN];

	if((strcmp(cfg_data.beep_device, "/dev/audio")) == 0)
		snprintf(beep_dir_path, MAX_PLEN, "%s", BEEP_DIR_AU);
	else if ((strcmp(cfg_data.beep_device, "/dev/dsp")) == 0)
		snprintf(beep_dir_path, MAX_PLEN, "%s", BEEP_DIR_VOC);
	
	gtk_clist_get_text(GTK_CLIST( a_list ), row, 1, &beep_file);

	if(strcmp(beep_file, "No beep") == 0) {
		cfg_data.beep_file[0] = '\0';
	} else {
		snprintf (cfg_data.beep_file, MAX_PLEN, "%s/%s", beep_dir_path, beep_file);
		play_beep();
	}

	/* update melon.cfg file for next startup */
	config_update();

	return;
}

gint set_beep_mode (GtkWidget *widget, gpointer mode)
{
	if (strcmp(mode, "ONETIME") == 0) {
		cfg_data.beep_mode = 0;
	} else if (strcmp(mode, "EACHTIME") == 0) {
		cfg_data.beep_mode = 1;
	}

	dialog_message(" Beep mode updated ");
	config_update();
	return 1;
}

gint set_beep_device (GtkWidget *widget, gpointer device)
{
	char msg[1024];

	strncpy(cfg_data.beep_device, device, MAX_PLEN);

	snprintf(msg, 1024, " Beep device is now %s ", cfg_data.beep_device);
	dialog_message(msg);
	config_update();
	return 1;
}

