/* Mednafen - Multi-system Emulator
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "main.h"

#define MK_COLOR_A(r,g,b,a) ( ((a)<<surface->format->Ashift) | ((r)<<surface->format->Rshift) | ((g) << surface->format->Gshift) |((b) << surface->format->Bshift))

uint32 DrawTextTrans(uint32 *dest, int pitch, uint32 width, uint8 *textmsg, uint32 fgcolor, int centered, bool want5x7 = 0);

static uint32 VirtualTime[128], DrawnTime[128], BlittedTime[128];
static uint32 VirtualIndex, DrawnIndex, BlittedIndex;

void FPS_Init(void)
{
 VirtualIndex = 0;
 DrawnIndex = 0;
 BlittedIndex = 0;

 memset(VirtualTime, 0, sizeof(VirtualTime));
 memset(DrawnTime, 0, sizeof(DrawnTime));
 memset(BlittedTime, 0, sizeof(BlittedTime));
}

void FPS_IncVirtual(void)
{
 VirtualTime[VirtualIndex] = SDL_GetTicks();
 VirtualIndex = (VirtualIndex + 1) & 127;
}

void FPS_IncDrawn(void)
{
 DrawnTime[DrawnIndex] = SDL_GetTicks();
 DrawnIndex = (DrawnIndex + 1) & 127;
}

void FPS_IncBlitted(void)
{
 BlittedTime[BlittedIndex] = SDL_GetTicks();
 BlittedIndex = (BlittedIndex + 1) & 127;
}

static bool isactive = 0;
bool FPS_IsActive(void)
{
 return(isactive);
}

void FPS_ToggleView(void)
{
 isactive ^= 1;
}

void FPS_Draw(SDL_Surface *surface, const SDL_Rect *rect)
{
 if(!isactive) return;

 uint32 curtime = SDL_GetTicks();
 uint32 vt_frames_drawn = 0, dt_frames_drawn = 0, bt_frames_drawn = 0;
 uint32 vt_mintime, dt_mintime, bt_mintime;
 char virtfps[64], drawnfps[64], blitfps[64];

 vt_mintime = dt_mintime = bt_mintime = curtime;

 for(int x = 0; x < 128; x++)
 {
  uint32 vt = VirtualTime[x];
  uint32 dt = DrawnTime[x];
  uint32 bt = BlittedTime[x];

  if(vt >= (curtime - 1000))
  {
   if(vt < vt_mintime) vt_mintime = vt;
   vt_frames_drawn++;
  }

  if(dt >= (curtime - 1000))
  {
   if(dt < dt_mintime) dt_mintime = dt;
   dt_frames_drawn++;
  }

  if(bt >= (curtime - 1000))
  {
   if(bt < bt_mintime) bt_mintime = bt;
   bt_frames_drawn++;
  }
 }

 if(curtime - vt_mintime)
  snprintf(virtfps, 64, "%f", (double)vt_frames_drawn * 1000 / (curtime - vt_mintime));
 else
  snprintf(virtfps, 64, "?");

 if(curtime - dt_mintime)
  snprintf(drawnfps, 64, "%f", (double)dt_frames_drawn * 1000 / (curtime - dt_mintime));
 else
  snprintf(drawnfps, 64, "?");

 if(curtime - bt_mintime)
  snprintf(blitfps, 64, "%f", (double)bt_frames_drawn * 1000 / (curtime - bt_mintime));
 else
  snprintf(blitfps, 64, "?");

 DrawTextTrans((uint32 *)surface->pixels + rect->x, surface->pitch, rect->w, (UTF8*)virtfps, MK_COLOR_A(0xFF, 0xFF, 0xFF, 0xFF), 0);
 DrawTextTrans((uint32 *)surface->pixels + rect->x + 13 * (surface->pitch >> 2), surface->pitch, rect->w, (UTF8*)drawnfps, MK_COLOR_A(0xFF, 0xFF, 0xFF, 0xFF), 0);
 DrawTextTrans((uint32 *)surface->pixels + rect->x + 13 * 2 * (surface->pitch >> 2), surface->pitch, rect->w, (UTF8*)blitfps, MK_COLOR_A(0xFF, 0xFF, 0xFF, 0xFF), 0);
}
