/*
 * Copyright (C) 2008 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
#include "recording_manager.hh"
#include "application.hh"
#include "exception_handler.hh"

ScheduledRecordingConfigurationSection::ScheduledRecordingConfigurationSection(Configuration& c) : configuration(c)
{
	XmlDocument& document = configuration.get_document();
	XPath xpath(document);
	XPathResult result(xpath.evaluate_expression("/application/scheduled_recordings"));
	if (result.get_count() == 0)
	{
		XmlNode root_node(document.get_root_node());
		scheduled_recordings_node = root_node.create_child_node("scheduled_recordings");
	}
	else
	{
		scheduled_recordings_node = result.get_result(0);
	}	
}

void ScheduledRecordingConfigurationSection::add(const ScheduledRecording& scheduled_recording)
{
	xmlNodePtr n = scheduled_recording.node;
	
	if (n == NULL)
	{
		XmlNode node(scheduled_recordings_node);
		
		if (get_scheduled_recording(scheduled_recording.description) != NULL)
		{
			throw Exception(_("Cannot create a new scheduled recording because there is an existing scheduled recording with that description."));
		}

		n = node.create_child_node("scheduled_recording");
	}
	
	XmlNode scheduled_recording_node(n);
	
	scheduled_recording_node.set_attribute("type",			scheduled_recording.type);
	scheduled_recording_node.set_attribute("channel",		scheduled_recording.channel_name);
	scheduled_recording_node.set_attribute("description",	scheduled_recording.description);
	scheduled_recording_node.set_attribute("start_time",	scheduled_recording.start_time);
	scheduled_recording_node.set_attribute("duration",		scheduled_recording.duration);
	
	configuration.set_dirty();
}

void ScheduledRecordingConfigurationSection::remove(const String& description)
{
	xmlNodePtr node = get_scheduled_recording(description);
	
	if (node == NULL)
	{
		throw Exception(_("Failed to get scheduled recording with description '%s'"), description.c_str());
	}

	XmlNode n(node);
	n.unlink();
}

xmlNodePtr ScheduledRecordingConfigurationSection::get_scheduled_recording(const String& description)
{
	xmlNodePtr result_node = NULL;
	
	String encoded_description = description;
	
	encoded_description = encoded_description.replace("&", "&amp;");
	encoded_description = encoded_description.replace("\"", "&quot;");
	encoded_description = encoded_description.replace("<", "&lt;");
	encoded_description = encoded_description.replace(">", "&gt;");
	encoded_description = encoded_description.replace("'", "&apos;");
	
	XPathResult result(get_scheduled_recordings());

	gsize count = result.get_count();
	for (guint index = 0; index < count; index++)
	{
		xmlNodePtr current_node = result.get_result(index);

		ScheduledRecording scheduled_recording(current_node);
		if (scheduled_recording.description == description)
		{
			result_node = current_node;
		}
	}
		
	return result_node;
}

xmlXPathObjectPtr ScheduledRecordingConfigurationSection::get_scheduled_recordings()
{
	XPath xpath(scheduled_recordings_node);
	return xpath.evaluate_expression("/application/scheduled_recordings/scheduled_recording");
}

RecordingManager::RecordingManager()
{
	Configuration& configuration = Application::get_current().get_configuration();
	scheduled_recording_configuration_section = new ScheduledRecordingConfigurationSection(configuration);
}

void RecordingManager::add_scheduled_recording(const ScheduledRecording& scheduled_recording)
{
	Application& application = Application::get_current();
	Configuration& configuration = application.get_configuration();

	bool fail = false;
	XPathResult scheduled_recordings = get_scheduled_recordings();
	int count = scheduled_recordings.get_count();
	for (int i = 0; i < count; i++)
	{
		if (scheduled_recordings.get_result(i) != scheduled_recording.node)
		{
			ScheduledRecording existing_scheduled_recording(scheduled_recordings.get_result(i));

			Channel& existing_channel = application.get_channel(existing_scheduled_recording.channel_name);

			gboolean conflict = check_conflict(scheduled_recording, existing_scheduled_recording);

			if (conflict && existing_channel.name != scheduled_recording.channel_name)
			{
				String message = String::format(
					_("This recording overlaps another scheduled recording with description '%s'"),
					existing_scheduled_recording.description.c_str());
				Application::show_error_message_dialog(message);
				fail = true;
				break;
			}
		}
	}
	
	if (!fail)
	{
		scheduled_recording_configuration_section->add(scheduled_recording);
		Log::write(_("Scheduled '%s' for recording"), scheduled_recording.description.c_str());
		configuration.save();
	}
}

void RecordingManager::remove_scheduled_recording(const String& description)
{
	scheduled_recording_configuration_section->remove(description);
}

gboolean RecordingManager::check_conflict(const ScheduledRecording& a, const ScheduledRecording& b)
{
	gboolean conflict = false;
	gboolean check_daily =
		a.type == RECORDING_TYPE_EVERY_DAY_NOT_WE ||
		a.type == RECORDING_TYPE_EVERY_DAY_WE ||
		b.type == RECORDING_TYPE_EVERY_DAY_NOT_WE ||
		b.type == RECORDING_TYPE_EVERY_DAY_WE;
	gboolean check_weekly =
		a.type == RECORDING_TYPE_EVERY_WEEK ||
		b.type == RECORDING_TYPE_EVERY_WEEK;
	
	guint a_end_time = a.start_time + a.duration;
	guint b_end_time = b.start_time + b.duration;
	guint a_start_time_day = a.start_time % SECONDS_IN_DAY;
	guint b_start_time_day = b.start_time % SECONDS_IN_DAY;
	guint a_end_time_day = (a.start_time % SECONDS_IN_DAY) + a.duration;
	guint b_end_time_day = (b.start_time % SECONDS_IN_DAY) + b.duration;

	conflict = is_overlapped(a.start_time, a_end_time, b.start_time, b_end_time);
	conflict = conflict || check_daily && is_overlapped(a_start_time_day, a_end_time_day, b_start_time_day, b_end_time_day);
	conflict = conflict || check_weekly && is_overlapped(a_start_time_day, a_end_time_day, b_start_time_day, b_end_time_day);

	return conflict;
}

gboolean RecordingManager::is_overlapped(guint a, guint b, guint c, guint d)
{
	gboolean test1 = a <= c && c <= b;
	gboolean test2 = a <= d && d <= b;
	gboolean test3 = c <= b && b <= d;
	gboolean test4 = c <= a && a <= d;
		
	return test1 || test2 || test3 || test4;
}

gboolean RecordingManager::is_in(guint value, guint start, guint end)
{
	gboolean result = start <= value && value <= end;

	Log::write(G_LOG_LEVEL_DEBUG, "is in: %d < %d < %d = %d", start, value, end, result);
	
	return result;
}

xmlXPathObjectPtr RecordingManager::get_scheduled_recordings()
{
	return scheduled_recording_configuration_section->get_scheduled_recordings();
}

ScheduledRecordingState RecordingManager::check(ScheduledRecording& scheduled_recording)
{
	ScheduledRecordingState result = SCHEDULED_RECORDING_STATE_PENDING;

	time_t now = DateTime::now_utc();
	
	int end_time = scheduled_recording.start_time + scheduled_recording.duration;

	if (end_time < now && scheduled_recording.type == RECORDING_TYPE_ONCE_OFF)
	{
		result = SCHEDULED_RECORDING_STATE_OLD;
	}
	else
	{
		if (scheduled_recording.is_current())
		{
			result = SCHEDULED_RECORDING_STATE_CURRENT;
		}
	}
	
	return result;
}

xmlNodePtr RecordingManager::check()
{
	xmlNodePtr result = NULL;
		
	XPathResult scheduled_recordings = get_scheduled_recordings();
	int count = scheduled_recordings.get_count();
	for (int i = 0; i < count; i++)
	{
		xmlNodePtr node = scheduled_recordings.get_result(i);
		ScheduledRecording scheduled_recording(node);
		ScheduledRecordingState state = check(scheduled_recording);
		switch (state)
		{
			case SCHEDULED_RECORDING_STATE_OLD:
				{
					Log::write(_("Removing old scheduled recording '%s'"), scheduled_recording.description.c_str());
					XmlNode scheduled_recording_node(scheduled_recording.node);
					scheduled_recording_node.unlink();
				}
				break;
			
			case SCHEDULED_RECORDING_STATE_CURRENT:
				if (result != NULL)
				{
					Log::write(_("Scheduled recording conflict"));
				}
				else
				{
					result = node;
				}
				break;
			
			case SCHEDULED_RECORDING_STATE_PENDING:
				break;
		}
	}
	
	return result;
}

gboolean RecordingManager::has_scheduled_recordings()
{
	XPathResult result = get_scheduled_recordings();
	return result.get_count() > 0;
}

xmlNodePtr RecordingManager::get_scheduled_recording(const String& description)
{
	return scheduled_recording_configuration_section->get_scheduled_recording(description);
}
