//
// A simple two-dimensional array template with bounds checking.
//
// Written by: Gary Holt   19 May 1997
//

#ifndef H_array
#define H_array 1
//
// A simple template for a 1D array of arbitrary type.  There are two reasons
// for using this instead of an array allocated with new:
// 1) It does bounds checking if the symbol NDEBUG is not defined.
// 2) The array goes away automatically when you're done with it.
//
template <class T>
class array
{
  T *arr_els;
  unsigned n_elements;
public:
  array() { arr_els = 0; n_elements = 0; } // Default constructor.
				// If you call this constructor, you must
				// call reallocate() or copy() to allocate
				// space for elements, or else there won't
				// be anything in the array.

//
// Constructor.  Specify the total number of elements in the array.
// Note that the last index will be length() - 1.
//
  array(unsigned arr_size) {	// Constructor:
    arr_els = new T[arr_size];	// Allocate the elements.
    n_elements = arr_size;	// Remember how many we allocated.
  }

//
// Index operator:
//
  T &operator[](unsigned idx) { 
#ifndef NDEBUG
    if (idx >= n_elements) // Illegal index?
      abort();			// Urk!
#endif
    return arr_els[idx];	// Return the element.
  }

  const T &operator[](unsigned idx) const { // This operator applies to
#ifndef NDEBUG
    if (idx >= n_elements)	// Illegal index?
      abort();			// Urk!
#endif
    return arr_els[idx];		    // constant arrays.
  }
//
// Index without range checking, even if debug is enabled:
//
        T &ix(unsigned idx)       { return arr_els[idx]; }
  const T &ix(unsigned idx) const { return arr_els[idx]; }

//
// Return the size of the array:
//
  unsigned length() const { return n_elements; }
  int last_idx() const { return n_elements - 1; }

//
// Change the allocation size.  Deletes the previous array elements and
// allocates space for the new number.  If 0 is specified, no space is
// allocated.
//
  void reallocate(unsigned new_arr_size);

//
// Assignment operator:
//
  array<T> &operator=(const array<T> &in_arr) {
    copy(in_arr.arr_els, in_arr.n_elements); // Copy element-by-element.
    return *this;		// Allow x=y=z.
  }

//
// Copy a pre-existing C-style array.  Arguments:
// 1) A pointer to the array.
// 2) The number of elements in the array.
//
  void copy(const T *, unsigned n_elements);

//
// Destructor:
//
  ~array() { delete[] arr_els; } // Free our memory.
};


//
// Change the allocation size.  Deletes the previous array elements and
// allocates space for the new number.  If 0 is specified, no space is
// allocated.
//
template <class T>
void
array<T>::reallocate(unsigned new_arr_size)
{
  delete[] arr_els;		// Discard previous elements.
  if (new_arr_size == 0)
    arr_els = 0;		// Don't allocate a new array.
  else
    arr_els = new T[new_arr_size]; // Allocate the elements.
  n_elements = new_arr_size;	// Remember how many we allocated.
}

//
// Assignment operator:
//
template <class T>
void
array<T>::copy(const T *in_arr, unsigned n_els)
{
  reallocate(n_els);		// Allocate the proper amount of space.

  for (unsigned idx = 0; idx < n_els; ++idx)
    arr_els[idx] = in_arr[idx];	// Copy each of the elements.
}

#endif /* ndef H_array */
