/* matchbox - a lightweight window manager

   Copyright 2002 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include "wm.h"

wm*
wm_new(int argc, char **argv)
{
   int i, *j;
      
   XSetWindowAttributes sattr; /* for root win */
   XSetWindowAttributes dattr; /* for dockbar  */
    
   Wm *w = NULL;

   int keys_to_grab[] = { XK_Return, XK_Tab, XK_l, XK_r,
			  XK_p, XK_n, XK_Left, XK_Right,
			  XK_Up, XK_Down, XK_L, XK_R, 0 };
   
   if ( (w = malloc(sizeof(Wm))) == NULL) err("err out of memory");
   
   wm_load_config(w, &argc, argv);
   
   if ((w->dpy = XOpenDisplay(w->config->display_name)) == NULL) {
      err("can't open display! check your DISPLAY variable.");
      exit(1);
   }
   
   XSetErrorHandler(handle_xerror); 

   w->screen = DefaultScreen(w->dpy);
   w->root   = RootWindow(w->dpy, w->screen);
   
   w->dpy_width  = DisplayWidth(w->dpy, w->screen);
   w->dpy_height = DisplayHeight(w->dpy, w->screen);
   
   w->wm_state = XInternAtom(w->dpy, "WM_STATE", False);
   w->wm_change_state = XInternAtom(w->dpy, "WM_CHANGE_STATE", False);
   w->wm_protos   = XInternAtom(w->dpy, "WM_PROTOCOLS", False);
   w->wm_delete   = XInternAtom(w->dpy, "WM_DELETE_WINDOW", False);
   w->wm_cmapwins = XInternAtom(w->dpy, "WM_COLORMAP_WINDOWS", False);
   w->mb_theme    = XInternAtom(w->dpy, "_MB_THEME", False);
   w->mb_command    = XInternAtom(w->dpy, "_MB_COMMAND", False);
   
   w->head_client    = NULL; /* A 'general' pointer for circular list */ 
   w->focused_client = NULL; /* Points to the currently focused client */
   w->main_client    = NULL; /* Points to the current 'main' client
				- eg the 'BIG' window - */
   
   w->atoms[WINDOW_TYPE_TOOLBAR]
      = XInternAtom(w->dpy, "_NET_WM_WINDOW_TYPE_TOOLBAR",False);
   w->atoms[WINDOW_TYPE_DOCK]
      = XInternAtom(w->dpy, "_NET_WM_WINDOW_TYPE_DOCK",False);
   w->atoms[WINDOW_TYPE_DIALOG]
      = XInternAtom(w->dpy, "_NET_WM_WINDOW_TYPE_DIALOG",False);

   /* -- needed by xstroke - broke for now --
   w->atoms[REC_PROTOCOLS]
      = XInternAtom(w->dpy, "RECOGNIZE_PROTOCOLS", False);
   w->atoms[REC_KEY]
      = XInternAtom(w->dpy, "RECOGNIZE_KEY", False);
   */
   
   sattr.event_mask =  SubstructureRedirectMask
                       |SubstructureNotifyMask
                       |StructureNotifyMask;
   
   XChangeWindowAttributes(w->dpy, w->root, CWEventMask, &sattr);

   /* grab keyboard */
   for (j = keys_to_grab; *j; j++)
      XGrabKey(w->dpy, XKeysymToKeycode(w->dpy, *j), w->config->modifier,
	       w->root, True, GrabModeAsync, GrabModeAsync);
   
    w->theme = theme_new(w, w->config->theme_file);
    
    /* make the dock bar window */ 

    if (w->config->use_dock)
    {
       i = w->theme->win_border_width;
       
       dattr.override_redirect = True;
       dattr.border_pixel = (w->theme->bg_col).pixel;
       dattr.event_mask = ChildMask|ButtonPressMask|ExposureMask;
       
       w->dockwin = XCreateWindow(w->dpy, w->root,
				  0, w->dpy_height 
				  - w->config->dock_maxheight -(2*i),
				  w->dpy_width-(2*i),
				  w->config->dock_maxheight,
				  i,
				  DefaultDepth(w->dpy, w->screen),
				  CopyFromParent,
				  DefaultVisual(w->dpy, w->screen),
				  CWOverrideRedirect|CWBorderPixel|CWEventMask,
				  &dattr);
#ifdef USE_XFT
       w->dockwin_offset = max( w->theme->toolbar_xftfont->ascent
				+ w->theme->toolbar_xftfont->descent,
				w->theme->buttons[BUTTON_CLOSE].height, 0
				) + w->theme->padding;
#else
       w->dockwin_offset = max( w->theme->toolbar_font->ascent
				+ w->theme->toolbar_font->descent,
				w->theme->buttons[BUTTON_CLOSE].height, 0
				) + w->theme->padding;
#endif
       w->head_client = dockbar_client_new(w, w->dockwin);
    
       w->head_client->next = w->head_client;
       w->head_client->prev = w->head_client;
       w->head_client->configure(w->head_client);

       // broke for now ..
       //wu_allow_xstroke( w, w->dockwin);
       
       XMapWindow(w->dpy, w->dockwin);
    }
    
    w->flags = 0;
    
    return w;
}

void
wm_usage(char *progname)
{
   printf("usage: %s [options ...]\n", progname);
   printf("\t-display <string> \n");
   printf("\t-theme <string> \n");
   printf("\t-modifier <mod1|mod2|ctrl>\n");    
   printf("\t-use_dock <yes|no>\n");    
   printf("\t-dockmax  <integer>\n");    
   printf("\t-dockmin  <integer>\n");     
   printf("\t-use_titlebar <yes|no>\n");
   printf("\n");
   printf("Compile time options:\n");
#ifdef DEBUG
   printf("\tdebug build     yes\n");
#else
   printf("\tdebug build     no\n");
#endif

#ifdef USE_XFT
   printf("\tXFT support     yes\n");
#else
   printf("\tXFT support     no\n");
#endif
   printf("\n(c) 2002 Matthew Allum\n");
   

   exit(0);
}

void
wm_load_config(wm *w, int *argc, char *argv[])
{
   static XrmDatabase rDB, cmdlnDB;
   char *enviroment;
   char *type;
   char buffer[20];
   
   XrmValue value;
   
   static int opTableEntries = 7;
   static XrmOptionDescRec opTable[] = {
      {"-theme",       ".theme",           XrmoptionSepArg, (XPointer) NULL},
      {"-modifier",    ".modifier",        XrmoptionSepArg, (XPointer) NULL},
      {"-use_dock",    ".dock",            XrmoptionSepArg, (XPointer) NULL},
      {"-dockmax",     ".dock.height.max", XrmoptionSepArg, (XPointer) NULL},
      {"-dockmin",     ".dock.height.min", XrmoptionSepArg, (XPointer) NULL},
      {"-use_titlebar",".titlebar",        XrmoptionSepArg, (XPointer) NULL},
      {"-display",     ".display",         XrmoptionSepArg, (XPointer) NULL},
   };
   
   XrmInitialize();

   rDB = XrmGetFileDatabase(CONFDEFAULTS);   

   XrmParseCommand(&cmdlnDB, opTable, opTableEntries, "matchbox", argc, argv); 
   if (*argc != 1) wm_usage(argv[0]);

   XrmCombineDatabase(cmdlnDB, &rDB, True);

   if ((enviroment = getenv("XENVIROMENT")) == NULL)
   {
      enviroment = (char *)getenv("HOME");
      strcat(enviroment, "/.Xdefaults");
   }
   XrmCombineFileDatabase(enviroment, &rDB, True);

   if ( (w->config = malloc(sizeof(Wm_config))) == NULL)
      err("err out of memory");

   /* defaults */
   w->config->use_dock = True;
   w->config->dock_maxheight = 16;
   w->config->dock_minheight = 4;
   w->config->use_title = True;
   w->config->modifier  = Mod2Mask;
   w->config->display_name[0] = '\0';

   if (XrmGetResource(rDB, "matchbox.display",
		      "Matchbox.Display",
		      &type, &value) == True)
   {
      strncpy(w->config->display_name, value.addr, (int) value.size);
      w->config->display_name[value.size] = '\0';
      dbg("cmd line: got display");
   } else {
      if (getenv("DISPLAY") != NULL)
	 strcpy(w->config->display_name, (char *)getenv("DISPLAY"));
   }

   if (XrmGetResource(rDB, "matchbox.theme",
		      "Matchbox.Theme",
		      &type, &value) == True)
   {
      strncpy(w->config->theme_file, value.addr, (int) value.size);
      w->config->theme_file[value.size] = '\0';
      dbg("cmb line: got theme :%s", w->config->theme_file);
   } else {
      strcpy(w->config->theme_file, DEFAULTTHEME);
   }
   
   if (XrmGetResource(rDB, "matchbox.dock", "Matchbox.Dock",
		     &type, &value) == True)
   {
      if(strncmp(value.addr, "no", (int) value.size) == 0)
      {
	 dbg("TURNING DOCK OFF");
	 w->config->use_dock = False;
      }
   }

   if (XrmGetResource(rDB, "matchbox.dock.height.max",
		      "Matchbox.Dock.Height.max",
		      &type, &value) == True)
   {
      strncpy(buffer, value.addr, (int) value.size);
      buffer[value.size] = '\0';
      w->config->dock_maxheight = atoi(buffer);
   }

   if (XrmGetResource(rDB, "matchbox.dock.height.min",
		      "Matchbox.Dock.Height.min",
		      &type, &value) == True)
   {
      strncpy(buffer, value.addr, (int) value.size);
      buffer[value.size] = '\0';
      w->config->dock_minheight = atoi(buffer);
   }

   if (XrmGetResource(rDB, "matchbox.titlebar", "Matchbox.Titlebar",
		      &type, &value) == True)
   {
      if(strncmp(value.addr, "no", (int) value.size) == 0)
      {
	 dbg("TURNING TITLE OFF");
	 w->config->use_title = False;
      }
   }

   if (XrmGetResource(rDB, "matchbox.modifier", "Matchbox.modifier",
		      &type, &value) == True)
   {
      if(strncmp(value.addr, "mod1", (int) value.size) == 0)
      {
	 w->config->modifier = Mod1Mask;
      } else if(strncmp(value.addr, "mod2", (int) value.size) == 0)
      {
	 w->config->modifier = Mod2Mask;
      } else if(strncmp(value.addr, "ctrl", (int) value.size) == 0)
      {
	 w->config->modifier = ControlMask;
      }
   }

}

void
wm_init_existing(Wm *w)
{
   unsigned int nwins, i;
   Window dummyw1, dummyw2, *wins;
   XWindowAttributes attr;
   Client *c;
   
   XQueryTree(w->dpy, w->root, &dummyw1, &dummyw2, &wins, &nwins);
   for (i = 0; i < nwins; i++) {
      XGetWindowAttributes(w->dpy, wins[i], &attr);
      if (!attr.override_redirect && attr.map_state == IsViewable)
      {
	 c = wm_make_new_client(w, wins[i]);
	 c->ignore_unmap++; /* why do existing wins unmap - on reparent ? */
      }
   }
   XFree(wins);
}


Client*
wm_find_client(Wm *w, Window win, int mode)
{
    Client *c;

    if (w->head_client == NULL) return NULL;

    if (mode == FRAME) {
       START_CLIENT_LOOP(w,c);
       if (c->frame == win) return c;
       END_CLIENT_LOOP(w,c);
    } else {
       START_CLIENT_LOOP(w,c);
       if (c->mapped) dbg("find: %s is mapped", c->name);
       if (c->window == win) return c;
       END_CLIENT_LOOP(w,c);
    }    

    return NULL;
}


void
wm_event_loop(Wm* w)
{
    XEvent ev;

    for (;;) {
        XNextEvent(w->dpy, &ev);
        switch (ev.type) {
	   case ButtonPress:
	      dbg("event loop: button press");
	      wm_handle_button_event(w, &ev.xbutton); break;
	   case MapRequest:
	      dbg("event loop: map");
	      wm_handle_map_request(w, &ev.xmaprequest); break;
	   case UnmapNotify:
	      dbg("event loop: unmap");
	      wm_handle_unmap_event(w, &ev.xunmap); break;
	   case Expose:
	      dbg("event loop: expose");
	      wm_handle_expose_event(w, &ev.xexpose); break;
	   case DestroyNotify:
	      dbg("event loop: destroy");
	      wm_handle_destroy_event(w, &ev.xdestroywindow); break;
	   case ConfigureRequest:
	      wm_handle_configure_request(w, &ev.xconfigurerequest); break;
	   case ConfigureNotify:
	      wm_handle_configure_notify(w, &ev.xconfigure); break;
	   case ClientMessage:
              wm_handle_client_message(w, &ev.xclient); break;
	   case KeyPress:
	      dbg("got keypress");
	      wm_handle_keypress(w, &ev.xkey); break;
	   case PropertyNotify:
	       wm_handle_property_change(w, &ev.xproperty); break;
	      /* TODO add these
            case ColormapNotify:
                handle_colormap_change(&ev.xcolormap); break;
	      */
	  default:
	     //dbg("unhandled event %d",ev.type);
	     break;
	}
    }
}

void
wm_handle_button_event(Wm *w, XButtonEvent *e)
{
   Client *p;
   Client *c = wm_find_client(w, e->window, FRAME);
   if (c)
   {
      c->button_press(c,e);
   }
   if ( !c || ( c && c->type != menu && c->type != mainwin))
   {
    /* remove menu if its up */
      if (w->flags & MENU_FLAG)
      {
	 START_CLIENT_LOOP(w,p);
	    if (p->type == menu)
	    { select_client_destroy(p); return; }
	 END_CLIENT_LOOP(w,p);
      }
   }
}

void
wm_handle_keypress(Wm *w, XKeyEvent *e)
{
   Client *p;
   KeySym key = XKeycodeToKeysym(w->dpy,e->keycode,0);

   dbg("got a key");
   
   switch (key) {
      case XK_L:
	 XGrabServer(w->dpy);
	 XSync(w->dpy, False);	    
	 START_CLIENT_LOOP(w,p);
	 if (p->type == toolbar) p->hide(p);
	 END_CLIENT_LOOP(w,p);
	 XUngrabServer(w->dpy);
	 break;
      case XK_R:
	 XGrabServer(w->dpy);
	 XSync(w->dpy, False);	    
	 START_CLIENT_LOOP(w,p);
	 if (p->type == toolbar) p->show(p);
	 END_CLIENT_LOOP(w,p);
	 XUngrabServer(w->dpy);
	 break;
      case XK_l:
      case XK_Down:
	 XGrabServer(w->dpy);
	 XSync(w->dpy, False);	    
	 START_CLIENT_LOOP(w,p);
	 if (p->type == toolbar && p->window != w->dockwin) p->hide(p);
	 END_CLIENT_LOOP(w,p);
	 XUngrabServer(w->dpy);
	 break;
      case XK_r:
      case XK_Up:
	 XGrabServer(w->dpy);
	 XSync(w->dpy, False);	    
	 START_CLIENT_LOOP(w,p);
	 if (p->type == toolbar && p->window != w->dockwin) p->show(p);
	 END_CLIENT_LOOP(w,p);
	 XUngrabServer(w->dpy);
	 break;
      case XK_Return:
	    if (w->main_client)
	       client_deliver_delete(w->main_client);
	    break;
      case XK_Left:
      case XK_p:
	    if (w->main_client)
	    {
	       XGrabServer(w->dpy);
	       XSync(w->dpy, False);	    
	       p = client_get_prev(w->main_client, mainwin);
	       w->main_client->hide(w->main_client);
	       p->show(p);
	       XUngrabServer(w->dpy);
	    }
	 break;
      case XK_Tab:
      case XK_n:
      case XK_Right:
	 if (w->main_client)
	 {
	    XGrabServer(w->dpy);
	    XSync(w->dpy, False);	    
	    p = client_get_next(w->main_client, mainwin);
	    w->main_client->hide(w->main_client);
	    p->show(p);
	    XUngrabServer(w->dpy);
	 }
	 break;
   }
   
}

void
wm_handle_configure_notify(Wm *w, XConfigureEvent *e)
{
   Client *p;
   int height_diff, width_diff, b;

   dbg("configure notify - UH !"); 
   if (e->window == w->root) /* screen rotation ? */
   {
         dbg("configure notify event called on root");
	 if (e->width  != w->dpy_width ||
	     e->height != w->dpy_height)
	 {
	    dbg("Root has changed size %i x %i", e->width, e->height);
	    height_diff = e->height - w->dpy_height;
	    width_diff  = e->width  - w->dpy_width;
	    w->dpy_width = e->width; 
	    w->dpy_height = e->height;
	    b = w->theme->win_border_width;
	    XGrabServer(w->dpy);
	    
	    START_CLIENT_LOOP(w,p);
	    switch (p->type)
	    {
	       case mainwin :
		  p->width += width_diff;
		  p->height += height_diff;
		  break;
	       case toolbar :
		  p->width += width_diff;
		  p->y += height_diff;
		  break;
	       case dialog :
	       case override :
	       case docked:
		  p->y += height_diff;
	       case menu:
	       case detached:
		  break;
	    }
	    p->move_resize(p);
	    p->redraw(p, False);
	    client_deliver_config(p);
	    END_CLIENT_LOOP(w,p);

	    XSync(w->dpy, False);
	    XUngrabServer(w->dpy);

	 }
   }
      
}

void
wm_handle_configure_request(Wm *w, XConfigureRequestEvent *e)
{
   /* this needs work - ideally no window should be allowed to
      resize itself, only iconise its self
      exception is docked clients ...?
   */
   Client *c = wm_find_client(w, e->window, WINDOW);
   if (!c) return;
   dbg("configure request event called on win %s", c->name);
   client_deliver_config(c); /* may fix gtk-menu after rotation */
   //if (client_get_state(c) == IconicState)
   if (c->type == docked)
   {
      /* docked windows allowed to resize */

      XWindowChanges xwc;

      xwc.x = e->x;
      xwc.y = e->y;
      xwc.width = e->width;
      xwc.height = e->height;
      xwc.border_width = e->border_width;
      xwc.sibling = e->above;
      xwc.stack_mode = e->detail;
      
      XConfigureWindow(w->dpy, e->window,
		       e->value_mask, &xwc);
      

      
      /* HACK HACK HACK HACK - most probably wrong !!! 

      XWindowAttributes attr;
      XWindowChanges wc;
      long icccm_mask;
	 
      dbg("configure request: got dim %i x %i", e->width, e->height); 
      wc.width = e->width;
      c->width = e->width;
      wc.height = e->height;
      c->height = c->height;
      wc.sibling = e->above;
      wc.stack_mode = e->detail;
      //client_deliver_config(c);
      //XConfigureWindow(w->dpy, e->window, e->value_mask, &wc);
      //docked_client_move_resize(c);

      XGetWindowAttributes(w->dpy, c->window, &attr);
      dbg("configure request: attrib %i x %i", attr.width, attr.height);

      if ( !XGetWMNormalHints(w->dpy, c->window, c->size, &icccm_mask) )
      {
	 c->width = attr.width;
	 c->height = attr.height;
      } else {
	 if (c->size->flags &  PMinSize) {
	    c->width  = c->size->min_width;
	    c->height = c->size->min_height;
	 }
	 else if (c->size->flags & PBaseSize) {
	    c->width  = c->size->base_width;
	    c->height = c->size->base_height;
	 } else {
	    c->width = attr.width;
	    c->height = attr.height;
	 }
      }
      dbg("configure request: normal hints %i x %i", c->width, c->height);
      //client_deliver_config(c);
      */
      
   } else {
      c->show(c);
               /* THIS IS AN EVIL HACK FOR NOW,
		 -- need to check docs for the above and
	            detail members of the event
		  From what I read this morning, above and
		  detail correspond to sibling and stacking_order
		  in a ConfigureWindow request (CDW).
		  
		  Yes but when the window is 'raised' externally
		  it seems a CDW is recieved rather than a map_request
		  Try with utils/monolaunch to see what I mean. 
	       */
   }
}



void
wm_handle_map_request(Wm *w, XMapRequestEvent *e)
{
   Client *c = wm_find_client(w, e->window, WINDOW);

   dbg("got map request");
   if (!c) {
      wm_make_new_client(w, e->window);
   } else {
      dbg("mapping %s\n", c->name);
      c->show(c);
      
   }
}

void
wm_handle_unmap_event(Wm *w, XUnmapEvent *e)
{
   XEvent ev;
   Client *p;
   Client *c = wm_find_client(w, e->window, WINDOW);
   if (!c) return;
   dbg("unmap on %i", c->frame);
   if (c->ignore_unmap)
      c->ignore_unmap--;
   else
   {
      XGrabServer(w->dpy);
      if (c->frame != c->window && c->frame) XUnmapWindow(w->dpy, c->frame);
      XSync(w->dpy, 0);
      /* check if the window was destroyed */
      if (!XCheckTypedWindowEvent(w->dpy, c->window, DestroyNotify,&ev))
      {
	 // The above doesn't seem to work
	 // I need to peek into the event queue to see if a destroy event is
	 // on the way 
	 //client_set_state(c, WithdrawnState);
	 ;
      }
      
      if (c->type == mainwin)
      {
	 p = client_get_prev(c, mainwin);
	 if (p != c) p->show(p);
      }
      XUngrabServer(w->dpy);
      c->mapped = False;
   }
   //c->hide(c);
   //wm_remove_client(w, c);
}


void
wm_handle_expose_event(Wm *w, XExposeEvent *e)
{
   //XEvent ev;
   Client *c = wm_find_client(w, e->window, FRAME);
    
   if (c && e->count == 0)
   {
      dbg("expose on %s", c->name);
      c->redraw(c, TRUE); 		  /* redraw title */
   }
}

void
wm_handle_destroy_event(Wm *w, XDestroyWindowEvent *e)
{
    Client *c = wm_find_client(w, e->window, WINDOW);
    if (!c) { dbg("cant find destroyed window"); return; }
    dbg("destroy event called on %s", c->name);
    wm_remove_client(w, c);
}

void
wm_handle_client_message(Wm *w, XClientMessageEvent *e)
{
    Client *c = wm_find_client(w, e->window, WINDOW);
    dbg("got client message request");
   if (e->window == w->root)
   {
      if (e->message_type == w->mb_command)
      {
	 /* check data for actual command TODO
	    e->format == 32 && e->data.l[0] == IconicState
	 */
	    Atom realType;
	    unsigned long n;
	    unsigned long extra;
	    int format;
	    int status;
	    char * value;

	    status = XGetWindowProperty(w->dpy, DefaultRootWindow(w->dpy),
					w->mb_theme, 0L, 512L, False,
					AnyPropertyType, &realType, &format,
					&n, &extra, (unsigned char **) &value);
   
	    if (status != Success || value == 0 || *value == 0 || n == 0)
	    {
	       dbg("no _MB_THEME set on root window");
	    } else {
	       dbg("switching theme to %s", value);
	       theme_switch(w, w->theme, value);
	    }
      }
   }


   if (c && e->message_type == w->wm_change_state &&
       e->format == 32 && e->data.l[0] == IconicState)
      c->hide(c);
}

void
wm_handle_property_change(Wm *w, XPropertyEvent *e)
{
   Atom wm_type_prop = XInternAtom(w->dpy, "_NET_WM_WINDOW_TYPE", False);
   Client *c = wm_find_client(w, e->window, WINDOW);

   dbg("property change on %s %i %i", c->name, e->atom, wm_type_prop);
   
   
   if (!c) return;


   if (e->atom == wm_type_prop) /* Deal with windows changing type - urg!  */
   { 				
      Atom realType;
      unsigned long n;
      unsigned long extra;
      int format;
      int status;
      Atom * value = NULL;
      status = XGetWindowProperty(w->dpy, c->window, wm_type_prop,
				  0L, 1000000L,
				  0, XA_ATOM, &realType, &format,
				  &n, &extra, (unsigned char **) &value);
      if (status == Success)
      {
	 if (realType == XA_ATOM && format == 32)
	 {
	    if (value[0] == w->atoms[WINDOW_TYPE_DOCK]
		&& w->config->use_dock)
	    {
	       if (c->type != docked)
		  {
		     Client *p;
		     client_morph(c, docked);
		     p = client_get_prev(c, mainwin);
		     if (p) p->show(p);
			
		  }
	       return;
	    }
	    if (value[0] == w->atoms[WINDOW_TYPE_TOOLBAR])
	    {
	       if (c->type != toolbar) client_morph(c, toolbar);
	       return;
	    }

	    /* we dont know what its changed too, so it becomes a mainwin */
	    if (c->type != mainwin)
	       {
		  if (c->type == docked) /* if it was a dock
					     move the dock back */
		  {
		     Client *p;
		     for (p=c->next; p != c; p=p->next) 
			if (p->type == docked && p->x > c->x)
			{
			   p->x -= ( c->width + c->wm->theme->padding);
			   p->move_resize(p);
			   client_deliver_config(p);
			}
 		  }
		  client_morph(c, mainwin);
	       }
	    return;
	 }
      } else {
	 dbg("property chage: status FAILED");
      }
   }
      
   
   switch (e->atom) {
      case XA_WM_NAME:
	 if (c->name) XFree(c->name);
	 XFetchName(w->dpy, c->window, &c->name);
	 c->redraw(c, FALSE);
	 break;
   }


}

Client*
wm_make_new_client(Wm *w, Window win)
{
   Window trans_win;
   Atom prop, realType;
   unsigned long n;
   unsigned long extra;
   int format;
   int status;
   Atom * value = NULL;
   client *c = NULL, *t = NULL;
   XSetWindowAttributes  attribs;
   
   XGrabServer(w->dpy);

   /* find out what sort of window it wants to be ... */
   prop = XInternAtom(w->dpy, "_NET_WM_WINDOW_TYPE", True);
   if (prop != None)
   {
      status = XGetWindowProperty(w->dpy, win, prop, 0L, 1000000L,
				  0, XA_ATOM, &realType, &format,
				  &n, &extra, (unsigned char **) &value);
      if (status == Success)
      {
	 if (realType == XA_ATOM && format == 32)
	 {
	    if (value[0] == w->atoms[WINDOW_TYPE_DOCK]
		&& w->config->use_dock)
	    {
	       dbg("got dock atom\n");
	       c = docked_client_new(w, win);

	    }
	    if (value[0] == w->atoms[WINDOW_TYPE_TOOLBAR])
	    {
	       dbg("got dock atom\n");
	       c = toolbar_client_new(w, win);
	    }
	 } 
      }
   }

   /* check for transient */
   XGetTransientForHint(w->dpy, win, &trans_win);
   if ( trans_win && (trans_win !=  RootWindow(w->dpy, w->screen) ) )
   {
      t = wm_find_client(w, trans_win, WINDOW);
      if (t) c = dialog_client_new(w, win, t);
   }
   
   if (c == NULL) /* nothing else so must be a main client */
      c = main_client_new(w, win);

   c->configure(c);

   wm_gravitate(c, 1);

   c->reparent(c);
   c->move_resize(c);
   client_deliver_config(c);

   attribs.do_not_propagate_mask = ButtonPressMask | ButtonReleaseMask;
   attribs.save_under = False;
   XChangeWindowAttributes(w->dpy, c->window,
			   CWDontPropagate|CWSaveUnder, &attribs);

   
   c->show(c);
   
   XSync(w->dpy, False);
   XUngrabServer(w->dpy);
   return c;
}

void
wm_remove_client(Wm *w, Client *c)
{
    
    XGrabServer(c->wm->dpy);
    XSetErrorHandler(ignore_xerror);
    c->destroy(c);
    XFlush(w->dpy);
    XSync(w->dpy, False);
    XSetErrorHandler(handle_xerror);
    XUngrabServer(w->dpy);
}

void
wm_gravitate(Client *c, int multiplier)
{
    int dy = 0;
    /*
    int gravity = (c->size->flags & PWinGravity) ?
        c->size->win_gravity : NorthWestGravity;

    switch (gravity) {
        case NorthWestGravity:
        case NorthEastGravity:
        case NorthGravity: 10 ; break;
         case CenterGravity: 10 ; break;
    }
    */

    c->y += multiplier * dy;
}

void
wm_restack(Wm *w, Client *c, signed int amount)
{
   client *p;
   XGrabServer(w->dpy);
   for (p=c->next; p != c; p=p->next)
   {
      if (p->y <= c->y)
      {
	 dbg("restacking %s", p->name);
	 switch (p->type)
	 {
	    case mainwin :
	       p->height += amount;
	       p->move_resize(p);
	       client_deliver_config(p);
	       break;
	    case toolbar :
	       p->y += amount;
	       p->move_resize(p);
	       client_deliver_config(p);
	       break;
	    case dialog :
	    case override :
	    case docked:
	    case menu:
	    case detached:
	       break;
	 }
      }
   }
   XSync(w->dpy, False);
   XUngrabServer(w->dpy);
}

int
wm_total_toolbar_height(Client *c) /* find the toolbar height */
{
   client *p;
   int x,y,ww,h;
   int t = 0;

   START_CLIENT_LOOP(c->wm,p)
      if (p->type == toolbar && c != p)
      {
	 p->get_coverage(p, &x, &y, &ww, &h);
	 t += h;
      }
   END_CLIENT_LOOP(c->wm,p)

   return t;
}




