/*
 * Copyright (c) 2001-2003 Shiman Associates Inc. All Rights Reserved.
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include "mas/mas.h"
#include "mas/mas_core.h"

void
usage( void )
{
    fprintf(stderr, "usage: masset [-h|-?] [-d id | -n name] key\n");
    fprintf(stderr, "options:\n");
    fprintf(stderr, "   -d id    do 'set' on device instance 'id'\n");
    fprintf(stderr, "   -n name  do 'set' on first device named 'name'\n");
    fprintf(stderr, "   key      action to perform\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "valid keys:\n");
    fprintf(stderr, "  play\n");
    fprintf(stderr, "  play_on_mark\n");
    fprintf(stderr, "  stop\n");
    fprintf(stderr, "  pause\n");
    fprintf(stderr, "  next_track\n");
    fprintf(stderr, "  prev_track\n");
    fprintf(stderr, "  flush\n");
    fprintf(stderr, "  rebuffer\n");
    fprintf(stderr, "\n");  
}

int
main(int argc, char* argv[])
{
    int32 err;
    mas_device_t  device;
    struct mas_package arg;
    struct mas_package nugget;
    char name[256];
    char* key;
    int32 device_instance;
    int has_name = FALSE;
    int has_id = FALSE;
    int parse_err = FALSE;
    char c;
    int done_opt = FALSE;
    
    
    while ( !done_opt )
    {
        if ( ( c = getopt( argc, argv, "h?d:n:" ) ) != EOF )
        {
            switch (c)
            {
                case 'n':
                if ( has_id )
                {
                    fprintf(stderr, "-d and -n options are mutually exclusive\n\n");
                    parse_err++;
                }
                else
                {
                    strncpy( name, optarg, 255 );
                    has_name = TRUE;
                    done_opt = TRUE;
                }
            
                break;
                case 'd':
                if ( has_name )
                {
                    fprintf(stderr, "-d and -n options are mutually exclusive\n\n");
                    parse_err++;
                }
                else
                {
                    device_instance = atoi( optarg );
                    has_id = TRUE;
                    done_opt = TRUE;
                }
                break;
                case 'h':
                parse_err++;
                break;
                case '?':
                parse_err++;
                break;
                default:
                parse_err++;
                break;
            }
            if ( parse_err )
            {
                usage();
                exit(2);
            }
        }
        else done_opt = TRUE;
    }
    
    /* we need a key, at least.  */
    if ( argc < 2 )
    {
        usage();
        exit(2);
    }

    key = argv[optind++];


    masc_log_verbosity( MAS_VERBLVL_INFO );

    /* initiate contact with MAS */
    err = mas_init();
    if (err < 0)
    {
	printf("\nconnection with server failed.\n");
	exit(1);
    }

    /*** retrieve device handle as specified */
    if ( !has_id && !has_name )
    {
        usage();
        exit(2);
    }
    else if ( has_name ) /* choose by name */
    {
        err = mas_asm_get_device_by_name( name, &device );
        if ( err < 0 )
        {
            masc_logerror( err, "can't retrieve device handle for '%s'", name);
            exit(1);
        }
        
    }
    else /* choose by ID number */
    {
        masc_log_message(0, "device id not supported yet");
        exit(1);
    }
    
    /* set the nugget */
    masc_entering_log_level("sending query");
    masc_log_message( 0, "key: %s", key );

    err = -1;
    
    while( 1 )
    {
        /* leaving out for now: set_file( mas_device_t device, char* filename ); */

        if( !strcmp(key, "play") )
        {
            err = mas_source_play( device );
            break;
        }

        if( !strcmp(key, "play_on_mark") )
        {
            err = mas_source_play_on_mark( device );
            break;
        }

        if( !strcmp(key, "stop") )
        {
            err = mas_source_stop( device );
            break;
        }

        if( !strcmp(key, "pause") )
        {
            err = mas_source_pause( device );
            break;
        }

        if( !strcmp(key, "next_track") )
        {
            err = mas_source_next_track( device );
            break;
        }

        if( !strcmp(key, "prev_track") )
        {
            err = mas_source_prev_track( device );
            break;
        }

        if( !strcmp(key, "flush") )
        {
            err = mas_source_flush( device );
            break;
        }

        if( !strcmp(key, "rebuffer") )
        {
            err = mas_source_rebuffer( device );
            break;
        }
        
        break;
    }

    if ( err < 0 )
    {
        masc_logerror( err, "invalid key '%s'?", key);
        exit(1);
    }
    masc_exiting_log_level();
    
    exit(0);
}
