/*
 * Copyright (c) 2001-2003 Shiman Associates Inc. All Rights Reserved.
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */
/*
 * $Id: mas_event.c,v 1.3 2003/06/26 20:53:19 rocko Exp $
 *
 * Copyright (c) 2000, 2001 by Shiman Associates Inc. and Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions: The above
 * copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the names of the authors or
 * copyright holders shall not be used in advertising or otherwise to
 * promote the sale, use or other dealings in this Software without
 * prior written authorization from the authors or copyright holders,
 * as applicable.
 *
 * All trademarks and registered trademarks mentioned herein are the
 * property of their respective owners. No right, title or interest in
 * or to any trademark, service mark, logo or trade name of the
 * authors or copyright holders or their licensors is granted.
 *
 */

#include <stdlib.h>
#include <string.h>
#include "mas_common.h"

int32
masc_setup_event( struct mas_event *event )
{
    if ( event == 0 )
	return mas_error(MERR_NULLPTR);

    memset( event, 0, sizeof *event );

    return 0;
}

int32
masc_strike_event( struct mas_event *event )
{
    if ( event == 0 )
	return mas_error(MERR_NULLPTR);

    if (event->predicate)
    {
        masc_rtfree(event->predicate);
        event->predicate = NULL;
    }
    
    if (event->port_dependencies)
    {
        masc_rtfree(event->port_dependencies);
        event->port_dependencies = NULL;
    }
    
    if (event->action_name)
    {
        masc_rtfree(event->action_name);
        event->action_name = NULL;
    }
    
    /* connect events on either side in queue */
    if (event->prev != 0)
        event->prev->next = event->next;
    if (event->next != 0)
        event->next->prev = event->prev;
    
    return 0;
}

/***************************************************************************
 * masc_insert_mas_event
 *
 * arguments:
 *  1. pointer to struct mas_event in queue, insert after this event
 *  2. pointer to struct mas_event, event to insert
 *
 *  Inserts the event after the specified event in a queue of events.
 *
 * returns: MERR_NULLPTR if insert_after pointer is zero
 *          MERR_NULLPTR if event pointer is zero
 *
 ***************************************************************************/
int32
masc_insert_event(struct mas_event* insert_after, struct mas_event* event)
{
    if ( insert_after == 0 ) return mas_error(MERR_NULLPTR);
    if ( event == 0 ) return mas_error(MERR_NULLPTR);

    event->next = insert_after->next;
    event->prev = insert_after;
    if ( insert_after->next )
        insert_after->next->prev = event;
    insert_after->next = event;

    return 0;
}

/***************************************************************************
 * masc_append_mas_event
 *
 * arguments:
 *  1. pointer to struct mas_event, head of a queue
 *  2. pointer to struct mas_event, event to append
 *
 *  Appends the event to the tail of a queue of events.
 *
 * returns: MERR_NULLPTR if head pointer is zero
 *          MERR_NULLPTR if event pointer is zero
 *
 ***************************************************************************/
int32
masc_append_event(struct mas_event* head, struct mas_event* event)
{
    struct mas_event* tail = head;

    if ( tail == 0 ) return mas_error(MERR_NULLPTR);
    if ( event == 0 ) return mas_error(MERR_NULLPTR);

    while( tail->next != NULL ) tail = tail->next;

    tail->next = event;
    event->prev = tail;

    return 0;
}

int32
masc_pack_event( struct mas_package *package, struct mas_event *event )
{
    masc_pushk_int32( package, "di", event->device_instance );
    masc_pushk_string( package, "an", event->action_name );
    if ( event->predicate )
        masc_pushk_payload( package, "pred", event->predicate, event->predicate_length );
    masc_pushk_int8( package, "t", event->type );
    if ( event->port_dependencies )
        masc_pushk_payload( package, "pdep", event->port_dependencies, event->num_port_dependencies * sizeof (int32) );
    if ( event->period )
        masc_pushk_uint32( package, "prd", event->period );
    if ( event->clkid )
        masc_pushk_int32( package, "clk", event->clkid );
    masc_pushk_int32( package, "pri", event->priority );
    masc_finalize_package( package );
    
    return package->error;
}

int32
masc_unpack_event( struct mas_package *package, struct mas_event *event )
{
    memset( event, 0, sizeof *event );
    
    if ( masc_test_members( package, "lacl", "di", "an", "t", "pri" ) != 0 )
        return mas_error(MERR_INVALID);
    
    masc_pullk_int32( package, "di", &event->device_instance );
    masc_pullk_string( package, "an", &event->action_name, TRUE );
    
    if ( masc_test_key( package, "pred" ) == 0 )
        masc_pullk_payload( package, "pred", &event->predicate, &event->predicate_length, TRUE );
    masc_pullk_int8( package, "t", &event->type );
    
    if ( masc_test_key( package, "pdep" ) == 0 )
    {
        masc_pullk_payload( package, "pdep", (void**)&event->port_dependencies, &event->num_port_dependencies, TRUE );
        event->num_port_dependencies /= sizeof (int32);
    }
    
    if ( masc_test_key( package, "prd" ) == 0 )
        masc_pullk_uint32( package, "prd", &event->period );
    
    if ( masc_test_key( package, "clk" ) == 0 )
        masc_pullk_int32( package, "clk", &event->clkid );
    
    masc_pullk_int32( package, "pri", &event->priority );

    return 0;
}
