/* Copyright (C) 1996 Bjoern Beutel. */

/* Description. =============================================================*/

/* This module contains options that can be set by both malaga and mallex. */

/* Includes. ================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <setjmp.h>
#include <string.h>
#include <glib.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "symbols.h"
#include "input.h"
#include "commands.h"
#include "rule_type.h"
#include "rules.h"
#include "value_parser.h"
#include "hangul.h"
#include "scanner.h"
#include "options.h"

/* Global variables. ========================================================*/

bool_t auto_variables; /* TRUE if variables are shown automatically */

bool_t use_display; 
/* TRUE if "variables" and "result" use the display process */

/* Commands. ================================================================*/

static void 
do_sort_records_option( string_t arguments )
/* Select the order in which attributes in a record are printed. */
{
  string_t argument;

  if (*arguments == EOS) 
  { 
    printf( "sort-records: %s\n", 
            attribute_order == INTERNAL_ORDER ? "internal" :
            attribute_order == ALPHABETIC_ORDER ? "alphabetic" :
            attribute_order == DEFINITION_ORDER ? "definition" : "" );
  }
  else 
  { 
    argument = parse_word( &arguments );
    if (strcmp_no_case( argument, "internal" ) == 0) 
      attribute_order = INTERNAL_ORDER;
    else if (strcmp_no_case( argument, "alphabetic" ) == 0)
      attribute_order = ALPHABETIC_ORDER;
    else if (strcmp_no_case( argument, "definition" ) == 0)
      attribute_order = DEFINITION_ORDER;
    else
    { 
      complain( "\"internal\", \"alphabetic\" or \"definition\" expected, "
		"not \"%s\".", argument );
    }
    free_mem( &argument );
  }
  parse_end( &arguments );
}

command_t sort_records_option = 
{ 
  "sort-records", do_sort_records_option,
  "Usage:\n"
  "  set sort-records internal -- Print attributes in internal order.\n"
  "  set sort-records alphabetic -- Print attributes in alphabetic order.\n"
  "  set sort-records definition -- Print attributes in symbol-table order.\n"
};

/*---------------------------------------------------------------------------*/

static void 
do_hidden_option( string_t arguments )
/* Hide specified attributes when printing values. */
{
  symbol_t *attribs;
  int_t i;
  string_t argument;

  if (*arguments == EOS) 
  { 
    /* Get list of hidden attributes, terminated by value SYMBOL_MAX. */
    attribs = get_hidden_attributes();

    printf( "hidden:" );
    if (attribs[0] == SYMBOL_MAX) 
      printf( " (none)" );
    else 
    { 
      for (i = 0; attribs[i] != SYMBOL_MAX; i++) 
	printf( " %s", get_symbol_name( attribs[i] ) );
    }
    printf( "\n" );
    free_mem( &attribs );
  } 
  else 
  { 
    while (*arguments != EOS) 
    { 
      argument = parse_word( &arguments );
      if (strcmp_no_case( argument, "none" ) == 0) 
	clear_hidden_attributes();
      else if (*argument == '+') 
	add_hidden_attribute( find_symbol( argument + 1 ) );
      else if (*argument == '-') 
	remove_hidden_attribute( find_symbol( argument + 1 ) );
      else
      {
	complain( "\"+...\", \"-...\" or \"none\" expected, not \"%s\".", 
		  argument );
      }
      free_mem( &argument );
    }
  }
}

command_t hidden_option = 
{ 
  "hidden", do_hidden_option,
  "Set hidden attributes in feature structures.\n"
  "Usage: set hidden ARGUMENT...\n"
  "ARGUMENT may be:\n"
  "  none -- Don't hide any attributes.\n"
  "  +ATTRIBUTE -- Hide ATTRIBUTE.\n"
  "  -ATTRIBUTE -- Don't hide ATTRIBUTE any longer.\n"
};

/*---------------------------------------------------------------------------*/

static void 
do_roman_hangul_option( string_t arguments )
/* Switch for Latin Hangul transcription. */
{
  if (*arguments == EOS) 
    printf( "roman-hangul: %s\n", (roman_hangul ? "yes" : "no") );
  else 
    roman_hangul = parse_yes_no( &arguments );
  parse_end( &arguments );
}

command_t roman_hangul_option = 
{ 
  "roman-hangul", do_roman_hangul_option,
  "This option only has an effect when the project file uses the option\n"
  "\"split-hangul-syllables: yes\".\n"
  "Usage:\n"
  "  set roman-hangul yes -- Display output in romanised Hangul.\n"
  "  set roman-hangul no -- Display output in Hangul syllables.\n"
};

/*---------------------------------------------------------------------------*/

static void 
do_use_display_option( string_t arguments )
/* Determine whether output is shown using the Display process. */
{
  bool_t u;

  if (*arguments == EOS) 
    printf( "use-display: %s\n", (use_display ? "yes" : "no") );
  else
  {
    u = parse_yes_no( &arguments );
    if (getenv("DISPLAY") != NULL)
      use_display = u;
    else if (in_command_loop)
      complain( "Environment variable DISPLAY is not set.");
  }
  parse_end( &arguments );
}

command_t use_display_option = 
{ 
  "use-display", do_use_display_option,
  "Usage:\n"
  "  set use-display yes -- Show output using display process.\n"
  "  set use-display no -- Show output using malaga itself.\n"
};

/*---------------------------------------------------------------------------*/

static void 
do_auto_variables_option( string_t arguments )
/* Determine if variables are displayed in debug mode. */
{
  if (*arguments == EOS) 
    printf( "auto-variables: %s\n", (auto_variables ? "yes" : "no") );
  else 
    auto_variables = parse_yes_no( &arguments );
  parse_end( &arguments );
}

command_t auto_variables_option = 
{ 
  "auto-variables variables", do_auto_variables_option,
  "Usage:\n"
  "  set auto-variables yes -- Show variables automatically in debug mode.\n"
  "  set auto-variables no -- Don't show variables automatically.\n"
};

/*---------------------------------------------------------------------------*/

static void 
do_switch_option( string_t arguments )
/* Set switches that can be read from Malaga rules. */
{
  symbol_t key;
  value_t value;
  string_t value_string, key_name;

  if (*arguments == EOS) 
  { 
    /* No switch name follows: list all switches. */
    get_first_switch( &value, &key );
    if (value == NULL) 
      printf( "switch: (none defined)\n" );
    else 
    { 
      do 
      { 
	key_name = get_symbol_name( key );
        value_string = value_to_readable( value, FALSE, 
					  11 + g_utf8_strlen( key_name, -1 ) );
        printf( "switch \"%s\": %s\n", key_name, value_string );
        free_mem( &value_string );
        get_next_switch( &value, &key );
      } while (value != NULL);
    }
  } 
  else 
  { 
    /* A switch name follows: read it. */
    key_name = parse_word( &arguments );
    if (*arguments == EOS) 
    { 
      /* Print value of given switch. */
      value = get_switch( find_symbol( key_name ) );
      value_string = value_to_readable( value, FALSE, 
					11 + g_utf8_strlen( key_name, -1 ) );
      printf( "switch \"%s\": %s\n", key_name, value_string );
      free_mem( &value_string );
    } 
    else 
    { 
      /* A value follows: read it. */
      set_scanner_input( arguments );
      TRY 
      { 
	parse_a_value();
	test_token( EOF );
      } 
      FINALLY 
	set_scanner_input( NULL );
      END_TRY;
      set_switch( find_symbol( key_name ), value_stack[ --top ] );
    }
    free_mem( &key_name );
  }
}

command_t switch_option = 
{ 
  "switch", do_switch_option, 
  "Set a switch that can be read from Malaga rule files.\n"
  "Usage: set switch SWITCH VALUE\n"
};

/* End of file. =============================================================*/
