// Copyright (C) 1994 The New York Group Theory Cooperative
// See magnus/doc/COPYRIGHT for the full notice.

// Contents: Definition of SubgroupRep and Subgroup classes.
//
// * provisional one-class implementation of the notion of subgroup
//
// Principal Authors: Stephane Collart, Roger Needham
//
// Status: Draft
//
// Description:
//
//   A subgroup stores a shared-reference `copy' of its parent group.
//   Enhancements to the parent group are thus available to its
//   subgroups. The subgroup is generated by (formal) words interpretable
//   in the generators of the parent group.
//
// Caveats:
//
// * If adding data members which store information computed about the
//   the subgroup, make sure that subgroup manipulators which modify the
//   semantics of the subgroup (for instance addGenerator()) either
//   recompute or erase any data which might become invalid.
//
// Restrictions:
//
// * The subgroup does not store names for its generators (generating
//   words): as the subgroup is mutable, this would involve finicky work.
//
// Bugs:
//
// * Being stored as a set, the generators do not have a fixed controllable
//   order when generators are added or deleted, making the meaning of
//   of formal words in the subgroup generators vary unpredictably.
//
// Revision History:
//
// * First implemented: oct 94.
//
// Notes:
//
// * If it becomes necessary to initialise subgroups with unspecified parent
//   groups, several alternatives are possible
//   - Group can be given a default constructor
//   - SubgroupRep can be made to derive from a representation
//     `UnspecifiedSubgroupRep' not requiring a parent group
//
// Questions:
//
// * It might be better to store the subgroup generators as a vector, in
//   order to have faster permanent direct access, for instance for
//   interpreting subgroup elements in the parent group. This would
//   make adjoining and removing generators less efficient. Another
//   possibility is taking seperate representations.
//

#ifndef _SUBGROUP_H_
#define _SUBGROUP_H_

#include "FGGroupRep.h"
#include "FGGroup.h"
#include "FPGroup.h"
#include "File.h"
#include "CosetEnumerator.h"

struct SubgroupRep : FGGroupRep {

  // Constructors:

  SubgroupRep( const FGGroup& parent )
  : FGGroupRep(0),
	 theParentGroup(parent)
  { }

  SubgroupRep( const FGGroup& parent, const VectorOf<Word>& gens )
  : FGGroupRep( gens.length() ),
	 theParentGroup( parent ),
	 theGenerators( gens )
  { }

  SubgroupRep( const FGGroup& parent, const SetOf<Word>& gens );
  // @rn Can do without this when we have container class conversions.


  // Representation methods:

  PureRep* clone( ) const { return new SubgroupRep(*this); }
  // overrides PureRep* FGGroupRep::clone();

  static const Type theSubgroupType;

  static Type type( ) { return theSubgroupType; }

  Type actualType( ) const { return type(); }
  // Overrides FGGroupRep::actualType( )


  // Operators:

private:

  SubgroupRep& operator = ( const SubgroupRep& );
  // disable assignment

public:

  // Methods dealing with group structure:
  // @stc these are implemented provisionally in a trivial way
 
  int order( ) const { return -1; }
  // overrides FGGroupRep::...

  Trichotomy isTrivial( ) const { return DONTKNOW; }
  // overrides FGGroupRep::...

  Trichotomy isFinite( ) const { return DONTKNOW; }
  // overrides FGGroupRep::...

  Trichotomy isInfinite( ) const { return DONTKNOW; }
  // overrides FGGroupRep::...

  Trichotomy isAbelian( ) const { return DONTKNOW; } //@rn can do
  // overrides FGGroupRep::...

  // Methods dealing with group elements:

  // Inherited from FGGroupRep:
  // Elt makeIdentity( ) const;

  // Inherited from FGGroupRep:
  // Bool isSyntacticIdentity(const Elt&) const

  Trichotomy areEqual(const Elt&, const Elt&) const { return DONTKNOW; } //@rn can do
  // overrides FGGroupRep::...
  // @stc provisional trivial implementation

  // Inherited from FGGroupRep:
  // Elt firstElt( ) const

  // Inherited from FGGroupRep:
  // Elt nextElt(const Elt&) const
  // beware: this deals with formal words in the subgroup generators:
  // to get an element in the parent group, apply eval
	 
  // Inherited from FGGroupRep:
  // Elt multiply(const Elt&, const Elt&) const
  // result is as actual type of args: if formal words in subgroup
  // generators, then such; if elt in parent group, then such
  // if mixed, then nonsense

  // Inherited from FGGroupRep:
  // Elt inverseOf(const Elt&) const
  // @stc this should really override with appropriate algorithm
	 
  // Inherited from FGGroupRep:
  // Elt raiseToPower(const Elt&, int) const
  // @stc this should really override with appropriate algorithm

  // Inherited from FGGroupRep:
  // Elt conjugateBy(const Elt&, const Elt&) const
  // @stc this should really override with appropriate algorithm

  // Inherited from FGGroupRep:
  // Elt commutator(const Elt&, const Elt&) const
  // @stc this should really override with appropriate algorithm

  Elt eval( const Word& w ) const;
  // overrides FGGroupRep::...
 
  Trichotomy wordProblem( const Word& w ) const;
  // overrides FGGroupRep::...
 
  Trichotomy conjugacyProblem( const Word& u, const Word& v ) const;
  // overrides FGGroupRep::...

  Word findRelator( );

  Bool redundantRelator(const Word& u);
  // This tries as best it can to decide whether the given word is a
  // product of conjugates of words from relators.

  // I/O:
 
  void printOn(ostream&) const { }
  // overrides FGGroupRep::...
  // @stc provisional

  virtual GroupRep* readFrom(istream&, Chars&) const { return 0; }
  // overrides FGGroupRep::...
  // @stc provisional: do not call...

  // Inherited from FGGroupRep:
  // virtual void printGenerator( ostream& ostr, int n ) const;
  // virtual void printGenerators( ostream& ostr ) const;
  // virtual void printWord( ostream& ostr, const Word& w ) const;


  // Data members:

  // Inherited from FGGroupRep:
  // int theOrder;
  // int theNumberOfGenerators;
  // VectorOf<Chars> theNamesOfGenerators;

  FGGroup theParentGroup;

  VectorOf<Word> theGenerators;

  // @rn Provisional members which support finding relators:

  Word lastWordTried;
  // The last cyclically reduced word we checked for relatorness.
  // Start with the empty word.

  SetOf<Word> relators;  // The relators we've found so far.

};


//------------------------- Subgroup ---------------------------//


class Subgroup : public FGGroup {

public:

  ///////////////////////////////////////////////////////
  //                                                   //
  //  Constructors                                     //
  //                                                   //
  ///////////////////////////////////////////////////////

  // @stc at some point, this default construtor may be wanted (see notes)
  // Subgroup( ) : FGGroup( new SubgroupRep() ) { }
  // to initialise an unspecified subgroup

  Subgroup( const FGGroup& parent )
  : FGGroup( new SubgroupRep( parent ) )
  { }
  // to make the trivial subgroup

  Subgroup( const FGGroup& parent, const VectorOf<Word>& gens )
  : FGGroup( new SubgroupRep(parent,gens) )
  { }
  // to make a finitely generated subgroup with a vector of unnamed
  // generators

  Subgroup( const FGGroup& parent, const SetOf<Word>& gens )
  : FGGroup( new SubgroupRep(parent,gens) )
  { }
  // to make a finitely generated subgroup with a set of unnamed
  // generators


  ///////////////////////////////////////////////////////
  //                                                   //
  //  Accessors                                        //
  //                                                   //
  ///////////////////////////////////////////////////////

  static Type type( ) { return SubgroupRep::type(); }

  // Inherited from FGGroup:
  // Chars FGGroup::nameOfGenerator( int i ) const;
  // VectorOf<Chars> FGGroup::namesOfGenerators( ) const;

  const FGGroup& parentGroup( ) const { return look()->theParentGroup; }

  const VectorOf<Word>& generators( ) const { return look()->theGenerators; }

  void setParentGroup( const Group& g ) { change()->theParentGroup = g; }

  void setGenerators( const VectorOf<Word>& s ) { change()->theGenerators = s; }

public:

  ///////////////////////////////////////////////////////
  //                                                   //
  //  Subgroup manipulators                            //
  //                                                   //
  ///////////////////////////////////////////////////////

  // NB manipulators which modify the semantics of the subgroup must
  // modify all necessary stored data to maintain consistency

  Subgroup& addGenerator( const Word& w );

  Subgroup& deleteGenerator( const Word& w );

  ///////////////////////////////////////////////////////
  //                                                   //
  //  Group structure methods                          //
  //                                                   //
  ///////////////////////////////////////////////////////
 
  // Inherited from FGGroup:
  // int order( ) const;             // pseudo-virtual
  // Trichotomy isTrivial( ) const;  // pseudo-virtual
  // Trichotomy isFinite( ) const;   // pseudo-virtual
  // Trichotomy isInfinite( ) const; // pseudo-virtual
  // Trichotomy isAbelian( ) const;  // pseudo-virtual
 
  ///////////////////////////////////////////////////////
  //                                                   //
  //  Methods which deal with group elements           //
  //                                                   //
  ///////////////////////////////////////////////////////

  // Inherited from FGGroup:
  // Elt makeIdentity( ) const;                               // pseudo-virtual
  // Bool isSyntacticIdentity(const Elt& e) const;            // pseudo-virtual
  // Trichotomy areEqual(const Elt& e1, const Elt& e2) const; // pseudo-virtual
  // Elt firstElt( ) const;                                   // pseudo-virtual
  // Elt nextElt(const Elt& e) const;                         // pseudo-virtual
  // Elt multiply(const Elt& e1, const Elt& e2) const;        // pseudo-virtual
  // Elt inverseOf(const Elt& e) const;                       // pseudo-virtual
  // Elt raiseToPower(const Elt& e, int n) const;             // pseudo-virtual
  // Elt conjugateBy(const Elt& e1, const Elt& e2) const;     // pseudo-virtual
  // Elt commutator(const Elt& e1, const Elt& e2) const;      // pseudo-virtual
  // Elt eval( const Word& w ) const;                         // pseudo-virtual
  // Trichotomy wordProblem( const Word& w ) const;           // pseudo-virtual
  // Trichotomy conjugacyProblem( const Word& u, const Word& v ) const;
                                                              // pseudo-virtual

  Word findRelator( ) { return enhance()->findRelator(); }
  // Attempts to find and return a relator for this group.
  // The words returned by successive calls will be distinct, but
  // they may be redundant as relators.
  // @rn We could use some means of querying whether this has any
  //     chance at all, e.g., is there at least a partial soln of the
  //     word problem for the parent group?

  Bool redundantRelator(const Word& u) {
	 return enhance()->redundantRelator(u);
  }
  
  ///////////////////////////////////////////////////////
  //                                                   //
  //  Methods which deal with sets of group elements   //
  //                                                   //
  ///////////////////////////////////////////////////////
 
  // Inherited from FGGroup:
  // SetOf<Elt> setMultiply(const SetOf<Elt>& S1, const SetOf<Elt>& S2) const;
  // SetOf<Elt> setMultiply(const Elt& e, const SetOf<Elt>& S) const;
  // SetOf<Elt> setMultiply(const SetOf<Elt>& S, const Elt& e) const;
  // SetOf<Elt> conjugateBy(const SetOf<Elt>& S1, const SetOf<Elt>& S2) const;
  // SetOf<Elt> conjugateBy(const Elt& e, const SetOf<Elt>& S) const;
  // SetOf<Elt> conjugateBy(const SetOf<Elt>& S, const Elt& e) const;
  // void closeUnderInverses(SetOf<Elt>& S) const;
  // void closeUnderCyclicPermutations(SetOf<Word>& S) const;
 
  ///////////////////////////////////////////////////////
  //                                                   //
  //  Representation access methods                    //
  //                                                   //
  ///////////////////////////////////////////////////////
 
private:

  // Shadow representation accessors to get representations of the
  // right type in the members of this class:

  const SubgroupRep* look( ) const {
	 return (const SubgroupRep*)GenericObject::look();
  }
  SubgroupRep* enhance( ) const {
	 return (SubgroupRep*)GenericObject::enhance();
  }
  SubgroupRep* change( ) {
	 return (SubgroupRep*)GenericObject::change();
  }

  // Special wrapping constructor to wrap new representations (returned
  // by eg. delegated methods) and for GenericObject initialisation by derived
  // classes:

  Subgroup( SubgroupRep* newrep ) : FGGroup(newrep) { }

};

#endif







