#												vim:sw=4:ts=4
# $Id: yaml.y.rb,v 1.12 2003/01/13 02:57:40 whythluckystiff Exp $
#
# 2003 rev. YAML Parser
# Grammar in Racc
#
# A thousand handshakes to Racc's author Minero Aoki.
#

class YAML::Parser

rule

atom	: word_rep
	 	| struct_rep
	 	| ANCHOR atom								
        { 
            if @options[:Model] == :Generic
                val[1].anchor = val[0]   
            end
            result = @anchor[ val[0] ] = val[1] 
        }
		| ALIAS										{ result = @anchor[ val[0] ] }

atom_or_empty   : atom
                | { result = new_node( 'null', nil ) }

#
# Words are broken out to distinguish them
# as keys in implicit maps and valid elements
# for the inline structures
#
word_rep	: TRANSFER word_rep						{ result = attach_transfer( val[0], val[1] ) }
			| WORD                                  { result = new_node( 'str', val[0] ) }
            | PLAIN                                 { result = process_plain( val[0] ) }
            | IOPEN PLAIN IEND                      { result = process_plain( val[1] ) }
			| "+"									{ result = new_node( 'boolean', true ) }
			| "-"									{ result = new_node( 'boolean', false ) }

#
# Any of these structures can be used as
# complex keys
#
struct_rep	: TRANSFER struct_rep					{ result = attach_transfer( val[0], val[1] ) }
			| scalar_block
			| implicit_seq
			| inline_seq
			| IOPEN inline_seq IEND                 { result = val[1] }
			| implicit_map
			| inline_map
			| IOPEN inline_map IEND                 { result = val[1] }

#
# Folding and indentation is handled in the tokenizer.
# We handle chomping here, though.
#
scalar_block	: IOPEN FSTART FOLD IEND
			 	{ 
                    result = new_node( 'str', process_block( val[1], val[2] ) )
				}
                | IOPEN FSTART IEND
                {
                    result = new_node( 'str', '' )
                }

#
# Implicit sequence 
#
implicit_seq	: IOPEN in_implicit_seq	IEND	{ result = new_node( 'seq', val[1] ) }

basic_seq       : '-' atom_or_empty             { result = val[1] }

in_implicit_seq : basic_seq 					{ result = val }
				| in_implicit_seq INDENT basic_seq
				{ 
					result.push val[2]
				}

#
# Inline sequences
#
inline_seq		: '[' in_inline_seq ']'  			{ result = new_node( 'seq', val[1] ) }
				| '[' ']'        					{ result = new_node( 'seq', [] ) }

in_inline_seq   : atom          					{ result = val }
                | ',' atom_or_empty                 { result = [ result = new_node( 'null', nil ), val[1] ] } 
                | in_inline_seq ',' atom_or_empty   { result.push val[2] }

#
# Implicit maps
#
implicit_map	: IOPEN in_implicit_map IEND		{ result = new_node( 'map', val[1] ) }

basic_mapping	: word_rep ':' atom_or_empty
				{ 
                    if val[0] == '<<'
                        result = [ :MERGE, val[2] ]
                    else
                        result = { val[0] => val[2] }
                    end
				}
				| '=' ':' atom_or_empty
				{
					result = [ :DEFAULT, val[2] ]
				}

complex_mapping : basic_mapping
				| '?' atom INDENT ':' atom_or_empty
				{
					result = { val[1] => val[4] }
				}

in_implicit_map : complex_mapping
				{
                    result = hash_update( {}, val[0] )
				}
				| in_implicit_map INDENT complex_mapping
				{
					result = hash_update( val[0], val[2] )
				}

#
# Inline maps
#
inline_map		: '{' in_inline_map '}'   			{ result = new_node( 'map', val[1] ) }
          		| '{' '}'							{ result = new_node( 'map', Hash.new ) }
         
in_inline_map	: basic_mapping 
				{
					result = hash_update( {}, val[0] )
				}
				| in_inline_map ',' basic_mapping
				{
					result = hash_update( val[0], val[2] )
				}

end

---- header = src/emitter.rb

---- inner

attr_accessor :finished, :directives, :options

#
# Reset the directives, EOF flag
#
def initialize( opts = {} )
	@directives = { 'YAML' => '1.0', 'TAB' => 'NONE' }
	@options = YAML::DEFAULTS.dup.update( opts )
	@finished = false
    @plain_sets = {
        :AllowInline => YAML.plain_re( ',}]' ),
        :AllowMapping => YAML.plain_re( ':' ),
        :AllowNone => YAML.plain_re
    }
end

#
# Iterator for parsing many documents 
#
def parse_documents( io, &doc_proc )
	@stream = YAML.make_stream( io )
    @lineno = 0
    @charno = 0
    @leftovers = nil
	@options[:Encoding] = @stream.utf_encoding
	while true
		o = _parse
		doc_proc.call( o )
		break if @finished
	end
end

#
# Basic single document parsing
# 
def parse( io )
	@stream = YAML.make_stream( io )
    @lineno = 0
    @charno = 0
    @leftovers = nil
	@options[:Encoding] = @stream.utf_encoding
	_parse
end

#
# Common tokenizer and parser
#
def _parse

	#
	# Parser instance variables
	#
	@anchor = {}
	@lvl = [ Level.new( -1, :Implicit, "" ) ]
	@separator = nil
	@last_tokens = []
	@plain_set = @plain_sets[ :AllowInline ]

    reset_indent
    reset_tokens

    # Tokenizer
    do_tokenize

    @finished = true if eof? and @leftovers.nil?
    return nil if @q.length.zero?

    add_end_indent -1
    close_tokens

    # @q.each { |tok|
    #     p [ tok.sym, tok.data ]
    # }

    # Parser
    begin
        do_parse
    rescue Racc::ParseError => rpe
        token = @last_tokens.last
        raise YAML::ParseError, "Parse error at line #{token.lineno}, char #{token.charno}: [#{token.sym}, #{token.data}]"
    end
end

def do_tokenize
    until eof? and @leftovers.nil? do
        if @leftovers
            yline = @leftovers
            @leftovers = nil
        else
            yline = readline.chomp
        end
        handle_indent( yline )
        until yline.empty? do
            handle_content( yline )
            if [ :End, :Pause ].include?( @lvl.last.status )
                @leftovers = yline
                return
            end
        end
    end
end

def handle_indent( yline )
    if ( indt_match = @lvl.last.indent_re.match( yline ) )
        @indent_raw << "\n" + indt_match[0]
        case indt_match[2]
            when /^#/
                yline.replace ""
            when /^-/
                @indent << "\n" + yline.slice!( 0, indt_match[1].length ) + " "
                @charno += indt_match[1].length
            else
                @indent << "\n" + yline.slice!( 0, indt_match[0].length )
                @charno += indt_match[0].length
        end
    end
end

def handle_content( yline )
    if @charno == 1
        case yline
            when /\A---( |$)/
                if @q.length.zero? 
                    reset_indent
                    @lvl.last.status = :Header
                    @charno += $&.length
                    yline.replace $'
                else
                    @lvl.last.status = :End
                end
                return
            when /\A\.\.\.( |$)/
                @lvl.last.status = :Pause
                @charno += $&.length
                yline.replace $'
                return
        end
    elsif @lvl.last.status == :Header
        if yline =~ /\A%(\w+):(\S+)/
            @directives[$1] = $2
            case $1
                when "YAML"
                    @options[:Version] = $2
                    @options[:UseVersion] = true
            end
            len = $&.length
            yline.slice!( 0, len )
            @charno += len
            return
        else
            @lvl.last.status = :Implicit
        end
    end
    if @lvl.last.status != :Block or @q.last.sym == :FSTART
        case yline
            when /\A#/
                # Throwaway comments
                yline.replace ""
                return
            when /\A +/
                # Throwaway space
                len = $&.length
                yline.slice!( 0, len )
                @charno += len
                return
        end
    end
    apply_indent
    if @lvl.last.status == :Block
        add_token :FOLD, yline
        yline.replace ""
        return
    end
    len = 0
    case yline
        when /\A[#{ Regexp::quote( SPACE_INDICATORS ) }]( |$)/
            # Space indicators
            len = $&.length
            c = yline[ 0, 1 ]
            if c == ':'
                if @lvl.last.status == :Inline
                    @plain_set = @plain_sets[ :AllowMapping ]
                else
                    @plain_set = @plain_sets[ :AllowInline ]
                    handle_seq_map_shortcut
                end
            elsif c == ',' and @lvl.last.status == :Inline
                @plain_set = @plain_sets[ :AllowNone ]
            end
            add_token c, c
		when /\A([>|])([-+\d]*)/
            len = $&.length
            fold = $&
            fold_type = $1
            @lvl.push Level.new( @lvl.last.spaces + 1, :Block, fold_type == "|" ? :Literal : :Folded )
            @q.push Token.new( :IOPEN, @lvl.last.spaces, @lineno, 1 )
            add_token :FSTART, fold
        when /\A([{\[])/
            len = $&.length
            tok = $&
            @lvl.push Level.new( @lvl.last.spaces + 2, :Inline, @lvl.last.domain )
            @plain_set = @plain_sets[ :AllowNone ]
            add_token tok, tok 
        when /\A([}\]])/
            len = $&.length
            tok = $&
            @lvl.pop
            @plain_set = @plain_sets[ :AllowInline ]
            add_token tok, tok 
        when /\A&(\w+)/
            len = $&.length
            add_token :ANCHOR, $1
        when /\A\*(\w+)/
            len = $&.length
            add_token :ALIAS, $1
        when /\A!(\S*)/
            len = $&.length
            ttype = $1
            if ttype =~ /(.*)\^(.+)/
                if $1.length > 0
                    @lvl.last.domain = $1
                    ttype = $1 + $2
                else
                    ttype = @lvl.last.domain + $2
                end
            end
            add_token :TRANSFER, ttype
        when /\A"((?:\\\"|[^\n"])*)/
            len = $&.length
            yline.slice!( 0, len )
            @charno += len
            qstr = $1.to_s
            qstr_indt = "+"
            while yline[0,1] != '"'
                if qstr[-1,1] == "\\"
                    qstr = qstr.chop!
                else
                    qstr << " "
                end
                yline = readline
                yline =~ /\A(\s#{qstr_indt})((?:\\\"|[^\n"])*)/
                len = $&.to_s.length
                yline.slice!( 0, len )
                @charno += len
                qstr << $2.to_s
            end
            len = 1
            add_token :WORD, YAML.unescape(qstr.gsub( /\\"/, '"' ))
        when /\A'((?:''|[^\n'])*)/
            len = $&.length
            yline.slice!( 0, len )
            @charno += len
            qstr = $1.to_s
            qstr_indt = "+"
            nl_mode = false
            while yline[0,1] != "'"
                qstr << " " if qstr[-1,1] != "\n"
                yline = readline
                yline =~ /\A(\s#{qstr_indt})((?:''|[^\n'])*)/
                qstr_indt = "{0,#{$1.length}}" if qstr_indt == "+"
                inner_str = $2.to_s.strip
                if inner_str == ""
                    qstr = ( nl_mode ? qstr : qstr[0..-2] ) + "\n"
                    nl_mode = true
                else
                    qstr << $2
                    nl_mode = false
                end
                len = $&.to_s.length
                yline.slice!( 0, len )
                @charno += len
            end
            len = 1
            add_token :WORD, qstr.gsub( "''", "'" )
        when @plain_set
            len = $&.length
            match = $&.strip
            if @q.last and @q.last.sym == :PLAIN
                @q.last.data += " " if @q.last.data !~ /\Z\n/
                @q.last.data += match
            else
                add_token :PLAIN, match
            end
        else
            len = 1
            c = yline[ 0, 1 ]
            add_token c, c
    end
    yline.slice!( 0, len )
    @charno += len
end

def handle_seq_map_shortcut
    backct = -1
    until @q[backct].nil? or [:IOPEN, :INDENT, :IEND].include?( @q[backct].sym ) do
        if @q[backct].sym == '-' and backct < -1
            @lvl.push Level.new( @q[backct + 1].charno - 1, @lvl.last.status, @lvl.last.domain )
            @q[backct + 1, 0] = Token.new( :IOPEN, @lvl.last.spaces, @lineno, @q[backct].charno )
            break
        end
        backct -= 1
    end
end

#
# Level descent class added 2002 Dec 31
#   The levels of indentation are tracked using this class.
#   Each level has an indentation space count, a status field
#   indicating what sort of data is contained at this depth,
#   and the domain anchored at this depth.
#
class Level
    attr_accessor :spaces, :status, :domain
    def initialize( ct, s, d )
        @spaces, @status, @domain = ct, s, d
    end
    def indent_re
        if @status == :Block
            if @spaces > 0
                /\A( {0,#{ @spaces - 1 }})(#[^\n]+|-( +(?!:)|$)| *)?/
            else
                /\A( *)(#[^\n]+)?/
            end
        else
            /\A( *)(#[^\n]+|-( +(?!:)|$))?/
        end
    end
end

#
# Token class added 2002 Dec 26.
#   Thanks to this class, we can now track line numbers
#   and store accessory data to our tokens.  This should
#   eventually assist our emitter in reforming a document
#   in its original form.
#
class Token
    attr_accessor :sym, :data, :lineno, :charno
    def initialize( s, d, ln, cn )
        @sym, @data, @lineno, @charno = s, d, ln, cn
    end
end

#
# Token stack ops
#
def reset_indent
    @indent, @indent_raw = "", ""
end

def reset_tokens
    @q = []
end

def add_token( sym, data )
    if sym == :FOLD 
        if @q.last.sym != :FOLD
            raise YAML::Error, "Improper fold!!"
        end
        if @q.last.data =~ / \Z/
            @lvl.last.domain = :FoldIndt if @lvl.last.domain == :Folded
        elsif @lvl.last.domain == :Folded
            @q.last.data.gsub!( /\n\Z\n/, "\n" ) unless @q.last.data.gsub!( /(.)\Z\n/, '\1 ' )
        elsif @lvl.last.domain == :FoldIndt
            @lvl.last.domain = :Folded
        end
        @q.last.data += data
    else
        @q.push Token.new( sym, data, @lineno, @charno )
    end
end

def apply_indent
    # Turn indent into a token
    unless @indent.empty?
        indt_len = /^ *\Z/.match( @indent )[0].length
        if @lvl.last.status == :Block and @q.last.sym == :FOLD
            add_fold_indent
        end
        if @lvl.last.spaces > indt_len
            add_end_indent indt_len
        end
        if @lvl.last.status == :Block
            fold_txt = ""
            if @q.last.sym == :FSTART
                if @q.last.data =~ /(\d+)/
                    if @lvl.last.spaces > 0
                        @lvl.last.spaces -= 1
                    end
                    indt_len = $1.to_i + @lvl.last.spaces
                end
                @lvl.last.spaces = indt_len
                @q.push Token.new( :FOLD, fold_txt, @lineno, 1 )
                @indent.slice!( 0, 1 )
                add_fold_indent
            end
        else
            indt_type = :INDENT
            if @lvl.last.spaces < indt_len
                indt_type = :IOPEN
            end
            if indt_type == :IOPEN and @q.last and @q.last.sym == :PLAIN
                unless @q.last.data.empty?
                    @q.last.data += @indent_raw.gsub( /^ +/, '' ).gsub( /\n(\n*)/, '\1' )
                end
            elsif @lvl.last.status != :Inline
                @plain_set = @plain_sets[ :AllowInline ]
                if indt_type == :IOPEN
                    @lvl.push Level.new( indt_len, @lvl.last.status, @lvl.last.domain )
                end
                @q.push Token.new( indt_type, indt_len, @lineno, 1 )
            end
        end
        reset_indent
    end
end

def close_tokens
    @q.push Token.new( false, '$', @lineno, @charno )
end

def pop_token
    @q.pop
end

def push_token( tok )
    @q.push tok
end

#
# Used by Racc
# The @last_token stored for error_reporting
#
def next_token
	@last_tokens.push @q.first
	if @last_tokens.length > 5
		@last_tokens.shift
	end
	tok = @q.shift
    [ tok.sym, tok.data ]
end

def add_end_indent( indt_len )
    while @lvl.last.spaces > indt_len
        @q.push Token.new( :IEND, @lvl.pop.spaces, @lineno, 1 )
    end
end

#
# Keep line number count
#
def readline
    @lineno += 1
    @charno = 1
    begin
        @stream.readline
    rescue IOError
    end
end

def eof?; @stream.eof?; end

#
# Handle Hash special keys
#
def hash_update( h, item )
	if item.class == Array
		# if h.class != YAML::SpecialHash
		# 	h = YAML::SpecialHash.new.update( h )
		# end
        if item[0] == :MERGE
            if Hash === item[1]
                h.update( item[1] )
            elsif Array === item[1]
                item[1].flatten.reverse.each { |hmerge|
                    h.update( hmerge )
                }
            end
		elsif item[0] == :DEFAULT
			h.default = item[1]
		# elsif item[0] == :COMMENT
		# 	h.comment = item[1]
		end
	elsif item.class == YAML::SpecialHash
		h = item.update( h )
	elsif item.is_a? Kernel::Hash
		h.update( item )
	else
		raise YAML::Error, "Invalid update to Hash with item: " + item.inspect
	end
	return h
end

def attach_transfer( meth, val )
    meth = YAML.unescape( meth )
    if @options[:Model] == :Generic
        val.type_id = meth
        val
    else
        YAML.transfer_method( meth, val ) 
    end
end

#
# Add a new node
#
def new_node( type_id, val )
    if @options[:Model] == :Generic
        val = YamlNode.new( type_id, val )
    end
    val
end

#
# Take block text and format based on the block def characters
#
def process_block( fold_type, fold_text )
    fold_text.chomp! if fold_type.include?( '|' )
    if fold_type.include?( '+' )
        fold_text << "\n"
    else
        fold_text.chomp!( '' )
        if fold_type.include?( '-' )
            fold_text.chomp!
        else
            fold_text << "\n" 
        end
    end
    return fold_text
end

def add_fold_indent
    if @lvl.last.spaces > 0
        @indent.gsub!( /^ {0,#{ @lvl.last.spaces }}/, '' )
    end
    @q.last.data += @indent
end

def process_plain( p )
    if @options[:Model] == :Generic
        new_node( YAML.detect_implicit( p ), p )
    else
        YAML.try_implicit( p ) 
    end
end
