/***************************************************************************
 $RCSfile: parser.cpp,v $
                             -------------------
    cvs         : $Id: parser.cpp,v 1.5 2004/07/01 08:42:37 cstim Exp $
    begin       : Tue Aug 28 2001
    copyright   : (C) 2001 by Martin Preuss
    email       : martin@aquamaniac.de
*/

/***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

/*
 */
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef __declspec
# if BUILDING_DLL
#  define DLLIMPORT __declspec (dllexport)
# else /* Not BUILDING_DLL */
#  define DLLIMPORT __declspec (dllimport)
# endif /* Not BUILDING_DLL */
#else
# define DLLIMPORT
#endif

#include <stdio.h>
#include <ctype.h>

#include "error.h"

#include "parser.h"
#include "stream.h"

namespace HBCI {

parser::parser(){
}


parser::~parser(){
}


Error parser::processString(string &s, unsigned int mode){
    unsigned int i;

    i=0;
    // skip leading blanks
    if (mode & PARSER_FLAGS_SKIP_BLANKS) {
        while(i<s.length()) {
            if (s.at(i)>=33)
                break;
            i++;
        }
        if (i)
            s.erase(0,i);
    } // skip blanks

    if (s.empty())
        return Error();
    // cut blanks at the end
    if (mode & PARSER_FLAGS_CUT_BLANKS) {
        for (i=s.length()-1; i>=0; i--)
            if (s.at(i)>32)
                break;
        if (i>=0)
            s.erase(i+1);
    } // if cut blanks

    // remove quotation marks
    if (!s.empty())
        if (mode & PARSER_FLAGS_REMOVE_QUOTATION_MARKS) {
            if (s.at(0)=='\"' && s.at(s.length()-1)=='\"')
                s=s.substr(1,s.length()-2);
            else if (s.at(0)=='\'' && s.at(s.length()-1)=='\'')
                s=s.substr(1,s.length()-2);
        }
    return Error();
}


Error parser::getString(string src,
                            string &dst,
                            string delimiters,
                            string nestchars,
                            unsigned int &pos,
                            unsigned int maxsize){
    string::size_type i;
    char waitfor[16];
    int wpos;

    if (pos>=src.length())
        return Error();

    wpos=-1;
    if (nestchars.length() &0x1)
        return Error("parser::getString()",
                       ERROR_LEVEL_NORMAL,
                       0,
                       ERROR_ADVISE_DONTKNOW,
                       "odd number of nesting chars");

    while(pos<src.length() && dst.length()<=maxsize) {
        // check if we found a delimiter (only if no nesting active)
        if (wpos<0)
            if (delimiters.find(src.at(pos))!=string::npos)
                // yes, we have one, return the result
                return Error();
        // check if we have a nest char
        i=nestchars.find(src.at(pos));
        if (i!=string::npos) {
            // ok, we found a nest-char
            /* If open/close chars are equal and we are waiting for that
             * special char then we handle it as a closing nest char */
            if (nestchars.at(i & ~0x1)==nestchars.at(i | 0x1))
                if (wpos>=0)
                    if (waitfor[wpos]==nestchars.at(i))
                        i|= 0x1;
            
            if (i & 0x1) {
                // odd, so nesting ends
                if (wpos<0)
                    // we are not waiting for chars, error
                    return Error("parser::getString()",
                                   ERROR_LEVEL_NORMAL,
                                   0,
                                   ERROR_ADVISE_DONTKNOW,
                                   "unbalanced nesting",
                                   "e.g. ')' without '('");
                // is it the char we are waiting for ?
                if (src.at(pos)!=waitfor[wpos])
                    // no, error
                    return Error("parser::getString()",
                                   ERROR_LEVEL_NORMAL,
                                   0,
                                   ERROR_ADVISE_DONTKNOW,
                                   "unexpected nesting char");
                // ok, close the current nesting
                wpos--;
            }
            else {
                // even, so nesting is to be started
                wpos++;
                if (wpos>=(int)sizeof(waitfor))
                    // too deep, error
                    return Error("parser::getString()",
                                   ERROR_LEVEL_NORMAL,
                                   0,
                                   ERROR_ADVISE_DONTKNOW,
                                   "nesting too deep");
                // save the char to wait for
                waitfor[wpos]=nestchars.at(i+1);
            }
        } // if nest char found

        // handle next char
        dst+=src.at(pos);
        pos++;
    } // while

    // we are at the end
    if (dst.length()>maxsize)
        // string too long
        return Error("parser::getString()",
                       ERROR_LEVEL_NORMAL,
                       0,
                       ERROR_ADVISE_DONTKNOW,
                       "buffer too small");
    if (wpos>=0)
        // still waiting for chars, so error
        return Error("parser::getString()",
                       ERROR_LEVEL_NORMAL,
                       0,
                       ERROR_ADVISE_DONTKNOW,
                       "unbalanced nesting",
                       "still open nesting");

    // otherwise we have done it
    return Error();
}


Error parser::getString(Stream *st,
                            string &dst,
                            string delimiters,
                            string nestchars,
                            unsigned int maxsize){
    string::size_type i;
    int j;
    char c;
    char waitfor[16];
    int wpos;

    wpos=-1;
    if (nestchars.length() &0x1)
        return Error("parser::getString()",
                       ERROR_LEVEL_NORMAL,
                       0,
                       ERROR_ADVISE_DONTKNOW,
                       "odd number of nesting chars");
    
    while(!st->eof() && dst.length()<=maxsize) {
        // read char from Stream
        j=st->peekChar();
        if (j==-1)
            // eof met, so return
            return Error();
        c=(char) j;

        // check if we found a delimiter (only if no nesting active)
        if (wpos<0)
            if (delimiters.find(c)!=string::npos)
                // yes, we have one, return the result
                return Error();
        // check if we have a nest char
        i=nestchars.find(c);
        if (i!=string::npos) {
            // ok, we found a nest-char
            /* If open/close chars are equal and we are waiting for that
             * special char then we handle it as a closing nest char */
            if (nestchars.at(i & ~0x1)==nestchars.at(i | 0x1))
                if (wpos>=0)
                    if (waitfor[wpos]==nestchars.at(i))
                        i|= 0x1;
            
            if (i & 0x1) {
                // odd, so nesting ends
                if (wpos<0)
                    // we are not waiting for chars, error
                    return Error("parser::getString()",
                                   ERROR_LEVEL_NORMAL,
                                   0,
                                   ERROR_ADVISE_DONTKNOW,
                                   "unbalanced nesting",
                                   "e.g. ')' without '('");
                // is it the char we are waiting for ?
                if (c!=waitfor[wpos])
                    // no, error
                    return Error("parser::getString()",
                                   ERROR_LEVEL_NORMAL,
                                   0,
                                   ERROR_ADVISE_DONTKNOW,
                                   "unexpected nesting char");
                // ok, close the current nesting
                wpos--;
            }
            else {
                // even, so nesting is to be started
                wpos++;
                if (wpos>=(int)sizeof(waitfor))
                    // too deep, error
                    return Error("parser::getString()",
                                   ERROR_LEVEL_NORMAL,
                                   0,
                                   ERROR_ADVISE_DONTKNOW,
                                   "nesting too deep");
                // save the char to wait for
                waitfor[wpos]=nestchars.at(i+1);
            }
        } // if nest char found

        // handle next char
        dst+=c;
        st->readChar();
    } // while

    // we are at the end
    if (dst.length()>maxsize)
        // string too long
        return Error("parser::getString()",
                       ERROR_LEVEL_NORMAL,
                       0,
                       ERROR_ADVISE_DONTKNOW,
                       "buffer too small");
    if (wpos>=0)
        // still waiting for chars, so error
        return Error("parser::getString()",
                       ERROR_LEVEL_NORMAL,
                       0,
                       ERROR_ADVISE_DONTKNOW,
                       "unbalanced nesting",
                       "still open nesting");

    // otherwise we have done it
    return Error();
}


int parser::_fromhex(char c){
    int i;

    c=toupper(c);
    if (c<'0' || c>'F')
        return -1;
    if (c<'A' && c>'9')
        return -1;
    i=c-'0';
    if (i>9) i-=7;
    return i;
}


void parser::_tohex(string &dst, char c){
    unsigned char i;

    i=c>>4;
    if (i>9) i+=7;
    i+='0';
    dst+=(char) i;
    i=c & 15;
    if (i>9) i+=7;
    i+='0';
    dst+=(char) i;
}

bool parser::_checkChar(char c){
    if (c<='Z' && c>='A')
        return true;
    if (c<='z' && c>='a')
        return true;
    if (c<='9' && c>='0')
        return true;
    if (c=='/' || c=='.' || c=='-' || c=='_' || c=='+')
        return true;
    return false;
}


Error parser::escapeHTTP(string &src, unsigned int &pos){
    string dst;
    while (pos<src.length()) {
        if (_checkChar(src.at(pos)))
            dst+=src.at(pos);
        else {
            dst+='%';
            _tohex(dst,src.at(pos));
        }
        pos++;
    } // while
    src=dst;
    return Error();
}


Error parser::unescapeHTTP(string &src, unsigned int &pos){
    string dst;
    int i, j;
    
    while (pos<src.length()) {
        if (src.at(pos)=='%') {
            if (pos+2>src.length())
                return Error("parser::getString()",
                               ERROR_LEVEL_NORMAL,
                               0,
                               ERROR_ADVISE_DONTKNOW,
                               "2 digits expected after '%'");
            pos++;
            i=_fromhex(src.at(pos));
            if (i==-1)
                return Error("parser::getString()",
                               ERROR_LEVEL_NORMAL,
                               0,
                               ERROR_ADVISE_DONTKNOW,
                               "2 digits expected after '%'");
            j=i<<4;
            pos++;
            i=_fromhex(src.at(pos));
            if (i==-1)
                return Error("parser::getString()",
                               ERROR_LEVEL_NORMAL,
                               0,
                               ERROR_ADVISE_DONTKNOW,
                               "2 digits expected after '%'");
            j+=i;
            dst+=(char) j;
        }
        else
            dst+=src.at(pos);
        pos++;
    } // while
    src=dst;
    return Error();
}


bool parser::_cmpSegment(const string &w, unsigned int &wpos,
                           const string &p, unsigned int &ppos,
                           bool sensecase,
                           unsigned int &matches) {
    char a;
    char b;

    a=0;
    b=0;

    while (wpos<w.length() && ppos<p.length()) {
        a=w.at(wpos);
        b=p.at(ppos);
        if (b=='*')
            return true;
        if (!sensecase) {
            a=toupper(a);
            b=toupper(b);
        }
        // count matches
        if (a==b)
            matches++;
        if (a!=b && b!='?')
            return false;
        wpos++;
        ppos++;
    }
    // both at end, would be ok
    if (wpos==w.length() && ppos==p.length())
        return true;
    // word ends, pattern doesnt, would be ok if pattern is '*' here
    if (wpos>=w.length() && ppos<p.length())
        if (p.at(ppos)=='*')
            return true;
    // otherwise no match ;-/
    return false;
}


bool parser::_findSegment(const string &w, unsigned int &wpos,
                            const string &p, unsigned int &ppos,
                            bool sensecase,
                            unsigned int &matches) {
    unsigned int lwpos, lppos, lmatches;

    lwpos=wpos;
    lppos=ppos;
    lmatches=matches;
    while(lwpos<w.length()) {
        ppos=lppos;
        wpos=lwpos;
        matches=lmatches;
        if (_cmpSegment(w,wpos,p,ppos,sensecase,matches))
            return true;
        lwpos++;
    }
    return false;
}


int parser::cmpPattern(const string &w, const string &p, bool sensecase) {
    unsigned int ppos;
    unsigned int wpos;
    unsigned int matches;

    ppos=wpos=matches=0;

    // compare until first occurrence of '*'
    if (!_cmpSegment(w,wpos,p,ppos,sensecase,matches))
        return -1;
    
    while(1) {
        // if pattern ends we have done it
        if (ppos>=p.length())
            return matches;
        // skip '*' in pattern
        ppos++;
        // if pattern ends behind '*' the word matches
        if (ppos>=p.length())
            return matches;
        // find next matching segment
        if (!_findSegment(w,wpos,p,ppos,sensecase,matches))
            return -1;
    } // while
    // I know, we'll never get to here ;-)
    return -1;
}


Error parser::parseRanges(string range,
                              list<string> &ranges) {
    string startpos;
    string endpos;
    string tmp;
    unsigned int pos;
    Error err;

    if (range.empty())
        // no range given, so just return
        return Error();
    pos=0;
    while(pos<range.length()) {
        startpos.erase();
        endpos.erase();

        // get startpos
        tmp.erase();
        err=parser::getString(range,
                                tmp,
                                "-,", // delimiters
                                "",  // nest chars
                                pos,
                                1024);
        if (!err.isOk())
            return err;
        err=parser::processString(tmp,
                                    PARSER_FLAGS_SKIP_BLANKS |
                                    PARSER_FLAGS_CUT_BLANKS |
                                    PARSER_FLAGS_REMOVE_QUOTATION_MARKS);
        if (!err.isOk())
            return err;
        if (!tmp.empty())
            startpos=tmp;

        // get endpos
        while(pos<range.length()) {
            // skip blanks
            if (range.at(pos)>32)
                break;
            pos++;
        } // while
        // check for "-"
        if (pos<range.length())
            if (range.at(pos)=='-') {
                // skip the "-"
                pos++;
                if (pos<range.length()) {

                    tmp.erase();
                    err=parser::getString(range,
                                            tmp,
                                            ",", // delimiters
                                            "",  // nest chars
                                            pos,
                                            1024);
                    if (!err.isOk())
                        return err;
                    err=parser::processString(tmp,
                                                PARSER_FLAGS_SKIP_BLANKS |
                                                PARSER_FLAGS_CUT_BLANKS |
                                                PARSER_FLAGS_REMOVE_QUOTATION_MARKS);
                    if (!err.isOk())
                        return err;
                    if (!tmp.empty())
                        // get the endpos
                        endpos=tmp;
                } // if endpos given
            } // if "-"
        // skip blanks
        while(pos<range.length()) {
            if (range.at(pos)>32)
                break;
            pos++;
        } // while

        // check for ","
        if (pos<range.length())
            if (range.at(pos)!=',')
                // error, if there is something behind the range
                return Error("http::getContentRange()",
                               ERROR_LEVEL_NORMAL,
                               0,
                               ERROR_ADVISE_DONTKNOW,
                               "extra data at end of line");
        // skip the comma
        pos++;
        // store the positions we found
        ranges.push_back(startpos);
        ranges.push_back(endpos);
    } // while still ranges
    return Error();
}


string parser::num2string(int number,
                            bool fillWithZero, int length) {
    string snumber;
    char numberText[20];
    sprintf(numberText, "%i", number);
    snumber = numberText;

    // fill with "0" (from left)
    if (fillWithZero && (int) snumber.length() < length) {
        string tmp(length - (int) snumber.length(),'0');
        snumber = tmp + snumber;
    }

    return snumber;
}

} // namespace HBCI
