/* outboxjobkeys.h
    copyright   : (C) 2001 by Martin Preuss
    email       : openhbci@aquamaniac.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


#ifndef OUTBOXJOBKEYS_H
#define OUTBOXJOBKEYS_H

/** @file outboxjobkeys.h
 *
 * @short Key management related Outboxjobs and their C wrappers. */

#ifdef __cplusplus
#include <string>
#include <list>
#endif /* __cplusplus */

#include <openhbci/dllimport.h>
#include <openhbci/pointer.h>
#include <openhbci/customer.h>
#include <openhbci/user.h>
#include <openhbci/bank.h>
#include <openhbci/account.h>
#include <openhbci/outboxjob.h>



#ifdef __cplusplus

namespace HBCI {

/**
 * @short Base class for key management
 * This job gets the servers keys or send ours. It is a base class for key
 * management.
 * @author Martin Preuss<martin@libchipcard.de>
 */
class DLLIMPORT OutboxJobKeys: public OutboxJob {
private:
    bool _sendkeys;

protected:
    const API *_hbciapi;
    Pointer<Job> _initjob;
    Pointer<Job> _exitjob;

    OutboxJobKeys(const API *api, Pointer<Customer> c, bool sk);
    ~OutboxJobKeys();

    virtual bool createHBCIJobs(Pointer<MessageQueue> mbox, int n=0);
    virtual bool evaluate();
    virtual bool commit(int msgNumber=HBCI_COMMIT_WHOLE_JOB);
    string description() const { return "Key management";};
    virtual bool isDialogJob() const { return true;};
    virtual int messages() const { return 2;};
    list<int> resultCodes() const;
};

} /* namespace HBCI */
#endif /* __cplusplus */


#ifdef __cplusplus

namespace HBCI {
/**
 * This job gets the servers keys.
 * @short Job that gets the server keys
 * @author Martin Preuss<martin@libchipcard.de>
 */
class DLLIMPORT OutboxJobGetKeys: public OutboxJobKeys {
public:
    OutboxJobGetKeys(const API *api, Pointer<Customer> c);
    ~OutboxJobGetKeys();

    string description() const { return "Get Keys";};
    JobProgressType type() const { return JOB_GET_KEYS; };
};

} /* namespace HBCI */
/** @ingroup HBCI_OutboxJobg */
typedef struct HBCI::OutboxJobGetKeys HBCI_OutboxJobGetKeys;

extern "C" {
#else /* __cplusplus */
    typedef struct HBCI_OutboxJobGetKeys HBCI_OutboxJobGetKeys;
#endif /* __cplusplus */
    /** @name HBCI_OutboxJobGetKeys 
     * @ingroup HBCI_OutboxJobg */
    /*@{*/
    /** Constructor */
    extern HBCI_OutboxJobGetKeys *
    HBCI_OutboxJobGetKeys_new(const HBCI_API *api, const HBCI_Customer *c);
    /** Upcast */
    extern HBCI_OutboxJob *
    HBCI_OutboxJobGetKeys_OutboxJob(HBCI_OutboxJobGetKeys *j);
    /*@}*/
#ifdef __cplusplus
}

namespace HBCI {

/**
 * This job sends our keys.
 * @short Job that sends the keys of one Customer
 * (FIXME: this should rather be a User) to the server
 * @author Martin Preuss<martin@libchipcard.de>
 */
class DLLIMPORT OutboxJobSendKeys: public OutboxJobKeys {
public:
    OutboxJobSendKeys(const API *api, Pointer<Customer> c);
    ~OutboxJobSendKeys();

    string description() const { return "Send Keys";};
    JobProgressType type() const { return JOB_SEND_KEYS; };
};

} /* namespace HBCI */
/** @ingroup HBCI_OutboxJobg */
typedef struct HBCI::OutboxJobSendKeys HBCI_OutboxJobSendKeys;

extern "C" {
#else /* __cplusplus */
    typedef struct HBCI_OutboxJobSendKeys HBCI_OutboxJobSendKeys;
#endif /* __cplusplus */
    /** @name HBCI_OutboxJobSendKeys 
     * @ingroup HBCI_OutboxJobg */
    /*@{*/
    /** Constructor */
    extern HBCI_OutboxJobSendKeys *
    HBCI_OutboxJobSendKeys_new(const HBCI_API *api, const HBCI_Customer *c);
    /** Upcast */
    extern HBCI_OutboxJob *
    HBCI_OutboxJobSendKeys_OutboxJob(HBCI_OutboxJobSendKeys *j);
    /*@}*/
#ifdef __cplusplus
}
#endif /* __cplusplus */

#ifdef __cplusplus

namespace HBCI {
/**
 * This job locks the cryptographic keys and tells the bankserver not to 
 * accept them anymore.
 * @short Job that locks both, the current crypt and sign key of a user
 * @author Fabian Kaiser<fabian@openhbci.de>
 */
class DLLIMPORT OutboxJobDisableKeys: public OutboxJobKeys {
protected:
  Pointer<Job> _job;

public:
  OutboxJobDisableKeys(const API *api, Pointer<Customer> c);
  ~OutboxJobDisableKeys();

  virtual bool createHBCIJobs(Pointer<MessageQueue> mbox, int n=0);
  bool commit(int msgNumber=HBCI_COMMIT_WHOLE_JOB);
  string description() const { return "Disable Keys";};
  int messages() const { return 3;};
  JobProgressType type() const { return JOB_DISABLE_KEYS; };
};

} /* namespace HBCI */
/** @ingroup HBCI_OutboxJobg */
typedef struct HBCI::OutboxJobDisableKeys HBCI_OutboxJobDisableKeys;

extern "C" {
#else /* __cplusplus */
    typedef struct HBCI_OutboxJobDisableKeys HBCI_OutboxJobDisableKeys;
#endif /* __cplusplus */
    /** @name HBCI_OutboxJobDisableKeys 
     * @ingroup HBCI_OutboxJobg */
    /*@{*/
    /** Constructor */
    extern HBCI_OutboxJobDisableKeys *
    HBCI_OutboxJobDisableKeys_new(const HBCI_API *api, const HBCI_Customer *c);
    /** Upcast */
    extern HBCI_OutboxJob *
    HBCI_OutboxJobDisableKeys_OutboxJob(HBCI_OutboxJobDisableKeys *j);
    /*@}*/
#ifdef __cplusplus
}

namespace HBCI {
/**
 * This job locks the cryptographic keys and tells the bankserver not to 
 * accept them anymore. Use this job instead of OutboxJobDisableKeys if you
 * don't have access to the keys anymore (e.g. if your key file is lost).
 * Note: Most banks do not support this job in order to prevent abuse (you
 * can lock out anybody if you know his/her userid;-)
 * @short Job that locks both, the current crypt and sign key of a user
 * @author Fabian Kaiser<fabian@openhbci.de>
 */
class DLLIMPORT OutboxJobDisableLostKeys: public OutboxJobDisableKeys {
private:
  int _keyNumber;
  int _keyVersion;

public:
  /**
   * Constructor.
   * @param api The HBCI-context
   * @param c The owner of the lost keys
   * @param keyNumber The number of the key to be disabled. If you
   * don't know the number, try "999"
   * @param keyVersion The version number of the key to be disabled. If
   * you don't know the version number, try "999"
   */
  OutboxJobDisableLostKeys(const API *api, Pointer<Customer> c,
			   int keyNumber=999, int keyVersion=999);
  ~OutboxJobDisableLostKeys();
  int messages() const { return 2;};
  bool createHBCIJobs(Pointer<MessageQueue> mbox, int n=0);
};

} /* namespace HBCI */
/** @ingroup HBCI_OutboxJobg */
typedef struct HBCI::OutboxJobDisableLostKeys HBCI_OutboxJobDisableLostKeys;

extern "C" {
#else /* __cplusplus */
    typedef struct HBCI_OutboxJobDisableLostKeys HBCI_OutboxJobDisableLostKeys;
#endif /* __cplusplus */
    /** @name HBCI_OutboxJobDisableLostKeys 
     * @ingroup HBCI_OutboxJobg */
    /*@{*/
    /**
     * Constructor.
     * @param api The HBCI-context
     * @param c The owner of the lost keys
     * @param keyNumber The number of the key to be disabled. If you
     * don't know the number, use the number "999" here.
     * @param keyVersion The version number of the key to be disabled. If
     * you don't know the version number, use the number "999" here.
     */
    extern HBCI_OutboxJobDisableLostKeys *
    HBCI_OutboxJobDisableLostKeys_new(const HBCI_API *api, 
				      const HBCI_Customer *c,
				      int keyNumber, 
				      int keyVersion);
    /** Upcast */
    extern HBCI_OutboxJob *
    HBCI_OutboxJobDisableLostKeys_OutboxJob(HBCI_OutboxJobDisableLostKeys *j);
    /*@}*/
#ifdef __cplusplus
}

namespace HBCI {
/**
 * The currently used keys are disabled and can not be used anymore, 
 * instead the new keys become valid.<br>
 * Before you call perform this job, make sure to call 
 * MediumRDHBase::createUserKeys(false) to create new keys (but not 
 * activate them, otherwise you won't be able to commit the new keys.
 * After the job has been executed successfully, call
 * MediumRDHBase::activateKeys() to enable the new keys
 * Note: Both - the crypto- and the signature-key are changed!
 * @short This job allows you to change your cryptographic keys
 * @author Fabian Kaiser<fabian@openhbci.de>
 */
class DLLIMPORT OutboxJobChangeKeys: public OutboxJobKeys {
protected:
  Pointer<Job> _job;

public:
    OutboxJobChangeKeys(const API *api, Pointer<Customer> c);
    ~OutboxJobChangeKeys();

    virtual bool createHBCIJobs(Pointer<MessageQueue> mbox, int n=0);
    bool commit(int msgNumber=HBCI_COMMIT_WHOLE_JOB);
    string description() const { return "Change Key";};
    int messages() const { return 3;};
    JobProgressType type() const { return JOB_CHANGE_KEYS; };
};

} /* namespace HBCI */
/** @ingroup HBCI_OutboxJobg */
typedef struct HBCI::OutboxJobChangeKeys HBCI_OutboxJobChangeKeys;

extern "C" {
#else /* __cplusplus */
    typedef struct HBCI_OutboxJobChangeKeys HBCI_OutboxJobChangeKeys;
#endif /* __cplusplus */
    /** @name HBCI_OutboxJobChangeKeys 
     * @ingroup HBCI_OutboxJobg */
    /*@{*/
    /** Constructor */
    extern HBCI_OutboxJobChangeKeys *
    HBCI_OutboxJobChangeKeys_new(const HBCI_API *api, const HBCI_Customer *c);
    /** Upcast */
    extern HBCI_OutboxJob *
    HBCI_OutboxJobChangeKeys_OutboxJob(HBCI_OutboxJobChangeKeys *j);
    /*@}*/
#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif



