/***************************************************************************
                          hbcivalue.cpp  -  description
                             -------------------
    begin                : Sat Aug 4 2001
    copyright            : (C) 2001 by fabian kaiser
    email                : fabian.kaiser@gmx.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef __declspec
# if BUILDING_DLL
#  define DLLIMPORT __declspec (dllexport)
# else /* Not BUILDING_DLL */
#  define DLLIMPORT __declspec (dllimport)
# endif /* Not BUILDING_DLL */
#else
# define DLLIMPORT
#endif


#include "value.h"

#include <stdio.h>
#include <locale.h>
#include <sstream>
#include "assert.h"

#include "hbci.h"
#include "hbcistring.h"
#include "error.h"

namespace HBCI {

Value::Value()
    : _value(0.0), _isvalid(false)
{
}


Value::Value(double value, const string &currency)
: _value(value)
, _currency(currency)
, _isvalid(true)
{
}


Value::Value(const string &strvalue)
:_value(0.00)
,_isvalid(true)
{
    // find the seperator betw. value and currency
    string::size_type i = strvalue.find(":");

    // extract currency if present
    if (i == string::npos) {
        _currency = "";
        i = strvalue.length();
    } else {
	// check for any malformed string-values
	if (strvalue.find_first_of(":") != strvalue.find_last_of(":"))
	    throw Error("HBCI::Value::Value(string)",
			ERROR_LEVEL_NORMAL,
			0,
			ERROR_ADVISE_RETRY,
			"Wrong string format (too many ':' in '"
			+ strvalue + "').");
	if (strvalue.length() == i+1)
	    _currency = "";
	else {
	    if (strvalue.length() < i+1+3)
		throw Error("HBCI::Value::Value(string)",
			    ERROR_LEVEL_NORMAL,
			    0,
			    ERROR_ADVISE_RETRY,
			    "Wrong string format (Currency too short in '"
			    + strvalue + "').");
	    
	    _currency = strvalue.substr(i+1, 3);
	}
    }

    // extract the value in a substring
    string valstr = strvalue.substr(0, i);

    if ((valstr.find(",") != string::npos) &&
	(valstr.find(".") != string::npos))
	throw Error("HBCI::Value::Value(string)",
		    ERROR_LEVEL_NORMAL,
		    0,
		    ERROR_ADVISE_RETRY,
		    "Wrong string format (',' and '.' mixed in '"
		    + valstr + "').");
    if (valstr.find_first_of(",") != valstr.find_last_of(","))
	throw Error("HBCI::Value::Value(string)",
		    ERROR_LEVEL_NORMAL,
		    0,
		    ERROR_ADVISE_RETRY,
		    "Wrong string format (too many ',' in '" + valstr + "').");

    // Convert "," to "."
    //if (valstr.find(",")!=string::npos)
    //cerr << "Found comma in '" << valstr << "' and replacing it.\n";
    i = valstr.find(",");
    if (i != string::npos)
	valstr.replace(i, 1, ".");

    if (valstr.find_first_of(".") != valstr.find_last_of("."))
	throw Error("HBCI::Value::Value(string)",
		    ERROR_LEVEL_NORMAL,
		    0,
		    ERROR_ADVISE_RETRY,
		    "Wrong string format (too many '.' in '"
		    + valstr + "').");

    // New stringstream to read the float value from
    istringstream ist(valstr);

    // Cache old locale and set numerics to C locale
    string oldloc = setlocale(LC_NUMERIC, 0l);
    setlocale(LC_NUMERIC, "C");
    
    // Read float (double) value according to C-locale rules.
    bool readsuccess = (ist >> _value);

    // Reset locale to cached string.
    setlocale(LC_NUMERIC, oldloc.c_str());

    // Throw HBCI::Error if that failed.
    if (!readsuccess)
	throw Error("HBCI::Value::Value(string)",
		    ERROR_LEVEL_NORMAL,
		    0,
		    ERROR_ADVISE_RETRY,
		    "Wrong string format (conversion string->float failed in '"
		    + valstr + "').");
}


string Value::toString() const {
    ostringstream ost1;

    // Cache old locale and set numerics to C locale
    string oldloc = setlocale(LC_NUMERIC, 0l);
    setlocale(LC_NUMERIC, "C");

    // Set output format to fixed with decimal point always-on, and
    // precision according to ISO-4217
    ost1.setf(ios::fixed|ios::showpoint, ios::floatfield);
    ost1.precision(currencyPrecision(_currency));
    
    // Output the value according to C-locale rules.
    ost1 << _value;

    // Reset locale to cached string.
    setlocale(LC_NUMERIC, oldloc.c_str());

    // Get a string copy from the stringstream
    string valstr = ost1.str();

    string::size_type i = valstr.find(".");
    if (i != string::npos) {
	// Convert "." to ","
	valstr.replace(i, 1, ",");
	
	// Delete ending 0s, as required by HBCI-spec
	while (valstr.find_last_of("0") == valstr.length()-1)
	    valstr.erase(valstr.find_last_of("0"), 1);
    }
    else
	// enforce decimal comma, as required by HBCI-spec
	valstr += ",";

    // Make sure this one doesn't give bad formats
    if (valstr.find_first_of(",") != valstr.find_last_of(","))
	throw Error("HBCI::Value::toString()",
		    ERROR_LEVEL_NORMAL,
		    0,
		    ERROR_ADVISE_RETRY,
		    "Error/Bug in conversion (sorry, too many ',' in '" 
		    + valstr + "').");

    // Return concatenated value plus colon plus currency
    return valstr + ":" + _currency;
}


string Value::toReadableString() const {
    ostringstream ost1;

    // Cache old locale and set numerics to C locale
    string oldloc = setlocale(LC_NUMERIC, 0l);
    setlocale(LC_NUMERIC, "C");

    // Set output format to fixed with decimal point always-on, and
    // precision according to ISO-4217
    ost1.setf(ios::fixed, ios::floatfield);
    ost1.precision(currencyPrecision(_currency));
    
    // Output the value according to C-locale rules.
    ost1 << _value;

    // Reset locale to cached string.
    setlocale(LC_NUMERIC, oldloc.c_str());

    // Get a string copy from the stringstream
    string valstr = ost1.str();

    // Convert "." to ","
    string::size_type i = valstr.find(".");
    if (i != string::npos)
	valstr.replace(i, 1, ",");

    // Another stringstream 
    ostringstream ost2;
    ost2 << valstr;

    if (_currency.length() > 0) 
	// Output the currency
	ost2 << " " << _currency;

    // Return result as a string copy of the stringstream
    return ost2.str();
}

unsigned int Value::currencyPrecision(const string& c) {
    if (c == "EUR")
	return 2;
    if ((c == "BEF") ||
	(c == "GRD") ||
	(c == "ITL") ||
	(c == "JPY") ||
	(c == "LUF") ||
	(c == "PTE") ||
	(c == "ESP"))
	return 0;
    
    if (Hbci::debugLevel()>15)
	fprintf(stderr,"Value::currencyPrecision(): Unknown currency '%s'.\n",
		c.c_str());
    return 2;
}

// bool Value::operator==(const Value &v) const{
//     return (
//         (_value==v._value) &&
//         (_currency==v._currency)
//            );
// }


} // namespace  HBCI

HBCI_Value *HBCI_Value_new_double(double value, 
				  const char *currency)
{
    return new HBCI::Value(value, string(currency));
}

int HBCI_Value_isValid(const HBCI_Value *v)
{
    assert(v);
    return v->isValid();
}
double HBCI_Value_getValue(const HBCI_Value *v)
{
    assert(v);
    return v->getValue();
}
const char* HBCI_Value_getCurrency(const HBCI_Value *v)
{
    assert(v);
    return v->getCurrency().c_str();
}
char* HBCI_Value_toReadableString(const HBCI_Value *v)
{
    assert(v);
    return hbci_strdup(v->toReadableString());
}


