/*
  hbciinteractorcb.h
  (C) 2002 by Christian Stimming <stimming@tuhh.de>

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef HBCIINTERACTORCB_H
#define HBCIINTERACTORCB_H

/** @file interactorcb.h 
 *
 * @short Callback-using HBCI::InteractorCB, intended to be used from
 * C wrapper @ref HBCI_InteractorCB.*/

#include <openhbci/dllimport.h>
#include <openhbci/pointer.h>
#include <openhbci/user.h>
#include <openhbci/interactor.h>

/**  Function to ask the user for the pin (password) of a security medium. 
 *
 * @return true (!=0) if ok, false (==0) if the user aborted
 *
 * @param user user for whom the pin is to be asked for.
 * Please note that this pointer may be zero.
 *
 * @param pin pointer to a char* pointer that should be set to point
 * to the buffer with the entered pin/password. This entered
 * pin/password buffer is going to be free'd by this class
 * immediately, so don't pass a pointer to some static buffer.
 *
 * @param minsize Minimum size of the pin, otherwise it will not be
 * accepted and the calling OpenHBCI function behaves just as if the
 * user aborted this dialog.
 *
 * @param newPin if true then the user should be asked for a new pin
 * (i.e. the pin should be entered twice).
 *
 * @param user_data The arbitrary data given in the class constructor.
 */
typedef int (*InputPinFuncTyp) (const HBCI_User *user, 
				char **pin, 
				int minsize, 
				int newPin,
				void *user_data);
/**
 * Function to tell a user something, and depending on the user's
 * reaction return true or false.
 *
 * @param user user for whom this function is called. Please note that
 * this pointer may be zero.
 * @param type The type of this medium.
 *
 * @param user_data The arbitrary data given in the class constructor.
 *
 * @return true if ok, false if user wants to abort.
 */
typedef int (*InputUserFuncTyp) (const HBCI_User *user,
				 MediumType type,
				 void *user_data);
/**
 * Function to log a message string.
 *
 * @param msg The message string to be logged.
 *
 * @param user_data The arbitrary data given in the class constructor.
 */
typedef void (*LoggingFuncTyp) (const char *msg,
				void *user_data);
/**
 * Function that simply returns true or false, depending on some state
 * of the system.
 *
 * @param user_data The arbitrary data given in the class constructor.
 */
typedef int (*BoolFuncTyp) (void *user_data);

/** Function that only notifies the application of some event.
 *
 * @param user_data The arbitrary data given in the class constructor.
 */
typedef void (*VoidFuncTyp) (void *user_data);

/** Function that notifies of something which is related to the given HBCI_User.
 *
 * @param user user for whom this function is called. Please note that
 * this pointer may be zero.
 * @param user_data The arbitrary data given in the class constructor.
 */
typedef void (*VoidUserFuncTyp) (const HBCI_User *user, 
				 void *user_data);


#ifdef __cplusplus

namespace HBCI {
/**  
 * @short Alternative implementation class for the Interactor interface.
 *
 * This class defines methods for all occasions where the core
 * OpenHBCI needs user interaction, as layouted through the
 * Interactor interface.
 * 
 * This is an implementation which stores function pointers
 * (callbacks) for each interaction method and calls them for each
 * method defined in the base class. If there hasn't been set any of
 * the correct callbacks, then the default implementation of the base
 * class is called.
 *
 * @author Christian Stimming <stimming@tuhh.de> */
class DLLIMPORT InteractorCB : public Interactor {
private:
    VoidFuncTyp _destrCB;
    InputPinFuncTyp _msgInputPinCB;
    InputUserFuncTyp _msgInsertMediumOrAbortCB;
    InputUserFuncTyp _msgInsertCorrectMediumOrAbortCB;
    LoggingFuncTyp _msgStateResponseCB;
    BoolFuncTyp _keepAliveCB;
    VoidUserFuncTyp _msgStartInputPinViaKeypadCB;
    VoidUserFuncTyp _msgFinishedInputPinViaKeypadCB;
    void *_user_data;

public:
    /** Constructor.
     *
     * Any of these function callbacks are allowed to be zero, in
     * which case the default implementation of HBCI::Interactor is
     * called.
     *
     * user_data is a pointer to arbitrary data that is going to be
     * passed on to each callback function. */
    InteractorCB(VoidFuncTyp destrCB,
		 InputPinFuncTyp msgInputPinCB,
		 InputUserFuncTyp msgInsertMediumOrAbortCB,
		 InputUserFuncTyp msgInsertCorrectMediumOrAbortCB,
		 LoggingFuncTyp msgStateResponseCB,
		 BoolFuncTyp keepAliveCB,
		 VoidUserFuncTyp msgStartInputPinViaKeypadCB,
		 VoidUserFuncTyp msgFinishedInputPinViaKeypadCB,
		 void *user_data);

    /** Destructor.
     * 
     * This will call the destrCB callbank function before actually
     * deleting this object. Use the destrCB to e.g. delete the data
     * pointed to by user_data. */
    ~InteractorCB();
    	    
    /** 
     * Ask the user for the pin of a security medium. This method is
     * called by the default Authentifikator. The default
     * implementation does nothing and returns false.
     *
     * @return true if ok, false if the user aborted
     * @param user user for whom the pin is to be asked for.
     * Please note that this pointer may be invalid !
     * @param pin reference to a variable to receive the pin
     * @param minsize minimum size of the pin.
     * @param newPin if true then the user should be asked for a new pin.
     * Most implementations will ask the user for the pin and then ask him
     * to confirm it (thus making the user doubly enter the pin).  */
    bool msgInputPin(Pointer<User> user,
		     string &pin,
		     int minsize,
		     bool newPin);

    /**
     * Tell the user to insert a chip card. The default
     * implementation does nothing and returns false.
     *
     * @param user user for whom the card is to be inserted.
     * Please note that this pointer may be invalid !
     * @param t The MediumType of the medium that should be inserted.
     * @return true if ok, false if user wants to abort.
     */
    bool msgInsertMediumOrAbort(Pointer<User> user,
				MediumType t);

    /**
     * Tell the user that:<ul>
     * <li>the currently inserted chip card is not the right one and</li>
     * <li>he/she should now insert the correct card</li>
     * </ul> The default
     * implementation does nothing and returns false.
     *
     * @return true if ok, false if user wants to abort.
     * @param user user for whom the card is to be inserted.
     * Please note that this pointer may be invalid !
     * @param t The MediumType of the medium that should be inserted.
     */
    bool msgInsertCorrectMediumOrAbort(Pointer<User> user,
                                       MediumType t);

    /**  
     * Prints a message that might be used for logging purpose. Does
     * not require interaction. The default implementation does
     * nothing and discards the message.  This method should be
     * overwritten by your application.  
     * @param msg the message to show */
    void msgStateResponse(const string &msg);

    /**
     * During actions which take a long time to perform (some seconds or
     * minutes) this method is called by OpenHBCI every once in a while. 
     * 
     * You can override this method to let your program do some GUI
     * update stuff. Also, if this method returns "false", then the
     * long term action will be aborted. Otherwise the action simply
     * continues.  
     *
     * The default implementation of the InteractorCB returns the
     * variable set by @ref abort() which is accessible also by @ref
     * aborted(). Thus the InteractorCB offery you the convenience
     * of only using @ref abort() instead of deriving your own
     * InteractorCB. If you overload this method, you probably
     * should take the value of @ref aborted() into account to comply
     * to that default behaviour.
     *
     * @author Martin Preuss<openhbci@aquamaniac.de>
     * @return false if user wants the action to be aborted, false otherwise */
    bool keepAlive();

    /**
     * This method is called directly before the user is supposed to enter the
     * pin into the keypad of a card reader. This is used to inform the user
     * that OpenHBCI started secure pin verification. A graphical application
     * might want to open a window that tells the user about this.
     * (new since 0.9.5)
     * @author Martin Preuss<martin@libchipcard.de>
     * @param user user for whom the pin is to be asked for.
     * Please note that this pointer may be invalid.
     */
    virtual void msgStartInputPinViaKeypad(Pointer<User> user);

    /**
     * This method is called directly after the secure pin verification (via
     * the card readers keypad) is finished. A graphical applicatition most
     * likely closes the notification window opened at
     * @ref msgStartInputPinViaKeypad().
     * (new since 0.9.5)
     * @author Martin Preuss<martin@libchipcard.de>
     * @param user user for whom the pin is to be asked for.
     * Please note that this pointer may be invalid.
     */
    virtual void msgFinishedInputPinViaKeypad(Pointer<User> user);
};

} /* namespace HBCI */
typedef struct HBCI::InteractorCB HBCI_InteractorCB;

extern "C" {
#else /* __cplusplus */
  typedef struct HBCI_InteractorCB HBCI_InteractorCB;
#endif /* __cplusplus */

    /** Constructor since OpenHBCI 0.9.5. See InteractorCB::InteractorCB().
     *
     * (Old constructor is still around to not break the compile.) */
    extern HBCI_InteractorCB* HBCI_InteractorCB_new2(
	VoidFuncTyp destrCB,
	InputPinFuncTyp msgInputPinCB,
	InputUserFuncTyp msgInsertMediumOrAbortCB,
	InputUserFuncTyp msgInsertCorrectMediumOrAbortCB,
	LoggingFuncTyp msgStateResponseCB,
	BoolFuncTyp keepAliveCB,
	VoidUserFuncTyp msgStartInputPinViaKeypadCB,
	VoidUserFuncTyp msgFinishedInputPinViaKeypadCB,
	void *user_data);

    /** Old Constructor. DEPRECATED. Rather use @ref HBCI_InteractorCB_new2. */
    extern HBCI_InteractorCB* HBCI_InteractorCB_new(
	VoidFuncTyp destrCB,
	InputPinFuncTyp msgInputPinCB,
	InputUserFuncTyp msgInsertMediumOrAbortCB,
	InputUserFuncTyp msgInsertCorrectMediumOrAbortCB,
	LoggingFuncTyp msgStateResponseCB,
	BoolFuncTyp keepAliveCB,
	void *user_data);

    /** Upcast. */
    extern HBCI_Interactor* 
    HBCI_InteractorCB_Interactor(HBCI_InteractorCB *h);

#ifdef __cplusplus
}
#endif /* __cplusplus */


#endif
