/***************************************************************************
                          hbciconnection.cpp  -  description
                             -------------------
    begin                : Thu Jul 26 2001
    copyright            : (C) 2001 by fabian kaiser
    email                : fabian.kaiser@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

/*
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <unistd.h>
#include "connection.h"
#include "hbcistring.h"


namespace HBCI {

time_t Connection::_timeout=CONNECTION_DEFAULT_TIMEOUT;


Connection::Connection(Hbci *hbci, string internetHost, int tcpIpPort)
    : _next_message_number(1)
    , tcpPort(tcpIpPort)
    , _hbci(hbci)
    , _sock(Socket(SOCKET_TYPE_TCP))
    , _addr(InetAddress(internetHost))
{
    mBox.setDescription("Connection::mBox (MessageQueue)");
}


Connection::~Connection(){
    // maybe the socket is still open
    close();
}


void Connection::setTimeOut(time_t t){
  _timeout=t;
}


Error Connection::open() {
  Error err;
  time_t startTime;

  // get start time for timeout calculation
  startTime=time(0);

  // connect, wait for 500 ms
  err=_sock.startConnect(_addr,tcpPort);
  while (_hbci->interactor().ref().keepAlive()) {
    err=_sock.checkConnect(500);
    if (err.isOk())
      return err;
    if (err.code()!=HBCI_ERROR_CODE_SOCKET_ERROR_TIMEOUT &&
	err.code()!=HBCI_ERROR_CODE_SOCKET_ERROR_INTERRUPT) {
      if (Hbci::debugLevel()>1)
	fprintf(stderr,"%s\n",err.errorString().c_str());
      return err;
    }
    if (difftime(time(0), startTime)>_timeout) {
      return Error("Connection::open",
		   ERROR_LEVEL_NORMAL,
		   HBCI_ERROR_CODE_SOCKET_ERROR_TIMEOUT,
		   ERROR_ADVISE_DONTKNOW,
		   "open timed out");
    }
  } // while

  // done
  if (Hbci::debugLevel()>1)
    fprintf(stderr,"User aborted connection attempt.\n");
  _sock.abortConnect();
  return Error("Connection::open()",
	       ERROR_LEVEL_NORMAL,
	       0,
	       ERROR_ADVISE_ABORT,
	       "User aborted connection attempt.");
}


bool Connection::close() {
    Error err;

    err=_sock.close();
    if (!err.isOk()) {
	if (Hbci::debugLevel()>1)
	    fprintf(stderr,"%s\n",err.errorString().c_str());
        return false;
    }
    return true;
}



bool Connection::sendMessage(Pointer<MessageQueue> jobs) {
    string boxstring;

    mBox = jobs;
    // create a message-string and send it to the inst.
    boxstring=jobs.ref().toString(_next_message_number++);
    return sendData(boxstring);
}


bool Connection::getResponse() {
    if (recieveData())
        return mBox.ref().setResponse(response);
    else
        return false;
}


bool Connection::sendData(string data) {
  Error err;
  time_t startTime;

  // get start time for timeout calculation
  startTime=time(0);

  // send data, wait for 500 ms
  while (_hbci->interactor().ref().keepAlive()) {
    err=_sock.writeData(data,500);
    if (err.isOk())
      return true;
    if (err.code()!=HBCI_ERROR_CODE_SOCKET_ERROR_TIMEOUT &&
	err.code()!=HBCI_ERROR_CODE_SOCKET_ERROR_INTERRUPT) {
      if (Hbci::debugLevel()>1)
	fprintf(stderr,"%s\n",err.errorString().c_str());
      return false;
    }
    if (difftime(time(0), startTime)>_timeout) {
      if (Hbci::debugLevel()>0)
	fprintf(stderr,
		"Connection::recieveData(): timeout\n");
      return false;
    }
  } // while

  return false;
}


bool Connection::recieveData() {
  int size = 1000;

  Pointer<Interactor> interactor;
  string result;
  string tmp;
  Error err;
  time_t startTime;

  // get start time for timeout calculation
  startTime=time(0);

  if (Hbci::debugLevel()>15)
    fprintf(stderr,"Connection::recieveData()\n");
  interactor=_hbci->interactor();


  // first read up to 30 bytes, wait for 500 ms
  size=30;
  while (interactor.ref().keepAlive() && size>0) {
    tmp.erase();
    err=_sock.readData(tmp,size,500);
    if (!err.isOk()) {
      if (err.code()!=HBCI_ERROR_CODE_SOCKET_ERROR_TIMEOUT) {
	if (Hbci::debugLevel()>1)
	  fprintf(stderr,"Error %d(%s)\n",
		  err.code(),
		  err.errorString().c_str());
	return false;
      }
    } // if error
    else {
      if (tmp.length()==0)
	// broken pipe
	return false;
      result+=tmp;
      size-=tmp.length();
    }
    if (difftime(time(0), startTime)>_timeout) {
      if (Hbci::debugLevel()>0)
	fprintf(stderr,
		"Connection::recieveData(): timeout\n");
      return false;
    }
  } // while
  if (size>0)
    return false;

  // now we should have at least 30 bytes, now parse the msg size
  tmp=String::nextDE(result, 0);
  tmp=String::nextDE(result, tmp.length()+1);
  size=atoi((const char*) tmp.c_str());

  // subtract the number of bytes we already have
  size-=result.length();

  // now read exact size bytes
  while (interactor.ref().keepAlive() && size>0) {
    tmp.erase();
    err=_sock.readData(tmp,size,500);
    if (!err.isOk()) {
      if (err.code()!=HBCI_ERROR_CODE_SOCKET_ERROR_TIMEOUT &&
	  err.code()!=HBCI_ERROR_CODE_SOCKET_ERROR_INTERRUPT) {
	if (Hbci::debugLevel()>1)
	  fprintf(stderr,"Readdata: %s\n",
		  err.errorString().c_str());
	return false;
      }
    } // if error
    else {
      if (tmp.length()==0)
	// broken pipe
	return false;
      result+=tmp;
      size-=tmp.length();
    }
    if (difftime(time(0), startTime)>_timeout) {
      if (Hbci::debugLevel()>0)
	fprintf(stderr,
		"Connection::recieveData(): timeout\n");
      return false;
    }
  } // while
  if (size>0)
    return false;

  // DEBUG
  if (Hbci::debugLevel()>5)
    fprintf(stderr, "RESPONSE:\n%s\n",
	    String::dumpToString(result).c_str());

  response=result;
    return true;
}

} /* namespace HBCI */
