(**
   Implements scaleable images. If you have a nice scaleable image that
   maybe widly used, add it here.

  TODO
  * More images, make existing images nicer
**)

MODULE VO:VecImage;

(*
    Implements some vector images.
    Copyright (C) 1997 Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)


IMPORT D := VO:Base:Display,
       U := VO:Base:Util,

       G := VO:Object;


CONST

  simpleCheck* =  0;
  amigaRadio * =  1;
  simpleCycle* =  2;
  cycle3D    * =  3;
  w95Radio   * =  4;
  w95Check   * =  5;
  arrowLeft  * =  6;
  arrowRight * =  7;
  arrowUp    * =  8;
  arrowDown  * =  9;
  knob       * = 10;
  hSlider    * = 11;
  vSlider    * = 12;
  return     * = 13;
  escape     * = 14;
  default    * = 15;
  w95Knob    * = 16;
  w95Up      * = 17;
  w95Down    * = 18;
  w95Left    * = 19;
  w95Right   * = 20;
  info       * = 21;
  atention   * = 22;
  warning    * = 23;
  question   * = 24;
  hMover     * = 25;
  vMover     * = 26;
  hLineIn3D  * = 27;
  vLineIn3D  * = 28;
  w95HMover  * = 29;
  w95VMover  * = 30;
  w95Combo   * = 31;
  combo      * = 32;
  led        * = 33;

  imageCount * = 34;

  checkmark  * =  w95Check;
  radio      * =  w95Radio;
  cycle      * =  simpleCycle;


TYPE
  Prefs*        = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc*    = RECORD (G.PrefsDesc)
                  END;

  ImageName     = ARRAY 30 OF CHAR;


  VecImage*     = POINTER TO VecImageDesc;
  VecImageDesc* = RECORD (G.ImageDesc)
                    type : LONGINT;
                  END;
VAR
  prefs*  : Prefs;
  images- : ARRAY imageCount OF ImageName;

  PROCEDURE (v : VecImage) Init*;

  BEGIN
    v.Init^;

    v.SetPrefs(prefs);
    
    v.type:=simpleCheck;
  END Init;

  PROCEDURE (v : VecImage) Set*(type : LONGINT);

  BEGIN
    IF v.visible THEN
      v.type:=type;
      v.Redraw;
    ELSE
      v.type:=type;
    END;

    v.SetFlags({G.stdFocus});

    (* Setting flags depending of image type *)

    CASE v.type OF
      led,         (* Currently the LED image does not draw differently when disabled.
                      This is just a hack to ease use of VO:State *)
      w95Check,
      w95Radio,
      w95Up,
      w95Down,
      w95Left,
      w95Right:
        v.SetFlags({G.canDisable});
    | hSlider,
      vSlider:
        v.RemoveFlags({G.stdFocus});
    ELSE
    END;

  END Set;

  PROCEDURE (v : VecImage) Get*():LONGINT;

  BEGIN
    RETURN v.type;
  END Get;


  PROCEDURE (v : VecImage) DrawRectangle(draw : D.DrawInfo;
                                         x,y,w,h : LONGINT; in : BOOLEAN);

  BEGIN
    IF in THEN
      draw.PushForeground(D.shadowColor);
    ELSE
      draw.PushForeground(D.shineColor);
    END;
    draw.DrawLine(x,y+h-1,x,y);
    draw.DrawLine(x+1,y,x+w-1,y);
    draw.PopForeground;

    IF in THEN
      draw.PushForeground(D.shineColor);
    ELSE
      draw.PushForeground(D.shadowColor);
    END;
    draw.DrawLine(x+w-1,y+1,x+w-1,y+h-1);
    draw.DrawLine(x+w-2,y+h-1,x+1,y+h-1);
    draw.PopForeground;
  END DrawRectangle;

  PROCEDURE (v : VecImage) CalcSize*;

  VAR
    unit : LONGINT;

  BEGIN
    IF D.display.spaceWidth>D.display.spaceHeight THEN
      unit:=D.display.spaceWidth;
    ELSE
      unit:=D.display.spaceHeight;
    END;


    CASE v.type OF
      simpleCycle:
        v.width:=3*unit;
        v.height:=3*unit;
    | cycle3D:
        v.width:=3*unit;
        v.height:=3*unit;
    | simpleCheck:
        v.width:=2*unit+4;
        v.height:=2*unit+4;
    | w95Check:
        v.width:=2*unit+2;
        v.height:=2*unit+2;
    | amigaRadio:
        v.width:=U.RoundUpEven(5*unit DIV 2)-1;
        v.height:=U.RoundUpEven(5*unit DIV 2)-1;
    | w95Radio:
        v.width:=U.RoundUpEven(9*unit DIV 4);
        v.height:=U.RoundUpEven(9*unit DIV 4);
    | arrowLeft,
      arrowRight,
      arrowUp,
      arrowDown:
        v.width:=(3*unit) DIV 2;
        v.height:=(3*unit) DIV 2;
    | w95Left,
      w95Right,
      w95Up,
      w95Down:
        v.width:=unit;
        v.height:=unit;
    | knob,
      w95Knob:
        v.width:=(5*unit) DIV 2;
        v.height:=(5*unit) DIV 2;
    | return,
      escape,
      default:
        v.width:=unit+6;
        v.height:=unit+6;
    | info,
      atention:
        v.width:=2*unit;
        v.height:=6*unit;
    | warning:
        v.width:=6*unit;
        v.height:=4*unit;
    | question:
        v.width:=2*unit;
        v.height:=6*unit;
    | hMover:
        v.width:=unit;
        v.height:=3*unit;
    | vMover:
        v.width:=3*unit;
        v.height:=unit;
    | hSlider:
        v.width:=U.RoundUpEven(5*unit);
        v.height:=U.MaxLong(4,2*unit);
    | vSlider:
        v.width:=U.MaxLong(4,2*unit);
        v.height:=U.RoundUpEven(5*unit);
    | hLineIn3D:
        v.width:=4;
        v.height:=2;
    | vLineIn3D:
        v.width:=2;
        v.height:=4;
    | w95HMover:
        v.width:=U.MaxLong(4,unit);
        v.height:=U.MaxLong(4,3*unit);
    | w95VMover:
        v.width:=U.MaxLong(4,3*unit);
        v.height:=U.MaxLong(4,unit);
    | w95Combo,
      combo:
        v.width:=(5*unit) DIV 2;
        v.height:=(5*unit) DIV 2;
    | led:
        v.width:=2*unit;
        v.height:=2*unit;
    END;
    v.minWidth:=v.width;
    v.minHeight:=v.height;

    v.CalcSize^;
  END CalcSize;

  PROCEDURE (v : VecImage) Draw* (x,y,width,height : LONGINT);

  VAR
    iy       : LONGINT;
    h,h4,h9,
    w,w6,w13,
    xs,ys    : LONGINT;
    points   : ARRAY 3 OF D.PointDesc;
    draw     : D.DrawInfo;

  BEGIN
    v.Draw^(x,y,width,height);

    IF ~v.Intersect(x,y,width,height) THEN
      RETURN;
    END;

    draw:=v.GetDrawInfo();

    CASE v.type OF
      simpleCheck:
        v.DrawBackground(v.x,v.y,v.width,v.height);

        IF D.selected IN draw.mode THEN
          draw.PushForeground(D.shadowColor);
        ELSE
          draw.PushForeground(D.shineColor);
        END;
        draw.DrawLine(v.x,v.y+v.height-1,v.x,v.y+1);
        draw.DrawLine(v.x,v.y,v.x+v.width-1,v.y);
        draw.DrawLine(v.x+1,v.y+v.height-2,v.x+1,v.y+2);
        draw.DrawLine(v.x+1,v.y+1,v.x+v.width-2,v.y+1);
        draw.PopForeground;

        IF D.selected IN draw.mode THEN
          draw.PushForeground(D.shineColor);
        ELSE
          draw.PushForeground(D.shadowColor);
        END;
        draw.DrawLine(v.x+v.width-1,v.y+1,v.x+v.width-1,v.y+v.height-1);
        draw.DrawLine(v.x+v.width-1,v.y+v.height-1,v.x,v.y+v.height-1);
        draw.DrawLine(v.x+v.width-2,v.y+2,v.x+v.width-2,v.y+v.height-2);
        draw.DrawLine(v.x+v.width-2,v.y+v.height-2,v.x+1,v.y+v.height-2);
        draw.PopForeground;

        IF D.selected IN draw.mode THEN
          draw.PushForeground(D.halfShadowColor);
          draw.FillRectangle(v.x+4,v.y+4,v.width-8,v.height-8);
          draw.PopForeground;
        END;

    | w95Check:
        IF D.disabled IN draw.mode THEN
          v.DrawBackground(v.x,v.y,v.width,v.height);
        ELSE
          draw.PushForeground(D.whiteColor);
          draw.FillRectangle(v.x,v.y,v.width,v.height);
          draw.PopForeground;
        END;

        draw.PushForeground(D.halfShadowColor);
        draw.DrawLine(v.x,v.y+v.height-1,v.x,v.y+1);
        draw.DrawLine(v.x,v.y,v.x+v.width-1,v.y);
        draw.PopForeground;

        draw.PushForeground(D.shadowColor);
        draw.DrawLine(v.x+1,v.y+v.height-2,v.x+1,v.y+2);
        draw.DrawLine(v.x+1,v.y+1,v.x+v.width-2,v.y+1);
        draw.PopForeground;

        draw.PushForeground(D.shineColor);
        draw.DrawLine(v.x+v.width-1,v.y+1,v.x+v.width-1,v.y+v.height-1);
        draw.DrawLine(v.x+v.width-1,v.y+v.height-1,v.x,v.y+v.height-1);
        draw.PopForeground;

        draw.PushForeground(v.background);
        draw.DrawLine(v.x+v.width-2,v.y+2,v.x+v.width-2,v.y+v.height-2);
        draw.DrawLine(v.x+v.width-2,v.y+v.height-2,v.x+1,v.y+v.height-2);
        draw.PopForeground;


        IF D.selected IN draw.mode THEN
          IF D.disabled IN draw.mode THEN
            draw.PushForeground(D.halfShadowColor);
          ELSE
            draw.PushForeground(D.shadowColor);
          END;
          draw.PushStyle(2,D.normalPen);
          draw.DrawLine(v.x+3,v.y+3,v.x+v.width-4,v.y+v.height-4);
          draw.DrawLine(v.x+v.width-4,v.y+3,v.x+3,v.y+v.height-4);
          draw.PopStyle;
          draw.PopForeground;
        END;

    | amigaRadio:
        v.DrawBackground(v.x,v.y,v.width,v.height);

        draw.PushStyle(2,D.normalPen);
        IF D.selected IN draw.mode THEN
          draw.PushForeground(D.shadowColor);
          draw.DrawArc(v.x+1,v.y+1,v.width-2,v.height-2,45*64,180*64);
          draw.PopForeground;
          draw.PushForeground(D.shineColor);
          draw.DrawArc(v.x+1,v.y+1,v.width-2,v.height-2,226*64,180*64);
          draw.PopForeground;
          draw.PushForeground(D.halfShadowColor);
          draw.FillArc(v.x+4,v.y+4,v.width-8,v.height-8,0,360*64);
          draw.PopForeground;
        ELSE
          draw.PushForeground(D.shineColor);
          draw.DrawArc(v.x+1,v.y+1,v.width-2,v.height-2,45*64,180*64);
          draw.PopForeground;
          draw.PushForeground(D.shadowColor);
          draw.DrawArc(v.x+1,v.y+1,v.width-2,v.height-2,226*64,180*64);
          draw.PopForeground;
        END;
        draw.PopStyle;

    | w95Radio:
        v.DrawBackground(v.x,v.y,v.width,v.height);

        IF D.disabled IN draw.mode THEN
          draw.PushForeground(v.background);
        ELSE
          draw.PushForeground(D.whiteColor);
        END;
        draw.FillArc(v.x,v.y,v.width,v.height,0,360*64);
        draw.PopForeground;

        draw.PushStyle(2,D.normalPen);
        draw.PushForeground(D.halfShadowColor);
        draw.DrawArc(v.x,v.y,v.width,v.height,45*64,180*64);
        draw.PopForeground;
        draw.PopStyle;

        draw.PushForeground(D.shadowColor);
        draw.DrawArc(v.x+1,v.y+1,v.width-2,v.height-2,45*64,180*64);
        draw.PopForeground;

        draw.PushStyle(2,D.normalPen);
        draw.PushForeground(D.shineColor);
        draw.DrawArc(v.x,v.y,v.width,v.height,225*64,180*64);
        draw.PopForeground;
        draw.PopStyle;

        draw.PushForeground(v.background);
        draw.DrawArc(v.x+1,v.y+1,v.width-2,v.height-2,225*64,180*64);
        draw.PopForeground;

        IF D.selected IN draw.mode THEN
          IF D.disabled IN draw.mode THEN
            draw.PushForeground(D.halfShadowColor);
          ELSE
            draw.PushForeground(D.shadowColor);
          END;
          draw.FillArc(v.x+4,v.y+4,v.width-8,v.height-8,0,360*64);
          draw.PopForeground;
        END;

    | simpleCycle:
        IF (D.selected IN draw.mode) & ~(G.noHighlight IN v.flags) THEN
          draw.PushForeground(D.fillColor);
          draw.FillRectangle(v.x,v.y,v.width,v.height);
          draw.PopForeground;
        ELSE
          v.DrawBackground(v.x,v.y,v.width,v.height);
        END;

        draw.PushForeground(D.textColor);
        draw.PushStyle(2,D.normalPen);

        w:=5*v.width DIV 10;
        h:=5*v.height DIV 10;
        h4:=4*h DIV 10;
        h9:=9*h DIV 10;
        w6:=6*w DIV 10;
        w13:=13*w DIV 10;
        xs:=v.x+v.width DIV 5;
        ys:=v.y+v.height DIV 4;
        draw.DrawArc(xs,ys,w,h,15*64,270*64);
        draw.DrawLine(xs+w,   ys+h9, xs+w6,  ys+h4);
        draw.DrawLine(xs+w6,  ys+h4, xs+w13, ys+h4);
        draw.DrawLine(xs+w13, ys+h4, xs+w,   ys+h9);

        draw.PopForeground;
        draw.PopStyle;

    | cycle3D:
        xs:=v.x+D.display.spaceWidth DIV 2;
        w:= v.width- 2*(D.display.spaceWidth DIV 2);
        ys:=v.y+D.display.spaceHeight;
        h:=v.height-2*D.display.spaceHeight;


        v.DrawBackground(v.x,v.y,v.width,v.height);

        draw.PushForeground(D.shineColor);
        draw.DrawLine(xs,ys+h-1,xs,ys+1);
        draw.DrawLine(xs,ys,xs+w-1,ys);

        draw.DrawLine(xs+1,ys+h-2,xs+1,ys+2);
        draw.DrawLine(xs+1,ys+1,xs+w-2,ys+1);
        draw.PopForeground;

        draw.PushForeground(D.shadowColor);
        draw.DrawLine(xs+w-1,ys+1,xs+w-1,ys+h-1);
        draw.DrawLine(xs+w-1,ys+h-1,xs,ys+h-1);

        draw.DrawLine(xs+w-2,ys+2,xs+w-2,ys+h-2);
        draw.DrawLine(xs+w-2,ys+h-2,xs+1,ys+h-2);
        draw.PopForeground;

    | arrowLeft:
        v.DrawBackground(v.x,v.y,v.width,v.height);

        IF D.selected IN draw.mode THEN
          draw.PushForeground(D.shineColor);
        ELSE
          draw.PushForeground(D.shadowColor);
        END;
        draw.DrawLine(v.x+v.width-1,v.y,v.x+v.width-1,v.y+v.height-1);
        draw.DrawLine(v.x+v.width-1,v.y+v.height-1,v.x,v.y+v.height DIV 2);
        draw.PopForeground;
        IF D.selected IN draw.mode THEN
          draw.PushForeground(D.shadowColor);
        ELSE
          draw.PushForeground(D.shineColor);
        END;
        draw.DrawLine(v.x+v.width-1,v.y,v.x,v.y+v.height DIV 2);
        draw.PopForeground;

    | arrowRight:
        v.DrawBackground(v.x,v.y,v.width,v.height);

        IF D.selected IN draw.mode THEN
          draw.PushForeground(D.shineColor);
        ELSE
          draw.PushForeground(D.shadowColor);
        END;
        draw.DrawLine(v.x,v.y+v.height-1,v.x+v.width-1,v.y+v.height DIV 2);
        draw.PopForeground;
        IF D.selected IN draw.mode THEN
          draw.PushForeground(D.shadowColor);
        ELSE
          draw.PushForeground(D.shineColor);
        END;
        draw.DrawLine(v.x,v.y+v.height-1,v.x,v.y);
        draw.DrawLine(v.x,v.y,v.x+v.width-1,v.y+v.height DIV 2);
        draw.PopForeground;

    | arrowUp:
        v.DrawBackground(v.x,v.y,v.width,v.height);

        IF D.selected IN draw.mode THEN
          draw.PushForeground(D.shineColor);
        ELSE
          draw.PushForeground(D.shadowColor);
        END;
        draw.DrawLine(v.x,v.y+v.height-1,v.x+v.width-1,v.y+v.height-1);
        draw.DrawLine(v.x+v.width-1,v.y+v.height-1,v.x+v.width DIV 2,v.y);
        draw.PopForeground;
        IF D.selected IN draw.mode THEN
          draw.PushForeground(D.shadowColor);
        ELSE
          draw.PushForeground(D.shineColor);
        END;
        draw.DrawLine(v.x,v.y+v.height-1,v.x+v.width DIV 2,v.y);
        draw.PopForeground;

    | arrowDown:
        v.DrawBackground(v.x,v.y,v.width,v.height);

        IF D.selected IN draw.mode THEN
          draw.PushForeground(D.shadowColor);
        ELSE
          draw.PushForeground(D.shineColor);
        END;
        draw.DrawLine(v.x+v.width-1,v.y,v.x,v.y);
        draw.DrawLine(v.x,v.y,v.x+v.width DIV 2,v.y+v.height-1);
        draw.PopForeground;
        IF D.selected IN draw.mode THEN
          draw.PushForeground(D.shineColor);
        ELSE
          draw.PushForeground(D.shadowColor);
        END;
        draw.DrawLine(v.x+v.width-1,v.y,v.x+v.width DIV 2,v.y+v.height-1);
        draw.PopForeground;

    | knob:
        draw.PushForeground(D.shineColor);
        draw.DrawLine(v.x,  v.y+v.height-1,v.x,          v.y+1);
        draw.DrawLine(v.x,  v.y,           v.x+v.width-1,v.y);
        draw.DrawLine(v.x+1,v.y+v.height-2,v.x+1,        v.y+2);
        draw.DrawLine(v.x+1,v.y+1,         v.x+v.width-2,v.y+1);
        draw.PopForeground;

        draw.PushForeground(D.shadowColor);
        draw.DrawLine(v.x+v.width-1,v.y+1,         v.x+v.width-1,v.y+v.height-1);
        draw.DrawLine(v.x+v.width-1,v.y+v.height-1,v.x,          v.y+v.height-1);
        draw.DrawLine(v.x+v.width-2,v.y+2,         v.x+v.width-2,v.y+v.height-2);
        draw.DrawLine(v.x+v.width-2,v.y+v.height-2,v.x+1,        v.y+v.height-2);
        draw.PopForeground;

        draw.PushForeground(D.buttonBackgroundColor);
        draw.FillRectangle(v.x+2,v.y+2,v.width-4,v.height-4);
        draw.PopForeground;

        IF (v.width>2*D.display.spaceWidth) & (v.height>2*D.display.spaceHeight) THEN
          draw.PushStyle(2,D.normalPen);
          draw.PushForeground(D.shadowColor);
          draw.DrawArc(v.x+(v.width-D.display.spaceWidth) DIV 2,v.y+(v.height-D.display.spaceHeight) DIV 2,
                       D.display.spaceWidth,D.display.spaceHeight,45*64,180*64);
          draw.PopForeground;

          draw.PushForeground(D.shineColor);
          draw.DrawArc(v.x+(v.width-D.display.spaceWidth) DIV 2,v.y+(v.height-D.display.spaceHeight) DIV 2,
                       D.display.spaceWidth,D.display.spaceHeight,226*64,180*64);
          draw.PopForeground;
          draw.PopStyle;
        END;

    | hSlider,
      vSlider:
        v.DrawRectangle(draw,v.x,v.y,v.width,v.height,FALSE);
        v.DrawRectangle(draw,v.x+1,v.y+1,v.width-2,v.height-2,FALSE);

        IF D.selected IN draw.mode THEN
          draw.PushForeground(D.fillColor);
        ELSIF v.DisplayFocus() THEN
          draw.PushForeground(D.focusColor);
        ELSE
          draw.PushForeground(D.buttonBackgroundColor);
        END;
        draw.FillRectangle(v.x+2,v.y+2,v.width-4,v.height-4);
        draw.PopForeground;

        IF v.type=hSlider THEN
          draw.PushForeground(D.shadowColor);
          draw.DrawLine(v.x+v.width DIV 2,v.y+2,v.x+v.width DIV 2,v.y+v.height-3);
          draw.PopForeground;
          draw.PushForeground(D.shineColor);
          draw.DrawLine(v.x+v.width DIV 2+1,v.y+2,v.x+v.width DIV 2+1,v.y+v.height-3);
          draw.PopForeground;
        ELSE
          draw.PushForeground(D.shadowColor);
          draw.DrawLine(v.x+2,v.y+v.height DIV 2,v.x+v.width-3,v.y+v.height DIV 2);
          draw.PopForeground;
          draw.PushForeground(D.shineColor);
          draw.DrawLine(v.x+2,v.y+v.height DIV 2+1,v.x+v.width-3,v.y+v.height DIV 2+1);
          draw.PopForeground;
        END;

    | w95Knob:
        draw.PushForeground(v.background);
        draw.DrawLine(v.x,  v.y+v.height-1,v.x,          v.y+1);
        draw.DrawLine(v.x,  v.y,           v.x+v.width-2,v.y);
        draw.PopForeground;

        draw.PushForeground(D.shineColor);
        draw.DrawLine(v.x+1,v.y+v.height-2,v.x+1,        v.y+2);
        draw.DrawLine(v.x+1,v.y+1,         v.x+v.width-3,v.y+1);
        draw.PopForeground;

        draw.PushForeground(D.shadowColor);
        draw.DrawLine(v.x+v.width-1,v.y,           v.x+v.width-1,v.y+v.height-1);
        draw.DrawLine(v.x+v.width-1,v.y+v.height-1,v.x,          v.y+v.height-1);
        draw.PopForeground;

        draw.PushForeground(D.halfShadowColor);
        draw.DrawLine(v.x+v.width-2,v.y+1,         v.x+v.width-2,v.y+v.height-2);
        draw.DrawLine(v.x+v.width-2,v.y+v.height-2,v.x+1,        v.y+v.height-2);
        draw.PopForeground;

        draw.PushForeground(D.buttonBackgroundColor);
        draw.FillRectangle(v.x+2,v.y+2,v.width-4,v.height-4);
        draw.PopForeground;

    | return:
        iy:=v.y+v.height;
        draw.PushForeground(D.shineColor);
        draw.DrawLine(v.x,iy-v.height DIV 2,v.x+v.width DIV 2,iy);
        draw.DrawLine(v.x+v.width DIV 2,iy,v.x+v.width DIV 2,iy-v.height DIV 3);
        draw.DrawLine(v.x+v.width DIV 2,iy-v.height DIV 3,v.x+v.width,iy-v.height DIV 3);
        draw.DrawLine(v.x+v.width,iy-v.height DIV 3,v.x+v.width,iy-v.height+v.height DIV 3);
        draw.PopForeground;

        draw.PushForeground(D.shadowColor);
        draw.DrawLine(v.x+v.width,iy-v.height+v.height DIV 3,v.x+v.width DIV 2,iy-v.height+v.height DIV 3);
        draw.DrawLine(v.x+v.width DIV 2,iy-v.height+v.height DIV 3,v.x+v.width DIV 2, iy-v.height);
        draw.DrawLine(v.x+v.width DIV 2,iy-v.height,v.x,iy-v.height DIV 2);
        draw.PopForeground;

    | escape:
        iy:=v.y+v.height;
        draw.PushForeground(D.shineColor);
        draw.DrawLine(v.x+v.width DIV 2, iy-v.height,v.x+v.width,iy-v.height DIV 2);
        draw.DrawLine(v.x+v.width,iy-v.height DIV 2,v.x+v.width-v.width DIV 3,iy-v.height DIV 2);
        draw.DrawLine(v.x+v.width-v.width DIV 3,iy-v.height DIV 2,v.x+v.width-v.width DIV 3, iy);
        draw.DrawLine(v.x+v.width-v.width DIV 3, iy,v.x+v.width DIV 3,iy);
        draw.PopForeground;

        draw.PushForeground(D.shadowColor);
        draw.DrawLine(v.x+v.width DIV 3,iy,v.x+v.width DIV 3,iy-v.height DIV 2);
        draw.DrawLine(v.x+v.width DIV 3,iy-v.height DIV 2,v.x,iy-v.height DIV 2);
        draw.DrawLine(v.x,iy-v.height DIV 2,v.x+v.width DIV 2,iy-v.height);
        draw.PopForeground;

    | default:
        v.DrawRectangle(draw,v.x,v.y,v.width,v.height,TRUE);

    | w95Left:
        v.DrawBackground(v.x,v.y,v.width,v.height);

        IF D.disabled IN draw.mode THEN
          draw.PushForeground(D.halfShineColor);
        ELSE
          draw.PushForeground(D.shadowColor);
        END;
        points[0].x:=SHORT(v.x+v.width DIV 4);
        points[0].y:=SHORT(v.y+v.height DIV 2);
        points[1].x:=SHORT(v.x+v.width-v.width DIV 4);
        points[1].y:=SHORT(v.y+v.height DIV 4);
        points[2].x:=SHORT(v.x+v.width-v.width DIV 4);
        points[2].y:=SHORT(v.y+v.height-v.height DIV 4);
        draw.FillPolygon(points,3);
        draw.PopForeground;

    | w95Right:
        v.DrawBackground(v.x,v.y,v.width,v.height);

        IF D.disabled IN draw.mode THEN
          draw.PushForeground(D.halfShineColor);
        ELSE
          draw.PushForeground(D.shadowColor);
        END;
        points[0].x:=SHORT(v.x+v.width-v.width DIV 4);
        points[0].y:=SHORT(v.y+v.height DIV 2);
        points[1].x:=SHORT(v.x+v.width DIV 4);
        points[1].y:=SHORT(v.y+v.height DIV 4);
        points[2].x:=SHORT(v.x+v.width DIV 4);
        points[2].y:=SHORT(v.y+v.height-v.height DIV 4);
        draw.FillPolygon(points,3);
        draw.PopForeground;

    | w95Up:
        v.DrawBackground(v.x,v.y,v.width,v.height);

        IF D.disabled IN draw.mode THEN
          draw.PushForeground(D.halfShineColor);
        ELSE
          draw.PushForeground(D.shadowColor);
        END;
        points[0].x:=SHORT(v.x+v.width DIV 2);
        points[0].y:=SHORT(v.y+v.height DIV 4);
        points[1].x:=SHORT(v.x+v.width DIV 4);
        points[1].y:=SHORT(v.y+v.height-v.height DIV 4);
        points[2].x:=SHORT(v.x+v.width-v.width DIV 4);
        points[2].y:=SHORT(v.y+v.height-v.height DIV 4);
        draw.FillPolygon(points,3);
        draw.PopForeground;

    | w95Down:
        v.DrawBackground(v.x,v.y,v.width,v.height);

        IF D.disabled IN draw.mode THEN
          draw.PushForeground(D.halfShineColor);
        ELSE
          draw.PushForeground(D.shadowColor);
        END;
        points[0].x:=SHORT(v.x+v.width DIV 4);
        points[0].y:=SHORT(v.y+v.height DIV 4);
        points[1].x:=SHORT(v.x+v.width-v.width DIV 4);
        points[1].y:=SHORT(v.y+v.height DIV 4);
        points[2].x:=SHORT(v.x+v.width DIV 2);
        points[2].y:=SHORT(v.y+v.height-v.height DIV 4);
        draw.FillPolygon(points,3);
        draw.PopForeground;

    | info:
        v.DrawBackground(v.x,v.y,v.width,v.height);

        h:=v.height DIV 6;
        draw.PushForeground(D.halfShadowColor);
        draw.FillRectangle(v.x+1,v.y+1,v.width-2,h-2);
        draw.PopForeground;
        v.DrawRectangle(draw,v.x,v.y,v.width,h,FALSE);

        h4:=(2*v.height) DIV 3;
        draw.PushForeground(D.halfShadowColor);
        draw.FillRectangle(v.x+1,v.y+2*h+1,v.width-2,h4-2);
        draw.PopForeground;
        v.DrawRectangle(draw,v.x,v.y+2*h,v.width,h4,FALSE);

    | atention:
        v.DrawBackground(v.x,v.y,v.width,v.height);

        h:=(2*v.height) DIV 3;
        draw.PushForeground(D.warnColor);
        draw.FillRectangle(v.x+1,v.y+1,v.width-2,h-2);
        draw.PopForeground;
        v.DrawRectangle(draw,v.x,v.y,v.width,h,FALSE);

        h4:=v.height DIV 6;
        draw.PushForeground(D.warnColor);
        draw.FillRectangle(v.x+1,v.y+h+h4+1,v.width-2,h4-2);
        draw.PopForeground;
        v.DrawRectangle(draw,v.x,v.y+h+h4,v.width,h4,FALSE);

    | warning:
        h:=v.height DIV 3;
        w:=v.width DIV 4;

        draw.PushForeground(D.shadowColor);
        draw.DrawLine(v.x+3*w,v.y,v.x+4*w,v.y+h);
        draw.DrawLine(v.x+4*w,v.y+h,v.x+4*w,v.y+2*h);
        draw.DrawLine(v.x+4*w,v.y+2*h,v.x+3*w,v.y+3*h);
        draw.DrawLine(v.x+3*w,v.y+3*h,v.x+w,v.y+3*h);
        draw.DrawLine(v.x+w,v.y+3*h,v.x,v.y+2*h);
        draw.PopForeground;

        draw.PushForeground(D.shineColor);
        draw.DrawLine(v.x,v.y+2*h,v.x,v.y+h);
        draw.DrawLine(v.x,v.y+h,v.x+w,v.y);
        draw.DrawLine(v.x+w,v.y,v.x+3*w,v.y);
        draw.PopForeground;

        (* line *)

        v.DrawRectangle(draw,v.x+w,v.y+h,2*w,h,TRUE);

    | question:
        v.DrawBackground(v.x,v.y,v.width,v.height);

        h:=v.height DIV 6;
        w:=v.width DIV 2;

        draw.PushForeground(D.shadowColor);
        draw.DrawLine(v.x+2*w,v.y,v.x+2*w,v.y+3*h);
        draw.DrawLine(v.x+2*w,v.y+3*h,v.x+w,v.y+3*h);
        draw.DrawLine(v.x+w,v.y+3*h,v.x+w,v.y+4*h);
        draw.DrawLine(v.x+w,v.y+4*h,v.x,v.y+4*h);
        draw.PopForeground;

        draw.PushForeground(D.shineColor);
        draw.DrawLine(v.x,v.y,v.x+2*w,v.y);

        draw.DrawLine(v.x,v.y+4*h,v.x,v.y+2*h);
        draw.DrawLine(v.x,v.y+2*h,v.x+w,v.y+2*h);
        draw.DrawLine(v.x+w,v.y+2*h,v.x+w,v.y+h);
        draw.PopForeground;

        draw.PushForeground(D.shadowColor);
        draw.DrawLine(v.x+w,v.y+h,v.x,v.y+h);
        draw.PopForeground;

        draw.PushForeground(D.shineColor);
        draw.DrawLine(v.x,v.y+h,v.x,v.y);
        draw.PopForeground;

        (* point *)

        v.DrawRectangle(draw,v.x,v.y+5*h,w,h,FALSE);

    | hMover:
        xs:=v.width DIV 2;
        ys:=2*D.display.spaceHeight;

        draw.PushForeground(D.shadowColor);
        draw.DrawLine(v.x+xs,v.y,v.x+xs,v.y+D.display.spaceHeight-1);
        draw.PopForeground;
        draw.PushForeground(D.shineColor);
        draw.DrawLine(v.x+xs+1,v.y,v.x+xs+1,v.y+D.display.spaceHeight-1);
        draw.PopForeground;

        v.DrawRectangle(draw,v.x+(v.width-D.display.spaceWidth) DIV 2+1,
                        v.y+D.display.spaceHeight,
                        D.display.spaceWidth,
                        D.display.spaceHeight,FALSE);

        draw.PushForeground(D.shadowColor);
        draw.DrawLine(v.x+xs,v.y+ys,v.x+xs,v.y+v.height-1);
        draw.PopForeground;
        draw.PushForeground(D.shineColor);
        draw.DrawLine(v.x+xs+1,v.y+ys,v.x+xs+1,v.y+v.height-1);
        draw.PopForeground;

    | vMover:
        xs:=2*D.display.spaceWidth;
        ys:=v.height DIV 2;

        draw.PushForeground(D.shadowColor);
        draw.DrawLine(v.x,v.y+ys,v.x+D.display.spaceWidth-1,v.y+ys);
        draw.PopForeground;
        draw.PushForeground(D.shineColor);
        draw.DrawLine(v.x,v.y+ys+1,v.x+D.display.spaceWidth-1,v.y+ys+1);
        draw.PopForeground;

        v.DrawRectangle(draw,v.x+D.display.spaceWidth,
                        v.y+(v.height-D.display.spaceHeight) DIV 2+1,
                        D.display.spaceWidth,
                        D.display.spaceHeight,FALSE);

        draw.PushForeground(D.shadowColor);
        draw.DrawLine(v.x+xs,v.y+ys,v.x+v.width-1,v.y+ys);
        draw.PopForeground;
        draw.PushForeground(D.shineColor);
        draw.DrawLine(v.x+xs,v.y+ys+1,v.x+v.width-1,v.y+ys+1);
        draw.PopForeground;

    | hLineIn3D:
       v.DrawRectangle(draw,v.x,v.y+(v.height-2) DIV 2,v.width,2,TRUE);

    | vLineIn3D:
       v.DrawRectangle(draw,v.x+(v.width-2) DIV 2,v.y,2,v.height,TRUE);

    | w95HMover,
      w95VMover:
        draw.PushForeground(v.background);
        draw.FillRectangle(v.x+2,v.y+2,v.width-4,v.height-4);
        v.DrawRectangle(draw,v.x,v.y,v.width,v.height,FALSE);
        v.DrawRectangle(draw,v.x+1,v.y+1,v.width-2,v.height-2,FALSE);
        draw.PopForeground;

    | w95Combo:
        v.DrawBackground(v.x,v.y,v.width,v.height);

        draw.PushForeground(v.background);
        draw.DrawLine(v.x,  v.y+v.height-1,v.x,          v.y+1);
        draw.DrawLine(v.x,  v.y,           v.x+v.width-2,v.y);
        draw.PopForeground;

        draw.PushForeground(D.shineColor);
        draw.DrawLine(v.x+1,v.y+v.height-2,v.x+1,        v.y+2);
        draw.DrawLine(v.x+1,v.y+1,         v.x+v.width-3,v.y+1);
        draw.PopForeground;

        draw.PushForeground(D.shadowColor);
        draw.DrawLine(v.x+v.width-1,v.y,           v.x+v.width-1,v.y+v.height-1);
        draw.DrawLine(v.x+v.width-1,v.y+v.height-1,v.x,          v.y+v.height-1);
        draw.PopForeground;

        draw.PushForeground(D.halfShadowColor);
        draw.DrawLine(v.x+v.width-2,v.y+1,         v.x+v.width-2,v.y+v.height-2);
        draw.DrawLine(v.x+v.width-2,v.y+v.height-2,v.x+1,        v.y+v.height-2);
        draw.PopForeground;

        IF D.disabled IN draw.mode THEN
          draw.PushForeground(D.halfShineColor);
        ELSE
          draw.PushForeground(D.shadowColor);
        END;
        points[0].x:=SHORT(v.x+v.width DIV 4);
        points[0].y:=SHORT(v.y+v.height DIV 4);
        points[1].x:=SHORT(v.x+v.width-v.width DIV 4);
        points[1].y:=SHORT(v.y+v.height DIV 4);
        points[2].x:=SHORT(v.x+v.width DIV 2);
        points[2].y:=SHORT(v.y+v.height-v.height DIV 4);
        draw.FillPolygon(points,3);
        draw.PopForeground;

    | combo:
        v.DrawBackground(v.x,v.y,v.width,v.height);

        IF D.disabled IN draw.mode THEN
          draw.PushForeground(D.halfShineColor);
        ELSE
          draw.PushForeground(D.shadowColor);
        END;
        points[0].x:=SHORT(v.x+v.width DIV 4);
        points[0].y:=SHORT(v.y+v.height DIV 4);
        points[1].x:=SHORT(v.x+v.width-v.width DIV 4);
        points[1].y:=SHORT(v.y+v.height DIV 4);
        points[2].x:=SHORT(v.x+v.width DIV 2);
        points[2].y:=SHORT(v.y+v.height-v.height DIV 4);
        draw.FillPolygon(points,3);
        draw.PopForeground;
    | led:
        v.DrawBackground(v.x,v.y,v.width,v.height);

        IF D.selected IN draw.mode THEN
          IF D.display.colorMode=D.monochromeMode THEN
            draw.PushForeground(D.whiteColor);
          ELSE
            draw.PushForeground(D.warnColor);
          END;
          draw.FillArc(v.x,v.y,v.width-1,v.height-1,0*64,360*64);
          draw.PopForeground;
        ELSE
          draw.PushForeground(D.shadowColor);
          draw.FillArc(v.x,v.y,v.width-1,v.height-1,0*64,360*64);
          draw.PopForeground;
        END;

        draw.PushForeground(D.shineColor);
        draw.DrawArc(v.x+v.width DIV 4,v.y+v.height DIV 4,
                     v.width DIV 2,v.height DIV 2,90*64,60*64);
        draw.PopForeground;

        draw.PushForeground(D.shadowColor);
        draw.DrawArc(v.x,v.y,v.width,v.height,0*64,360*64);
        draw.PopForeground;
    END;
  END Draw;

  PROCEDURE (v :  VecImage) Hide*;

  BEGIN
    IF v.visible THEN
      v.DrawHide;
      v.Hide^;
    END;
  END Hide;

  PROCEDURE (v : VecImage) DrawFocus*;

  BEGIN
    IF (v.type=hSlider) OR (v.type=vSlider) THEN
      v.SetFlags({G.showFocus});
      v.Redraw;
    ELSE
      v.DrawFocus^;
    END;
  END DrawFocus;

  PROCEDURE (v : VecImage) HideFocus*;

  BEGIN
    IF (v.type=hSlider) OR (v.type=vSlider) THEN
      v.RemoveFlags({G.showFocus});
      v.Redraw;
    ELSE
      v.HideFocus^;
    END;
  END HideFocus;

  PROCEDURE GetImageEntry*(name : ARRAY OF CHAR):LONGINT;

  VAR
    x : LONGINT;

  BEGIN
    FOR x:=0 TO imageCount-1 DO
      IF name=images[x] THEN
        RETURN x;
      END;
    END;
    RETURN -1;
  END GetImageEntry;

  PROCEDURE CreateVecImage*():VecImage;

  VAR
    vecImage : VecImage;

  BEGIN
    NEW(vecImage);
    vecImage.Init;

    RETURN vecImage;
  END CreateVecImage;

BEGIN
  NEW(prefs);
  prefs.Init;

  images[simpleCheck]:="simpleCheck";
  images[amigaRadio] :="amigaRadio";
  images[simpleCycle]:="simpleCycle";
  images[cycle3D]    :="cycle3D";
  images[w95Radio]   :="w95Radio";
  images[w95Check]   :="w95Check";
  images[arrowLeft]  :="arrowLeft";
  images[arrowRight] :="arrowRight";
  images[arrowUp]    :="arrowUp";
  images[arrowDown]  :="arrowDown";
  images[knob]       :="knob";
  images[hSlider]    :="hSlider";
  images[vSlider]    :="vSlider";
  images[return]     :="return";
  images[escape]     :="escape";
  images[default]    :="default";
  images[w95Knob]    :="w95Knob";
  images[w95Up]      :="w95Up";
  images[w95Down]    :="w95Down";
  images[w95Left]    :="w95Left";
  images[w95Right]   :="w95Right";
  images[info]       :="info";
  images[atention]   :="atention";
  images[warning]    :="warning";
  images[question]   :="question";
  images[hMover]     :="hMover";
  images[vMover]     :="vMover";
  images[hLineIn3D]  :="hLineIn3D";
  images[vLineIn3D]  :="vLineIn3D";
  images[w95HMover]  :="w95HMover";
  images[w95VMover]  :="w95VMover";
  images[w95Combo]   :="w95Combo";
  images[combo]      :="combo";
  images[led]        :="led";
END VO:VecImage.