(**
   Implements horizontal and vertical knobs.

  TODO
  * Support for prior, next, home, end
**)

MODULE VO:Knob;

(*
    Implements a knob widget to be used in scroll gadget.
    Copyright (C) 2000  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT A  := VO:Base:Adjustment,
       D  := VO:Base:Display,
       E  := VO:Base:Event,
       F  := VO:Base:Frame,
       O  := VO:Base:Object,
       U  := VO:Base:Util,

       G  := VO:Object,
       V  := VO:VecImage;


CONST
  decAction* = 0;
  incAction* = 1;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the cycle is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 knob* : LONGINT;
               END;


  (* I try to ensure that top, visible and total are always
    valid models. But it is impossible to ensure that
    the values are correct, too, because they may be changed
    by clients. *)


  Knob*     = POINTER TO KnobDesc;
  KnobDesc* = RECORD (G.GadgetDesc)
                knob       : V.VecImage;
                offset     : LONGINT;
                corr       : LONGINT;
                adjustment : A.Adjustment;
                vert,
                selected   : BOOLEAN;
              END;

CONST
  knobPatWidth  = 16;
  knobPatHeight = 4;

VAR
  prefs*      : Prefs;
  knobPattern : ARRAY 8 OF CHAR;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frame:=F.none;
    p.knob:=V.w95Knob;
  END Init;

  PROCEDURE (k : Knob) Init*;

  BEGIN
    k.Init^;

    k.SetPrefs(prefs);

    k.SetFlags({G.canFocus});

    k.vert:=TRUE;

    k.adjustment:=NIL;

    k.selected:=FALSE;

    k.offset:=0;

    k.corr:=0;

    NEW(k.knob);
    k.knob.Init;
    k.knob.SetParent(k);
    k.knob.SetFlags({G.horizontalFlex,G.verticalFlex});
  END Init;

  PROCEDURE (k : Knob) Set*(vert : BOOLEAN);

  BEGIN
    k.vert:=vert;
  END Set;

  PROCEDURE (k : Knob) SetOffset*(offset : LONGINT);

  BEGIN
    k.corr:=offset;
  END SetOffset;

  PROCEDURE (k : Knob) SetModel*(model : O.Model);

  BEGIN
    IF k.adjustment#NIL THEN
      k.UnattachModel(k.adjustment.GetTopModel());
      k.UnattachModel(k.adjustment.GetVisibleModel());
      k.UnattachModel(k.adjustment.GetTotalModel())
    END;
    IF (model#NIL) & (model IS A.Adjustment) THEN
      k.adjustment:=model(A.Adjustment);
      k.AttachModel(k.adjustment.GetTopModel());
      k.AttachModel(k.adjustment.GetVisibleModel());
      k.AttachModel(k.adjustment.GetTotalModel());
    END;
  END SetModel;

  (**
    This function is used to check if an argument to SetModel
    was successfully accepted.
   **)

  PROCEDURE (k : Knob) ModelAccepted * (m : O.Model):BOOLEAN;

  BEGIN
    RETURN m=k.adjustment
  END ModelAccepted;


  PROCEDURE (k : Knob) CalcSize*;

  BEGIN
    k.knob.Set(k.prefs(Prefs).knob);
    k.knob.CalcSize;

    IF k.vert THEN
      k.width:=k.knob.oWidth;
      k.height:=k.knob.oWidth;
    ELSE
      k.width:=k.knob.oHeight;
      k.height:=k.knob.oHeight;
    END;

    k.minWidth:=k.width;
    k.minHeight:=k.height;

    k.CalcSize^;
  END CalcSize;


  PROCEDURE (k : Knob) HandleMouseEvent*(event : E.MouseEvent;
                                         VAR grab : G.Object):BOOLEAN;

  VAR
    new : LONGINT;

  BEGIN
    IF ~k.visible OR k.disabled THEN
      RETURN FALSE;
    END;

    IF (k.adjustment=NIL) OR ~k.adjustment.IsValid()
      OR (k.adjustment.GetVisible()=k.adjustment.GetTotal()) THEN
      RETURN FALSE;
    END;


    WITH event : E.ButtonEvent DO

      IF (event.type=E.mouseDown) & k.PointIsIn(event.x,event.y)
      & (event.button=E.button1) THEN
        IF k.knob.PointIsIn(event.x,event.y) THEN
          IF k.vert THEN
            k.offset:=event.y-k.knob.y;
          ELSE
            k.offset:=event.x-k.knob.x;
          END;
          k.selected:=TRUE;
          k.Redraw;
        ELSE
          IF k.vert THEN
            IF event.y<k.knob.y THEN
              IF k.adjustment.GetTop()+k.corr>k.adjustment.GetVisible() THEN
                k.adjustment.SetTop(k.adjustment.GetTop()-k.adjustment.GetVisible());
              ELSE
                k.adjustment.SetTop(1-k.corr);
              END;
            ELSIF event.y>k.knob.y+k.knob.height THEN
              IF k.adjustment.GetTop()+k.corr+k.adjustment.GetVisible()<k.adjustment.GetTotal()-k.adjustment.GetVisible()+1 THEN
                k.adjustment.SetTop(k.adjustment.GetTop()+k.adjustment.GetVisible());
              ELSE
                k.adjustment.SetTop(k.adjustment.GetTotal()-k.corr-k.adjustment.GetVisible()+1);
              END;
            END;
          ELSE
            IF event.x<k.knob.x THEN
              IF k.adjustment.GetTop()+k.corr>k.adjustment.GetVisible() THEN
                k.adjustment.SetTop(k.adjustment.GetTop()-k.adjustment.GetVisible());
              ELSE
                k.adjustment.SetTop(1-k.corr);
              END;
            ELSIF event.x>k.knob.x+k.knob.width THEN
              IF k.adjustment.GetTop()+k.corr+k.adjustment.GetVisible()<k.adjustment.GetTotal()-k.adjustment.GetVisible()+1 THEN
                k.adjustment.SetTop(k.adjustment.GetTop()+k.adjustment.GetVisible());
              ELSE
                k.adjustment.SetTop(k.adjustment.GetTotal()-k.adjustment.GetVisible()-k.corr+1);
              END;
            END;
          END;

        END;
        grab:=k;
        RETURN TRUE;
      ELSIF (event.type=E.mouseUp) & (event.button=E.button1) THEN
        k.selected:=FALSE;

        grab:=NIL;
        RETURN TRUE;
      END;
    | event : E.MotionEvent DO
      IF k.selected THEN
        IF k.vert THEN
          new:=k.adjustment.GetTop()+k.corr
                 +((event.y-k.knob.y-k.offset)*(k.adjustment.GetTotal()+k.corr))
                    DIV k.height + 1;
          IF (new>0) & (new<=k.adjustment.GetTotal()+k.corr-k.adjustment.GetVisible()+1) THEN
            k.adjustment.SetTop(new);
          ELSIF new<=0 THEN
            k.adjustment.SetTop(1-k.corr);
          ELSE
            k.adjustment.SetTop(k.adjustment.GetTotal()+k.corr-k.adjustment.GetVisible()+1);
          END;
        ELSE
          new:=k.adjustment.GetTop()+k.corr
                 +((event.x-k.knob.x-k.offset)*(k.adjustment.GetTotal()+k.corr))
                    DIV k.width + 1;
          IF (new>0) & (new<=k.adjustment.GetTotal()+k.corr-k.adjustment.GetVisible()+1) THEN
            k.adjustment.SetTop(new);
          ELSIF new<=0 THEN
            k.adjustment.SetTop(1-k.corr);
          ELSE
            k.adjustment.SetTop(k.adjustment.GetTotal()+k.corr-k.adjustment.GetVisible()+1);
          END;
        END;
      END;

      RETURN TRUE;
    ELSE
    END;
    RETURN FALSE;
  END HandleMouseEvent;

  PROCEDURE (k : Knob) HandleKeyEvent*(event : E.KeyEvent):BOOLEAN;

  VAR
    keysym   : LONGINT;

  BEGIN
    IF event.type=E.keyDown THEN
      keysym:=event.GetKey();
      IF k.vert & (keysym=E.up) THEN
        IF k.adjustment.GetTop()+k.corr>1 THEN
          k.adjustment.DecTop;
        END;
      ELSIF k.vert & (keysym=E.down) THEN
        IF k.adjustment.GetTop()+k.corr<=k.adjustment.GetTotal()+k.corr-k.adjustment.GetVisible() THEN
          k.adjustment.IncTop;
        END;
      ELSIF ~k.vert & (keysym=E.left) THEN
        IF k.adjustment.GetTop()+k.corr>1 THEN
          k.adjustment.DecTop;
        END;
      ELSIF ~k.vert & (keysym=E.left) THEN
        IF k.adjustment.GetTop()+k.corr<=k.adjustment.GetTotal()+k.corr-k.adjustment.GetVisible() THEN
          k.adjustment.IncTop;
        END;
      ELSE
        RETURN FALSE;
      END;
      RETURN TRUE;
    END;
    RETURN FALSE;
  END HandleKeyEvent;

  PROCEDURE (k : Knob) Receive*(message : O.Message);

  BEGIN
    WITH
      message : O.Action DO
        IF message.action=decAction THEN
          IF (k.adjustment#NIL) & (k.adjustment.GetTop()+k.corr>1) THEN
            k.adjustment.DecTop;
          END;
        ELSIF message.action=incAction THEN
          IF (k.adjustment#NIL)
          &  (k.adjustment.GetTop()+k.corr<=k.adjustment.GetTotal()+k.corr-k.adjustment.GetVisible()) THEN
            k.adjustment.IncTop;
          END;
        END;
    ELSE
      k.Receive^(message);
    END;
  END Receive;

  PROCEDURE (k : Knob) DrawKnob;

  VAR
    kSize,
    kStart,
    bSize   : LONGINT;
    draw    : D.DrawInfo;

  BEGIN
    draw:=k.GetDrawInfo();

    IF k.vert THEN
      bSize:=k.height;
    ELSE
      bSize:=k.width;
    END;

    IF (k.adjustment#NIL) & k.adjustment.IsValid()
     & ~(k.adjustment.GetVisible()=k.adjustment.GetTotal()) THEN

      IF k.adjustment.GetTotal()+k.corr=0 THEN
        kSize:=bSize;
        kStart:=0;
      ELSE
        kSize:=U.RoundDiv((bSize*k.adjustment.GetVisible()),k.adjustment.GetTotal()+k.corr);
        kStart:=U.RoundDiv(bSize*(k.adjustment.GetTop()+k.corr-1),k.adjustment.GetTotal()+k.corr);
      END;

      IF k.vert THEN
        IF kSize<k.knob.minWidth THEN
          kSize:=k.knob.minWidth;
        END;
      ELSE
        IF kSize<k.knob.minHeight THEN
          kSize:=k.knob.minHeight;
        END;
      END;

      IF kSize>bSize THEN
        kSize:=bSize;
      END;

      IF kStart+kSize>bSize THEN
        kStart:=bSize-kSize;
      END;

(*      s.knob.Hide; (* Does not work :-| *)*)

      IF k.vert THEN
        k.knob.MoveResize(k.x,k.y+kStart,k.width,kSize);
      ELSE
        k.knob.MoveResize(k.x+kStart,k.y,kSize,k.height);
      END;
      k.knob.Draw(k.oX,k.oY,k.oWidth,k.oHeight);

      draw.PushForeground(D.halfShineColor);
      draw.PushBackground(D.shineColor);
      draw.PushPattern(knobPattern,knobPatWidth,knobPatHeight,D.fbPattern);
      IF k.vert THEN
        draw.FillRectangle(k.x,k.y,k.width,k.knob.y-k.y);

        draw.FillRectangle(k.x,k.knob.y+k.knob.oHeight,
                           k.width,bSize-k.knob.oHeight-kStart);
      ELSE
        draw.FillRectangle(k.x,k.y,k.knob.x-k.x,k.height);

        draw.FillRectangle(k.knob.x+k.knob.oWidth,k.y,
                           bSize-k.knob.oWidth-kStart,k.height);
      END;
      draw.PopPattern;
      draw.PopForeground;
      draw.PopBackground;

    ELSE (* no models *)
      draw.PushForeground(D.halfShineColor);
      draw.PushBackground(D.shineColor);
      draw.PushPattern(knobPattern,knobPatWidth,knobPatHeight,D.fbPattern);
      IF k.vert THEN
        draw.FillRectangle(k.x,k.y,k.width,bSize);
      ELSE
        draw.FillRectangle(k.x,k.y,bSize,k.height);
      END;
      draw.PopPattern;
      draw.PopForeground;
      draw.PopBackground;
    END;
  END DrawKnob;

  PROCEDURE (k : Knob) Draw*(x,y,w,h : LONGINT);

  BEGIN
    k.Draw^(x,y,w,h);

    IF ~k.Intersect(x,y,w,h) THEN
      RETURN;
    END;

    k.DrawKnob;

    IF k.disabled THEN
      k.DrawDisabled;
    END;
  END Draw;


  PROCEDURE (k : Knob) Hide*;

  BEGIN
    IF k.visible THEN
      k.knob.Hide;
      k.DrawHide;
      k.Hide^;
    END;
  END Hide;

  PROCEDURE (k : Knob) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF k.visible & ~k.disabled THEN
      k.DrawKnob;
    END;
  END Resync;

  PROCEDURE CreateKnob*():Knob;

  VAR
    knob : Knob;

  BEGIN
    NEW(knob);
    knob.Init;

    RETURN knob;
  END CreateKnob;

BEGIN
  knobPattern[0]:=CHR(204); (* 11001100 *)
  knobPattern[1]:=CHR(204); (* 11001100 *)
  knobPattern[2]:=CHR(204); (* 11001100 *)
  knobPattern[3]:=CHR(204); (* 11001100 *)
  knobPattern[4]:=CHR(51);  (* 00110011 *)
  knobPattern[5]:=CHR(51);  (* 00110011 *)
  knobPattern[6]:=CHR(51);  (* 00110011 *)
  knobPattern[7]:=CHR(51);  (* 00110011 *)

  NEW(prefs);
  prefs.Init;
END VO:Knob.