(**
   A combobox gadget.
**)

MODULE VO:Combo;

(*
    Implements a combobox gadget.
    Copyright (C) 2000  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D  := VO:Base:Display,
       E  := VO:Base:Event,
       F  := VO:Base:Frame,
       O  := VO:Base:Object,
       U  := VO:Base:Util,
       Z  := VO:Base:Size,

       TM := VO:Model:Table,
       VM := VO:Model:Value,

       G  := VO:Object,
       T  := VO:Table,
       TV := VO:TableView,
       TX := VO:Text,
       V  := VO:VecImage,
       W  := VO:Window;

CONST
  selectedMsg* = 0;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the cycle is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 image*      : LONGINT;
                 imageRight* : BOOLEAN;
               END;

  Combo*     = POINTER TO ComboDesc;

  Popup      = POINTER TO PopupDesc;
  PopupDesc  = RECORD (W.WindowDesc)
                 tableModel : TM.TableModel;
                 table      : T.Table;
                 combo      : Combo;
               END;

  ComboDesc* = RECORD (G.GadgetDesc)
                 image       : V.VecImage;

                 value-      : G.Object;
                 tableModel- : TM.TableModel;
                 model       : VM.ValueModel;
                 popup       : Popup;
               END;

  TextCombo*     = POINTER TO TextComboDesc;
  TextComboDesc* = RECORD (ComboDesc)
                   END;

  (* --- *)

  SelectedMsg*      = POINTER TO SelectedMsgDesc;
  SelectedMsgDesc*  = RECORD (O.MessageDesc)
                      END;


VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frame:=F.ridge;
    p.image:=V.w95Combo;
    p.image:=V.combo;
    p.imageRight:=TRUE;
  END Init;

  PROCEDURE (c : Combo) Init*;

  BEGIN
    c.Init^;

    c.SetBackground(D.tableBackgroundColor);

    c.SetFlags({G.canFocus});
    c.RemoveFlags({G.stdFocus});

    c.SetPrefs(prefs);

    NEW(c.image);
    c.image.Init;
    c.image.SetParent(c);
    c.image.SetFlags({G.horizontalFlex,G.verticalFlex});

    c.value:=NIL;
    c.tableModel:=NIL;
    c.model:=NIL;
    c.popup:=NIL;
  END Init;

  PROCEDURE (c : Combo) SetTableModel*(model : TM.TableModel);

  BEGIN
    c.tableModel:=model;
    c.tableModel.SetSelectionType(TM.singleLineSelect);
  END SetTableModel;

  PROCEDURE (c : Combo) SetValueObject*(value : G.Object);

  BEGIN
    c.value:=value;
    c.value.SetParent(c);
  END SetValueObject;

  (* copied from Group *)
  (**
    Returns the object that coveres the given point and that supports
    drag and drop of data.

    If drag is TRUE, when want to find a object that we can drag data from,
    else we want an object to drop data on.
  **)
(*
  PROCEDURE (g : Combo) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):G.Object;

  VAR
    object,
    return  : G.Object;

  BEGIN
    object:=g.list;
    WHILE object#NIL DO
      return:=object.GetDnDObject(x,y,drag);
      IF return#NIL THEN
        RETURN return;
      END;
      object:=object.next;
    END;
    RETURN g.GetDnDObject^(x,y,drag);
  END GetDnDObject;
*)

  PROCEDURE (c : Combo) CopySelection*(row : LONGINT);

  VAR
    text : U.Text;
    x    : LONGINT;

  BEGIN
    IF c.model#NIL THEN
      IF (c.model.type=VM.longint)
      OR (c.model.type=VM.integer)
      OR (c.model.type=VM.shortint) THEN
        c.model.SetLongint(row-1);
      ELSIF c.model.type=VM.text THEN
        text:=NIL;
        x:=1;
        WHILE (text=NIL) & (x<=c.tableModel.GetColumns()) DO
          text:=c.tableModel.GetText(x,row);
          INC(x);
        END;

        IF text#NIL THEN
          c.model.SetText(text);
        END;
      END;
    END;

    IF c.tableModel#NIL THEN
      c.tableModel.SelectRow(row);
    END;
  END CopySelection;

  (**
    Assign the model for the cycle object. The cycle object will show the nth entry
    given by the value of the integer model.
  **)

  PROCEDURE (c : Combo) SetModel*(model : O.Model);

  BEGIN
    IF c.model#NIL THEN
      c.UnattachModel(c.model);
    END;
    IF (model#NIL) & (model IS VM.ValueModel) THEN
      c.model:=model(VM.ValueModel);
      c.AttachModel(c.model);
      IF ((c.model.type=VM.longint)
      OR (c.model.type=VM.integer)
      OR (c.model.type=VM.shortint)) & ~c.model.IsNull() THEN
        IF c.tableModel#NIL THEN
          c.tableModel.SelectRow(c.model.GetLongint());
        END;

        c.CopySelection(c.model.GetLongint());
      END;
    ELSE
      c.model:=NIL;
    END;
  END SetModel;

  (**
    This function is used to check if an argument to SetModel
    was successfully accepted.
   **)

  PROCEDURE (c : Combo) ModelAccepted * (m : O.Model):BOOLEAN;

  BEGIN
    RETURN m=c.model
  END ModelAccepted;

  PROCEDURE (c : Combo) CalcSize*;

  VAR
    w,mw,h,mh : LONGINT;

  BEGIN
    c.image.Set(c.prefs(Prefs).image);
    c.image.CalcSize;


    c.height:=0; (* TODO!*)
    c.minHeight:=c.height;

    w :=0; h :=0;
    mw:=0; mh:=0;

    IF c.value.StdFocus() THEN
      c.SetFlags({G.stdFocus});
    END;

    c.value.SetFlags({G.horizontalFlex,G.verticalFlex});

    c.value.CalcSize;
    w:=c.value.width;
    mw:=c.value.minWidth;
    h:=c.value.height;
    mh:=c.value.minHeight;

    IF ~c.StdFocus() & c.MayFocus() THEN
      (* tell the object to reserve space for focus displaying *)
      c.value.SetFlags({G.mayFocus});
    END;

    h:=U.MaxLong(h+D.display.spaceHeight,c.image.height);
    mh:=U.MaxLong(mh+D.display.spaceHeight,c.image.height);
    INC(c.height,h);
    INC(c.minHeight,mh);

    (* A little trick to make cycle-Button a little bit nicer *)
    c.image.Resize(c.height,c.height);
    c.width:=c.image.width+D.display.spaceWidth;
    c.minWidth:=c.width;

    INC(c.width,w+D.display.spaceWidth);
    INC(c.minWidth,mw+D.display.spaceWidth);

    c.CalcSize^;
  END CalcSize;

  PROCEDURE (c : Combo) InitTable*(table : T.Table);

  BEGIN

  END InitTable;

  PROCEDURE (p : Popup) PreInit*;

  VAR
    closeMsg : W.Msg2Exit;

  BEGIN
    p.table:=T.CreateTable();
    p.table.SetFlags({G.horizontalFlex(*,G.verticalFlex*)});
    p.table.SetMinHeight(Z.unit,30);
    p.table.SetModel(p.tableModel);
    p.table.SetShowScroller(FALSE,TRUE);

    p.combo.InitTable(p.table);

      (* if an entry gets selection with the mouse, close the popup *)
      NEW(closeMsg);
      closeMsg.destination:=p;
      p.table.table.AddHandler(closeMsg,TV.mouseSelectionMsg);

    p.AddFocusObject(p.table.table);

    p.SetTop(p.table);

    p.PreInit^;
  END PreInit;

  PROCEDURE (p : Popup) Maped*;

  BEGIN
    p.Maped^;

    IF p.tableModel.sy>0 THEN
      p.table.table.MakeVisible(1,p.tableModel.sy);
    END;
  END Maped;

  PROCEDURE (c : Combo) OpenPopup;

  BEGIN
    IF c.popup=NIL THEN
      NEW(c.popup);
      c.popup.Init;
      c.popup.SetStyle(W.popup);
      c.popup.SetParent(c.GetWindow());
      c.popup.SetReference(c);
      c.popup.tableModel:=c.tableModel;
      c.popup.combo:=c;
    END;

    c.popup.Open;

    c.popup.EventLoop;

    c.popup.Close;

    c.CopySelection(c.tableModel.sy);
  END OpenPopup;

  PROCEDURE (c : Combo) HandleMouseEvent*(event : E.MouseEvent;
                                          VAR grab : G.Object):BOOLEAN;

  BEGIN
    IF ~c.visible OR c.disabled(* OR (c.model=NIL) OR c.model.IsNull()*) THEN
      RETURN FALSE;
    END;

    WITH event : E.ButtonEvent DO
      IF (event.type=E.mouseDown) & c.PointIsIn(event.x,event.y)
      & (event.button=E.button1) THEN
        c.OpenPopup;
        RETURN TRUE;
      END;
    ELSE
    END;

    RETURN FALSE;
  END HandleMouseEvent;


  PROCEDURE (c : Combo) HandleKeyEvent*(event : E.KeyEvent):BOOLEAN;

  VAR
    keysym : LONGINT;

  BEGIN
    IF event.type=E.keyDown THEN
      keysym:=event.GetKey();
      IF (keysym=E.down) THEN
        IF (c.tableModel.sy<c.tableModel.GetRows()) THEN
          c.tableModel.SelectRow(c.tableModel.sy+1);
          c.CopySelection(c.tableModel.sy);
        END;
        RETURN TRUE;
      ELSIF keysym=E.up THEN
        IF (c.tableModel.sy>1) THEN
          c.tableModel.SelectRow(c.tableModel.sy-1);
          c.CopySelection(c.tableModel.sy);
        END;
        RETURN TRUE;
      END;
    END;
    RETURN FALSE;
  END HandleKeyEvent;

  PROCEDURE (c : Combo) Layout*;

  VAR
    width : LONGINT;

  BEGIN
    width:=c.width-c.image.width;
    c.value.Resize(width,c.height);

    IF c.HasFocus() THEN
      (* tell the object to display focus *)
      c.value.SetFlags({G.showFocus});
    ELSE
      (* tell the object to not display focus *)
      c.value.RemoveFlags({G.showFocus});
    END;

    IF c.prefs(Prefs).imageRight THEN
      c.value.Move(c.x+(width-c.value.width) DIV 2,
                   c.y+(c.height-c.value.height) DIV 2);
    ELSE
      c.value.Move(c.x+c.image.width+(width-c.value.width) DIV 2,
                   c.y+(c.height-c.value.height) DIV 2);
    END;

    IF c.prefs(Prefs).imageRight THEN
      c.image.Move(c.x+c.width-c.image.width,c.y);
    ELSE
      c.image.Move(c.x,c.y);
    END;
    c.image.Resize(-1,c.height);

    c.Layout^;
  END Layout;

  PROCEDURE (c : Combo) Draw*(x,y,w,h : LONGINT);

  BEGIN
    c.Draw^(x,y,w,h);

    IF ~c.Intersect(x,y,w,h) THEN
      RETURN;
    END;

    c.value.Draw(x,y,w,h);
    c.image.Draw(x,y,w,h);

    IF c.disabled THEN
      c.DrawDisabled;
    END;
  END Draw;

  PROCEDURE (c : Combo) DrawFocus*;

  BEGIN
    (* If our image can draw a keyboard focus, delegate it *)
    IF  ~c.value.StdFocus() THEN
      c.value.DrawFocus;
    ELSE
      (* Delegate drawing to the baseclass *)
      c.DrawFocus^;
    END;
  END DrawFocus;

  (**
    Hide the keyboard focus.
  **)

  PROCEDURE (c : Combo) HideFocus*;

  BEGIN
    (* If our image can draw a keyboard focus, delegate it *)
    IF ~c.value.StdFocus() THEN
      c.value.HideFocus;
    ELSE
      (* Delegate drawing to the baseclass *)
     c.HideFocus^;
    END;
  END HideFocus;

  PROCEDURE (c : Combo) Hide*;

  BEGIN
    IF c.visible THEN
      c.image.Hide;
      c.value.Hide;
      c.Hide^;
    END;
  END Hide;

  PROCEDURE (c : Combo) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF (model=c.model) THEN
      IF ((c.model.type=VM.longint)
      OR (c.model.type=VM.integer)
      OR (c.model.type=VM.shortint)) & ~c.model.IsNull() THEN
        c.CopySelection(c.model.GetLongint()+1);
      END;
    END;
  END Resync;

  PROCEDURE (c : TextCombo) Init*;

  VAR
    text : TX.Text;

  BEGIN
    c.Init^;

    text:=TX.CreateText();
    c.SetValueObject(text);

  END Init;

  PROCEDURE (c : TextCombo) CopySelection*(row : LONGINT);

  VAR
    text : U.Text;

  BEGIN
    c.CopySelection^(row);

    IF row>0 THEN
      text:=c.tableModel.GetText(1,row);
      IF text#NIL THEN
        c.value(TX.Text).SetText(text^);
      ELSE
        c.value(TX.Text).SetText("");
      END;
    ELSE
      c.value(TX.Text).SetText("");
    END;
  END CopySelection;


  PROCEDURE CreateCombo*():Combo;

  VAR
    combo : Combo;

  BEGIN
    NEW(combo);
    combo.Init;

    RETURN combo;
  END CreateCombo;

  PROCEDURE CreateTextCombo*():TextCombo;

  VAR
    combo : TextCombo;

  BEGIN
    NEW(combo);
    combo.Init;

    RETURN combo;
  END CreateTextCombo;


BEGIN
  NEW(prefs);
  prefs.Init;

END VO:Combo.