#include <string.h>
#include "libnjb.h"
#include "njb_error.h"
#include "defs.h"
#include "base.h"
#include "byteorder.h"
#include "eax.h"

extern int __sub_depth;

// Routine to unpack EAX block from NJB1 series device
eax_t *eax_unpack(void *data, size_t nbytes)
{
	__dsub= "eax_unpack";
	eax_t *eax; 
	unsigned char *dp= (unsigned char *) data;
	int i= 0;

	__enter;

	eax= malloc(sizeof(eax_t));
	if ( eax == NULL ) {
		NJB_ERROR(EO_NOMEM);
		__leave;
		return NULL;
	}

	memset(eax, 0, sizeof(eax_t));

	eax->volume= (u_int8_t) *dp++;
	eax->volumemin = 0;
	eax->volumemax = 99;
	eax->muted= (u_int8_t) *dp++;
	eax->eq_status= (u_int8_t) *dp++;

	eax->bass= (int8_t) *dp++;
	eax->bassmin= (int8_t) *dp++;
	eax->bassmax= (int8_t) *dp++;

	eax->midrange= (int8_t) *dp++;
	eax->midrangemin= (int8_t) *dp++;
	eax->midrangemax= (int8_t) *dp++;

	eax->treble= (int8_t) *dp++;
	eax->treblemin= (int8_t) *dp++;
	eax->treblemax= (int8_t) *dp++;

	eax->nfreq= (u_int8_t) *dp++;
	eax->freqset= (u_int8_t) *dp++;
	eax->frequencies= malloc(eax->nfreq*2);
	if ( eax->frequencies == NULL ) {
		NJB_ERROR(EO_NOMEM);
		eax_destroy(eax);
		__leave;
		return NULL;
	}

	for (i= 0; i< eax->nfreq; i++) {
		eax->frequencies[i] = njb1_bytes_to_16bit(&dp[0]);
		dp+= 2;
	}

	eax->neffects= (u_int8_t) *dp++;
	eax->acteffect= (u_int8_t) *dp++;
	eax->effects= (char **) malloc(eax->neffects * sizeof(char *));
	if ( eax->effects == NULL ) {
		NJB_ERROR(EO_NOMEM);
		eax_destroy(eax);
		__leave;
		return NULL;
	}
	memset(eax->effects, 0, eax->neffects * sizeof(char *));

	eax->effamts= (u_int8_t *) malloc(eax->neffects);
	if ( eax->effamts == NULL ) {
		NJB_ERROR(EO_NOMEM);
		eax_destroy(eax);
		__leave;
		return NULL;
	}
	memset(eax->effamts, 0, eax->neffects);

	for (i= 0; i<eax->neffects; i++) {
		u_int8_t efflen= (u_int8_t) *dp++;

		eax->effects[i]= (char *) malloc(efflen+1);
		if ( eax->effects[i] == NULL ) {
			NJB_ERROR(EO_NOMEM);
			eax_destroy(eax);
			__leave;
			return NULL;
		}

		memcpy(eax->effects[i], dp, efflen);
		eax->effects[i][efflen]= '\0';
		dp+= efflen;

		eax->effamts[i]= (u_int8_t) *dp++;
	}

	eax->nphone= (u_int8_t) *dp++;
	eax->actphone= (u_int8_t) *dp++;
	eax->phonemodes= (char **) malloc (eax->nphone * sizeof(char *));
	if ( eax->phonemodes == NULL ) {
		NJB_ERROR(EO_NOMEM);
		eax_destroy(eax);
		__leave;
		return NULL;
	}

	memset(eax->phonemodes, 0, eax->nphone * sizeof(char *));
	for (i= 0; i< eax->nphone; i++) {
		u_int8_t phonelen= (u_int8_t) *dp++;
		eax->phonemodes[i]= (char *) malloc(phonelen+1);
		if ( eax->phonemodes[i] == NULL ) {
			NJB_ERROR(EO_NOMEM);
			eax_destroy(eax);
			__leave;
			return NULL;
		}

		memcpy(eax->phonemodes[i], dp, phonelen);
		eax->phonemodes[i][phonelen]= '\0';
		dp+= phonelen;
	}

	eax->nrear= (u_int8_t) *dp++;
	eax->actrear= (u_int8_t) *dp++;
	eax->rears= (char **) malloc(eax->nrear * sizeof(char *));
	if ( eax->rears == NULL ) {
		NJB_ERROR(EO_NOMEM);
		eax_destroy(eax);
		__leave;
		return NULL;
	}

	memset(eax->rears, 0, eax->nrear * sizeof(char *));
	for (i= 0; i< eax->nrear; i++) {
		u_int8_t rearlen= (u_int8_t) *dp++;
		eax->rears[i]= (char *) malloc(rearlen+1);
		if ( eax->rears[i] == NULL ) {
			NJB_ERROR(EO_NOMEM);
			eax_destroy(eax);
			__leave;
			return NULL;
		}

		memcpy(eax->rears[i], dp, rearlen);
		eax->rears[i][rearlen]= '\0';
		dp+= rearlen;
	}

	__leave;
	return eax;
}

void eax_destroy(eax_t * eax)
{
	/* Free the memory used by the EAX structure 
	 * extra care taken for NULL checking since
	 * it is used in case an error occurs in
	 * the parsing in protocol.c */
	u_int8_t i;

	if (eax != NULL) {
		if (eax->rears != NULL) {
			for (i = 0; i < eax->nrear; i++) {
				if (eax->rears[i] != NULL)
					free(eax->rears[i]);
			}
			free(eax->rears);
		}
		if (eax->phonemodes != NULL) {
			for (i = 0; i < eax->nphone; i++) {
				if (eax->phonemodes[i] != NULL)
					free(eax->phonemodes[i]);
			}
			free(eax->phonemodes);
		}
		if (eax->effects != NULL) {
			for (i = 0; i < eax->neffects; i++) {
				if (eax->effects[i] != NULL)
					free(eax->effects[i]);
			}
			free(eax->effects);
		}
		if (eax->effamts != NULL)
			free(eax->effamts);
		if (eax->frequencies != NULL)
			free(eax->frequencies);
		free(eax);
	}
}

// Routine to refresh EAX for an NJB1 series device
void eax_refresh(eax_t *eax, unsigned char *data, size_t numbytes)
{
	/*
	 * The NJB1 has a special "refresh" command that is perhaps better
	 * left unused, but here it is anyway.
	 */
	char *dp= (char *) data;
	int i;

	if (numbytes != 15)
		return;

	dp += 5;
	eax->volume= (u_int8_t) *dp++;
	eax->muted= (u_int8_t) *dp++;
	eax->eq_status= (u_int8_t) *dp++;
	eax->bass= (int8_t) *dp++;
	eax->midrange= (int8_t) *dp++;
	eax->treble= (int8_t) *dp++;
	eax->freqset= (u_int8_t) *dp++;
	eax->acteffect= (u_int8_t) *dp++;
	for (i= 0; i<eax->neffects; i++) {
		eax->effamts[i]= (u_int8_t) *dp;
	}
	dp++;
	eax->actphone= (u_int8_t) *dp++;
	eax->actrear= (u_int8_t) *dp;
}

static void eax_add_to_state(njb_state_t *state, njb_eax_t *eax)
{
  /* Add the new EAX to our state */
  if (state->first_eax == NULL) {
    state->first_eax = eax;
    state->next_eax = NULL;
  } else if (state->next_eax == NULL) {
    state->first_eax->next = eax;
    state->next_eax = eax;
  } else {
    state->next_eax->next = eax;
    state->next_eax = eax;
  }
}

// Convert a 16-bit integer to a string
static char *sixteen_to_string_hz(u_int16_t value)
{
  char buffer[16];
  snprintf(buffer, 16, "%d Hz", value);
  buffer[15] = '\0';
  return strdup(buffer);
}

// The new EAX API for the NJB1 device
// The EAX effects are added to the linked list in the state
int eax_unpack_new_api(void *data, size_t nbytes, njb_state_t *state)
{
  __dsub= "eax_unpack_new_api";
  njb_eax_t *eax, *volumeeax, *mutingeax; 
  unsigned char *dp = (unsigned char *) data;
  u_int16_t frequencies;
  u_int16_t effects;
  u_int16_t effect_amount;
  u_int16_t phonemodes;
  u_int16_t rearmodes;
  int8_t tmp8;
  int i = 0;

  __enter;
  
  // Volume EAX
  volumeeax = new_eax_type();
  volumeeax->number = NJB_SOUND_SET_VOLUME;
  volumeeax->name = strdup("Volume");
  volumeeax->group = 0x00;
  volumeeax->scaleable = 0x01;
  volumeeax->current_scalevalue = (u_int16_t) *dp++;
  volumeeax->min_scalevalue = 0;
  volumeeax->max_scalevalue = 100;
  // Do not add to state until after Muting's been added

  // Muting EAX
  mutingeax = new_eax_type();
  mutingeax->number = NJB_SOUND_SET_MUTING;
  mutingeax->name = strdup("Muted");
  mutingeax->group = 0x00;
  mutingeax->selectable = 0x01;
  mutingeax->min_selectionvalue = 0x0000;
  mutingeax->max_selectionvalue = 0x0001;
  mutingeax->current_selectionvalue = (u_int16_t) *dp++;
  mutingeax->selection_names = (char **) malloc(2 * sizeof(char *));
  mutingeax->selection_names[0] = strdup("Off");
  mutingeax->selection_names[1] = strdup("On");

  eax_add_to_state(state, mutingeax);
  eax_add_to_state(state, volumeeax);

  // Equalizer activation EAX
  eax = new_eax_type();
  eax->number = NJB_SOUND_SET_EQSTATUS;
  eax->name = strdup("Equalizer active");
  eax->group = 0x01;
  eax->selectable = 0x01;
  eax->min_selectionvalue = 0x0000;
  eax->max_selectionvalue = 0x0001;
  eax->current_selectionvalue = (u_int16_t) *dp++;
  eax->selection_names = (char **) malloc(2 * sizeof(char *));
  eax->selection_names[0] = strdup("Off");
  eax->selection_names[1] = strdup("On");
  eax_add_to_state(state, eax);
  
  // Equalizer bass EAX
  eax = new_eax_type();
  eax->number = NJB_SOUND_SET_BASS;
  eax->name = strdup("Bass");
  eax->group = 0x01;
  eax->scaleable = 0x01;
  tmp8 = (int8_t) *dp++;
  eax->current_scalevalue = (int16_t) tmp8;
  tmp8 = (int8_t) *dp++;
  eax->min_scalevalue = (int16_t) tmp8;
  tmp8 = (int8_t) *dp++;
  eax->max_scalevalue = (int16_t) tmp8;
  eax_add_to_state(state, eax);

  // Equalizer midrange EAX
  eax = new_eax_type();
  eax->number = NJB_SOUND_SET_MIDRANGE;
  eax->name = strdup("Midrange");
  eax->group = 0x01;
  eax->scaleable = 0x01;
  tmp8 = (int8_t) *dp++;
  eax->current_scalevalue = (int16_t) tmp8;
  tmp8 = (int8_t) *dp++;
  eax->min_scalevalue = (int16_t) tmp8;
  tmp8 = (int8_t) *dp++;
  eax->max_scalevalue = (int16_t) tmp8;
  eax_add_to_state(state, eax);

  // Equalizer treble EAX
  eax = new_eax_type();
  eax->number = NJB_SOUND_SET_TREBLE;
  eax->name = strdup("Treble");
  eax->group = 0x01;
  eax->scaleable = 0x01;
  tmp8 = (int8_t) *dp++;
  eax->current_scalevalue = (int16_t) tmp8;
  tmp8 = (int8_t) *dp++;
  eax->min_scalevalue = (int16_t) tmp8;
  tmp8 = (int8_t) *dp++;
  eax->max_scalevalue = (int16_t) tmp8;
  eax_add_to_state(state, eax);

  // Equalizer center frequency
  eax = new_eax_type();
  eax->number = NJB_SOUND_SET_MIDFREQ;
  eax->name = strdup("Midrange center frequency");
  eax->group = 0x01;
  eax->selectable = 0x01;
  eax->min_selectionvalue = 0x0000;
  frequencies = (u_int16_t) *dp++;
  eax->max_selectionvalue = frequencies - 1;
  eax->current_selectionvalue = (u_int16_t) *dp++;
  // We have to conjure these strings ourselves
  eax->selection_names = (char **) malloc(frequencies * sizeof(char *));
  for (i = 0; i < frequencies; i ++) {
    eax->selection_names[i] = sixteen_to_string_hz(njb1_bytes_to_16bit(&dp[0]));
    dp += 2;
  }
  eax_add_to_state(state, eax);

  // EAX Effects
  eax = new_eax_type();
  eax->number = NJB_SOUND_SET_EAX;
  eax->name = strdup("EAX effect");
  eax->group = 0x02;
  eax->selectable = 0x01;
  eax->min_selectionvalue = 0x0000;
  effects = (u_int16_t) *dp++;
  eax->max_selectionvalue = effects - 1;
  eax->current_selectionvalue = (u_int16_t) *dp++;
  // We have to conjure these strings ourselves
  eax->selection_names = (char **) malloc(effects * sizeof(char *));
  effect_amount = 0;
  for (i = 0; i < effects; i ++) {
    u_int8_t efflen= (u_int8_t) *dp++;
    
    eax->selection_names[i]= (char *) malloc(efflen+1);
    memcpy(eax->selection_names[i], dp, efflen);
    eax->selection_names[i][efflen] = '\0';
    dp += efflen;
    // This is actually the same for all effects, so we
    // only need one scale value for it (below).
    effect_amount = (u_int16_t) *dp++;
  }
  eax_add_to_state(state, eax);

  // Also add a scale value for the EAX level
  eax = new_eax_type();
  eax->number = NJB_SOUND_SET_EAXAMT;
  eax->name = strdup("EAX effect level");
  eax->group = 0x02;
  eax->scaleable = 0x01;
  eax->min_scalevalue = 0x0000;
  eax->max_scalevalue = 100;
  eax->current_scalevalue = effect_amount;
  eax_add_to_state(state, eax);

  // Headphone modes
  eax = new_eax_type();
  eax->number = NJB_SOUND_SET_HEADPHONE;
  eax->name = strdup("Headphone mode");
  eax->group = 0x03;
  eax->selectable = 0x01;
  eax->min_selectionvalue = 0x0000;
  phonemodes = (u_int16_t) *dp++;
  eax->max_selectionvalue = phonemodes - 1;
  eax->current_selectionvalue = (u_int16_t) *dp++;
  // We have to conjure these strings ourselves
  eax->selection_names = (char **) malloc(phonemodes * sizeof(char *));
  for (i = 0; i < phonemodes; i ++) {
    u_int8_t len= (u_int8_t) *dp++;
    
    eax->selection_names[i]= (char *) malloc(len+1);
    memcpy(eax->selection_names[i], dp, len);
    eax->selection_names[i][len] = '\0';
    dp += len;
  }
  eax_add_to_state(state, eax);

  // Rear speaker modes
  eax = new_eax_type();
  eax->number = NJB_SOUND_SET_REAR;
  eax->name = strdup("Rear speaker mode");
  eax->group = 0x03;
  eax->selectable = 0x01;
  eax->min_selectionvalue = 0x0000;
  rearmodes = (u_int16_t) *dp++;
  eax->max_selectionvalue = rearmodes - 1;
  eax->current_selectionvalue = (u_int16_t) *dp++;
  // We have to conjure these strings ourselves
  eax->selection_names = (char **) malloc(rearmodes * sizeof(char *));
  for (i = 0; i < rearmodes; i ++) {
    u_int8_t len= (u_int8_t) *dp++;
    
    eax->selection_names[i]= (char *) malloc(len+1);
    memcpy(eax->selection_names[i], dp, len);
    eax->selection_names[i][len] = '\0';
    dp += len;
  }
  eax_add_to_state(state, eax);

  // At last rewind the EAX list.
  state->next_eax = state->first_eax;

  return 0;
}

// Dummy EAX unpacking routine for an NJB3 series device,
// since this API is deprecated, only zeroes are returned.
eax_t *eax_unpack3(void *data)
{
  __dsub = "eax_unpack3";
  eax_t *eax; 
  // unsigned char *dp = (unsigned char *) data;
  // int i = 0;
  
  __enter;
  
  eax= malloc(sizeof(eax_t));
  if ( eax == NULL ) {
    NJB_ERROR(EO_NOMEM);
    __leave;
    return NULL;
  }
  
  memset(eax, 0, sizeof(eax_t));
  
  eax->volume = 0;
  eax->volumemin = 0;
  eax->volumemax = 99;

  eax->muted = 0;
  eax->eq_status = 0;
  
  eax->bass = 0;
  eax->bassmin = 0;
  eax->bassmax = 0;
  
  eax->midrange = 0;
  eax->midrangemin = 0;
  eax->midrangemax = 0;
  
  eax->treble = 0;
  eax->treblemin = 0;
  eax->treblemax = 0;
  
  eax->nfreq = 0;
  eax->freqset = 0;
  eax->frequencies = malloc(eax->nfreq*2);
  
  eax->neffects = 0;
  eax->acteffect = 0;
  eax->effects = (char **) malloc(eax->neffects * sizeof(char *));
  memset(eax->effects, 0, eax->neffects * sizeof(char *));
  eax->effamts = (u_int8_t *) malloc(eax->neffects);
  memset(eax->effamts, 0, eax->neffects);
  
  eax->nphone = 0;
  eax->actphone = 0;
  eax->phonemodes = (char **) malloc (eax->nphone * sizeof(char *));
  memset(eax->phonemodes, 0, eax->nphone * sizeof(char *));
  
  eax->nrear = 0;
  eax->actrear = 0;
  eax->rears = (char **) malloc(eax->nrear * sizeof(char *));
  memset(eax->rears, 0, eax->nrear * sizeof(char *));

  __leave;
  return eax;
}

njb_eax_t *new_eax_type(void)
{
  __dsub = "new_eax_type";
  njb_eax_t *eax; 
  
  __enter;
  
  eax = malloc(sizeof(njb_eax_t));
  if ( eax == NULL ) {
    NJB_ERROR(EO_NOMEM);
    __leave;
    return NULL;
  }
  
  memset(eax, 0, sizeof(njb_eax_t));
  eax->number = 0;
  eax->name = NULL;
  eax->group = 0;
  eax->exclusive = 0;
  eax->selectable = 0;
  eax->current_selectionvalue = 0;
  eax->min_selectionvalue = 0;
  eax->max_selectionvalue = 0;
  eax->selection_names = NULL;
  eax->scaleable = 0;
  eax->current_scalevalue = 0;
  eax->min_scalevalue = 0;
  eax->max_scalevalue = 0;
  eax->next = NULL;
  __leave;
  return eax;  
}

void destroy_eax_type(njb_eax_t *eax)
{
  if (eax == NULL)
    return;
  /* Free the EAX effect name */
  if (eax->name != NULL) {
    free(eax->name);
  }
  /* Free the EAX selection names */
  if (eax->selectable > 0) {
    u_int8_t i;
    for (i = 0; i < (eax->max_selectionvalue - eax->min_selectionvalue); i++) {
      if (eax->selection_names[i] != NULL) {
	free(eax->selection_names[i]);
      }
    }
    if (eax->selection_names != NULL) {
      free(eax->selection_names);
    }
  }
  free(eax);
  return;
}
