/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This file is part of libgpiv.
   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  



---------------------------------------------------------------
FILENAME:                io.c
LIBRARY:                 libgpiv:
EXTERNAL FUNCTIONS:      gpiv_io_make_fname,
			 gpiv_fread_image, 
			 gpiv_fwrite_image, 
			 gpiv_read_image, 
                         gpiv_write_image, 
			 gpiv_fread_ascii_image, 
			 gpiv_fwrite_ascii_image, 
                         gpiv_read_ascii_image, 
			 gpiv_write_ascii_image
			 gpiv_count_pivdata,
                         gpiv_fcount_pivdata, 
                         gpiv_find_pivdata_origin,
			 gpiv_find_pivdata_scale,
                         gpiv_read_pivdata, 
			 gpiv_fread_pivdata, 
                         gpiv_write_pivdata, 
			 gpiv_fwrite_pivdata, 
                         gpiv_write_scdata, 
			 gpiv_fwrite_scdata, 
			 gpiv_fwrite_griddata,
			 gpiv_print_histo,
			 gpiv_fprint_histo,

                         gpiv_fread_pgm_image
                         gpiv_fwrite_pgm_image

                         gpiv_fcreate_hdf5
                         gpiv_fread_hdf5_image,
                         gpiv_fwrite_hdf5_image,

                         gpiv_fread_hdf5_parameters,
                         gpiv_fwrite_hdf5_parameters,
                         gpiv_fcount_hdf5_data
                         gpiv_fread_hdf5_piv_position,
                         gpiv_fwrite_hdf5_piv_position,
			 gpiv_fread_hdf5_pivdata, 
			 gpiv_fwrite_hdf5_pivdata, 
                         gpiv_fread_hdf5_sc_position,
                         gpiv_fwrite_hdf5_sc_position,
                         gpiv_fread_hdf5_scdata, 
			 gpiv_fwrite_hdf5_scdata, 
                         gpiv_fread_hdf5_histo, 
			 gpiv_fwrite_hdf5_histo

                         gpiv_fread_davis_image,

LOCAL FUNCTIONS:

LAST MODIFICATION DATE:  $Id: io.c,v 1.13 2006/01/31 13:30:13 gerber Exp $
 --------------------------------------------------------------- */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <assert.h>
#include <hdf5.h>

#include <gpiv.h>


void 
gpiv_io_make_fname(char *fname_base, 
		   char *EXT, 
		   char *fname_out
		   )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *  Constructs (output) filename from base name and exetsion 
 *
 * INPUTS:
 *      fname_base:     file base name
 *      EXT:            file extension name
 * OUTPUTS:
 *      fname_out:      completed filename
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
  snprintf(fname_out, GPIV_MAX_CHARS, "%s%s", fname_base, EXT);
}



char *
gpiv_fread_image(char *fname,
		 guint16 **img1,
		 guint16 **img2, 
		 GpivImagePar image_par
		 ) 
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Read raw image data from file
 *
 * INPUTS:
 *      fname:          pointer to complete filename to be written
 *      image_par:      image parameters containing number of rows an columns
 *
 * OUTPUTS:
 *      img1:           first binary image
 *      img2:           second binary image
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    FILE *fp;
    char *err_msg = NULL;
    int nrows =  image_par.nrows;
    int ncolumns =  image_par.ncolumns;
    int depth = image_par.depth;
    int x_corr = image_par.x_corr;
    guint8 **img1_dum, **img2_dum;
    gint i, j;

/* gpiv_warning("gpiv_fread_image:: 0 imgsize = %d size = %d", sizeof(img1), sizeof(guint16) * (nrows-1) * ncolumns); */
/* fflush(stdout); */
    assert(img1 != NULL);

    if (image_par.x_corr) 
        assert(img2[0] != NULL);

    if ((fp = fopen(fname, "rb")) == NULL) {
	err_msg = "GPIV_FREAD_IMAGE: Failure opening for input";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }


    if (depth <= 8) {
        img1_dum = gpiv_matrix_guint8(nrows, ncolumns);
        fread(img1_dum[0], ncolumns * nrows * (sizeof(guint8)), 1, fp);
        if (x_corr) {
            img2_dum = gpiv_matrix_guint8(nrows, ncolumns);
            fread(img2_dum[0], ncolumns * nrows * (sizeof(guint8)), 1, fp);
        } else {
            img2_dum = img1_dum;
        }

        for (i = 0; i < nrows; i++) {
            for (j = 0; j < ncolumns; j++) {
                img1[i][j]  = img1_dum[i][j];
                img2[i][j] = img2_dum[i][j];
            }
        }
        
        gpiv_free_matrix_guint8(img1_dum);
        if (x_corr) {
            gpiv_free_matrix_guint8(img2_dum);
        }
        

    } else {
        fread(img1[0], ncolumns * nrows * (sizeof(guint16)), 1, fp);
        if (x_corr) {
            fread(img2[0], ncolumns * nrows * (sizeof(guint16)), 1, fp);
        } else {
            img2 = img1;
        }
    }

    fclose(fp);
    return err_msg;
}



char *
gpiv_fwrite_image(char *fname,
		  guint16 **img1, 
		  guint16 **img2, 
		  GpivImagePar image_par
		  )
/*  ---------------------------------------------------------------------------
 * DESCRIPTION:
 *     writes binary image to file
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      fname:          pointer to complete filename to be written
 *      img1:           first 2-dimensional matrix containing nrows x ncoloms pixels
 *      img2:           second 2-dimensional matrix containing nrows x ncoloms pixels
 *      image_par:      image parameters containing number of rows an columns
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    FILE *fp;
    char *err_msg = NULL;
    unsigned char **l_img1 = NULL, **l_img2 = NULL;
    int nrows =  image_par.nrows;
    int ncolumns =  image_par.ncolumns;
    int depth = image_par.depth;
    int x_corr = image_par.x_corr;
    gint i, j;
    assert(img1[0] != NULL);
    assert(img2[0] != NULL);


    if ((fp = fopen(fname, "wb")) == NULL) {
	err_msg = "GPIV_FWRITE_IMAGE: Failure opening for input";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    if (depth <= 8) {
        l_img1 = gpiv_ucmatrix(image_par.nrows, image_par.ncolumns);
        if (x_corr) {
            l_img2 = gpiv_ucmatrix(image_par.nrows, image_par.ncolumns);
        }
    }

    for (i = 0; i < image_par.nrows; i++) {
        for (j = 0; j < image_par.ncolumns; j++) {
            l_img1[i][j] = img1[i][j];
            l_img2[i][j] = img2[i][j];
/*             g_message("gpiv_fwrite_image:: l_img1[%d][%d] = %d", */
/*                       i, j, (int)l_img1[i][j]); */
/*             if (l_img1[i][j] >= 245.0 ) { */
/*                 l_img1[i][j] = 0; */
/*             } */
/*             if (l_img2[i][j] >= 245.0 ) { */
/*                 l_img2[i][j] = 0; */
/*             } */
        }
    }

    if (depth <= 8) {
        fwrite(l_img1[0], 1, ncolumns * nrows * (sizeof(char)), fp);
        if (x_corr) {
            fwrite(l_img2[0], 1, ncolumns * nrows * (sizeof(char)), fp);
        }
    } else {
        fwrite(img1[0], ncolumns * nrows * (sizeof(guint16)), 1, fp);
        if (x_corr) {
            fwrite(img2[0], ncolumns * nrows * (sizeof(guint16)), 1, fp);
        }
    }

    if (depth <= 8) {
        gpiv_free_ucmatrix(l_img1);
        if (x_corr) {
            gpiv_free_ucmatrix(l_img2);
        }
    }
    fclose(fp);
    return err_msg;
}



void
gpiv_read_image(guint16 **img1, 
		guint16 **img2, 
		GpivImagePar image_par
		)
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Reads binary image from stdin 
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      image_par:      image parameters containing number of rows an columns
 *
 * OUTPUTS:
 *      img1:           first 2-dimensional matrix containing nrows x ncoloms pixels
 *      img2:           second 2-dimensional matrix containing nrows x ncoloms pixels
 *---------------------------------------------------------------------------*/
{
    int nrows =  image_par.nrows;
    int ncolumns =  image_par.ncolumns;
    int depth = image_par.depth;
    int x_corr = image_par.x_corr;
    guint8 **img1_dum, **img2_dum;
    int i, j;

    assert(img1[0] != NULL);
    if (image_par.x_corr) 
        assert(img2[0] != NULL);

    if (depth <= 8) {
        img1_dum = gpiv_matrix_guint8(nrows, ncolumns);
        read(0, img1_dum[0], ncolumns * nrows * (sizeof(guint8)));
        if (x_corr) {
            img2_dum = gpiv_matrix_guint8(nrows, ncolumns);
            read(0, img2_dum[0], ncolumns * nrows * (sizeof(guint8)));
        } else {
            img2_dum = img1_dum;
        }

        for (i = 0; i < nrows; i++) {
            for (j = 0; j < ncolumns; j++) {
                img1[i][j]  = img1_dum[i][j];
                img2[i][j] = img2_dum[i][j];
            }
        }
        
        gpiv_free_matrix_guint8(img1_dum);
        if (x_corr) {
            gpiv_free_matrix_guint8(img2_dum);
        }
        


    } else {
        read(0, img1[0], ncolumns * nrows * (sizeof(guint16)));
        if (x_corr) {
            read(0, img2[0], ncolumns * nrows * (sizeof(guint16)));
        } else {
            img2 = img1;
        }
    }
}



void
gpiv_write_image(guint16 **img1, 
		 guint16 **img2, 
		 GpivImagePar image_par
		 )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Writess binary image to stdout 
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      img1:           first binary image
 *      img2:           second binary image
 *      image_par:      image parameters containing number of rows an columns
 *---------------------------------------------------------------------------*/
{
    int nrows =  image_par.nrows;
    int ncolumns =  image_par.ncolumns;
    int depth = image_par.depth;
    int x_corr = image_par.x_corr;

    assert(img1[0] != NULL);
    assert(img2[0] != NULL);

    if (depth <= 8) {
        write(1, img1[0], ncolumns * nrows * (sizeof(guint8)));
        if (x_corr) {
            write(1, img2[0], ncolumns * nrows * (sizeof(guint8)));
        }
    } else {
        write(1, img1[0], ncolumns * nrows * (sizeof(guint16)));
        if (x_corr) {
            write(1, img2[0], ncolumns * nrows * (sizeof(guint16)));
        }
    }
}



char *
gpiv_fread_ascii_image(char *fname,
		       guint16 **img, 
		       GpivImagePar image_par
		       )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Reads ascii image from file 
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      fname:          pointer to complete filename to be read
 *      image_par:      image parameters containing number of rows an columns
 *
 * OUTPUTS:
 *      img:            2-dimensional matrix containing nrows x ncoloms pixels
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    int row, column, ira, irb;
    float fdum;
    FILE *fp;
    char cdum, *err_msg = NULL;
    int nrows =  image_par.nrows;
    int ncolumns =  image_par.ncolumns;

    assert(img[0] != NULL);

    if ((fp = fopen(fname, "rb")) == NULL) {
	err_msg = "GPIV_FREAD_ASCII_IMAGE: Failure opening for input";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    for (row = 0; row < nrows; row++) {
	fscanf(fp, " %c", &cdum);
	for (column = 0; column < ncolumns; column++) {
	    fscanf(fp, GPIV_ASCII_IMG_FMT, &ira, &irb, &fdum);
	    img[row][column] = fdum;
	}
    }

    fclose(fp);
    return err_msg;
}



char *
gpiv_fwrite_ascii_image(char *fname,
			guint16 **img, 
			GpivImagePar image_par
			)
/*---------------------------------------------------------------------------- 
 * DESCRIPTION:
 *     Writess ascii image from file 
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      fname:          pointer to complete filename to be written
 *      img:            2-dimensional matrix containing nrows x ncoloms pixels
 *      image_par:      image parameters containing number of rows an columns
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int row, column, ira = 0, irb = 0;
    double fdum = 0.0;
    FILE *fp;
    char cdum = '\0';
    int nrows =  image_par.nrows;
    int ncolumns =  image_par.ncolumns;

    assert(img[0] != NULL);

    if ((fp = fopen(fname, "rb")) == NULL) {
	err_msg = "GPIV_FWRITE_ASCII_IMAGE: Failure opening %s for input";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    for (row = 0; row < nrows; row++) {
	fprintf(fp, "%c", cdum);
	for (column = 0; column < ncolumns; column++) {
	    fprintf(fp, GPIV_ASCII_IMG_FMT, ira, irb, fdum);
	    img[row][column] = fdum;
	}
    }

    fclose(fp);
    return err_msg;
}



void
gpiv_read_ascii_image(guint16 **img, 
		      GpivImagePar image_par
		      )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Reads ascii image from stdin 
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      nrows:         number of rows in image
 *      ncolumns:      number of columns in image
 *
 * OUTPUTS:
 *      img:            2-dimensional matrix containing nrows x ncoloms pixels
 *---------------------------------------------------------------------------*/
{
    int row = 0, column = 0, ira = 0, irb = 0;
    char cdum = '\0';
    float fdum = 0.0;
    int nrows =  image_par.nrows;
    int ncolumns =  image_par.ncolumns;

    assert(img[0] != NULL);

    for (row = 0; row < nrows; row++) {
	scanf("%c", &cdum);
	for (column = 0; column < ncolumns; column++) {
	    scanf(GPIV_ASCII_IMG_FMT, &ira, &irb, &fdum);
	    img[row][column] = fdum;
	}
    }

}




void
gpiv_write_ascii_image(guint16 **img, 
		       GpivImagePar image_par
		       )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Writes ascii image from stdin 
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 *  INPUTS:
 *      img:            2-dimensional matrix containing nrows x ncoloms pixels
 *      nrows:          number of rows in image
 *      ncolumns:       number of columns in image
 *---------------------------------------------------------------------------*/
{
    int row = 0, column = 0, ira = 0, irb = 0;
    char cdum = '\0';
    double fdum = 0;
    int nrows =  image_par.nrows;
    int ncolumns =  image_par.ncolumns;

    assert(img[0] != NULL);

    for (row = 0; row < nrows; row++) {
	printf("%c", cdum);
	for (column = 0; column < ncolumns; column++) {
	    printf(GPIV_ASCII_IMG_FMT, ira, irb, fdum);
	    img[row][column] = fdum;
	}
    }

}




int 
gpiv_count_pivdata(GpivPivData *piv_data, 
		   char data_line[GPIV_MAX_LINES][GPIV_MAX_CHARS],
		   int *ndata_lines
		   )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     - Reads data from STDIN to find array length of x-and y data, 
 *       provided that both x and y-data are in incremental or decremental 
 *       order
 *     - Determines if scaled data have been used in order to determine which 
 *       format/header has to be used for writing the data
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      piv_data:       piv data to be read
 *
 * OUTPUTS:
 *      piv.n, piv,y:   number of x and y- data
 *      data_line:      all lines that are read from STDIN
 *      nlines:         total number of lines, including comment lines
 *
 * RETURNS:
 *     scale:          0 or 1 for (non) scaled data
 *---------------------------------------------------------------------------*/
{
  int increment = 0, scale = 0;
  float x = 0, y = 0;
  float dat_x = -10.0e5, dat_y = -10.0e5;
  int line_i = 0, line_nr = 0; 

  
  piv_data->nx = 0;
  piv_data->ny = 0;

  while (fgets(data_line[line_i], GPIV_MAX_CHARS, stdin) != NULL) {
    if (data_line[line_i][0] == '#' && scale == 0) {
      scale = gpiv_find_pivdata_scale(data_line[line_i]);
    }
    if (data_line[line_i][0] != '#' && data_line[line_i][0] != '\n' && 
	data_line[line_i][0] != '\t') {
      sscanf(data_line[line_i],"%f %f",&x, &y);
      if(line_nr == 0) {
	dat_x=x;
	dat_y=y;
      }
      if(line_nr == 1) {
	if (x > dat_x || y > dat_y) {
	  increment = 1;
	} else {
	  increment = 0;
	}
      }
      line_nr++;
      if (increment == 1) {
	if (x > dat_x){
	  dat_x=x;
	  piv_data->nx++;
	}
	if (y > dat_y) {
	  dat_y=y;
	  piv_data->ny++;
	}
      } else {
	if (x < dat_x){
	  dat_x=x;
	  piv_data->nx++;
	}
	if (y < dat_y) {
	  dat_y=y;
	  piv_data->ny++;
	}
      }
    }
    line_i++;
  }

/*
 * Total amount of lines, including comment and empty one's
 */
  *ndata_lines = line_i;
  piv_data->nx++;
  piv_data->ny++;

  return scale;
}


int
gpiv_fcount_pivdata(char *fname, 
		    GpivPivData *piv_data
		    )
/*-----------------------------------------------------------------------------
 *DESCRIPTION:
 *
 *     - Reads PIV data from ASCII file to find array length of x-and y data, 
 *       provided that both x and y-data are in incremental or decremental 
 *       order
 *     - Determines if scaled data have been used in order to determine which 
 *       format/header has to be used for writing the data
 *
 * PROTOTYPE LOCATATION:
 *      io.h
 *
 * INPUTS:
 *      fname:          pointer to complete filename containing data
 *
 * OUTPUTS:
 *      piv_data:       pointer to piv_data to be read
 *
 * RETURNS:
 *      scale:          0 or 1 for (non) scaled data
 *---------------------------------------------------------------------------*/
{
    FILE *fp_in;
    int increment = 0, scale = 0;
    float x = 0, y = 0;
    float dat_x = -10.0e5, dat_y = -10.0e5;
    char line_dum[GPIV_MAX_CHARS];
    int line_i = 0, line_nr = 0; 

    piv_data->nx = 0;
    piv_data->ny = 0;

    if((fp_in = fopen(fname,"r")) == NULL) { 
        gpiv_warning("GPIV_FCOUNT_PIVDATA: Failure opening for input"); 
        return -1;
    } 
 
    
    while (fgets(line_dum, GPIV_MAX_CHARS, fp_in) != NULL) {
        if (line_dum[0] == '#' && scale == 0) {
            scale=gpiv_find_pivdata_scale(line_dum);
        }
        line_i++;
        if (line_dum[0] != '#' && line_dum[0] != '\n' && line_dum[0] != '\t') {
            sscanf(line_dum,"%f %f",&x, &y);
            if(line_nr == 0) {
                dat_x=x;
                dat_y=y;
            }

            if(line_nr == 1) {
                if (x > dat_x || y > dat_y) {
                    increment = 1;
                } else {
                    increment = 0;
                }
            }

            line_nr++;

            if (increment == 1) {
                if (x > dat_x){
                    dat_x=x;
                    piv_data->nx++;
                }
                if (y > dat_y) {
                    dat_y=y;
                    piv_data->ny++;
                }
            } else {
                if (x < dat_x){
                    dat_x=x;
                    piv_data->nx++;
                }
                if (y < dat_y) {
                    dat_y=y;
                    piv_data->ny++;
                }
            }

        }
    }

/*   nlines = line_nr; */
    piv_data->nx++;
    piv_data->ny++;
    fclose(fp_in);
    
    return scale;
}



int 
gpiv_find_pivdata_origin (char line[GPIV_MAX_CHARS]
			  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Determines the name of the program that generated the data
 *
 * PROTOTYPE LOCATATION:
 *      io.h
 *
 * INPUTS:
 *      line:           character line containing program that generated data
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      pdf:            enumerator of data origin
 *---------------------------------------------------------------------------*/
{
    if (strstr(line,"rr") != '\0') {
        pdf = GPIV_RR;
    } else if (strstr(line,"DaVis") != '\0') { 
        pdf = GPIV_DAV;
    } else {
/* fprintf (stderr,"\n%s %s warning: No format defined: taking 'rr' format", */
/* 	     LIBNAME,function_name); */
        pdf = GPIV_RR;
    }

    return pdf;
}



int 
gpiv_find_pivdata_scale (char line[GPIV_MAX_CHARS]
			 )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Determines if the data have been scaled in order to read x-and y 
 *     positions as floating point varibles
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      line:           character line containing program that generated data
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      scale:          1 if scaled, else 0
 *---------------------------------------------------------------------------*/
{
    int scale = 0;
    
    if (strstr(line,"scale") != '\0') {
        scale=1;
    }

    return scale;
}



void
gpiv_read_pivdata (GpivPivData *piv_data, 
		   char data_line[GPIV_MAX_LINES][GPIV_MAX_CHARS], 
		   int ndata_lines,
		   char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS], 
		   int *ncomment_lines
		   )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads data from stdin
 *
 * PROTOTYPE LOCATATION:
 *      io.h
 *
 * INPUTS:
 *      fname:          complete filename
 *      piv:            struct piv_data containing all variables in a datafile
 *                      piv.nx and piv.ny have to be known.
 *      comment_line:   lines contain comment (staring with '#' sign at first 
 *                      column)
 *      ncomment_lines: number of comment lines
 *
 * OUTPUTS:
 *      comment_line:   comment lines (lines starting with '#'), except header
 *      ncomment_lines: number of comment lines
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    int i = 0, j = 0, nc = 0, line_nr = 0;

    assert(piv_data->point_x != NULL);
    assert(piv_data->point_y != NULL);
    assert(piv_data->dx != NULL);
    assert(piv_data->dy != NULL);
    assert(piv_data->snr != NULL);
    assert(piv_data->peak_no != NULL);


    while ((i < piv_data->ny) && (j < piv_data->nx)) {
        if (data_line[line_nr][0] == '#' && 
            (strstr(data_line[line_nr],"x(m)") == '\0') && 
            (strstr(data_line[line_nr],"x(px)") == '\0')) {
            strcpy(comment_line[nc],data_line[line_nr]);
            nc++;
        } else if (data_line[line_nr][0] != '\n' && 
                   data_line[line_nr][0] != '\t' && 
                   (strstr(data_line[line_nr],"x(m)") == '\0') && 
                   (strstr(data_line[line_nr],"x(px)") == '\0')) {
            sscanf(data_line[line_nr],"%f %f %f %f %f %d", &piv_data->point_x[i][j], 
                   &piv_data->point_y[i][j],&piv_data->dx[i][j], &piv_data->dy[i][j], 
                   &piv_data->snr[i][j], &piv_data->peak_no[i][j]);
            if (j < piv_data->nx - 1) {
                j++; 
            } else if (i < piv_data->ny - 1) { 
                j=0; 
                i++; 
            } else {
                break; 
            }
        }
        line_nr++;
    }

    *ncomment_lines = nc;

}




char *
gpiv_fread_pivdata (char *fname, 
		    GpivPivData *piv_data, 
		    char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS], 
		    int *ncomment_lines
		    )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads piv data from ascii data file
 *
 * PROTOTYPE LOCATATION:
 *      io.h
 *
 * INPUTS:
 *      fname:          complete filename
 *
 * OUTPUTS:
 *      piv:            struct piv_data containing all variables in a datafile
 *                     piv.nx and piv.ny have to be known.
 *      comment_line:   lines contain comment (staring with '#' sign at first 
 *                     column)
 *      ncomment_lines: number of comment lines
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int i = 0, j = 0, nc = 0;
    FILE *fp;
    char line_dum[GPIV_MAX_CHARS];
    
    assert(piv_data->point_x != NULL);
    assert(piv_data->point_y != NULL);
    assert(piv_data->dx != NULL);
    assert(piv_data->dy != NULL);
    assert(piv_data->snr != NULL);
    assert(piv_data->peak_no != NULL);


    if((fp=fopen(fname,"r")) == NULL) { 
        err_msg = "GPIV_FREAD_PIVDATA: Failure opening %s for input"; 
        gpiv_warning("%s", err_msg);
        return err_msg;
    } 
    
    while (fgets(line_dum, GPIV_MAX_CHARS, fp) != NULL) {
        if (line_dum[0] == '#' && (strstr(line_dum,"x(m)") == '\0') && 
            (strstr(line_dum,"x(px)") == '\0')) {
            strcpy(comment_line[nc],line_dum);
            nc++;
        } else if (line_dum[0] != '\n' && line_dum[0] != '\t' && 
                   (strstr(line_dum,"x(m)") == '\0') && 
                   (strstr(line_dum,"x(px)") == '\0')) {
            sscanf(line_dum,"%f %f %f %f %f %d", &piv_data->point_x[i][j], 
                   &piv_data->point_y[i][j],&piv_data->dx[i][j], &piv_data->dy[i][j], 
                   &piv_data->snr[i][j], &piv_data->peak_no[i][j]);
            if (j < piv_data->nx - 1) {
                j++; 
     } else if (i < piv_data->ny - 1) { 
         j=0; 
         i++; 
     } else {
         break; 
     }
        }
    }
    fclose(fp);
    *ncomment_lines = nc;


    return err_msg;
}




void 
gpiv_write_pivdata (GpivPivData * piv_data,
		    char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS], 
		    int ncomment_lines, 
		    int scale, 
		    char *RCSID
		    )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Writes PIV data to stdout
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      piv_data:       struct piv_data containing all variables in a datafile
 *      comment_line:   lines contain comment (staring with '#' sign at first column)
 *      ncomment_lines: number of comment lines
 *      scale:          parameter to write piv.point_x and piv.point_y as int or as float
 *      RCSID:          identifier of program that generated datafile
 *---------------------------------------------------------------------------*/
{
    int i, j;

    assert(piv_data->point_x != NULL);
    assert(piv_data->point_y != NULL);
    assert(piv_data->dx != NULL);
    assert(piv_data->dy != NULL);
    assert(piv_data->snr != NULL);
    assert(piv_data->peak_no != NULL);


/*
 * Writing header and data
 */
    for (i=0; i < ncomment_lines; i++) printf("%s", comment_line[i]); 
    printf("# %s\n", RCSID);

    if (scale == 1) {
        printf("\n# x(m)        y(m)         Vx(m/s)      Vy(m/s)      snr        peak#\n");
        for (i = 0; i < piv_data->ny; i++) { 
            for (j = 0; j < piv_data->nx; j++) {
                printf(GPIV_PIV_S_FMT, 
                       piv_data->point_x[i][j], 
                       piv_data->point_y[i][j], 
                       piv_data->dx[i][j], 
                       piv_data->dy[i][j], 
                       piv_data->snr[i][j], 
                       piv_data->peak_no[i][j]);
            }
        }
        
    } else {
        printf("\n# x(px) y(px)    dx(px)       dy(px)       snr  peak#\n");
        for (i = 0; i < piv_data->ny; i++) {
            for (j = 0; j < piv_data->nx; j++) {
                printf(GPIV_PIV_FMT, 
                       piv_data->point_x[i][j], 
                       piv_data->point_y[i][j], 
                       piv_data->dx[i][j], 
                       piv_data->dy[i][j], 
                       piv_data->snr[i][j], 
                       piv_data->peak_no[i][j]
                       );      
            }
        }
    }

    fflush (stdout);
}




char *
gpiv_fwrite_pivdata (char *fname, 
		     GpivPivData *piv_data, 
		     char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS], 
		     int ncomment_lines, 
		     int scale, 
		     char *RCSID
		     )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes PIV data to file
 *
 * PROTOTYPE LOCATATION:
 *      io.h
 *
 * INPUTS:
 *      fname:          complete filename
 *      piv_data:       struct piv_data containing all variables in a datafile
 *      comment_line:   lines contain comment (staring with '#' sign at first 
 *                      column)
 *      ncomment_lines: number of comment lines
 *      scale:          parameter to write piv.point_x and piv.point_y as int 
 *                      or as float
 *      RCSID:          identifier of program that generated datafile
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    FILE *fp;
    int i, j;
    
    assert(piv_data->point_x != NULL);
    assert(piv_data->point_y != NULL);
    assert(piv_data->dx != NULL);
    assert(piv_data->dy != NULL);
    assert(piv_data->snr != NULL);
    assert(piv_data->peak_no != NULL);
    

    if((fp=fopen(fname,"w")) == NULL) { 
        err_msg = "GPIV_FWRITE_PIVDATA: Failure opening for output";
        gpiv_warning("%s", err_msg);
        return err_msg;
    } 
    

/*
 * Writing header and data
 */
    for (i=0; i < ncomment_lines; i++) fprintf(fp,"%s", comment_line[i]); 
    fprintf(fp,"# %s\n", RCSID);

    if (scale == 1) {
        fprintf(fp,"\n# x(m)        y(m)         Vx(m/s)      Vy(m/s)      snr        peak#\n");
        for (i = 0; i < piv_data->ny; i++) {
            for (j = 0; j < piv_data->nx; j++) { 
                fprintf(fp, GPIV_PIV_S_FMT, 
                        piv_data->point_x[i][j], 
                        piv_data->point_y[i][j], 
                        piv_data->dx[i][j], 
                        piv_data->dy[i][j], 
                        piv_data->snr[i][j], 
                        piv_data->peak_no[i][j]);      
            }
        }

    } else {
        fprintf(fp,"\n# x(px) y(px)    dx(px)       dy(px)       snr  peak#\n");
        for (i = 0; i < piv_data->ny; i++) {
            for (j = 0; j < piv_data->nx; j++) { 
                fprintf(fp, GPIV_PIV_FMT, 
                        piv_data->point_x[i][j], 
                        piv_data->point_y[i][j], 
                        piv_data->dx[i][j], 
                        piv_data->dy[i][j], 
                        piv_data->snr[i][j], 
                        piv_data->peak_no[i][j]);      
            }
        }
    }
    
    fclose(fp);
    return err_msg;
}



char *
gpiv_fread_scdata (char *fname, 
                   GpivScalarData * scalar, 
                   char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS], 
                   int *ncomment_lines
                   )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads scalar data from ascii data file
 *
 * PROTOTYPE LOCATATION:
 *      io.h
 *
 * INPUTS:
 *      fname:          complete filename
 *
 * OUTPUTS:
 *      scalar_data:    struct piv_data containing all variables in a datafile
 *                      piv.nx and piv.ny have to be known.
 *      comment_line:   lines contain comment (staring with '#' sign at first column)
 *      ncomment_lines: number of comment lines
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int i = 0, j = 0, nc = 0;
    FILE *fp;
    char line_dum[GPIV_MAX_CHARS];
    
    assert(scalar->point_x != NULL);
    assert(scalar->point_y != NULL);
    assert(scalar->scalar != NULL);
    assert(scalar->flag != NULL);


    if((fp=fopen(fname,"r")) == NULL) { 
        err_msg = "GPIV_FREAD_SCDATA: Failure opening for input"; 
        gpiv_warning("%s", err_msg);
        return err_msg;
    } 
    
    while (fgets(line_dum, GPIV_MAX_CHARS, fp) != NULL) {
        if (line_dum[0] == '#' && (strstr(line_dum,"x(m)") == '\0') && 
            (strstr(line_dum,"x(px)") == '\0')) {
            strcpy(comment_line[nc],line_dum);
            nc++;
        } else if (line_dum[0] != '\n' && line_dum[0] != '\t' && 
                   (strstr(line_dum,"x(m)") == '\0') && 
                   (strstr(line_dum,"x(px)") == '\0')) {
            sscanf(line_dum,"%f %f %f %d", &scalar->point_x[i][j], 
                   &scalar->point_y[i][j],&scalar->scalar[i][j], 
                   &scalar->flag[i][j]);
            if (j < scalar->nx - 1) {
                j++; 
     } else if (i < scalar->ny - 1) { 
         j=0; 
         i++; 
     } else {
         break; 
     }
        }
    }
    fclose(fp);
    *ncomment_lines = nc;


    return err_msg;
}




void
gpiv_write_scdata (GpivScalarData * scalar, 
		   char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS], 
		   int ncomment_lines, 
		   int scale, 
		   char *RCSID
		   )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes scalar data to stdout
 *
 * PROTOTYPE LOCATATION:
 *      io.h
 *
 * INPUTS:
 *      scalar:         struct GpivScalarData containing all variables in a datafile
 *      comment_line:   lines contain comment (staring with '#' sign at first 
 *                      column)
 *      ncomment_lines: number of comment lines
 *      scale:          parameter to write scalar.point_x and scalar.point_y as
 *                      int or as float
 *      RCSID:          identifier of program that generated datafile
 *---------------------------------------------------------------------------*/
{
    int i, j;

    assert(scalar->point_x != NULL);
    assert(scalar->point_y != NULL);
    assert(scalar->scalar != NULL);
    assert(scalar->flag != NULL);


/*
 *  Writing comment lines
 */
    for (i=0; i < ncomment_lines; i++) printf("%s", comment_line[i]); 

/*
 * Writing header and data of scalar data
 */
    if (scale == 1) {
        printf("\n#       x(m)         y(m)        scalar      peak#\n");
        for (i=0; i< scalar->ny; i++) {
            for (j=0; j< scalar->nx; j++) { 
                printf(GPIV_SCALAR_S_FMT, scalar->point_x[i][j], 
                       scalar->point_y[i][j], 
                       scalar->scalar[i][j], scalar->flag[i][j]);      
            }
        }
        
    } else {
        printf("\n# x(px) y(px) scalar      peak#\n");
        for (i=0; i< scalar->ny; i++) {
            for (j=0; j< scalar->nx; j++) { 
              printf(GPIV_SCALAR_FMT, scalar->point_x[i][j], 
                     scalar->point_y[i][j], 
                     scalar->scalar[i][j], scalar->flag[i][j]);      
            }
        }
    }
    
}




char *
gpiv_fwrite_scdata (char *fname, 
		    GpivScalarData * scalar, 
		    char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS], 
		    int ncomment_lines, 
		    int scale, 
		    char *RCSID
		    )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes scalar data to file
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      scalar:         struct GpivScalarData containing all variables in a datafile
 *      comment_line:   lines contain comment (staring with '#' sign at first 
 *                      column)
 *      ncomment_lines: number of comment lines
 *      scale:          parameter to write scalar.point_x and scalar.point_y as
 *                      int or as float
 *      RCSID:          identifier of program that generated datafile
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    FILE *fp;
    int i, j;
    
    assert(scalar->point_x != NULL);
    assert(scalar->point_y != NULL);
    assert(scalar->scalar != NULL);
    assert(scalar->flag != NULL);


/*
 * Opens output file
 */
    if((fp=fopen(fname,"w")) == NULL) { 
        err_msg = "GPIV_FWRITE_SCDATA: Failure opening for output";
        gpiv_warning("%s", err_msg);
        return err_msg;
    } 


/*
 * Writing comment lines
 */
    for (i=0; i < ncomment_lines; i++) fprintf(fp,"%s", comment_line[i]); 

/*
 * Writing header and data of vorticity
 */
    if (scale == 1) {
        fprintf(fp,"\n#       x(m)         y(m)        scalar      peak#\n");
        for (i=0; i< scalar->ny; i++) {
            for (j=0; j< scalar->nx; j++) { 
                fprintf(fp,GPIV_SCALAR_S_FMT, scalar->point_x[i][j], 
                        scalar->point_y[i][j], 
                        scalar->scalar[i][j],scalar->flag[i][j]);      
            }
        }

    } else {
        fprintf(fp,"\n# x(px) y(px) scalar      peak#\n");
        for (i=0; i< scalar->ny; i++) {
            for (j=0; j< scalar->nx; j++) { 
                fprintf(fp,GPIV_SCALAR_FMT, scalar->point_x[i][j], 
                        scalar->point_y[i][j], 
                        scalar->scalar[i][j],scalar->flag[i][j]);      
            }
        }
    }
    
    fclose(fp);
    return err_msg;
}



char *
gpiv_fwrite_sc_griddata (char *fname, 
			 GpivScalarData * scalar, 
			 char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS], 
			 int ncomment_lines, 
			 int scale, 
			 char *RCSID
			 )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Writes scalar data to file in grid format for gnuplot
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      scalar:         struct GpivScalarData containing all variables in a datafile
 *      comment_line:   lines contain comment (staring with '#' sign at first 
 *                      column)
 *      ncomment_lines: number of comment lines
 *      scale:          parameter to write scalar.point_x and scalar.point_y as
 *                      int or as float
 *      RCSID:          identifier of program that generated datafile
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    FILE *fp;
    int i, j;
    
    assert(scalar->point_x != NULL);
    assert(scalar->point_y != NULL);
    assert(scalar->scalar != NULL);
    assert(scalar->flag != NULL);
    

/*
 * Opens output file
 */
    if((fp=fopen(fname,"w")) == NULL) { 
        err_msg = "gpiv_fwrite_sc_griddata: Failure opening for output"; 
        gpiv_warning("%s", err_msg);
        return err_msg;
  } 


/*
 * Writing comment lines
 */
    for (i=0; i < ncomment_lines; i++) fprintf(fp,"%s", comment_line[i]); 
    
/*
 * Writing header and data of vorticity
 */
    if (scale == 1) {
        fprintf(fp,"\n#       x(m)         y(m)        scalar      peak#\n");
        for (i=0; i< scalar->ny; i++) {
            fprintf(fp,"\n");
            for (j=0; j< scalar->nx; j++) { 
                fprintf(fp,GPIV_SCALAR_S_FMT, scalar->point_x[i][j], 
                        scalar->point_y[i][j], 
                        scalar->scalar[i][j], scalar->flag[i][j]);      
            }
        }

    } else {
        fprintf(fp,"\n# x(px) y(px) scalar      peak#\n");
        for (i=0; i< scalar->ny; i++) {
            fprintf(fp,"\n");
            for (j=0; j< scalar->nx; j++) { 
                fprintf(fp,GPIV_SCALAR_FMT, scalar->point_x[i][j], 
                        scalar->point_y[i][j], 
                        scalar->scalar[i][j],scalar->flag[i][j]);      
            }
        }
    }
    
    fclose(fp);
    return err_msg;
}



char *
gpiv_fwrite_sc_mtvgriddata (char *fname, 
                            GpivScalarData * scalar, 
                            char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS], 
                            int ncomment_lines, 
                            int scale, 
                            char *RCSID
                            )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Writes scalar data to file in grid format for plotmtv
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      scalar:         struct GpivScalarData containing all variables in a datafile
 *      comment_line:   lines contain comment (staring with '#' sign at first 
 *                      column)
 *      ncomment_lines: number of comment lines
 *      scale:          parameter to write scalar.point_x and scalar.point_y as
 *                      int or as float
 *      RCSID:          identifier of program that generated datafile
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
  char *err_msg = NULL;
  FILE *fp;
  int i, j;

  assert(scalar->point_x != NULL);
  assert(scalar->point_y != NULL);
  assert(scalar->scalar != NULL);
/*   assert(scalar->flag != NULL); */


/*
 * Opens output file
 */
  if((fp=fopen(fname,"w")) == NULL) { 
      err_msg = "GPIV_FWRITE_SC_MTVGRIDDATA: Failure opening for output";
      gpiv_warning("%s", err_msg);
      return err_msg;
  } 


/*
 * Writing header and data of griddata
 */
  fprintf(fp, "$ DATA=CONTOUR\n");
  fprintf(fp, "%% contfill=T\n");
  fprintf(fp, "%% nx=%d xmin=%f xmax=%f\n", scalar->nx, 
          scalar->point_x[0][0], scalar->point_x[0][scalar->nx - 1]);
  fprintf(fp, "%% ny=%d ymin=%f ymax=%f\n", scalar->ny, 
          scalar->point_y[0][0], scalar->point_y[scalar->ny - 1][0]);

  for (i=0; i < scalar->ny; i++) {
      fprintf(fp, "\n");
      for (j=0; j < scalar->nx; j++) { 
 	  fprintf(fp, "%f ", scalar->scalar[i][j]);      
      }
  }


  fclose(fp);

  return err_msg;
}



void
gpiv_print_histo(GpivBinData *klass, 
		 char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS], 
		 int ncomment_lines, 
		 int scale, 
		 char *RCSID
		 )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Writing histogram data to stdout
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      fname:          complete filename
 *      klass:          struct GpivBinData containing the values for each bin
 *      comment_line:   lines contain comment (staring with '#' sign at first
 *                      column)
 *      ncomment_lines: number of comment lines
 *      RCSID:          identifier of program that generated the output datafile
 *---------------------------------------------------------------------------*/
{
  int i, *count=klass->count, nbins=klass->nbins;
  float *centre=klass->centre;

  assert(klass->count != NULL);
  assert(klass->bound != NULL);
  assert(klass->centre != NULL);
  

  printf ("\n#%s", RCSID);
  printf ("\n# bin  frequency\n");
  for (i = 0; i < nbins; i++) {
    printf("\n   %f          %d",centre[i], count[i]); 
/*     printf("%d %d\n",i, count[i]); */
  }

}



void
gpiv_fprint_histo(char *fname, 
		  GpivBinData *klass, 
		  char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS], 
		  int ncomment_lines, 
		  int scale, 
		  char *RCSID
		  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Writing histogram data to ouput file
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      fname:          complete filename
 *      klass:          struct bin containing the values for each bin
 *      comment_line:   lines contain comment (staring with '#' sign at first
 *                      column)
 *      ncomment_lines: number of comment lines
 *      RCSID:         identifier of program that generated the output datafile
 *---------------------------------------------------------------------------*/
{
  FILE *fp;
  int i, *count=klass->count, nbins=klass->nbins;
  float *centre=klass->centre;

  assert(klass->count != NULL);
  assert(klass->bound != NULL);
  assert(klass->centre != NULL);
  

  if ((fp = fopen (fname, "wb")) == NULL) {
      fprintf (stderr,"Failure opening %s for output", fname);
      exit(1);
    }

  fprintf (fp,"\n#%s", RCSID);
  fprintf (fp,"\n# bin  frequency\n");
  for (i = 0; i < nbins; i++) {
    fprintf(fp,"\n   %f          %d", centre[i], count[i]); 
  }

  fclose(fp);
}




void
gpiv_print_cumhisto_eqdatbin(GpivBinData *klass, 
                             char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS], 
                             int ncomment_lines, 
                             int scale, 
                             char *RCSID
                             )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Writing cumulative histogram data with an equal number of date per bin 
 *     or klass to stdout. Special output for validation; work around to print
 *     float data as y-values.
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      klass:          struct GpivBinData containing the values for each bin
 *      comment_line:   lines contain comment (staring with '#' sign at first 
 *                      column)
 *      ncomment_lines: number of comment lines
 *      RCSID:          identifier of program that generated the output datafile
 *---------------------------------------------------------------------------*/
{
  int i, nbins=klass->nbins;
  float *bound=klass->bound, *centre=klass->centre;

  assert(klass->bound != NULL);
  assert(klass->centre != NULL);
  

  for (i = 0; i < ncomment_lines; i++) {
      printf ("\n#%s", comment_line[i]);
  }
  printf ("\n#%s", RCSID);
  printf ("\n# bound      value\n");
  for (i = 0; i < nbins; i++) {
      printf ("\n   %f          %f", bound[i], centre[i]); 
  }

}



void
gpiv_fprint_cumhisto_eqdatbin(char *fname, 
                              GpivBinData *klass, 
                              char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS], 
                              int ncomment_lines, 
                              int scale, 
                              char *RCSID
                              )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Writing cumulative histogram data with an equal number of date per bin 
 *     or klass to ouput file. Special output for validation; work around to
 *     print float data as y-values.
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      fname:          complete filename
 *      klass:          struct bin containing the values for each bin
 *      comment_line:   lines contain comment (staring with '#' sign at first 
 *                      column)
 *      ncomment_lines: number of comment lines
 *      RCSID:         identifier of program that generated the output datafile
 *---------------------------------------------------------------------------*/
{
  FILE *fp;
  int i, nbins=klass->nbins;
  float *bound=klass->bound, *centre=klass->centre;

  assert(klass->bound != NULL);
  assert(klass->centre != NULL);
  

  if ((fp = fopen (fname, "wb")) == NULL) {
      fprintf (stderr,"Failure opening %s for output", fname);
      exit(1);
    }


  for (i = 0; i < ncomment_lines; i++) {
      fprintf (fp,"\n#%s", comment_line[i]);
  }
  fprintf (fp,"\n#%s", RCSID);
  fprintf (fp,"\n# bound      value\n");
  for (i = 0; i < nbins; i++) {
      fprintf (fp,"\n   %f          %f", bound[i], centre[i]); 
  }

  fclose(fp);
}



char *
gpiv_fread_pgm_image(char *fname,
                     guint16 **img1,
                     guint16 **img2, 
                     GpivImagePar * image_par,
                     gint line_nr
                     ) 
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Read raw portable graymap (pgm) format image data from file
 *
 * INPUTS:
 *      fname:          pointer to complete filename to be written
 *      image_par:      image parameters containing number of rows an columns
 *
 * OUTPUTS:
 *      img1:           first binary image, memory will be allocated here
 *      img2:           second binary image, memory will be allocated here
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    FILE *fp;
    char *err_msg = NULL, img_type[GPIV_MAX_CHARS];
    guint8 **img1_dum, **img2_dum;
    gint i, j;
    char line[GPIV_MAX_CHARS];

/* fflush(stdout); */
    gpiv_warning("gpiv_fread_pgm_image:: 0 0");
/*     assert(img1[0] == NULL); */
/*     assert(img2[0] == NULL); */

    gpiv_warning("gpiv_fread_pgmimage:: 0 ");
    if ((fp = fopen(fname, "rb")) == NULL) {
	err_msg = "GPIV_FREAD_IMAGE: Failure opening for input";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }


    fscanf(fp, "%s", img_type);
    if (strcmp(img_type, "P5") != 0) {
        err_msg = "gpiv_fread_pgmimage: Not a raw pgm image";
        return err_msg;
    }

    gpiv_warning("gpiv_fread_pgmimage:: 1b ");
/*     gpiv_img_read_pgmheader (fp, image_par); */
/*     gpiv_warning("gpiv_fread_pgmimage:: 1c "); */

    if ((err_msg = gpiv_img_check_header_read(*image_par)) != NULL) {
        gpiv_error ("%s", err_msg); 
    }


gpiv_warning("gpiv_fread_pgmimage:: 2 ncols = %d nrows = %d", 
             image_par->ncolumns, image_par->nrows);

/*     img1 = gpiv_alloc_img(*image_par); */
/*     if (image_par->x_corr) { */
/*         img2 = gpiv_alloc_img(*image_par); */
/*     } */
    
    for (i = 0; i <line_nr + 1; i++) {
        fgets(line, GPIV_MAX_CHARS, fp);
    }

    if (image_par->depth <= 8) {
        img1_dum = gpiv_matrix_guint8(image_par->nrows, image_par->ncolumns);
        fread(img1_dum[0], image_par->ncolumns * image_par->nrows * (sizeof(guint8)), 1, fp);
        if (image_par->x_corr) {
            img2_dum = gpiv_matrix_guint8(image_par->nrows, image_par->ncolumns);
            fread(img2_dum[0], image_par->ncolumns * image_par->nrows * 
                  (sizeof(guint8)), 1, fp);
        } else {
            img2_dum = img1_dum;
        }

        for (i = 0; i < image_par->nrows; i++) {
            for (j = 0; j < image_par->ncolumns; j++) {
                img1[i][j]  = img1_dum[i][j];
                img2[i][j] = img2_dum[i][j];
            }
        }
        
        gpiv_free_matrix_guint8(img1_dum);
        if (image_par->x_corr) {
            gpiv_free_matrix_guint8(img2_dum);
        }
        

    } else {
        fread(img1[0], image_par->ncolumns * image_par->nrows * (sizeof(guint16)), 1, fp);
        if (image_par->x_corr) {
            fread(img2[0], image_par->ncolumns * image_par->nrows * (sizeof(guint16)), 1, 
                  fp);
        } else {
            img2 = img1;
        }
    }

    fclose(fp);
    return err_msg;
}



char *
gpiv_fwrite_pgm_image(char *fname,
                      guint16 **img1, 
                      guint16 **img2, 
                      GpivImagePar image_par
                      )
/*  ---------------------------------------------------------------------------
 * DESCRIPTION:
 *     writes binary portable graymap (pgm) format image to file
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      fname:          pointer to complete filename to be written
 *      img1:           first 2-dimensional matrix containing nrows x ncoloms pixels
 *      img2:           second 2-dimensional matrix containing nrows x ncoloms pixels
 *      image_par:      image parameters containing number of rows an columns
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    FILE *fp;
    gchar *err_msg = NULL;
    unsigned char **l_img1 = NULL, **l_img2 = NULL;
    gint nrows =  image_par.nrows;
    gint ncolumns =  image_par.ncolumns;
    gint depth = image_par.depth;
    gint x_corr = image_par.x_corr;
    gint i, j;
    assert(img1[0] != NULL);
    assert(img2[0] != NULL);


    if ((fp = fopen(fname, "wb")) == NULL) {
	err_msg = "GPIV_FWRITE_IMAGE: Failure opening for input";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    fprintf(fp, "P5\n");
    gpiv_img_fprint_pgmheader (fp, image_par); 

    if (depth <= 8) {
        l_img1 = gpiv_ucmatrix(image_par.nrows, image_par.ncolumns);
        if (x_corr) {
            l_img2 = gpiv_ucmatrix(image_par.nrows, image_par.ncolumns);
        }
    }

    for (i = 0; i < image_par.nrows; i++) {
        for (j = 0; j < image_par.ncolumns; j++) {
            l_img1[i][j] = img1[i][j];
            l_img2[i][j] = img2[i][j];
        }
    }

    if (depth <= 8) {
        fwrite(l_img1[0],  nrows, ncolumns * (sizeof(char)), fp);
       if (x_corr) {
            fwrite(l_img2[0], nrows, ncolumns * (sizeof(char)), fp);
       }

    } else {
        fwrite(img1[0], ncolumns * nrows * (sizeof(guint16)), 1, fp);
        if (x_corr) {
            fwrite(img2[0], ncolumns * nrows * (sizeof(guint16)), 1, fp);
        }
    }

    if (depth <= 8) {
        gpiv_free_ucmatrix(l_img1);
        if (x_corr) {
            gpiv_free_ucmatrix(l_img2);
        }
    }
    fclose(fp);
    return err_msg;
#undef MAX_ROWLENGTH
}



char *
gpiv_fcreate_hdf5 (char *fname
                  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Creates an hdf5 data file with DATA and PARAMETERS groups
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      fname:          complete filename
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
/*
 * HDF declarations
 */
    hid_t       file_id, group_id; 
    herr_t      status;

    if ((file_id = H5Fcreate(fname, H5F_ACC_TRUNC, H5P_DEFAULT, H5P_DEFAULT))
        < 0) {
        err_msg = "GPIV_FCREATE_HDF5: H5Fcreate unable to create file";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    group_id = H5Gcreate (file_id, "IMAGE", 0);
    status = H5Gclose (group_id);
    group_id = H5Gcreate (file_id, "POSITIONS", 0);
    status = H5Gclose (group_id);
    group_id = H5Gcreate (file_id, "PIV", 0);
    status = H5Gclose (group_id);
/*     group_id = H5Gcreate (file_id, "VECTORS", 0); */
/*     status = H5Gclose (group_id); */
    group_id = H5Gcreate (file_id, "SCALARS", 0);
    status = H5Gclose (group_id);

    status = H5Fclose(file_id); 
    return err_msg;
}



char *
gpiv_fread_hdf5_image(char *fname, 
#ifdef HD5_IMAGE_INT
                       int **img1, 
                       int **img2, 
#else
                       guint16 **img1, 
                       guint16 **img2, 
#endif
                      GpivImagePar * image_par
                       )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Reads IMAGE data to a hdf5 data file
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      fname:          complete filename
 *      image_par:      image parameters
 *
 * OUTPUTS
 *      img:            2-dimensional matrix containing nrows x ncoloms pixel 
 *                      values
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{    
    char *err_msg = NULL;
    int i, j, rank_x, rank_y, rank_vx, rank_vy;
    float *point_x_hdf = NULL, *point_y_hdf = NULL;

/*static char *msg_error = NULL;/*

/*
 * HDF declarations
 */
    hid_t       file_id, group_id, attribute_id, dataset_id, 
        dataspace_id; 
    hsize_t     dims[2];
    herr_t      status;


    assert(img1[0] != NULL);
    if (image_par->x_corr) assert(img2[0] != NULL);

    if ((i = H5Fis_hdf5(fname)) == 0)  {
        err_msg = "GPIV_FREAD_HDF5_IMAGE: not an hdf5 file";
        gpiv_warning("%s", err_msg);
       return err_msg;
    }

    file_id = H5Fopen(fname, H5F_ACC_RDONLY, H5P_DEFAULT);
    group_id = H5Gopen (file_id, "IMAGE");

/*
 * image #1
 */
    dataset_id = H5Dopen(group_id, "#1");
    dataspace_id = H5Dget_space(dataset_id);
#ifdef HD5_IMAGE_INT
    if ((status = H5Dread(dataset_id, H5T_NATIVE_INT, H5S_ALL, H5S_ALL, 
                          H5P_DEFAULT, img1[0])) < 0) {
        err_msg = "GPIV_FREAD_HDF5_IMAGE: failing H5Dread";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
#else
    if ((status = H5Dread(dataset_id, H5T_NATIVE_USHORT, H5S_ALL, H5S_ALL, 
                          H5P_DEFAULT, img1[0])) < 0) {
        err_msg = "GPIV_FREAD_HDF5_IMAGE: failing H5Dread";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
#endif
/*     msg_error = "H5Dread: no image data";
 *     GpivIo
 *     gpiv_io
 */
    status = H5Sclose(dataspace_id);
    status = H5Dclose(dataset_id);

/*
 * image #2
 */
    if (image_par->x_corr) {
        dataset_id = H5Dopen(group_id, "#2");
        dataspace_id = H5Dget_space(dataset_id);
#ifdef HD5_IMAGE_INT
        status = H5Dread(dataset_id, H5T_NATIVE_INT, H5S_ALL, H5S_ALL, 
                         H5P_DEFAULT, img2[0]);
#else
        status = H5Dread(dataset_id, H5T_NATIVE_USHORT, H5S_ALL, H5S_ALL, 
                         H5P_DEFAULT, img2[0]);
#endif
        status = H5Sclose(dataspace_id);
        status = H5Dclose(dataset_id);
    } else {
	img2 = img1;
    }



    H5Gclose (group_id);
    status = H5Fclose(file_id);
    return err_msg;
}



char *
gpiv_fwrite_hdf5_image(char *fname, 
#ifdef HD5_IMAGE_INT
                       int **img1, 
                       int **img2, 
#else
                       guint16 **img1, 
                       guint16 **img2, 
#endif
                       GpivImagePar * image_par
                       )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes IMAGE data to file in hdf version 5 format
 *
 * PROTOTYPE LOCATATION:
 *      io.h
 *
 * INPUTS:
 *      fname:          complete filename
 *      img:            2-dimensional matrix containing nrows x ncoloms pixel 
 *                      values
 *      image_par:      image parameters
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int i, j, k, line_nr=0, image_white_is_zero = 0;
    float *point_x_hdf = NULL;
/*
 * HDF file, dataset,-space identifier
 */
    hid_t       file_id, group_id, dataset_id, dataspace_id, 
        attribute_id, atype; 
    hsize_t     dims[2];
    herr_t      status;

    assert(img1[0] != NULL);
    if (image_par->x_corr) assert(img2[0] != NULL);


    if ((i = H5Fis_hdf5(fname)) == 0) {
        err_msg = "GPIV_FWRITE_HDF5_IMAGE: not an hdf5 file";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    file_id = H5Fopen(fname, H5F_ACC_RDWR, H5P_DEFAULT);
    group_id = H5Gopen (file_id, "IMAGE");
/*     H5Gset_comment (group_id, ".", RCSID); */

/*
 * Hdf required image specifications
 */
    dims[0] = 1;

    dataspace_id = H5Screate(H5S_SCALAR);
    atype = H5Tcopy(H5T_C_S1);
    H5Tset_size(atype, 5);
    attribute_id = H5Acreate(group_id, "CLASS", atype, 
                             dataspace_id, H5P_DEFAULT);
    status = H5Awrite(attribute_id, atype, 
                      "IMAGE"); 
    status = H5Aclose(attribute_id); 
    status = H5Sclose(dataspace_id);


    dataspace_id = H5Screate(H5S_SCALAR);
    atype = H5Tcopy(H5T_C_S1);
    H5Tset_size(atype, 15);
    attribute_id = H5Acreate(group_id, "SUBCLASS", atype, 
                             dataspace_id, H5P_DEFAULT);
    status = H5Awrite(attribute_id, atype, 
                      "IMAGE_GRAYSCALE"); 
    status = H5Aclose(attribute_id); 
    status = H5Sclose(dataspace_id);


    dataspace_id = H5Screate_simple(1, dims, NULL);
    attribute_id = H5Acreate(group_id, "IMAGE_WHITE_IS_ZERO", 
                             H5T_NATIVE_USHORT, dataspace_id, H5P_DEFAULT);
    status = H5Awrite(attribute_id, H5T_NATIVE_USHORT, &image_white_is_zero); 
    status = H5Aclose(attribute_id); 
    status = H5Sclose(dataspace_id);


    dataspace_id = H5Screate(H5S_SCALAR);
    atype = H5Tcopy(H5T_C_S1);
    H5Tset_size(atype, 3);
    attribute_id = H5Acreate(group_id, "IMAGE_VERSION", atype, 
                             dataspace_id, H5P_DEFAULT);
    status = H5Awrite(attribute_id, atype, 
                      "1.2"); 
    status = H5Aclose(attribute_id); 
    status = H5Sclose(dataspace_id);


 /*
 * Create the data space.
 */
   dims[0] = image_par->ncolumns; 
   dims[1] = image_par->nrows;
   dataspace_id = H5Screate_simple(2, dims, NULL);


/* 
 * image #1
 */
#ifdef HD5_IMAGE_INT
   dataset_id = H5Dcreate(group_id, "#1", H5T_NATIVE_INT, 
			  dataspace_id, H5P_DEFAULT);
   status = H5Dwrite(dataset_id, H5T_NATIVE_INT, H5S_ALL, H5S_ALL, 
		     H5P_DEFAULT, &img1[0][0]);
#else
   dataset_id = H5Dcreate(group_id, "#1", H5T_NATIVE_USHORT, 
			  dataspace_id, H5P_DEFAULT);
   status = H5Dwrite(dataset_id, H5T_NATIVE_USHORT, H5S_ALL, H5S_ALL, 
		     H5P_DEFAULT, &img1[0][0]);
/*    dataset_id = H5Dcreate(group_id, "#1", H5T_NATIVE_UCHAR,  */
/* 			  dataspace_id, H5P_DEFAULT); */
/*    status = H5Dwrite(dataset_id, H5T_NATIVE_UCHAR, H5S_ALL, H5S_ALL,  */
/* 		     H5P_DEFAULT, &img1[0][0]); */
#endif
   status = H5Dclose(dataset_id);


/* 
 * image #2
 */
   if (image_par->x_corr) {
#ifdef HD5_IMAGE_INT
       dataset_id = H5Dcreate(group_id, "#2", H5T_NATIVE_INT, 
                              dataspace_id, H5P_DEFAULT);
       status = H5Dwrite(dataset_id, H5T_NATIVE_INT, H5S_ALL, H5S_ALL, 
                         H5P_DEFAULT, &img2[0][0]);
#else
       dataset_id = H5Dcreate(group_id, "#2", H5T_NATIVE_USHORT, 
                              dataspace_id, H5P_DEFAULT);
       status = H5Dwrite(dataset_id, H5T_NATIVE_USHORT, H5S_ALL, H5S_ALL, 
                         H5P_DEFAULT, &img2[0][0]);
#endif
       status = H5Dclose(dataset_id);
   }


   status = H5Sclose(dataspace_id);
   status = H5Gclose (group_id);
   status = H5Fclose(file_id);
   return err_msg;
}



char *
gpiv_fread_hdf5_parameters (char *fname,
                            const char *par_key,
                            void * pstruct
                            )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads parameters from hdf5 data file
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      group_id:       hdf group identity
 *      par_key:        key for apropiate parameter
 *
 * OUTPUTS:
 *      pstruct:        parameter structure
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;    

    if (strcmp(par_key, GPIV_EVAL_PAR_KEY) == 0) {
        gpiv_piv_fread_hdf5_parameters(fname, pstruct);
    } else {
        err_msg = "GPIV_SCAN_PARAMETERS: called with unexisting key";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

  return err_msg;
}



char *
gpiv_fwrite_hdf5_parameters (char *fname,
                             const char *par_key,
                             void * pstruct
                             )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Writes parameters to hdf5 data file
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      group_id:       hdf group identity
 *      par_key:        key for apropiate parameter
 *      pstruct:        parameter structure
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{    
    char *err_msg = NULL;
    if (strcmp(par_key, GPIV_EVAL_PAR_KEY) == 0) {
        gpiv_piv_fwrite_hdf5_parameters(fname, pstruct);
    } else {
        err_msg = "GPIV_FWRITE_HDF5_PARAMETERS: called with unexisting key";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    return err_msg;
}


char *
gpiv_fcount_hdf5_data(char *fname,
                      int * ny,
                      int * nx
                      )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Reads array lengths of 2-dimensional data
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *     fname:          complete filename
 *
 * OUTPUTS:
 *     ny:             length of first array index
 *     nx:             length of second array index
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int i, j, rank_x, rank_y, rank_vx, rank_vy;
    float *point_x_hdf = NULL, *point_y_hdf = NULL;
/*
 * HDF declarations
 */
    hid_t       file_id, group_id, dataset_id, dataspace_id; 
    hsize_t     dims[2];
    herr_t      status;

    if ((i = H5Fis_hdf5(fname)) == 0) {
        err_msg = "GPIV_FCOUNT_HDF5_PIVDATA: not an hdf5 file";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    file_id = H5Fopen(fname, H5F_ACC_RDONLY, H5P_DEFAULT);
    group_id = H5Gopen (file_id, "POSITIONS");

/*
 * getting nx
 */
    dataset_id = H5Dopen(group_id, "point_x");
    dataspace_id = H5Dget_space(dataset_id);
    rank_x = H5Sget_simple_extent_ndims (dataspace_id);
    if (rank_x != 1) {
        err_msg = "GPIV_FCOUNT_HDF5_PIVDATA: rank_x != 1";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    H5Sget_simple_extent_dims(dataspace_id, dims, NULL);
    *nx = dims[0];
    status = H5Dclose(dataset_id);
    status = H5Sclose(dataspace_id);
	
/*
 * getting ny
 */
    dataset_id = H5Dopen(group_id, "point_y");
    dataspace_id = H5Dget_space(dataset_id);
    rank_y = H5Sget_simple_extent_ndims (dataspace_id);
    if (rank_y != 1) {
        err_msg = "GPIV_FCOUNT_HDF5_PIVDATA: rank_y != 1";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    H5Sget_simple_extent_dims(dataspace_id, dims, NULL);
    *ny = dims[0];
    status = H5Dclose(dataset_id);
    status = H5Sclose(dataspace_id);


    H5Gclose (group_id);
    status = H5Fclose(file_id); 
    return err_msg;
}



char *
gpiv_fread_hdf5_piv_position(char *fname, 
                             GpivPivData *piv_data,
                             char *RCSID,
                             int *scale
                             )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Reads position data from hdf5 data file into GpivPivData struct
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *     fname:          complete filename
 *
 * OUTPUTS:
 *     piv_data:        parameter structure
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int i, j, rank_x, rank_y, rank_vx, rank_vy;
    float *point_x_hdf = NULL, *point_y_hdf = NULL;
/*
 * HDF declarations
 */
    hid_t       file_id, group_id, attribute_id, dataset_id, 
        dataspace_id; 
    hsize_t     dims[2];
    herr_t      status;


    assert(piv_data->point_x != NULL);
    assert(piv_data->point_y != NULL);


    if ((i = H5Fis_hdf5(fname)) == 0)  {
        err_msg = "GPIV_FREAD_HDF5_PIV_POSITION: not an hdf5 file";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    file_id = H5Fopen(fname, H5F_ACC_RDONLY, H5P_DEFAULT);
/*
 * POSITION group with RCSID comment and scale attribute
 */
    group_id = H5Gopen (file_id, "POSITIONS");
    H5Gget_comment (group_id, ".", GPIV_MAX_CHARS, RCSID);
    attribute_id = H5Aopen_name(group_id, "scale");
    H5Aread(attribute_id, H5T_NATIVE_INT, scale); 
    H5Aclose(attribute_id); 

/*
 * point_x
 */
    dataset_id = H5Dopen(group_id, "point_x");
    dataspace_id = H5Dget_space(dataset_id);
    rank_x = H5Sget_simple_extent_ndims (dataspace_id);
    if (rank_x != 1) {
        err_msg = "GPIV_FREAD_HDF5_PIV_POSITION: rank_x != 1";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    H5Sget_simple_extent_dims(dataspace_id, dims, NULL);
    piv_data->nx = dims[0];
    point_x_hdf = gpiv_vector(piv_data->nx) ;
    status = H5Dread(dataset_id, H5T_NATIVE_FLOAT, H5S_ALL, H5S_ALL, 
                     H5P_DEFAULT, point_x_hdf);
    status = H5Dclose(dataset_id);
    status = H5Sclose(dataspace_id);
	

/*
 * point_y
 */
    dataset_id = H5Dopen(group_id, "point_y");
    dataspace_id = H5Dget_space(dataset_id);
    rank_y = H5Sget_simple_extent_ndims (dataspace_id);
    if (rank_y != 1) {
        err_msg = "GPIV_FREAD_HDF5_PIV_POSITION: rank_y != 1";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    H5Sget_simple_extent_dims(dataspace_id, dims, NULL);
    piv_data->ny = dims[0];
    point_y_hdf = gpiv_vector(piv_data->ny) ;
    status = H5Dread(dataset_id, H5T_NATIVE_FLOAT, H5S_ALL, H5S_ALL, 
                     H5P_DEFAULT, point_y_hdf);
    status = H5Dclose(dataset_id);
    status = H5Sclose(dataspace_id);

/*
 * filling GpivPivData structure with point_x an point_y
 */
    for (i = 0; i < piv_data->ny; i++) {
            for (j = 0; j < piv_data->nx; j++) {
                piv_data->point_x[i][j] = point_x_hdf[j];
                piv_data->point_y[i][j] = point_y_hdf[i];
            }
    }
    gpiv_free_vector(point_x_hdf);
    gpiv_free_vector(point_y_hdf);


    H5Gclose (group_id);
    status = H5Fclose(file_id);
    return err_msg;
}


char *
gpiv_fwrite_hdf5_piv_position(char *fname, 
                              GpivPivData *piv_data,
                              char *RCSID,
                              int *scale 
                              )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Writess position data to hdf5 data file into GpivPivData struct
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *     fname:          complete filename
 *
 * OUTPUTS:
 *     piv:            parameter structure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int i, j, rank_x, rank_y, rank_vx, rank_vy;
    float *point_x_hdf = NULL, *point_y_hdf = NULL;

/*
 * HDF declarations
 */
    hid_t       file_id, group_id, attribute_id, dataset_id, 
        dataspace_id, str_MC; 
    hsize_t     dims[2], dims_attrib[1];
    herr_t      status;


    assert(piv_data->point_x != NULL);
    assert(piv_data->point_y != NULL);


    if ((i = H5Fis_hdf5(fname)) == 0) {
        err_msg = "GPIV_FWRITE_HDF5_PIV_POSITION: not an hdf5 file";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    file_id = H5Fopen(fname, H5F_ACC_RDWR, H5P_DEFAULT);

/*
 * POSITION group with RCSID comment and scale attribute
 */
    group_id = H5Gopen (file_id, "POSITIONS");
    H5Gset_comment (group_id, ".", RCSID);
    dims_attrib[0] = 1;
    dataspace_id = H5Screate_simple(1, dims_attrib, NULL);
    attribute_id = H5Acreate(group_id, "scale", H5T_NATIVE_INT, 
                             dataspace_id, H5P_DEFAULT);
    H5Awrite(attribute_id, H5T_NATIVE_INT, scale); 
    H5Aclose(attribute_id); 

/*
 * data dimension
 */
    str_MC = H5Tcopy (H5T_C_S1);
    H5Tset_size (str_MC, GPIV_MAX_CHARS);
    dataspace_id = H5Screate_simple(1, dims_attrib, NULL);
    attribute_id = H5Acreate(group_id, "dimension", str_MC, 
                             dataspace_id, H5P_DEFAULT);
    if (*scale) {
        H5Awrite(attribute_id, str_MC, "Positions in m");
    } else {
        H5Awrite(attribute_id, str_MC, "Positions in pixels");
    }

    H5Aclose(attribute_id); 

/*
 * point_x
 */
    point_x_hdf = gpiv_vector(piv_data->nx);
    for (i = 0; i < piv_data->nx; i++) point_x_hdf[i] = piv_data->point_x[0][i];
    dims[0] = piv_data->nx; 
    dataspace_id = H5Screate_simple(1, dims, NULL);
    dataset_id = H5Dcreate(group_id, "point_x", H5T_NATIVE_FLOAT, dataspace_id, 
                           H5P_DEFAULT);
    status = H5Dwrite(dataset_id, H5T_NATIVE_FLOAT, H5S_ALL, H5S_ALL, 
                      H5P_DEFAULT, point_x_hdf);
    status = H5Dclose(dataset_id);
    status = H5Sclose(dataspace_id);
    gpiv_free_vector(point_x_hdf);

/*
 * point_y
 */
    point_y_hdf = gpiv_vector(piv_data->ny);
    for (i = 0; i < piv_data->ny; i++) point_y_hdf[i] = piv_data->point_y[i][0];
    dims[0] = piv_data->ny; 
    dataspace_id = H5Screate_simple(1, dims, NULL);
    dataset_id = H5Dcreate(group_id, "point_y", H5T_NATIVE_FLOAT, 
                           dataspace_id, H5P_DEFAULT);
    status = H5Dwrite(dataset_id, H5T_NATIVE_FLOAT, H5S_ALL, H5S_ALL, 
                      H5P_DEFAULT, point_y_hdf);
    status = H5Dclose(dataset_id);
    status = H5Sclose(dataspace_id);
    gpiv_free_vector(point_y_hdf);

/* 
 *Closing group and file
 */
    status = H5Gclose (group_id);
    status = H5Fclose(file_id); 

    return err_msg;
}



char *
gpiv_fread_hdf5_pivdata (char *fname, 
                         GpivPivData *piv_data, 
                         char *DATA_KEY,
                         char *RCSID,
                         int *scale
                         )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Reads piv data from hdf5 data file
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *     fname:          complete filename
 *
 * OUTPUTS:
 *     piv_data:        struct piv_data containing all variables in a datafile
 *                      piv.nx and piv.ny have to be known.
 *     comment_line:    lines contain comment (staring with '#' sign at first
 *                      column)
 *     ncomment_lines:  number of comment lines
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{    
    char * err_msg = NULL;
    int i, j, rank_x, rank_y, rank_vx, rank_vy;
    float *point_x_hdf = NULL, *point_y_hdf = NULL;
/*
 * HDF declarations
 */
    hid_t       file_id, group_id, attribute_id, dataset_id, 
        dataspace_id; 
    hsize_t     dims[2];
    herr_t      status;


/*     assert(piv_data->nx == 0); */
/*     assert(piv_data->nx == 0); */
    assert(piv_data->point_x != NULL);
    assert(piv_data->point_y != NULL);
    assert(piv_data->dx != NULL);
    assert(piv_data->dy != NULL);
    assert(piv_data->snr != NULL);
    assert(piv_data->peak_no != NULL);


    if ((i = H5Fis_hdf5(fname)) == 0)  {
        err_msg = "GPIV_FREAD_HDF5_PIVDATA: not an hdf5 file";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    file_id = H5Fopen(fname, H5F_ACC_RDONLY, H5P_DEFAULT);

/*
 * PIV group with RCSID comment and scale attribute
 */
    group_id = H5Gopen (file_id, DATA_KEY);
    H5Gget_comment (group_id, ".", GPIV_MAX_CHARS, RCSID);
    attribute_id = H5Aopen_name(group_id, "scale");
    H5Aread(attribute_id, H5T_NATIVE_INT, scale); 
    H5Aclose(attribute_id); 


/*
 * dx
 */
    dataset_id = H5Dopen(group_id, "dx");
    dataspace_id = H5Dget_space(dataset_id);
    status = H5Dread(dataset_id, H5T_NATIVE_FLOAT, H5S_ALL, H5S_ALL, 
                     H5P_DEFAULT, piv_data->dx[0]);
    status = H5Dclose(dataset_id);
    status = H5Sclose(dataspace_id);

/*
 * dy
 */
    dataset_id = H5Dopen(group_id, "dy");
    dataspace_id = H5Dget_space(dataset_id);
    status = H5Dread(dataset_id, H5T_NATIVE_FLOAT, H5S_ALL, H5S_ALL, 
                     H5P_DEFAULT, piv_data->dy[0]);
    status = H5Dclose(dataset_id);
    status = H5Sclose(dataspace_id);

/*
 * snr
 */
    dataset_id = H5Dopen(group_id, "snr");
    dataspace_id = H5Dget_space(dataset_id);
    status = H5Dread(dataset_id, H5T_NATIVE_FLOAT, H5S_ALL, H5S_ALL, 
                     H5P_DEFAULT, piv_data->snr[0]);
    status = H5Dclose(dataset_id);
    status = H5Sclose(dataspace_id);

/*
 * peak_no
 */
    dataset_id = H5Dopen(group_id, "peak_no");
    dataspace_id = H5Dget_space(dataset_id);
    status = H5Dread(dataset_id, H5T_NATIVE_INT, H5S_ALL, H5S_ALL, 
                     H5P_DEFAULT, piv_data->peak_no[0]);
    status = H5Dclose(dataset_id);
    status = H5Sclose(dataspace_id);



    status = H5Gclose (group_id);
    status = H5Fclose(file_id); 
    return err_msg;
}



char *
gpiv_fwrite_hdf5_pivdata(char *fname, 
                         GpivPivData *piv_data, 
                         char *DATA_KEY,
                         char *RCSID,
                         int *scale 
                         )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Writes PIV data to file in hdf version 5 format
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *     fname:          complete filename
 *     piv:            struct piv_data containing all variables in a datafile
 *     DATA_KEY:       location of the data in the DATA group in the hdf file
 *     RCSID:          identifier of program that generated datafile
 *     scale:          parameter to write piv.point_x and piv.point_y as int 
 *                     or as float
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int i, j, k, line_nr = 0;
    float *point_x_hdf = NULL,  *point_y_hdf = NULL;
    const  char *string/* [GPIV_MAX_CHARS] */ = "RcsId";
    char *dir /*=  "/point_x" */;
    int point = 5;

/*
 * HDF file, dataset,-space identifier
 */
    hid_t       file_id, dataset_id, dataspace_id, 
        group_id, attribute_id,
        str_MC; 
    hsize_t     dims[2], dims_attrib[1];
    herr_t      status;	      


    assert(piv_data->point_x != NULL);
    assert(piv_data->point_y != NULL);
    assert(piv_data->dx != NULL);
    assert(piv_data->dy != NULL);
    assert(piv_data->snr != NULL);
    assert(piv_data->peak_no != NULL);

    if ((i = H5Fis_hdf5(fname)) == 0) {
        err_msg = "GPIV_FWRITE_HDF5_PIVDATA: not an hdf5 file";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
     file_id = H5Fopen(fname, H5F_ACC_RDWR, H5P_DEFAULT);

/*
 * PIV group with RCSID comment and scale attribute
 */
    group_id = H5Gopen (file_id, DATA_KEY);
    H5Gset_comment (group_id, ".", RCSID);
    dims_attrib[0] = 1;
    dataspace_id = H5Screate_simple(1, dims_attrib, NULL);
    attribute_id = H5Acreate(group_id, "scale", H5T_NATIVE_INT, 
                             dataspace_id, H5P_DEFAULT);
    H5Awrite(attribute_id, H5T_NATIVE_INT, scale); 
    H5Aclose(attribute_id); 
    H5Sclose(dataspace_id);

/*
 * data dimension
 */
    str_MC = H5Tcopy (H5T_C_S1);
    H5Tset_size (str_MC, GPIV_MAX_CHARS);
    dataspace_id = H5Screate_simple(1, dims_attrib, NULL);
    attribute_id = H5Acreate(group_id, "dimension", str_MC, 
                             dataspace_id, H5P_DEFAULT);
    if (*scale) {
        H5Awrite(attribute_id, str_MC, "PIV estimators in m/s");
    } else {
        H5Awrite(attribute_id, str_MC, "PIV estimators in pixels");
    }
    H5Aclose(attribute_id); 
    H5Sclose(dataspace_id);


    dims[0] = piv_data->ny; 
    dims[1] = piv_data->nx;

/* 
 * dx
 */
   dataspace_id = H5Screate_simple(2, dims, NULL);
   dataset_id = H5Dcreate(group_id, "dx", H5T_NATIVE_FLOAT, 
                          dataspace_id, H5P_DEFAULT);
   status = H5Dwrite(dataset_id, H5T_NATIVE_FLOAT, H5S_ALL, H5S_ALL, 
		     H5P_DEFAULT, &piv_data->dx[0][0]);
   status = H5Dclose(dataset_id);
   status = H5Sclose(dataspace_id);

/*
 * dy
 */
   dataspace_id = H5Screate_simple(2, dims, NULL);
   dataset_id = H5Dcreate(group_id, "dy", H5T_NATIVE_FLOAT, 
                              dataspace_id, H5P_DEFAULT);
   status = H5Dwrite(dataset_id, H5T_NATIVE_FLOAT, H5S_ALL, H5S_ALL, 
		     H5P_DEFAULT,&piv_data->dy[0][0]);
   status = H5Dclose(dataset_id);
   status = H5Sclose(dataspace_id);

/*
 * snr
 */
   dataspace_id = H5Screate_simple(2, dims, NULL);
   dataset_id = H5Dcreate(group_id, "snr", H5T_NATIVE_FLOAT, 
			  dataspace_id, H5P_DEFAULT);
   status = H5Dwrite(dataset_id, H5T_NATIVE_FLOAT, H5S_ALL, H5S_ALL, 
		     H5P_DEFAULT,&piv_data->snr[0][0]);
   status = H5Dclose(dataset_id);
   status = H5Sclose(dataspace_id);

/*
 * peak_no
 */
   dataspace_id = H5Screate_simple(2, dims, NULL);
   dataset_id = H5Dcreate(group_id, "peak_no", H5T_NATIVE_INT, dataspace_id, 
			  H5P_DEFAULT);
   status = H5Dwrite(dataset_id, H5T_NATIVE_INT, H5S_ALL, H5S_ALL, H5P_DEFAULT,
		     &piv_data->peak_no[0][0]);
   status = H5Dclose(dataset_id);
   status = H5Sclose(dataspace_id);


   status = H5Gclose (group_id);
   status = H5Fclose(file_id); 
   return err_msg;
}


char *
gpiv_fread_hdf5_sc_position(char *fname, 
                             GpivScalarData *scdata
                             )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *    Reads position data from hdf5 data file into GpivScalarData struct
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *     fname:          complete filename
 *
 * OUTPUTS:
 *     scdata:         parameter structure
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL; 
    int i, j, rank_x, rank_y, rank_vx, rank_vy;
    float *point_x_hdf = NULL, *point_y_hdf = NULL;
/*
 * HDF declarations
 */
    hid_t       file_id, group_id, attribute_id, dataset_id, dataspace_id; 
    hsize_t     dims[2];
    herr_t      status;


    assert(scdata->point_x != NULL);
    assert(scdata->point_y != NULL);


    if ((i = H5Fis_hdf5(fname)) == 0) {
        err_msg = "GPIV_FREAD_HDF5_SC_POSITION: not an hdf5 file";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    file_id = H5Fopen(fname, H5F_ACC_RDONLY, H5P_DEFAULT);
    group_id = H5Gopen (file_id, "POSITIONS");

/*
 * point_x
 */
    dataset_id = H5Dopen(group_id, "point_x");
    dataspace_id = H5Dget_space(dataset_id);
    rank_x = H5Sget_simple_extent_ndims (dataspace_id);
    if (rank_x != 1) {
        err_msg = "GPIV_FREAD_HDF5_SC_POSITION: rank_x != 1";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    H5Sget_simple_extent_dims(dataspace_id, dims, NULL);
    scdata->nx = dims[0];
    point_x_hdf = gpiv_vector(scdata->nx) ;
    status = H5Dread(dataset_id, H5T_NATIVE_FLOAT, H5S_ALL, H5S_ALL, 
                     H5P_DEFAULT, point_x_hdf);
    status = H5Dclose(dataset_id);
    status = H5Sclose(dataspace_id);
	

/*
 * point_y
 */
    dataset_id = H5Dopen(group_id, "point_y");
    dataspace_id = H5Dget_space(dataset_id);
    rank_y = H5Sget_simple_extent_ndims (dataspace_id);
    if (rank_y != 1) {
        err_msg = "GPIV_FREAD_HDF5_SC_POSITION: rank_y != 1";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    H5Sget_simple_extent_dims(dataspace_id, dims, NULL);
    scdata->ny = dims[0];
    point_y_hdf = gpiv_vector(scdata->ny) ;
    status = H5Dread(dataset_id, H5T_NATIVE_FLOAT, H5S_ALL, H5S_ALL, 
                     H5P_DEFAULT, point_y_hdf);
    status = H5Dclose(dataset_id);
    status = H5Sclose(dataspace_id);

/*
 * filling GpivPivData structure with point_x an point_y
 */
    for (i = 0; i < scdata->ny; i++) {
            for (j = 0; j < scdata->nx; j++) {
                scdata->point_x[i][j] = point_x_hdf[j];
                scdata->point_y[i][j] = point_y_hdf[i];
            }
    }
    gpiv_free_vector(point_x_hdf);
    gpiv_free_vector(point_y_hdf);


    H5Gclose (group_id);
    status = H5Fclose(file_id);
    return err_msg;
}


char *
gpiv_fwrite_hdf5_sc_position(char *fname, 
                             GpivScalarData *scdata,
                              char *RCSID,
                              int *scale
                             )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *    Writess position data to hdf5 data file into GpivScalarData struct
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      fname:           complete filename
 *
 * OUTPUTS:
 *       scdata:         parameter structure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int i, j, rank_x, rank_y, rank_vx, rank_vy;
    float *point_x_hdf = NULL, *point_y_hdf = NULL;
/*
 * HDF declarations
 */
    hid_t       file_id, group_id, attribute_id, 
        dataset_id, dataspace_id, str_MC; 
    hsize_t     dims[2], dims_attrib[1];
    herr_t      status;


    assert(scdata->point_x != NULL);
    assert(scdata->point_y != NULL);


    if ((i = H5Fis_hdf5(fname)) == 0) {
        err_msg = "GPIV_FWRITE_HDF5_SC_POSITION: not an hdf5 file";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    file_id = H5Fcreate(fname, H5F_ACC_TRUNC, H5P_DEFAULT, H5P_DEFAULT);
    group_id = H5Gcreate (file_id, "POSITIONS", 0);
    H5Gset_comment (group_id, ".", RCSID);
    dims_attrib[0] = 1;
    dataspace_id = H5Screate_simple(1, dims_attrib, NULL);
    attribute_id = H5Acreate(group_id, "scale", H5T_NATIVE_INT, 
                             dataspace_id, H5P_DEFAULT);
    H5Awrite(attribute_id, H5T_NATIVE_INT, scale); 
    H5Aclose(attribute_id); 


/*
 * data dimension
 */
    str_MC = H5Tcopy (H5T_C_S1);
    H5Tset_size (str_MC, GPIV_MAX_CHARS);
    dims_attrib[0] = 1;
    dataspace_id = H5Screate_simple(1, dims_attrib, NULL);
    attribute_id = H5Acreate(group_id, "dimension", str_MC, 
                             dataspace_id, H5P_DEFAULT);
    if (*scale) {
        H5Awrite(attribute_id, str_MC, "Positions in m");
    } else {
        H5Awrite(attribute_id, str_MC, "Positions in pixels");
    }

    H5Aclose(attribute_id); 

/*
 * point_x
 */
    point_x_hdf = gpiv_vector(scdata->nx);
    for (i = 0; i < scdata->nx; i++) point_x_hdf[i] = scdata->point_x[0][i];
    dims[0] = scdata->nx; 
    dataspace_id = H5Screate_simple(1, dims, NULL);
    dataset_id = H5Dcreate(group_id, "point_x", H5T_NATIVE_FLOAT, dataspace_id, 
                           H5P_DEFAULT);
    status = H5Dwrite(dataset_id, H5T_NATIVE_FLOAT, H5S_ALL, H5S_ALL, 
                      H5P_DEFAULT, point_x_hdf);
    status = H5Dclose(dataset_id);
    status = H5Sclose(dataspace_id);
    gpiv_free_vector(point_x_hdf);

/*
 * point_y
 */
    point_y_hdf = gpiv_vector(scdata->ny);
    for (i = 0; i < scdata->ny; i++) point_y_hdf[i] = scdata->point_y[i][0];
    dims[0] = scdata->ny; 
    dataspace_id = H5Screate_simple(1, dims, NULL);
    dataset_id = H5Dcreate(group_id, "point_y", H5T_NATIVE_FLOAT, 
                           dataspace_id, H5P_DEFAULT);
    status = H5Dwrite(dataset_id, H5T_NATIVE_FLOAT, H5S_ALL, H5S_ALL, 
                      H5P_DEFAULT, point_y_hdf);
    status = H5Dclose(dataset_id);
    status = H5Sclose(dataspace_id);
    gpiv_free_vector(point_y_hdf);


    status = H5Gclose (group_id);
/*     group_id = H5Gcreate (file_id, "DATA", 0); */
/*     status = H5Gclose (group_id); */
/*     group_id = H5Gcreate (file_id, "PARAMETERS", 0); */
/*     status = H5Gclose (group_id); */
    status = H5Fclose(file_id); 
    return err_msg;
}



char *
gpiv_fread_hdf5_scdata(char *fname, 
                       GpivScalarData *scalar_data,
                       char *DATA_KEY,
                       char *RCSID
                       )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Writes SCALAR data to a hdf5 data file
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *      fname:          complete filename
 *      DATA_KEY:       key to specify data type/origin (NORMAL_STRAIN, ...)
 *      RCSID:         identifier of program that generated the output datafile
 *
 * OUTPUTS:
 *     scalar_data
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int i, j, rank_x, rank_y, rank_vx, rank_vy;
    float *point_x_hdf = NULL, *point_y_hdf = NULL;
/*
 * HDF declarations
 */
    hid_t       file_id, group_data, group_id, attribute_id, dataset_id, 
        dataspace_id; 
    hsize_t     dims[2];
    herr_t      status;


  assert(scalar_data->point_x != NULL);
  assert(scalar_data->point_y != NULL);
  assert(scalar_data->scalar != NULL);
  assert(scalar_data->flag != NULL);

    if ((i = H5Fis_hdf5(fname)) == 0) {
        err_msg = "GPIV_FREAD_HDF5_SCDATA: not an hdf5 file";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    file_id = H5Fopen(fname, H5F_ACC_RDONLY, H5P_DEFAULT);
    group_data = H5Gopen (file_id, "SCALARS");
    group_id = H5Gopen (group_data, DATA_KEY);
    H5Gget_comment (group_id, ".", GPIV_MAX_CHARS, RCSID);

/*
 * scalar
 */
    dataset_id = H5Dopen(group_id, "scalar");
    dataspace_id = H5Dget_space(dataset_id);
    status = H5Dread(dataset_id, H5T_NATIVE_FLOAT, H5S_ALL, H5S_ALL, 
                     H5P_DEFAULT, scalar_data->scalar[0]);
    status = H5Dclose(dataset_id);
    status = H5Sclose(dataspace_id);

/*
 * peak_no
 */
    dataset_id = H5Dopen(group_id, "flag");
    dataspace_id = H5Dget_space(dataset_id);
    status = H5Dread(dataset_id, H5T_NATIVE_INT, H5S_ALL, H5S_ALL, 
                     H5P_DEFAULT, scalar_data->flag[0]);
    status = H5Dclose(dataset_id);
    status = H5Sclose(dataspace_id);


    H5Gclose (group_id);
    H5Gclose (group_data);
    status = H5Fclose(file_id);
    return err_msg;
}



char *
gpiv_fwrite_hdf5_scdata(char *fname, 
                        GpivScalarData * scalar_data,
                        char * DATA_KEY,
                        char *RCSID
                        )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Writes SCALAR data to file in hdf version 5 format
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *     scalar:         struct GpivScalarData containing all variables in a datafile
 *     DATA_KEY:       identifier of data type
 *     RCSID:          identifier of program that generated datafile
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int i, j, k, line_nr=0;
    float *point_x_hdf = NULL;
/*
 * HDF file, dataset,-space identifier
 */
    hid_t       file_id, group_data, group_id, dataset_id, dataspace_id; 
    hsize_t     dims[2];
    herr_t      status;
    
    assert(scalar_data->point_x != NULL);
    assert(scalar_data->point_y != NULL);
    assert(scalar_data->scalar != NULL);
    assert(scalar_data->flag != NULL);
    
    
    if ((i = H5Fis_hdf5(fname)) == 0) {
        err_msg = "GPIV_FWRITE_HDF5_SCDATA: not an hdf5 file";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    file_id = H5Fopen(fname, H5F_ACC_RDWR, H5P_DEFAULT);
    group_data = H5Gopen (file_id, "SCALARS");
    group_id = H5Gcreate (group_data, DATA_KEY, 0);
    H5Gset_comment (group_id, ".", RCSID);

/*
 * Create the data space.
 */
   dims[0] = scalar_data->ny; 
   dims[1] = scalar_data->nx;


/* 
 * scalar
 */

   dataspace_id = H5Screate_simple(2, dims, NULL);
   dataset_id = H5Dcreate(group_id, "scalar", H5T_NATIVE_FLOAT, 
			  dataspace_id, H5P_DEFAULT);
   status = H5Dwrite(dataset_id, H5T_NATIVE_FLOAT, H5S_ALL, H5S_ALL, 
		     H5P_DEFAULT, &scalar_data->scalar[0][0]);
   status = H5Dclose(dataset_id);
   status = H5Sclose(dataspace_id);

/*
 * flag
 */

   dataspace_id = H5Screate_simple(2, dims, NULL);
   dataset_id = H5Dcreate(group_id, "flag", H5T_NATIVE_INT, dataspace_id, 
			  H5P_DEFAULT);
   status = H5Dwrite(dataset_id, H5T_NATIVE_INT, H5S_ALL, H5S_ALL, H5P_DEFAULT,
		     &scalar_data->flag[0][0]);
   status = H5Dclose(dataset_id);
   status = H5Sclose(dataspace_id);


   status = H5Gclose (group_id);
   status = H5Gclose (group_data);
   status = H5Fclose(file_id);
   return err_msg;
}



char *
gpiv_fread_hdf5_histo(char *fname, 
                       GpivBinData *klass, 
                       char * DATA_KEY,
                       char *RCSID
                       )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Reads histogram data to ouput file in hdf version 5 format
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *     fname:          complete filename
 *     klass:          struct bin containing the values for each bin
 *     DATA_KEY:       identifier of data type (PEAK_LOCK, RESIDUS, ...)
 *     RCSID:          identifier of program that generated the output datafile
 *---------------------------------------------------------------------------*/
{
    char * err_msg = NULL;
    int i;

/*
 * HDF declarations
 */
    hid_t       file_id, group_data, group_id, attribute_id, dataset_id, 
        dataspace_id; 
    hsize_t     dims[2];
    herr_t      status;
    
    assert(klass->count != NULL);
    assert(klass->bound != NULL);
    assert(klass->centre != NULL);
  

    if ((i = H5Fis_hdf5(fname)) == 0)  {
        err_msg = "GPIV_FREAD_HDF5_HISTO: not an hdf5 file";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    file_id = H5Fopen(fname, H5F_ACC_RDONLY, H5P_DEFAULT);
    group_data = H5Gopen (file_id, "SCALARS");
    group_id = H5Gopen (group_data, DATA_KEY);
    H5Gget_comment (group_id, ".", GPIV_MAX_CHARS, RCSID);

/*
 * centre
 */
    dataset_id = H5Dopen(group_id, "centre");
    dataspace_id = H5Dget_space(dataset_id);
    status = H5Dread(dataset_id, H5T_NATIVE_FLOAT, H5S_ALL, H5S_ALL, 
                     H5P_DEFAULT, klass->centre);
    status = H5Dclose(dataset_id);
    status = H5Sclose(dataspace_id);

/*
 * count
 */
    dataset_id = H5Dopen(group_id, "count");
    dataspace_id = H5Dget_space(dataset_id);
    status = H5Dread(dataset_id, H5T_NATIVE_INT, H5S_ALL, H5S_ALL, 
                     H5P_DEFAULT, klass->count);
    status = H5Dclose(dataset_id);
    status = H5Sclose(dataspace_id);

    H5Gclose (group_id);
    H5Gclose (group_data);
    status = H5Fclose(file_id);
    return err_msg;
 }



char *
gpiv_fwrite_hdf5_histo(char *fname, 
                       GpivBinData *klass, 
                       char * DATA_KEY,
                       char *RCSID
                       )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Writes histogram data to ouput file in hdf version 5 format
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *     fname:          complete filename
 *     klass:          struct bin containing the values for each bin
 *     DATA_KEY:       identifier of data type (PEAK_LOCK, RESIDUS, ...)
 *     RCSID:          identifier of program that generated the output datafile
 *
 * RETURNS:
 *     character pointer to NULL on successfull operations or to error message
 *     on error
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int i;

/*
 * HDF declarations
 */
    hid_t       file_id, group_data, group_id, attribute_id, dataset_id, 
        dataspace_id; 
    hsize_t     dims[1];
    herr_t      status;
    
    assert(klass->count != NULL);
    assert(klass->bound != NULL);
    assert(klass->centre != NULL);
  

    if ((i = H5Fis_hdf5(fname)) == 0)  {
        err_msg = "GPIV_FWRITE_HDF5_HISTO: not an hdf5 file";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    file_id = H5Fopen(fname, H5F_ACC_RDWR, H5P_DEFAULT);
    group_data = H5Gopen (file_id, "SCALARS");
    group_id = H5Gopen (group_data, DATA_KEY);
    H5Gset_comment (group_id, ".", RCSID);

/*
 * Create the data space.
 */
   dims[0] = klass->nbins; 
/*
 * centre
 */
    dataspace_id = H5Screate_simple(1, dims, NULL);
    dataset_id = H5Dcreate(group_id, "centre", H5T_NATIVE_FLOAT, 
                           dataspace_id, H5P_DEFAULT);
    status = H5Dwrite(dataset_id, H5T_NATIVE_FLOAT, H5S_ALL, H5S_ALL, 
                     H5P_DEFAULT, &klass->centre[0]);
    status = H5Dclose(dataset_id);
    status = H5Sclose(dataspace_id);

/*
 * count
 */
    dataspace_id = H5Screate_simple(1, dims, NULL);
    dataset_id = H5Dcreate(group_id, "count", H5T_NATIVE_INT, 
                          dataspace_id, H5P_DEFAULT);
    status = H5Dread(dataset_id, H5T_NATIVE_INT, H5S_ALL, H5S_ALL, 
                     H5P_DEFAULT, &klass->count[0]);
    status = H5Dclose(dataset_id);
    status = H5Sclose(dataspace_id);

    H5Gclose (group_id);
    H5Gclose (group_data);
    status = H5Fclose(file_id);
    return err_msg;
}




char *
gpiv_fread_davis_image(char *fname,
                       guint16 **img1,
                       guint16 **img2, 
                       GpivImagePar *image_par
                       ) 
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads Davis formatted image, with ext .IMG, from file
 *
 * PROTOTYPE LOCATATION:
 *     io.h
 *
 * INPUTS:
 *     fname:          pointer to complete filename to be read
 *     image_par:      containing image dimensions and other image specs
 *
 * OUTPUTS:
 *     img1:            image data of first image
 *     img2:            image data of second image
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    FILE *fp;
    char *err_msg = NULL;
    gint i, j;

    assert(img1 != NULL);
    assert(img2 != NULL);

    if ((fp = fopen(fname, "rb")) == NULL) {
	err_msg = "GPIV_FREAD_DAVIS_IMAGE: Failure opening for input";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    fseek(fp, 256, SEEK_SET);
    fread(img1[0], sizeof(guint16) * image_par->ncolumns * image_par->nrows, 
          1, fp);
    fread(img2[0], sizeof(guint16) * image_par->ncolumns * image_par->nrows, 
          1, fp);


    fclose(fp);
    return err_msg;
}



/* #undef LIBNAME */
/* #undef ASCII_IMG_FMT */
/* #undef GPIV_PIV_FMT */
/* #undef GPIV_PIV_S_FMT */
/* #undef GPIV_SCALAR_FMT */
/* #undef GPIV_SCALAR_S_FMT */
/* #undef GPIV_MAX_CHARS */
/* #undef GPIV_MAX_LINES */
/* #undef EXT_DAT_OUT */
