/**
 * @file get-presence.c Queries the presence for a user.
 *
 * @Copyright (C) 2004-2006 Christian Hammond
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA
 */
#include <libgalago/galago.h>
#include <stdio.h>
#include <string.h>

static const char *
get_name_from_status_type(GalagoStatusType type)
{
	switch (type)
	{
		case GALAGO_STATUS_UNSET:         return "Unset";
		case GALAGO_STATUS_OFFLINE:       return "Offline";
		case GALAGO_STATUS_AVAILABLE:     return "Available";
		case GALAGO_STATUS_AWAY:          return "Away";
		case GALAGO_STATUS_EXTENDED_AWAY: return "Extended Away";
		case GALAGO_STATUS_HIDDEN:        return "Hidden";
		default:
			return NULL;
	};
}

int
main(int argc, char **argv)
{
	GList *l;
	char *username, *service_id;
	GalagoPresence *presence;
	GalagoStatus *status;
	GalagoAccount *account;
	GalagoService *service;
	GalagoPerson *person;

	if (argc != 3)
	{
		printf("Usage: get-presence service-id username\n");
		exit(1);
	}

	service_id = argv[1];
	username   = argv[2];

	galago_init("get-presence", GALAGO_INIT_CLIENT);

	service = galago_get_service(service_id, GALAGO_REMOTE, TRUE);

	if (service == NULL)
	{
		printf("Unknown service %s\n", service_id);
		exit(1);
	}

	account = galago_service_get_account(service, username, TRUE);

	if (account == NULL)
	{
		printf("Unknown account %s on service %s\n", username, service_id);
		exit(1);
	}

	presence = galago_account_get_presence(account, TRUE);

	person   = galago_account_get_person(account);

	printf("Service:          %s\n", galago_service_get_name(service));
	printf("Username:         %s\n", galago_account_get_username(account));
	printf("Display name:     %s\n", galago_account_get_display_name(account));

	if (presence == NULL)
	{
		printf("No presence\n");
	}
	else
	{
		gboolean idle = galago_presence_is_idle(presence);

		printf("Idle:             %s\n", idle ? "yes" : "no");

		if (idle)
		{
		printf("Idle Start Time:  %d\n",
				   (int)galago_presence_get_idle_start_time(presence));
			printf("Idle Time Length: %d\n",
				   (int)galago_presence_get_idle_time(presence));
		}

		printf("Status(es):\n");

		for (l = galago_presence_get_statuses(presence);
			 l != NULL;
			 l = l->next)
		{
			GList *attrs, *l2;

			status = (GalagoStatus *)l->data;

			printf("\tType:    %s\n",
				   get_name_from_status_type(galago_status_get_primitive(status)));
			printf("\tID:      %s\n", galago_status_get_id(status));
			printf("\tName:    %s\n", galago_status_get_name(status));

			attrs = galago_object_get_attributes(GALAGO_OBJECT(status));

			if (attrs != NULL)
			{
				printf("\tAttributes:\n");

				for (l2 = attrs; l2 != NULL; l2 = l2->next)
				{
					GalagoKeyValue *key_value = (GalagoKeyValue *)l2->data;

					if (G_VALUE_HOLDS_INT(key_value->value))
					{
						printf("\t\t%s: %d\n",
							   key_value->key,
							   g_value_get_int(key_value->value));
					}
					else if (G_VALUE_HOLDS_STRING(key_value->value))
					{
						printf("\t\t%s: %s\n",
							   key_value->key,
							   g_value_get_string(key_value->value));
					}
					else if (G_VALUE_HOLDS_DOUBLE(key_value->value))
					{
						printf("\t\t%s: %f\n",
							   key_value->key,
							   g_value_get_double(key_value->value));
					}
					else if (G_VALUE_HOLDS_BOOLEAN(key_value->value))
					{
						printf("\t\t%s: %s\n",
							   key_value->key,
							   g_value_get_boolean(key_value->value)
							   ? "true" : "false");
					}
				}
			}
		}
	}

	return 0;
}
