/**
 * @file check-libgalago.c Unit tests
 *
 * @Copyright (C) 2004-2006 Christian Hammond
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA
 */
#include <libgalago/galago.h>
#include <check.h>
#include <string.h>
#include <stdio.h>
#include <stdarg.h>

static GalagoPerson *
make_dummy_person(void)
{
	return galago_create_person("dummy-person");
}

static GalagoPerson *
make_dummy_person_with_id(const char *id)
{
	return galago_create_person(id);
}

static GalagoService *
make_dummy_service(void)
{
	return galago_create_service("dummy-service", "Dummy Service", 0);
}

static GalagoAccount *
make_dummy_account(void)
{
	GalagoPerson *person = make_dummy_person();
	GalagoService *service = make_dummy_service();

	return galago_service_create_account(service, person, "dummy-account");
}

static GalagoImage *
make_dummy_image(void)
{
	return galago_image_new_from_file("avatar.png");
}

static GalagoPresence *
make_dummy_presence(void)
{
	return galago_account_create_presence(make_dummy_account());
}

static GalagoStatus *
make_dummy_status(void)
{
	return galago_status_new(GALAGO_STATUS_AWAY, "away", "Away", TRUE);
}

/**************************************************************************
 * libgalago tests
 **************************************************************************/
START_TEST(test_init_reinit)
{
	fail_unless(galago_init("check-libgalago", GALAGO_INIT_CLIENT),
				"Unable to initialize libgalago");

	galago_uninit();

	fail_unless(galago_init("check-libgalago", GALAGO_INIT_CLIENT),
				"Unable to re-initialize libgalago");
}
END_TEST

/*
 * Ticket #36: Uninitting and re-initting after a GalagoService was
 *             created causes D-BUS issues
 */
START_TEST(test_init_reinit_with_service)
{
	fail_unless(galago_init("check-libgalago", GALAGO_INIT_FEED),
				"Unable to initialize libgalago");
	galago_create_service("dummy-service-1", "Dummy Service 1", 0);
	galago_uninit();

	fail_unless(galago_init("check-libgalago", GALAGO_INIT_FEED),
				"Unable to re-initialize libgalago");
	galago_uninit();
}
END_TEST


/**************************************************************************
 * Test object creation
 **************************************************************************/
START_TEST(test_create_person)
{
	GalagoPerson *person;

	galago_init("check-libgalago", GALAGO_INIT_FEED);

	person = make_dummy_person();
	fail_unless(person != NULL, "Unable to create person");
	fail_unless(galago_object_get_dbus_path(GALAGO_OBJECT(person)) != NULL,
				"Object path not set on the new person");

	person = make_dummy_person_with_id(NULL);
	fail_unless(person != NULL, "Unable to create ID-less person");
	fail_unless(galago_object_get_dbus_path(GALAGO_OBJECT(person)) != NULL,
				"Object path not set on the new ID-less person");
}
END_TEST

START_TEST(test_create_service)
{
	GalagoService *service;

	galago_init("check-libgalago", GALAGO_INIT_FEED);

	service = make_dummy_service();
	fail_unless(service != NULL, "Unable to create service");
	fail_unless(galago_object_get_dbus_path(GALAGO_OBJECT(service)) != NULL,
				"Object path not set on the new service");
}
END_TEST

START_TEST(test_create_account)
{
	GalagoAccount *account;

	galago_init("check-libgalago", GALAGO_INIT_FEED);

	account = make_dummy_account();
	fail_unless(account != NULL, "Unable to create account");
	fail_unless(galago_object_get_dbus_path(GALAGO_OBJECT(account)) != NULL,
				"Object path not set on the new account");
}
END_TEST

START_TEST(test_create_presence)
{
	GalagoPresence *presence;

	galago_init("check-libgalago", GALAGO_INIT_FEED);

	presence = make_dummy_presence();
	fail_unless(presence != NULL, "Unable to create presence");
	fail_unless(galago_object_get_dbus_path(GALAGO_OBJECT(presence)) != NULL,
				"Object path not set on the new presence");
}
END_TEST


/**************************************************************************
 * Test D-BUS object signature support
 **************************************************************************/
static void
test_signature(GType type, GalagoObject *object)
{
	DBusMessage *message;
	DBusMessageIter iter;
	const char *obj_sig;
	char *sig;

	fail_unless(object != NULL, "Object was NULL");
	fail_unless(GALAGO_IS_OBJECT(object), "Object was not a GalagoObject");

	message = dbus_message_new_method_call(GALAGO_DBUS_SERVICE,
										   "/foo", "foo.bar", "foobar");
	fail_unless(message != NULL, "Message was unable to be created");

	obj_sig = galago_object_type_get_dbus_signature(type);
	fail_unless(obj_sig != NULL, "D-BUS signature is NULL");

	dbus_message_iter_init_append(message, &iter);
	galago_dbus_message_iter_append_object(&iter, GALAGO_OBJECT(object));

	dbus_message_iter_init(message, &iter);
	sig = dbus_message_iter_get_signature(&iter);
	fail_unless(sig != NULL, "Unable to retrieve signature from iterator");

	fail_unless(!strcmp(sig, obj_sig),
				"Class-provided signature ('%s') does not match "
				"iterator-provided signature ('%s')",
				obj_sig, sig);

	g_free(sig);
	g_object_unref(G_OBJECT(object));
	dbus_message_unref(message);
}

START_TEST(test_dbus_signature_account)
{
	galago_init("check-libgalago", GALAGO_INIT_CLIENT);
	test_signature(GALAGO_TYPE_ACCOUNT, GALAGO_OBJECT(make_dummy_account()));
}
END_TEST

START_TEST(test_dbus_signature_image)
{
	galago_init("check-libgalago", GALAGO_INIT_CLIENT);
	test_signature(GALAGO_TYPE_IMAGE, GALAGO_OBJECT(make_dummy_image()));
}
END_TEST

START_TEST(test_dbus_signature_person)
{
	galago_init("check-libgalago", GALAGO_INIT_CLIENT);
	test_signature(GALAGO_TYPE_PERSON, GALAGO_OBJECT(make_dummy_person()));
}
END_TEST

START_TEST(test_dbus_signature_presence)
{
	galago_init("check-libgalago", GALAGO_INIT_CLIENT);
	test_signature(GALAGO_TYPE_PRESENCE, GALAGO_OBJECT(make_dummy_presence()));
}
END_TEST

START_TEST(test_dbus_signature_service)
{
	galago_init("check-libgalago", GALAGO_INIT_CLIENT);
	test_signature(GALAGO_TYPE_SERVICE, GALAGO_OBJECT(make_dummy_service()));
}
END_TEST

START_TEST(test_dbus_signature_status)
{
	galago_init("check-libgalago", GALAGO_INIT_CLIENT);
	test_signature(GALAGO_TYPE_STATUS, GALAGO_OBJECT(make_dummy_status()));
}
END_TEST


/**************************************************************************
 * Test the Person object
 **************************************************************************/
static GalagoAccount *
dummy_calc_priority_account(GObject *unused, GalagoPerson *person)
{
	return NULL;
}

static GalagoAccount *
custom_calc_priority_account(GObject *unused, GalagoPerson *person)
{
	GList *accounts = galago_person_get_accounts(person, TRUE);

	fail_unless(accounts != NULL, "Account list of person is missing!");
	return accounts->data;
}

START_TEST(test_person_priority_accounts)
{
	GalagoPerson *person;
	GalagoService *service;
	GalagoPresence *presence;
	GalagoAccount *account1, *account2;
	GalagoAccount *priority_account;
	gulong conn_id;

	galago_init("check-libgalago", GALAGO_INIT_CLIENT);

	person = make_dummy_person();
	service = make_dummy_service();

	account1 = galago_service_create_account(service, person, "account-1");
	presence = galago_account_create_presence(account1);
	galago_presence_set_idle(presence, TRUE, time(NULL));

	account2 = galago_service_create_account(service, person, "account-2");
	presence = galago_account_create_presence(account2);

	priority_account = galago_person_get_priority_account(person);

	fail_unless(priority_account != NULL,
				"Returned priority account was NULL");
	fail_unless(priority_account == account2,
				"Default priority account calculation failed");

	/* Test with a dummy handler. */
	conn_id = g_signal_connect(galago_get_core(), "calc_priority_account",
							   G_CALLBACK(dummy_calc_priority_account), NULL);
	priority_account = galago_person_get_priority_account(person);
	fail_unless(priority_account != NULL,
				"Returned priority account was NULL");
	fail_unless(priority_account == account2,
				"Dummy priority account fallback calculation failed");
	g_signal_handler_disconnect(galago_get_core(), conn_id);


	/* Test with a real handler. */
	conn_id = g_signal_connect(galago_get_core(), "calc_priority_account",
							   G_CALLBACK(custom_calc_priority_account), NULL);
	priority_account = galago_person_get_priority_account(person);
	fail_unless(priority_account != NULL,
				"Returned priority account was NULL");
	fail_unless(priority_account == account1,
				"Custom priority account calculation failed");
	g_signal_handler_disconnect(galago_get_core(), conn_id);
}
END_TEST


/**************************************************************************
 * Test the Service object
 **************************************************************************/
static void
test_service_normalize_with(GalagoServiceFlags flags, const char *expectedStr)
{
	GalagoService *service;
	char *username;

	service = galago_create_service("test-service", "Test Service", flags);
	username = galago_service_normalize(service, "JoeBob  Smith/Home");
	fail_unless(!strcmp(username, expectedStr),
				"Service normalization failed with flags %d. Resulting string "
				"\"%s\" didn't match expected string \"%s\".",
				flags, username, expectedStr);
	g_free(username);
	g_object_unref(G_OBJECT(service));
}

START_TEST(test_service_normalize)
{
	galago_init("check-libgalago", GALAGO_INIT_CLIENT);

	test_service_normalize_with(0,
								"joebobsmith/home");
	test_service_normalize_with(GALAGO_PRESERVE_SPACES,
								"joebob  smith/home");
	test_service_normalize_with(GALAGO_PRESERVE_CASE,
								"JoeBobSmith/Home");
	test_service_normalize_with(GALAGO_STRIP_SLASH,
								"joebobsmith");
	test_service_normalize_with(GALAGO_PRESERVE_SPACES |
								GALAGO_PRESERVE_CASE |
								GALAGO_STRIP_SLASH,
								"JoeBob  Smith");
}
END_TEST


/**************************************************************************
 * Test synchronous calls
 **************************************************************************/
START_TEST(test_service_get_account)
{
	GalagoService *service;
	GalagoAccount *account;

	galago_init("check-libgalago", GALAGO_INIT_CLIENT);

	service = galago_get_service("aim", GALAGO_REMOTE, TRUE);
	fail_unless(service != NULL,
				"Couldn't fetch aim service. Is presence-feed running?");

	account = galago_service_get_account(service, "GalagoAIMUser", TRUE);
	fail_unless(service != NULL,
				"Couldn't get account GalagoAIMUser. "
				"Is presence-feed running?");
}
END_TEST


/**************************************************************************
 * Test asynchronous calls
 **************************************************************************/
static void
service_received_cb(GalagoCore *core, GalagoService *service,
					gpointer user_data)
{
	GMainLoop *loop = (GMainLoop *)user_data;

	fail_unless(service != NULL && GALAGO_IS_SERVICE(service) &&
				!strcmp(galago_service_get_id(service), "aim"),
				"Retrieved incorrect service parameter.");

	g_main_loop_quit(loop);
}

START_TEST(test_core_get_service_async)
{
	GMainLoop *loop;

	loop = g_main_loop_new(NULL, FALSE);
	galago_init("check-libgalago", GALAGO_INIT_CLIENT);
	galago_get_service_async("aim", service_received_cb, loop, NULL);
	g_main_loop_run(loop);
}
END_TEST


static void
account_received_cb(GalagoService *service, GalagoAccount *account,
					gpointer user_data)
{
	GMainLoop *loop = (GMainLoop *)user_data;

	fail_unless(service != NULL && GALAGO_IS_SERVICE(service) &&
				!strcmp(galago_service_get_id(service), "aim"),
				"Retrieved incorrect service parameter.");

	fail_unless(account != NULL && GALAGO_IS_ACCOUNT(account) &&
				!strcmp(galago_account_get_username(account), "GalagoAIMUser"),
				"Retrieved incorrect account parameter.");

	g_main_loop_quit(loop);
}

START_TEST(test_service_get_account_async)
{
	GalagoService *service;
	GMainLoop *loop;

	loop = g_main_loop_new(NULL, FALSE);

	galago_init("check-libgalago", GALAGO_INIT_CLIENT);

	service = galago_get_service("aim", GALAGO_REMOTE, TRUE);
	fail_unless(service != NULL,
				"Couldn't fetch aim service. Is presence-feed running?");

	galago_service_get_account_async(service, "GalagoAIMUser",
									 account_received_cb, loop, NULL);

	g_main_loop_run(loop);
}
END_TEST

static void
presence_received_cb(GalagoAccount *account, GalagoPresence *presence,
					 gpointer user_data)
{
	GMainLoop *loop = (GMainLoop *)user_data;

	fail_unless(account != NULL && GALAGO_IS_ACCOUNT(account) &&
				!strcmp(galago_account_get_username(account), "GalagoAIMUser"),
				"Retrieved incorrect account parameter.");

	fail_unless(presence != NULL && GALAGO_IS_PRESENCE(presence),
				"Retrieved incorrect presence parameter.");

	g_main_loop_quit(loop);
}

START_TEST(test_account_get_presence_async)
{
	GalagoService *service;
	GalagoAccount *account;
	GMainLoop *loop;

	loop = g_main_loop_new(NULL, FALSE);

	galago_init("check-libgalago", GALAGO_INIT_CLIENT);

	service = galago_get_service("aim", GALAGO_REMOTE, TRUE);
	fail_unless(service != NULL,
				"Couldn't fetch aim service. Is presence-feed running?");

	account = galago_service_get_account(service, "GalagoAIMUser", TRUE);
	fail_unless(service != NULL,
				"Couldn't fetch GalagoAIMUser. Is presence-feed running?");

	galago_account_get_presence_async(account, presence_received_cb, loop,
									  NULL);

	g_main_loop_run(loop);
}
END_TEST

/**************************************************************************
 * Core tests
 **************************************************************************/
START_TEST(test_core)
{
	fail_unless(1 == 1, "core test suite");
}
END_TEST

#define ADD_TCASE(name, func) \
	tc = tcase_create(name); \
	tcase_add_test(tc, (func)); \
	suite_add_tcase(s, tc)

static Suite *
make_libgalago_suite(void)
{
	Suite *s;
	TCase *tc;

	s = suite_create("libgalago");

	/* Core test */
	ADD_TCASE("core", test_core);

	/* libgalago tests */
	ADD_TCASE("init_reinit", test_init_reinit);
	ADD_TCASE("init_reinit_with_service", test_init_reinit_with_service);

	/* Test object creation */
	ADD_TCASE("create_person", test_create_person);
	ADD_TCASE("create_service", test_create_service);
	ADD_TCASE("create_account", test_create_account);
	ADD_TCASE("create_presence", test_create_presence);

	/* Test D-BUS object signature support */
	ADD_TCASE("dbus_signature_account",  test_dbus_signature_account);
	ADD_TCASE("dbus_signature_image",    test_dbus_signature_image);
	ADD_TCASE("dbus_signature_person",   test_dbus_signature_person);
	ADD_TCASE("dbus_signature_presence", test_dbus_signature_presence);
	ADD_TCASE("dbus_signature_service",  test_dbus_signature_service);
	ADD_TCASE("dbus_signature_status",   test_dbus_signature_status);

	/* Test the Person object */
	ADD_TCASE("person_priority_accounts", test_person_priority_accounts);

	/* Test the Service object */
	ADD_TCASE("service_normalize", test_service_normalize);

	/* Test synchronous queries */
	ADD_TCASE("service_get_account", test_service_get_account);

	/* Test asynchronous queries */
	ADD_TCASE("core_get_service_async", test_core_get_service_async);
	ADD_TCASE("service_get_account_async", test_service_get_account_async);
	ADD_TCASE("account_get_presence_async", test_account_get_presence_async);

	return s;
}

static void
error_handler(const gchar *log_domain, GLogLevelFlags log_level,
			  const gchar *message, gpointer user_data)
{
	fail(message);
}

int
main(int argc, char **argv)
{
	Suite *s;
	SRunner *sr;
	int nf;

	g_type_init();

	g_log_set_handler(NULL,
					  G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL |
					  G_LOG_FLAG_RECURSION,
					  error_handler, NULL);
	g_log_set_handler("GLib",
					  G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL |
					  G_LOG_FLAG_RECURSION,
					  error_handler, NULL);
	g_log_set_handler("Galago",
					  G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL |
					  G_LOG_FLAG_RECURSION,
					  error_handler, NULL);

	s = make_libgalago_suite();
	sr = srunner_create(s);

	srunner_set_log(sr, "check-libgalago.log");
	srunner_run_all(sr, CK_NORMAL);

	nf = srunner_ntests_failed(sr);

	srunner_free(sr);

	return (nf == 0 ? EXIT_SUCCESS : EXIT_FAILURE);
}
