/** \file coveb-tree.c */
/*
 * coveb : An open source van Emde Boas Priority Queue library
 * by Rudi Cilibrasi (cilibrar@cilibrar.com)
 *
 * Distributed under the BSD license.  Copyright 2008 Rudi Cilibrasi.
 *
 * v 0.71
 *
 * Uses stratified trees inspired by van Emde Boas to hold a priority queue
 * of unsigned 32-bit integer values.  Standard operations are supported.
 * Speed and memory use are optimized for large numbers of elements and
 * high queue throughput.  The data structure is designed to take advantage
 * of multilevel caches of any size; this property is called being
 * "cache oblivious".
 *
 * Instantiation:
 *
 * struct coveb_bq *q;
 *
 *   q = coveb_bq_new();
 *
 * This allocates a new cache-oblivious bitwise priority queue. To free, use:
 *
 * Deallocation:
 *
 *   coveb_bq_free(q);
 *
 * Insertion of 32-bit unsigned integer values:
 *
 * void coveb_bq_insert(struct coveb_bq *q, uint32_t x);
 *
 *    places a new value into the queue, or does nothing if it's already there.
 *
 * Removal of values:
 *
 * uint32_t coveb_bq_extractmin(struct coveb_bq *q);
 *
 *    removes the smallest value (min) from the queue and returns it.  It
 *    is illegal to try to remove a value from an empty (size = 0) queue.
 *    This is the fastest way to remove elements.
 *
 * uint32_t coveb_bq_remove(struct coveb_bq *q, uint32_t x);
 *
 *    This generic function removes any specific value from the queue.
 *    It returns 0 if the object was already in the queue, and 1 if not.
 *
 * Queue status monitoring:
 *
 *  When the queue is completely full (containing four 'binary' gigasamples),
 *  it consumes about one gigabyte of memory.  This means that it uses about 2
 *  bits per sample on average when full.  This is only about twice as big as
 *  the literal storage cost of a typical queue state.  Because this size
 *  is not reportable as a 32-bit integer, the following function is
 *  available to detect the full queue condition:
 *
 *  uint32_t coveb_bq_addresses_full(const struct coveb_bq *q);
 *
 *  Returns 1 if all 2 ** 32 elements are in the queue, or 0 otherwise.
 *  When the queue is not full, the normal size function is exactly accurate:
 *
 * uint32_t coveb_bq_size(const struct coveb_bq *q);
 *
 *    returns the count of integers currently stored in the queue.  This
 *  can be used to see if an element can be extracted safely (size > 0).
 *  Note that in the special case where the queue is completely full and
 *  therefore the number of elements in the queue is one greater than the
 *  maximum value of an unsigned 32-bit integer, the size function will
 *  return instead one less than the actual number of elements.  To
 *  detect this uncommon condition use coveb_bq_addresses_full.
 *
 * Extrema functions
 *
 * uint32_t coveb_bq_max(const struct coveb_bq *q);
 *
 *  returns the largest integer stored in the queue without removal.
 *
 * uint32_t coveb_bq_min(const struct coveb_bq *q);
 *
 *  returns the smallest integer stored in the queue without removal.
 *
 * Rangefinding functions
 *
 * void coveb_bq_locate_smallest_not_less_than(const struct coveb_bq *q,
 * uint32_t incl_lower_bound, uint32_t *result_x, uint32_t *gotresult);
 *
 *   searches for an integer stored in the queue that is not less than
 *   incl_lower_bound.  If there is more than one such integer, the smallest
 *   is returned.  If an integer is found, *gotresult will be set to 1 and
 *   *result_x will hold the integer that was found.  If no such integer is
 *   found in the queue, then *gotresult with be set to 0 and *result_x will
 *   be untouched.  The queue is not modified in any case.
 *
 * void coveb_bq_successor(const struct coveb_bq *q,
 * uint32_t excl_lower_bound, uint32_t *result_x, uint32_t *gotresult);
 *
 *   This function scans upward for an integer that is greater than
 *   excl_lower_bound.  It is otherwise the same as the
 *   coveb_bq_locate_smallest_not_less_than function.
 */

#include <stdio.h>
#include <string.h>
//#include <assert.h>
#include <stdlib.h>
#include <stdint.h>


#include "coveb-tree.h"

#include "bigmac.h"

#ifndef OUTERCALLOC
#define OUTERCALLOC(a,b) calloc(a,b)
#endif
#ifndef OUTERFREE
#define OUTERFREE(a) free(a)
#endif
#ifndef INNERCALLOC
#define INNERCALLOC(a,b) calloc(a,b)
#endif
#ifndef INNERFREE
#define INNERFREE(a) free(a)
#endif

/* Override FAILOUT_FUNC(msg) before #include this file to control error
 * reporting. */

void coveb_failout(const char *msg) {
#ifdef FAILOUT_FUNC
  FAILOUT_FUNC(msg);
#else
  fprintf(stderr, "Error: %s\n", msg);
  exit(1);
#endif
}

/**
 * \mainpage
 *
 * libcoveb by Rudi Cilibrasi (cilibrar@cilibrar.com)
 *
 * The open source cache oblivious van Emde Boas priority queue and tree
 * library.
 *
 * libcoveb is an optimized ordered container for 32-bit unsigned keys that
 * supports fast insertion and removal from one end as well as fast range
 * location.
 *
 * This library has two queue types coveb_bq and coveb_mq.  One is called the
 * bitwise queue (coveb_bq) , and the other is called the mapping queue
 * (coveb_mq).  The bitwise queue is the smallest and fastest, but stores only
 * 32-bit unsigned integers with no extra data.  The mapping priority queue is
 * able to store payloads of specific sizes attached to each entry that hold
 * whatever is appropriate according to user defined needs.  \sa coveb_bq_new()
 * and coveb_mq_new() for more details.
 *
 * libcoveb has been optimized for high performance in any caching situation.
 * It uses customized allocators for maximum speed.  It supports O(log (log N))
 * type performance for range location, insertion, and extraction of the
 * minimum value.  It can outperform balanced binary trees and hash tables in
 * many situations, depending on the number of elements involved and the
 * distribution of the data.  The data structure used in libcoveb is like a
 * stratified tree.  If you have more than ten thousand items or so, this
 * data structure may speed up performance compared to other comparison based
 * structures like trees.
 */

/* Optimized allocator */
static inline struct FastAllocController *fac_create(void);
static inline void *faa_alloc_block(struct FastAllocArena *faa);
static inline void *faa_calloc_block(struct FastAllocArena *faa);
static inline void *fac_calloc_block(struct FastAllocController *fac, uint32_t bs);
static inline void *fac_alloc_block(struct FastAllocController *fac, uint32_t bs);
static inline void fac_free_block(struct FastAllocController *fac, void *blockaddr, uint32_t bs);
static inline void fac_free_all_blocks(struct FastAllocController *fac);

static inline void coveb_bq_free_ex(struct coveb_bq *q, uint32_t f_outerblocktoo);

/* Internal bitwise queue functions */
static inline void coveb__ibq_insert(struct vEBPNode *v, uint32_t x);
static inline uint32_t coveb__ibq_remove(struct vEBPNode *v, uint32_t m);
static inline struct vEBPNode *coveb__ibq_create(uint32_t fullbitsize,  struct FastAllocController *fac, uint32_t x);
static inline COVEB_KEYTYPE coveb__ibq_min(const struct vEBPNode *v);
static inline uint32_t coveb__ibq_extractmin(struct vEBPNode *v);

static inline uint32_t coveb__ibq_contains(const struct vEBPNode *v, COVEB_KEYTYPE x);
static inline uint32_t coveb__ibq_size(const struct vEBPNode *v);
static inline uint32_t coveb__ibq_recompose(int k, uint32_t hi, uint32_t lo);
static inline uint32_t coveb__ibq_addresses_full(const struct vEBPNode *q);
static inline void coveb__ibq_decompose(uint32_t inpnum, int k, uint32_t *hi, uint32_t *lo);
static inline uint32_t direct_table_min(const struct vEBSimpleBitTable *v);
static inline void coveb__ibq_locate_smallest_not_less_than(const struct vEBPNode *v, uint32_t incl_lower_bound, uint32_t *result_x, uint32_t *gotresult);
static inline void direct_table_locate_smallest_not_less_than(const struct vEBSimpleBitTable *v, uint32_t incl_lower_bound, uint32_t *result_x, uint32_t *gotresult);

static const unsigned char CovebBitsSetTable256[] =
{
0,1,1,2,1,2,2,3,1,2,2,3,2,3,3,4, 1,2,2,3,2,3,3,4,2,3,3,4,3,4,4,5,
1,2,2,3,2,3,3,4,2,3,3,4,3,4,4,5, 2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6,
1,2,2,3,2,3,3,4,2,3,3,4,3,4,4,5, 2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6,
2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6, 3,4,4,5,4,5,5,6,4,5,5,6,5,6,6,7,
1,2,2,3,2,3,3,4,2,3,3,4,3,4,4,5, 2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6,
2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6, 3,4,4,5,4,5,5,6,4,5,5,6,5,6,6,7,
2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6, 3,4,4,5,4,5,5,6,4,5,5,6,5,6,6,7,
3,4,4,5,4,5,5,6,4,5,5,6,5,6,6,7, 4,5,5,6,5,6,6,7,5,6,6,7,6,7,7,8
};

static const unsigned char CovebLogTable256[] =
{
0,0,1,1,2,2,2,2,3,3,3,3,3,3,3,3, 4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,
5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5, 5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6, 6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6, 6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7, 7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7, 7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7, 7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7, 7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7
};

/* Inline function definitions for maximum speed. */

/**
 * Find the maximum element in the queue.
 * \param q pointer to input bitwise priority queue
 *
 * \returns 32-bit unsigned integer representing the maximum value in the queue.
 *
 * It is illegal to try to take the maximum value from an empty queue.
 */
uint32_t coveb_bq_max(const struct coveb_bq *q)
{
  if (q->tree == NULL) {
    coveb_failout("Cannot take maximum of empty queue.");
    return 0;
  }
  if (coveb__ibq_size(q->tree) == 1)
    return coveb_bq_min(q);
  return q->tree->fn._max;
}

/**
 * Find the minimum element in the queue.
 * \param q pointer to input bitwise priority queue
 *
 * \returns 32-bit unsigned integer representing the minimum value in the queue.
 *
 * It is illegal to try to take the minimum value from an empty queue.
 */
uint32_t coveb_bq_min(const struct coveb_bq *q)
{
  if (q->tree == 0) {
    coveb_failout("Cannot find min element of an empty queue.");
    return 0;
  }
  return coveb__ibq_min(q->tree);
}


/**
 * Remove an element from the queue.
 * \param q pointer to input bitwise priority queue
 * \param x value to be removed
 *
 * \returns unsigned integer flag indicating whether the operation
 * succeeded in removing an element (0) or failed to remove an element (1).
 *
 * It is illegal to try to remove an element from an empty queue.
 * If the element to be removed is not in the queue, the call has no effect
 * and the value 1 is returned.
 */
uint32_t coveb_bq_remove(struct coveb_bq *q, uint32_t x)
{
  if (q->tree == 0) {
    coveb_failout("Cannot extract element from an empty queue.");
    return 0;
  }
  COVEB_KEYTYPE rv;
  if (coveb__ibq_size(q->tree) == 1) {
    rv = coveb__ibq_min(q->tree);
    if (rv != x)
      return 1;
    coveb_bq_free_ex(q, 0);
    return 0;
  }
  else
    rv = coveb__ibq_remove(q->tree, x);
  return rv;
}

/**
 * Remove the smallest element from the queue.
 * \param q pointer to input bitwise priority queue
 *
 * \returns 32-bit unsigned integer representing the element that was removed
 *
 * It is illegal to call this function with an empty (coveb_bq_size() == 0)
 * queue.  The value that was removed from the queue is returned.
 */
uint32_t coveb_bq_extractmin(struct coveb_bq *q)
{
  if (q->tree == 0) {
    coveb_failout("Cannot extract element from an empty queue.");
    return 0;
  }
  COVEB_KEYTYPE rv;
  if (coveb__ibq_size(q->tree) == 1) {
    rv = coveb__ibq_min(q->tree);
    coveb_bq_free_ex(q, 0);
  }
  else
    rv = coveb__ibq_extractmin(q->tree);
  return rv;
}

/**
 * Test if the queue contains all 32-bit unsigned integers or not.
 *
 * \param q bitwise priority queue to be tested
 * \returns 1 if the queue is completely full and contains all possible
 * 32-bit unsigned integers, or 0 otherwise.
 *
 * This function is necessary to determine if the queue contains
 * 2 ** 32 or (2 ** 32) - 1 entries.  When the queue has either of these
 * two sizes, coveb_bq_size() returns (2 ** 32) - 1 to avoid integer overflow.
 * This function provides a way to get the exact size in this exceptional case.
 *
 * \sa coveb_bq_size()
 */
uint32_t coveb_bq_addresses_full(const struct coveb_bq *q)
{
  if (q->tree == 0)
    return 0;
  return coveb__ibq_addresses_full(q->tree);
}

static inline uint32_t coveb__ibq_addresses_full(const struct vEBPNode *q)
{
  return (q->t._flags & MASK_TOTFULL) != 0;
}

/**
 * Returns the number of elements stored in a priority queue.
 * \param q pointer to input bitwise priority queue
 *
 * \returns 32-bit unsigned integer representing the number of elements stored.
 *
 * An empty queue returns a size of 0.
 *
 * \sa coveb_bq_addresses_full()
 *
 */
uint32_t coveb_bq_size(const struct coveb_bq *q)
{
  if (q->tree == 0)
    return 0;
  return coveb__ibq_size(q->tree);
}

/**
 * Determines whether or not the queue contains a specific element.
 * \param q pointer to input bitwise priority queue
 * \param x 32-bit unsigned integer to search for in the queue
 *
 * \returns unsigned integer flag indicating if the value was found (1)
 * or not (0).
 *
 */
uint32_t coveb_bq_contains(const struct coveb_bq *q, uint32_t x)
{
  if (q->tree == 0)
    return 0;
  return coveb__ibq_contains(q->tree, x);
}

/**
 * Allocates a new bitwise priority queue.  A bitwise priority queue holds
 * a set of 32-bit unsigned integers.
 * \returns pointer to a new and empty bitwise priority queue.
 */
struct coveb_bq *coveb_bq_new(void) {
  struct coveb_bq *result = (struct coveb_bq *) OUTERCALLOC(sizeof(*result), 1);
  return result;
}

/**
 * Copies a bitwise priority queue.
 * \param q bitwise priority queue to be copied
 * \returns pointer to a new and identical bitwise priority queue
 */
struct coveb_bq *coveb_bq_clone(const struct coveb_bq *q) {
  struct coveb_bq *result = coveb_bq_new();
  if (coveb_bq_size(q) == 0)
    return result;
  uint32_t proberes = 0, probeflag, cur = coveb_bq_min(q);
  do {
    coveb_bq_insert(result, cur);
    coveb_bq_successor(q, cur, &proberes, &probeflag);
    cur = proberes;
  } while (probeflag);
  return result;
}

static inline void faa_free_block(struct FastAllocArena *faa, void *blockaddr) {
  struct FastAllocFreeNode *fafn = (struct FastAllocFreeNode *) blockaddr;
  fafn->nxt = faa->freelist;
  faa->freelist = fafn;
}

static inline void *faa_calloc_block(struct FastAllocArena *faa) {
  void *rv = faa_alloc_block(faa);
  memset(rv, 0, faa->blocksize);
  return rv;
}

static inline void *faa_alloc_block(struct FastAllocArena *faa) {
  if (faa->freelist == NULL) {
    if (faa->blocksperchunk < faa->maxbpc) {
      if (faa->blocksperchunk * 2 < faa->maxbpc)
        faa->blocksperchunk *= 2;
      else
        faa->blocksperchunk = faa->maxbpc;
    }
    int totsize = faa->blocksize * faa->blocksperchunk + sizeof(void *);
    void *bigv = INNERCALLOC(1, totsize);
    if (bigv == NULL) {
      coveb_failout("Out of memory.");
      return NULL;
    }
    struct FastAllocFreeNode *bigfafn = (struct FastAllocFreeNode *) bigv;
    bigfafn->nxt = faa->bigchunks;
    faa->bigchunks = bigfafn;
    bigfafn = (struct FastAllocFreeNode *) ((void **)bigv + 1);
    uint32_t i;
    for (i = 0; i < faa->blocksperchunk; i += 1) {
      bigfafn->nxt = faa->freelist;
      faa->freelist = bigfafn;
      bigfafn =       (struct FastAllocFreeNode *)
                  (((char *)bigfafn) + faa->blocksize);
    }
  }
  struct FastAllocFreeNode *rv = faa->freelist;
  faa->freelist = faa->freelist->nxt;
  return rv;
}

static inline void faa_free_everything(struct FastAllocArena *dest)
{
  struct FastAllocFreeNode *cur;
  for (cur = dest->bigchunks; cur != NULL;) {
    struct FastAllocFreeNode *faanext = cur->nxt;
    INNERFREE(cur);
    cur = faanext;
  }
  dest->freelist = NULL;
  dest->bigchunks = NULL;
}

static inline void faa_init(struct FastAllocArena *dest, uint32_t bs, uint32_t bpc)
{
  dest->blocksize = bs;
  dest->blocksperchunk = 1;
  dest->maxbpc = bpc;
  dest->freelist = NULL;
  dest->bigchunks = NULL;
}

static inline struct FastAllocArena *search_fac_for_faa_with_size(struct FastAllocController *fac, uint32_t bs)
{
  int i;
  for (i = 0; i < fac->faa_tot; i += 1)
    if (fac->faa[i].blocksize == bs) {
      return &fac->faa[i];
    }
  return NULL;
}

static inline void create_new_faa_for_size(struct FastAllocController *fac, uint32_t bs)
{
  if (fac->faa_tot == MAXBLOCKSIZES) {
    coveb_failout("Out of memory sizes.");
    return;
  }
  faa_init(&fac->faa[fac->faa_tot], bs, bpcrec(bs));
  fac->faa_tot += 1;
}

static inline struct FastAllocController *fac_create(void)
{
  struct FastAllocController *fac = (struct FastAllocController *) OUTERCALLOC(sizeof(*fac), 1);
  create_new_faa_for_size(fac, sizeof(struct vEBPNode));
  create_new_faa_for_size(fac, sizeof(COVEB_KEYTYPE) * (1 << 3));
  create_new_faa_for_size(fac, sizeof(COVEB_KEYTYPE) * (1 << 8));
  create_new_faa_for_size(fac, sizeof(COVEB_KEYTYPE) * (1 << 16));
  return fac;
}

static inline void *fac_calloc_block(struct FastAllocController *fac, uint32_t bs)
{
  struct FastAllocArena *faa;
  faa = search_fac_for_faa_with_size(fac, bs);
  if (faa == NULL) {
    create_new_faa_for_size(fac, bs);
    faa = search_fac_for_faa_with_size(fac, bs);
  }
  return faa_calloc_block(faa);
}

static inline void fac_free_block(struct FastAllocController *fac, void *blockaddr, uint32_t bs)
{
  struct FastAllocArena *faa;
  faa = search_fac_for_faa_with_size(fac, bs);
  faa_free_block(faa, blockaddr);
}

static inline void fac_free_all_blocks(struct FastAllocController *fac)
{
  int i;
  for (i = 0; i < fac->faa_tot; i += 1)
    faa_free_everything(&fac->faa[i]);
  OUTERFREE(fac);
}

static inline void *fac_alloc_block(struct FastAllocController *fac, uint32_t bs)
{
/*
  if (bs < sizeof(void *)) {
    coveb_failout("Cannot allocate miniscule memory blocks.");
    return NULL;
  }
*/
  struct FastAllocArena *faa;
  faa = search_fac_for_faa_with_size(fac, bs);
  if (faa == NULL) {
    create_new_faa_for_size(fac, bs);
    faa = search_fac_for_faa_with_size(fac, bs);
  }
  void *rv = faa_alloc_block(faa);
  return rv;
}

static inline uint32_t direct_table_remove(struct vEBSimpleBitTable *v, uint32_t x) {
  uint32_t mask = (1 << x);
  uint32_t res = !(v->_bits & mask);
  v->_bits &= ~mask;
  return res;
}

static inline void direct_table_locate_smallest_not_less_than(const struct vEBSimpleBitTable *v, uint32_t incl_lower_bound, uint32_t *result_x, uint32_t *gotresult)
{
  uint32_t i;
  *gotresult = 0;
  for (i = incl_lower_bound; i < 32; i += 1) {
    if (direct_table_contains(v, i)) {
      *result_x = i;
      *gotresult = 1;
      return;
    }
  }
  return;
}

static inline void coveb__ibq_locate_smallest_not_less_than(const struct vEBPNode *v, uint32_t incl_lower_bound, uint32_t *result_x, uint32_t *gotresult)
{
  *gotresult = 0;
  if (incl_lower_bound == 0) {
    *result_x = coveb__ibq_min(v);
    *gotresult = 1;
    return;
  }

  if (is_single(v)) {
    if (v->sn._x >= incl_lower_bound) {
      *result_x = v->sn._x;
      *gotresult = 1;
      //assert(*result_x >= incl_lower_bound);
    }
    return; // we can assume *gotresult starts at 0
  }
  if (v->fn._min >= incl_lower_bound) {
      *result_x = v->fn._min;
      *gotresult = 1;
      //assert(*result_x >= incl_lower_bound);
      return;
  }
  uint32_t mh, ml, th, tl,k;
  k = get_fullbitsize(v);
  coveb__ibq_decompose(incl_lower_bound, k, &th, &tl);
  mh = TOP_MIN(v);
  if (mh > th) {
    ml = BOT_MIN(v, mh);
    *result_x = coveb__ibq_recompose(k, mh, ml);
    *gotresult = 1;
      //assert(*result_x >= incl_lower_bound);
    return;
  }
  uint32_t botres, lowersuccess;
  uint32_t lowend;
  //assert(mh <= th);
  uint32_t mth, topres, topsuccess;
  mth = th;
  nextphase:
  TOP_LOCATESNLT(v, mth, &topres, &topsuccess);
  if (topsuccess != 0) {
    //assert(topres >= th);
    lowend = (topres > th) ? 0 : tl;
    BOT_LOCATESNLT(v, topres, lowend, &botres, &lowersuccess);
    if (lowersuccess == 0) {
      mth = topres + 1;
      goto nextphase;
    }
    *result_x = coveb__ibq_recompose(k, topres, botres);
    *gotresult = 1;
      //assert(*result_x >= incl_lower_bound);
    return;
  }

  // Max found no result

  return;
}

static void coveb__ibq_free_node(struct vEBPNode *v);

static inline void custom_free_fastmap(struct FastAllocController *fac, struct vEBPNode **fm, int numelems) {
#if USE_FAST_ALLOCATOR
  fac_free_block(fac, fm, numelems * sizeof(void *));
#else
  INNERFREE(fm);
#endif
}

static inline void custom_free_node(struct vEBPNode *v) {
#if USE_FAST_ALLOCATOR
  fac_free_block(v->t._fac, v, sizeof(*v));
#else
  INNERFREE(v);
#endif
}

static inline struct vEBPNode **custom_alloc_fastmap(struct FastAllocController *fac, uint32_t howmanyelems) {
#if USE_FAST_ALLOCATOR
/*
  if (fac == 0) {
    coveb_failout("No allocator passed in to alloc_fastmap");
    return NULL;
  }
*/
  return (struct vEBPNode **) fac_calloc_block(fac, howmanyelems * sizeof(struct vEBPNode *));
#else
  return (struct vEBPNode **) INNERCALLOC(howmanyelems, sizeof(struct vEBPNode *));
#endif
}

static inline struct vEBPNode *custom_alloc_node(struct FastAllocController *fac) {
#if USE_FAST_ALLOCATOR
  return (struct vEBPNode *) fac_calloc_block(fac, sizeof(struct vEBPNode));
#else
  return (struct vEBPNode *) INNERCALLOC(1, sizeof(struct vEBPNode));
#endif
}

/**
 * Deallocate (or free) a bitwise priority queue.
 * \param q bitwise priority queue to be deallocated
 * \returns nothing
 */
void coveb_bq_free(struct coveb_bq *q)
{
  coveb_bq_free_ex(q, 1);
}

static void coveb_bq_free_ex(struct coveb_bq *d, uint32_t outerblocktoo) {
  if (d->fac) {
    fac_free_all_blocks(d->fac);
    d->fac  = NULL;
    d->tree = NULL;
  }
  if (d->tree) {
    coveb__ibq_free_node(d->tree);
    d->tree = NULL;
  }
  if (outerblocktoo)
    OUTERFREE(d);
}

static inline void coveb__ibq_decompose(uint32_t inpnum, int k, uint32_t *hi, uint32_t *lo) {
  uint32_t blo = BOT_K_FROM_K(k);
  uint32_t lomask = (1u << blo) - 1;
  *lo = inpnum & lomask;
  *hi = inpnum >> blo;
}

static inline uint32_t coveb__ibq_recompose(int k, uint32_t hi, uint32_t lo) {
  uint32_t blo = BOT_K_FROM_K(k);
  return (hi << blo) | lo;
}

/**
 * Find the element in the queue after a given lower boundary point.
 * \param q bitwise priority queue to be scanned
 * \param num 32-bit unsigned integer lower boundary point
 * \param result_x pointer to 32-bit unsigned integer output result buffer
 * \param gotresult pointer to 32-bit unsigned integer flag
 * \returns nothing
 *
 * If an entry is found in the queue that is greater than num, the smallest
 * is returned in *result_x and *gotresult will be set to 1.  If no element is
 * found above num, then *gotresult will be set to 0.  This function can be
 * used to iterate through the queue.
 *
 * \sa coveb_bq_locate_smallest_not_less_than()
 */
void coveb_bq_successor(const struct coveb_bq *q, uint32_t num, uint32_t *result_x, uint32_t *gotresult)
{
  if (num == 0xFFFFFFFF) {
    *gotresult = 0;
    return;
  }
  coveb_bq_locate_smallest_not_less_than(q,num+1,result_x,gotresult);
}

/**
 * Find the smallest element in the queue at least as big as a given lower
 * boundary point.
 *
 * \param q bitwise priority queue to be scanned
 * \param incl_lower_bound 32-bit unsigned integer lower boundary point
 * \param result_x pointer to 32-bit unsigned integer output result buffer
 * \param gotresult pointer to 32-bit unsigned integer flag
 * \returns nothing
 *
 * If an entry is found in the queue that is greater than or equal to
 * incl_lower_bound, the smallest such entry is returned in *result_x and
 * *gotresult will be set to 1.  If no element is found above num, then
 * *gotresult will be set to 0.  This function can be used to iterate through
 * the queue.
 *
 * \sa coveb_bq_successor()
 */
void coveb_bq_locate_smallest_not_less_than(const struct coveb_bq *q, uint32_t incl_lower_bound, uint32_t *result_x, uint32_t *gotresult)
{
  *gotresult = 0;
  if (q->tree == NULL) {
    return;
  }
  coveb__ibq_locate_smallest_not_less_than(q->tree, incl_lower_bound, result_x, gotresult);
}

/**
 * Insert an element into the queue.
 *
 * \param q bitwise priority queue to be scanned
 * \param x 32-bit unsigned integer to be inserted
 * \returns nothing
 *
 * This function inserts the element x into the queue.
 * If the element is already in the queue, this function has no effect.
 *
 * \sa coveb_bq_extractmin()
 */
void coveb_bq_insert(struct coveb_bq *q, uint32_t x)
{
  if (q->tree != NULL) {
    coveb__ibq_insert(q->tree, x);
    return;
  }
  if (q->fac == NULL)
    q->fac = fac_create();
  q->tree = coveb__ibq_create(32, q->fac, x);
}

static inline void fastmap_erase_subnode(struct vEBPNode *v, uint32_t ind) {
  v->fn._bot[ind] = NULL;
}

static inline void fastmap_store_subnode(struct vEBPNode *v, uint32_t ind, struct vEBPNode *subnode) {
  v->fn._bot[ind] = subnode;
}

static inline struct vEBPNode *fastmap_fetch_subnode(struct vEBPNode *v,uint32_t ind) {
  return v->fn._bot[ind];
}

static inline uint32_t fastmap_fetch_bits(struct vEBPNode *v,uint32_t ind) {
  return (uint32_t) (v->fn._bot[ind]);
}

static inline void fastmap_store_bits(struct vEBPNode *v, uint32_t ind, uint32_t bits) {
  v->fn._bot[ind] = (struct vEBPNode *) bits;
}

static inline void fastmap_initialize(struct vEBPNode *v, uint32_t howManyElems) {
  v->fn._bot = custom_alloc_fastmap(v->t._fac, howManyElems);
}

static inline void direct_table_initialize(struct vEBSimpleBitTable *v, uint32_t x) {
  v->_bits = 1 << x;
}

static inline void direct_table_free(struct vEBSimpleBitTable *v) {
  v->_bits = 0;
}

static inline void direct_table_insert(struct vEBSimpleBitTable *v, uint32_t x) {
  v->_bits |= 1 << x;
}

static inline uint32_t get_log2(uint32_t x) {
  uint32_t r;
  uint32_t t, tt;

  if ( (tt = x >> 16) )
    r = (t = tt >> 8) ? 24 + CovebLogTable256[t] : 16 + CovebLogTable256[tt];
  else
    r = (t = x >> 8) ? 8 + CovebLogTable256[t] : CovebLogTable256[x];
  return r;
}

static inline uint32_t direct_table_extractmin(struct vEBSimpleBitTable *v) {
  uint32_t b = v->_bits;
  uint32_t j = b & (b - 1);
  uint32_t rv = get_log2(j ^ b);
  v->_bits = j;
  return rv;
}

static inline uint32_t direct_table_size(const struct vEBSimpleBitTable *v) {
  unsigned char * p = (unsigned char *) &v->_bits;
  return CovebBitsSetTable256[p[0]] + CovebBitsSetTable256[p[1]] +
         CovebBitsSetTable256[p[2]] + CovebBitsSetTable256[p[3]];
}

static inline uint32_t direct_table_max(const struct vEBSimpleBitTable *v) {
    int32_t i;
    uint32_t b = v->_bits;
    for (i = 31; i >= 0; i -= 1) {
      if (b & 0x80000000 )
        return i;
      b <<= 1;
    }
    coveb_failout("Cannot find set bit in max.");
    return 0;
}

static inline uint32_t direct_table_min(const struct vEBSimpleBitTable *v) {
    uint32_t b = v->_bits;
    uint32_t j = b & (b - 1);
    uint32_t rv = get_log2(j ^ b);
    return rv;
}

static inline COVEB_KEYTYPE coveb__ibq_size(const struct vEBPNode *v) {
  return v->t._size;
}

static inline COVEB_KEYTYPE coveb__ibq_min(const struct vEBPNode *v) {
  if (is_single(v))
    return v->sn._x;
  return v->fn._min;
}

struct vEBPSearchResult coveb_search(struct vEBPNode *v, COVEB_KEYTYPE x) {
  struct vEBPSearchResult result, subr;
  result.x = x;
  result.sn = NULL;
  result.atmin = NULL;
  result.ondirectdata = NULL;
  result.is_notfound = 1;
  if (is_single(v)) {
    if (v->sn._x == x) {
      result.is_notfound = 0;
      result.sn = v;
    }
    return result;
  }
  if (v->fn._min == x) {
      result.is_notfound = 0;
      result.atmin = v;
      return result;
  }
  uint32_t k = get_fullbitsize(v);
  uint32_t hi, lo;
  coveb__ibq_decompose(x, k, &hi, &lo);
  if (!TOP_CONTAINS(v, hi))
    return result;
  if (BOT_IS_DIRECT(k)) {
    if (direct_table_contains(GET_DIRECT(v->fn._bot[hi]), lo)) {
      result.is_notfound = 0;
      result.ondirectdata = v;
    }
    return result;
  }
  subr = coveb_search(v->fn._bot[hi], lo);
  if (subr.is_notfound)
    return result;
  subr.x = result.x;
  return subr;
  }

static inline uint32_t coveb__ibq_contains(const struct vEBPNode *v, COVEB_KEYTYPE x) {
  if (is_single(v))
    return (v->sn._x == x);
  if (v->fn._min == x)
    return 1;
  uint32_t k = get_fullbitsize(v);
  uint32_t hi, lo;
  coveb__ibq_decompose(x, k, &hi, &lo);
  if (!TOP_CONTAINS(v, hi))
    return 0;
  if (!BOT_CONTAINS(v, hi, lo))
    return 0;
  return 1;
}

static inline struct vEBPNode *coveb__ibq_create(uint32_t fullbitsize,  struct FastAllocController *fac, uint32_t x) {
  struct vEBPNode *v = custom_alloc_node(fac);
  v->t._fullbitsize = fullbitsize;
  v->t._size = 1;
  v->sn._x = x;
  v->t._fac = fac;
  return v;
}

static inline void coveb_demote(struct vEBPNode *v) {
  uint32_t k = get_fullbitsize(v);
  uint32_t botind = TOP_MIN(v);
  if (!TOP_IS_DIRECT(k))
    coveb__ibq_free_node(v->fn._top);
  if (!BOT_IS_DIRECT(k)) {
    coveb__ibq_free_node(v->fn._bot[botind]);
  }
  v->fn._top = 0;
  v->fn._bot[botind] = 0;
  custom_free_fastmap(v->t._fac, v->fn._bot, 1 << TOP_K_FROM_K(k));
  v->t._size = 1;
}

static inline void coveb_promote(struct vEBPNode *v, uint32_t x) {
  uint32_t k = get_fullbitsize(v);
  uint32_t hi, lo;
  uint32_t minprime, nextval;
  minprime = (x < v->sn._x) ? x : v->sn._x;
  nextval = (x > v->sn._x) ? x : v->sn._x;
  coveb__ibq_decompose(nextval, k, &hi, &lo);
  v->fn._min = minprime;
  v->fn._max = nextval;
  if (TOP_IS_DIRECT(k))
    direct_table_initialize(GET_DIRECT(v->fn._top), hi);
  else
    v->fn._top = coveb__ibq_create(TOP_K_FROM_K(k), v->t._fac, hi);
  v->fn._bot = custom_alloc_fastmap(v->t._fac, 1 << TOP_K_FROM_K(k));
  if (BOT_IS_DIRECT(k))
    direct_table_insert(GET_DIRECT(v->fn._bot[hi]), lo);
  else
    fastmap_store_subnode(v, hi, coveb__ibq_create(BOT_K_FROM_K(k), v->t._fac, lo));
  v->t._size = 2;
}

static inline void coveb__ibq_insert(struct vEBPNode *v, uint32_t x) {
  uint32_t k = get_fullbitsize(v);
  if (v->fn._min == x)
    return;
  if (is_single(v)) {
    if (v->sn._x == x)
      return;
    coveb_promote(v, x);
    return;
  }
  if (x < v->fn._min) {
    uint32_t tmp;
    tmp = x;
    x = v->fn._min;
    v->fn._min = tmp;
  }
  uint32_t hi, lo;
  coveb__ibq_decompose(x, k, &hi, &lo);
  uint32_t tophad, bothad = 0;
  tophad = TOP_CONTAINS(v, hi);
  if (tophad)
    bothad = BOT_CONTAINS(v, hi, lo);
  if (tophad && bothad)
    return;
  if (!tophad)
    TOP_INSERT(v, hi);
  if (!bothad) {
    if (!BOT_IS_DIRECT(k) && v->fn._bot[hi] == 0)
      fastmap_store_subnode(v, hi, coveb__ibq_create(BOT_K_FROM_K(k), v->t._fac, lo));
    else
      BOT_INSERT(v, hi, lo);
  }
  if ((v->t._flags & MASK_TOTFULL) == 0) {
    if (v->t._size == 0xFFFFFFFF)
      v->t._flags |= MASK_TOTFULL;
    else
    v->t._size += 1;
  }
  if (x > v->fn._max)
    v->fn._max = x;
  return;
}

static inline uint32_t coveb__ibq_remove(struct vEBPNode *v, uint32_t m) {
  if (coveb__ibq_size(v) < 2) {
    coveb_failout("Cannot remove last element from queue.");
    return 1;
  }
  uint32_t k = get_fullbitsize(v);
  if (m == v->fn._min) {
    coveb__ibq_extractmin(v);
    return 0;
  }
  uint32_t rv = !coveb__ibq_contains(v,m);
  if (rv)
    return rv;
  uint32_t nextval = v->fn._min;
  COVEB_KEYTYPE sh, sl;
  coveb__ibq_decompose(m, k, &sh, &sl);
  if (BOT_SIZE(v, sh) == 1) {
    if (TOP_SIZE(v) == 1) {
      coveb_demote(v);
      v->sn._x = nextval;
      return rv;
    } else {
      if (BOT_IS_DIRECT(k))
        direct_table_remove(GET_DIRECT(v->fn._bot[sh]), sl);
      else
        coveb__ibq_free_node(fastmap_fetch_subnode(v, sh));
      fastmap_erase_subnode(v, sh);
      TOP_REMOVE(v,sh);
      goto dropsize;
    }
  } else {
    BOT_REMOVE(v, sh, sl);
  }
  dropsize:
  if (v->t._flags & MASK_TOTFULL)
    v->t._flags &= ~MASK_TOTFULL;
  else
    v->t._size -= 1;
  if (m != v->fn._max)
    return rv;
  uint32_t lowend = v->fn._min, hiend = v->fn._max;
  while (hiend - lowend > 2) {
    uint32_t probenum = (lowend >> 1) + (hiend >> 1), proberesnum, proberesflag;
    coveb__ibq_locate_smallest_not_less_than(v, probenum, &proberesnum, &proberesflag);
    if (proberesflag)
      lowend = proberesnum;
    else
      hiend = probenum;
  }
  v->fn._max = lowend;
  return rv;
}

static inline uint32_t coveb__ibq_extractmin(struct vEBPNode *v) {
  if (coveb__ibq_size(v) < 2) {
    coveb_failout("Cannot extract last element from queue.");
    return 0;
  }
  uint32_t k = get_fullbitsize(v);
  COVEB_KEYTYPE sh = TOP_MIN(v);
  COVEB_KEYTYPE sl = BOT_MIN(v, sh);
  COVEB_KEYTYPE nextmin = coveb__ibq_recompose(k, sh, sl);
  COVEB_KEYTYPE rv = v->fn._min;
  if (BOT_SIZE(v, sh) == 1) {
    if (TOP_SIZE(v) == 1) {
      coveb_demote(v);
      v->sn._x = nextmin;
      return rv;
    } else {
      TOP_EXTRACTMIN(v);
      if (!BOT_IS_DIRECT(k))
        coveb__ibq_free_node(fastmap_fetch_subnode(v, sh));
      fastmap_erase_subnode(v, sh);
    }
  } else {
    BOT_EXTRACTMIN(v, sh);
  }
  v->fn._min = nextmin;
  if (v->t._flags & MASK_TOTFULL)
    v->t._flags &= ~MASK_TOTFULL;
  else
    v->t._size -= 1;
  return rv;
}

void coveb__ibq_free_node(struct vEBPNode *v) {
  while (!is_single(v))
    coveb__ibq_extractmin(v);
  custom_free_node(v);
}

