/*
  CLAW - a C++ Library Absolutely Wonderful

  CLAW is a free library without any particular aim but being useful to 
  anyone.

  Copyright (C) 2005-2008 Julien Jorge

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: julien_jorge@yahoo.fr
*/
/**
 * \file string_algorithm.tpp
 * \brief Implementation of the algorithms on strings.
 * \author Julien Jorge
 */
#include <sstream>

/*----------------------------------------------------------------------------*/
/**
 * \brief A portable version of std::getline( is, str, '\n' ) that removes a
 *        tailing '\r'.
 * \param is The stream in which we read.
 * \param str The line read from the stream.
 */
template<typename StreamType, typename StringType>
StreamType& claw::text::getline( StreamType& is, StringType& str )
{
  std::getline( is, str );

  if ( !str.empty() )
    if ( str[ str.size() - 1 ] == typename StringType::value_type('\r') )
      str.erase( str.size() - 1 );

  return is;
} // getline()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove characters at the begining of a string.
 * \param str The string to modify.
 * \param s The characters to remove.
 */
template<class StringType>
void claw::text::trim_left( StringType& str,
                            const typename StringType::value_type* const s )
{
  typename StringType::size_type p = str.find_first_not_of(s);

  if (p != StringType::npos)
    str = str.substr(p);
} // trim_left()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove characters at the end of a string.
 * \param str The string to modify.
 * \param s The characters to remove.
 */
template<class StringType>
void claw::text::trim_right( StringType& str,
                             const typename StringType::value_type* const s )
{
  typename StringType::size_type p = str.find_last_not_of(s);

  if (p != StringType::npos)
    str = str.substr( 0, p+1 );
} // trim_right()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove characters at the begining end at the end of a string.
 * \param str The string to modify.
 * \param s The characters to remove.
 */
template<class StringType>
void claw::text::trim( StringType& str,
                       const typename StringType::value_type* const s )
{
  typename StringType::size_type first = str.find_first_not_of(s);
  typename StringType::size_type last  = str.find_last_not_of(s);

  if (first != StringType::npos)
    str = str.substr( first, last - first + 1 );
} // trim()

/*----------------------------------------------------------------------------*/
/**
 * \brief Squeeze successive characters of a string into one character.
 * \param str The string to modify.
 * \param s The characters to remove.
 *
 * \b Example :
 * <tt>
 * std::string s("word aaa bbb abab");
 * claw::squeeze( s, "ab" );
 * std::cout << s << std::end; // result is "word a b abab"
 * </tt>
 */
template<class StringType>
void claw::text::squeeze( StringType& str,
                          const typename StringType::value_type* const s )
{
  typedef typename StringType::size_type size_type;

  size_type first(0);

  do
    {
      first = str.find_first_of(s, first);

      if ( first != StringType::npos )
        {
          size_type last = str.find_first_not_of(str[first], first+1);

          if ( last == StringType::npos )
            str = str.substr(0, first+1);
          else if ( last - first > 1 )
            str = str.substr(0, first+1) + str.substr(last);

          ++first;
        }
    }
  while ( (first != StringType::npos) && (first != str.length()) );
} // squeeze()

/*----------------------------------------------------------------------------*/
/**
 * \brief Test if the content of a string is immediately convertible to a type.
 * \param str The string to test.
 */
template<typename T, class StringType>
bool claw::text::is_of_type( const StringType& str )
{
  std::basic_istringstream< typename StringType::value_type,
    typename StringType::traits_type,
    typename StringType::allocator_type > iss(str);

  T val;
  bool result = false;

  if ( iss >> val )
    result = iss.eof();

  return result;
} // is_of_type()

/*----------------------------------------------------------------------------*/
/**
 * \brief Split a string into several substrings, according to a given
 *        separator.
 * \param sequence A back insertion sequence in which the substrings are added.
 * \param str The string to split.
 * \param sep The separator on which the string is splitted.
 */
template<class BackInsertion, class StringType>
void claw::text::split( BackInsertion& sequence, const StringType& str,
			const typename StringType::value_type sep )
{
  StringType line;
  std::basic_istringstream< typename StringType::value_type,
    typename StringType::traits_type,
    typename StringType::allocator_type > iss(str);

  while ( std::getline(iss, line, sep) )
    sequence.push_back(line);
} // split()
