/*
  CLAW - a C++ Library Absolutely Wonderful

  CLAW is a free library without any particular aim but being useful to 
  anyone.

  Copyright (C) 2005-2008 Julien Jorge

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: julien_jorge@yahoo.fr
*/
/**
 * \file meta.hpp
 * \brief Structures for meta-programming.
 * \author Julien Jorge
 */
#ifndef __CLAW_META_HPP__
#define __CLAW_META_HPP__

namespace claw
{
  namespace meta
  {
    /**
     * \brief This predicate tells if two types are the sames.
     *
     * \b Template \b parameters
     * - \a T, \a U The types to compare.
     *
     * The result can be retrieved via the if_then_else::result type.
     *
     * \author Julien Jorge
     */
    template<typename T, typename U>
    struct same_type
    {
      enum
	{
	  result = false
	};
    }; // struct same_type [false]

    template<typename T>
    struct same_type<T,T>
    {
      enum
	{
	  result = true
	};
    }; // struct same_type [true]

    /**
     * \brief Empty type.
     */
    struct no_type {};

    /**
     * \brief Use this class to make a list of types.
     *
     * \b Template \b parameters
     * - \a Head The first type of the list.
     * - \a Queue A list of the remaining types.
     *
     * All type_list methods suppose that the list ends withe the type no_type.
     *
     * \b Example
     *
     * <tt> type_list< int, typelist<float, no_type> ></tt> is a list of two
     * types. The last type (<tt>no_type</tt>) is used to indicate the end of
     * the list.
     *
     * \author Julien Jorge
     */
    template<typename Head, typename Queue>
    struct type_list
    {
      typedef Head head_type;
      typedef Queue queue_type;
    }; // struct type_list

    /**
     * \brief Check if a type is present in a type_list.
     *
     * \b Template \b paramters
     * - \a T The type to find.
     * - \a List The list in which we search the type.
     *
     * \b Type \b requirements
     * - \a List is either no_type, either a type_list<> ended with no_type.
     *
     * \author Julien Jorge
     */
    template<typename T, typename List>
    struct type_list_find
    {
      enum
	{
	  result = same_type<T, typename List::head_type>::result
	  || type_list_find<T, typename List::queue_type>::result
	};
    }; // struct type_list_find

    template<typename T>
    struct type_list_find<T, no_type>
    {
      enum
	{
	  result = same_type<T, no_type>::result
	};
    }; // struct type_list_find

    /**
     * \brief This class checks if each type in a type_list<> is present only
     *        once.
     *
     * \author Julien Jorge
     */
    template<typename List>
    struct type_list_is_a_set
    {
      enum
	{
	  result = !type_list_find<typename List::head_type,
	                           typename List::queue_type>::result 
	  && type_list_is_a_set<typename List::queue_type>::result
	};
    }; // struct type_list_is_a_set

    template<>
    struct type_list_is_a_set<no_type>
    {
      enum
	{
	  result = true
	};
    }; // struct type_list_is_a_set [no_type]

    /**
     * \brief This structure allows to do conditional typing.
     *
     * \b Template \b parameters
     * - \a Cond The result of the predicate to check.
     * - \a ThenPart The type defined when the predicate is true.
     * - \a ElsePart The type defined when the predicate is false.
     *
     * The result can be retrieved via the if_then_else::result type.
     *
     * \author Julien Jorge
     */
    template<bool Cond, typename ThenPart, typename ElsePart>
    struct if_then_else
    {

    }; // struct if_then_else

    template<typename ThenPart, typename ElsePart>
    struct if_then_else<true, ThenPart, ElsePart>
    {
      typedef ThenPart result;
    }; // struct if_then_else [true]

    template<typename ThenPart, typename ElsePart>
    struct if_then_else<false, ThenPart, ElsePart>
    {
      typedef ElsePart result;
    }; // struct if_then_else [false]

  } // namespace meta
} // namespace claw

#endif // __CLAW_META_HPP__
