/*
 *  Linux snipes, a text-based maze-oriented game for linux.
 *  Copyright (C) 1997 Jeremy Boulton.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Jeremy Boulton is reachable via electronic mail at
 *  boultonj@ugcs.caltech.edu.
 */

#include <stdlib.h>
#include <curses.h>
#include <assert.h>

#include "snipes.h"
#include "coords.h"
#include "collision.h"
#include "walls.h"
#include "hive.h"
#include "chars.h"
#include "screen.h"

#define MAX_WEAPONS	100
#define DELAY_TIME	1

#define BOUNCE_MIN	4
#define BOUNCE_VARIABLE	7

/* FIXME
 * 
 * A note about diagonals:  In the DOS version of the game, diagonal
 * shots don't go exactly diagonal.  delta.y is always 1, but delta.x
 * varies between 2 and 3 in the pattern { 2, 2, 3, 2, 3 }.  Since
 * the height of cells in the DOS version is 5 (distance between
 * horizontal walls, including the walls themselves), this means that
 * the pattern has the same period as the horizontal walls.  If a
 * weapon is even with a horizontal wall, it's next delta (going
 * upward) will be 2, then 2 again, and so on.  The sum of the numbers
 * in the pattern is 12, so that between the position at T[0] and
 * the position at T[5], 13 spaces are covered.  This is very close
 * to the distance between two vertical walls, including the walls
 * themselves, which is 12.  Just to make that perfectly clear:
 * 
 *                  |  1234567890123      |
 * ==================          |   *      |
 *                  |           o         |
 *                  |         *           |
 *                  |      o              |
 *                  |    *                |
 * ==================  o       =====================
 *                   OO
 *                   \/1234567890123
 */


int num_weapons;

int enable_weapon_kill_wall=0;
int enable_weapon_bounce=0;
int friendly_weapon_color=0;
int enemy_weapon_color=0;

struct location {
  char in_use;
  char has_collided;
  int friendly_fire;
  int bounce_count;
  int display_char_type;
  coordinate position;
  coordinate direction;
  int delay_counter;
} weapon[MAX_WEAPONS];


void init_weapons( int fcolor, int ecolor, int enable_bounce,
		   int enable_kill_wall )
{
  int i;

  friendly_weapon_color = fcolor;
  enemy_weapon_color = ecolor;
  enable_weapon_bounce = enable_bounce;
  enable_weapon_kill_wall = enable_kill_wall;
  
  num_weapons = 0;

  for( i=0; i<MAX_WEAPONS; i++ )
    weapon[i].in_use = 0;
}


/* weapon_update_has_collided( int index )
 * 
 * The weapon is not on the collision map when this is called.  That
 * way we can check it for collisions with other weapons.  If it does
 * collide with other weapons, those other weapons have already been
 * added to the collision map in this round of processing.  Since
 * weapons are processed in increasing index number order, we check
 * all lower numbered weapons' positions and eliminate whichever
 * ones are at the same location.
 */

void weapon_update_has_collided( int index )
{
  int j;

  weapon[index].has_collided = check_collisions( weapon[index].position.x,
						 weapon[index].position.y, 1, 1,
						 MAZE_OBJECT_PLAYER|
						 MAZE_OBJECT_SNIPE|
						 MAZE_OBJECT_SNIPE_GHOST|
						 MAZE_OBJECT_SNIPE_MAKER|
						 MAZE_OBJECT_WALL|
						 MAZE_OBJECT_SNIPE_BULLET|
						 MAZE_OBJECT_PLAYER_BULLET );
  
  /* Check collisions with other weapons so that those other weapons also
   * get marked as having collided.
   */
  if( weapon[index].has_collided &
      (MAZE_OBJECT_SNIPE_BULLET|MAZE_OBJECT_PLAYER_BULLET) )
    for( j=index-1; j>=0; j-- )
      if( weapon[j].in_use &&
	  weapon[j].position.x == weapon[index].position.x &&
	  weapon[j].position.y == weapon[index].position.y )
	weapon[j].has_collided=1;
}


void weapon_remove_from_collision_map( int index )
{
  if( weapon[index].friendly_fire )
    collision_map_remove_obj( weapon[index].position.x,
			      weapon[index].position.y, 1, 1,
			      MAZE_OBJECT_PLAYER_BULLET );
  else
    collision_map_remove_obj( weapon[index].position.x,
			      weapon[index].position.y, 1, 1,
			      MAZE_OBJECT_SNIPE_BULLET );
}


void weapon_add_to_collision_map( int index )
{
  if( weapon[index].friendly_fire )
    collision_map_place_obj( weapon[index].position.x,
			     weapon[index].position.y, 1, 1,
			     MAZE_OBJECT_PLAYER_BULLET );
  else
    collision_map_place_obj( weapon[index].position.x,
			     weapon[index].position.y, 1, 1,
			     MAZE_OBJECT_SNIPE_BULLET );
}



void add_weapon( coordinate p, coordinate direction, int ff )
{
  int i;
  
  if( num_weapons == MAX_WEAPONS )
    return;

  for( i=0; i<MAX_WEAPONS; i++ )
    if( !weapon[i].in_use ) {
      weapon[i].in_use = 1;
      weapon[i].has_collided = 0;
      weapon[i].position = p;
      weapon[i].direction = direction;
      weapon[i].delay_counter = DELAY_TIME - 1;
      weapon[i].friendly_fire = ff;
      weapon[i].bounce_count = BOUNCE_MIN + random()%BOUNCE_VARIABLE;
      if( ff )
	weapon[i].display_char_type = 0;
      else {
	if( direction.y == -1 )
	  weapon[i].display_char_type = 0;
	if( direction.y ==  1 )
	  weapon[i].display_char_type = 1;
	if( direction.x == -1 )
	  weapon[i].display_char_type = 2;
	if( direction.x ==  1 )
	  weapon[i].display_char_type = 3;

	/* Diagonal cases */
	if( direction.x + direction.y == 2 )
	  weapon[i].display_char_type = 5;
	if( direction.x + direction.y == -2 )
	  weapon[i].display_char_type = 5;
	if( direction.x + direction.y == 0 )
	  weapon[i].display_char_type = 4;
      }
      weapon_add_to_collision_map( i );
      num_weapons++;
      break;
    }
}


/* remove_weapon( int index, screen_coords *sc, wall_info *wi )
 * 
 * Move a weapon to unused status.  Update the collision map
 * and inform any walls or hives that the weapon may have collided
 * with.  The walls are only updated if weapons destroy walls
 * in this level.
 */

void remove_weapon( int index, screen_coords *sc, wall_info *wi )
{
  weapon_remove_from_collision_map( index );
  weapon[index].in_use = 0;
  num_weapons--;

  if( weapon[index].friendly_fire ) {
    /* Check for collisions with walls and inform the walls about them. */
    if( enable_weapon_kill_wall ) {
      if( weapon[index].direction.x == 0 || weapon[index].direction.y == 0 ) {
	if( check_collisions( weapon[index].position.x,
			      weapon[index].position.y, 1, 1,
			      MAZE_OBJECT_WALL ) )
	  set_wall_segment( wi, weapon[index].position.x,
			    weapon[index].position.y, 0 );
      }
    }
  }

  /* Check for collisions with hives and inform the hives about them. */
  if( check_collisions( weapon[index].position.x, weapon[index].position.y,
			1, 1, MAZE_OBJECT_SNIPE_MAKER ) )
    kill_hive( weapon[index].position );
}


void show_weapons( screen_coords *sc )
{
  int i;
  coordinate screen_pos;
  
  for( i=0; i<MAX_WEAPONS; i++ )
    if( weapon[i].in_use && !weapon[i].has_collided )
      if( coords_check_on_screen( sc, weapon[i].position, 0 ) ) {
	coords_to_screen_pos( sc, weapon[i].position, &screen_pos );

	screen_move( screen_pos.y, screen_pos.x );
	if( weapon[i].friendly_fire ) {
	  screen_setcolorpair(friendly_weapon_color);

	  switch ( weapon[i].display_char_type )
	  {
	    case 0:
	      screen_addch( PLAYER_BULLET_1 );
	      break;
	    case 1:
	      screen_addch( PLAYER_BULLET_2 );
	      break;
	  }
	}
	else {
	  screen_setcolorpair(enemy_weapon_color);

	  switch ( weapon[i].display_char_type )
	  {
	    case 0:
	      screen_addch( WEAPON_UP );
	      break;
	    case 1:
	      screen_addch( WEAPON_DOWN );
	      break;
	    case 2:
	      screen_addch( WEAPON_LEFT );
	      break;
	    case 3:
	      screen_addch( WEAPON_RIGHT );
	      break;
	    case 4:
	      screen_addch( WEAPON_DIAG_POS );
	      break;
	    case 5:
	      screen_addch( WEAPON_DIAG_NEG );
	      break;
	  }
	}
      }
}


/* diagonal_bounce_check( int index, screen_coords *sc )
 * 
 * This function changes the direction of a weapon if it
 * is travelling diagonally and hits a wall.  The direction
 * is not changed, however, if the weapon has already bounced
 * enough times.
 */

void diagonal_bounce_check( int index, screen_coords *sc )
{
  coordinate td, fd, tp = weapon[index].position;

  /* Bounce check */
  fd = weapon[index].direction;

  /* The weapon only bounces if it is travelling diagonally */
  if( fd.x != 0 && fd.y != 0 ) {
    coords_add_delta( sc, &tp, fd );
    
    /* The weapon bounces if, continuing diagonally, it would hit
     * a wall.
     */
    if( check_collisions( tp.x, tp.y, 1, 1, MAZE_OBJECT_WALL ) ) {
      /* If bounce_count reaches 0, the weapon has bounced enough
       * so just let it hit the wall.  Otherwise, make it bounce.
       */
      if( --weapon[index].bounce_count != 0 ) {
	/* Bouncing off a horizontal wall? */
	td = weapon[index].direction;
	td.x = 0;
	tp = weapon[index].position;
	coords_add_delta( sc, &tp, td );
	if( check_collisions( tp.x, tp.y, 1, 1, MAZE_OBJECT_WALL ) )
	  fd.y = -fd.y;

	/* Bouncing off a vertical wall? */
	td = weapon[index].direction;
	td.y = 0;
	tp = weapon[index].position;
	coords_add_delta( sc, &tp, td );
	if( check_collisions( tp.x, tp.y, 1, 1, MAZE_OBJECT_WALL ) )
	  fd.x = -fd.x;

	weapon[index].direction = fd;
      }
    }
  }
}


/* move_weapon( int index, screen_coords *sc, wall_info *wi )
 * 
 * Weapon should not be in the collision map when this is called
 * and will not be in the collision map when this returns.
 */

void move_weapon( int index, screen_coords *sc, wall_info *wi )
{
  if( weapon[index].friendly_fire ) {
    weapon[index].display_char_type = weapon[index].display_char_type?0:1;

    if( enable_weapon_bounce )
      diagonal_bounce_check( index, sc );
    coords_add_delta( sc, &weapon[index].position, weapon[index].direction );
    weapon_update_has_collided( index );
  }
  else {
    coords_add_delta( sc, &weapon[index].position, weapon[index].direction );
    weapon_update_has_collided( index );
  }
}


void move_weapons( screen_coords *sc, wall_info *wi )
{
  int i;
  
  /* Take all weapons out of the collision map.  Any weapons which
   * collided last time are eliminated this time.
   */
  for( i=0; i<MAX_WEAPONS; i++ ) {
    if( weapon[i].in_use ) {
      if( weapon[i].has_collided )
	remove_weapon( i, sc, wi );
      else
	weapon_remove_from_collision_map( i );
    }
  }

  /* For all the weapons that are left, make sure that nothing
   * has collided with them.  If not, see if they need to be
   * moved and, if so, move them.  Upon moving the weapon, again
   * check for collisions.
   */
  for( i=0; i<MAX_WEAPONS; i++ ) {
    if( weapon[i].in_use ) {

      /* See if something ran into the weapon.  If so, the weapon is
       * marked as having collided and will be eliminated next time
       * the weapons are moved.
       */
      weapon_update_has_collided( i );
      if( weapon[i].has_collided ) {
	/* Put weapon back in collision map in case other weapons
	 * collide with it.
	 */
	weapon_add_to_collision_map( i );
	continue;
      }

      /* Move the weapon if the weapon is ready to move. */
      if( ++weapon[i].delay_counter == DELAY_TIME ) {
	weapon[i].delay_counter = 0;
	move_weapon( i, sc, wi );
	/* FIXME: should the next line be outside the loop so all
	 * weapons get added back to the collision map?  Probably
	 * although I suspect that since DELAY_TIME is 1 it doesn't
	 * result in any different behavior at the moment.
	 */
	weapon_add_to_collision_map( i );
      }
    }
  }
}
